## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "alpha_colouring.tcl"
 #                                    created: 04/12/98 {22:45:38 PM} 
 #                                last update: 12/14/2001 {13:05:13 PM}}} 
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html/Alphatk.html
 #  
 # Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on use and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##

namespace eval tw {}

set tw::colorList [list blue cyan green magenta red white yellow]

# Don't remove marks or selection when recolouring.
set tw::nonColorMarkRegexp {^(mark:|(back)?sel$)}

proc colorIndex {color} {
    if {[string range $color 0 5] == "color_"} {
	regsub "color_" $color "" colorIndex
    } else {
	set colorIndex [lsearch -exact $::tw::colorList $color]
	incr colorIndex
    }
    return $colorIndex
}

# regModeKeywords [options] <mode> <keyword list> - Set keywords and comments 
#  that 
#  Alpha can recognize to color them. Specifically, in mode <mode>, every 
#  keyword specified in the list is colored non-black (blue, by default). 
#  Comments, if specified by '-e' or '-b' below, are colored red by defualt.
#  
#    -a			Keywords will be *added* to existing mode
#    			keywords.  The new keywords can be a different
#    			color than older keywords.  This flag can also
#    			be used to modify other attributes, but it
#    			cannot be used to modify colors of existing
#    			keywords.
#    				
#    -m <c>		Specify a magic character. Every word beginning with 
#                       the magic character is a keyword.
#    -e <commentstr>	Specify a string that begins comments that last to 
#			the end of the line. 
#    -b <comment beg> <comment end>	
#                       Specify a pair of strings that bracket a 
#			comment.
#    -c <color>		Comment color.
#    -k <color>		Keyword color.
#    -s <color>		String color. Alpha can color single-line
#			strings only, using the simple heuristic
#			of assuming the first two double quotes
#			constitute a string. 
#    -i <char>		Specify a character to display differently.
#			Commonly used for curly braces, etc.
#    -I <color>		Color of above characters.
#	Keywords must be less than 20 characters long.
proc regModeKeywords {args} {
    getOpts {-m -e {-b 2} -c -f -k -s -i -I} "lappend"
    set mode [lindex $args 0]
    global ${mode}::keywords ${mode}::keywordsopts ${mode}::multiComment
    global ${mode}::specialColors
    # spaces, starting the lineRegexp/lineVars vars
    append  lineRegexp "^(\[ \t\]*)"
    lappend lineVars space
    # -a option: adding additional colors
    if {![info exists opts(-a)]} {
	catch {unset ${mode}::keywords}
    } else {
	# new options override all old options which we stored in
	# the given array
	if {[info exists ${mode}::keywordsopts]} {
	    foreach o [array names ${mode}::keywordsopts] {
		ensureset opts($o) [set ${mode}::keywordsopts($o)]
	    }
	    unset ${mode}::keywordsopts
	}
    }
    # -e option:  single comment character.
    if {[info exists opts(-e)]} {
	set comment [lindex $opts(-e) 0]
	if {[string length $comment]} {
	    append  lineRegexp "\(?:\(" [quote::Regfind $comment] "\).*\)?"
	    lappend lineVars comment
	}
	# Set the comment color.
	if {[info exists opts(-c)]} {
	    set ${mode}::specialColors(comment) color[colorIndex $opts(-c)]
	} else {
	    set ${mode}::specialColors(comment) color5
	}
    }
    # -b option:  paired (bracketed) comments.
    if {[info exists opts(-b)]} {
	set ${mode}::multiComment [lindex $opts(-b) 0]
	set mC0 [lindex [set ${mode}::multiComment] 0]
	set mC1 [lindex [set ${mode}::multiComment] 1]
	# Make sure that they're both non-empty
	set emptyMulti 0
	foreach char [list $mC0 $mC1] {
	    if {![string length $char]} {set emptyMulti 1 ; break}
	}
	if {$emptyMulti} {
	    # multi comment chars were empty, so unset completely
	    unset opts(-b)
	    unset ${mode}::multiComment
	} else {
	    append lineRegexp "(?:([quote::Regfind $mC0]).*)?"
	    lappend lineVars multicomment
	}
	if {[info exists opts(-c)]} {
	    set ${mode}::specialColors(comment) color[colorIndex $opts(-c)]
	} else {
	    set ${mode}::specialColors(comment) color5
	}
    }
    # -s option:  colorizing quoted strings.
    if {[info exists opts(-s)]} {
	set ${mode}::specialColors(quote) color[colorIndex $opts(-s)]
	append  lineRegexp "((?!\\B)\"(?:\[^\\B\"\]|\\B.)*(?:\"|\\B?\\Z))?"
	lappend lineVars quote
    }
    # -k option:  colorizing keywords.  Also used for magic characters.
    if {![info exists opts(-k)]} {set opts(-k) blue}
    set keywordColor [colorIndex $opts(-k)]
    if {$keywordColor == 0} {
	foreach kw [lindex $args 1] {
	    catch {unset ${mode}::keywords($kw)}
	}
    } else {
	foreach kw [lindex $args 1] {
	    set ${mode}::keywords($kw) color$keywordColor
	}
    }
    # -m option:  magic color.
    if {[info exists opts(-m)]} {
	global ${mode}::magicPrefix
	if {$keywordColor == 0} {
	    catch {unset ${mode}::magicPrefix}
	    catch {unset ${mode}::magicColor}
	    # So we don't remember this option next time
	    unset opts(-m)
	} else {
	    set ${mode}::magicPrefix [lindex $opts(-m) 0]
	    set ${mode}::magicColor color$keywordColor
	}
    }
    # -i option:  special characters.  Requires an -I flag as well.
    if {[info exists opts(-i)]} {
	global ${mode}::specialChars
	if {![info exists opts(-I)]} {set opts(-I) 0}
	foreach char $opts(-i) {
	    set ${mode}::specialChars($char) color[colorIndex $opts(-I)]
	}
    }
    # text, ending the lineRegexp/lineVars vars.
    append lineRegexp "(.*)\$"
    lappend lineVars txt
    # The following two lines used to be wrapped in this,
    # but I don't think we want it:
    #if {![info exists ${mode}::lineRegexp]} {}
    namespace eval $mode [list set lineRegexp $lineRegexp]
    namespace eval $mode [list set lineVars $lineVars]
    # remember all the old options
    array set ${mode}::keywordsopts [array get opts]

    if {![llength [winNames -f]]} { return }
    # Now schedule a recolouring event, but only if there isn't
    # already one pending.
    set script [list tw::recolourAllModeWindows $mode]
    foreach af [after info] {
	if {[after info $af] == [list $script idle]} {
	    return
	}
    }
    after idle $script
}

proc tw::recolourAllModeWindows {mode} {
    # Now, for any window which is open, with this mode active,
    # remove all colour-related tags and recolour.
    global win::Modes win::tk
    foreach w [array names win::Modes] {
	if {$win::Modes($w) == $mode} {
	    # If this doesn't exist, we probably pre-initialised
	    # the win::Modes array with a mode for a window we're
	    # about to create, and we're just now loading up the
	    # mode for that window.
	    if {[info exists win::tk($w)]} {
		set tkw $win::tk($w)
		foreach tag [$tkw tag names] {
		    if {![regexp -- $::tw::nonColorMarkRegexp $tag]} {
			set range [$tkw tag ranges $tag]
			if {[llength $range]} {
			    eval [list $tkw tag remove $tag] $range
			}
		    }
		}
		if {[wm state [winfo toplevel $tkw]] == "normal"} {
		    # If it is visible
		    ::tw::arrangeToColour $tkw {} {}
		}
	    }
	}
    }
}
