## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "alpha_commands.tcl"
 #                                    created: 04/12/98 {23:17:46 PM} 
 #                                last update: 12/07/2001 {20:20:25 PM} 
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html
 #  
 # Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##
#==============================================================================
#= Alpha Commands
#==============================================================================
#
#In this list of routines, text between '<' and '>' is a placeholder for a 
#required parameter, text between '[' and ']' is a placeholder for an 
#optional parameter, and the '|' signifies a choice of two or more 
#alternatives.  A '+' signifies that the previous symbol can be present one 
#or more times, while a '*" means zero or more times.  Some commands have no 
#parameters, and are only expected to be called interactively.
#

#  other  #

# findTag - prompt user for a function name and attempt 
#  to use the file 'cTAGS' to locate the function's 
#  definition
proc findTag {} {
    if {[llength [winNames -f]]} {
	set name [getSelect]
    } else {
	set name ""
    }
    set name [prompt "Find which name?" $name]
    echo "unimplemented: findTag $name"
}
# createTagFile - searches all files in current file set 
#  and saves the locations of any function declarations
#  in a file called 'cTAGS'.
proc createTagFile {} {
    echo "unimplemented: createTagFile"
}

proc addAlphaChars {args} {echo "addAlphaChars $args"}

# colors <fore red> <fore green> <fore blue> <back red> <back green> <back blue>
proc colors {args} {echo "colors $args"}
# getFileInfo <file> <arr> - Given a file name, creates an array called 
#  'arr' in the current context, containing fields 'created', 'creator', 
#  'modified', 'type', 'datalen', and 'resourcelen'. 'created' and 
#  'modified' are in a form suitable for the command 'mtime'.
proc getFileInfo {f a} {
    upvar $a arr
    file stat $f arr
    regsub -all -- "-" [file attributes $f] "" l
    array set arr $l
    set arr(creator) $f
    set arr(type) TEXT
    set arr(modified) [file mtime $f]
    set arr(datalen) [file size $f]
    set arr(created) $arr(ctime)
}

# setFileInfo <file> <field> [arg] - Allows some information to be set 
#  for files. Settable fields are 'modified', 'created', 'creator', and 
#  'type' all take an argument. 'resourcelen' can be set, but doesn't take 
#  an argument and just removes the resource fork.
proc setFileInfo {file f {arg ""}} {
    if {![file exists $file]} {
	error "No such file: $file"
    }
    switch -- $f {
	"modified" {
	    if {[info tclversion] > 8.2} {
		file mtime $file $arg
	    } else {
		alertnote "Unimplemented: setFileInfo $file modified; upgrade to Tcl 8.3 or newer"
	    }
	}
	"created" {
	    echo "Unimplemented: setFileInfo $file created"
	}
	"resourcelen" {
	    echo "Unimplemented: setFileInfo $file resourcelen"
	}
	default {
	    file attributes $file -$f $arg
	}
    }
}

proc getModifiers {args} {return 0}
# insertColorEscape <pos> <color ind> [hypertext func] - Create a color 
#  or style "point" for documentation purposes. Look at the file 
#  "docColors.tcl" for examples. The hypertext func is only used when the 
#  "point" is underline. See 'getColors' for info about the current file.
proc insertColorEscape {pos color {func ""}} {
    global alphaPriv
    if {$color == 0 || $color == 12} {
	text::color \
	  $alphaPriv(insertColorEscapePos) $pos \
	  color$alphaPriv(insertColorEscapeColor) \
	  $alphaPriv(insertColorEscapeFunc)
	unset alphaPriv(insertColorEscapePos) 
	unset alphaPriv(insertColorEscapeColor)
	unset alphaPriv(insertColorEscapeFunc) 
    } else {
	set alphaPriv(insertColorEscapePos) $pos
	set alphaPriv(insertColorEscapeColor) $color
	set alphaPriv(insertColorEscapeFunc) $func
    }
}
proc removeColorEscapes {} {
    # currently does nothing
}

namespace eval text {}

proc text::hyper {from to hyper} {
    text::color $from $to 15 $hyper
}

proc text::color {from to colour {hyper ""}} {
    switch -- $colour {
	black       {set colour 0}
	blue        {set colour 1}
	cyan        {set colour 2}
	green       {set colour 3}
	magenta     {set colour 4}
	red         {set colour 5}
	white       {set colour 6}
	yellow      {set colour 7}
	bold        {set colour 8}
	condensed   {set colour 9}
	extended    {set colour 10}
	italic      {set colour 11}
	normal      {set colour 12}
	outline     {set colour 13}
	shadow      {set colour 14}
	underline   {set colour 15}
    }
    text_cmd tag add color$colour $from $to
    if {$colour == 15 && [string length $hyper]} {
	# do hyper
	text_cmd addHyper $from $to $hyper
    }
}

# This isn't required by Tcl only Alpha.
proc linkVar {args} {}

# execute - prompt user for a function or macro. The 
#  tab key acts as a "completion" command.
proc execute {} {
    uplevel \#0 [status::prompt "execute:"]
}

proc beep {args} {bell}

proc selectLimits {} {
    if {[set res [text_cmd tag ranges sel]] == ""} {
	return [text_cmd tag ranges backsel]
    } else {
	return $res
    }
}

proc rgbToColor {triplet} {
    set rgb "#"
    foreach c $triplet {
	append rgb [format "%04x" $c]
    }
    return $rgb
}

proc setRGB {col args} {
    global win::tktitle
    if {[llength $args] != 3} {
	# not sure what this command with no args is supposed to do
	error "Bad args to setRGB"
    }
    set rgb [rgbToColor $args]
    if {![regsub "color_" $col "color" col]} {
	set idx [lsearch -exact {blue cyan green magenta red white yellow} $col]
	if {$idx < 0} {
	    if {$col == "background" || $col == "foreground"} {
		foreach c [array names win::tktitle] {
		    $c configure -$col "$rgb"
		}
		return
	    } else {
		puts stderr "Bad colour $col"
	    }
	}
	incr idx
	set col "color$idx"
    }
    
    foreach c [array names win::tktitle] {
	$c tag configure $col -foreground "$rgb"
    }
}

# breakIntoLines <string> - return 'string' with 
#  carriage returns and spaces inserted to satisfy 
#  'leftFillColumn' and 'fillColumn' variables.
proc breakIntoLines {t} {
    global leftFillColumn fillColumn
    set unset {}
    if {![info exists leftFillColumn]} {
	set leftFillColumn 0
	lappend unset leftFillColumn
    }
    if {![info exists fillColumn]} {
	set fillColumn 70
	lappend unset fillColumn
    }
    set width [expr {$fillColumn - $leftFillColumn + 1}]
    if {$t == ""} { return $t }
    regsub -all "\n" $t "\r" t
    #regsub -all "  +" $t " " t
    append t " "
    while 1 {
	if {$t == ""} {
	    break
	}
	set first [string first "\r" $t]
	if {$first != -1 && $first < $width} {
	    append res [string range $t 0 $first]
	    set t [string trimleft [string range $t [expr {$first +1}] end] " "]
	    continue
	}
	set a [string range $t 0 $width]
	set where [string last " " $a]
	if {$where == -1} {
	    set t [string range $t [expr {$width +1}] end]
	} else {
	    set t [string trimleft [string range $t [expr {$where +1}] end] " "]
	    set a [string range $a 0 [expr {$where -1}]]
	}
	append res $a "\r"
    }
    set left [string range "                 " 1 $leftFillColumn]
    regsub -all "(^|\r)" $res "&$left" res
    foreach v $unset {
	unset $v
    }
    return [string trimright $res]
}

# watchCursor - turns the cursor into a a watch cursor.
proc watchCursor {} {
    global hasWatch
    # not a very good implementation.  May choose wrong window
    # and should really use an idletask not an 'after'
    if {![info exists hasWatch]} {
	set hasWatch 1
	if {[win::Current] != ""} {
	    text_cmd configure -cursor watch
	    after 1000 "text_cmd configure -cursor xterm ; unset hasWatch"
	} else {
	    . configure -cursor watch
	    after 1000 ". configure -cursor xterm ; unset hasWatch"
	}
	return 1
    } else {
	return 0
    }
}

# wc <file>... - counts chars, words, lines of input files.
proc wc {args} {
    foreach f $args {
	wordCount [file::readAll $f]
    }
}

#  save quit undo  #

# quit - quits ALPHA
if {[info commands __quit] == ""} {
    rename quit __quit
}
proc quit {args} {
    # need to check files aren't dirty
    global win::NumDirty
    if {[info exists win::NumDirty] && ($win::NumDirty > 0)} {
	if {![dialog::yesno "There are windows with unsaved changes.  Are you \
	  sure you wish to quit?"]} {
	    return
	}
    }
    global win::Active
    if {[info exists win::Active]} {
	while {[llength $win::Active]} {
	    killWindow [lindex $win::Active 0] 0 "Discard Changes"
	}
    }
    quitHook
    __quit
}

proc saveAll {} {
    global win::Active
    foreach w $win::Active {
	save $w
    }
}


# redo - redo the next action that has been undone but 
#  not redone
proc redo {} {
    global win::tk win::Current
    tw::redo $win::tk($win::Current)
}
# undo - undo the last action that has not been undone
proc undo {} {
    global win::tk win::Current
    tw::undo $win::tk($win::Current)
}

#  Basic gui stuff  #

# winNames [-f] - return a TCL list of all open windows. If '-f' option 
#  specified, complete pathnames are returned.
proc winNames {{full ""}} {
    global win::Active
    if {$full == "-f"} { return $win::Active }
    set res {}
    foreach f $win::Active {
	if {[win::IsFile $f]} {
	    lappend res [file tail $f]
	} else {
	    lappend res $f
	}
    }
    set res
}

# moveWin [win name] <left> <top> - moves current or specified window. 
#  The window name can be "StatusWin".
proc moveWin {x y args} {
    if {[llength $args]} {
	set win [winTailToFullName $x]
	set x $y
	set y [lindex $args 0]
    } else {
	global win::Active
	set win [lindex $win::Active 0]
    }
    global win::tk
    wm geometry [winfo toplevel $win::tk($win)] +${x}+${y}
    # For some reason, without this update, if we do a moveWin, but
    # then immediately throw up a dialog, the moveWin is ignored.
    # It would be good to track down exactly where the problem happens...
    update idletasks
}

# sizeWin [win name] <width> <height> - sets size of current or specified window.
#  The window name can be "StatusWin", although only the width can be 
#  changed.
proc sizeWin {w h args} {
    if {[llength $args]} {
	set win [winTailToFullName $w]
	set w $h
	set h [lindex $args 0]
    } else {
	global win::Active
	set win [lindex $win::Active 0]
    }
    global win::tk
    wm geometry [winfo toplevel $win::tk($win)] ${w}x${h}
}
# new [-g <l> <t> <w> <h>] [-n <name>] - opens an untitled window.
#Can optionally provide left and top coordinates, plus width and
#height.  All or none.
proc new {args} {
    set opts(-n) "untitled"
    getOpts [list {-g 4} -n]
    
    if {![info exists opts(-g)]} {
	global defWidth defHeight defTop defLeft
	set opts(-g) [list $defLeft $defTop $defWidth $defHeight]
    }
    
    alpha::embedInto $opts(-n) [concat [list toplevel $opts(-n)] $opts(-g)]
}
# sendToBack <winName> - Send named window to back.
proc sendToBack {w} {echo "sendToBack $w"}
# setWinInfo [-w <win>] <field> <arg> - Sets a piece of data about either 
#  the current or a specified window. Settable fields 'platform', 'state', 
#  'read-only', 'tabsize', 'dirty', and 'shell'. 'shell' means that dirty 
#  flag ignored and undo off.
proc setWinInfo {field arg args} {
    if {$field == "-w"} {
	set win $arg
	set field [lindex $args 0]
	set arg [lindex $args 1]
    } else {
	set win [win::Current]
    }
    switch -- $field {
	"platform" {
	    echo "Ignored 'platform' argument to setWinInfo"
	}
	"state" {
	    echo "Ignored 'state' argument to setWinInfo"
	}
	"readonly" -
	"read-only" {
	    global ::win::tk
	    tw::read_only $win::tk($win) $arg
	}
	"tabsize" {
	    global ::win::tk
	    tw::setTabSize $win::tk($win) $arg
	}
	"dirty" {
	    global ::win::tk
	    tw::dirty $win::tk($win) $arg
	}
	"shell" {
	    global ::tw::shell ::win::tk
	    global ::tw::[set win::tk($win)]
	    if {$arg} {
		set ::tw::[set win::tk($win)](shell) 1
	    } else {
		catch {unset ::tw::[set win::tk($win)](shell)}
	    }
	    
	}
	"encoding" {
	    global ::win::tk
	    tw::encoding $win::tk($win) $arg
	}
	default {
	    error "Bad arg '$field' to setWinInfo"
	}
    }
}
# splitWindow [percent] - toggle having window split into two panes. 
#  Optional arg specifies percent of window to allocate to the first pane. 
proc splitWindow {args} {
    text_cmd toggleSplit
}

# toggleScrollbar - toggles horizontal scrollbar on frontmost window. 
#  Will not succeed if scrollbar scrolled.
proc toggleScrollbar {} {
    text_cmd horizScrollbar
}

proc winTailToFullName {n} {
    global win::tk
    if {[file exists $n] || \
      ([regsub { <[0-9]+>$} $n {} name] && [file exists $name])} {
	set n [file nativename $n]
    }
    if {![info exists win::tk($n)]} {
	# it was just the tail of the name
	foreach nm [array names win::tk] {
	    if {[file tail $nm] == $n} {
		return $nm
	    }
	}
	if {![info exists win::tk($n)]} {
	    error "Window $n not found!"
	}
    }
    return $n
}

# bringToFront <winName> - Bring named window to front.
proc bringToFront {n {deactivate 1}} {
    global win::tk win::Active
    
    set n [winTailToFullName $n]

    if {[set old [lindex $win::Active 0]] != ""} {
	if {$old != $n} {
	    # if this flag wasn't set, we just killed the last window
	    if {$deactivate} {
		deactivateHook $n
	    }
	}
    }
    set w [winfo toplevel $win::tk($n)]

    if {![string length [$w cget -use]]} {
	wm deiconify $w
    }
    
    # if this flag wasn't set, we killed the last window, so always
    # want to activate this one.
    if {!$deactivate || ($old != $n)} {
	#puts stderr "bringToFront $old -> $n"
	activateHook $n
    }
    
    # This 'update idletasks' is required with Tcl8.3 on Windows
    raise $w ; update idletasks ; focus $win::tk($n)
}

# closeAll - close all windows
proc closeAll {} {
    global win::Active
    foreach w $win::Active {
	catch {killWindow $w}
    }
}

# nextWindow - select next window
proc nextWindow {} {
    global win::Active
    bringToFront [lindex $win::Active 1]
}
# otherPane - If window is split, select the other pane.
proc otherPane {args} {
    text_cmd otherPane
}

# prevWindow - select previous window
proc prevWindow {args} {
    global win::Active
    bringToFront [lindex $win::Active 1]
}

# getGeometry [win] - return a TCL list containing the left 
#  edge of the current window, the top, the width, and height.
proc getGeometry {{w ""}} {
    global win::tk win::Current
    if {$w == ""} {
	set w $win::Current
    } else {
	set w [winTailToFullName $w]
    }
    
    set g [split [winfo geometry [winfo toplevel $win::tk($w)]] "x+."]
    return [concat [lrange $g 2 3] [lrange $g 0 1]]
}
# getMainDevice - return a list containing the left, top, right, and 
#  bottom of the rectangle defining the main device.
proc getMainDevice {} {
    return [list 0 0 [winfo screenwidth .] [winfo screenheight .]]
}
# getWinInfo [-w <win>] <arr> - Creates an array in current context 
#  containing info about either the current or a specified window. Array 
#  has fields 'state', 'platform', 'read-only', 'tabsize', 'split', 
#  'linesdisp' (num lines that can be seen in the window), 'currline' 
#  (first line displayed), and 'dirty'.
proc getWinInfo {ar args} {
    switch -- [llength $args] {
	0 { 
	    global win::Active win::tk
	    set w [lindex $win::Active 0]
	    uplevel [list upvar \#0 ::tw::[set win::tk($w)] $ar]
	}
	2 {
	    set w [lindex $args 0] 
	    set ar [lindex $args 1]
	    uplevel [list upvar \#0 ::tw::[set win::tk($w)] $ar]
	}
	default {
	    error "wrong args to getWinInfo"
	}
    }
    set tkw $win::tk($w)
    set lines [expr {int([$tkw index end])}]
    set yview [$tkw yview]
    set currline [expr {int($lines * [lindex $yview 0])}]
    set linesdisp [expr {int($lines * ([lindex $yview 1] - [lindex $yview 0]))}]
    uplevel [list set ${ar}(currline) $currline]
    uplevel [list set ${ar}(linesdisp) $linesdisp]
    return ""
}
# icon [-f <winName>] [-c|-o|-t|-q] [-g <h> <v>] - Having to do w/ 
#  iconifying windows. '-c' means close (iconify) window, '-o' open, '-t' 
#  toggle open/close, '-q' returns either a '1' for an iconified window or a 
#  '0' for an uniconified window, and '-g' moves the icon to horizontal 
#  position <h> and vertical position 'v'. Options are executed as they 
#  are parsed, so the '-f' option, if present, should always be first. 
proc icon {args} {
    getOpts {-f}
    if {[info exists opts(-f)]} {
	set w $opts(-f)
    } else {
	set w [win::Current]
    }
    global win::tk
    set w [winfo toplevel $win::tk($w)]
    set state [wm state $w]
    if {[info exists opts(-q)]} {
	if {$state == "normal"} { return 0 } else {
	    return 1
	}
    } elseif {[info exists opts(-c)]} {
	if {![string length [$w cget -use]]} {
	    wm iconify $w
	}
    } elseif {[info exists opts(-o)]} {
	if {![string length [$w cget -use]]} {
	    wm deiconify $w
	}
    } elseif {[info exists opts(-t)]} {
	if {$state == "normal"} { 
	    if {![string length [$w cget -use]]} {
		wm iconify $w
	    }
	} else {
	    if {![string length [$w cget -use]]} {
		wm deiconify $w
	    }
	}
    }
}
# killWindow - kill current window
proc killWindow {{wn ""} {destroy_in_progress 0} {dirty_behaviour ""}} {
    global win::tk win::Active win::tktitle
    if {$wn == ""} {
	set wn [lindex $win::Active 0]
    }
    if {[info exists win::tk($wn)]} {
	if {[winfo exists $win::tk($wn)]} {
	    set w [winfo toplevel $win::tk($wn)]
	} else {
	    regexp {^\.[^.]+} $win::tk($wn) w
	}
	if {[winfo exists $w]} {
	    getWinInfo -w $wn winfo
	    if {$winfo(dirty)} {
		if {$dirty_behaviour == ""} {
		    set dirty_behaviour [buttonAlert \
		      "That window has unsaved changes. \
		      What shall I do?" "Discard Changes" "Save first" "Cancel"]
		}
		switch -- $dirty_behaviour {
		    "Discard Changes" {
			# do nothing
		    }
		    "Save first" {
			save
		    }
		    "Cancel" {
			error "Cancelled"
		    }
		}
	    }
	    if {[winfo exists $win::tk($wn)]} {
		# remove any possible bindings which may trigger
		# side-effects (esp. for destroy)
		bindtags $win::tk($wn) $win::tk($wn)
	    }
	    hook::callAll preCloseHook * $wn
	    #wm withdraw $w
	    # All sorts of nasty recursive loops can arise if we don't remove
	    # these two bindings.  Such loops generally result in wish crashing
	    # (obviously not ideal behaviour, and it ought to catch the infinite
	    # loop, but anyway, we should write nice code too ;-)
	    ::bind $w <Destroy> ""
	    wm protocol $w WM_DELETE_WINDOW ""
	    bindtags $w $w
	    tw::windowCleanup $win::tk($wn)
	    if {!$destroy_in_progress} {
		destroy $w
	    }
	} else {
	    hook::callAll preCloseHook * $wn
	    if {[winfo exists $win::tk($wn)]} {
		# remove any possible bindings which may trigger
		# side-effects (esp. for destroy)
		bindtags $win::tk($wn) $win::tk($wn)
	    }
	    tw::windowCleanup $win::tk($wn)
	}
    } else {
	echo "Couldn't cleanup $wn"
    }
    
    
    if {[catch {closeHook $wn} err]} {
	echo "Bad error in closehook; please report bug: $err"
    }
    set twidget $win::tk($wn)

    unset win::tktitle($twidget)
    unset win::tk($wn)
    global tw::$twidget
    catch {unset tw::$twidget}
    # remove the wrapper proc
    if {[info commands $twidget] != ""} {
	rename $twidget ""
    }
    if {[llength $win::Active]} {
	bringToFront [lindex $win::Active 0] 0
	update
    }
}

#  Time and timing  #
# now
#  Returns the current time as Macintosh seconds. This is the number 
#  of seconds that have elapsed since Midnight Jan 1, 1904.
proc now {} {clock seconds}
# mtime <time> [long|short|abbrev]
#  Returns a date and time string using the Macintosh International Utilities. The 
#  long/short/abbrev specification corresponds to the date. These are the following 
#  formats:
#	short		3/16/92 9:20:46 PM
#	abbrev	Mon, Mar 16, 1992 9:20:49 PM
#	long		Monday, March 16, 1992 9:20:43 PM
#  The returned value actually is in the form of a list. To get text as 
#  above, run the result through 'join', as in "join [mtime [now] short]".
proc mtime {when {how "short"} {gmt 0}} {
    switch -- $how {
	"long" {
	    set c [clock format $when -gmt $gmt]
	    return [list [concat [lrange $c 0 2] [lindex $c end]] [lindex $c 3]]
	}
	"short" {
	    return [clock format $when -format [list "%m/%d/%Y" "%H:%M:%S %p"] -gmt $gmt]
	}
	"abbrev" {
	    echo "mtime $when $how not complete"
	    return [clock format $when -format "%m/%d/%Y %H:%M:%S %p" -gmt $gmt]
	}
    }
}
# ticks
#  Returns the current TickCount. Ticks are 60ths of a seconds. TickCount is the 
#  number of ticks since the Macintosh was started. The command:
#		puts stdout [expr "[ticks] / 60"]
#  will print the number of seconds since the Macintosh was booted.
proc ticks {} {clock clicks}
# iterationCount - allows actions to be repeated many times. "control-u 44 
#  =" inserts 44 '='s into current window.  Also can be used to execute any 
#  function or macro (including the keyboard macro) many times.  Defaults to 
#  4.
proc iterationCount {args} {echo "iterationCount $args"}

#  Printing  #
# pageSetup - display the printing PageSetup dialog.
proc pageSetup {} {
    alertnote "There are currently no 'page setup' options.  Just print away..."
}
# print - print front window
proc print {{f ""}} {
    global printerFont printerFontSize
    if {$f == ""} {
	set f [win::Current]
    }
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"unix" {
	    exec enscript $f
	}
	"windows" {
	    if {$printerFont != ""} {
		print_file $f 0 "$printerFont $printerFontSize"
	    } else {
		print_file $f 0
	    }
	}
	"macintosh" {
	}
    }
    
}

proc printAll {} {
    global win::Active
    foreach f $win::Active {
	print $f
    }
}

#  Filesystem  #

# cp <fromName>+ <toName>
#  This command will copy the file fromName and name the new file toName, 
#  overwriting any existing file. This command copies both data forks, and 
#  the Finder information. 
proc cp {args} {eval file copy $args}
# mkdir <name> - creates a directory (folder) named 
#  'name' in the current directory.
proc mkdir {args} {eval file mkdir $args}
# moveFile <fromName> <toName>
#  This command will move the file fromName to toName, overwriting any 
#  existing file. The move can not be made across volume (disk drives) 
#  boundaries. 
proc moveFile {args} {eval file rename $args}
proc copyFile {args} {eval file copy $args}
# removeFile <fileName>
#  This command will delete the file or folder 'fileName'.
proc removeFile {args} {eval file delete $args}
# rmdir <dirname> - remove a directory (folder)
proc rmdir {args} {eval file delete $args}



#  Not that important  #

# # largestPrefix <list> - Returns the longest prefix contained in all 
#  strings of 'list'.
proc largestPrefix {list} {
    # we only use this where the list is alphabetical
    set first [lindex $list 0]
    set last [lindex $list end]
    set len [string length $first]
    set i 0
    while {[string index $first $i] == [string index $last $i]} {
	if {$i == $len} {
	    break
	}
	incr i
    }
    return [string range $first 0 [expr {$i -1}]]
}
# abortEm - aborts whatever is currently happening
proc abortEm {} {
    global alpha::abort
    set alpha::abort 1
}

#  Even less important  #

# zapInvisibles - removes chars < ascii 32, except for
#  LF's and CR's.
proc zapInvisibles {args} {echo "zapInvisibles $args"}
# abbrev <label> <string> [<mode>] - register a label for <string>. See 
#'execAbbrev'.
proc abbrev {args} {echo "abbrev $args"}
# backColor - set background color
proc backColor {args} {echo "backColor $args"}
# dumpColors - dump current foreground and background
#  colors into the current buffer in Alpha-readable
#  format.
proc dumpColors {args} {echo "dumpColors $args"}
# currentPosition - displays current and total bytes.
proc currentPosition {} {
    alertnote "Current: [getPos], maximum: [maxPos]"
}
# execAbbrev - looks at current word and tries to expand it. Labels are 
#  specified using 'abbrev'.
proc execAbbrev {args} {echo "execAbbrev $args"}
# fileInfo - prompts for a file, and displays type, 
#  creator, sizes of both data and resource forks, last
#  modification time, and creation time
proc fileInfo {} {
    set f [getfile]
    foreach {a v} [file attributes $f] {
	append res "[string range $a 1 end] : $v\n"
    }
    alertnote $res
}
# fileRemove - prompts for a file, and removes it
proc fileRemove {} {file delete [getfile "Delete which file?"]}
# freeMem - give a rough approximation of the current 
#  memory reserves of ALPHA
proc freeMem {args} {echo "freeMem $args"}
# getAscii - displays the ASCII code for character at 
#  current insertion point
proc getAscii {args} {echo "getAscii $args"}
# getColors - returns list of colors/hypertext for current document. 
#  Format is list of lists, each sublist consisting of file offset, color 
#  index, and possibly a hypertext command.
proc getColors {} {
    global win::tk
    set res {}
    foreach {on color index} [text_cmd dump -tag [minPos] [maxPos]] {
	set color [string range $color 5 end]
	if {![regexp {[0-9]} [string index $color 0]]} {
	    set color [lsearch -exact {blue cyan green magenta red white yellow} $color]
	    incr color
	    if {$color == 0} {
		set color 1
	    }
	}
	if {$on == "tagon"} {
	    if {$color == 15} {
		set hyper [array get ::tw::$win::tk([win::Current]) hyper:$index:*]
		set hyper [lindex $hyper 1]
		if {[string length $hyper]} {
		    lappend res [list $index $color $hyper]
		} else {
		    lappend res [list $index $color]
		}
	    } else {
		lappend res [list $index $color]
	    }
	} else {
	    if {$color == 15} {
		lappend res [list $index 12]
	    } else {
		lappend res [list $index 0]
	    }
	}
    }
    set res
}
# insertAscii - prompts for an ASCII code and inserts
#  into text.
proc insertAscii {args} {echo "insertAscii $args"}
# insertFile - prompts for a file name and inserts the
#  corresponding file into the current window. Not
#  undoable.
proc insertFile {args} {echo "insertFile $args"}
# insertPathName - present the user w/ a SFGetFIle dialog 
#  and paste the complete path-name of the chosen file
#  into the current window
proc insertPathName {args} {echo "insertPathName $args"}
# mousePos - Returns list <row,col> of mouse position, if the mouse is 
#  currently over the active window. Otherwise, return error (catch w/ 
#  'catch').
proc mousePos {args} {echo "mousePos $args"}
# thinkReference <-t|-l> <think reference page> - Interact with Think 
#  Reference 2.0. '-t' retrieves a template without leaving Alpha, '-l' 
#  switches to Think Reference and goes to the specified page.
proc thinkReference {args} {echo "thinkReference $args"}


