## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "alpha_dialogs.tcl"
 #                                    created: 04/11/98 {17:32:52 PM} 
 #                                last update: 12/12/2001 {11:36:49 AM} 
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html
 #  
 # Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##

#  Dialogs  #

# alertnote message_string
#  This command will display message_string in a standard Macintosh alert box.
proc alertnote {message} {
    tk_messageBox -message $message -parent [dialog::findParent]
    update
}
# colorTriple [<prompt>] [<red> <green> <blue>] - Prompts user to choose 
#  color. If specified, the input RGB value is used as the initial color on 
#  the colorpicker.
proc colorTriple {{prompt ""} args} {
    if {[llength $args] == 0} {
	set init 000
    } elseif {[llength $args] == 3} {
	foreach c $args {
	    set c1 [format %1X [expr {$c / 256}]]
	    if {[string length $c1] == 1} {set c1 "0$c1"}
	    append init $c1
	}
    } else {
	error "Bad args to colorTriple"
    }
    if {$prompt != ""} {
	set res [tk_chooseColor -title $prompt -parent [dialog::findParent] \
	  -initialcolor "#${init}"]
    } else {
	set res [tk_chooseColor -parent [dialog::findParent] \
	  -initialcolor "#${init}"]
    }
    if {$res == ""} { error "Cancelled" }
    foreach c [split [string range $res 1 end] ""] {
	lappend numbers [format %1d 0x$c]
    }
    set red [expr {[lindex $numbers 0] * 4096 + [lindex $numbers 1] * 256}]
    set green [expr {[lindex $numbers 2] * 4096 + [lindex $numbers 3] * 256}]
    set blue [expr {[lindex $numbers 4] * 4096 + [lindex $numbers 5] * 256}]
    return [list $red $green $blue]
}

# findFile [<path>] - open a file in a new window. An optional path parameter
#  specifies a default directory or file.
proc findFile {{default ""}} {
    if {$default == ""} { set default [file dirname [win::Current]]}
    if {[file isdirectory $default]} {
	edit [tk_getOpenFile -parent [dialog::findParent] \
	  -initialdir $default -filetypes [findFileTypes]]
    } else {
	edit [tk_getOpenFile -parent [dialog::findParent] \
	  -initialfile $default -filetypes [findFileTypes]]
    }
}
# getfile [<prompt>] [<path>]
#  This command will display an SFGetFile() and return the full path name of the 
#  selected file, or an error if CANCEL button was selected.  An optional path 
#  parameter specifies a default directory or file.
proc getfile {args} {
    set opts(-types) ""
    getOpts [list -types]
    set title "Find file"
    set default ""
    if {[llength $args]} {
	set title [lindex $args 0]
	if {[llength $args] > 1} {
	    set default [lindex $args 1]
	    if {[llength $args] > 2} {
		return -code error "Too many arguments"
	    }
	}
    }
    if {$default == ""} { set default [file dirname [win::Current]]}
    if {[file isdirectory $default]} {
	set res [tk_getOpenFile -title $title -initialdir $default \
	  -parent [dialog::findParent] -filetypes [findFileTypes $opts(-types)]]
    } elseif {[file isfile $default]} {
	set res [tk_getOpenFile -title $title -initialfile $default \
	  -parent [dialog::findParent] -filetypes [findFileTypes $opts(-types)]]
    } else {
	set res [tk_getOpenFile -title $title \
	  -parent [dialog::findParent] -filetypes [findFileTypes $opts(-types)]]
    }
    if {$res == ""} {
	error "Cancelled"
    } else {
	return $res
    }
}

proc findFileTypes {{useThesePatterns ""}} {
    if {[llength $useThesePatterns]} {
	lappend filetypes [list "Allowed files" $useThesePatterns]
	return $filetypes
    } else {
	global tcl_platform openAllFiles
	if {$tcl_platform(platform) == "macintosh"} {
	    if {$openAllFiles} {
		return ""
	    } else {
		return [list [list "Text files" "" "TEXT"]]
	    }
	} else {
	    if {$tcl_platform(platform) == "windows"} {
		lappend filetypes [list "All Files" "*.*"]
	    } else {
		lappend filetypes [list "All Files" "*"] [list "Invisible Files" ".*"]
	    }
	    global filepats
	    foreach m [lsort -dictionary [array names filepats]] {
		if {$filepats($m) != ""} {
		    lappend filetypes [list "$m files" $filepats($m)]
		}
	    }
	    return $filetypes
	}
    }
}

# getChar - waits for a keystroke, returns ascii.
# This implementation is a bit complex, since I just copied it from
# the more general status::prompt procedure and made it return after
# one key.
bind KeyCapture <KeyPress>  {set alphaPriv(key) [list 1 %A %K %N]}
bind KeyCapture <KeyRelease>  {set alphaPriv(key) [list 0 %A %K %N]}

proc getChar {{representation "key"}} {
    global alphaPriv alpha::modifiers alpha::mods
    set alphaPriv(status) ""
    set oldFocus [focus]
    set oldGrab [grab current .status]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab -global .status
    catch {destroy .status.e}
    entry .status.e -textvariable alphaPriv(status)
    pack .status.e -side left
    trace variable alphaPriv(status) w status::_helper
    # have problem in that the bindings below trigger and screw us
    # Not sure why the keycapture binding prevents alphaPriv(done)
    # from being set to 1
    bind .status.e <Return> "set alphaPriv(done) 1 ; set alphaPriv(key) {}; break"
    bindtags .status.e [concat .status.e KeyCapture [bindtags .status.e]]
    focus .status.e
    set alphaPriv(mods) 0
    if {[info exists alphaPriv(done)]} {unset alphaPriv(done)}
    while 1 {
	vwait alphaPriv(key)
	set statuscontents $alphaPriv(status)
	#echo "$alphaPriv(key)"
	if {[info exists alphaPriv(done)]} {
	    # real key press
	    if {$alphaPriv(done) == 1} {
		trace vdelete alphaPriv(status) w status::_helper
		break;
	    }
	    unset alphaPriv(done)
	} else {
	    regsub -all -- {_[LR]} [set keycode [lindex $alphaPriv(key) 2]] "" keycode
	    if {[info exists alpha::mods($keycode)]} {
		if {[lindex $alphaPriv(key) 0]} {
		    # pressed so add to list of mods
		    set alphaPriv(mods) [expr {$alphaPriv(mods) | $alpha::mods($keycode)}]
		} else {
		    # released so remove
		    set alphaPriv(mods) [expr {($alphaPriv(mods) | $alpha::mods($keycode)) ^ $alpha::mods($keycode)}]
		}
		unset alphaPriv(key)
		continue
	    } else {
		# it was a real key
		if {$representation == "anything" && [lindex $alphaPriv(key) 0]} {
		    append statuscontents $keycode
		} else {
		    unset alphaPriv(key)
		    continue
		}
		
	    }
	}
	set first ""
	set last ""
	regexp -- {^(.*)(.)$} $statuscontents "" first last
	break
	
	
    }
    if {[info exists alphaPriv(done)]} {unset alphaPriv(done)}
    catch {unset alphaPriv(key)}
    catch {destroy .status.e}
    catch {grab release .status}
    dialog::restoreFocus $oldFocus
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    switch -- $representation {
	"key" {
	    return $last
	}
	default {
	    error "No other representations supported"
	}
    }	

}

namespace eval status {}

proc status::makePrompt {w} {
    #catch {destroy $w}
    #label $w -text "" -background [default::color lightbackground] \
    #  -borderwidth 0 -padx 0
    #pack $w -side left
    grab $w
    bind all <Button-1> [list focus -force $w]
    
    # The status bar nearly works without any of this stuff, just with the two
    # simple 'press/release' bindings below.  However, it doesn't quite work,
    # because if a modifier-key is already held down when the status::prompt
    # is first called, then we never notice that with the simpler code.
    # For example if 'ctrl-s' is used for an interactive search, a sequence
    # of 'ctrl-s ctrl-w' in which the 'ctrl' is never released is seen as
    # ctrl-s w.  The more complex code handles this case by explicitly binding
    # to each possible modifier.
    global alpha::modifier_keys alpha::mods
    for {set i 0} {$i < 16} {incr i} {
	set key ""
	set themods 0
	set count 1
	foreach mod [list Shift Control [lindex $alpha::modifier_keys 0] [lindex $alpha::modifier_keys 2]] {
	    if {[expr {$i & $count}]} {
		append key "$mod-"
		set themods [expr {$themods | $alpha::mods($mod)}]
	    }
	    set count [expr {2*$count}]
	}
	bind $w <${key}KeyPress> "status::pressed 1 %A %K %N $themods ; break"
	bind $w <${key}KeyRelease> "status::pressed 0 %A %K %N $themods ; break"
    }
    
    #bind $w <KeyPress> "status::pressed 1 %A %K %N"
    #bind $w <KeyRelease> "status::pressed 0 %A %K %N"
    bind $w <Return> "set alphaPriv(done) 1 ; set alphaPriv(key) {}; break"
}

proc status::pressed {press ascii mods other {modval 0}} {
    global alphaPriv
    if {$modval != 0} {
	if {$press} {
	    set alphaPriv(mods) [expr {$alphaPriv(mods) | $modval}]
	} else {
	    set alphaPriv(mods) [expr {($alphaPriv(mods) | $modval) ^ $modval}]
	}
    }
    set alphaPriv(key) [list $press $ascii $mods $other $modval]
    #puts stdout "pressed: $press $ascii $mods $other $modval"
}

proc status::flash {color {wait 100}} {
    set oldbg [.status cget -bg]
    .status configure -background $color
    .status.text configure -background $color
    update
    after $wait "set waiting 1"
    vwait waiting
    .status configure -background $oldbg
    .status.text configure -background $oldbg
}

proc coreKeyPrompt {prompt} {
    global alphaPriv alpha::modifiers alpha::mods
    set oldFocus [focus]
    set oldGrab [grab current .status]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    #message $prompt
    status::makePrompt .status.text
    set err 0
    set res ""
    
    # have problem in that the bindings below trigger and screw us
    # Not sure why the keycapture binding prevents alphaPriv(done)
    # from being set to 1
    focus .status.text
    set alphaPriv(mods) 0
    if {[info exists alphaPriv(done)]} {unset alphaPriv(done)}
    set statuscontents ""
    while {1} {
	vwait alphaPriv(key)
	if {[info exists alphaPriv(done)]} {
	    break
	} else {
	    regsub -all -- {_[LR]} [set keycode [lindex $alphaPriv(key) 2]] "" keycode
	    if {[info exists alpha::mods($keycode)]} {
		if {[lindex $alphaPriv(key) 0]} {
		    # pressed so add to list of mods
		    set alphaPriv(mods) [expr {$alphaPriv(mods) | $alpha::mods($keycode)}]
		} else {
		    # released so remove
		    set alphaPriv(mods) [expr {($alphaPriv(mods) | $alpha::mods($keycode)) ^ $alpha::mods($keycode)}]
		}
		unset alphaPriv(key)
		continue
	    } else {
		# it was a real key
		if {[lindex $alphaPriv(key) 0]} {
		    if {[string length $keycode] > 1} {
			# So we don't append 'question', but rather ?
			append statuscontents [lindex $alphaPriv(key) 1]
		    } else {
			append statuscontents $keycode
		    }
		} else {
		    unset alphaPriv(key)
		    continue
		}
		
	    }
	    set first ""
	    set last ""
	    regexp -- {^(.*)(.)$} $statuscontents "" first last
	    set res [list $last $alphaPriv(mods)]
	    break
	}
    }
    
    if {[info exists alphaPriv(done)]} {unset alphaPriv(done)}
    catch {unset alphaPriv(key)}
    catch {grab release .status.text}
    dialog::restoreFocus $oldFocus
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    bind all <Button-1> ""
    return -code $err $res
}

#proc status::prompt {prompt {func ""} {add "key"}}
if {0} {
proc status::prompt {args} {
    global alphaPriv alpha::modifiers alpha::mods
    set oldFocus [focus]
    set oldGrab [grab current .status]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    set opts(-add) key
    getOpts {-command -add}
    switch -- [llength $args] {
	default {
	    error "status::prompt ?-f -add what -command script? prompt ?oldfunc? ?add?"
	}
	1 {
	    set prompt [lindex $args 0]
	    set func ""
	}
	2 {
	    foreach {prompt func} $args {}
	}
	3 {
	    foreach {prompt func opts(-add)} $args {}
	}
    }
    if {[info exists opts(-f)]} {
	status::flash black
    }
    if {[info exists func] && [string length $func]} {
	set oldstyle 1
    } else {
	catch {unset func}
	set oldstyle 0
    }
    
    message $prompt
    makePrompt .status.text
    set err 0
    set res ""
    
    # have problem in that the bindings below trigger and screw us
    # Not sure why the keycapture binding prevents alphaPriv(done)
    # from being set to 1
    focus .status.text
    set alphaPriv(mods) 0
    if {[info exists alphaPriv(done)]} {unset alphaPriv(done)}
    set statuscontents ""
    while 1 {
	if {!$oldstyle} {
	    set statuscontents ""
	}
	vwait alphaPriv(key)
	if {[info exists alphaPriv(done)]} {
	    break
	} else {
	    regsub -all -- {_[LR]} [set keycode [lindex $alphaPriv(key) 2]] "" keycode
	    if {[info exists alpha::mods($keycode)]} {
		if {[lindex $alphaPriv(key) 0]} {
		    # pressed so add to list of mods
		    set alphaPriv(mods) [expr {$alphaPriv(mods) | $alpha::mods($keycode)}]
		} else {
		    # released so remove
		    set alphaPriv(mods) [expr {($alphaPriv(mods) | $alpha::mods($keycode)) ^ $alpha::mods($keycode)}]
		}
		unset alphaPriv(key)
		continue
	    } else {
		# it was a real key
		if {[lindex $alphaPriv(key) 0]} {
		    if {[string length $keycode] > 1} {
			# So we don't append 'question', but rather ?
			append statuscontents [lindex $alphaPriv(key) 1]
		    } else {
			append statuscontents $keycode
		    }
		} else {
		    unset alphaPriv(key)
		    continue
		}
		
	    }
	}
	#puts stderr "$statuscontents , $alphaPriv(key)"
	set first ""
	set last ""
	regexp -- {^(.*)(.)$} $statuscontents "" first last
	if {$oldstyle} {
	    set addargs [list $first]
	} else {
	    set addargs {}
	}
	switch -- $opts(-add) {
	    "key" {
		lappend addargs $last
	    }
	    "modifiers" -
	    "anything" {
		lappend addargs $last $alphaPriv(mods)
	    }
	}
	if {[info exists func]} {
	    if {[info exists opts(-debug)]} {
		tclLog "$func $addargs"
	    }
	    if {[set err [catch [list uplevel 1 $func $addargs] res]]} {
		if {[info exists opts(-debug)]} {
		    global errorInfo
		    tclLog "$res $errorInfo"
		}
		break
	    }
	} elseif {[info exists opts(-command)]} {
	    if {[info exists opts(-debug)]} {
		tclLog "$opts(-command) $addargs"
	    }
	    if {[set err [catch [list uplevel 1 $opts(-command) $addargs] res]]} {
		if {[info exists opts(-debug)]} {
		    global errorInfo
		    tclLog "$res $errorInfo"
		}
		break
	    }
	} else {
	    append res $statuscontents
	}
	message "[.status.text cget -text]$res"
	if {[info exists alphaPriv(key)]} {
	    unset alphaPriv(key)
	}
	
    }
    catch {unset alphaPriv(done)}
    catch {unset alphaPriv(key)}
    catch {grab release .status.text}
    dialog::restoreFocus $oldFocus
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    bind all <Button-1> ""
    return -code $err $res
}
}

# statusPrompt <prompt> [<func>] - Prompt in the status window. If 'func' 
#  is present, call this routine at each key-press with the current 
#  contents of the status line and the key, insert into statusline 
#  whatever is returned by the func. Command-v pastes the current (<80 
#  char) clipboard contents on the status line.
proc statusPrompt {prompt {func ""} args} {
    global alphaPriv
    set alphaPriv(status) ""
    set oldFocus [focus]
    set oldGrab [grab current .status]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    set flag ""
    if {$prompt == "-f" || $prompt == "--"} {
	set flag $prompt
	set prompt $func
	set func [lindex $args 0]
    }
    if {$flag == "-f"} {
	status::flash black
    }
    message $prompt
    catch {destroy .status.e}
    entry .status.e -textvariable alphaPriv(status)
    pack .status.e -side left
    grab .status.e
    trace variable alphaPriv(status) w status::_helper
    bind .status.e <Return> "set alphaPriv(done) 1"
    focus .status.e
    bind all <Button-1> [list focus -force .status.e]

    while 1 {
	vwait alphaPriv(done)
	if {$alphaPriv(done) == 1} {
	    trace vdelete alphaPriv(status) w status::_helper
	    break;
	}
	set first ""
	set last ""
	regexp -- {^(.*)(.)$} $alphaPriv(status) "" first last
	if {$func != ""} {
	    if {[catch [list uplevel 1 $func [list $first $last]] res]} {
		trace vdelete alphaPriv(status) w status::_helper
		break;
	    }
	}
	unset alphaPriv(done)
    }
    if {[info exists alphaPriv(done)]} {unset alphaPriv(done)}
    catch {destroy .status.e}
    catch {grab release .status.e}
    dialog::restoreFocus $oldFocus
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    bind all <Button-1> ""
    return $alphaPriv(status)
}

proc status::_helper {args} {
    global alphaPriv
    if {![info exists alphaPriv(done)]} {
	set alphaPriv(done) 0
    }
    
}

# askyesno [-c] prompt
#  This command will display a Macintosh alert box with 'prompt' displayed
#  with the push buttons Yes and No. The command will return the 
#  string "yes" or "no". The '-c' flag specifies that a cancel button be 
#  used as well.
proc askyesno {text {other ""}} {
    set root [dialog::findRoot]
    set buttons {yes no}
    if {$other != ""} {
	if {$text == "-c"} {
	    lappend buttons "cancel"
	    set text $other
	} else {
	    error "bad args"
	}
    }
    global tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	set i [eval [list tk_dialog ${root}.f "" $text stop 0] $buttons]
    } else {
	set i [eval [list tk_dialog ${root}.f "" $text "" 0] $buttons]
    }
    return [lindex $buttons $i]
}
# buttonAlert <prompt> [<button>] - Create a dialog w/ the specified 
#  buttons, returning the one selected.
proc buttonAlert {prompt args} {
    set root [dialog::findRoot]
    global tcl_platform 
    if {$tcl_platform(platform) == "macintosh"} {
	set i [eval [list tk_dialog ${root}.f "" $prompt stop 0] $args]
    } else {
	set i [eval [list tk_dialog ${root}.f "" $prompt "" 0] $args]
    }
    return [lindex $args $i]
}

# dialog [<-w width>|<-h height>|<-b title l t r b>|<-c title val l t r b>|
#			<-t text l t r b>|<-e text l t r b>|<-r text val l t r b>|
#			<-p l t r b>]+ 
#  Create and display a dialog.  '-w' and '-h' allow width and height of 
#  dialog window to be set.  '-b', '-c', '-r', '-t', '-e' and '-p' allow 
#  buttons, checkboxes, radio buttons, static text, editable text and gray 
#  outlines to be created, respectively.  All control types (except gray 
#  outlines) require specification of a title or text, together with left, 
#  top, right, and bottom coordinates.  Checkboxes and radioboxes have an 
#  additional parameter, the default value.  At least one button must be 
#  specified.  The return value is a list containing resulting values for 
#  all buttons, radioboxes, checkboxes, and editable textboxes (static text 
#  is ignored).  Buttons have value '1' if chosen, '0' otherwise.  The 
#  dialog box exits at the first button press.
#
proc dialog {d1 d1d d2 d2d args} {
    #puts "dialog $d1 $d1d $d2 $d2d [lrange $args 0 end]"
    global tcl_platform alphaPriv
    
    # Set up a unique dialog window, and dialog storage array.
    # This allows this procedure to be re-entrant.
    set dialogNumber 1
    while {[info exists alphaPriv(atkdialog$dialogNumber)]} {
	incr dialogNumber
    }
    set dial atkdialog$dialogNumber
    global $dial
    set root [dialog::findRoot]
    set w [set alphaPriv($dial) ${root}.dl$dialogNumber]
    catch {destroy $w}
    # Remember old focus
    set oldFocus [focus]

    toplevel $w -class Dialog
    wm title $w ""
    wm iconname $w Dialog
    wm protocol $w WM_DELETE_WINDOW { }

    # Dialog boxes should be transient with respect to their parent,
    # so that they will always stay on top of their parent window.  However,
    # some window managers will create the window as withdrawn if the parent
    # window is withdrawn or iconified.  Combined with the grab we put on the
    # window, this can hang the entire application.  Therefore we only make
    # the dialog transient if the parent is viewable.
    if {[winfo viewable [winfo toplevel [winfo parent $w]]]} {
	wm transient $w [winfo toplevel [winfo parent $w]]
    }    
    if {$tcl_platform(platform) == "macintosh"} {
	::tk::unsupported::MacWindowStyle style $w movableDBoxProc
    }

    # extract -w, -h
    set arg($d1) $d1d
    set arg($d2) $d2d
    set _w [expr {([winfo screenwidth $w] - $arg(-w))/2}]
    if {$_w < 0} { set _w 0}
    set _h [expr {([winfo screenheight $w] - $arg(-h))/2}]
    if {$_h < 0} { set _h 0}
    wm geometry $w $arg(-w)x$arg(-h)+${_w}+${_h}

    set i 0
    set j 0
    set val 0
    set havebutton ""
    set haveentry ""
    set in $w
    set nothingYet 1
    set ${dial}(pageStyle) "normal"
    set ${dial}(window) $w
    
    set len [llength $args]
    if {[lindex $args [expr {$len -2}]] == "-help"} {
	set help [lindex $args end]
    }
    set got_dnd 0

    if {[catch {
	while 1 {
	switch -- [set type [lindex $args $i]] {
	    "-b" {
		set label [lindex $args [expr {$i +1}]]
		button $w.d$j -text $label \
		  -command "set ${dial}(button) $val"
		# first button is the default
		if {$havebutton == ""} {
		    bind OverallDialog <Return> "
			$w.d$j configure -state active -relief sunken
			update idletasks
			after 100
			set ${dial}(button) $val
			break
		    "
		    $w.d$j configure -default active
		    set havebutton $j
		}
		if {$label == "Cancel"} {
		    bind $w <Escape> "
		    $w.d$j configure -state active -relief sunken
		    update idletasks
		    after 100
		    set ${dial}(button) $val
		    break
		    "
		    wm protocol $w WM_DELETE_WINDOW [list event generate $w <Escape>]
		}
		incr val
		incr i 2
		if {[lindex $args $i] == "-set"} {
		    foreach {cmd item} [lindex $args [incr i]] {}
		    if {[regexp {^(\+|\-)} $item]} {
			set item [expr {$item + $j}]
		    }
		    # If we're changing a future item, it won't exist
		    # so we must evaluate this at the end.
		    lappend lateEval "$w.d$j configure -command \
		      \[[list dialog::createItemSetCommand $w.d$item $cmd]\]"
		    incr i
		}
		# A delay flag makes a button disabled until a certain time
		if {[lindex $args $i] == "-delay"} {
		    set time [lindex $args [incr i]]
		    incr i
		    $w.d$j configure -state disabled
		    after $time [list catch [list $w.d$j configure -state active]]
		}
	    }
	    "-l" {
		set listitems [lindex $args [expr {$i +1}]]
		set showitems [lindex $args [expr {$i +2}]]
		listbox $w.d$j -height $showitems 
 		eval [list $w.d$j insert end] $listitems
		set isList($val) $w.d$j
		incr i 3
		if {[lindex $args $i] == "-dnd"} {
		    incr i
		    dialog::bindDnd $w.d$j [lindex $args $i] \
		      [lindex $args [incr i]]
		    set got_dnd 1
		    incr i
		}
	    }
	    "-c" {
		if {[lindex $args [expr {$i+2}]] == ""} {
		    set defval 0
		} else {
		    set defval [lindex $args [expr {$i+2}]]
		}
		set ${dial}(var$val) $defval
		checkbutton $w.d$j -text \
		  [lindex $args [expr {$i +1}]] \
		  -variable ${dial}(var$val) -anchor w 
		incr val
		incr i 3
	    }
	    "-t" {
		set label [lindex $args [expr {$i +1}]]
		label $w.d$j -text $label \
		  -anchor w -wraplength [expr {$arg(-w) -20}] -justify left
		incr i 2
		if {[lindex $args $i] == "-dnd"} {
		    incr i
		    dialog::bindDnd $w.d$j [lindex $args $i] \
		      [lindex $args [incr i]]
		    set got_dnd 1
		    incr i
		}
	    }
	    "-i" {
		set label [lindex $args [expr {$i +1}]]
		label $w.d$j -image $label \
		  -anchor w
		incr i 2
	    }
	    "-e" {
		set show ""
		set econtents [lindex $args [expr {$i +1}]]
		incr i 2
		while {1} {
		    switch -- [lindex $args $i] {
			"-password" {
			    set show "\u2022"
			    incr i
			}
			"-dnd" {
			    incr i
			    dialog::bindDnd $w.d$j [lindex $args $i] \
			      [lindex $args [incr i]]
			    set got_dnd 1
			    incr i
			}
			"--" {
			    incr i
			    break
			}
			default {break}
		    }
		}
		set eheight [expr {[lindex $args [expr {$i +3}]] - \
		  [lindex $args [expr {$i + 1}]]}]
		global defaultFont fontSize 
		if {$eheight > 20} {
		    if {$show != ""} {
			return -code error "Can't have multi-line\
			  password fields"
		    }
		    # multi-line; use text widget
		    text $w.d$j -wrap char -relief sunken \
		      -bd 2 -highlightthickness 0 -font "$defaultFont $fontSize"
		    $w.d$j insert end $econtents
		    set isText($val) $w.d$j
		} else {
		    if {$eheight < 3} {
			set show "\u2022"
		    }
		    set ${dial}(var$val) $econtents
		    entry $w.d$j -textvariable ${dial}(var$val) \
		      -font "$defaultFont $fontSize" -show $show
		}
		if {$haveentry == ""} {
		    set haveentry $j
		}
		incr val
	    }
	    "-mt" -
	    "-m" {
		if {$type == "-mt"} {
		    set menutitle [lindex $args [expr {$i+1}]]
		    incr i
		}
		set items [lindex $args [expr {$i +1}]]
		set curr [lindex $items 0]
		set items [lrange $items 1 end]
		if {[lsearch -exact $items $curr] == -1} {
		    set curr [lindex $items 0]
		}
		set ${dial}(var$val) $curr
		if {![llength $items]} { 
		    if {$curr == ""} {
			label $w.d$j -text "(no options available)"
		    } else {
			label $w.d$j -text "$curr"
		    }
		} else {
		    set style [set ${dial}(pageStyle)]
		    # Only allow first multi-choice item to be a tabbed
		    # dialog.
		    if {[info exists lastvar]} {
			set style normal
		    }
		    switch -- $style {
			"normal" {
			    # Alpha's option-menus auto-shrink to the size of
			    # the largest item they contain, so we cheat
			    set auto_size 1
			    # Make the option menu.
			    eval alpha_optionMenu $w.d$j ${dial}(var$val) $items
			}
			"tabbed" {
			    package require Iwidgets
			    iwidgets::tabnotebook $w.d$j -clientdata ${dial}(var$val)
			    foreach item $items {
				$w.d$j add -label $item
			    }
			}
		    }
		    # To handle '-n' options, we must set this special variable
		    if {![info exists lastvar]} {
			set lastvar var$val
			set ${dial}(allitems) $items
			set ${dial}(pageController) $w.d$j
			bind $w <Down> "dialog_pane_change $dial $w 1 var$val cursor"
			bind $w <Up> "dialog_pane_change $dial $w -1 var$val cursor"
		    }
		}
		incr val
		incr i 2
	    }
	    "-copyto" {
		set to [lindex $args [expr {$i +1}]]
		if {[regexp {^(\+|\-)} $to]} {
		    set to [expr {$to + $j -1}]
		}
		set copyto([expr {$j -1}]) $to
		trace variable ${dial}(var[expr {$val -1}]) w \
		  "dialog::copyTo $dial $to"
		incr i 2
		continue
	    }
	    "-M" {
		incr i
		continue
	    }
	    "-v" {
		incr i
		continue
	    }
	    "-T" {
		set title [lindex $args [expr {$i +1}]]
		incr i 2
		continue
	    }
	    "-p" {
		#echo "dialog option -p ignored"
		incr i 5
		continue
	    }
	    "-pagestyle" {
		set ${dial}(pageStyle) [lindex $args [expr {$i +1}]]
		if {[lsearch -exact [list "normal" "tabbed"] [set ${dial}(pageStyle)]] == -1} {
		    return -code error "Bad page style '[set ${dial}(pageStyle)]'; should be\
		      'normal' or 'tabbed'"
		}
		incr i 2
		continue
	    }
	    "-n" {
		set isMultiPage 1
		set f [dialog_pageitem_to_frame $dial $w.subf \
		  [lindex $args [expr {$i +1}]]]
		frame $f
		if {$in == $w} {
		    trace variable ${dial}($lastvar) w \
		      "dialog_pane_change $dial $w"
		}
		set in $f
		incr i 2
		continue
	    }
	    "-help" {
		# we usually have extracted the help in advance.
		if {![info exists help]} {
		    set help [lindex $args [expr {$i +1}]]
		}
		incr i 2
		continue
	    }
	    "-r" {
		set te [lindex $args [expr {$i +1}]]
		set ${dial}(var$val) "radiobutton $te"
		radiobutton $w.d$j -text $te \
		  -value $te -anchor w 
		if {[lindex $args [expr {$i+2}]]} {
		    global selectedButton
		    set selectedButton $te
		}
		incr val
		incr i 3
	    }
	    default {
		echo "dialog $d1 $d1d $d2 $d2d $args"
		destroy $w
		error "dialog:argument [lindex $args $i] not handled"
	    }
	    "" {
		# Usually only reached if we have a multi-page dialog which ends
		# immediately after a new page, and isn't robustly constructed.
		break
	    }
	    
	}
	if {$nothingYet} {
	    if {[lsearch -exact [list -t -p] $type] == -1} {
		set nothingYet 0
		# This is the first active item
		bind $w <KeyPress> "dialog::a_key $w.d$j %A"
	    }
	}
	
	if {[lindex $args $i] == "-font"} {
	    # ignore font specifications for the moment.
	    incr i 2
	}
	foreach {l t r b} [lrange $args $i [expr {$i+3}]] {}
	# Special password handling
	if {[info exists eheight] && ($show != "")} {
	    if {($b - $t) < 10} {incr b 10}
	    unset eheight
	}
	if {[info exists auto_size]} {
	    place $w.d$j -in $in -x $l -y $t -height [expr {$b -$t}]
	    unset auto_size
	    if {[info exists menutitle]} {
		label $w.dm$j -text $menutitle
		set lwidth [winfo reqwidth $w.dm$j]
		place $w.dm${j} -in $in -x [expr {$l - $lwidth}] -y $t -height [expr {$b - $t}]
		unset menutitle
	    }
	} else {
	    if {($r == $l)} {
		place $w.d$j -in $in -x $l -y $t
		set r [expr {$l + [winfo reqwidth $w.d$j]}]
	    }
	    set hh [expr {$b - $t}]
	    if {$j == $havebutton && $hh < 21 && ($tcl_platform(platform) != "windows")} {
		set hdiff [expr {26 - $hh}]
		set t [expr {$t - $hdiff/2}]
		set hh 26
	    }
	    place $w.d$j -in $in -x $l -y $t -width [expr {$r -$l}] -height $hh
	}
	bindtags $w.d$j [concat [list "OverallDialog"] [bindtags $w.d$j]]
	incr i 4
	if {[info exists help]} {
	    if {$type != "-t" && $type != "-p" && $type != "-l"} {
		# add the first help index
		set helpitem [lindex $help 0]
		set help [lrange $help 1 end]
		if {[string length $helpitem]} {
		    eval balloon::help $w.d$j [split $helpitem "|"]
		}
	    }
	}
	incr j

	# reached end?
	if {[lindex $args $i] == ""} {break}
    }
    if {$havebutton == ""} {
	destroy $w
	error "Must have a button in the dialog!"
    }
    
    if {[info exists lateEval]} {
	foreach script $lateEval {
	    eval $script
	}
    }
    dialog::evaluateDnds

    global tcl_platform
    if {($tcl_platform(platform) == "windows") && $got_dnd} {
	# workaround bug in dnd
	if {[info exists isMultiPage] && [info exists lastvar]} {
	    wm geometry $w +5000+5000
	    dialog_pane_change $dial "" "" $lastvar cycle
	}
    }
    
    if {$in != $w} {
	# have to place the correct item
	dialog_pane_change $dial $w "" $lastvar ""
    }

    # end of big 'catch'
    } err]} {
	destroy $w
	return -code error $err
    }

    # 5. Create a <Destroy> binding for the window that sets the
    # button variable to -1;  this is needed in case something happens
    # that destroys the window, such as its parent window being destroyed.

    bind $w <Destroy> {set ${dial}(button) -1}
    
    # 6. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    # If the window has an icon, now is the time to set it
    alpha::setIcon $w
    set x [expr {[winfo screenwidth $w]/2 - $arg(-w)/2 \
	    - [winfo vrootx [winfo parent $w]]}]
    set y [expr {[winfo screenheight $w]/2 - $arg(-h)/2 \
	    - [winfo vrooty [winfo parent $w]]}]
    wm geom $w +$x+$y
    if {[info exists title]} {
	if {$tcl_platform(platform) == "windows"} {
	    # The standard Windows titlebar font doesn't seem to have
	    # an ellipsis... except for Windows 2000 (= NT 5.0)
	    # I need to know whether this is required on Win95/98...
	    if {($tcl_platform(os) != "Windows NT") \
	      || ($tcl_platform(osVersion) != 5.0)} {
		regsub -all "" $title "..." title
	    }
	}
	wm title $w $title
    }
    wm deiconify $w
    
    # 7. Set a grab and claim the focus too.

    set oldGrab [grab current $w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $w
    if {$haveentry != ""} {
	focus $w.d$haveentry
	if {[winfo class $w.d$haveentry] == "Entry"} {
	    $w.d$haveentry selection range 0 end
	} else {
	    # must be text widget
	    if {[string length [$w.d$haveentry get 1.0 "end -1c"]]} {
		$w.d$haveentry tag add sel 1.0 "end -1c"
	    }
	}
    } else {
	focus $w.d$havebutton
    }
    
    global alpha::modifier_keys
    bind OverallDialog <[lindex $alpha::modifier_keys 0]-KeyPress> \
      "dialog::cmd_key $w %K"
    # 8. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.

    bindtags $w [concat [list "OverallDialog"] [bindtags $w]]
    bind OverallDialog <Control-Return> \
      "dialog::redirectToFocus <Return> ; break"
    bind OverallDialog <Control-Tab> \
      "dialog::redirectToFocus <Tab> ; break"
    bind OverallDialog <Control-Shift-Tab> \
      "dialog::redirectToFocus <Shift-Tab> ; break"
    bind OverallDialog <Tab> {dialog::tabToWindow [tk_focusNext %W] ; break}
    bind OverallDialog <Shift-Tab> {dialog::tabToWindow [tk_focusPrev %W] ; break}

    #puts "$w, [focus]"
    #puts "[bindtags $w], [bindtags [focus]]"
    #bind $w <F1> {puts [focus] ; puts "tags: [bindtags [focus]]"}
    
    tkwait variable ${dial}(button)
    #parray ${dial}
    if {[info exists lastvar]} {
	catch {trace vdelete ${dial}($lastvar) w "dialog_pane_change $dial $w"}
	catch {unset ${dial}(allitems)}
    }
    dialog::restoreFocus $oldFocus
    set res {}
    for {set k 0} {$k < $val} {incr k} {
	if {[info exists isText($k)]} {
	    lappend res [$isText($k) get 1.0 "end -1c"]
	} elseif {[info exists isList($k)]} {
	    lappend res [$isList($k) get 0 end]
	} else {
	    if {[info exists ${dial}(var$k)]} {
		if {[string range [set ${dial}(var$k)] 0 10] == "radiobutton"} {
		    global selectedButton
		    if {$selectedButton == [string range [set ${dial}(var$k)] 12 end]} {
			lappend res 1
		    } else {
			lappend res 0
		    }
		} else {
		    #echo "alphaPriv(var$k) = [set alphaPriv(var$k)]"
		    set tmpres [set ${dial}(var$k)]
		    if {[info exists ${dial}(mapalphaPriv(var$k):$tmpres)]} {
			set actual [set ${dial}(mapalphaPriv(var$k):$tmpres)]
			unset ${dial}(mapalphaPriv(var$k):$tmpres)
			set tmpres $actual
		    }
		    lappend res $tmpres
		}
		unset ${dial}(var$k)
	    } else {
		if {[set ${dial}(button)] == $k} {
		    lappend res 1
		} else {
		    lappend res 0
		}
	    }
	} 
    }
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# ${dial}(button) doesn't get reset by it.
	wm withdraw $w
	bind $w <Destroy> {}
	destroy $w
    }
    # This is a hack to make sure our variables are unset.
    # Unfortunately, despite the fact that we unset them correctly
    # above, when we destroy the window, the tk_optionMenu items
    # may be set again....
    foreach v [array names ${dial}] {
	if {[string match "var*" $v]} { unset ${dial}($v) }
	if {[string match "map*" $v]} { unset ${dial}($v) }
    }
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    update idletasks
    unset alphaPriv($dial)
    unset $dial

    return $res

}

namespace eval dialog {}

# Copy of Tk's private 'tk::TabToWindow'.
proc dialog::tabToWindow  {w} {
    if {[string equal [winfo class $w] Entry]} {
	$w selection range 0 end
	$w icursor end
    }
    focus $w
}

# Used to help perform over-rides of events.  We want 'Return' and
# 'Tab' in a dialog to have a special effect, but then we want the
# user to be able to use 'Control-Return/Tab' for the original
# purpose of Tab/Return.
proc dialog::redirectToFocus {event} {
    set w [focus]
    if {$w == ""} {return}
    #puts "redirect $w $event"
    set tags [bindtags $w]
    set idx [lsearch -exact $tags "OverallDialog"]
    if {$idx < 0} {
	event generate $w $event
	return
    }
    bindtags $w [lreplace $tags $idx $idx]
    catch {event generate $w $event}
    catch {
	bindtags $w $tags
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::a_key" --
 # 
 #  When the user presses a straightforward key in a dialog, this procedure
 #  is called.  If the first item in the dialog is suitable for keyboard
 #  matching (e.g. it is a popup menu or a listbox), we try to match
 #  the item the user is spelling out.
 # -------------------------------------------------------------------------
 ##
proc dialog::a_key {w key} {
    variable string_so_far
    #puts stdout "$w $key [winfo class $w]"
    if {[lsearch -exact [list Menubutton Listbox] [winfo class $w]] == -1} {
	return
    }
    append string_so_far $key
    #puts stderr $string_so_far
    after cancel [list set dialog::string_so_far ""]
    after 1000 [list set dialog::string_so_far ""]
    switch -- [winfo class $w] {
	"Menubutton" {
	    set m [$w cget -menu]
	    set last [$m index end]
	    for {set i 0} {$i < $last} {incr i} {
		regsub -all " " [string tolower [$m entrycget $i -label]] "" item
		if {[string first $string_so_far $item] == 0} {
		    $m invoke $i
		    return
		}
	    }
	}
	"Listbox" {
	    set last [$w index end]
	    for {set i 0} {$i < $last} {incr i} {
		regsub -all " " [string tolower [$w get $i]] "" item
		if {[string first $string_so_far $item] == 0} {
		    set cur [$w curselection]
		    if {[llength $cur]} {
			eval $w selection clear $cur
		    }
		    $w selection set $i
		    $w see $i
		    return
		}
	    }
	}
    }
}

proc dialog::cmd_key {w key} {
    set key [string tolower $key]
    foreach child [winfo children $w] {
	switch -- [winfo class $child] {
	    "Checkbutton" -
	    "Button" {
		if {[string tolower [string index [$child cget -text] 0]] == $key} {
		    # contents of tk::ButtonInvoke
		    if {[string compare [$child cget -state] "disabled"]} {
			set oldRelief [$child cget -relief]
			set oldState [$child cget -state]
			$child configure -state active -relief sunken
			update idletasks
			after 100
			$child configure -state $oldState -relief $oldRelief
			uplevel #0 [list $child invoke]
		    }
		    return
		}
	    }
	}
    }
}

proc dialog::restoreFocus {to} {
    if {![catch {win::Current} cw]} {
	global ::win::tk
	catch {focus $::win::tk($cw)}
    } else {
	catch {focus $to}
    }
}

# These two procs have to destroy .startup at least under
# Gnome on Linux, otherwise startup messages/errors are
# hidden underneath the splash screen.
proc dialog::findRoot {} {
    set root [focus]
    if {[string length $root] < 2} {
	set root ""
    }
    if {[winfo exists .startup]} {
	global tcl_platform
	if {$tcl_platform(platform) == "unix"} {
	    destroy .startup
	    return ""
	}
	return .startup
    } else { 
	return $root
    }
}

proc dialog::findParent {} {
    set root [focus]
    if {![string length $root]} {
	set root .
    }
    if {[winfo exists .startup]} {
	global tcl_platform
	if {$tcl_platform(platform) == "unix"} {
	    destroy .startup
	    return .
	}
	return .startup
    } else { 
	return $root
    }
}

proc dialog::createItemSetCommand {item cmd} {
    if {[winfo class $item] == "Listbox"} {
	return [list dialog::itemSet [list dialog::listboxSetContents $item] $cmd]
    } else {
	return [list dialog::itemSet [list dialog::labelSetContents $item] $cmd]
    }
}

proc dialog::labelSetContents {w varInfo val} {
    set width [winfo width $w]
    set valwidth [font measure [$w cget -font] $val]
    #puts [list $w $var $varInfo $width $valwidth]
    if {$valwidth > $width} {
	set view [dialog::abbreviate $varInfo $val $width]
    } else {
	set view $val
    }
    $w configure -text $view
}

proc dialog::listboxSetContents {w varInfo val} {
    #puts [list dialog::listboxSetContents $w $varInfo $val $view]
    $w delete 0 end
    eval [list $w insert end] $val
}

proc dialog::evaluateDnds {} {
    update idletasks
    variable dnds
    if {[info exists dnds]} {
	catch {eval [join $dnds "\n"]}
	unset dnds
    }
}

proc dialog::bindDnd {w dial varInfo} {
    variable dnds
    
    set mimetype [dialog::valGetMimeType $varInfo]
    if {![string length $mimetype]} {
	catch {puts stderr "Ignored dnd binding on $varInfo"}
	return
    }
    
    set cmdCheck [list dialog::itemAcceptable $varInfo]
    set cmdSet [dialog::valGetDropAction $varInfo]
    lappend cmdSet $dial $varInfo
    
    lappend dnds [list BindDialogItem $w \
      [dialog::createItemSetCommand $w $cmdSet] \
      $cmdCheck $mimetype 30]
}

proc dialog::copyTo {dial item var elt op} {
    global $dial isText
    if {[info exists isText($item)]} {
	set w $isText($item)
	$w delete 0 end
	$w insert 0 [set ${dial}($elt)]
	$w select range 0 end    
    } elseif {[info exists ${dial}(var$item)]} {
	# Set to nothing first, so we clear the current selection.
	set ${dial}(var$item) ""
	# Now set to the correct value
	set ${dial}(var$item) [set ${dial}($elt)]
    } else {
	alertnote "dialog::copyTo error"
    }
}

# Following proc modified from:
# optMenu.tcl --
#
# This file defines the procedure tk_optionMenu, which creates
# an option button and its associated menu.
#
# RCS: @(#) $Id$
#
# Copyright (c) 1994 The Regents of the University of California.
# Copyright (c) 1994 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

# tk_optionMenu --
# This procedure creates an option button named $w and an associated
# menu.  Together they provide the functionality of Motif option menus:
# they can be used to select one of many values, and the current value
# appears in the global variable varName, as well as in the text of
# the option menubutton.  The name of the menu is returned as the
# procedure's result, so that the caller can use it to change configuration
# options on the menu or otherwise manipulate it.
#
# Arguments:
# w -			The name to use for the menubutton.
# varName -		Global variable to hold the currently selected value.
# args -		Any number of values for the option, must be >= 1,
#                       and at least one must be non-dimmed.

proc alpha_optionMenu {w varName args} {
    global multiColumnMenusEveryNItems
    upvar #0 $varName var

    foreach firstVal $args {
	if {!([string index $firstVal 0] == "\(")} {
	    set firstValue $firstVal
	    break
	}
    }
    if {![info exists firstValue]} {
	error "No legal value for option menu given!"
    }
    if {![info exists var]} {
	set var $firstValue
    }
    menubutton $w -textvariable $varName -indicatoron 1 -menu $w.menu \
      -relief raised -bd 2 -highlightthickness 2 -anchor c \
      -direction flush
    menu $w.menu -tearoff 0
    set count $multiColumnMenusEveryNItems
    foreach i $args {
	incr count -1
	if {([string range $i 0 1] == "\(-") || ($i == "-")} {
	    $w.menu add separator
	} else {
	    if {[string index $i 0] == "\("} {
		$w.menu add radiobutton -label [string range $i 1 end] \
		  -variable $varName -state disabled
	    } else {
		$w.menu add radiobutton -label $i -variable $varName
	    }
	    if {$count < 0} {
		set count $multiColumnMenusEveryNItems
		$w.menu entryconfigure $i -columnbreak 1
	    }
	}
    }
    return $w.menu
}

proc dialog_pageitem_to_frame {dial prefix name} {
    global $dial
    switch -- [set ${dial}(pageStyle)] {
	"normal" {
	    # Have to get rid of extra '.' or ' ' which happen to be in the name
	    regsub -all "\[\. \]" $name "X" fr
	    set pane $prefix$fr
	    return $pane
	}
	"tabbed" {
	    [set ${dial}(pageController)] childsite $name
	}
    }
}

proc dialog_pane_change {dial w dmy elt op} {
    global $dial
    if {$op == "cursor"} {
	# we used a cursor key
	set idx [lsearch -exact [set ${dial}(allitems)] [set ${dial}($elt)]]
	set len [llength [set ${dial}(allitems)]]
	while {1} {
	    # we don't wrap around (else could use: ($idx + $dmy +$len)%$len )
	    incr idx $dmy
	    if {$idx < 0 || $idx >= $len} {
		return
	    }
	    # Don't stop on separators
	    if {![regexp {^(\(|-$)} [lindex [set ${dial}(allitems)] $idx]]} {
		break
	    }
	}
	set ${dial}($elt) [lindex [set ${dial}(allitems)] $idx]
	return
    } elseif {$op == "cycle"} {
	# Used to work around bug in Windows dnd.
	set orig [set ${dial}($elt)]
	set len [llength [set ${dial}(allitems)]]
	for {set idx 0} {$idx < $len} {incr idx} {
	    if {![regexp {^(\(|-$)} [lindex [set ${dial}(allitems)] $idx]]} {
		set ${dial}($elt) [lindex [set ${dial}(allitems)] $idx]
		update idletasks
	    }
	}
	set ${dial}($elt) $orig
	return
    }
    switch -- [set ${dial}(pageStyle)] {
	"normal" {
	    dialog_normal_pane_change $dial $w $elt
	}
	"tabbed" {
	    dialog_tabbed_pane_change $dial $w $elt
	}
    }
}

proc dialog_normal_pane_change {dial w elt} {
    global $dial
    foreach pane [info commands $w.subf*] {
	place forget $pane
    }
    set pane [dialog_pageitem_to_frame $dial $w.subf [set ${dial}($elt)]]
    if {[winfo exists $pane]} {
	place $pane -in $w -x 0 -y 0
    }
}

proc dialog_tabbed_pane_change {dial w elt} {
    global dial
    [set ${dial}(pageController)] select $elt
}

# get_directory [-p <prompt>]
#
#  This command will display a standard Macintosh file dialog and
#  request the user select a folder.  The command will return the
#  selected folder's full path name, or an error if the Cancel button
#  was selected.
proc get_directory {{start ""} {title "Pick a directory"} args} {
    if {$start == "-p"} {
	set start [lindex $args 0]
    }
    if {[llength [info commands tk_chooseDirectory]]} {
	if {[file exists $start]} {
	    set f [tk_chooseDirectory -title $title \
	      -parent [dialog::findParent] -initialdir $start]
	} else {
	    set f [tk_chooseDirectory -title $title \
	      -parent [dialog::findParent]]
	}
	if {$f != ""} {
	    global tcl_platform
	    if {$tcl_platform(platform) == "windows"} {
		# Catch this in case we're in a vfs.
		catch {set f [file attributes $f -longname]}
		return $f
	    } else {
		return $f
	    }
	} else {
	    error "Cancelled"
	}
    } else {
	echo "You should install 'Tk 8.3 or newer' to be able\
	  to choose a directory directly."
	if {$title == "Pick a directory"} {
	    set title "Pick a file in the directory you want"
	}
	set f [tk_getOpenFile -title $title -initialdir $start]
	if {$f != ""} {
	    return [file dirname $f]
	} else {
	    error "Cancelled"
	}
    }
}

# getPathName - prompt the user with an SFGetFile dialog and return 
#  complete pathname.
proc getPathName {} {
    return [tk_getOpenFile -parent [dialog::findParent]]
}
# putfile <prompt> <original>
#
#  This command will display an SFPutFile() and return the full path
#  name of the selected file, or an empty string if CANCEL button was
#  selected.  Original is the default name displayed for the user.
proc putfile {{title "Enter save file"} {where ""}} {
    return [tk_getSaveFile -title $title -initialfile $where \
      -parent [dialog::findParent]]
}

proc gotoLine {} {
    if {![llength [winNames -f]]} {return}
    set y 80
    set res [eval dialog -w 250 -h 110 -t [list "Goto line:"] 10 10 245 30 \
      -e [list ""] 20 50 190 70 [dialog::okcancel 10 y]]
    if {[lindex $res 1]} {
	goto "[lindex $res 0].0"
	bringToFront [win::Current]
    }
}

# prompt <prompt> <default> [<name> <menu item>*] - prompt dialog to 
#  the user with a prompt string and a default value. The prompt dialog can 
#  optionally include a popup menu specified by 'name' and the succeeding 
#  strings. Selection of the popup menu items inserts the item text into the 
#  editable dialog item. 'Prompt' returns the value of the editable item. 
#  If the 'Cancel' button is selected, the tcl returns an error and your 
#  script will be stopped unless you execute the command from 'catch'.
proc prompt {prompt default args} {
    if {![llength $args]} {
	set y 12
	eval lappend dialog [dialog::text $prompt 10 y 30] \
	  [dialog::edit $default 20 y 300] [dialog::okcancel 10 y]
	set res [eval dialog -w 250 -h 110 $dialog]
	if {[lindex $res 1]} {
	    return [lindex $res 0]
	} else {
	    error "Cancelled"
	}
    } else {
	set name [lindex $args 0]
	set args [lrange $args 1 end]
	set y 12
	eval lappend dialog [dialog::text $prompt 10 y 30] \
	  [dialog::edit $default 20 y 300] [dialog::text $name 10 y] \
	  [dialog::menu 50 y $args [lindex $args 0]] -copyto 0 \
	  [dialog::okcancel 10 y]
	set res [eval dialog -w 250 -h $y $dialog]
	if {[lindex $res 2]} {
	    return [lindex $res 0]
	} else {
	    error "Cancelled"
	}
    }
}

# setFontsTabs - bring up font and tab dialog
proc setFontsTabs {} {
    catch {text_cmd setFontsTabs}
}

proc chooseFontTab {font size tabsize} {
    set y 12
    eval lappend dialog \
      [dialog::text "Font:" 10 y] \
      [dialog::menu 50 y [flag::options defaultFont] $font] \
      [dialog::text "Size:" 10 y] \
      [dialog::menu 50 y [flag::options fontSize] $size] \
      [dialog::text "Tabsize:" 10 y] \
      [dialog::edit $tabsize 50 y 4] \
      [dialog::okcancel 10 y]
    set res [eval dialog -w 250 -h $y $dialog]
    if {[lindex $res 3]} {
	return [lrange $res 0 2]
    } else {
	error "Cancelled"
    }
    
}


# getline <prompt> <default>
#  This command will display a Macintosh alert box with prompt displayed, a 
#  text edit field with default initially in the field, and with the push 
#  buttons OK, Cancel.. The command will return the text entered into the 
#  text edit field by the user, or an empty string if the user selected the 
#  Cancel button. 
proc getline {{prompt Prompt} {default {}}} {
    set y 10
    set d [dialog::text $prompt 10 y 300]
    eval lappend d [dialog::edit $default 20 y 30 3]
    incr y 10
    eval lappend d [dialog::okcancel 10 y]
    set res [eval dialog -w 340 -h $y $d]
    if {[lindex $res 1]} {
	return [lindex $res 0]
    } else {
	return ""
    }
}

proc dialog::navigateList {w itemName args} {
    set selected [$w curselection]
    switch -- $itemName {
	up {
	    # Select the previous item, clearing any prior selections.
	    # The 'l' argument indicates multiple list items are ok.
	    set l [lindex $args 0]
	    if {![llength $selected]} {
		$w selection set end
	    } else {
		set last [lindex $selected 0]
		if {!$l} {$w selection clear 0 [$w size]}
		if {$last > 0} {incr last -1}
		$w selection set $last
	    }
	    $w see [lindex [$w curselection] 0]
	}
	down {
	    # Select the next item, clearing any prior selections.
	    # The 'l' argument indicates multiple list items are ok.
	    set l [lindex $args 0]
	    if {![llength $selected]} {
		$w selection set 0
	    } else {
		set last [lindex $selected end]
		if {!$l} {$w selection clear 0 [$w size]}
		if {$last < [expr {[$w size] -1}]} {incr last}
		$w selection set $last
	    }
	    $w see [lindex [$w curselection] 0]
	}
	pageDown {
	    # If current selection is already at the bottom of the page
	    # view, scroll down one page.  Select the bottom item.
	    set bottomitem [$w nearest [winfo height $w]]
	    $w selection clear 0 [$w size]
	    if {![lcontains selected $bottomitem]} {
		# bottom item not selected
		$w selection set $bottomitem
	    } else {
		$w yview scroll 1 pages
		$w selection set [$w nearest [winfo height $w]]
	    }
	}
	pageUp {
	    # If current selection is already at the top of the page view,
	    # scroll down one page.  Select the top item.
	    set topitem [$w nearest 0]
	    $w selection clear 0 [$w size]
	    if {![lcontains selected $topitem]} {
		# top item not selected
		$w selection set $topitem
	    } else {
		$w yview scroll -1 pages
		$w selection set [$w nearest 0]
	    }
	}
	home {
	    # Select the first item.
	    $w selection clear 0 [$w size]
	    $w selection set 0
	    $w see 0    
	}
	end {
	    # Select the last item.
	    $w selection clear 0 [$w size]
	    $w selection set end
	    $w see end
	}
	scrollDown {
	    # Scroll the list down one, leaving selection unchanged.
	    $w yview scroll  1 units
	}
	scrollUp {
	    # Scroll the list up one, leaving selection unchanged.
	    $w yview scroll -1 units
	}
	mouseWheel {
	    # Scroll the list, leaving selection unchanged.
	    set dir [lindex $args 0]
	    set scrollAmount 5
	    $w yview scroll [expr {- ($dir / 120) * $scrollAmount}] units
	}
    }
}

proc dialog::image {name x yy} {
    upvar $yy y
    if {$x == ""} {
	unset x
	upvar x x
    }
    set res [list -i $name $x $y]
    incr x [::image width $name]
    incr y [::image height $name]
    lappend res $x $y
    return $res
}

# listpick [-p <prompt>] [-l] [-L <def list>] <list>
#  This command will display a dialog with the list displayed in a List Manager 
#  list. If the user presses the Cancel button, an empty string is returned. If 
#  the user selects the Open button, or double clicks an item in the list, that 
#  item will be returned. If '-l' is specified, than the return is a list of 
#  items.
proc listpick {args} {
    global tcl_platform alphaPriv
    set root [dialog::findRoot]
    set w ${root}.dl
    catch {destroy $w}
    # Remember old focus
    set oldFocus [focus]
    
    toplevel $w -class Dialog
    wm title $w ""
    wm iconname $w Dialog
    wm protocol $w WM_DELETE_WINDOW { }

    # Dialog boxes should be transient with respect to their parent,
    # so that they will always stay on top of their parent window.  However,
    # some window managers will create the window as withdrawn if the parent
    # window is withdrawn or iconified.  Combined with the grab we put on the
    # window, this can hang the entire application.  Therefore we only make
    # the dialog transient if the parent is viewable.
    if {[winfo viewable [winfo toplevel [winfo parent $w]]]} {
	wm transient $w [winfo toplevel [winfo parent $w]]
    }    
    if {$tcl_platform(platform) == "macintosh"} {
	::tk::unsupported::MacWindowStyle style $w movableDBoxProc
    }

    getOpts {-p -L -w -h -T}
    if {![info exists opts(-p)]} {
	set opts(-p) "Please pick one:"
    }
    
    if {[info exists opts(-T)]} {
	set title $opts(-T)
	if {$tcl_platform(platform) == "windows"} {
	    # The standard Windows titlebar font doesn't seem to have
	    # an ellipsis... except for Windows 2000 (= NT 5.0)
	    # I need to know whether this is required on Win95/98...
	    if {($tcl_platform(os) != "Windows NT") || ($tcl_platform(osVersion) != 5.0)} {
		regsub -all "" $title "..." title
	    }
	}
	wm title $w $title
    }
    
    label $w.msg -wraplength 4i -justify left  -text $opts(-p)
    pack $w.msg -side top
    
    frame $w.buttons
    pack $w.buttons -side bottom -fill x -pady 2m
    button $w.buttons.ok -text Ok -command "set alphaPriv(button) 1" \
      -default active
    bind $w <Return> "
    $w.buttons.ok configure -state active -relief sunken
    update idletasks
    after 100
    set alphaPriv(button) 1
    "
    
    button $w.buttons.cancel -text Cancel -command "set alphaPriv(button) 0"
    bind $w <Escape> "
    $w.buttons.cancel configure -state active -relief sunken
    update idletasks
    after 100
    set alphaPriv(button) 0
    "
    wm protocol $w WM_DELETE_WINDOW [list event generate $w <Escape>]
    pack $w.buttons.ok $w.buttons.cancel -side left -expand 1
    
    frame $w.frame -borderwidth .5c
    pack $w.frame -side top -expand yes -fill both
    
    scrollbar $w.frame.scroll -command "$w.frame.list yview"
    if {[info exists opts(-l)]} {
	set selectmode "extended"
    } else {
	set selectmode "browse"
    }
    listbox $w.frame.list -yscroll "$w.frame.scroll set" \
      -setgrid 1 -height 12 -selectmode $selectmode -width 30
    
    pack $w.frame.scroll -side right -fill y
    pack $w.frame.list -side left -expand 1 -fill both

    bind $w.frame.list <Double-Button-1> {set alphaPriv(button) 1}
    # This enables navigating the list by typing in a key, clearing any
    # previous selection.
    bind $w <KeyPress>       "dialog::a_key        $w.frame.list %A"
    # These change the selected item, clearing any previous selections.
    bind $w <Down>           "dialog::navigateList $w.frame.list down 0"
    bind $w <Up>             "dialog::navigateList $w.frame.list up   0"
    # These enable using the shift key to extend the list of
    # multiple selections one item up/down.
    if {[info exists opts(-l)]} {
	bind $w <Shift-Down> "dialog::navigateList $w.frame.list down 1"
	bind $w <Shift-Up>   "dialog::navigateList $w.frame.list up   1"
    }
    # These enable using page up/down and home/end to navigate the
    # list, changing the selection as well.
    bind $w <Next>           "dialog::navigateList $w.frame.list pageDown"
    bind $w <Prior>          "dialog::navigateList $w.frame.list pageUp"
    bind $w <Home>           "dialog::navigateList $w.frame.list home"
    bind $w <End>            "dialog::navigateList $w.frame.list end"
    # These enable scrolling up/down in the list without actually changing
    # the selection.
    bind $w <Control-Down>   "dialog::navigateList $w.frame.list scrollDown"
    bind $w <Control-Up>     "dialog::navigateList $w.frame.list scrollUp"
    bind $w <MouseWheel>     "dialog::navigateList $w.frame.list mouseWheel %D"

    # args is a list of a list
    eval $w.frame.list insert 0 [lindex $args 0]
    # Determine (a) which item(s) will be initially selected, and
    # (b) which item should appear at the top of the view page.
    set idx [list ]
    if {[info exists opts(-L)]} {
	if {[info exists opts(-l)]} {
	    # Multiple selections are possible.
	    foreach itm $opts(-L) {
		# Select all default items found ...
		set _idx [lsearch -exact [lindex $args 0] $itm]
		if {$_idx >= 0} {lappend idx $_idx}
	    }
	} else {
	    # Only one default is possible.
	    set _idx [lsearch -exact [lindex $args 0] $opts(-L)]
	    if {$_idx >= 0} {lappend idx $_idx}
	}
    }
    if {![llength $idx]} {set idx [list "0"]}
    
    bind $w <Destroy> {set alphaPriv(button) -1}

    # 6. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    # If the window has an icon, now is the time to set it
    alpha::setIcon $w
    set x [expr {[winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
      - [winfo vrootx [winfo parent $w]]}]
    set y [expr {[winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
      - [winfo vrooty [winfo parent $w]]}]
    wm geom $w +$x+$y
    
    # Show the first selected item
    $w.frame.list see [lindex $idx 0]

    update
    wm deiconify $w

    # 7. Set a grab and claim the focus too.

    set oldGrab [grab current $w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $w
    focus $w.buttons.ok
    
    # Set the selection (if we do this earlier it can be reset by
    # the grab, I think).
    foreach _idx $idx {$w.frame.list selection set $_idx}

    # 8. Wait for the user to respond, then restore the focus and
    # return the index of the selected button.  Restore the focus
    # before deleting the window, since otherwise the window manager
    # may take the focus away so we can't redirect it.  Finally,
    # restore any grab that was in effect.

    tkwait variable alphaPriv(button)
    
    if {[info exists opts(-l)]} {
	set res [list]
	foreach itm [$w.frame.list curselection] {
	    lappend res [$w.frame.list get $itm]
	}
    } else {
	if {[$w.frame.list curselection] != ""} {
	    set res [$w.frame.list get [$w.frame.list curselection]]
	} else {
	    set res ""
	}
    }
    
    dialog::restoreFocus $oldFocus
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# alphaPriv(button) doesn't get reset by it.

	bind $w <Destroy> {}
	destroy $w
    }
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    if {$alphaPriv(button) == 1} {
	return $res
    } else {
	error "Cancelled!"
    }
}
    
# message <string> - prints 'string' on the status line.
proc message {t args} {
    set color "black"
    switch -- [llength $args] {
	0 {
	    # do nothing
	}
	1 {
	    set flag $t
	    set t [lindex $args 0]
	    if {$flag == "-error"} {
		set color "red"
	    }
	}
	default {
	    error "too many arguments to message"
	}
    }
    regsub -all "\[\r\n\]" $t " " t
    if {![winfo exists .status.text]} {
	label .status.text
	pack .status.text -side left
    }
    .status.text configure -text [::msgcat::mc $t] -foreground $color
    update idletasks
}

