## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "alpha_io.tcl"
 #                                    created: 03/01/2000 {15:19:43 PM} 
 #                                last update: 12/07/2001 {20:22:46 PM} 
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html
 #  
 # Copyright (c) 2000-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # Right now only a handful of procedures actually read in the contents
 # of a file, or write out the contents of a text widget to file.
 # 
 # These are: edit, revert, save and saveAs.  We route all of the i/o
 # of these commands through putsTextToFile and readTextFromFile, which
 # in turn use alphaOpen.  This ensures we have common control over
 # encodings, error handling etc.
 # 
 # It is extremely important that the variable 'alpha::encoding' is
 # not tampered with!
 # 
 # ###################################################################
 ##

#

## 
 # -------------------------------------------------------------------------
 # 
 # "alphaOpen" --
 # 
 #  Open a file, for whatever purpose, and set the encoding correctly.
 #  
 #  We interact with encodings as follows:
 #  
 #  If 'alpha::encoding' is set to "", then this proc will set that
 #  variable to the perceived encoding for the file.
 #  
 #  If 'alpha::encoding' is set to some other value, the the encoding
 #  for the file is assumed to be that value, and alpha::encoding is
 #  then unset.
 #  
 #  If 'alpha::encoding' is unset, then this proc behaves identically
 #  to 'open', with the exception of the 'alpha::encodingFor' test, which
 #  might over-ride things depending on what defaults have been specified
 #  for certain directories.
 # -------------------------------------------------------------------------
 ##
proc alphaOpen {filename args} {
    global alpha::encoding
    set fid [eval [list open $filename] $args]

    if {[info exists alpha::encoding] \
      && ([string length $alpha::encoding] > 0)} {
	fconfigure $fid -encoding $alpha::encoding
    } else {
	set enc [alpha::encodingFor $filename]
	if {[string length $enc]} {
	    catch {fconfigure $fid -encoding $enc}
	}
    }
    if {[info exists alpha::encoding]} {
	set alpha::encoding [fconfigure $fid -encoding]
    }
    # Return without overhead of 'return'
    set fid
}

## 
 # -------------------------------------------------------------------------
 # 
 # "putsTextToFile" --
 # 
 #  Returns 1 for success, or 0 for failure.
 # -------------------------------------------------------------------------
 ##
proc putsTextToFile {filename text {enc ""}} {
    set overwriting [file exists $filename]
    
    if {$overwriting} {
	if {[file isdirectory $filename]} {
	    alertnote "A directory with the same name already exists!  Save aborted."
	    return 0
	}
	if {[file exists [file join [file dirname $filename] __tmp_Alpha]]} {
	    alertnote "Alphatk's temporary file for safe saving already\
	      exists in this directory.  If this problem persists, please\
	      report a bug.  I will remove the file now."
	    catch {file delete [file join [file dirname $filename] __tmp_Alpha]}
	}
	if {[catch {file rename -force $filename \
	  [file join [file dirname $filename] __tmp_Alpha]}]} {
	    alertnote "Couldn't remove old file.  Save aborted."
	    return 0
	}
    }
    if {$enc != ""} {
	global alpha::encoding
	set alpha::encoding $enc
    }
    if {[catch {alphaOpen $filename w} fout]} {
	if {[info exists alpha::encoding]} { unset alpha::encoding }
	alertnote "Sorry, couldn't open the file for writing! Save aborted."
	if {$overwriting} {
	    catch {file rename -force [file join \
	      [file dirname $filename] __tmp_Alpha] $filename}
	}
	return 0
    }
    if {[info exists alpha::encoding]} { unset alpha::encoding }
    
    if {[fconfigure $fout -encoding] == "unicode"} {
	puts -nonewline $fout \uFEFF
    }
    
    if {[catch {puts -nonewline $fout $text} res]} {
	catch {close $fout}
	catch {file delete -force $filename}
	if {$overwriting} {
	    catch {file rename -force \
	      [file join [file dirname $filename] __tmp_Alpha] $filename}
	}
	alertnote "Couldn't save; had filesystem error: $res"
    } else {
	close $fout
	catch {file delete -force [file join [file dirname $filename] __tmp_Alpha]}
	message "Wrote [file tail $filename], [string length $text] characters."
    }
    return 1
}

proc wordswap {data} {
    binary scan $data s* elements
    return [binary format S* $elements]
}

proc readTextFromFile {filename {enc ""}} {
    global tcl_platform
    if {$enc != ""} {
	global alpha::encoding
	set alpha::encoding $enc
    }
    set fin [alphaOpen $filename r]
    # the alphaOpen may have adjusted the encoding
    set enc [fconfigure $fin -encoding]

    if {$enc == "unicode"} {
	fconfigure $fin -encoding binary
    }
    if {[catch {read $fin} text]} {
	catch {close $fin}
	error $text
    }
    close $fin

    if {$enc == "unicode"} {
	if {[binary scan $text S bom] == 1} {
	    if {$bom == -257} {
		if {$tcl_platform(byteOrder) == "littleEndian"} {
		    set text [wordswap [string range $text 2 end]]
		} else {
		    set text [string range $text 2 end]
		}
	    } elseif {$bom == -2} {
		if {$tcl_platform(byteOrder) == "littleEndian"} {
		    set text [string range $text 2 end]
		} else {
		    set text [wordswap [string range $text 2 end]]
		}
	    } elseif {$tcl_platform(byteOrder) == "littleEndian"} {
		set text [wordswap $text]
	    }
	}
	set text [encoding convertfrom unicode $text]
    }
    
    # Do some nasty stuff on windows.
    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	if {![regexp "\[^\n\]\n\[^\n\]" $text] && [regexp "\n\n" $text]} {
	    regsub -all "\n\n" $text "\n" text
	}
    }
    # Return without overhead of 'return'
    set text
}

# edit [-r] [-m] [-c] [-w] [-g <l> <t> <w> <h>] <name> - Open a file in new 
#  window. '-c' means don't prompt for duplicate win if file already open.
#  '-r' means open the file read-only. '-m' means omit the function titlebar 
#  menu and present only the marks titlebar menu, which is labeled with the 
#  contents of 'markLabel'. The '-g' option allows left and top coords to 
#  be specified, plus width, and height. All or none. '-w' allows you to
#  bypass the "Wrap" dialog for files with long rows.
proc edit {args} {
    global win::tk
    
    global defWidth defHeight defTop defLeft
    set opts(-g) [list $defLeft $defTop $defWidth $defHeight]

    getOpts {{-g 4} -tabsize}
    set fn [lindex $args end]
    if {![file exists $fn]} {
	# Check if this is an encoding issue, in which we were
	# called by the system with a natively encoding string,
	# which therefore hasn't been properly converted to utf.
	set ftry [encoding convertfrom [encoding system] $fn]
	if {[file exists $ftry]} {
	    set fn $ftry
	}
    }
    set n [file::ensureStandardPath $fn]
    if {$n == ""} {return}
    set name $n
    set exists 0
    if {[info exists win::tk($n)]} {
	set exists 1
    } else {
	foreach existname [array names win::tk] {
	    if {[win::StripCount $existname] == $n} {
		set n $existname
		set exists 1
		break
	    }
	}
    }
    if {$exists} {
	if {[info exists opts(-c)] || ([dialog::yesno -n "Open a duplicate" \
	  -y "Go to existing window" "That window is already open!"])} {
	    bringToFront $n
	    return
	}
    }
    append name [win::CountFor [file tail $n]]

    if {[info exists opts(-tabsize)]} {
	win::setInitialConfig $name tabsize $opts(-tabsize)
    }
    filePreOpeningHook $n $name
    # doesn't check window exists, or file tail name clash
    global alpha::encoding win::config
    set alpha::encoding ""
    if {[info exists win::config($name)]} {
	foreach item $win::config($name) {
	    if {[lindex $item 0] == "encoding"} {
		set alpha::encoding [lindex $item 1]
	    }
	}
    }
    # Set the window title
    global showFullPathsInWindowTitles
    if {$showFullPathsInWindowTitles} {
	set title $n
    } else {
	if {[win::IsFile $n]} {
	    set title [file tail $n]
	} else {
	    set title $n
	}
    }

    alpha::embedInto \
      -text [readTextFromFile $n] -encoding $alpha::encoding \
      $name [concat [list toplevel $title] $opts(-g)]
    
    unset alpha::encoding

    if {[info exists opts(-r)] || ![file writable $n]} {
	winReadOnly
    }
    update
}

# revert - revert the file to its last saved version
proc revert {args} {
    if {[llength $args] && [lindex $args 0] == "-w"} {
	set n [lindex $args 1]
    } else {
	set n [win::Current]
    }

    global win::tk
    set filename [win::StripCount $n]
    if {[file exists $filename]} {
	set w $win::tk($n)
	if {[catch {readTextFromFile $filename [tw::encoding $w]} text]} {
	    if {[info exists alpha::encoding]} { unset alpha::encoding }
	    alertnote "Couldn't read the saved file's contents!  Error: $text"
	    return
	}
	if {[info exists alpha::encoding]} { unset alpha::encoding }
	getWinInfo -w $n wi
	set topl $wi(currline)
	setWinInfo -w $n read-only 0
	deleteText -w $n [minPos] [maxPos]
	insertText -w $n $text
	setWinInfo -w $n dirty 0
	revertHook $n
	display -w $n [rowColToPos -w $n $topl 0]
	update idletasks
	::tw::arrangeToColour $w
	if {![file writable $filename]} {
	    winReadOnly $n
	}
	message "File window '$n' synchronised with version currently on disk"
    } else {
	error "No such file!"
    }
}


# save - save current window (or given window)
proc save {{w ""}} {
    global win::Active win::tk
    if {$w == ""} {
	set w [lindex $win::Active 0]
    }
    getWinInfo -w $w info
    if {$info(read-only)} {
	return
    }
    set wn [win::StripCount $w]
    if {![file exists $wn]} {
	saveAs [file join [pwd] $wn]
	return
    }
    saveHook $w
    if {![putsTextToFile $wn [text_wcmd $w get 1.0 "end -1c"] \
      [tw::encoding $win::tk($w)]]} {
	return
    }
    ::tw::save $win::tk($w)
    savePostHook $w
}

# saveAs [def name] - save current window with new name. Optionally takes 
#  a default filename. Returns complete path of saved file, if ok hit, 
#  otherwise TCL_ERROR returned.
proc saveAs {{default ""} args} {
    global win::Active win::tk win::tktitle showFullPathsInWindowTitles \
      tcl_platform
    set w [lindex $win::Active 0]
    # get new stuff
    if {[llength $args]} {
	if {$default == "-f" && ([llength $args] == 1)} {
	    set name [lindex $args 0]
	} else {
	    error "bad args to saveAs"
	}
    } else {
	if {$default == ""} { set default [win::StripCount $w] }
	if {![file exists $default]} {
	    set default [file join [file dirname $default] \
	      [file::makeNameLegal [file tail $default]]]

	    global win::Modified
	    set next [lindex $win::Active 1]
	    if {[info exists win::Modified($next)]} {
		set default [file join [file dirname $next] [file tail $default]]
	    }
	}
	set name [tk_getSaveFile -initialfile $default -filetypes [findFileTypes]]
    }
    if {$name == ""} {
	error "Cancelled"
    }
    set tail [file tail $name]
    if {[file::makeNameLegal $tail] != $tail} {
	return -code error "Illegal file name '$tail' to saveAs"
    }
    # The win::tk array entry includes any <2> duplicates etc.
    
    # Now we need to sort out what encoding to save the file in.  The
    # main issue is there might be different default encodings for the
    # current location and the new location.
    set currEnc [tw::encoding $win::tk($w)]
    set useEnc $currEnc
    set saveAsEnc [alpha::encodingFor $name]
    if {($saveAsEnc != "") && ($saveAsEnc != $currEnc)} {
	if {$currEnc == ""} {
	    set useEnc $saveAsEnc
	} else {
	    # ask user
	    if {[dialog::yesno -n "Use old $currEnc" \
	      "The default encoding for the location\
	      to which you are saving that file is '$saveAsEnc'. Would\
	      you like to save this file in that encoding?"]} {
		set useEnc $saveAsEnc
	    }
	}
    }
    # Finally save the file.
    if {![putsTextToFile $name [text_cmd get 1.0 "end -1c"] $useEnc]} {
	return
    }
    
    set tkw $win::tk($w)
    unset win::tk($w)
    # Avoid confusion on windows.
    if {$tcl_platform(platform) == "windows"} {
	# Catch this in case we're in a vfs.
	catch {set name [file attributes $name -longname]}
    }
    set name [file nativename $name]
    set filename $name
    append name [win::CountFor [file tail $name] 1]

    # get old mode
    global win::Modes
    set oldmode $win::Modes($w)
    
    set win::tk($name) $tkw
    set win::tktitle($tkw) $name
    if {[$tkw statusConfigure image] == "lock"} {
	tw::read_only $tkw 0
    }
    if {$showFullPathsInWindowTitles} {
	wm title [winfo toplevel $win::tk($name)] $name
    } else {
	wm title [winfo toplevel $win::tk($name)] [file tail $name]
    }
    wm protocol [winfo toplevel $tk($name)] WM_DELETE_WINDOW \
      [list killWindow $name]

    if {($currEnc == "") || ($useEnc != $currEnc)} {
	# Once a file has been saved, it has an encoding associated
	# with it.  This is either the encoding we forced on the file,
	# or the system encoding if none was set.
	if {$useEnc != ""} {
	    $tkw encoding $useEnc
	} else {
	    $tkw encoding [encoding system]
	}
    }
    
    saveasHook $w $name
    
    # adjust dirty, undo, redo data.
    ::tw::save $tkw
    
    savePostHook $name

    win::modeHasChanged $name $oldmode $win::Modes($name)
}
