## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "alpha_macros.tcl"
 #                                    created: 04/12/98 {23:17:46 PM} 
 #                                last update: 10/22/2001 {17:44:51 PM} 
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html
 #  
 # Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##

#  macros  #

namespace eval macro {}

# dumpMacro - prompts for a name and then dumps a tcl proc representation 
#  of the current keyboard macro into the current window.
proc dumpMacro {args} {echo "dumpMacro $args"}
# startKeyboardMacro - start recording keyboard macro
proc macro::startRecording {} {
    global macro::recording alpha::currentMacroContents
    if {$::macro::recording} {
	# We were already recording.
	set macro::recording 0
        status::errorMsg "Cancelled -- recording was already in progress."
    }
    set macro::recording 1
    set alpha::currentMacroContents [list]
    message "Defining Keyboard Macro"
    setupMacroRecording
}
# endKeyboardMacro - stop recording keyboard macro
proc macro::endRecording {} {
    
    global macro::recording alpha::currentMacroContents
    
    catch {endMacroRecording}
    if {!$::macro::recording} {
	set macro::recording 0
	status::errorMsg "Not recording."
    } 
    if {$::macro::recording != "1"} {
	# Recording was suspended, but never resumed !!
	# Must have been an error somewhere.
	set macro::recording 0
	macro::current ""
	status::errorMsg "Cancelled -- the recording of the macro must have failed."
    } 
    set lastCommand   ""
    set macroCommands [list ]
    foreach command $alpha::currentMacroContents {
	# Clean up 'insertText' commands by combining them into a string.
	set pat {^ *insertText +}
	if {[regexp $pat $command] && [regexp $pat $lastCommand]} {
	    set newInsert [quote::Regfind [lindex $command 1]]
	    set oldInsert [lindex $lastCommand 1]
	    if {$newInsert == {\\\\}} {
		set command "    insertText \{\\\\\}"
	    } else {
		set macroCommands \
		  [lreplace $macroCommands end end]
		set command "    insertText \"${oldInsert}${newInsert}\""
	    }
	}
	lappend macroCommands \n${command}
	set lastCommand $command
    } 
    macro::current "proc macroName \{\} \{[join $macroCommands]\n\}"
    # This has to come last to properly dim the macro menus.
    set macro::recording 0
    status::msg "Keyboard Macro Complete"
}

proc macro::recording {} {return $::macro::recording}

set macro::recording 0
  
proc ::alpha::macroRecord {{command ""}} {

    if {!$::macro::recording || [info level] > "2"} {
	return
    } elseif {$::macro::recording == "1"} {
	if {![string length $command]} {set command [info level 1]}
	if {[regexp {(^:*__record__)|endRecording} $command]} {return}
	lappend ::alpha::currentMacroContents "    $command"
    }
}

# This is how we avoid recording 'nested' commands.

proc ::alpha::suspendRecording {} {
    if {[regexp {^:*__record__} [info level 1]]} {return}
    incr ::macro::recording 1
}

proc ::alpha::resumeRecording {} {
    if {[regexp {^:*__record__} [info level 1]]} {return}
    incr ::macro::recording -1
}
  
set ::macro::_Current ""

# We need to add to this list!

proc listMacroProcs {} {
    
    # Make sure that the file 'indentation.tcl' has been loaded.
    auto_load bind::CarriageReturn
    
    # Note that many of these would be added if they're part of a menu.
    set macroRecordList {
	backSpace backwardChar backwardCharSelect backwardDeleteWord
	backwardWord backwardWordSelect balance beginningBufferSelect
	beginningLineSelect beginningOfBuffer beginningOfLine capitalizeRegion
	capitalizeWord centerRedraw clear closeAll copy cut deleteChar
	deleteSelection deleteWord deleteText downcaseRegion downcaseWord
	endBufferSelect endLineSelect endOfBuffer endOfLine enterReplaceString
	enterSearchString enterSelection evaluate exchangePointAndMark find
	findAgain findAgainBackward findFile findInNextFile findTag float
	forwardChar forwardCharSelect forwardWord forwardWordSelect
	gotoMark gotoNamedMark gotoTMark insertText insertToTop killLine
	killWindow nextLine nextLineSelect nextSentence
	nextWindow oneSpace openLine otherPane pageBack pageForward paste
	performSearch placeText previousLine prevLineSelect prevSentence
	prevWindow print redo replace replaceAll replace&FindAgain
	replaceString replaceText scrollDownLine scrollLeftCol scrollRightCol
	scrollUpLine search searchString select setMark setNamedMark
	shiftLeftRegion shiftRightRegion spacesToTabs splitWindow tabsToSpaces
	toggleScrollbar undo upcaseRegion upcaseWord wrap wrapText yank zoom
    }
    set namespaces [list bind menu]
    foreach ns $namespaces {
	foreach proc [namespace eval $ns [list info procs]] {
	    lappend macroRecordList ${ns}::${proc}
	}
    }
    return [list $namespaces $macroRecordList]
}

proc setupMacroRecording {} {

    set procsLists [listMacroProcs]
    namespace eval __record__ {}
    foreach ns [lindex $procsLists 0] {namespace eval __record__::$ns {}}
    foreach proc [lindex $procsLists 1] {
	if {![llength [info proc $proc]]} {continue}
	if {[catch {rename $proc __record__::$proc}]} {continue}
	proc $proc {args} {
	    alpha::suspendRecording
	    set proc __record__::[lindex [info level 0] 0]
	    if {[catch {uplevel 1 [list $proc] $args} result]} {
		set caught 1
	    } else {
	        set caught 0
	    }
	    alpha::resumeRecording
	    alpha::macroRecord
	    if {$caught} {
		error $result
	    } else {
		return $result
	    }
	}
    } 
    # Now a special case for inserting text.  If we simply redefine
    # tkTextInsert as above we will always insert into the original window !!
    if {[llength [info proc tkTextInsert]]} {
	if {![catch {rename tkTextInsert __record__::tkTextInsert}]} {
	    proc tkTextInsert {w s} {
		set proc __record__::tkTextInsert
		set result [__record__::tkTextInsert $w $s]
		if {[string length $s]} {
		    alpha::macroRecord "insertText \{[quote::Regsub $s]\}"
		} 
		return $result
	    }
	}
    } elseif {[llength [info proc tk::TextInsert]]} {
	namespace eval __record__::tk {}
	if {[catch {rename tk::TextInsert __record__::tk::TextInsert}]} {
	    proc tkTextInsert {w s} {
		set proc __record__::tk::TextInsert
		set result [__record__::tk::TextInsert $w $s]
		if {[string length $s]} {
		    alpha::macroRecord "insertText \{[quote::Regfind $s]\}"
		} 
		return $result
	    }
	}
    }
}

proc endMacroRecording {} {
 
    set recordingList [lindex [listMacroProcs] 1]
    foreach proc $recordingList {
	# We catch this in case additional procs have been added since the
	# recording actually started.
	if {[info proc $proc] == "" || [info proc __record__::$proc] == ""} {
	    continue	
	}
	rename $proc "" 
	rename __record__::$proc $proc
    }
    if {[llength [info proc __record__::tkTextInsert]]} {
	rename tkTextInsert ""
	rename __record__::tkTextInsert tkTextInsert
    } elseif {[llength [info proc __record__::tk::TextInsert]]} {
	rename tk::TextInsert ""
	rename __record__::tk::TextInsert tk::TextInsert
    }
    catch {namespace delete __record__ }
}
