## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "alpha_menus.tcl"
 #                                    created: 04/12/98 {23:17:46 PM} 
 #                                last update: 11/19/2001 {11:02:19 AM} 
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html
 #  
 # Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##
 
#  menus  #

# Check if we have compound menus.  Later we can replace this
# with a simple 'info tclversion'.
menu .test
.test add command -label "test"
if {[catch {.test entryconfigure test -compound left}]} {
    set alpha::noCompoundMenus 1
} else {
    set alpha::noCompoundMenus 0
}
destroy .test

# addMenuItem [-m] [-l <meta-characters>] <menu name> <item name> - Convert
#  item to menu form and add to specified menu. If '-m' specified, menu 
#  form conversion not done. The '-l' option allows you to use menu meta 
#  characters as text of menu items. If the '-l' option is used, the 
#  <meta-characters> string is interpreted for menu item attributes, and 
#  meta-characters in <item name> are included in the menu item text 
#  literally. For example: 
#  	addMenuItem -m -l "/B" "Hello/C" 
#  would appear as the text "Hello/C" in the menu, and have "B" as its 
#  command equivalent.
proc addMenuItem {name item args} {
    # doesn't currently take account of the proc attached to the given
    # menu.
    set accel ""
    if {[llength $args]} {
	if {$name == "-m"} { 
	    set mflag 1
	    if {$item == "-l"} {
		foreach {accel name item index} $args {}
	    } else {
		set name $item 
		set item [lindex $args 0]
		set index [lindex $args 1]
	    }
	} elseif {$name == "-l"} {
	    set accel $item
	    foreach {name item index} $args {}
	} else {
	    set index [lindex $args 0]
	}
	if {$index == ""} {
	    set index "add"
	} else {
	    set index [list insert $index]
	}
    }
    global alpha_mprocs
    if {[info exists alpha_mprocs($name)]} {
	set mproc $alpha_mprocs($name)
    } else {
	set mproc "menu::generalProc"
    }
    foreach {name n} [menu_tags $name] {}
    if {$item == "\(-"} {
	eval [list $n] $index separator
    } else {
	if {[info exists mflag]} { 
	    set label $item
	} else {
	    set label [quote::Menuify $item]
	}
	if {$accel != ""} {
	    foreach {accelerator accel} [alpha::accelerator [string range $accel 1 end] ""] {}
	    set accel [list -accelerator $accel]
	    if {[info exists accelerator] && ($accelerator != "")} {
		regsub -all "Ctrl" $accelerator "Control" accelerator
		bind all "<${accelerator}>" [list $mproc $name $item]
	    }
	}
	eval [list $n] $index \
	  command -label [list [::msgcat::mc $label]] \
	  -command [list [list $mproc $name $item]] \
	  $accel
    }
    
}

proc addHelpMenu {args} {
    global multiColumnMenusEveryNItems alpha::helpMenuCounter
    if {![info exists alpha::helpMenuCounter]} {
	set alpha::helpMenuCounter $multiColumnMenusEveryNItems
    }
    foreach item $args {
	eval .menubar.help add [alpha::menuItem .menubar.help $item "helpMenu"]
	incr alpha::helpMenuCounter -1
	if {$alpha::helpMenuCounter <= 0} {
	    if {![catch {.menubar.help entryconfigure end -columnbreak 1}]} {
		set alpha::helpMenuCounter $multiColumnMenusEveryNItems
	    }
	}
    }
}

proc alpha::menuItem {parent item {proc ""}} {
    if {$item != "(-"} {
	if {[string first "Menu " $item] == 0} {
	    # should be a submenu
	    set name [lindex $item 2]
	    set items [lindex $item 3]
	    regsub -all {\.} $name {} n
	    regsub -all " " $n "_" n
	    set subparent $parent.m$n
	    global tearoffMenus
	    ::menu $subparent -tearoff $tearoffMenus
	    foreach subitem $items {
		eval $subparent add \
		  [alpha::menuItem $subparent $subitem [list helpMenu $name]]
	    }
	    return [list cascade -label [::msgcat::mc $name] -menu $subparent]
	} else {
	    return [list command -label [::msgcat::mc $item] \
	      -command "$proc [list $item]"]
	}
    } else {
	return [list "separator"]
    }
}

proc alpha::accelerator {key mods} {
    variable command_key 
    variable option_key
    variable keymap
    variable menukeymap
    if {$key == ""} {return ""}
    regsub -all {<[SE]} $mods "" mods
    if {$mods != ""} {
	regsub "<I" $mods "${option_key}-" mods
	regsub "<U" $mods "Shift-" mods
	regsub "<B" $mods "Ctrl-" mods
	regsub "<O" $mods "${command_key}-" mods
    }
    if {[regexp {[a-z]} $key]} {
	global keys::func tcl_platform
	# Enter is invalid on non-MacOS 
	if {$key == "a" && ($tcl_platform(platform) != "macintosh")} {
	    return ""
	}
	set rest [lindex $keys::func [expr {[text::Ascii $key] -97}]]
	if {$rest == "Delete"} {
	    set rest BackSpace
	} elseif {$rest == "Fwd Del"} {
	    set rest Delete
	}
	set menu $mods$rest
	append mods KeyPress-$rest
    } else {
	if {![info exists alpha::menukeymap($key)]} {
	    if {$mods == "" || $mods == "Shift-"} { 
		append mods $command_key "-" 
	    }
	    set menu $mods$key
	    if {[info exists alpha::keymap($key)]} {
		set key $alpha::keymap($key)
	    }
	} else {
	    set menu $mods$key
	    set key $alpha::menukeymap($key)
	}
	append mods KeyPress- $key
    }
    return [list $mods $menu]
}

# deleteMenuItem [-m] <menu name> <item name> - Convert item to menu form 
#  and delete from specified menu. If '-m' specified, menu form conversion 
#  not done. 
proc deleteMenuItem {name item args} {
    if {[llength $args]} {
	if {$name == "-m"} { 
	    set mflag 1
	    set name $item 
	    set item [lindex $args 0]
	}
    }
    foreach {name n} [menu_tags $name] {}
    if {[info exists mflag]} { 
	set label $item
    } else {
	set label [quote::Menuify $item]
    }
    
    set i [_menuItemToIndex $n $label]
    $n delete $i
}

# Tk uses either a variety of index lookup techniques to find a
# menu item, or a pattern lookup.  There is no built in technique
# to find the index of an item with a particular textual label.
# This command effectively implements that.
proc _menuItemToIndex {menucmd label} {
    # Find a pattern-quoted name of the label
    set label [quote::Find [::msgcat::mc $label]]
    if {[string index $label 0] != "\\"} {
	# The label might be mistaken for an integer index, or 'end'
	# or something like that.  Since it is a 'pattern', we can
	# simply add a literal to the first character so it won't
	# be mis-interpreted.
	set label "\\$label"
    }
    return [$menucmd index $label]
}

proc renameMenuItem {name item newitem args} {
    if {[llength $args]} {
	if {$name == "-m"} { 
	    set mflag 1
	    set name $item 
	    set item $newitem
	    set newitem [lindex $args 0]
	}
    }
    foreach {name n} [menu_tags $name] {}
    if {[info exists mflag]} { 
	set label $item
    } else {
	set label [quote::Menuify $item]
	set newitem [quote::Menuify $newitem]
    }
    set i [_menuItemToIndex $n $label]
    $n entryconfigure $i -label [::msgcat::mc $newitem]
}

# enableMenuItem <menuName> <item text> <on|off> - Either enable or 
#  disable the menu item of user menu 'menuName' that has text '<item 
#  text>'. Note that unless the menu is not only created, but also already
#  inserted, this command has no effect. 
proc enableMenuItem {name item on args} {
    if {[llength $args]} {
	if {$name == "-m"} {
             set mflag 1
	     set name $item
             set item $on
	     set on [lindex $args 0]
        }
    }
    
    foreach {name n} [menu_tags $name] {}
    if {![winfo exists $n]} { error "No such menu $name" }
    if {$item == ""} {
	# it's the whole menu
	set index [_menuItemToIndex .menubar $name]
	set menu .menubar
    } else {
	set menu $n
	if {[info exists mflag]} {
	    set index [_menuItemToIndex $n $item]
	} else {
	    set index [_menuItemToIndex $n [quote::Menuify $item]]
	}
    }

    $menu entryconfigure $index \
      -state [expr {$on ? "normal" : "disabled"}]
    
}

# insertMenu <name> - insert the previously created user menu 'name' into 
#  the menuBar. 
proc insertMenu {n} {
    foreach {n w} [menu_tags $n] {}
    if {![winfo exists $w]} {
	menu $w
    }
    # There is no command to check for the existence of a menu item,
    # so we use 'catch' to check whether the item is already in the
    # menu.
    if {[catchNoClobber {_menuItemToIndex .menubar $n}]} {
	# insert it just before the Help menu.
	.menubar insert Help cascade -label "$n" -menu $w
    }
}
# markMenuItem [-m] <menuName> <item text> <on|off> [<mark char>] - 
#  Either mark or unmark
#  the menu item of user menu 'menuName' that has text '<item text>'. 
#  Note that unless the menu is not only created, but also already
#  been inserted, this command has no effect. 
proc markMenuItem {m item on {char ""} args} {
    if {$m == "-m"} {
	set m $item ; set item $on ; set on $char ; set char [lindex $args 0]
    } else {
	set item [quote::Menuify $item]
    }
    foreach {m widg} [menu_tags $m] {}
    set m $widg
    if {$on == "on"} { set on 1 } elseif {$on == "off"} { set on 0 }
    if {[catch {_menuItemToIndex $m $item} index]} {
	catch {puts stderr "No such item '$item' in menu $m, while [info level 1]"}
	return ""
    }
    set label [$m entrycget $index -label]
    global alphaDummy
    switch -- [$m type $index] {
	"radiobutton" {
	    if {$on} {
		set alphaDummy(menu,$m) $label
	    }
	}
	"checkbutton" {
	    set alphaDummy(menu,$m,$label) $on
	}
	default {
	    # this is so the user can create the item as an ordinary
	    # menu item, and then later turn it into a checkbutton
	    # without any problems.
	    $m insert $index checkbutton \
	      -command [$m entrycget $index -command] \
	      -label [::msgcat::mc $label] \
	      -accelerator [$m entrycget $index -accelerator] \
	      -columnbreak [$m entrycget $index -columnbreak] \
	      -state normal \
	      -variable alphaDummy(menu,$m,$label)
	    set alphaDummy(menu,$m,$label) $on
	    $m delete [expr {$index +1}]
	}
    }
    return ""
}

# Menu [-s] [-n <name>] [-i <num] [-m] [-M <mode>] [-p <procname>] <list of menu items> - 
#  Defines a new menu named 'name' (if provided w/ '-n' option). The menu is not 
#  yet inserted into the menubar. The menu commands may be nested for 
#  heirarchical menus, see 'AlphaBits.tcl' for examples. Alpha massages the 
#  function names to make them look better in the menus. 
#  '-c'			Ignore any menu meta-chars. Can also be done on a per-item basis 
#  				by appending an ampersand ('&') to the end of an item.
#  '-s'			Use system menu definition proc (faster).
#  '-n <num>'    Resource id of icon suite to use for menu title. 'ics#' 
#                is the only resource that is really necessary.
#  '-n <name>'	Name the menu. Not necessary for submenus.
#  '-m'			No menu form. If not supplied, each menu item is split into 
#  				words at each capitalized letter.
#  '-p <name>' 	The tcl proc named by 'name' is called w/ the menu's name
#  				and the item's contents when the item is chosen.
#  '-M <mode>'	Specifies that any bindings created by the menu are 
#  				specific to a given mode. This is important because mode-specific
#  				bindings over-ride global bindings.
proc Menu {args} {
    set ma [lindex $args end]
    set args [lreplace $args end end]
    getOpts {-n -M -p -t -h}
    if {[info exists opts(-p)]} {
	lappend proc $opts(-p)
	global alpha_mprocs
	set alpha_mprocs($opts(-n)) $proc
    } else {
	set proc ""
    }
    #if {[info exists opts(-M)]} { lappend proc -M $opts(-m) }
    #if {[info exists opts(-m)]} { lappend proc -m }
    set noNameError "Menu needs '-n name'"
    if {![info exists opts(-n)]} {
	return -code error $noNameError
    }
    foreach {opts(-n) widg} [menu_tags $opts(-n)] {}
    set m $widg
    if {[winfo exists $m]} {
	destroy $m
    }
    global tearoffMenus
    menu $m -tearoff $tearoffMenus
    #puts "$m $opts(-n) [menu_tags $opts(-n)]"
    global multiColumnMenusEveryNItems
    set count $multiColumnMenusEveryNItems
    foreach item $ma {
	incr count -1
	# special 'catch' so we don't trip on 'lindex $item 0'
	set submenu 0
	if {![catch {lindex $item 0} tmp] && $tmp == "Menu" && \
	  [llength $item] > 1} {
	    if {[catch {eval $item} res]} {
		if {$res != $noNameError} {
		    return -code error $res
		} else {
		    # treat it as a menu item, not a menu
		}
	    } else {
		set submenu 1
	    }
	}
	
	if {$submenu} {
	    foreach {mm label} $res {}
	    if {![info exists opts(-m)]} {
		set label [quote::Menuify $label]
	    }
	    $m add cascade -label [::msgcat::mc $label] -menu $mm
	} elseif {[info exists opts(-c)]} {
	    # ignore all meta characters
	    if {[string range $item 0 1] == "\(-"} {
		$m add separator
		continue
	    } elseif {[regexp "^ *-+ *\$" $item]} {
		$m add separator
		continue
	    } else {
		set state "normal"
		set entrytype command
		set isOn 0
		set label $item

		if {![info exists opts(-m)]} {
		    set label [quote::Menuify $label]
		}
		# '-c' doesn't strip ellipses.
		#regexp {^(.*)\u2026$} $item "" item
		if {[info exists opts(-t)]} {
		    set entrytype $opts(-t)
		}
		if {$proc == ""} {
		    $m add $entrytype -label [::msgcat::mc $label] \
		      -command "$item" -state $state
		} else {
		    $m add $entrytype -label [::msgcat::mc $label] \
		      -command "$proc [list $opts(-n) $item]" \
		      -state $state
		}
		if {$entrytype == "radiobutton"} {
		    global alphaDummy
		    if {$isOn} {
			set alphaDummy(menu,$m) $label
		    }
		    $m entryconfigure end -variable \
		      alphaDummy(menu,$m) -value $label 
		    # I used to think we needed '-command ""' here,
		    # but in fact that's not true.  While we don't
		    # need the command to have the variable set correctly,
		    # we do need it so Alpha's other procedures can
		    # register the change, and perhaps schedule this
		    # item to be saved as a modified preference.
		} elseif {$entrytype == "checkbutton"} {
		    global alphaDummy
		    set alphaDummy(menu,$m,$label) $isOn
		    $m entryconfigure end -variable \
		      alphaDummy(menu,$m,$label)
		}
	    }
	} else {
	    switch -regexp -- $item {
		"/." {
		    regexp {/(.)} $item "" key
		    regsub "/[quote::Regfind ${key}]" $item "" item 
		    if {$key == "\x1e"} {
			# special case 'icon'
			set key ""
			set icon 1
		    }
		    if {[regsub {^\(} $item "" item]} {
			set state "disabled"
		    } else {
			set state "normal"
		    }
		    regexp {^((<[UIOCSEB])*)} $item "" mods
		    set item [string range $item [string length $mods] end]					
		    if {[regexp {^(.*)(\\?&|\^.)$} $item "" item other]} {
			# If this isn't true it's the \\?& branch.
			if {!$::alpha::noCompoundMenus \
			  && ([string index $other 0] == "^")} {
			    if {[info exists icon]} {
				# add icon and set compound
				scan [string index $other 1] %c int
				# magic number according to which Alpha on MacOS behaves!
				incr int 208
				set image ics8$int
				set ok 1
				if {[catch {image width $image}]} {
				    set ok 0
				    global HOME
				    set ifile [file join $HOME AlphaGifs Alpha$int.gif]
				    if {[file exists $ifile]} {
					if {![catch {image create photo $image -file $ifile}]} {
					    set ok 1
					}
				    }
				}
				if {!$ok} {
				    unset image
				}
			    } else {
				# add bitmap and set compound
				set int [string index $other 1]
				incr int 256
				set image ICON$int
				set ok 1
				if {[catch {image width $image}]} {
				    set ok 0
				    global HOME
				    set ifile [file join $HOME AlphaGifs Alpha$int.gif]
				    if {[file exists $ifile]} {
					if {![catch {image create photo $image -file $ifile}]} {
					    set ok 1
					}
				    }
				}
				if {!$ok} {
				    unset image
				}
			    }
			}
		    }
		    if {[regexp {^!(.)} $item "" markc]} {
			switch -- $markc {
			    "\u2022" {
				error "Can't use command keys and indicators"
				set entrytype checkbutton
				set item [string range $item 2 end]
				set label $item
				set isOn 1
			    }
			    " " {
				error "Can't use command keys and indicators"
				set entrytype checkbutton
				set item [string range $item 2 end]
				set label $item
				set isOn 0
			    }
			    default {
				set item [string range $item 2 end]
				set label "$markc   $item"
			    }
			}
		    } else {
			if {![info exists opts(-m)]} {
			    set label [quote::Menuify $item]
			} else {
			    set label $item
			}
		    }
		    regexp {^(.*)\u2026$} $item "" item
		    
		    if {$proc == ""} {
			set cmd $item
		    } else {
			set cmd "$proc [list $opts(-n) $item]"
		    }
		    set accelerator ""
		    set accel ""
		    foreach {accelerator accel} [alpha::accelerator $key $mods] {}
		    $m add command -label [::msgcat::mc $label] -command $cmd \
		      -accelerator $accel -state $state
		    
		    if {[info exists image]} {
			$m entryconfigure end -compound left -image $image
		    }
		    
		    if {$accelerator != ""} {
			regsub -all "Ctrl" $accelerator "Control" accelerator
			
			if {[string first "Shift" $accelerator] == -1} {
			    if {[regexp {[A-Z]$} $accelerator last]} {
				regsub {[A-Z]$} $accelerator [string tolower $last] accelerator
			    }
			}
			set to "Alpha"
			if {[info exists opts(-M)]} {
			    set to "$opts(-M)AlphaStyle"
			}
			
			bind $to "<${accelerator}>" "[list $m invoke $label] ; break"
		    }
		}
		"^ *-+ *\$" -
		{^\(-} {
		    $m add separator
		    continue
		}
		default {
		    regsub {^(<[UIOCSEB])*} $item "" item
		    regexp {^(.*)(&|\^.)$} $item "" item
		    set state [expr {[regexp {^\((.*)} $item "" item] ? \
		      "disabled" : "normal"}]
		    set entrytype command
		    set isOn 0
		    if {[regexp {^!(.)} $item "" markc]} {
			switch -- $markc {
			    "\u2022" {
				set entrytype checkbutton
				set item [string range $item 2 end]
				set label $item
				set isOn 1
			    }
			    " " {
				set entrytype checkbutton
				set item [string range $item 2 end]
				set label $item
				set isOn 0
			    }
			    default {
				set item [string range $item 2 end]
				set label "$markc   $item"
			    }
			}
		    } else {
			set label $item
		    }
		    if {![info exists opts(-m)]} {
			set label [quote::Menuify $label]
		    } 
		    regexp {^(.*)\u2026$} $item "" item
		    if {[info exists opts(-t)]} {
			set entrytype $opts(-t)
		    }
		    if {$proc == ""} {
			$m add $entrytype -label [::msgcat::mc $label] \
			  -command "$item" -state $state
		    } else {
			$m add $entrytype -label [::msgcat::mc $label] \
			  -command "$proc [list $opts(-n) $item]" \
			  -state $state
		    }
		    if {$entrytype == "radiobutton"} {
			global alphaDummy
			if {$isOn} {
			    set alphaDummy(menu,$m) $label
			}
			$m entryconfigure end -variable \
			  alphaDummy(menu,$m) -value $label
			# I used to think we needed '-command ""' here,
			# but in fact that's not true.  While we don't
			# need the command to have the variable set correctly,
			# we do need it so Alpha's other procedures can
			# register the change, and perhaps schedule this
			# item to be saved as a modified preference.
		    } elseif {$entrytype == "checkbutton"} {
			global alphaDummy
			set alphaDummy(menu,$m,$label) $isOn
			$m entryconfigure end -variable \
			  alphaDummy(menu,$m,$label)
		    }
		}
	    }
	    # that was the end of the switch

	}
	# that was the end of the if 'cascade' else 'switch'
	if {$count <= 0} {
	    set idx [_menuItemToIndex $m $label]
	    set count [expr {$multiColumnMenusEveryNItems -1}]
	    while {1} {
		# This will fail if, e.g., it is a 'separator' menu entry.
		if {![catch {$m entryconfigure $idx -columnbreak 1}]} {
		    break
		}
		incr count -1
		incr idx -1
		if {$idx < 2} {
		    break
		}
	    }
	} 
    }
    return [list $m $opts(-n)]
}
# removeMenu <name> - remove menu 'name' from menubar, except those 
#  specified by previous 'makeMenuPermanent' calls.
proc removeMenu {n} {
    .menubar delete [lindex [menu_tags $n] 0]
}

namespace eval alphatk {}

proc alphatk::itemsToMenu {m cmdscript items} {
    foreach item $items {
	switch -regexp -- $item {
	    "/." {
		regexp {/(.)} $item "" key
		regsub "/[quote::Regfind ${key}]" $item "" item 
		if {$key == "\x1e"} {
		    # special case 'icon'
		    set icon 1
		    set key ""
		}
		if {[regexp {^(.*)(\\?&|\^.)$} $item "" item other]} {
		    # If this isn't true it's the \\?& branch.
		    # If this isn't true it's the \\?& branch.
		    if {!$::alpha::noCompoundMenus \
		      && ([string index $other 0] == "^")} {
			if {[info exists icon]} {
			    # add icon and set compound
			    scan [string index $other 1] %c int
			    # magic number according to which Alpha on MacOS behaves!
			    incr int 208
			    set image ics8$int
			    set ok 1
			    if {[catch {image width $image}]} {
				set ok 0
				global HOME
				set ifile [file join $HOME AlphaGifs Alpha$int.gif]
				if {[file exists $ifile]} {
				    if {![catch {image create photo $image -file $ifile}]} {
					set ok 1
				    }
				}
			    }
			    if {!$ok} {
				unset image
			    }
			} else {
			    # add bitmap and set compound
			    set int [string index $other 1]
			    incr int 256
			    set image ICON$int
			    set ok 1
			    if {[catch {image width $image}]} {
				set ok 0
				global HOME
				set ifile [file join $HOME AlphaGifs Alpha$int.gif]
				if {[file exists $ifile]} {
				    if {![catch {image create photo $image -file $ifile}]} {
					set ok 1
				    }
				}
			    }
			    if {!$ok} {
				unset image
			    }
			}
		    }
		}
		set state [expr {[regexp {^\((.*)} $item "" item] ? \
		  "disabled" : "normal"}]
		    
		set label [quote::Menuify $item]
		regexp {^(.*)\u2026$} $item "" item
		$m add command -label [::msgcat::mc $label] \
		  -command "$cmdscript $item" -state $state
		if {[info exists image]} {
		    $m entryconfigure end -compound left -image $image
		}
	    }
	    {\(-} {
		$m add separator
		continue
	    }
	    default {
		regsub {^(<[UIOCSEB])*} $item "" item
		regexp {^(.*)(&|\^.)$} $item "" item
		set state [expr {[regexp {^\((.*)} $item "" item] ? \
		  "disabled" : "normal"}]
		set label [quote::Menuify $item]
		regexp {^(.*)\u2026$} $item "" item
		if {[info exists opts(-t)]} {
		    set entrytype $opts(-t)
		}
		$m add command -label [::msgcat::mc $label] \
		  -command "$cmdscript $item" -state $state
	    }
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menu_tags" --
 # 
 #  This very important procedure takes a standard Alpha menu name, and
 #  converts it into a list of two items: the actual visible textual
 #  label to use for the menu, and the suffix to use for the Tk menu
 #  widget.  The actual menu will be created with '$suffix'.
 #  
 #  To avoid nameclashes between converted toplevel items starting
 #  with a bullet, and submenus, we prepend 'top' or 'm' as appropriate.
 # -------------------------------------------------------------------------
 ##
proc menu_tags {n} {
    global menuFunnynames 
    if {[info exists menuFunnynames($n)]} {
	return [lindex $menuFunnynames($n) 1]
    }
    if {[regexp {^[\u2022]} $n]} {
	global index::feature
	if {[info exists index::feature]} {
	    foreach m [array names index::feature] {
		if {![catch [list uplevel \#0 [list set $m]] res]} {
		    if {$res == $n} {
			regexp {(.*)Menu$} $m "" menuFunnynames($n)
			set menuFunnynames($n) [quote::Menuify $menuFunnynames($n)]
			# Little hack to get us a nice menu name for the open windows menu
			if {$menuFunnynames($n) == "Open Windows"} {
			    set menuFunnynames($n) "Windows"
			}
			set root [::msgcat::mc $menuFunnynames($n)]
			break
		    }
		}
	    }
	    # This is now obsolete, since this menu is a feature.
	    if {![info exists root]} {
		set menuFunnynames($n) "Windows"
		set root [::msgcat::mc "Windows"]
	    }
	    regsub -all {\.} $root {} rr
	    regsub -all " " $rr "_" rr
	    set res [list $root .menubar.top$rr]
	    if {[info exists menuFunnynames($n)]} {
		set menuFunnynames($n) [list $menuFunnynames($n) $res]
	    }
	    return $res
	}
	alertnote "Bug in Alphatk: please report menu_tags problem"
    } else {
	set root [::msgcat::mc $n]
	regsub -all {\.} $root {} rr
	regsub -all " " $rr "_" rr
	return [list $root .menubar.m$rr]
    }
}

proc menu_set_widget {n w} {
    global menuFunnynames
    set menuFunnynames($n) [list $n [list "temp" $w]]
}

#  bindings  #

# deleteModeBindings <mode> - Delete all bindings for specified mode.
proc deleteModeBindings {args} {echo "deleteModeBindings $args"}
# describeBinding - display the binding of the next typed key sequence. 
proc describeBinding {} {
    set key [alpha::waitForKey]
    if {![describeBindingHelper $key]} {
	alertnote "No binding for $key!"
    }
}

proc describeBindingHelper {key {noshift 0}} {
    global mode
    foreach tag [list ${mode}AlphaStyle AlphaStyle Alpha ${mode}AlphaMenuStyle AlphaMenuStyle AlphaMenu] {
	if {[bind $tag "<${key}>"] != ""} {
	    if {$noshift} {
		alertnote "$key (without 'Shift', though) is\
		  bound to [bind $tag <${key}>]"
	    } else {
		alertnote "$key bound to [bind $tag <${key}>]"
	    }
	    return 1
	}
    }
    if {[string first "Shift-" $key] != -1} {
	regsub -all "Shift-" $key "" key
	return [describeBindingHelper $key 1]
    }
    return 0
}

# keyAscii - insert ascii representation (in decimal)
#  of the keydown event, plus a modifier string, if 
#  necessary.
proc keyAscii {} {
    # Not quite right, but better than nothing.
    keyCode
}
# keyCode - insert the key code along w/ a string 
#  representing and modifiers into the current window.
#  Can be used to create bindings in 'Alphabits'.
proc keyCode {} {
    set key [alpha::waitForKey]
    set items [split $key -]
    set mods {}
    global alpha::keymap alpha::command_key alpha::option_key
    foreach k [array names alpha::keymap] {
	if {$alpha::keymap($k) == [lindex $items end]} {
	    set char $k
	    break
	}
    }
    if {![info exists char]} {
	set char [lindex $items end]
    }
    foreach mod [lrange $items 1 end-1] {
	if {$mod == "Shift"} {
	    append mods "s"
	} elseif {$mod == "Control"} {
	    append mods "z"
	} elseif {$mod == $alpha::command_key} {
	    append mods "c"
	} elseif {$mod == $alpha::option_key} {
	    append mods "o"
	} else {
	    error "Illegal modifier '$mod'"
	}
    }
    set result "'$char'"
    if {[string length $mods]} {
	append result " <$mods>"
    }
    insertText $result
}

array set alpha::mods [list Ctrl 144 Control 144 Shift 34 Option 72 \
  Cmd 1 Command 1 Meta 72 Alt 1]
set alpha::modifiers [list Ctrl Control Shift Option Alt Command Cmd Meta]
proc alpha::waitForKey {} {
    global alphaPriv
    variable modifiers
    set oldFocus [focus]
    set oldGrab [grab current .status]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    message "Press any key combination."
    grab -global .status
    bind .status <KeyPress> {set alphaPriv(done) [list %A %K %N]}
    focus .status

    while 1 {
	vwait alphaPriv(done)
	#echo $alphaPriv(done)
	regsub -all -- {_[LR]} [set keycode [lindex $alphaPriv(done) 1]] "" keycode
	append key "-" $keycode
	if {[lsearch -exact $modifiers $keycode] == -1} {
	    break
	}
    }
    # set ascii [lindex $alphaPriv(done) 0]
    
    unset alphaPriv(done)
    bind .status <KeyPress> ""

    catch {grab release .status}
    catch {focus $oldFocus}
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    regsub -all {_[LR]} $key "" key
    if {[string first "Shift" $key] == -1} {
	if {[regexp {[A-Z]$} $key last]} {
	    regsub {[A-Z]$} $key [string tolower $last] key
	}
    }
    set key [string range $key 1 end]
    foreach mod $modifiers {
	regsub -all -- "(-$mod)+" $key "-$mod" key
    }
    message "Got keypress: $key"
    return $key
}

array set alpha::asciikeymap [list 0x03 Enter 0x20 " " 0x08 0x33 0x7f deleteChar]

# ascii (see bindings).
proc ascii {key args} {
    global alpha::asciikeymap
    if {[info exists alpha::asciikeymap($key)]} {
	set key $alpha::asciikeymap($key)
    } else {
	set key [text::Ascii $key 1]
    }
    uplevel Bind [list $key] $args
}
# unascii  (see bindings)
proc unascii {key args} {
    global alpha::asciikeymap
    if {[info exists alpha::asciikeymap($key)]} {
	set key $alpha::asciikeymap($key)
    } else {
	set key [text::Ascii $key 1]
    }
    uplevel unBind [list $key] $args
}

# bindingList - list all current bindings.
proc bindingList {} {
    echo "bindingList not perfect yet"
    global mode
    set tags [list ${mode}AlphaStyle AlphaStyle Alpha ${mode}AlphaMenuStyle AlphaMenuStyle AlphaMenu]
    set res {}
    foreach tag $tags {
	foreach b [bind $tag] {
	    set fn [bind $tag $b]
	    lappend res [list Bind $tag $b $fn]
	}
    }
    return [join $res "\r"]
}

# unBind  (see bindings)
proc unBind {key mods args} {
    # blank out the script and send it to Bind to be zapped.
    if {[string index $mods 0] == "<"} {
	set args [lreplace $args 0 0 ""]
    } else {
	set mods ""
    }
    
    eval [list Bind $key $mods] $args
}

# bind  (see bindings)
proc Bind {key mods args} {
    global alpha::keymap alpha::command_key alpha::option_key \
      tcl_platform alpha::menukeymap
    switch -- [llength $args] {
	0 { 
	    set script $mods 
	    set mods "" 
	    set bmode "" 
	}
	1 {
	    if {[string index $mods 0] == "<"} {
		set script [lindex $args 0]
		set bmode ""
	    } else {
		set script $mods
		set mods ""
		set bmode [lindex $args 0]
	    }
	}
	2 { 
	    set script [lindex $args 0]
	    set bmode [lindex $args 1] 
	}
	default {
	    error "Too many args to 'Bind'"
	}
    }
    set bind "<"
    # Build up list of modifiers, including list of
    # modifiers we don't have (which might be needed below).
    set not {}
    set yes {}
    if {[regexp {s} $mods]} {lappend yes "Shift"} else {
	lappend not "Shift"
    }
    if {[regexp {z} $mods]} {lappend yes "Control"} else {
	lappend not "Control"
    }
    if {[regexp {o} $mods]} {
	lappend yes $alpha::option_key
	set have_option 1
    } else {
	lappend not $alpha::option_key
    }
    if {[regexp {c} $mods]} {lappend yes $alpha::command_key} else {
	lappend not $alpha::command_key
    }
    if {[llength $yes]} {
	append bind [join $yes "-"] "-"
    }
    
    regexp "'(.)'" $key "" key
    if {[string length $key] > 1 && [regexp {^[a-z]} $key] \
      && ($key != "space") && ($key != "enter")} {
	set key "[string toupper [string index $key 0]][string range $key 1 end]"
    }
    if {[info exists alpha::keymap($key)]} {
	set key $alpha::keymap($key)
    } elseif {[info exists alpha::menukeymap($key)]} {
	set key $alpha::menukeymap($key)
    }
    append bind "KeyPress-" $key ">"
    if {$script == "prefixChar"} {
	set script "prefixChar ${bmode}Prefix-[string toupper $key]"
    } elseif {$script == "startEscape"} {
	set script "prefixChar ${bmode}Prefix-e"
    }

    if {[regexp {[eA-Z]} $mods prefix]} {
	append bmode Prefix- $prefix
	# auto-bind the prefix char if it's not currently set.
	# Alpha seems not to bother to bind ctrl-c automatically, for instance.
	if {[regexp {[A-Z]} $prefix]} {
	    if {[bind Alpha <Control-KeyPress-[string tolower $prefix]>] == ""} {
		message "We have no prefixChar binding for 'Ctrl-$prefix', so\
		  subsequent bindings will be ignored."
		#bind Alpha <Control-KeyPress-[string tolower $prefix]> \
		#"prefixChar Prefix-$prefix ; break"
	    }
	}
    } else {
	if {[string length $bmode]} {
	    set modeSpecific $bmode
	    append bmode AlphaStyle
	} else {
	    append bmode Alpha
	}
    }
    #echo [list bind $bmode $bind $script]
    if {$script != ""} { append script " ; break" }
    if {$tcl_platform(platform) == "windows" && [info exists have_option]} {
	set ignore "no meta key"
    }
    if {$key == "Enter"} {
	set ignore "no Enter key"
    }
    if {[info exists ignore]} {
	echo "FYI: keyboard has $ignore; ignoring [list bind $bmode $bind $script]"
	return
    }
    if {[catch [list bind $bmode $bind $script]]} {
	global badkeylog ; lappend badkeylog "$bmode $bind $script"
	return
    }
    if {[string first "Shift" $bind] != -1} {
       if {[regexp -- {-[a-z]>$} $bind last]} {
	   regsub {[a-z]>$} $bind [string toupper $last] bind
	   if {[catch [list bind $bmode $bind $script]]} {
	      global badkeylog ; lappend badkeylog "$bmode $bind $script"
	      return
	   }
       }
    }
    if {[info exists modeSpecific]} {
	# Now we need to look for bindings at the 'Alpha' level
	# which are more specific than this one.
	set l [llength $not]
	if {$l} {
	    set combos [expr {1 << $l}]
	    set bind [string range $bind 1 end]
	    for {set i 1} {$i < $combos} {incr i} {
		set mods "<"
		for {set j 0} {$j < $l} {incr j} {
		    if {$i & (1 << $j)} {
			append mods [lindex $not $j] "-"
		    }
		}
		append mods $bind
		set bound [bind Alpha $mods]
		#puts stdout "Check: $i, $combos, $j, $mods"
		if {[string length $bound]} {
		    # Got a more specific match.  We bind it
		    # again for this mode specific case, unless
		    # the current mode had already got something
		    # bound to that keypress.
		    #puts "got 'bind $bmode $mods $bound'"
		    if {[bind $bmode $mods] == ""} {
			bind $bmode $mods $bound
		    }
		}
	    }
	}
    } else {
	# Now we need to look for mode-specific bindings which
	# are less specific than this one
	set l [llength $yes]
	if {$l} {
	    set justkey [string range $bind \
	      [expr {1+[string last "-" $bind]}] end]
	    # This loop can be pretty time consuming, but we do our
	    # best to abort early.
	    foreach m [mode::listAll] {
		append m AlphaStyle
		if {![llength [bind $m]]} {continue}
		if {[string length [bind $m $bind]]} {
		    continue
		}
		
		set combos [expr {1 << $l -1}]
		for {set i 0} {$i < $combos} {incr i} {
		    set mods "<"
		    for {set j 0} {$j < $l} {incr j} {
			if {$i & (1 << $j)} {
			    append mods [lindex $yes $j] "-"
			}
		    }
		    append mods $justkey
		    set bound [bind $m $mods]
		    #puts stdout "Check: ${m} $i, $combos, $j, $mods"
		    if {[string length $bound]} {
			#puts "got $bound"
			# Got a less specific match.  We bind it
			# again for this mode specific case.
			bind $m $bind $script
			continue
		    }
		}
	    }
	}
	
    }
}
# need 0x21 0x29 0x34 0x24 0x1b Kpad=
array set alpha::menukeymap [list]
array set alpha::keymap [list 0x27 quoteright 0x2f period \
  - minus + plus * asterisk \
  0x31 space "\r" Return " " space 0x33 BackSpace \
  0x24 Return "\n" backslash_n_is_not_allowed \
  "\t" Tab 0x30 Tab "" Left "" Right 0x7b Left 0x7c Right \
  Del Delete Esc Escape 0x7d Down 0x7e Up 0x13 @ \
  Pgup Prior 0x74 Prior Pgdn Next 0x79 Next . period , comma \
  "\]" bracketright "\[" bracketleft = equal ? question "/" slash \
  ' quoteright ` quoteleft "\\" backslash ";" semicolon \
  Kpad/ KP_Divide Kpad* KP_Multiply Kpad- KP_Subtract Kpad+ KP_Add \
  KpadEnter KP_Enter Kpad.  KP_Decimal Kpad1 KP_1 Kpad2 KP_2 Kpad3 KP_3 \
  Kpad4 KP_4 Kpad5 KP_5 Kpad6 KP_6 Kpad7 KP_7 Kpad8 KP_8 Kpad9 KP_9 \
  Kpad0 KP_0 Kpad= KP_Equal ( parenleft ) parenright < less > greater \
  ]

# Keypad Numlock  NUMOCK            (Not NUMLOCK)

# prefixChar - used to further modify the next keystroke 
#  combination, in the same manner as using the shift key 
#  in the next keystroke
proc prefixChar {{bt ""}} {
    if {$bt == ""} {
	alpha::errorAlert "Prefix char called without argument; shouldn't happen!"
    } else {
	message "Prefix..."
	text_cmd binding_capture $bt
    }
}
# startEscape - used to further modify the next 
#  keystroke combination, in the same manner as using the 
#  shift key in the next keystroke
proc startEscape {args} {
    alertnote "Shouldn't call 'startEscape' -- diverted to prefix char!"
}

# float -m <menu> [<-h|-w|-l|-t|-M> <val>] [-n winname] [-z tag] -
#  Takes a created menu (not necessarily in the menubar), and makes a 
#  floating window out of it. Returns integer tag that is used to remove 
#  the window. NOT DYNAMIC!  W/ no options returns all currently defined menus.
#  Displayed w/ system floating window WDEF if system 7.5, plainDBox 
#  otherwise. -h through -M change width, height, left margin, top margin, and
#  margin between buttons. -z allows a ten-char tag to be specified for 
#  'floatShowHide'.
proc float {args} {
    # Default location
    set opts(-t) 0
    set opts(-l) 0
    getOpts [list -m -h -w -l -t -M -n -z]
    if {[llength $args]} {
	return -code error "Bad arguments '$args'"
    }
    if {![info exists opts(-m)]} {
	return -code error "No '-m menu' argument to float"
    }
    
    foreach {m widg} [menu_tags $opts(-m)] {}
    set m $widg
    if {[info tclversion] < 8.4} {
	return [tkTearOffMenu $m $opts(-l) $opts(-t)]
    } else {
	return [tk::TearOffMenu $m $opts(-l) $opts(-t)]
    }
}
# floatShowHide <on|off> <tag> - Shows or hides all floats w/ specified 
#  tag. Tags of current modes are automatically shown/hidden.
proc floatShowHide {args} {
    global tearoffMenus
    if {$tearoffMenus} {
	echo "floatShowHide $args"
    }
}
# unfloat <float num> - removes specified floating window. W/ no options 
#  lists all floating windows.
proc unfloat {{floatName ""}} {
    global menu::floats
    if {![string length floatName]} {
	return [array names menu::floats]
    } elseif {![info exists menu::floats($floatName)]} {
	error "'$floatName' is not a floating menu."
    } else {
	destroy $floatName
    }
}

proc closeFloat {} {
    global menu::floats
    unfloat [lindex [lsort [array names menu::floats]] end]
}
