## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "alpha_windows.tcl"
 #                                    created: 04/12/98 {22:45:38 PM} 
 #                                
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html
 #  
 # Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on use and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##

namespace eval win {}
namespace eval tw {}

# This allows some global things to happen in a slightly simpler
# fashion even when there are no windows around.  I don't think
# it has any adverse side-effects.  It associates the Tk window '.'
# with the Alpha window "" (i.e. no window).
set win::tk() .
# This command is eventually called when you try to execute
# something (e.g. getPos) which requires an open window when
# there isn't an open window.  So, it's convenient to have
# it throw this error.
proc ::tw::. {args} {
    error "No window is open (while executing '$args')"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "alpha::divertTkCallToWindow" --
 # 
 #  Used to divert the Tk callbacks from the enhanced text widget to
 #  callbacks into the AlphaTcl library which use window names
 #  (win/file paths with optional <2>) as file identifiers.
 #  
 #  We basically convert from a Tk widget path (.al0.text1) to a window
 #  name, and leave the rest of the arguments intact, passing them to
 #  'cmd'.
 # -------------------------------------------------------------------------
 ##
proc alpha::divertTkCallToWindow {cmd w args} {
    eval $cmd [list $::win::tktitle($w)] $args
}

proc ::alphatk_lockClick {f on} {
    if {$on} {
	hook::callAll lockHook "" $f
    } else {
	if {[file exists [win::StripCount $f]]} {
	    file attributes [win::StripCount $f] -readonly 0
	}
	hook::callAll unlockHook "" $f
    }
}

proc ::alphatk_hyper {cmd} {
    placeBookmark
    uplevel \#0 $cmd
}

proc splitter {w} {
    $w configure -relief sunken
    text_cmd toggleSplit 
}

proc toggleLineNumbers {} {
    text_cmd toggleLineNumbers
}

## 
 # -------------------------------------------------------------------------
 # 
 # "alpha::embedInto" --
 # 
 #  To embed Alphatk's functionality within any Tk window, one must
 #  pass the following arguments:
 #  
 #  alpha::embedInto ?-text contents? ?-encoding enc? ?-mode m? name id|tkwin
 #  
 #  so, optionally:
 #  
 #   -mode m
 #   -encoding enc
 #   -text contents
 #   
 #  and two required arguments:
 #   
 #   name -- name/title Alphatk should use for the window.
 #   into -- either a tk window path or a window id (if it is the
 #           window id of a tk container in a separate process, it
 #           must have been created with '-container 1'), or a 
 #           toplevel description which is a list of six elements,
 #           "toplevel" title l t w h
 #   
 #  Unfortunately embedded windows currently seem to have trouble
 #  with notification of their destruction (i.e. win::kill is never
 #  called).
 # -------------------------------------------------------------------------
 ##
proc alpha::embedInto {args} {
    set opts(-text) ""
    set opts(-encoding) ""
    getOpts [list -text -encoding -mode]
    if {[llength $args] != 2} {
	error "Bad arguments: should be \"alpha::embedInto\
	  ?-text contents? ?-encoding enc? ?-mode m? name id|tkwin\""
    }
    foreach {name into} $args {break}
    if {[info exists opts(-mode)]} {
	set win::Modes($name) $opts(-mode)
    }
    switch -- [string index $into 0] {
	"." {
	    # It's a tk window.  Should be a frame or toplevel
	    coreWindow $name $into $opts(-text) $opts(-encoding)
	}
	"0" {
	    # It's a window id.  This should be either a Tk window
	    # which has '-container 1' set, or just some window in
	    # some other application.
	    set w [_uniqueToplevel -use $into]
	    coreWindow $name $w $opts(-text) $opts(-encoding)
	}
	default {
	    if {[catch {lindex $into 0}] \
	      || ([lindex $into 0] != "toplevel") \
	      || ([llength $into] != 6)} {
		error "Should be a window id (0x...), a tk window path, \
		  or a description of a toplevel (toplevel title l t w h)"
	    }
	    foreach {dummy title x y w h} $into {}

	    # Create the toplevel
	    set ww [_uniqueToplevel]
	    wm withdraw $ww

	    # Calculate and set window geometry and title
	    set geom ${w}x${h}+${x}+${y}
	    wm geometry $ww $geom
	    wm title $ww $title
	    update idletasks

	    coreWindow $name $ww $opts(-text) $opts(-encoding)

	}
    }
    registerWindowWithAlphaTcl $name
}

set controllerMargin 0

## 
 # -------------------------------------------------------------------------
 # 
 # "alpha::coreWindow" --
 # 
 #  To embed Alphatk's functionality within any Tk window, one must
 #  take the following sequence of actions.  
 #  
 #  (i) optionally set 'win::Modes($name)' to the mode to use for
 #  this window.  If not set, Alphatk will determine one for you
 #  based upon the extension of '$name' (the default is Text mode,
 #  of course).
 #  
 #  (ii) call 'alpha::coreWindow $name $w $text $encoding'
 #  
 #  (iii) call 'registerWindowWithAlphaTcl $n'
 #  
 #  That is sufficient!  Or you can call 'alpha::embedInto' which
 #  will do all of the above for you.
 #  
 #  Here '$w' is the arbitrary Tk window, '$name' is the name of the
 #  window, and $text, $encoding are the contents and encoding of the
 #  window (optional).
 #  
 # -------------------------------------------------------------------------
 ##
proc alpha::coreWindow {name w {text ""} {encoding ""}} {
    global tcl_platform win::Modes defaultFont fontSize tabSize
    if {![info exists win::Modes($name)]} {
	set win::Modes($name) [win::FindMode $name]
    }
    
    set textwin [tw::MakeWindow $w $text -encoding $encoding \
      -font "$defaultFont $fontSize" -tabsize $tabSize]
    
    # Note: 'width 1' is too narrow, but 'width 2' is a little larger
    # than is necessary.  Hence we use width 1 here, but then use
    # grid -sticky ew to let them expand to the available space.  This
    # allows the right margin to size itself to the width of the scrollbar,
    # which can vary depending on the current appearance/colour-scheme.
    menubutton [tw::Toolbar $w path vcs] -text "V" -padx 0 -pady 0 -width 1 -relief ridge \
      -activebackground [color activebackground] -menu [tw::Toolbar $w path vcs].menu \
      -activeforeground [color activeforeground]
    menubutton [tw::Toolbar $w path marks] -text "M" -padx 0 -pady 0 -width 1 -relief ridge \
      -activebackground [color activebackground] -menu [tw::Toolbar $w path marks].menu \
      -activeforeground [color activeforeground]
    menubutton [tw::Toolbar $w path func] -text "\{ \}" -padx 0 -pady 0 -width 1 -relief ridge \
      -activebackground [color activebackground] -menu [tw::Toolbar $w path func].menu \
      -activeforeground [color activeforeground]
    menubutton [tw::Toolbar $w path files] -text "f" -padx 0 -pady 0 -width 1 -relief ridge \
      -activebackground [color activebackground] -menu [tw::Toolbar $w path files].menu \
      -activeforeground [color activeforeground]
    bindtags [tw::Toolbar $w path vcs] [concat Vcspopup [bindtags [tw::Toolbar $w path vcs]]]
    bindtags [tw::Toolbar $w path marks] [concat Markspopup [bindtags [tw::Toolbar $w path marks]]]
    bindtags [tw::Toolbar $w path func] [concat Funcspopup [bindtags [tw::Toolbar $w path func]]]
    bindtags [tw::Toolbar $w path files] [concat Filespopup [bindtags [tw::Toolbar $w path files]]]

    if {$tcl_platform(platform) == "macintosh"} {
	# Makes things look a little better on MacOS, which has a
	# rather larger default font.
	[tw::Toolbar $w path vcs] configure -font "Monaco 9"
	[tw::Toolbar $w path marks] configure -font "Monaco 9"
	[tw::Toolbar $w path func] configure -font "Times 9" -pady 2 -text "\{\}"
	[tw::Toolbar $w path files] configure -font "Monaco 9"
    }
    
    tw::Toolbar $w add vcs
    tw::Toolbar $w add marks
    tw::Toolbar $w add func
    tw::Toolbar $w add files
    
    update idletasks
    alpha::setIcon $w
    global win::tk win::tktitle
    set win::tk($name) $textwin
    set win::tktitle($textwin) $name
    ::winCreatedHook $name
    
    if {[winfo toplevel $w] == $w} {
	wm protocol $w WM_DELETE_WINDOW [list killWindow $name]
	global useGlobalMenuBarOnly
	if {![info exists useGlobalMenuBarOnly] || !$useGlobalMenuBarOnly} {
	    if {$tcl_platform(platform) != "macintosh"} {
		$w configure -menu .menubar
	    }
	}
    }
    # We could just use a 'Toplevel' class binding, but then we spread
    # assumptions of the name of the text window to more parts of the
    # code.
    bind $w <Destroy> [list win::kill $textwin]
    
    focus $textwin
    return $w
}

# This is the ONLY piece of code that cares the given window
# is called '.alN'.  If any 'args' are given, they are passed
# to the toplevel's creation command (e.g. -use 0x... for embedding)
proc alpha::_uniqueToplevel {args} {
    set nn 0
    while {[winfo exists .al$nn]} {incr nn}
    eval [list toplevel [set w .al$nn]] $args
    return $w
}

proc flash {char} {
    if {[text::isEscaped]} {
	# it's a literal character
	return
    }
    set pos [pos::math [getPos] - 1]
    if {[catch {matchIt $char $pos} matched]} {
	beep
	message $matched
	return
    } else {
	blink $matched
    }
}

proc win::diskModified {name {mod 1} {diff 0}} {
    variable tk
    set w $tk($name)
    if {$mod} {
	$w statusConfigure background [default::color activebackground]
    } else {
	$w statusConfigure background [default::color background]
    }
}

proc win::openHook {name} {
    variable Modes 
    variable tk

    set w $tk($name)
    bindtags $w [concat $Modes($name)AlphaStyle [bindtags $w]]
    ::tw::arrangeToColour $w {} {}
}

proc win::modeHasChanged {name oldmode newmode} {
    variable tk
    set w $tk($name)
    append oldmode "AlphaStyle"
    append newmode "AlphaStyle"
    set oldtags [bindtags $w]
    set idx [lsearch -exact $oldtags $oldmode]
    set newtags [lreplace $oldtags $idx $idx $newmode]
    bindtags $w $newtags
}

proc win::kill {w} {
    variable tktitle
    if {[info exists tktitle($w)]} {
	killWindow $tktitle($w) 1
    }
}

# This version is probably better.
proc text_cmd {cmd args} {
    global tw::split
    if {![info exists tw::split([set w [focus]])]} {
	global win::tk win::Active
	set w $win::tk([lindex $win::Active 0])
    }
    uplevel 1 [list tw::$cmd $w] $args
}

proc text_cmd {cmd args} {
    global win::tktitle
    if {![info exists win::tktitle([set w [focus]])]} {
	global win::tk win::Active
	set w $win::tk([lindex $win::Active 0])
    }
    uplevel 1 [list tw::$cmd $w] $args
}

proc text_cmds {args} {
    global win::tktitle
    if {![info exists win::tktitle([set w [focus]])]} {
	global win::tk win::Active
	set w $win::tk([lindex $win::Active 0])
    }
    foreach cmd $args {
	set rest [lrange $cmd 1 end]
	set cmd [lindex $cmd 0]
	uplevel 1 [list tw::$cmd $w] $rest
    }
}

proc text_wcmd {ww cmd args} {
    global win::tk 
    set w $win::tk($ww)
    uplevel 1 [list tw::$cmd $w] $args
}

proc text_wcmds {ww args} {
    global win::tk
    set w $win::tk($ww)
    foreach cmd $args {
	set rest [lrange $cmd 1 end]
	set cmd [lindex $cmd 0]
	uplevel 1 [list tw::$cmd $w] $rest
    }
}

proc win::titleBarPopup {w x y} {
    set path [::getTitleBarPath]
    destroy .titlepop
    menu .titlepop -tearoff 0
    set build {}
    foreach item [file split $path] {
	set build [file join $build $item]
	.titlepop insert 0 command -label $item -command [list titlebar $build]
    }
    tk_popup .titlepop $x $y 0
}

switch -- $tcl_platform(platform) {
    "unix" -
    "windows" {
	bind AlphaStyle <Alt-Double-Button-1> {cmdDoubleClick}
	bind AlphaStyle <Alt-Control-Double-Button-1> {cmdDoubleClick -1 -1 0 0 1}
	bind AlphaStyle <Alt-Shift-Double-Button-1> {cmdDoubleClick -1 -1 0 1 0}
	bind AlphaStyle <Alt-Control-Shift-Double-Button-1> {cmdDoubleClick -1 -1 0 1 1}
	bind AlphaStyle <Button-3> {win::titleBarPopup %w %X %Y}
    }
    "macintosh" {
	bind AlphaStyle <Mod1-Double-Button-1> {cmdDoubleClick}
    }
}

bind Markspopup <Button-1> "menu_set_widget marksMenu \"%W.menu\" ; ::marksMenuHook"
bind Funcspopup <Button-1> "menu_set_widget parseMenu \"%W.menu\" ; ::parseMenuHook"
bind Vcspopup <Button-1> "menu_set_widget vcsMenu \"%W.menu\" ; ::vcsMenuHook"
bind Filespopup <Button-1> "menu_set_widget relatedFilesMenu \"%W.menu\" ; ::relatedFilesMenuHook"

bind Splitter <Enter> "%W configure -highlightbackground \[color activebackground\]"
bind Splitter <Leave> "%W configure -highlightbackground \[color background\]"
bind Splitter <Button-1> "splitter %W"
bind Splitter <ButtonRelease-1> "%W configure -relief raised"
if {($tcl_platform(platform) == "windows")} {
    bind Alpha <Control-z> {undo ; break}
    bind Alpha <Control-Shift-z> {redo ; break}
    bind Alpha <Control-x> {cut ; break}
    bind Alpha <Control-c> {copy ; break}
    bind Alpha <Control-v> {paste ; break}
}
bind AlphaStyle <<Paste>> {paste ; break}
bind AlphaStyle <<Cut>> {cut ; break}
bind AlphaStyle <<Copy>> {copy ; break}
bind AlphaStyle <<Clear>> {clear ; break}
balloon::help Splitter "Click here to split/unsplit the window"
balloon::help Vcspopup "Click here to access version control functions"
balloon::help Markspopup "Click here to access file marks"
balloon::help Funcspopup "Click here to access functions in this window"
balloon::help Filespopup "Click here to access related files"
balloon::help Lock "Shows modified/dirty status of the window.\
  A red dot indicates unsaved changes (and you can click here to save),\
  a dark background indicates that\
  the file has changed on disk, and a lock indicates the window is not editable."
