# This file (unlike most of the rest of Alphatk core)
# is distributed under a BSD style license.

option add *Balloonhelp*background yellow
option add *Balloonhelp*foreground black

## 
 # -------------------------------------------------------------------------
 # 
 # "Balloon Help" --
 # 
 #  Simple way of attaching balloon help to any tk window or class
 #  (anything we can use bind or bindtags on).
 #  
 #  The balloon is coloured appropriately, follows the mouse, and
 #  tries hard to ensure it isn't placed half off the screen.
 # -------------------------------------------------------------------------
 ##
namespace eval balloon {
    variable info
    variable time 650
    
    # If balloon help is set for windows which contain as children
    # other windows which also have balloon help, Tk doesn't give us
    # the kind of leave-enter events we want.  We keep a stack of
    # current nested balloon help requests to deal with this situation
    # properly.
    set info(stack) ""
    
    ## 
     # ---------------------------------------------------------------------
     # 
     # "help" --
     # 
     # Register some information with a particular tk window or tk class
     # (anything to which 'bind' and 'bindtags' may be applied).
     # 
     # For simplest possible balloon help, 'args' is just a single argument
     # containing the text to show when the user's mouse dwells on the item.
     # 
     # For most items, args will contain 2 arguments, in order:
     # 
     # 1) Text for item when enabled
     # 2) Text for item when disabled
     # 
     # However, for checkboxes, the above two pieces of text are used when
     # the item is 'off/unchecked', and an additional two arguments may
     # be given:
     # 
     # 3) Text for item when enabled and on
     # 4) Text for item when disabled and on.
     # 
     # If either of items 2/4 is empty, the default is to use items 1/3
     # respectively, and if item 3 is item, the default is to use item 1.
     # If the item finally chosen to be shown is empty, no balloon is shown.
     # ----------------------------------------------------------------------
     ##
    proc help {item args} {
	variable info

	if {[winfo exists $item]} {
	    set hitem "Help-$item"
	    bindtags $item [concat [list $hitem] [bindtags $item]]
	    bind $item <Destroy> "balloon::end $item %W"
	} else {
	    set hitem $item
	}
	bind $hitem <Any-Enter> \
	  "balloon::_start $item %W; [bind $item <Any-Enter>]" 
	bind $hitem <Motion> "balloon::_follow %W ; [bind $item <Motion>]" 
	bind $hitem <Any-Leave> \
	  "balloon::_hide $item %W; [bind $item <Any-Leave>]"
	bind $hitem <Unmap> "balloon::_hide $item %W ; [bind $item <Unmap>]"
	bind $hitem <Button-1> \
	  "balloon::_button $item %W ; [bind $item <Button-1>]"
	
	set info($item,text) $args
    }

    proc forgetClass {item} {
	variable info
	foreach b {Any-Enter Any-Leave Button-1 Motion Unmap} {
	    regsub {^[^;]+ ; } [bind $item <$b>] {} old
	    bind $item <$b> $old
	}
	unset info($item,text)
    }
    
    proc forget {} {
	variable info
	if {[info exists info]} {
	    unset info
	}
    }
    
    proc end {item w} {
	variable info
	if {$item == $w} {
	    unset info($item,text)
	    _remove
	    
	    catch {
		bind $item <Any-Enter> ""
		bind $item <Any-Leave> ""
		bind $item <Unmap> ""
		bind $item <Button-1> ""
		bind $item <Destroy> ""
	    }
	} else {
	    _remove
	}
    }

    proc _button {item w} {
	variable info
	if {[info exists info($w,after)]} {
	    after cancel $info($w,after)
	    if {[winfo exists .balloon.l]} {
		_check $item $w
	    }
	}
    }
    
    proc _start {item w} {
	variable info 
	variable time
	
	set top [lindex $info(stack) end]
	if {$top != $w} {
	    lappend info(stack) $w
	}
	
	if {[info exists info(after)]} {
	    # If a previous balloon is still showing, but we have
	    # already left it's window, and entered a new window, we
	    # should show the new balloon immediately.
	    
	    # Cancel destruction of previous balloon
	    after cancel $info(after)
	    unset info(after)
	    # Show new balloon
	    _show $item $w
	} elseif {[llength $info(stack)] > 1} {
	    # Show new balloon immediately
	    _show $item $w
	} else {
	    # No balloon is currently showing, so wait before presenting
	    # the user with their first balloon
	    set info($w,after) [after $time "balloon::_show $item $w"]
	}
    }

    proc _remove {} {
	# This procedure may be called twice if we 'end' balloons,
	# while an 'after' call is still outstanding.  So we just
	# need to make sure that it doesn't mind a second call.
	::destroy .balloon
	variable info
	if {[info exists info(after)]} {
	    after cancel $info(after)
	    unset info(after)
	}
    }
    
    proc _follow {w} {
	if {[winfo exists .balloon]} {
	    _checkPosition [expr {[winfo pointerx $w] + 20}] \
	      [expr {[winfo pointery $w] + 20}]
	}
    }
    
    proc _checkPosition {posx posy} {
	if {[winfo exists .balloon]} {
	    set width [winfo reqwidth .balloon]
	    # If right side is offscreen, move to the left...
	    if {$posx + $width > [winfo screenwidth .balloon]} {
		set posx [expr {$posx - 40 - $width}]
	    }
	    # ... but not too far.
	    if {$posx < 0} { set posx 0 }
	    
	    set height [winfo reqheight .balloon]
	    # If bottom is offscreen, move up...
	    if {$posy + $height > [winfo screenheight .balloon]} {
		set posy [expr {$posy - 40 - $height}]
	    }
	    # ... but not too far.
	    if {$posy < 0} { set posy 0 }
	    wm geometry .balloon "+${posx}+${posy}"
	    update idletasks
	}
    }
    
    proc _show {item w} {
	if {![winfo exists $w]} {
	    _hide $item $w
	    end $item $w
	    return
	}
	
	variable info
	variable time
	
	set top [lindex $info(stack) end]
	if {$top != $w} {
	    # If we're not top of the stack, reschedule
	    set info($w,after) [after $time "balloon::_show $item $w"]
	    return
	}
	
	# the position of the balloon window
	set posx [expr {[winfo pointerx $w] + 20}]
	set posy [expr {[winfo pointery $w] + 20}]

	# The .balloon often exists from the previous balloon
	::destroy .balloon
	global tcl_platform
	if {$tcl_platform(platform) == "macintosh"} {
	    toplevel .balloon -relief ridge -borderwidth 2 \
	      -class Balloonhelp ; ::tk::unsupported::MacWindowStyle\
	      style .balloon floating none
	} else {
	    toplevel .balloon -relief ridge -borderwidth 2 -class Balloonhelp 
	}
	.balloon configure -background [default::color activebackground]
	
	wm withdraw .balloon 
	wm overrideredirect .balloon 1
	if {$::tcl_platform(platform) == "macintosh"} {
	    ::tk::unsupported::MacWindowStyle style .balloon\
	      floating sideTitlebar
	}
	wm geometry .balloon "+${posx}+${posy}"

	set text [_text $item $w]
	if {[string length $text] > 500} {
	    set wraplength 3i
	} else {
	    set wraplength 2i
	}
	label .balloon.l -text $text \
	  -bg [default::color lighterbackground] -foreground black \
	  -bd 0 -font "helvetica 10" \
	  -justify left -wraplength $wraplength -padx 6 -relief solid
	pack .balloon.l

	# make it visible
	if {$text != ""} {
	    # This update is needed for tcl 8.2.1 or newer on Windows,
	    # which otherwise flashes the full window for a split second
	    # before collapsing it.
	    update idletasks
	    _checkPosition $posx $posy
	    # Unfortunately the above line can also destroy the balloon!
	    if {[winfo exists .balloon]} {
		wm deiconify .balloon
	    }
	}
	
	set info($w,after) [after $time "balloon::_check $item $w"]
    }
    
    proc _text {item w} {
	variable info
	# In principle we allow four different pieces of text to be
	# associated with each item.  In practice we only use 1 or 2.
	# This flexibility is nicely compatible with MacOS balloons.
	
	set offset 0
	if {![catch {$w cget -state} state]} {
	    if {$state == "disabled"} {
		incr offset 1
	    }
	}
	
	switch -- [winfo class $w] {
	    "Checkbutton" {
		# Checkbuttons display a different help text if the
		# box is checked or not.
		set value [uplevel \#0 set [$w cget -variable]]
		if {$value == [$w cget -onvalue]} {
		    incr offset 2
		}
	    }
	}
	
	set text [lindex $info($item,text) $offset]
	if {$text == "" && ($offset & 1)} {
	    incr offset -1
	    set text [lindex $info($item,text) $offset]
	    if {$text == "" && ($offset & 2)} {
		incr offset -2
	    }
	    set text [lindex $info($item,text) $offset]
	}
	return $text
    }
    
    proc _check {item w} {
	variable info
	variable time
	if {![winfo exists $w]} {
	    catch {_hide $item $w}
	    end $item $w
	} else {
	    if {[winfo exists .balloon.l]} {
		set top [lindex $info(stack) end]
		if {$top == $w} {
		    set text [_text $item $w]
		    if {$text == ""} {
			wm withdraw .balloon
		    } else {
			.balloon.l configure -text $text
			wm deiconify .balloon
		    }
		}
		set info($w,after) [after $time "balloon::_check $item $w"]
	    }
	}
    }
    
    proc _hide {item w} {
	variable info 
	variable time
	if {[info exists info($w,after)]} {
	    after cancel $info($w,after)
	    unset info($w,after)
	}
	set top [lindex $info(stack) end]
	if {$top == $w} {
	    set info(stack) [lreplace $info(stack) end end]
	}
	if {![llength $info(stack)]} {
	    set info(after) [after $time "balloon::_remove"]
	}
    }
    
}
