## -*-Tcl-*- (nowrap)
 # ==========================================================================
 #  Statistical Modes - an extension package for Alpha
 # 
 #  FILE: "SASCompletions.tcl"
 #                                    created: 05/14/00 {01:48:41 pm} 
 #                                last update: 12/20/00 {03:35:58 pm} 
 #  Description: 
 # 
 #  This file will be sourced automatically, immediately after the _first_
 #  time sasMode.tcl is sourced.  This file declare completions items and
 #  procedures for SAS mode.
 #  
 #  Author: Craig Barton Upright
 #  E-mail: <cupright@princeton.edu>
 #    mail: Princeton University, Department of Sociology
 #          Princeton, New Jersey 08544
 #     www: <http://www.princeton.edu/~cupright>
 #  
 # Copyright (c) 2000  Craig Barton Upright
 # 
 # This program is free software; you can redistribute it and/or modify
 # it under the terms of the GNU General Public License as published by
 # the Free Software Foundation; either version 2 of the License, or
 # (at your option) any later version.
 # 
 # This program is distributed in the hope that it will be useful,
 # but WITHOUT ANY WARRANTY; without even the implied warranty of
 # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 # GNU General Public License for more details.
 # 
 # You should have received a copy of the GNU General Public License
 # along with this program; if not, write to the Free Software
 # Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 # 
 # ==========================================================================
 ##

proc SASCompletions.tcl {} {}

# Setting the order of precedence for completions.

set completions(SAS) {
    completion::cmd Command completion::electric completion::word
}

#=============================================================================
# 
# SAS::Completion::Command
# 
# (1)  The lists of commands has already been created.
# (2)  Check to see if the command is preceded by a <p'> or <P'>, which 
#      indicates that the command is preceded by "proc " or "PROC ".
# (3)  Complete the insertion as defined by the variables 
#      SASCommandElectrics($lastword)
# 


proc SAS::Completion::Command {} {
    
    global SASbothCaseCommands SASCommandElectrics SASSyntaxMessage
    
    set lastword [completion::lastWord where]
    if {[lsearch -exact $SASbothCaseCommands $lastword] == -1} {
        return 0
    }
    set oneBack  [pos::math $where - 1]
    set twoBack  [pos::math $where - 2]
    
    set oneBackChar [lookAt $oneBack]
    set twoBackChar [lookAt $twoBack]
    
    # Do we have a defined completion?
    if {[info exists SASCommandElectrics($lastword)]} {
        set complete $SASCommandElectrics($lastword)
    } else {
        set complete " "
    } 
    append complete  " ;\r"
    # Do we have a message to put in the status bar?
    if {[info exists SASSyntaxMessage($lastword)]} {
        set sm $SASSyntaxMessage($lastword)
    } else {
        set sm ""
    } 
    # Now create the electric insertion.
    if {($twoBackChar == "p" || $twoBackChar == "P") && $oneBackChar== "'"} {
        # Is this a p'<command> Or a P'<COMMAND> contraction?
        if {$twoBackChar == "P"} {
            set p "PROC"
        } else {
            set p "proc"
        } 
        deleteText $twoBack [getPos]
        set commandInsertion "$p $lastword $complete"
    } else {
        # No, so just insert defined completion.
        set commandInsertion " $complete"
    }
    elec::Insertion $commandInsertion
    # Putting a message in the status bar with syntax information.
    message "$sm"
    return 1
}


#=============================================================================
# 
#  SAS Command Electrics   #
# 
# -- further completions beyond the above proc.
# 
# These are distinguished from "SASelectrics" because we want them to take
# place after the SAS::Completions::Command takes place, not before.
# 
# Each completion will already have $lastword and a space, "$lastword ", and
# be followed with " ;\r"
# 

#    Specific completions  #

# This list could easily be expanded by somebody that used the program more
# than twice a year.  The SPSSCompletions.tcl and the SttaCompletions.tcl
# files have more complicated command-specific examples.  Feel free to
# contribute more SAS completions!
# 

array set SASCommandElectrics {

    filename         " \"\""
    infile           " lrecl =  "
    libname          " "
    options          " = "
    proc             ""
    run              ""
}


#    ... and then ALL CAP completions  #

foreach lowerCaseCommand [array names SASCommandElectrics] {
    set upperCaseCommmand   [string toupper $lowerCaseCommand]
    set upperCaseCompletion [string toupper $SASCommandElectrics($lowerCaseCommand)]
    regsub -all {\\R} $upperCaseCompletion {\\r} upperCaseCompletion
    regsub -all {\\T} $upperCaseCompletion {\\t} upperCaseCompletion
    set SASCommandElectrics($upperCaseCommmand)  "$upperCaseCompletion" 
}

catch {unset upperCaseCommmand}
catch {unset upperCaseCompletion}

#=============================================================================
# 
#    Syntax messages  #
# 
# Make sure that [,],{,},#, and " have preceding backslashes.
# 
# As of this writing I don't when I'll have time to pursue this project.
# 
# The following examples demonstrate how one could include syntax messages,
# which would appear in the status bar during completions and via
# command-control-double-click.  The syntax can be obtained from the manual 
# pages of SAS, by entering 
# 
# help <command> ;
# 
# or from an OnlineDoc web site.
# 

array set SASSyntaxMessage {

    aceclus     "proc aceclus proportion=p | threshold=t < options > ;"
    model       "model dependents=independents < / options > ;"
}

#    ... and then ALL CAP syntax messages  #

foreach lowerCaseCommand [array names SASSyntaxMessage] {
    set upperCaseCommmand      [string toupper $lowerCaseCommand]
    set upperCaseSyntaxMessage [string toupper $SASSyntaxMessage($lowerCaseCommand)]
    set SASSyntaxMessage($upperCaseCommmand)  "$upperCaseSyntaxMessage"
}

catch {unset upperCaseCommmand}
catch {unset upperCaseSyntaxMessage}

#=============================================================================
# 
# .
