## -*-Tcl-*- (nowrap)
 # ==========================================================================
 #  Statistical Modes - an extension package for Alpha
 # 
 #  FILE: "SPSSCompletions.tcl"
 #                                    created: 05/14/00 {01:48:41 pm} 
 #                                last update: 12/20/00 {03:59:34 pm} 
 #  Description: 
 # 
 #  This file will be sourced automatically, immediately after the _first_
 #  time SPSSMode.tcl is sourced.  This file declare completions items and
 #  procedures for SPSS mode.
 #  
 #  Author: Craig Barton Upright
 #  E-mail: <cupright@princeton.edu>
 #    mail: Princeton University, Department of Sociology
 #          Princeton, New Jersey 08544
 #     www: <http://www.princeton.edu/~cupright>
 #  
 # Copyright (c) 2000  Craig Barton Upright
 # 
 # This program is free software; you can redistribute it and/or modify
 # it under the terms of the GNU General Public License as published by
 # the Free Software Foundation; either version 2 of the License, or
 # (at your option) any later version.
 # 
 # This program is distributed in the hope that it will be useful,
 # but WITHOUT ANY WARRANTY; without even the implied warranty of
 # MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 # GNU General Public License for more details.
 # 
 # You should have received a copy of the GNU General Public License
 # along with this program; if not, write to the Free Software
 # Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 # 
 # ==========================================================================
 ##

proc SPSSCompletions.tcl {} {}

# Setting the order of precedence for completions.

set completions(SPSS) {
    contraction completion::cmd Command completion::electric completion::word
}

#=============================================================================
# 
# SPSS::Completion::Command
# 
# (1)  The lists of commands has already been created.
# (2)  Check to see if the command is preceded by a <'> or </>, which indicates 
#      that the user wants an argument and not a command.
# (3)  If so, then leave it at the word completion.
# (4)  Othewise, complete the insertion as defined by the variables 
#      SPSScommandElectrics($lastword)
# 

proc SPSS::Completion::Command {} {

    global SPSScmds SPSSCommandElectrics SPSSSyntaxMessage

    set lastword [completion::lastWord where]
    if {[lsearch -exact $SPSScmds $lastword] == -1} {
        return 0
    }
    set oneBack     [pos::math $where - 1]
    set oneBackChar [lookAt $oneBack]

    # Do we have a defined completion?
    if {[info exists SPSSCommandElectrics($lastword)]} {
        set complete $SPSSCommandElectrics($lastword)
    } else {
        set complete " .\r"
    } 
    # Do we have a message to put in the status bar?
    if {[info exists SPSSSyntaxMessage($lastword)]} {
        set sm $SPSSSyntaxMessage($lastword)
    } else {
        set sm ""
    } 
    # Now create the electric insertion.
    if {$oneBackChar == "'"} {
        # Is this a <'><keyword> completion?
        # Then insert as an argument, not as a command.
        deleteText $oneBack [getPos]
        set commandInsertion "$lastword "
        set sm ""
    } elseif {$oneBackChar == "/"} {
        # Is this a </><keyword> completion?
        # Then insert as an argument, not as a command.
        set commandInsertion " "
        set sm ""
    } else {
        # No, such just insert defined completion
        set commandInsertion " $complete"
    }    
    elec::Insertion $commandInsertion
    
    # Putting a message in the status bar with syntax information
    message "$sm"
    return 1
}


#=============================================================================
# 
#  SPSS Command Electrics   #
# 
# -- further completions beyond the above proc.
# 
# These are distinguished from "SPSSelectrics" because we want them to take
# place after the SPSS::Completions::Command takes place, not before.
# 
# Each completion will already have $lastword and a space, "$lastword "
# 

#    Specific completions, multi  #
# 
# Multiple word commands.
# 

# "add" can be either "files" or "value labels"
set SPSSCommandElectrics(add)           ""
set SPSSCommandElectrics(begin)         "data"
set SPSSCommandElectrics(clear)         "transformations .\r"
set SPSSCommandElectrics(drop)          "documents .\r"
# "do" can be either "if" or "repeat"
set SPSSCommandElectrics(do)            ""
set SPSSCommandElectrics(data)          "list"
# "end" can be either "if" or "file type"
set SPSSCommandElectrics(end)           ""
# "file" can be "handle", "label" or "type"
set SPSSCommandElectrics(file)          ""
# "get" can be "bmdp", "osiris", "sas", or "translate"
set SPSSCommandElectrics(get)           ""
set SPSSCommandElectrics(input)         "program"
set SPSSCommandElectrics(keyed)         "data"
set SPSSCommandElectrics(logistic)      "regression"
set SPSSCommandElectrics(match)         "files"
set SPSSCommandElectrics(mult)          "response"
set SPSSCommandElectrics(missing        "values"
set SPSSCommandElectrics(n)             "of cases"
set SPSSCommandElectrics(nonpar)        "corr"
set SPSSCommandElectrics(npar)          "tests"
set SPSSCommandElectrics(partial)       "corr"
# "print" can be "eject", "formats" or "space"
set SPSSCommandElectrics(print)         ""
set SPSSCommandElectrics(procedure)     "output"
set SPSSCommandElectrics(quick)         "cluster"
set SPSSCommandElectrics(repeating)     "data"
set SPSSCommandElectrics(record)        "type"
set SPSSCommandElectrics(rename)        "variables"
set SPSSCommandElectrics(sort)          "cases"
set SPSSCommandElectrics(split)         "file"
set SPSSCommandElectrics(select)        "if"
# "save" can be either "scss" or "translate", or nothing
set SPSSCommandElectrics(save)          ""
set SPSSCommandElectrics(variable)      "labels"
set SPSSCommandElectrics(value)         "labels"
set SPSSCommandElectrics(write)         "formats"

#    Specific completions, single  #
# 
# One word (and the tails of mult-word) commands.
# 

array set SPSSCommandElectrics {
    
    aggregate       "outfile = \"\"\r\t/break =  .\r"
    alscal          " =  .\r"
    anova           "variables =  by (,) .\r"
    autorecode      "variables = \r\tinto  .\r"
    bmdp            " .\r"
    break           ".\r"
    cluster         " .\r"
    compute         " =  .\r"
    correlations    "variables =  .\r"
    count           " =  () .\r"
    crosstabs       "tables =  by  .\r"
    decriptives     "variables =  .\r"
    discriminant    "groups = (,)\r\tvariables =  .\r"
    documents       " .\r"
    edit            ".\r"
    examine         "variables =  .\r"
    execute         ".\r"
    export          "outfile = \"\" .\r"
    factor          "variables =  .\r"
    files           "file = \"\" .\r"
    flip            "variables =  .\r"
    formats         " () .\r"
    frequencies     "variables =  .\r"
    graph           ""
    handle          " /  .\r"
    if              "() .\r"
    import          "file = \"\"\ .\r"
    labels          " \"\" .\r"
    list            "variables =  .\r"
    loglinear       " (,)  .\r"
    osiris          "dictionary =  .\r"
    recode          " ( = ) .\r"
    regression      "    .\r"
    sas             "data =  .\r"
    scss            "masterfile = \"\" .\r"
    translate       "file = \"\" .\r"
    value           "labels  \"\"\"\" .\r"
    variable        "labels  \"\" .\r"
}

#    Specific "argument" completions  #

array set SPSSCommandElectrics {
    
    case            ".\r"
    file            "= \"\" .\r"
    label           " "
    outfile         "= \"\""

}


#    ... and then ALL CAP completions  #

foreach lowerCaseCommand [array names SPSSCommandElectrics] {
    set upperCaseCommmand   [string toupper $lowerCaseCommand]
    set upperCaseCompletion [string toupper $SPSSCommandElectrics($lowerCaseCommand)]
    regsub {\\R} $upperCaseCompletion {\\r} upperCaseCompletion
    regsub {\\T} $upperCaseCompletion {\\t} upperCaseCompletion
    set SPSSCommandElectrics($upperCaseCommmand)  "$upperCaseCompletion" 
}

catch {unset upperCaseCommmand}
catch {unset upperCaseCompletion}

#=============================================================================
# 
#  Function Completions  #
# 


foreach genericCompletion $SPSSlowerCaseFunctions {
    set SPSSelectrics($genericCompletion) "()"
}

#=============================================================================
# 
#  Contractions  #
# 
# These are for two and three word commands.  Hitting the completion key
# again, after these have been completed, will invoke further completions.
# 

array set SPSSelectrics {
    
    a'f                    "kill0add files"
    a'vl                   "kill0add value labels"
    b'd                    "kill0begin data"
    c't                    "kill0clear transformations"
    d'd                    "kill0drop documents"
    d'i                    "kill0do if"
    d'l                    "kill0data list"
    d'r                    "kill0do repeat"
    e'ft                   "kill0end file type"
    e'r                    "kill0end repeat"
    f'h                    "kill0file handle"
    f'l                    "kill0file label"
    f't                    "kill0file type"
    g'b                    "kill0get bmdp"
    g'o                    "kill0get osiris"
    g's                    "kill0get sas"
    g'sc                   "kill0get scss"
    g't                    "kill0get translate"
    i'p                    "kill0input program"
    k'dl                   "kill0keyed data"
    l'r                    "kill0logistic regression"
    m'f                    "kill0match files"
    m'r                    "kill0mult response"
    m'v                    "kill0missing values"
    n'oc                   "kill0n of cases"
    n'c                    "kill0nonpar corr"
    n't                    "kill0npar tests"
    p'c                    "kill0partial corr"
    p'e                    "kill0print eject"
    p'f                    "kill0print formats"
    p'o                    "kill0procedure output"
    p's                    "kill0print space"
    q'c                    "kill0quick cluster"
    r'd                    "kill0repeating data"
    r't                    "kill0record type"
    r'v                    "kill0rename variables"
    s'c                    "kill0sort cases"
    s'f                    "kill0split file"
    s'i                    "kill0select if"
    s's                    "kill0save sas"
    s't                    "kill0save translate"
    vr'l                   "kill0variable labels"
    vl'l                   "kill0value labels"
    w'f                    "kill0write formats"
}

# And now upper case electrics.

foreach lowerCaseCompletion [array names SPSSelectrics] {
    set upperCaseCommmand   [string toupper $lowerCaseCompletion]
    set upperCaseCompletion [string toupper $SPSSelectrics($lowerCaseCompletion)]
    regsub -all {\\R}   $upperCaseCompletion {\\r}   upperCaseCompletion
    regsub -all {\\T}   $upperCaseCompletion {\\t}   upperCaseCompletion
    regsub -all {KILL} $upperCaseCompletion {kill} upperCaseCompletion
    set SPSSelectrics($upperCaseCommmand)  "$upperCaseCompletion"
}

catch {unset upperCaseCommmand}
catch {unset upperCaseCompletion}

#=============================================================================
# 
#    Syntax messages  #
# 
# Make sure that [,],{,},#, and " have preceding backslashes.
# 

# specific messages --

# As of this writing I don't when I'll have time to pursue this project.
# 
# The following examples demonstrate how one could include syntax messages,
# which would appear in the status bar during completions and via
# command-control-double-click.  The syntax can be obtained from the manual 
# pages of SPSS, by entering 
# 
# help <command>
# syntax
# 

array set SPSSSyntaxMessage {
    
    anova           "anova \[variables=\] varlist by varlist(min,max)...varlist(min,max) <options>"
    correlations    "correlations \[variables=\] varlist \[with varlist\] \[/varlist...\]  <options>"
}

#    ... and then ALL CAP syntax messages  #

foreach lowerCaseCommand [array names SPSSSyntaxMessage] {
    set upperCaseCommmand      [string toupper $lowerCaseCommand]
    set upperCaseSyntaxMessage [string toupper $SPSSSyntaxMessage($lowerCaseCommand)]
    set SPSSSyntaxMessage($upperCaseCommmand)  "$upperCaseSyntaxMessage"
}

catch {unset upperCaseCommmand}
catch {unset upperCaseCompletion}

#=============================================================================
# 
# .
