## -*-Tcl-*-
 # ###################################################################
 #  Vince's Additions - an extension package for Alpha
 # 
 #	FILE: "filesetsMenu.tcl"
 #					created: 20/7/96 {6:22:25 pm} 
 #				   last update: 11/02/2001 {15:33:06 PM} 
 #	Author:	Vince Darley
 #	E-mail:	<vince@santafe.edu>
 #	  mail:	317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #	   www:	<http://www.santafe.edu/~vince/>
 #	
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  24/3/96  VMD 1.0 update of Pete's original to allow mode-specific filesets
 #  27/3/96  VMD 1.1 added hierarchial filesets, and checks for unique menus
 #  13/6/96  VMD 1.2 memory efficiency improvements with 'fileSets' array
 #  10/3/97  VMD 1.3 added 'procedural' fsets, including 'Open Windows'
 #  6/4/97   VMD 1.31 various fixes incorporated - thanks!
 #  11/7/97  VMD 1.4 added cache for the fileset menu, improved wc proc.
 #  15/7/97  VMD 1.41 better handling of out-of-date filesets, and dir opening
 #  15/7/97  VMD 1.42 placed cache in separate file.
 #  21/7/97  VMD 1.43 added glob patterns to ignore for directory filesets
 #  22/7/97  VMD 1.5 more sophisticated menu caching.  No more long rebuilds!
 #  10/9/97  VMD 1.6 simplified some stuff for new Alpha-Tcl
 #  7/12/97  VMD 1.6.1 makes use of winNumDirty flag
 #  12/1/98  VMD 1.6.2 removes special treatment of *recent*
 #  15/1/1999  VMD 1.7.2 a year of improvements....
 #  2000-03-13 VMD 1.7.7 Removed default filesets, fixed for no filesets.
 #  
 #  Version 2.0 separates the filesets menu from the core idea of a 
 #  'fileset', which is used elsewhere in Alpha.  The code in this
 #  file is just concerned with the filesets menu, and simply makes
 #  use of the nice core filesets code.
 # ###################################################################
 ##

alpha::menu filesetMenu 2.0.5 global "131" {
    # We need this core code.
    alpha::package require filesets
    
    # A type can have the option of being unsorted (e.g. tex-filesets)
    newPref flag sortFilesetItems 0 "fileset"
    # Visual formatting may be of relevance to some types
    newPref flag indentFilesetItems 0 "fileset"
    # Use the variable 'filesetSortOrder' to determine the visual
    # structure of the fileset menu
    newPref flag sortFilesetsByType 0 "fileset" filesetMenu::rebuildSome
    # When a file is selected from the menu, do we try and keep
    # 'currFileSet' accurate?
    newPref flag autoAdjustFileset 1 "fileset"
    # Filesets may include non-text files.  Alpha will tell the finder
    # to open these if they are selected.
    newPref flag includeNonTextFiles 0 "fileset" filesetMenu::rebuildSome

    # The filesets not to show in the menu
    ensureset filesetsNotInMenu "Help"
    ensureset fsMenuDeletions [list]
    
    namespace eval filesetMenu {}
    set filesetMenu::haveSubMenus 0
    # This will autoload this file and then run this procedure
    filesetMenu::build
    # Make sure any deleted items in the main menu are removed at
    # startup.  We have to do this with a hook, because in Alpha
    # (but not Alphatk), 'deleteMenuItem' only works for menu items
    # which have already been inserted into the menu bar.
    hook::register startupHook [join $fsMenuDeletions "\n"]
} {
    # insert just before 'open remote' item
    menu::insert File items "<B<O/OopenRemote" "<I<O/OopenViaFileset"
    hook::register fileset-delete filesetMenu::fsetDeleted *
    hook::register fileset-update filesetMenu::fsetUpdated *
    hook::register fileset-new filesetMenu::fsetNew *
    hook::register fileset-current filesetMenu::changeCurrent *
    hook::register fileset-uncache filesetMenu::fsetUncache *
} {
    menu::uninsert File items "<B<O/OopenRemote" "<I<O/OopenViaFileset"
    hook::deregister fileset-delete filesetMenu::fsetDeleted *
    hook::deregister fileset-update filesetMenu::fsetUpdated *
    hook::deregister fileset-new filesetMenu::fsetNew *
    hook::deregister fileset-current filesetMenu::changeCurrent *
    hook::deregister fileset-uncache filesetMenu::fsetUncache *
} uninstall {this-file} help {file "Filesets Help"} maintainer {
    "Vince Darley" vince@santafe.edu <http://www.santafe.edu/~vince/>
}

proc filesetMenu {} {}

namespace eval filesetMenu {}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"rebuildAllFilesets" --
 #	
 # This does a complete rebuild of all information.  
 # 
 # If 'use_cache' is 1, we try to use caches of information where 
 # possible, so all filesets may not actually be rebuilt.
 # 
 # Note: this procedure cannot just be a simple loop over
 # 'UpdateAFileset', because we need to rebuild the actual
 # menu, and this may have an unknown structure.
 # 
 # At one point I wrote: The problem is that the names of menus may
 # actually change (spaces added/deleted).  This is not a problem for
 # the fileset menu, but is a problem for any filesets which have been
 # added to other menus, since they won't know that they need to be
 # rebuilt.
 # -------------------------------------------------------------------------
 ##
proc rebuildAllFilesets { {use_cache 0} } {
    global filesetMenu filesetmodeVars filesetsNotInMenu \
      fileSets filesetMenu::haveSubMenus fsMenuCache
    
    if {!$use_cache} {
	foreach f [lsort [fileset::names]] {
	    fileset::uncache $f
	}
    }
    message "Rebuilding filesets menu"
    
    set filesetMenu::haveSubMenus 0
    set problems {}
    if {$filesetmodeVars(sortFilesetsByType)} {
	global filesetSortOrder
	# just make file-sets for those we don't want in the menu
	foreach f $filesetsNotInMenu {
	    fileset::make $f 0
	}
	set used $filesetsNotInMenu
	set sp [filesetMenu::sortedOrder $filesetSortOrder used]
	set sets [lindex $sp 0]
	set problems [lindex $sp 1]
    } else {
	set sets {}
	foreach f [lsort [fileset::names]] {
	    set doMenu [expr {![lcontains filesetsNotInMenu $f]}]
	    set menu [fileset::make $f 1]
	    if {$doMenu} {
		if {[llength $menu]} {
		    lappend sets $menu
		} else {
		    lappend problems $f
		}
	    }
	}			
    }
    
    if {[llength $problems]} {
	foreach fset $problems {
	    filesetMenu::_hideOrShow $fset
	}
	message "The following filesets had problems and will be\
	  hidden: $problems"
    }

    # cache the fileset menu
    set m [list Menu -m -n $filesetMenu -p filesetMenu::menuProc \
      [concat {{/'Edit File} {Menu -n Utilities {}}} "Help" \
      "\(-" $sets]]
    
    cache::create fsMenu2.0 
    cache::add fsMenu2.0 "eval" $m 
    global fsListOfSubmenus fsSubMenuInfo fsMenuDeletions
    set fsMenuDeletions [list]
    prefs::modified fsListOfSubmenus fsSubMenuInfo fileSets fsMenuCache \
      fsMenuDeletions
    eval $m
    
    filesetMenu::rebuildUtils
    
    message ""
}


## 
 # -------------------------------------------------------------------------
 # 
 # "filesetMenu::fsetNew" --
 # 
 #  If we've added, or renamed a fileset.  In most cases we must
 #  rebuild everything (due to limitations in Alpha), but for
 #  'procedural' filesets, we can just do the utilities menu.
 # -------------------------------------------------------------------------
 ##
proc filesetMenu::fsetNew {name} {
    if {[fileset::getKindFromFset $name] == "procedural"} {
	global filesetsNotInMenu
	if {[lsearch $filesetsNotInMenu $name] == -1} {
	    lappend filesetsNotInMenu $name
	    prefs::modified filesetsNotInMenu
	}
	filesetMenu::rebuildUtils
    } else {
	rebuildAllFilesets 1
	message "The fileset \"$name\" has been added to the main fileset menu."
    }
}

proc filesetMenu::changeCurrent {from to} {
    # These may fail if one or other is a temporary fileset
    markMenuItem -m choose $from off
    markMenuItem -m choose $to on
}

proc filesetMenu::fsetUncache {fset} {
    global fileSets fsListOfSubmenus fsSubMenuInfo

    if {[info exists fsListOfSubmenus($fset)]} {
	# if the fileset already has a base menu, use that:
	foreach n $fsListOfSubmenus($fset) {
	    catch [list unset fsSubMenuInfo($n)]
	    prefs::modified fsSubMenuInfo($n)
	}
	catch [list unset fsListOfSubmenus($fset)]
	prefs::modified fsListOfSubmenus($fset)
    }
    if {[info exists fileSets($fset)]} {
	catch [list unset fileSets($fset)]
	prefs::modified fileSets($fset)
    }
}

proc filesetMenu::fsetUpdated {fset {m ""}} {
    if {![llength $m]} {
	return
    }

    global fileSets fsListOfSubmenus fsSubMenuInfo

    # we could rebuild the menu with this: but we don't
    cache::add fsMenu2.0 "eval" $m
    if {[info exists fsListOfSubmenus($fset)]} {
	# if the fileset already has a base menu, use that:
	foreach n $fsListOfSubmenus($fset) {
	    prefs::modified fsSubMenuInfo($n)
	}
	prefs::modified fsListOfSubmenus($fset)
    }
    if {[info exists fileSets($fset)]} {
	prefs::modified fileSets($fset)
    }
    eval $m
}

proc filesetMenu::fsetDeleted {fset} {
    global filesetsNotInMenu
    
    set err [catch {filesetMenu::remove $fset}]
    
    if {[set l [lsearch -exact $filesetsNotInMenu $fset]] != -1} {
	set filesetsNotInMenu [lreplace $filesetsNotInMenu $l $l]
	prefs::modified filesetsNotInMenu
	deleteMenuItem -m choose $fset
	deleteMenuItem -m hideFileset $fset
	return
    }
    if {$err} {
	# it's on a submenu or somewhere else so we just have
	# to do the lot!
	global errorInfo
	rebuildAllFilesets 1
    } else {
	deleteMenuItem -m choose $fset
	deleteMenuItem -m hideFileset $fset
    }

}


#  Menu procedures  #

## 
 # -------------------------------------------------------------------------
 #	 
 #	"filesetSortOrder" --
 #	
 #  The structure of this variable dictates how the fileset menu is
 #  structured:
 #		   
 #		   '{pattern p}' 
 #			   lists all filesets which match 'p'
 #		   '-' 
 #			   adds	a separator line
 #		   '{list of types}' 
 #			   lists all filesets of those types.
 #		   '{submenu name sub-order-list}' 
 #			   adds	a submenu with name 'name' and recursively
 #			   adds	filesets to that submenu as given by the 
 #			   sub-order.
 #			   
 #  Leading, trailing and double separators are automatically removed.
 #	 
 # -------------------------------------------------------------------------
 ##
ensureset filesetSortOrder { {pattern *Core} {pattern Packages} \
	{pattern Menus} {pattern Modes} {pattern Preferences} \
	- {tex} - {pattern *.cc} {submenu Headers {pattern *.h}} \
	- {fromDirectory think codewarrior ftp \
	fromOpenWindows fromHierarchy} * } 

proc filesetMenu::remove {fset} {
    global fsListOfSubmenus fsSubMenuInfo filesetMenu fsMenuDeletions
    # find its menu:
    if {[info exists fsListOfSubmenus($fset)]} {
	foreach m $fsListOfSubmenus($fset) {
	    # remove info about it's name
	    if {[info exists fsSubMenuInfo($m)]} {
		unset fsSubMenuInfo($m)
		prefs::modified fsSubMenuInfo($m)
	    }
	}
	set base [lindex $fsListOfSubmenus($fset) 0]
	unset fsListOfSubmenus($fset)
	prefs::modified fsListOfSubmenus($fset)
	#cache::snippetRemove $fset _fsmenu_
	# this will fail if it's on a submenu or if it isn't a menu at all
	deleteMenuItem -m $filesetMenu $base
	lappend fsMenuDeletions [list deleteMenuItem -m $filesetMenu $base]
	prefs::modified fsMenuDeletions
    } else {
	# I think I do nothing
    }
    
}

proc filesetMenu::removeInMenu {fset} {
    global fsListOfSubmenus fsSubMenuInfo filesetMenu fsMenuDeletions
    # find its menu:
    if {[info exists fsListOfSubmenus($fset)]} {
	set base [lindex $fsListOfSubmenus($fset) 0]
	# this will fail if it's on a submenu or if it isn't a menu at all
	markMenuItem -m hideFileset $fset on
	deleteMenuItem -m $filesetMenu $base
	lappend fsMenuDeletions [list deleteMenuItem -m $filesetMenu $base]
	prefs::modified fsMenuDeletions
    } else {
	# I think I do nothing
    }
}

## 
 # Global procedures to deal with the fact that Alpha can only have one
 # menu with each given name.  This is only a problem in dealing with
 # user-defined menus such as fileset menus, tex-package menus, ...
 ##

## 
 # -------------------------------------------------------------------------
 #	 
 #	"filesetMenu::makeSub" --
 #	
 # If desired this is the only procedure you need use --- it returns a menu
 # creation string, taking account of the unique name requirement and will
 # make sure your procedure 'proc' is called with the real menu name! 
 # -------------------------------------------------------------------------
 ##
proc filesetMenu::makeSub {fset name proc args} {
    if {[string length $proc] > 1 } {
	return [concat {Menu -m -n} \
	  [list [filesetMenu::registerName $fset $name $proc]] \
	  -p filesetMenu::subProc $args]
    } else {
	return [concat {Menu -m -n} \
	  [list [filesetMenu::registerName $fset $name]] $args]
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"filesetMenu::registerName" --
 #	
 # Call to ensure unique fileset submenu names.  We just add spaces as
 # appropriate and keep track of everything for you!  Filesets which have
 # multiple menus _must_ register the main menu first. 
 # -------------------------------------------------------------------------
 ##
proc filesetMenu::registerName {fset name {proc ""} {top 1}} {
    global fsSubMenuInfo fsListOfSubmenus
    if {$top && ($fset == $name) && [info exists fsListOfSubmenus($fset)] } {
	# if the fileset already has a base menu, use that,
	# but only if this is the base level of the fileset menu
	# (otherwise we have trouble if submenus contain the
	# directories with the same name as the top menu).
	foreach n $fsListOfSubmenus($fset) {
	    if { [string trimright $n] == $fset } {
		set base $n
	    } 
	    unset fsSubMenuInfo($n)
	}
	unset fsListOfSubmenus($fset)
    }
    set original $name					
    if {[info exists base]} {
	set name $base
    } else {
	# I add at least one space to _all_ hierarchical submenus now.
	# This is so I won't clash with any current or future modes
	# which should never normally add spaces themselves.
	append name " "
	while {[info exists fsSubMenuInfo($name)]} {
	    append name " "
	}		
    }
    
    if {[info tclversion] < 8.0 && ([string length $name] > 31)} {
	# Menu names can only be 31 characters long
	set name "[string trimright $name]"
	while {[string length $name] > 31} {
	    regsub -- ".\$" $name  name
	}
    }
    
    set fsSubMenuInfo($name) [list $fset $original $proc]
    # build list of a fileset's menus
    lappend fsListOfSubmenus($fset) "$name"
    
    return $name
}

## 
 # -------------------------------------------------------------------------
 #	 
 #	"filesetMenu::subProc" --
 #	
 # This procedure is implicitly used to deal with ensuring unique sub-menu
 # names.  It calls the procedure you asked for, with the name of the menu
 # you think you're using. 
 # -------------------------------------------------------------------------
 ##
proc filesetMenu::subProc {menu item} {
    global fsSubMenuInfo
    set l $fsSubMenuInfo($menu)
    set realProc [lindex $l 2]
    if {[info commands $realProc] == ""} {auto_load "$realProc"}
    # try to call the proc with three arguments (fileset is 1st)
    if {[llength [info args $realProc]] == 2} {
	$realProc [lindex $l 1] "$item"
    } else {
	$realProc [lindex $l 0] [lindex $l 1] "$item"
    }
}


proc filesetMenu::menuProc {menu item} {
    switch -- $item {
	"Edit File" {
	    file::openViaFileset
	} 
	"Help" {
	    help::openFile "Filesets Help"
	}
    }
}

proc filesetMenu::utilsProc { menu item } {
    global filesetUtils currFileSet
    if {![info exists filesetUtils($item)] \
      && [info exists filesetUtils(${item})]} {
	append item ""
    }
    if {[info exists filesetUtils($item)]} {
	# it's a utility
	set utilDesc $filesetUtils($item)
	set allowedTypes [lindex $utilDesc 0]
	if {[string match $allowedTypes [fileset::type $currFileSet]]} {
	    return [eval [lindex $utilDesc 1]]
	} else {
	    beep
	    message "That utility can't be applied to the current file-set."
	    return
	}
    } else {
	$item
    }
}

proc filesetMenu::sortedOrder {order usedvar} {
    upvar $usedvar used
    global filesetmodeVars filesetsNotInMenu
    set sets {}
    set problems {}
    
    foreach item $order {
	switch -- [lindex $item 0] {
	    "-" { 
		# add divider
		lappend sets "\(-" 
		continue
	    } 
	    "*" {
		# add all the rest
		set subset {}
		foreach s [fileset::names] {
		    if {![lcontains used $s]}  {
			lappend subset $s
			lappend used $s
		    }
		}
		foreach f [lsort $subset] {
		    set fmenu [fileset::make $f 1]
		    if {[llength $fmenu]} {
			lappend sets $fmenu
		    } else {
			lappend problems $f
			lappend filesetsNotInMenu $f
		    }
		}
	    } 
	    "pattern" {
		# find all which match a given pattern
		set patt [lindex $item 1]
		set subset {}
		foreach s [fileset::names] {
		    if {![lcontains used $s]}  {
			if {[string match $patt $s]} {
			    lappend subset $s
			    lappend used $s
			}
		    }
		}
		foreach f [lsort $subset] {
		    set fmenu [fileset::make $f 1]
		    if {[llength $fmenu]} {
			lappend sets $fmenu
		    } else {
			lappend problems $f
			lappend filesetsNotInMenu $f
		    }
		}
		
	    }
	    "submenu" {
		global filesetMenu::haveSubMenus
		set filesetMenu::haveSubMenus 1
		# add a submenu with name following and sub-order
		set name [lindex $item 1]
		set suborder [lrange $item 2 end]		  	
		# we make kind of a pretend fileset here.
		set sp [filesetMenu::sortedOrder $suborder used]
		set subsets [lindex $sp 0]
		set problems [lindex $sp 1]
		if { $subsets != "" } {
		    lappend sets [filesetMenu::makeSub $name $name \
		      fileset::openItemProc $subsets]
		}
	    }
	    "default" {		
		set subset {} 
		foreach s [fileset::names] {
		    if {[lcontains item [fileset::type $s]] \
		      && ![lcontains used $s]}  {
			lappend subset $s
			lappend used $s
		    }
		}
		foreach f [lsort $subset] {
		    set fmenu [fileset::make $f 1]
		    if {[llength $fmenu]} {
			lappend sets $fmenu
		    } else {
			lappend problems $f
			lappend filesetsNotInMenu $f
		    }
		}
	    }
	}
	
    }
    # remove multiple and leading, trailing '-' in case there were gaps
    regsub -all "\\\(-\( \\\(-\)+" $sets "\(-" sets
    while { [lindex $sets 0] == "\(-" } { set sets [lrange $sets 1 end] }
    set l [expr {[llength $sets] -1}]
    if { [lindex $sets $l] == "\(-" } { set sets [lrange $sets 0 [incr l -1]] }
    
    return [list $sets $problems]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "filesetMenu::build" --
 # 
 #  This is the procedure called at each startup.  It used to be called
 #  'rebuildFilesetMenu', but that is misleading, hence the current
 #  name.  Its job is to build the menu, not worrying about whether
 #  the information it uses is current/valid/cached or whatever.
 #  It is the job of 'rebuild/update filesets' to find new information
 #  if desired.
 #  
 #  Reads the fileset menu from the cache if it exists.  This speeds up
 #  start-up by quite a bit.
 # -------------------------------------------------------------------------
 ##
proc filesetMenu::build {} { 
    message "Building filesets"
    if {[cache::exists fsMenu2.0]} {
	uplevel \#0 cache::readContents fsMenu2.0
	filesetMenu::rebuildUtils
    } else {
	rebuildAllFilesets 1
    }
    
}
	
## 
 # -------------------------------------------------------------------------
 #	 
 #	"filesetMenu::rebuildSome" --
 #	
 # If given '*' rebuild the entire menu, else rebuild only those types
 # given.  This is generally useful to avoid excessive rebuilding when
 # flags are adjusted
 # -------------------------------------------------------------------------
 ##
proc filesetMenu::rebuildSome {args} {
    rebuildAllFilesets		
}

proc filesetMenu::rebuildUtils {} {
    global filesetUtils 
    
    set itemList [list]
    foreach fset [lsort -ignore [fileset::names]] {
	if {[fileset::getKindFromFset $fset] == "procedural"} {continue}
	lappend itemList $fset
    }

    Menu -n "Utilities" -p filesetMenu::utilsProc [concat \
      "editFilesets" \
      "editAFileset" \
      "newFileset" \
      "deleteFileset" \
      "printFileset" \
      "<S<EupdateAFileset" \
      "<SupdateCurrentFileset" \
      "rebuildAllFilesets" \
      [list [menu::makeFlagMenu choose list currFileSet]] \
      [list [list Menu -n hideFileset -m -p filesetMenu::hideOrShow $itemList]]\
      [list [menu::makeFlagMenu filesetFlags array filesetmodeVars]] \
      "\(-" \
      "/T<I<OfindTag" \
      "createTagFile" \
      "\(-" \
      [lsort [array names filesetUtils]] \
      ]

    filesetMenu::utilsMarksTicks
}

proc filesetMenu::hideOrShow {menu item} {
    global filesetsNotInMenu
    
    # Workaround removal of ellipsis from menu items.
    if {![fileset::exists $item]} {
	if {[fileset::exists "${item}"]} {
	    append item ""
	}
    }
    
    if {[catch {filesetMenu::_hideOrShow $item} ret]} {
	# error
	alertnote $ret
	return
    }
    
    if {$ret} {
	rebuildAllFilesets 1
    }
    
    # Trying to unhide something may actually fail, so
    # we always check the real list.
    if {[lcontains filesetsNotInMenu $item]} {
	message "The fileset \"$item\" is now hidden."
    } else {
	message "The fileset \"$item\" is now in the main fileset menu."
    }
}

proc filesetMenu::_hideOrShow {fset} {
    global filesetsNotInMenu
    
    if {![fileset::exists $fset]} {
	return -code error "No such fileset \"$fset\""
    }
    set rebuild 0
    
    if {[lcontains filesetsNotInMenu $fset]} {
	if {[fileset::getKindFromFset $fset] == "procedural"} {
	    return -code error "Sorry, '$fset' is a 'procedural'\
	      fileset, and those filesets are completely dynamic\
	      and cannot appear in menus."
	}
	set idx [lsearch $filesetsNotInMenu $fset]
	set filesetsNotInMenu [lreplace $filesetsNotInMenu $idx $idx]		
	markMenuItem -m hideFileset $fset off
	# would be better if we could just insert it
	set rebuild 1
    } else {
	lappend filesetsNotInMenu $fset
	markMenuItem -m hideFileset $fset on
	if {[catch {filesetMenu::removeInMenu $fset}]} {
	    set rebuild 1
	}
    }
    prefs::modified filesetsNotInMenu
    return $rebuild
}

proc filesetMenu::ensureNoProceduralFilesets {} {
    global filesetsNotInMenu
    set notIn {}
    foreach s $filesetsNotInMenu {
	if {[fileset::exists $s]} {
	    lappend notIn $s
	}
    }
    if {[llength $notIn] != [llength $filesetsNotInMenu]} {
	set filesetsNotInMenu $notIn
	prefs::modified filesetsNotInMenu
    }
    foreach s [fileset::names] {
	if {[fileset::getKindFromFset $s] == "procedural"} {
	    if {[lsearch -exact $filesetsNotInMenu $s] == -1} {
		lappend filesetsNotInMenu $s
	    }
	}
    }
}

proc filesetMenu::utilsMarksTicks {} {
    global filesetsNotInMenu
    foreach name $filesetsNotInMenu {
	if {[fileset::getKindFromFset $name] == "procedural"} {continue}
	markMenuItem -m hideFileset $name on
    }
}

filesetMenu::ensureNoProceduralFilesets
