# File : "macMenuFiles.tcl"
#                        Created : 2001-01-22 21:35:09
#              Last modification : 2001-09-07 08:54:36
# Author : Bernard Desgraupes
# e-mail : <berdesg@easynet.fr>
# www : <http://perso.easynet.fr/~berdesg/>
# 
# (c) Copyright : Bernard Desgraupes, 2001
#         All rights reserved.
# This software is free software. See licensing terms in the MacMenu Help file.
#
# Description : this file is part of the macMenu package for Alpha.
# It contains the procedures related to files management with macMenu.
# 

namespace eval mac {}


# # # Managing files # # #
# ========================

# Select
# ------
proc mac::selectFiles {} {
    if ![mac::selectDialog Select] return
    mac::selectProc
}

proc mac::selectProc {} {
    mac::buildFiles&Folders
    mac::setselect 1
    mac::finderToFront
    return
}

proc mac::unselectFiles {} {
    if {![mac::selectDialog Unselect]} {return} 
    mac::unselectProc
}

proc mac::unselectProc {} {
    mac::buildFiles&Folders
    mac::setselect 0
    mac::finderToFront
    return
}

proc mac::setselect {toggle} {
    global mac::folderslist mac_params mac_contents mac::objlist
    # Process in top folder
    if {[info exists mac_contents($mac_params(srcfold))]} {
	# Open the top folder's finder window
	catch {AEBuild -r 'MACS' aevt odoc ---- [tclAE::build::propertyObject cwnd \
	  [tclAE::build::nameObject cfol [tclAE::build::TEXT $mac_params(srcfold)]]]} res
	foreach f $mac_contents($mac_params(srcfold)) {
	    if {[mac::discriminate $f]} {
		catch {AEBuild -r 'MACS' core setd ---- [tclAE::build::propertyObject issl \
		  [tclAE::build::filename [file join $mac_params(srcfold) $f]]] data 'bool'(0${toggle})} res  
	    }
	} 
    }
    # Process in subfolders
    foreach fold ${mac::folderslist} {
	set fold [file join $mac_params(srcfold) $fold]
	if {[info exists mac_contents($fold)]} {
	# Open the subfolder's finder window 
	catch {AEBuild -r 'MACS' aevt odoc ---- [tclAE::build::propertyObject cwnd \
	  [tclAE::build::nameObject cfol [tclAE::build::TEXT $fold]]]} res
	    foreach f $mac_contents($fold) {
		if {[mac::discriminate [file join $fold $f]]} {
		catch {AEBuild -r 'MACS' core setd ---- [tclAE::build::propertyObject issl \
		  [tclAE::build::filename [file join $fold $f]]] data 'bool'(0${toggle})} res  
		}
	    } 
	}
    } 
}

	
# Duplicate
# ---------
proc mac::duplicateFiles {} {
    if ![mac::selectDialog Duplicate] return 
    mac::duplicateProc
}

proc mac::duplicateProc {} {
    global mac::fileslist mac::objlist
    mac::buildFilesList
    set mac::objlist {}
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    lappend mac::objlist [tclAE::build::filename $f]
	} 
    } 
    mac::duplicateByChunks
    message "Duplicate event sent OK."
}

proc mac::duplicateByChunks {} {
    global mac::objlist macchunksize
    if {![set len [llength ${mac::objlist}]]} {return} 
    message "Duplicating selected files."
    set nbchunks [expr $len / $macchunksize]
    set ini [expr {$nbchunks * $macchunksize}]
    set last [expr $len - 1]
    while {[expr $ini >= 0]} {
	set subobjlist [lrange ${mac::objlist} $ini $last]
	catch {AEBuild -r 'MACS' core clon ---- [tclAE::build::List ${mac::objlist} -untyped]} 
	set last [expr $ini - 1]
	set ini [expr {$last - $macchunksize} + 1]
    }
}


# Copy and Move
# -------------
proc mac::copyFiles {} {
    if ![mac::moveDialog Copy] return
    if ![mac::moveOrCopy clon] return
    message "Copy event sent OK."
}

proc mac::moveFiles {} {
    if ![mac::moveDialog Move] return
    if ![mac::moveOrCopy move] return
    message "Move event sent OK."
}

proc mac::moveOrCopy {event} {
    global mac::folderslist mac_params mac_contents mac::objlist
    if {$mac_params(trgtfold)==""} {
	alertnote "Empty target folder."
	return 0
    } 
    mac::buildFiles&Folders
    if {![file exists $mac_params(trgtfold):]} {
	mac::makeDirs $mac_params(trgtfold)
    } 
    set mac::objlist {}
    if {[info exists mac_contents($mac_params(srcfold))]} {
	foreach f $mac_contents($mac_params(srcfold)) {
	    if {[mac::discriminate [file join $mac_params(srcfold) $f]]} {
		lappend mac::objlist [tclAE::build::filename [file join $mac_params(srcfold) $f]]
	    }
	} 
    } 
    mac::processByChunks $event ${mac::objlist} $mac_params(trgtfold)
    foreach fold ${mac::folderslist} {
	set mac::objlist {}
	set srcsubfold [file join $mac_params(srcfold) $fold]
	if {[expr ![file exists [file join $mac_params(trgtfold) $fold]] \
	  && {[set mac_contents($srcsubfold)]!=""}]} {
	    mac::makeDirs [file join $mac_params(trgtfold) $fold]
	} 
	if {[info exists mac_contents($srcsubfold)]} {
	    foreach f $mac_contents($srcsubfold) {
		if {[mac::discriminate [file join $mac_params(srcfold) $fold $f]]} {
		    lappend mac::objlist [tclAE::build::filename [file join $mac_params(srcfold) $fold $f]]
		}
	    } 
	}
	mac::processByChunks $event ${mac::objlist} [file join $mac_params(trgtfold) $fold]
    } 
    return 1
}

proc mac::processByChunks {event theobjlist targetdir} {
    global macchunksize mac_params
    if {![set len [llength $theobjlist]]} {return} 
    message "Moving/Copying to $targetdir"
    set nbchunks [expr $len / $macchunksize]
    set ini [expr {$nbchunks * $macchunksize}]
    set last [expr $len - 1]
    while {[expr $ini >= 0]} {
	set subobjlist [lrange $theobjlist $ini $last]
	catch {AEBuild -r 'MACS' core $event ---- [tclAE::build::List $subobjlist -untyped] \
	  insh [tclAE::build::nameObject cfol [tclAE::build::TEXT $targetdir]] \
	  alrp 'bool'(0$mac_params(overwrite))} res
	if {[mac::testError $res "in $targetdir"]} return
	set last [expr $ini - 1]
	set ini [expr {$last - $macchunksize} + 1]
    }
}


# Rename
# ------
proc mac::renameFiles {} {
    if ![mac::renameDialog Rename] {return} 
    mac::renameProc
    message "Rename event sent OK"
}

proc mac::renameProc {} {
    global mac::folderslist mac::namelist mac::renamelist mac_params mac_contents 
    mac::buildFiles&Folders
    set mac_params(currnum) $mac_params(digitopt)
    set mac::namelist {}
    set mac::renamelist {}
    foreach f $mac_contents($mac_params(srcfold)) {
	set mac_params(paddvalue) [string length [llength $mac_contents($mac_params(srcfold))]]
	if {[mac::discriminate [file join $mac_params(srcfold) $f]]} {
	    regsub $mac_params(regex) $f $mac_params(replace) res
	    lappend mac::namelist $f
	    lappend mac::renamelist [mac::applyRenameOptions $res]
	}
    } 
    if {![mac::processRenamelist $mac_params(srcfold)]} return
    foreach fold ${mac::folderslist} {
	set mac_params(currnum) $mac_params(digitopt)
	set mac::namelist {}
	set mac::renamelist {}
	foreach f $mac_contents([file join $mac_params(srcfold) $fold]) {
	    set mac_params(paddvalue) [string length [llength $mac_contents([file join $mac_params(srcfold) $fold])]]
	    if {[mac::discriminate [file join $mac_params(srcfold) $fold $f]]} {
		if {$mac_params(casestr)==""} {
		    regsub $mac_params(regex) $f $mac_params(replace) res
		} else {
		    regsub $mac_params(casestr) $mac_params(regex) $f $mac_params(replace) res
		}
		lappend mac::namelist $f
		lappend mac::renamelist [mac::applyRenameOptions $res]
	    }
	} 
	if {![mac::processRenamelist [file join $mac_params(srcfold) $fold]]} return
    } 
}

# Renaming options are : trucating, numbering and casing (applied in this order)
proc mac::applyRenameOptions {name} {
    global mac::folderslist mac::renamelist mac_params mac_contents 
    if !$mac_params(addopts) {return $name}
    if $mac_params(truncating) {set name [mac::truncate $name]}
    if $mac_params(numbering) {set name [mac::numbering $name]}
    if $mac_params(casing) {set name [mac::casing $name]}
    return $name
}

proc mac::truncate {name} {
    global mac_params  
    set num [split $mac_params(truncexp) .]
    set str [split $name .]
    switch [llength $num] {
	1 {return [string range $name 0 [expr [lindex $num 0]-1]]}
	2 {
	    switch [llength $str] {
		1 {
		    return [string range $name 0 [expr [lindex $num 0]-1]]
		}
		default {
		    regexp {^(.*)\.(.*)$} $name dum lt rt
		    return "[string range $lt 0 \
		      [expr [lindex $num 0]-1]].[string range $rt 0 [expr [lindex $num 1]-1]]"
		}
	    } 
	}
    }
}

proc mac::numbering {name} {
    global mac_params
    if {$mac_params(paddopt)} {
	eval set affix "[format %0.$mac_params(paddvalue)d $mac_params(currnum)]"
    } else {
	set affix "$mac_params(currnum)"
	}
    if {!$mac_params(incropt) && [expr $mac_params(currnum) > 0]} {
    set affix "-$affix"
    }
    if {$mac_params(whereopt)} {
	set result "$name$affix"
    } else {
	set result "$affix$name"
    }
    incr mac_params(currnum)
    return $result
}

proc mac::casing {name} {
    global mac_params
    switch $mac_params(caseopt) {
	0 {return [string toupper $name]}
	1 {return [string tolower $name]}
	2 {
	    set str [split $name]
	    set name ""
	    foreach w $str {
		append name " [string toupper [string index $w 0 ]][string tolower [string range $w 1 end]]"
	    } 
	    return [string trim $name]
	}
	3 {return "[string toupper [string index $name 0 ]][string tolower [string range $name 1 end]]"}
    }
}

proc mac::processRenamelist {fold} {
    global mac::namelist mac::renamelist mac_params  
    if {![set len [llength ${mac::renamelist}]]} {return 1}
    set i 1
    while {[expr $i <= $len]} {
	if {[lindex ${mac::renamelist} $i]==[lindex ${mac::renamelist} [expr $i - 1]]} {
	    alertnote "Naming conflict : two files renamed\r[lindex ${mac::renamelist} $i]\
	      in folder $fold."
	    return 0
	} 
	incr i
    }
    for {set i 0} {$i<$len} {incr i} {
	set oldname [lindex ${mac::namelist} $i]
	if {[set newname [lindex ${mac::renamelist} $i]]==""} {
	    alertnote "Empty new name for [file join $mac_params(srcfold) $fold [lindex ${mac::namelist} $i]]"
	    return 0
	}
	catch {AEBuild -r 'MACS' core setd ---- [tclAE::build::propertyObject pnam \
	  [tclAE::build::filename [file join $fold $oldname]]] \
	  data "$newname"} res
	if [mac::testError $res] {return 1}
    }
    return 1
}


# Trash
# -----
proc mac::trashFiles {} {
    if ![mac::selectDialog Trash] {return} 
    mac::trashProc
    message "Trash event sent OK."
}

proc mac::trashProc {} {
    global mac::fileslist mac_params
    mac::buildFilesList
    set mac_params(lasttrashed) ""
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    lappend mac_params(lasttrashed) $f
	    message "Trashing $f"
	    catch {AEBuild -r 'MACS' core delo ---- [tclAE::build::filename $f]} 
	}
    } 
}

proc mac::untrashFiles {} {
    global mac::fileslist mac_params untrashlist utvalues
    set mac_params(trashnb) [tclAE::build::resultData 'MACS' core cnte ---- [tclAE::build::propertyObject trsh 'null'()] kocl type(cobj)]
    if {!$mac_params(trashnb)} {
	alertnote "Trash is empty."
	return
    } 
    if ![mac::untrashDialog] return 
    mac::untrashProc
}

# The Put Away event has been changed from 'FNDR sput' to 'fndr ptwy'. 'sput' happens to work
# with individual files, 'ptwy' with the whole contents of the trash.
proc mac::untrashProc {} {
    global mac::fileslist mac_params untrashlist utvalues
    if {[expr {$mac_params(lasttrashed)==""}] && [expr {$mac_params(untrashopt)==0}]} {
	alertnote "No files trashed from macMenu."
	return
    } 
    switch $mac_params(untrashopt) {
	0 {
	    message "Untrashing last files..."
	    set rootvol [lindex [file split [lindex $mac_params(lasttrashed) 0]] 0]
	    foreach f $mac_params(lasttrashed) {
		set f [file tail $f]
		catch {AEBuild -r 'MACS' FNDR sput ---- [tclAE::build::alis [file join $rootvol Trash]]\
		  fsel [tclAE::build::alis [file join $rootvol Trash $f]]}
	    }
	}
	1 {
	    message "Untrashing..."
	    catch {AEBuild -r 'MACS' fndr ptwy ---- [tclAE::build::indexObject cobj "abso('all ')" \
	      [tclAE::build::propertyObject trsh [tclAE::build::nullObject]]] } res
	    if [mac::testError $res] return
	}
    }
    set mac_params(lasttrashed) ""
    message "Untrash event sent OK."
}


# Alias
# -----
proc mac::aliasFiles {} {
    if ![mac::moveDialog Alias] return 
    mac::aliasProc
}

# We make a distinction between two situations :
# - if no target folder is specified or if it is the same as the source folder, then
#   the aliases are made in the same folders and subfolders as the original files 
# - if a target folder is specified then all the aliases are sent to this folder 
#   no matter where the original files are located in the source folder, i-e even
#   if the 'In hierarchy' flag is used.
proc mac::aliasProc {} {
    global mac::fileslist mac_params
    mac::buildFilesList    
    set onspot [expr {$mac_params(srcfold)==$mac_params(trgtfold)} || {$mac_params(trgtfold)==""}]
    if {[expr !$onspot && ![file exists $mac_params(trgtfold)]]} {
	mac::makeDirs $mac_params(trgtfold)
    } 
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    message "Making alias for $f"
	    if {$onspot} {
		catch {AEBuild -r 'MACS' FNDR sali ---- [tclAE::build::alis [file dirname $f]] \
		  fsel [tclAE::build::alis $f]} res
	    } else {
		catch {AEBuild -r 'MACS' core crel \
		  kocl type(alia) \
		  insh [tclAE::build::nameObject cfol [tclAE::build::TEXT $mac_params(trgtfold)]] \
		  data [tclAE::build::filename $f]} res 
	    }
	}
    } 
    message "Make Aliases event sent OK."
}

proc mac::removeAliases {} {
    if ![mac::unaliasDialog Remove] return 
    mac::rmAliasProc
}

proc mac::rmAliasProc {} {
    global mac::folderslist mac_params
    mac::buildFoldersList
    set mac_params(lasttrashed) ""
    foreach fol ${mac::folderslist} {
	message "Removing aliases from $fol"
	set nb [tclAE::build::resultData 'MACS' core cnte ---- [tclAE::build::nameObject cfol [tclAE::build::TEXT $fol]] kocl type(alia)]
	if $nb {
	    set aliaslist [tclAE::build::resultData 'MACS' core getd rtyp TEXT \
	      ---- [tclAE::build::indexObject alia "abso('all ')" [tclAE::build::nameObject cfol [tclAE::build::TEXT $fol]]]]
	    if {$nb==1} {set aliaslist [list $aliaslist]}
	    foreach al $aliaslist {
		lappend mac_params(lasttrashed) $al
		catch {AEBuild -r 'MACS' core delo ---- [tclAE::build::filename $al]} 
		message "Trashing $al"
	    }
	}
    }
    message "Aliases removed."
}  


# Lock
# ----
proc mac::lockFiles {} {
    if ![mac::selectDialog Lock] return 
    mac::lockProc
}

proc mac::lockProc {} {
    mac::setlock 1
    message "Lock event sent OK."
}

proc mac::unlockFiles {} {
    if ![mac::selectDialog Unlock] return 
    mac::unlockProc
}

proc mac::unlockProc {} {
    mac::setlock 0
    message "Unlock event sent OK."
}

proc mac::setlock {{toggle 1}} {
    global mac::fileslist
    mac::buildFilesList
    set action(1) "Locking"
    set action(0) "Unlocking"
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    message "[set action($toggle)] $f"
	    catch {AEBuild -r 'MACS' core setd ---- [tclAE::build::propertyObject aslk \
	      [tclAE::build::filename $f]] data 'bool'(0${toggle})} res
	}
    } 
}


# Change creator
# --------------
proc mac::changeCreator {{dir ""}} {
    if ![mac::changeCreatorDialog Change] return
    mac::changeCreatorProc $dir
    message "Change creator event sent OK"
}

proc mac::changeCreatorProc {{dir ""}} {
    global mac::fileslist mac_params maccreatorslist
    mac::buildFilesList
    set creator [string range [lindex $maccreatorslist $mac_params(creatoridx)] 0 3]
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    message "Changing creator for $f"
	    catch {AEBuild -r 'MACS' core setd ---- [tclAE::build::propertyObject fcrt [tclAE::build::filename $f]] \
	      data 'type'($creator)} res
	}
    } 
}


# Change type
# -----------
proc mac::changeType {{dir ""}} {
    if ![mac::changeTypeDialog Change] return
    mac::changeTypeProc $dir
    message "Change type event sent OK"
}

proc mac::changeTypeProc {{dir ""}} {
    global mac::fileslist mac_params mactypeslist
    mac::buildFilesList
    set type [string range [lindex $mactypeslist $mac_params(typeidx)] 0 3]
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    message "Changing type for $f"
	    catch {AEBuild -r 'MACS' core setd ---- [tclAE::build::propertyObject asty [tclAE::build::filename $f]] \
	      data 'type'($type)} res
	}
    } 
}


# Transtype (convert line endings)
# --------------------------------
proc mac::transtypeFiles {} {
    global mac_params
    if ![mac::transtypeDialog Transtype] return
    mac::transtypeProc
    message "Transtyping done"
}

proc mac::transtypeProc {} {
    global mac::fileslist mac_params
    set mess "Source and target types are identical \
      ($mac_params(fromeol) to $mac_params(toeol))."
    if {$mac_params(fromeol)==$mac_params(toeol)} {
	if {$mac_params(fromshell)} {
	    return $mess
	} else {
	    alertnote $mess
	    return
	} 
    }
    mac::buildFilesList
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    mac::doTranstype $f
	}
    } 
    return
}

# The code of this proc is borrowed from the file::convertLineEndings proc
# written by Johan Linde in fileManipulation.tcl (7.5d19). It is simply
# adapted to the MacMenu context.
proc mac::doTranstype {file} {
    global mac_params
    set contents [file::readAll $file]
    if {[regexp {\n\r} $contents]} {
	set thisType "win"
    } elseif {[regexp {\r} $contents]} {
	set thisType "unix"
    } else {
	set thisType "mac"
    }
    if {$thisType==$mac_params(toeol)} {return}
    if {$mac_params(fromeol)!="all" && $thisType!=$mac_params(fromeol)} {return}
    message "Transtyping $file"
    switch $thisType {
	"mac" {
	    if {$mac_params(toeol) == "win"} {
		regsub -all "\n" $contents "\r\n" contents
	    }
	}
	"unix" {
	    if {$mac_params(toeol) == "win"} {
		regsub -all "\r" $contents "\r\n" contents
	    }
	}
	"win" {
	    switch -- $mac_params(toeol) {
		"mac" {
		    regsub -all "\n\r" $contents "\r" contents
		}
		"unix" {
		    regsub -all "\n\r" $contents "\n" contents
		}
	    }
	}
    }
    if {$mac_params(toeol) == "mac"} {
	file::writeAll $file $contents 1
    } else {
	set fid [tclAE::build::resultData 'MACS' rdwr open ---- [makeAlis $file] perm true()]
	AEBuild 'MACS' rdwr writ ---- [tclAE::build::TEXT $contents] refn $fid
	if {$thisType == "win"} {
	    AEBuild 'MACS' rdwr seof ---- $fid set2 [string length $contents]
	}
	AEBuild 'MACS' rdwr clos ---- $fid
    }
}


# List
# ----
proc mac::listFiles {} {
    global mac_params
    if ![mac::listDialog List] return
    mac::listProc
    message "List files done."
}

proc mac::listProc {} {
    global mac_params sortbylist
    mac::buildFilesList
    message "Listing files..."
    set newlist [mac::listBuildProc]
    set result [mac::listFilesHeader]
    if {[llength $newlist]} {
	append result "\n[llength $newlist] files"
	if {$mac_params(sortbyidx)} {
	    append result "\nSorted by '[lindex $sortbylist $mac_params(sortbyidx)]'"
	} 
	append result "\n\n[join $newlist "\n"]"
    } else {
	append result "\n\nNo files"
    }
    if {!$mac_params(fromshell)} {
	new -n "Files list" -info $result
    }
    return $result
}

proc mac::listBuildProc {} {
    global mac::fileslist mac_params sortcodelist sortbylist
    mac::buildFilesList
    message "Listing files..."
    set newlist {}
    foreach f ${mac::fileslist} {
	if {[mac::discriminate $f]} {
	    lappend newlist $f
	}
    } 
    if {![llength $newlist]} {return ""} 
    if {$mac_params(sortbyidx)} {
	message "Sorting by '[lindex $sortbylist $mac_params(sortbyidx)]'. Wait..."
	return [mac::listDoSort $newlist [lindex $sortcodelist $mac_params(sortbyidx)]]
    } else {
	return $newlist
    }
}

# This proc sorts the files list according  to  the  criterion  specified  by
# 'by'. The admissible values for 'by' are : asmo,  ascd,  ptsz,  kind,  labi
# (vers and  comt  purposely  omitted  here)  corresponding  respectively  to
# "Modification date", "Creation date", "Size", "Kind", "Labels".
# If the "Incl Crit" (Include Criterion) checkbox is checked,  the  value  of
# the criterion will be reported with each item of the sorted list.
proc mac::listDoSort {list by} {
    global  mac_params
    set criterionlist ""
    # Let's build a temporary list prepending to each item the value
    # of the property
    foreach f $list {
	if {![catch {set res [AEBuild -r 'MACS' core getd ---- \
	  [tclAE::build::propertyObject $by [tclAE::build::filename $f]]]}]} {
	    lappend criterionlist "$res$f"
	}
    } 
    # Sort the list. Biggest or newest comes first.
    if {$by=="ptsz" || $by=="asmo" || $by=="ascd"} {
	set criterionlist [lsort -decr $criterionlist]
    } else {
	set criterionlist [lsort $criterionlist]
    }
    # Handle the prefix
    set resultlist ""
    if {$mac_params(criterion)} {
	foreach f $criterionlist {
	    # We're interested only in the second and fourth parenthesing groups:
	    if {[regexp {aevt\\ansr[^]+(|)?([^]+)(|)?\)?\}(.*)} $f dum a criterion b filename]} {
		switch $by {
		    asmo - ascd {
			set criterion [mtime [format %i 0x$criterion]]
		    }
		    ptsz {
			set criterion [format %i 0x$criterion]
		    }
		}
		lappend resultlist "$criterion\t$filename"
	    } else {
		lappend resultlist "\t$f"
	    }
	} 
    } else {
	foreach f $criterionlist {
	    regsub {aevt\\ansr[^\}]+\}} $f "" f
	    lappend resultlist $f
	} 
    }
    return $resultlist
}

proc mac::listFilesHeader {} {
    global mac::fileslist mac_params alternlist compvaluelist compdatelist
    set result "Filter : $mac_params(regex)\n"
    append result "Folder : $mac_params(srcfold)\n"
    if {$mac_params(subfolds)==1} {append result "Nesting level : $mac_params(nest)\n"} 
    if {$mac_params(subfolds)==2} {append result "All subfolders\n"} 
    if {$mac_params(iscase)} {append result "case sensitive"} else {append result "case insensitive"}
    if {$mac_params(isneg)} {append result " - negate filter"} 
    if {$mac_params(addconds)} {
	if {$mac_params(asty)!=""} {append result "\nType [lindex $alternlist $mac_params(isasty)] $mac_params(asty)"}
	if {$mac_params(fcrt)!=""} {append result "\nCreator [lindex $alternlist $mac_params(isfcrt)] $mac_params(fcrt)"}
	if {$mac_params(ascd)!=""} {append result "\nCreation date [lindex $compdatelist $mac_params(isascd)] $mac_params(ascd)"}
	if {$mac_params(asmo)!=""} {append result "\nModification date [lindex $compdatelist $mac_params(isasmo)] $mac_params(asmo)"}
	if {$mac_params(size)!=""} {append result "\nSize [lindex $compvaluelist $mac_params(issize)] $mac_params(size)"}
    } 
    return $result
}

