# File : "macMenuShell.tcl"
#                        Created : 2001-01-22 21:35:13
#              Last modification : 2001-08-29 21:46:18
# Author : Bernard Desgraupes
# e-mail : <berdesg@easynet.fr>
# www : <http://perso.easynet.fr/~berdesg/>
# 
# (c) Copyright : Bernard Desgraupes, 2001
#         All rights reserved.
# This software is free software. See licensing terms in the MacMenu Help file.
#
#  Description : this file is part of the macMenu package for Alpha.
# It contains procedures to create a shell to make the macMenu
# functionalities available from a command line.



namespace eval mac {}
namespace eval macsh {}


# # # Shell definition # # #
# ==========================

proc mac::macShell {} {
    Shel::start "mac" {*Mac Shell*} \
      "WELCOME TO MAC SHELL\r\tType help to get info about commands.\r"
    # Binding MacShell completions to 'tab' key after the shellMode is loaded.
    Bind 0x30 	{macsh::completion} Shel
    message "Bound the \"tab\" key to macShell completion"
}

proc mac::evaluate {cmdline} {
    history add $cmdline
    if {$cmdline==""} return
    set cmd [lindex [split $cmdline] 0]
    regsub -- "-all" $cmdline "-all 1" cmdline
    if {[info tclversion] < 8.0} {
	set prefix "macsh::"
    } else {
	set prefix "::macsh::"
    }
    if {[info commands "$prefix$cmd"]!=""} {
	catch {eval $prefix$cmdline} res
	return $res
    } else {
	# If the command is unknown to Mac Shell, try to 
	# evaluate $cmdline as a Tcl instruction
	return [Alpha::evaluate $cmdline]
    }
}

proc mac::Prompt {} {
    global Shel::startPrompt Shel::endPrompt
    return "${Shel::startPrompt}[file tail [string trimright [macsh::pwd] {:}]]${Shel::endPrompt} "
}


# # # MacShell commands # # #
# ===========================

proc macsh::help {{subcmd ""}} {
    global mac::commands mac::options subcmds HOME index::feature
    switch $subcmd {
	options {
	    set res ""
	    foreach o [mac::optionsHelp] {
		append res "$o\n"
	    } 
	    return $res
	}
	more {
	    set helpname [file join $HOME Help "Mac Menu Help"]
	    if {[file exists $helpname]} {
		if {[info tclversion] < 8.0} {
		    edit $helpname
		} else {
		    ::edit $helpname
		}
		set start [minPos]
		if {![catch {search -f 1 -s -r 1 "^Mac Shell Syntax" $start} res]} {
		    goto [lindex $res 0]
		} else {goto [minPos]}
		winunequalHor
	    } else {
		return "Can't find file \'Mac Menu Help\'"
	    }
	}
	bindings {return [mac::bindingsInfoString]}
	tutorial {
	    mac::macMenuTutorialInfo
	    winunequalHor
	}
	version {return [lindex [set index::feature(macMenu)] 0]}
	default {
	    set res "Commands available in Mac shell :\n"
	    foreach c ${mac::commands} {
		if {[info exists subcmds($c)]} {
		    foreach s $subcmds($c) {
			switch $c {
			    "files" {
				set str [lindex $s 1]
				regsub -all "(.)" $str "-\\1 " str
				append res "\t$c [lindex $s 0]\t<$str-all>\n"
			    }
			    default {
				append res "\t$c $s\n"
			    }
			}
			
		    } 
		} else {
		    append res "\t$c\n"
		} 
	    }
	    append res "Plain Tcl commands are also interpreted in Mac shell."
	    return $res
	}
    }
}

proc macsh::files {subcmd args} {
    global mac_params maccreatorslist mactypeslist maceolslist macMenumodeVars
    set mac_params(fromshell) 1
    set opts(-c) ""
    set opts(-f) ".*"
    set opts(-i) 0
    set opts(-l) 0
    set opts(-n) 0
    set opts(-r) &
    set opts(-s) "[macsh::pwd]"
    set opts(-t) ""
    set opts(-o) $macMenumodeVars(overwriteIfExists)
    set opts(-k) ""
    set opts(-d) 0
    set opts(-b) ""
    set opts(-x) ""
    set opts(-all) 0
    getOpts {-f -s -l -i -n -t -o -r -c -k -d -b -x -all}
    if {[file exists [file join [macsh::pwd] $opts(-s)]] && [file isdirectory [file join [macsh::pwd] $opts(-s)]]} {
	set mac_params(srcfold) [file join [macsh::pwd] $opts(-s)]
    } else {
	set mac_params(srcfold) $opts(-s)
    }
    set mac_params(iscase) [expr !$opts(-i)]
    set mac_params(casestr) [expr {$mac_params(iscase) ? "":"-nocase"}]
    set mac_params(isneg) $opts(-n)
    set mac_params(overwrite) $opts(-o)
    if {$opts(-l)=="all"} {
	set mac_params(subfolds) 2
    } elseif {[expr $opts(-l) > 0]} {
	set mac_params(subfolds) 1
	set mac_params(nest) $opts(-l)
    } else {
	set mac_params(subfolds) $opts(-l)
    }
    mac::getSortingOption $opts(-b)
    set rgx $opts(-f)
    set rgx [string trimleft $rgx ^]
    set rgx [string trimright $rgx $]
    if {$rgx==""} {
	set rgx ".*"
    } 
    set mac_params(regex) "^$rgx\$"
    # If option -all is given, it avverrides any -f option.
    # It is equivalent to '-f .* -n 0'.
    if {$opts(-all)==1} {
	set mac_params(regex) ".*"
	set mac_params(neg) 0
    } 
    switch $subcmd {
	copy -
	move {
	    set mac_params(trgtfold) $opts(-t)
	    if {$subcmd=="copy"} {
		set subcmd clon
	    } 
	    if {$opts(-t)==""} {
		return "No target folder for 'files $subcmd' : option -t missing."
	    } 
	    mac::moveOrCopy $subcmd
	}
	rename {
	    if {$opts(-r)==""} {
		return "Replacement expression (option -r) is empty."
	    } 
	    set mac_params(replace) $opts(-r)
	    set mac_params(caseopt) [lsearch  [list u l w f] $opts(-k)]
	    set mac_params(numbering) $opts(-d)
	    set oldexp $mac_params(truncexp)
	    set mac_params(truncexp) $opts(-x)
	    if {$mac_params(caseopt)=="-1"} {
		if {$opts(-k)!=""} {
		    return "Bad value for option -k : should be u, l, w, or f" 
		} 
		set mac_params(casing) 0
	    } else {
		set mac_params(casing) 1
	    }
	    if {$mac_params(truncexp)==""} {
		set mac_params(truncating) 0
		set mac_params(truncexp) $oldexp
	    } else {
		set mac_params(truncating) 1
	    }
	    set mac_params(addopts) [expr $mac_params(casing) + $mac_params(numbering) + $mac_params(truncating)]
	    mac::renameProc
	}
	duplicate -
	trash -
	lock -
	unlock -
	alias -
	select -
	unselect -
	list {
	    return [mac::${subcmd}Proc]
	}
	change {
	    set thecreator $opts(-c)
	    set thetype $opts(-t)
	    if {$thecreator=="" && $thetype==""} {
		return "No creator or type specified for 'files $subcmd' : option -c or -t missing."
	    } 
	    if {$thecreator!=""} {
		set tmplist $maccreatorslist
		set maccreatorslist [lreplace $maccreatorslist 0 0 $thecreator]
		set mac_params(creatoridx) 0
		mac::changeCreatorProc $mac_params(srcfold)
		set maccreatorslist tmplist
	    } 
	    if {$thetype!=""} {
		set tmplist $mactypeslist
		set mactypeslist [lreplace $mactypeslist 0 0 $thetype]
		set mac_params(typeidx) 0
		mac::changeTypeProc $mac_params(srcfold)
		set mactypeslist tmplist
	    } 
	}
	transtype {
	    set mac_params(fromeol) $opts(-o)
	    set mac_params(toeol) $opts(-t)
	    if {[lsearch -exact $maceolslist $mac_params(fromeol)]=="-1" && $mac_params(fromeol)!="all"} {
	        return "Unknown type '$mac_params(fromeol)'. Should be mac, unix, win or all."
	    } 
	    if {[lsearch -exact $maceolslist $mac_params(toeol)]=="-1"} {
	        return "Unknown target type '$mac_params(toeol)'. Should be mac, unix or win."
	    } 
	    return [mac::transtypeProc]
	    }
	untrash {
	    set mac_params(untrashopt) $opts(-all)
	    mac::untrashProc
	}
	unalias {
	    mac::rmAliasProc
	}
	default {return "Unknown subcommand $subcmd"}	
    }
    return
}

proc macsh::infos {subcmd {item ""}} {
    global mac_params
    set res ""
    if {[expr {$item==""} && {$subcmd!="hardware"}]} {
	return "Missing path : should be 'info $subcmd <path>'"
    } 
    switch $subcmd {
	"file" {
	    if {[file dirname $item]==":"} {
		set item [file join $mac_params(pwd) $item]
	    } 
	    if {![file exists $item] || ![file isfile $item]} {
		return "Can't find file $item"
	    }
	    mac::getFilesInfo $item
	}
	"folder" {
	    set item [string trimright $item :]
	    if {[file dirname $item]==":"} {
		set item [file join $mac_params(pwd) $item]
	    } 
	    if {![file exists $item] || ![file isdirectory $item]} {
		return "Can't find folder $item"
	    }
	    mac::getFolderInfo $item
	    if {$mac_params(isshared)} {
	        mac::getFolderSharInfo $item
	    } 
	}
	"disk" -
	"volume" {
	    set subcmd volume
	    mac::getVolumeInfo $item
	    if {$mac_params(isshared)} {
	        mac::getVolumeSharInfo $item
	    } 
	}
	"appl" {
	    if {[file dirname $item]==":"} {
		set item [file join $mac_params(pwd) $item]
	    } 
	    if {![file exists $item] || ![file isfile $item]} {
		return "Can't find file $item"
	    }
	    if ![mac::getApplInfo $item] {return "[file tail $item] does not have type APPL"}
	}
	"process" {
	    if {[lsearch [mac::getProcessesList] $item]==-1} {
		return "$subcmd $item is not a running process" 
	    } 
	    mac::getProcessInfo $item
	}
	"hardware" {
	    mac::getHardwareInfo 
	}
	default {return "files : unknown subcommand $subcmd"}
    }
    global mac${subcmd}info mac_description
    foreach t [lsort -ignore [array names mac${subcmd}info]] {
	append res "$mac_description($t): [set mac${subcmd}info($t)]\n"
    } 
    return $res
}

proc macsh::empty {} {
    mac::emptyTrash
    return
}

proc macsh::eject {} {mac::eject}

proc macsh::restart {} {mac::restart}

proc macsh::shutdown {} {mac::shutDown}


# # # Unix-shell-like commands # # #
# ==================================

# -------------------------------------------------------------------------
# cd : change directory
# SYNTAX : 
# cd		change to Alpha's home directory
# cd .		change to directory of frontmost window (not including the shell)
# cd ..		change to parent directory. 
#     To go several levels up, type several .. separated by colons. E-g :
#     cd ..:..       etc.
# Alternatively
# cd :		change to parent directory. 
#     To go several levels up, type several : like
#     cd :::         etc.
# cd blah	change to subfolder blah of current folder if blah exists
# cd abs_path	change to directory corresponding to absolute path "abs_path"
# -------------------------------------------------------------------------
proc macsh::cd {{item ""}} {
    global HOME mac_params
    if {[regexp "^:+$" $item]} {
        regsub -all : $item ..: item
    } 
    set item [string trimright $item :]
    if {$item=="."} {
	set topwin [lindex [winNames -f] 1]
	if {$topwin!=""} {
	    set mac_params(pwd) [file dirname $topwin]
	} else {
	    macsh::cd
	}
    } elseif {$item==""} {
	set mac_params(pwd) $HOME
    } elseif {[regexp {^\.\.(:\.\.)*$} $item]} {
	set nb [llength [split $item :]]
	for {set i 0} {$i<$nb} {incr i} {
	    set dir [file dirname $mac_params(pwd)]
	    if {$dir!=":"} {
		set mac_params(pwd) $dir
	    } 
	}
    } else {
	set dir [file join $mac_params(pwd) $item]
	if {[file exists $dir] && [file isdirectory $dir]} {
	    set mac_params(pwd) $dir
	} elseif {[file exists $item] && [file isdirectory $item]} {
	    set mac_params(pwd) $item
	} elseif {[regexp "$item:" [mac::getAllVolsList]]} {
	    set mac_params(pwd) $item:
	} else {
	    return "Can't find folder $item"
	}
    } 
    return $mac_params(pwd)
}     

# -------------------------------------------------------------------------
# pwd : print working directory
# Returns the complete path of the current folder.
# -------------------------------------------------------------------------
proc macsh::pwd {} {
    global mac_params
    return $mac_params(pwd)
}

# -------------------------------------------------------------------------
# ls : list
# List all files and subfolders in the current folder
# -------------------------------------------------------------------------
proc macsh::ls {} {
    global mac_params
    set reslist [glob -nocomplain  -dir "[file join $mac_params(pwd)]" *] 
    if {[llength $reslist]} {
	return [join $reslist "\n"] 
    } 
    return "Empty folder"
}

# -------------------------------------------------------------------------
# ld : list directories
# List subfolders in the current folder
# -------------------------------------------------------------------------
proc macsh::ld {} {
    global mac_params
    set reslist [glob -nocomplain -types d -dir "[file join $mac_params(pwd)]" *]
    if {[llength $reslist]} {
	return [join $reslist "\n"] 
    } 
    return "No subfolders in $mac_params(pwd)"
}

# -------------------------------------------------------------------------
# mkdir : make directory
# The mkdir command creates a new subfloder in  the  current  folder.  If  no
# argument is given it will create an 'untitled' folder (same as using  cmd-N
# in the Finder). If an argument is specified, it will be the name  given  to
# the new subfolder.
# -------------------------------------------------------------------------
proc macsh::mkdir {{item ""}} {
    global mac_params
    set res ""
    set item [string trimright $item :]
    if {$item==""} {
	catch {AEBuild -r 'MACS' core crel \
	  insh [tclAE::build::nameObject cfol [tclAE::build::TEXT $mac_params(pwd)]] \
	  kocl type(cfol)} res 
	regexp {.*([^]+)} $res dum res
	return [file join $mac_params(pwd) $res]
    } else {
	if {[file dirname $item]==":"} {
	    set item [file join $mac_params(pwd) $item]
	} 
    }
    if {[file exists $item]} {
	return "[file tail $item] already exists in $mac_params(pwd)"
    }
    if {![catch {file mkdir $item}]} {return $item}
    return "mkdir failed"
}

# -------------------------------------------------------------------------
# edit : edit a file
# Give the name of the file : if it is in the current directory, the proc 
# will complete the path. You can use the completion mechanism to enter the
# name of the file : type the first letters, then hit the Tab key.
# If the 'edit' command is used with no argument, you are prompted to 
# select a file to edit.
# -------------------------------------------------------------------------
if {[info tclversion] < 8.0} {
    proc macsh::edit {{item ""}} {
	global HOME mac_params
	set item [string trimright $item :]
	set filename [file join $mac_params(pwd) $item]
	if {$item==""} {
	    if {[catch {set item [getfile "Select a file"]}]} {return ""}
	    edit -w $item
	} elseif {[file exists $filename] && [file isfile $filename]} {
	    edit -w $filename
	} elseif {[file exists $item] && [file isfile $item]} {
	    edit -w $item
	} else {
	    return "Error: Can't open $item"
	}
	return $item
    }     
} else {
    proc macsh::edit {{item ""}} {
	global HOME mac_params
	set item [string trimright $item :]
	set filename [file join $mac_params(pwd) $item]
	if {$item==""} {
	    if {[catch {set item [getfile "Select a file"]}]} {return ""}
	    ::edit -w $item
	} elseif {[file exists $filename] && [file isfile $filename]} {
	    ::edit -w $filename
	} elseif {[file exists $item] && [file isfile $item]} {
	    ::edit -w $item
	} else {
	    return "Error: Can't open $item"
	}
	return $item
    }     
}


# # # Completion mechanism # # #
# ==============================
# 
# For some reason,  Shel  mode  does  not  handle  Alpha's  usual  completion
# mechanism so we write our own proc.
# 
# -------------------------------------------------------------------------
# The completion mechanism is bound to the TAB key like in Unix shells.
# Type the first letters of a command or of a file's or folder's name and hit
# the TAB key : the procedure will try to  complete,  looking  either  for  a
# command name or for the relative or absolute  path  of  a  file  or  folder
# included in the current folder. A complete path is supposed to start with a
# double quote. If there are several possibilities, a pick list is displayed.
# If the result is the path of a file or of a folder, its name is enclosed in
# double quotes to protect blank spaces.
# -------------------------------------------------------------------------
proc macsh::completion {} {
    global macshellcmds mac_params
    set reslist ""
    set inipos [getPos]
    set path [mac::subPath posit]    
    set word [completion::lastWord pos]
    if {[file dirname $path]!=":"} {
	set tmplist [glob -nocomplain -dir "[file join [file dirname $path]]" *]
	set start $posit
	set islong 1
    } else {
	set tmplist [glob -nocomplain -dir "[file join $mac_params(pwd)]" *]
	set start $pos
	set islong 0
    }
    foreach sf $tmplist {
	set sftail [file tail $sf]
	if {[regexp "^$word" $sftail]} {
	    if {[file isdirectory $sf]} {append sftail :}
	    if {$islong} {
		lappend reslist "$sftail"
	    } else {
		lappend reslist "\"$sftail\""
	    }
	} 
    } 
    foreach cmd $macshellcmds {
	if {[regexp "^$word" $cmd]} {
	    lappend reslist $cmd
	} 
    } 
    switch [llength $reslist] {
	0 {
	    message "No completion found"
	    return
	}
	1 {set res [lindex $reslist 0]}
	default {
	    set res [listpick  -p "Complete with :" $reslist]
	    if {$res==""} {return}
	}
    }
    select $start $inipos
    deleteSelection
    if {$islong} {
        set res "[file join [file dirname $path] $res]"
    } 
    insertText $res
    if {[lookAt [pos::math [getPos] - 1]] == "\""} {
	goto [pos::math [getPos] -1]
    }
}

proc mac::subPath {pos} {
    upvar $pos inipos
    set start [getPos]
    set iniline [lineStart $start]
    if {![catch {search -s -f 0 -m 0 {"} $start} res]} {
	if {[pos::compare [lindex $res 0]<$iniline]} {
	    return ""
	} 
	set inipos [lindex $res 1]
	return [getText $inipos $start]
    }
    if {![catch {search -s -f 1 " " $iniline} res]} {
	set inipos [lindex $res 1]
    } else {
	set inipos $iniline
    }
    return ""
}

