# File : "macMenuUtils.tcl"
#                        Created : 2001-01-22 21:34:18
#              Last modification : 2001-10-18 13:35:58
# Author : Bernard Desgraupes
# e-mail : <berdesg@easynet.fr>
# www : <http://webperso.easyconnect.fr/berdesg/>
# 
# (c) Copyright : Bernard Desgraupes, 2001
#         All rights reserved.
# This software is free software. See licensing terms in the MacMenu Help file.
#
# Description : this file is part of the macMenu package for Alpha.
# It contains various utility procs used by macMenu.

namespace eval mac {}


# # # Building files list procs # # #
# ===================================

proc mac::buildFilesList {} {
    global mac::fileslist mac_params
    set mac::fileslist ""
    switch $mac_params(subfolds) {
	0 {mac::getFilesinHier $mac_params(srcfold) 0}
	1 {mac::getFilesinHier $mac_params(srcfold) $mac_params(nest)}
	2 {mac::getFilesinHier $mac_params(srcfold) 1 1}
    }
    message "Processing files list..."
}

proc mac::getFilesinHier {dir depth {unlimited 0}} {
    global mac::fileslist mac_params
    if {![file exists $dir]} {
	alertnote "Can't find folder $dir"
	return
    } 
    message "Building files list..."
    set dir [mac::strToPath $dir]
    set tmplist [glob -nocomplain -dir [file join $dir] *]   
    foreach f $tmplist {
	set dp $depth
	# Here we say ![file isdirectory $f] instead of [file isfile $f] because we want
	# to include aliases in the list :
	if {![file isdirectory $f]} {
	    if {![expr [mac::testRegex $f $mac_params(casestr)] - !$mac_params(isneg)]} {
		lappend mac::fileslist $f
	    } 
	} elseif {[file isdirectory $f] && [expr $depth > 0]} {
	    if {!$unlimited} {incr dp -1} 
	    mac::getFilesinHier $f $dp $unlimited
	}
    }
}

proc mac::buildFiles&Folders {} {
    global mac::folderslist mac_params mac_contents
    if {[info exists mac_contents]} {unset mac_contents}
    set mac_params(trgtfold) [mac::strToPath $mac_params(trgtfold)]
    set mac::folderslist ""
    switch $mac_params(subfolds) {
	0 {mac::getFoldersContents $mac_params(srcfold) 0}
	1 {mac::getFoldersContents $mac_params(srcfold) $mac_params(nest)}
	2 {mac::getFoldersContents $mac_params(srcfold) 1 1}
    }
    message "Processing files list..."
}

proc mac::getFoldersContents {dir depth {unlimited 0}} {
    global mac::folderslist mac_params mac_contents
    if {![file exists $dir]} {
	alertnote "Can't find folder $dir."
	return
    } 
    message "Building files list..."
    set dir [mac::strToPath $dir]
    set tmplist [glob -nocomplain -dir [file join $dir] *]   
    set mac_contents($dir) ""
    foreach f $tmplist {
	set dp $depth
	# Here we say ![file isdirectory $f] instead of [file isfile $f] because we want
	# to include aliases in the list :
	if {![file isdirectory $f]} {
	    if {![expr [mac::testRegex $f $mac_params(casestr)] - !$mac_params(isneg)]} {
		lappend mac_contents($dir) [mac::relFilename $dir $f]
	    } 
	} elseif {[file isdirectory $f] && [expr $depth > 0]} {
	    if {!$unlimited} {incr dp -1} 
	    lappend mac::folderslist ":[mac::relFilename $mac_params(srcfold) $f]"
	    mac::getFoldersContents $f $dp $unlimited
	}
    }
    return ${mac::folderslist}
}

proc mac::buildFoldersList {} {
    global mac::folderslist mac_params
    if {![file exists $mac_params(srcfold)]} {
	alertnote "Can't find folder $mac_params(srcfold)"
	return
    } 
    set mac::folderslist [list $mac_params(srcfold)]
    switch $mac_params(subfolds) {
	0 {return}
	1 {mac::getFoldersinHier $mac_params(srcfold) $mac_params(nest)}
	2 {mac::getFoldersinHier $mac_params(srcfold) 1 1}
    }
}

proc mac::getFoldersinHier {dir depth {unlimited 0}} {
    global mac::folderslist mac_params
    if !$depth return
    message "Building subfolders list..."
    set dir [mac::strToPath $dir]
    set tmplist [glob -nocomplain -types d -dir [file join $dir] *]   
    set mac::folderslist [concat ${mac::folderslist} $tmplist]
    foreach f $tmplist {
	set dp $depth
	if {!$unlimited} {incr dp -1} 
	mac::getFoldersinHier $f $dp $unlimited
    }
}

# -------------------------------------------------------------------------
# When a target folder does not exist, it is created by this proc. Any
# intermediate non existing subfolder must be created as well.
# -------------------------------------------------------------------------
proc mac::makeDirs {path} {
    set path [string trimright $path :]
    set items [split $path ":"]
    set len [llength $items]
    set vol [lrange $items 0 0]
    for {set i 1} {$i<$len} {incr i} {
	eval set foldpath \[file join $vol: [lrange $items 1 $i]\]
	if {![file exists $foldpath]} {
	    file mkdir $foldpath
	}
    }
}

# -------------------------------------------------------------------------
# Truncate to get path relatively to the source folder.
# Modified 2001-07-16 by Frdric Boulanger (thanks) to fix a bug when a 
# dir name contains special regexp chars.
# -------------------------------------------------------------------------
proc mac::relFilename {dir file} {
   if {[regexp "^[file join [quote::Regfind ${dir}] \(.*\)]\$" $file dum res]} {
     return $res
   } else {
     return $file
   }
}

# -------------------------------------------------------------------------
# Add a trailing colon if name contains no colon (volumes)
# -------------------------------------------------------------------------
proc mac::strToPath {dir} {
    set dir [string trimright $dir :]
    if {![regexp : $dir] && $dir!=""} {append dir :} 
    return $dir
}

# -------------------------------------------------------------------------
# Procs to apply the additional conditions
# -------------------------------------------------------------------------
proc mac::testRegex {f {case ""}} {
    global mac_params
    if {$case==""} {
	return [regexp $mac_params(regex) [file tail $f]]
    } else {
	return [regexp $case $mac_params(regex) [file tail $f]]
    }
}

proc mac::discriminate {file} {
    global mac_params
    if !$mac_params(addconds) {return 1}
    return [expr [mac::testDate ascd $file] && [mac::testDate asmo $file] && \
      [mac::testSize $file] && [mac::testType asty $file] && \
      [mac::testType fcrt $file] ]
}

proc mac::testType {type file} {
    global mac_params
    if {$mac_params($type)==""} {
        return 1
    } 
    set thetype [mac::findItemProperty $type file $file]
    return [expr ![expr $mac_params(is$type) - ![string compare $mac_params($type) $thetype]]]
}

proc mac::testDate {date file} {
    global mac_params
    if {$mac_params($date)==""} {
        return 1
    } 
    set thedate [mac::getItemDate $date file $file short]
    return [expr ![expr $mac_params(is$date) - [mac::dateCompare $thedate $mac_params($date)]]]
}

proc mac::testSize {file} {
    global mac_params
    if {$mac_params(size)==""} {
        return 1
    } 
    set thebytes [expr $mac_params(size) * 1024] 
    set thesize [format %1d 0x[mac::findItemProperty ptsz file $file]]
    return [expr ![expr $mac_params(issize) - [expr $thebytes < $thesize]]]
}
 


# # # Misc actions procs # # #
# ============================

proc mac::emptyTrash {} {
    catch {AEBuild -r 'MACS' fndr empt} res
    mac::testError $res
}

proc mac::shutDown {} {AEBuild -r 'MACS' fndr shut}

proc mac::restart {} {AEBuild -r 'MACS' fndr rest}

proc mac::showClipboard {} {
	catch {AEBuild -r 'MACS' FNDR shcl}  
	mac::finderToFront
}

proc mac::eject {} {
    set ejectVolsList [mac::getEjectVolsList]
    set l [llength $ejectVolsList]
    switch $l {
	0 {
	    alertnote "No ejectable volumes."
	    return
	}
	1 {mac::doEject $ejectVolsList}
	default {
	    message "Choose a volume to eject."
	    set volume [listpick  -p "Volume to eject :" $ejectVolsList]
	    if {$volume!=""} {
	        mac::doEject $volume
	    } 
	}
    }
}

# -------------------------------------------------------------------------
# With the 'fndr ejct' Apple Event, the volume is ejected but not unmounted :
# the finder asks to reinsert the disk if it has been changed. To avoid  this
# mess, we use 'core delo' instead.
# -------------------------------------------------------------------------
proc mac::doEject {vol} {
    set vol [string trim $vol "\{\}"]
    catch {AEBuild -r 'MACS' core delo ---- [tclAE::build::nameObject cdis [tclAE::build::TEXT $vol] ]} res
    if {[mac::testError $res]} return
    message "$vol ejected."
}

# -------------------------------------------------------------------------
# Make list of all mounted volumes
# -------------------------------------------------------------------------
proc mac::getAllVolsList {} {
    return [tclAE::build::resultData 'MACS' core getd rtyp TEXT ---- [tclAE::build::indexObject cdis "abso('all ')"]] 
}

# -------------------------------------------------------------------------
# Make list of ejectable volumes
# -------------------------------------------------------------------------
proc mac::getEjectVolsList {} {
    set ejlist ""
    message "Looking for ejectable volumes..."
    foreach v [mac::getAllVolsList] {
        if {[tclAE::build::resultData 'MACS' core getd ---- [tclAE::build::propertyObject isej\
	       [tclAE::build::nameObject cdis [tclAE::build::TEXT $v] ]]]} {
            lappend ejlist $v
        } 
    } 
    return $ejlist
}


# # # Misc utility procs # # #
# ============================

# -------------------------------------------------------------------------
# Switch to the Finder (making it the frontmost process).
# -------------------------------------------------------------------------
proc mac::finderToFront {} {
    AEBuild -r 'MACS' core setd ---- [tclAE::build::propertyObject pisf \
	  [tclAE::build::nullObject]] data 'bool'(01)
}

# -------------------------------------------------------------------------
# Make list of running processes.
# -------------------------------------------------------------------------
proc mac::getProcessesList {} {
    return [tclAE::build::resultData 'MACS' core getd rtyp TEXT ---- [tclAE::build::indexObject prcs "abso('all ')"]] 
}

# -------------------------------------------------------------------------
# Proc to find a specific property for a certain type of item. 
# Type can be : file, cfol, cdis, appf, prcs.
# -------------------------------------------------------------------------
proc mac::findItemProperty {prop type item} {
    if {[catch {set res [tclAE::build::resultData 'MACS' core getd ---- [tclAE::build::propertyObject $prop \
      [tclAE::build::nameObject $type [tclAE::build::TEXT $item]]]]}]} {
	return ""
    } else {
	return $res
    }
}

# -------------------------------------------------------------------------
# Implement the ToolBox Gestalt() function.
# -------------------------------------------------------------------------
proc mac::gestalt {selector} {
    if {[catch {set res [tclAE::build::resultData 'MACS' fndr gstl ---- 'type'($selector)]}]} {
	return ""
    } else {
	return $res
    }
}

# -------------------------------------------------------------------------
# This proc formats the result of mac::gestalt according to
# the specified type.
# Type can be : size, vers, name, mhz, sysa, bool, cpu.
# -------------------------------------------------------------------------
proc mac::gestaltRead {selector type} {
    set res [mac::gestalt $selector]
    if {$res==""} {return ""}
    set ansr ""
    switch $type {
	"size" {
	    set ansr "$res bytes"
	}
	"vers" {
	    set ansr [string trimright [format %1x $res] 0]
	    set ansr [join [split $ansr {}] .]
	}
	"name" {
	    set ansr [format %1x $res]
	    set name ""
	    set len [string length $ansr]
	    for {set i 0} {$i<$len} {incr i 2} {
	        append name [format %c 0x[string range $ansr $i [expr $i+ 1]]]
	    }
	    set ansr $name
	}
	"mhz" {
	    set len [string length $res]
	    set ansr "[string range $res 0 [expr $len - 7]].[string range $res [expr $len - 6] end] Mhz"
	}
	"sysa" {
	    if {$res==1} {
		set ansr "68k"
	    } else {
		set ansr "PowerPC"
	    }
	}
	"bool" {set ansr [expr $res>0]}
	"cpu" {
	    switch [format %1x $res] {
		0  {set ansr "68000"}
		1  {set ansr "68010"}
		2  {set ansr "68020"}
		3  {set ansr "68030"}
		4  {set ansr "68040"}
		101  {set ansr "601"}
		103  {set ansr "603"}
		104  {set ansr "604"}
		106  {set ansr "603e"}
		107  {set ansr "603ev"}
		108  {set ansr "750 (G3)"}
		109  {set ansr "604e"}
		10A  {set ansr "604ev (Mach5)"}
		10C  {set ansr "Max (Altivec)"}
		defautl {set ansr "unknown"}
	    }
	}
	default {set ansr $res}
    }
    return $ansr
}

# -------------------------------------------------------------------------
# Get the size on disk of an item. Prop is ptsz (logical), phys (physical) 
# for a file or a folder; capa (capacity), frsp (free space) for a disk.
# Type is file, cfol or cdis.
# Reply's form : aevt\ansr{'----':comp(000000000D9F6E00)}
# The largest value accepted by Tcl is 2147483647 (= 0x7fffffff).
# If the returned hexa value is larger than 0x7fffffff then the command
# 'format' returns a negative integer. If it is larger than 0xffffffff it 
# rises an error. 
# So, if the result is larger than 268435455 (=0xfffffff), the following
# proc calculates the result in megabytes.
# There is currently a problem with tclAE::build::resultData in Alpha8 so
# we use a workaround until it is solved.
# -------------------------------------------------------------------------
proc mac::getItemSize {prop type item} {
    if {[info tclversion] < 8.0} {
	set res [mac::findItemProperty $prop $type $item]
    } else {
	set res [AEBuild -r 'MACS' core getd ---- [tclAE::build::propertyObject $prop \
	  [tclAE::build::nameObject $type [tclAE::build::TEXT $item]]]]
	regexp {([0-9a-fA-F]+)} $res dum res
    }
    set res [string trimleft $res 0]
    set len [string length $res]
    if {[expr $len < 8]} {
	set rep "[format %1d 0x$res] bytes"
    } else {
	# Integer part : trim the five last digits
	set entire [string range $res 0 [expr $len - 6]]
	# Decimal : take only the fifth digit from the right and divide by 16.
	set decimal [expr 0x[string range $res  [expr $len - 5] [expr $len - 5]]]
	set rep "[expr [format %1d 0x$entire]+[expr $decimal.0 / 16]] Mb"
    }
    return $rep
}

# -------------------------------------------------------------------------
# Get the date of an item. Prop is ascd (creation) or asmo (modification).
# Type is file, cfol, cdis or appf.
# Reply's form : aevt\ansr{'----':ldt (00000000B7445E5E)}
# -------------------------------------------------------------------------
proc mac::getItemDate {prop type item {l ""}} {
    if {[info tclversion] < 8.0} {
	set res [mac::findItemProperty $prop $type $item]
    } else {
	set res [AEBuild -r 'MACS' core getd ---- [tclAE::build::propertyObject $prop \
	  [tclAE::build::nameObject $type [tclAE::build::TEXT $item]]]]
	regexp {([0-9a-fA-F]+)} $res dum res
    }
    if {$l=="short"} {
	return [ISOTime::ISODate [format %1d 0x$res]]
    } else {
	return [ISOTime::ISODateAndTimeRelaxed [format %1d 0x$res]]
    }
}

# -------------------------------------------------------------------------
# Date comparison. Dates must be in the ISOTime format "YYYY-MM-DD"
# as returned by the ISOTime.tcl procs (separator is irrelevant).
# Returns 0 if datea < dateb, 1 if datea == dateb, 2 if datea > dateb
# -------------------------------------------------------------------------
proc mac::dateCompare {datea dateb} {
    regexp {(\d\d\d\d).(\d\d).(\d\d)} $datea dum yya mma dda
    regexp {(\d\d\d\d).(\d\d).(\d\d)} $dateb dum yyb mmb ddb
    if {[expr {$yya < $yyb} || {[expr $yya == $yyb] && [expr $mma < $mmb]} \
      || {[expr $yya == $yyb] && [expr $mma == $mmb] && [expr $dda < $ddb]}]} {return 0}
    if {[expr {$yya > $yyb} || {[expr $yya == $yyb] && [expr $mma > $mmb]} \
      || {[expr $yya == $yyb] && [expr $mma == $mmb] && [expr $dda > $ddb]}]} {return 2}
    return 1
}

# -------------------------------------------------------------------------
# Output the contents of an info window
# -------------------------------------------------------------------------
proc mac::sendTextInfo {} {
    global macinfotext
    new -n "* macMenu Info *" -info $macinfotext
    return 1
}
    
# -------------------------------------------------------------------------
# Update some variables when prefs are modified
# -------------------------------------------------------------------------
proc mac::shadowPrefs {name} {
    global macMenumodeVars mac_params predefext 
    global ispredef inimaccreatorslist maccreatorslist macchunksize
    set maccreatorslist $inimaccreatorslist
    if {$macMenumodeVars(additionalTypes)!=""} {
	foreach type $macMenumodeVars(additionalTypes) {
	    lappend maccreatorslist $type
	} 
    }
    set predefext $macMenumodeVars(predefExtensions)
    set predefext [string trimright $predefext]
    foreach e $predefext {
	set ispredef($e) 0
    } 
    set macchunksize $macMenumodeVars(chunksSize)
    set mac_params(overwrite) $macMenumodeVars(overwriteIfExists)
}

# -------------------------------------------------------------------------
# Rewrite the filtering expression when predefined extensions have been
# selected.
# -------------------------------------------------------------------------
proc mac::updateFilterExpr {} {
    global dialogargs dialogtitle dialogy mac_params predefext ispredef
    set extlist ""
    foreach e [array names ispredef] {
	if {$ispredef($e)} {
	    lappend extlist $e
	} 
    } 
    if {$mac_params(otherexts)!=""} {
	regsub -all {  +} $mac_params(otherexts) " " str
	set str [string trim $str]
	set others [split $str]
	foreach e $others {
	    lappend extlist $e
	} 
    } 
    if {$extlist!=""} {
	set mac_params(regex) ".*\\.\([join $extlist |]\)$"
    } 
}

# -------------------------------------------------------------------------
# Check for errors returned in the Apple Event reply
# -------------------------------------------------------------------------
proc mac::testError {res {addinfo ""}} {
    global tileLeft tileTop tileWidth errorHeight
    if {[regexp {err(s|n):} $res]} {
	regsub {aevt\\ansr\{.*errs:} $res "" res
	regsub {\}$} $res "" res
	new  -g $tileLeft $tileTop [expr $tileWidth*.6] [expr $errorHeight] \
	  -n "* Error Info *" -info "Apple Event error message :\n$res\n$addinfo" 
	return 1
    } else {
	return 0
    }
}


# -------------------------------------------------------------------------
# Break long path names in the info windows
# -------------------------------------------------------------------------
proc mac::pathLine {dir} {
    global dialogargs dialogy interv
    if {[expr [string length $dir] < 30]} {
	set y $dialogy
	eval lappend dialogargs [list [dialog::text "$dir: " 190 y ]]
	return
    } 
    set line [file split $dir]
    set line [lreplace $line 0 0 [string trimright [lindex $line 0] :]]
    set result ""
    set i 0
    while {[string length $result] < 30} {
	set result [join [lrange $line 0 $i] :]
	incr i
    }
    set y $dialogy
    eval lappend dialogargs [list [dialog::text "[join [lrange $line 0 [expr $i-1]] :]: " 190 y ]]
    set next [join [lrange $line $i end] :]
    if {[llength $next]!=0} {
	set dialogy [expr $dialogy + 16]
	mac::pathLine $next
    } 
}

# -------------------------------------------------------------------------
# Associate one of the letters m, c, s, k, l, v, o to a sorting option
# for "files list". An improper or empty 'letter' will set 
# mac_params(sortbyidx) to 0 which means no sorting.
# -------------------------------------------------------------------------
proc mac::getSortingOption {letter} {
    global mac_params
    if {[set pos [lsearch -exact [list "" "" m c s k l v o] [string range $letter 0 0]]]=="-1"} {
	set pos 0
    } 
    set mac_params(sortbyidx) $pos
}