## -*-Tcl-*-  (nowrap)
 # ==========================================================================
 #  FILE: "mailMenu.tcl"
 # 
 #                                    created: 04/24/1996 {12:08:43 PM} 
 #                                last update: 10/31/2001 {12:20:08 PM}
 # Author: Pete Keheler
 # 
 # Includes contributions from Juan Falguaras, Vince Darley, Jon Guyer,
 # and others.
 # 
 # Since Tcl 8.x provides a mail package in tcllib, we are rewriting this
 # with a plug-in mail architecture, so that we can use either Eudora or Tcl
 # or ...  
 # 
 # Version 0.2 includes the first steps.
 # 
 # ==========================================================================
 # 
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #   Redistributions of source code must retain the above copyright
 #    notice, this list of conditions and the following disclaimer.
 # 
 #   Redistributions in binary form must reproduce the above copyright
 #    notice, this list of conditions and the following disclaimer in the
 #    documentation and/or other materials provided with the distribution.
 # 
 #   Neither the name of Alpha/Alphatk nor the names of its contributors may
 #    be used to endorse or promote products derived from this software
 #    without specific prior written permission.
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # DAMAGE.
 # 
 # ==========================================================================
 # 
 # TO DO:
 # 
 # Take the folders (Not ended in " Folder") from the Eudora Folder
 #   (recursively) to build the MailBoxes.  Not ask for it to Eudora via AE.
 #     
 # Version History (abridged):
 # 
 #  modified by  rev    reason
 #  -------- --- ------ -----------
 #  08/07/98 JF  1.1     Added eudoraNicknamesFolder pref var to hold the path
 #                         to the Eudora "Nicknames Folder" and added this
 #                         folder in the scanning for "euNicknames" array
 #                         (look for # JF)
 #  08/21/98 JF  1.2     Rewrite of finishNickname with 'prompt::statusLineComplete'
 #   			   and deleting substituteNickname
 #    
 # Eudora Menu 1.2 became Mail Mode (w/ Mail Menu) 0.1
 #    
 #  ??/??/?? VD  0.1.7   Jon updated various apple-events issues, Vince fixed
 #                         some problems for Alphatk.
 #  ??/??/?? VD  0.2.0   There are now three possible plug-ins: eudora
 # 			   (works as before with apple-events), tclmail (not
 # 			   really implemented, but can use the smtp library
 # 			   from tcllib to send mail directly), and 'system',
 # 			   which simply uses your system configuration for
 # 			   'mailto:' urls to send mail.
 #  06/30/01 cbu 0.2.1   More support for user defined prefix string.
 #                       More support for user defined colors.
 #                       'set newDocTypes...' here instead of newDocuments.tcl.
 # 
 # ==========================================================================
 # 
 # Class message:	A message
 # Elements:		field by name
 # Properties:
 # 
 #  "body"       <string>        -- the body of the message
 #  "priority"   <integer>       -- the priority
 #  "label"      <integer>       -- the index of the label
 #  "status"     <unread/already read/replied/forwarded/redirected/
 #               not sendable/sendable/queued/sent/never sent>
 #                               -- the message status
 #  "sender"     <string>        -- the sender as appearing in the message summary
 #  "date"       <string [r/o]>  -- the date as appearing in the message summary
 #  "subject"    <string>        -- the subject as appearing in the message summary
 #  "size"       <integer [r/o]> -- the size of the message
 #  "outgoing"   <boolean [r/o]> -- is the message is outgoing?
 #  "signature"  <none/standard/alternate>
 #  
 #                               -- which signature the message should have
 #  "QP"             <boolean>   -- is Eudora allowed to encode text?
 #  "return receipt" <boolean>   -- is a return receipt is requested?
 #  "wrap"           <boolean>   -- should the text be wrapped when sent?
 #  "tab expansion"  <boolean>   -- should tabs get expanded to spaces?
 #  "keep copy"      <boolean>   -- should a copy should be kept after message
 #                                  is sent?
 #  "preserve macintosh info"  <boolean> -- should Macintosh information
 #                                          always be sent with attachments?
 #  "attachment encoding"      <AppleDouble/AppleSingle/BinHex/uuencode> 
 #                                       -- the type of encoding to use for
 #                                          attachments
 #  "show all headers"         <boolean> -- should all headers be visible?
 #  "transliteration table"    <integer> -- the resource id of the transliteration
 #                                          table
 #  "will be fetched"          <boolean> -- will the message be [re]fetched
 #                                          on next check?
 #  "will be deleted"          <boolean> -- will the message be deleted from
 #  					   server on next check?
 # ==========================================================================
 ##

alpha::mode Mail 0.2.2 mailMenu {} mailMenu {
    addMenu mailMenu 138 Mail
    set unixMode(rmail) {Mail}
    namespace eval browse {}
    set browse::GotoProc(*MAILBOX*) mailGotoMatch
    if {$tcl_platform(platform) == "macintosh" || \
      ($tcl_platform(platform) == "unix" && $tcl_platform(os) == "Darwin")} {
	alpha::package require tclAE 2.0b7
	eventHandler CSOm eNot "eudoraHandler"
    }
    set "eMailer(Mail Menu)" {Mail::newEmailWindow}
    menu::buildProc mailMenu Mail::buildMenu
    set Mail::handlers(Eudora) eudora
    if {[info tclversion] >= 8.0} {
	# This is direct SMTP access via tcllib, which could
	# eventually become something like emacs's 'rmail'.
	set Mail::handlers(tcllib) tclmail
    }
    set Mail::handlers(simple) mailsystem
    # Insert a new option in the 'New Document' prompt.
    set {newDocTypes(New Email Message)} mailNewMsg
    # Handler for mail services
    newPref var mailHandler "Eudora" global Mail::handlerChanged Mail::handlers array
    newPref v commentsContinuation 1 Mail "" \
      [list "only at line start" "spaces allowed" "anywhere"] index
} uninstall {
    this-file
} help {
    file "Mail Help"
} maintainer {
}

proc mailMenu {} {}

namespace eval Mail   {}
namespace eval ::Mail {}

if {![info exists trashName]} {set trashName "Trash"}

# Standard prefs used by various Alpha procs.

newPref flag    wordWrap                {1}     Mail
newPref flag    autoMark                {0}     Mail
newPref var     wordBreak               {[a-zA-Z0-9:]+}  Mail
newPref var     wordBreakPreface        {[^a-zA-Z0-9:]}  Mail

# These could probably use some explanation ...

newPref flag    tossOnQueue             {1}     Mail
newPref flag    switchOnQueue           {0}     Mail

# The character to use for quoted text.
newPref var     prefixString            {> }    Mail    {Mail::setCommentChars}
newPref folder  eudoraPrefFolder        [file join [file dirname $PREFS] "Eudora Folder"] Mail
newPref file    eudoraNicknames         ""      Mail
newPref folder  eudoraNicknamesFolder   ""      Mail

# Use this preference to locate your mail application.
newPref sig     mailSig                 CSOm

# The color of quoted text (as determined by the Prefix String preference.
newPref color 	quoteColor       	{red}   Mail    {Mail::colorizeMail}

# The color of Heading: keywords (such as Subject, To, etc)
newPref color 	headingColor     	{blue}  Mail    {Mail::colorizeMail}

newPref color  stringColor		{none}  Mail    {Mail::colorizeMail}

# This is essentially a 'dummy' call, so that all of any other regModeKeywords
# can be adds (using -a).
regModeKeywords -s $MailmodeVars(stringColor) Mail {}

set Mail::Keywords {{Subject:} {To:} {From:} {Cc:} {Date:} {Sender:}}

# Set (change) the comment characters for Mail mode.

proc Mail::setCommentChars {{pref ""}} {

    global MailmodeVars Mail::commentCharacters 
    global Mail::startPara Mail::endPara

    set pS [string trim $MailmodeVars(prefixString)]

    set Mail::commentCharacters(General)   "$pS "
    set Mail::commentCharacters(Paragraph) [list "$pS$pS " " $pS$pS" " $pS "]
    set Mail::commentCharacters(Box)       [list $pS 1 $pS  1 $pS 3]

    regModeKeywords -a -e $pS Mail {}

    set Mail::startPara "^\[ \t\]*([quote::Regfind $pS]|\$)"
    set Mail::endPara   "^\[ \t\]*([quote::Regfind $pS]|\$)"

    if {$pref != ""} {refresh}
}

# Call this now
Mail::setCommentChars

# ===========================================================================
# 
# Mail::colorizeMail Lisp.
# 
# Used to update preferences, and could be called in a <mode>Prefs.tcl file
# 

proc Mail::colorizeMail {{pref ""}} {
    
    global MailmodeVars Mail::Keywords
    
    set cC [string trim $MailmodeVars(prefixString)]
    
    regModeKeywords -a -e $cC -c $MailmodeVars(quoteColor) \
      -k $MailmodeVars(headingColor) Mail [set Mail::Keywords]

    if {$pref != ""} {refresh}
}

# Call this now.
Mail::colorizeMail

# bind tab
Bind 0x30   mailTab Mail

#===============================================================================
# Utility routines.
proc eudoraFolder {} { nameObject euMF {'TEXT'()} [nullObject] }
proc mailboxByName {name} { nameObject euMB "$name" [eudoraFolder] }
proc mailboxByIndex {ind} { indexObject euMB $ind [eudoraFolder] }
proc eudoraMessage {msg_id mailbox} { indexObject euMS $msg_id [mailboxByName $mailbox] }
proc mailboxProperty {prop mailbox} { objectProperty 'CSOm' $prop [mailboxByName $mailbox] }
proc messageProperty {prop msg_id mailbox} { objectProperty 'CSOm' $prop [eudoraMessage $msg_id $mailbox] }

# Get path-name for indicated mailbox
proc mailboxPathName {name} {
    extractPath [mailboxProperty euFS $name]
}

proc mailboxPathIndex {ind} {
    set res [objectProperty 'CSOm' euFS [mailboxByIndex $ind]]
    return [extractPath $res]
}

#===============================================================================

if {![info exists eudoraBoxes]} {
    set eudoraBoxes {In Out}
}

# JF 98/08/07
if {$MailmodeVars(eudoraNicknames) == ""} {
    if {[file exists [file join $MailmodeVars(eudoraPrefFolder) "Eudora Nicknames"]]} {
	set MailmodeVars(eudoraNicknames) [file join $MailmodeVars(eudoraPrefFolder) "Eudora Nicknames"]
    }
}
if {$MailmodeVars(eudoraNicknamesFolder) == ""} {
    if {[file exists [file join $MailmodeVars(eudoraPrefFolder) "Nicknames Folder"]]} {
	set MailmodeVars(eudoraNicknamesFolder) [file join $MailmodeVars(eudoraPrefFolder) "Nicknames Folder"]
    }
}



proc updateNicknames {{arg ""}} {
    global euNicknames MailmodeVars
    # JF 98/08/07
    if {(![info exists MailmodeVars(eudoraNicknames)] && ![info exists MailmodeVars(eudoraNicknamesFolder)])\
      || \
      (![file exists $MailmodeVars(eudoraNicknames)] && ![file exists $MailmodeVars(eudoraNicknamesFolder)])} {
	if {![string length $arg]} {
	    alertnote {Please locate the file "Eudora Nicknames" or/and folder "Nicknames Folder" using Mail mode's prefs.}
	}
	return
    }
    
    if {[file exists $MailmodeVars(eudoraNicknames)]} {
	set fd [open $MailmodeVars(eudoraNicknames)]
	foreach a [split [read $fd] "\n"] {
	    if {[llength $a] && [lindex $a 0] == "alias"} {
		set euNicknames([lindex $a 1]) [lindex $a 2]
	    }
	}
	close $fd
    }
    if {[string length $MailmodeVars(eudoraNicknamesFolder)] && [file isdirectory $MailmodeVars(eudoraNicknamesFolder)]} {
	foreach f [glob -dir $MailmodeVars(eudoraNicknamesFolder) *] {
	    set fd [open "$f"]
	    foreach a [split [read $fd] "\n"] {
		if {[regexp {\{\}\[\]} $a]} {
		    alertnote "ERROR: Braces in $f"
		    close $fd
		    return
		} else {
		    if {[llength $a] && [lindex $a 0] == "alias"} {
			set euNicknames([lindex $a 1]) [lindex $a 2]
		    }
		}
	    }
	    close $fd
	}
    }
}
updateNicknames quiet

proc Mail::newEmailWindow {to} {
    # This will handle 'to' created by 'url::mailto', as well as plain
    # email addresses.
    set args [list ]
    regexp {^(.+)\?(.+$)} $to allofit to args
    regsub {mailto:} $to {} to
    set mailFields  [join [split $args &]]]
    set emailFields [list cc bcc subject body]
    foreach emailField $emailFields {
	set efieldText ""
	foreach mailField $mailFields {
	    set pat "^${emailField}=(.*)$"
	    if {[regexp $pat $mailField allofit fieldText]} {
		set efieldText [quote::Unurl $fieldText]
		break
	    } 
	}
	lappend mailArgs $efieldText
    }
    eval mailNewMsg [list $to] $mailArgs
}

proc finishNickname {} {
    global euNicknames
    set pos [getPos]
    backwardWord
    if {[pos::compare [getPos] == [minPos]]} {
	set preMailNick ""
    } else {
	set preMailNick [getText [getPos] $pos]
    }
    goto $pos
    set s [prompt::fromChoices "Nick" $preMailNick -list [array names euNicknames]]
    if {$preMailNick != ""} {
	backwardDeleteWord
    }
    insertText $s
}

proc editNicknames {} {
    global MailmodeVars
    # JF 98/08/07
    if {![info exists MailmodeVars(eudoraNicknamesFolder)]} {
	edit -w "$MailmodeVars(eudoraNicknames)"
    } else {
	edit -w [getfile "Which of them?" "$MailmodeVars(eudoraNicknamesFolder)"]
    }
}


proc openMailbox {} {
    global eudoraBoxes eudoraLastFolder
    if {[info exists eudoraLastFolder]} {
	set fold $eudoraLastFolder
    } else {
	set fold [lindex $eudoraBoxes 0]
    }
    set eudoraLastFolder [prompt::fromChoices "Open mailbox" $fold -list $eudoraBoxes]
    mailBoxProc dummy $eudoraLastFolder
    prefs::modified eudoraLastFolder
}

proc moveToMailbox {} {
    global eudoraLastFolder trashName eudoraBoxes
    
    if {[info exists eudoraLastFolder]} {
	set fold $eudoraLastFolder
    } else {
	set fold $trashName
    }
    
    set folder [prompt::fromChoices "Move to" $fold -list $eudoraBoxes]
    if {[string length $folder]} {
	doFolderMove $folder
	set eudoraLastFolder $folder
	prefs::modified eudoraLastFolder
    } else {
	doFolderMove $fold
    }
}

#  generic handling  #

proc Mail::menuProc {menu item} {
    global mailSig
    switch -- $item {
	"eudora"		{app::launchFore $mailSig}
	"backgroundEudora"	{launch [nameFromAppl $mailSig]}
	"help"			{help::openFile "Mail Help"}
	"cc"			{mailCc}
	"bcc"			{mailBcc}
	"newMessage"		{mailNewMsg}
	"reply"			{mailReplymsg}
	"replyToAll"		{mailReplymsg 1}
	"send"			{Mail::action checkSystem; mailSendCreatedMsg}
	"flushOutbox"		{Mail::action checkSystem; Mail::action FlushOut}
	"checkForMail"		{Mail::action checkSystem; mailCheck}
	"startNotifying" 	{Mail::action checkSystem; startNotifying}
	"stopNotifying"		{Mail::action checkSystem; stopNotifying}
	"moveToTrash"		{mailTrashmsg}
	"Move To Trash"		{mailTrashmsg}
	"Create New"		{eudoraNewMailbox}
	"Delete"		{eudoraDeleteMailbox}
	default			$item
    }
}

proc mailFlagsProc {menu flag} {
    global mail$flag trashName
    
    if {$flag == "trashName"} {
	set trashName [prompt "Trash folder name:" $trashName]
	prefs::modified trashName
	return
    }
    
    
    set mail$flag [expr 1 - [set mail$flag]]
    prefs::modified mail$flag
    markMenuItem mailFlags $flag [set mail$flag]
}

proc Mail::buildMenu {} {
    global mailMenu mailHandler
    if {$mailHandler == "Eudora"} {
	set ma [list \
	  "<SbackgroundEudora" \
	  "<Seudora"]
    } else {
	set ma [list]
    }
    
    eval lappend ma [list \
      [menu::makeFlagMenu mailHandler list mailHandler] \
      "help" \
      "\(-" \
      "/e<U<BnewMessage" \
      "cc" \
      "bcc" \
      "\(-" \
      "/f<U<Bsend"]
    
    set subs [list]
    if {$mailHandler == "Eudora"} {
	eval lappend ma [list \
	  "/g<U<S<Breply" \
	  "/g<U<I<S<BreplyToAll" \
	  "\(-" \
	  "/,<SfinishNickname" \
	  "editNicknames" \
	  "updateNicknames" \
	  "\(-" \
	  "updateMailboxLists" \
	  {Menu -n open -p mailBoxProc -m {}} \
	  {Menu -n moveTo -p transferProc -m {}} \
	  "\(-" \
	  {Menu -n tellEudora -p Mail::menuProc {
	      "/k<UflushOutbox"
	      "/l<UcheckForMail"
	      "(-"
	      startNotifying
	      stopNotifying}
	  } \
	  [list Menu -n mailFlags -p mailFlagsProc {}]]
	lappend subs mailBoxMenu moveToMenu mailFlagsMenu
    }
    
    return [list build $ma Mail::menuProc \
      $subs $mailMenu]
}

if {![info exists mailflushOnCheck]} {set mailflushOnCheck 1}
if {![info exists mailalertOnIncoming]} {set mailalertOnIncoming 1}
if {![info exists mailimmediateSend]} {set mailimmediateSend 0}

menu::buildProc mailBoxMenu Mail::buildMailBoxMenu
menu::buildProc moveToMenu Mail::buildMoveToMenu
menu::buildProc mailFlagsMenu Mail::buildMailFlagsMenu

proc Mail::buildMailBoxMenu {} {
    global eudoraBoxes
    set ma [concat [list "/h<U<BOpen Mailbox" "\(-"] $eudoraBoxes]
    return [list build $ma mailBoxProc "" open]
}

proc Mail::buildMoveToMenu {} {
    global eudoraBoxes
    set ma [concat [list "/i<U<BMove To Trash" "/j<U<BMove To Mailbox" "\(-"] $eudoraBoxes]
    return [list build $ma transferProc "" moveTo]
}

proc Mail::buildMailFlagsMenu {} {
    global mailflushOnCheck mailalertOnIncoming mailimmediateSend
    foreach m [list flushOnCheck alertOnIncoming immediateSend] {
	if {[set mail$m]} {
	    lappend ma "!$m"
	} else {
	    lappend ma $m
	}
    }
    lappend ma "--" trashName
    return [list build $ma mailFlagsProc "" mailFlags]
}

menu::buildSome mailMenu

proc Mail::handlerChanged {to} {
    menu::buildSome mailMenu
}

proc mailSendCreatedMsg {} {
    global MailmodeVars mailimmediateSend
    
    set res [search -s -f 1 -r 0 {>==text follows this line==<} [minPos]]
    set header [getText [minPos] [lindex $res 0]]
    
    if {![regexp {To: ([^\r\n]*)} $header dummy to]} {set to ""}
    if {![regexp {[Cc]+: ([^\r\n]*)} $header dummy cc]} {set cc ""}
    if {![regexp {Subject: ([^\r\n]*)} $header dummy subject]} {set subject ""}
    
    set body [getText [pos::math [lindex $res 1] + 1] [maxPos]]
    
    Mail::action CreateMsg
    Mail::action SetField to $to
    Mail::action SetField cc $cc
    Mail::action SetField subject $subject
    Mail::action SetField body $body
    
    if {$MailmodeVars(tossOnQueue)} {
	setWinInfo dirty 0
	killWindow
    }
    
    Mail::action QueueLast
    
    if {$MailmodeVars(switchOnQueue)} {
	switchTo [Mail::action checkSystem]
    }
    
    if {$mailimmediateSend} {
	Mail::action FlushOut
    } else {
	message "Message queued"
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "Mail::action" --
 # 
 #  Current actions are as follows:
 #  
 #  checkSystem: 
 #    takes no arguments, and ensures the current mail handler is
 #    available.  Returns a mail-handler specific piece of information
 #    which can be used to identify the handler to a greater level of
 #    detail (e.g. 'eudora' returns the name of the actual application,
 #    'tcllib' returns nothing)
 #  
 #  xxx:
 #    yyy.
 #    
 # -------------------------------------------------------------------------
 ##
proc Mail::action {args} {
    global mailHandler Mail::handlers
    if {[info tclversion] < 8.0} {
	set ns [set Mail::handlers($mailHandler)]
	set cmd [lindex $args 0]
	eval ${ns}::${cmd} [lrange $args 1 end]
    } else {
	namespace eval ::$Mail::handlers($mailHandler) $args
    }
}

proc mailCc {} {
    if {![catch {set res [search -s -f 1 -r 1 -i 1 {cc:.*} [minPos]]}]} {
	goto [lindex $res 0]
	endOfLine
    } else {
	beginningOfBuffer
	nextLine
	insertText "Cc: \r"
	backwardChar
    }
}

proc mailBcc {} {
    if {![catch {set res [search -s -f 1 -r 1 -i 1 {bcc:.*} [minPos]]}]} {
	goto [lindex $res 0]
	endOfLine
    } else {
	beginningOfBuffer
	nextLine
	insertText "Bcc: \r"
	backwardChar
    }
}

#  tcl mail handling  #

namespace eval tclmail {}

proc tclmail::checkSystem {} {
    ::package require smtp
}

#====================================================================#
# sendmail procedure (created from code: tSTMPmail.tcl) 25-06-99
# Vince copied from comp.lang.tcl.  Will need to make significant
# modifications.
#====================================================================#

# highlevel command to email message to the logged in user

proc tclmail::send_message { subject {message ""} } {
    set email $login::user(EMAIL_ADDRESS)
    if {$email == ""} {
	set email "pc@acs.co.nz"
    }
    sendmail $email $email -subject $subject -body $message
}

#====================================================================#

# the arguments to "sendmail" are the options below
# eg: -subject value, sets op(subject) $value

# eg: sendmail pc pc@acs.co.nz -body "test message"
# note: body & subject default to "none" if not specified

proc tclmail::sendmail { from to args } {
    # set default options
    # wingate95 ip address for host (quicker than hostname for linux box)
    set op(smtphost) "192.168.3.8"
    set op(port) 25
    set op(from) $from
    set op(authuser) "acs.co.nz"
    set op(to) $to
    set op(subject) "none"
    set op(timeout) 5
    set op(blocking) 1
    set op(body) "none"

    # process command line arguments
    foreach {option value} $args {
	set option [string range $option 1 end]
	set op($option) $value
    }

    # now establish connection
    set sock [socket $op(smtphost) $op(port)]
    fconfigure $sock -blocking $op(blocking)
    bs_CheckMailResponse $sock $op(timeout) 220

    # authenticate..
    bs_TalkToSm $sock $op(timeout) "HELO $op(authuser)" 250
    bs_TalkToSm $sock $op(timeout) "MAIL FROM: <$op(from)>" 250

    # to's
    bs_TalkToSm $sock $op(timeout) "RCPT TO: <$op(to)>" 250

    # data
    bs_TalkToSm $sock $op(timeout) "DATA" 354

    # now send email (header, body etc)
    bs_TalkToSm $sock $op(timeout) "To: $op(to)" 0
    bs_TalkToSm $sock $op(timeout) "From: $op(from)" 0
    bs_TalkToSm $sock $op(timeout) "Date: [clock format [clock seconds] -format "%a, %e %b %Y %T"] +1200" 0
    bs_TalkToSm $sock $op(timeout) "Subject: $op(subject)" 0

    bs_TalkToSm $sock $op(timeout) "\n$op(body)" 0
    bs_TalkToSm $sock $op(timeout) "." 250

    # now close connection
    bs_TalkToSm $sock $op(timeout) "QUIT" 221
    catch {close $sock}
}

#====================================================================#

# handles a complete smtp transaction

proc tclmail::bs_TalkToSm { sock time_out what should } {
    puts $sock $what
    flush $sock
    if {$should != 0} {
	if {[catch {bs_CheckMailResponse $sock $time_out $should} e_info]} {
	    error "Response Err When sending -- $what -- : $e_info"
	}
    }
}

#====================================================================#

# looks for returns within timeout. fblocking might show up strange
# on some platforms...

proc tclmail::bs_CheckMailResponse {sock time_out should} {
    set res ""
    set ttime 0
    while {1} {
	set ttt [time {set res [gets $sock]}]
	if {$res != ""} break
	incr ttime 100
	after 100
	if {$ttime > $time_out } {
	    error "tSMTPmail: Err When Sending: Exceeded Timeout of $time_out Sec."
	    break
	}
    }
    if {[lindex $res 0] != $should} {
	error "Wrong Response: Should be $should, is [lindex $res 0]\n"
	return 1
    }
    return 0
}


#  simple mail system handling  #

namespace eval mailsystem {}

proc mailsystem::checkSystem {} {
    return 1
}

proc mailsystem::CreateMsg {} {
    global mailsystem::msgInfo
    catch {unset mailsystem::msgInfo}
}

proc mailsystem::SetField {fld to} {
    global mailsystem::msgInfo
    set mailsystem::msgInfo($fld) $to
}

proc mailsystem::QueueLast {} {
    global mailsystem::msgInfo
    set address [set mailsystem::msgInfo(to)]
    unset mailsystem::msgInfo(to)
    set url [eval [list url::mailto $address] [array get mailsystem::msgInfo]]
    url::execute $url
}

#  eudora mail handling  #

namespace eval eudora {}

proc eudora::checkSystem {} {
    global mailSig
    set name [nameFromAppl $mailSig]
    launch $name
    return [file tail $name]
}

proc eudora::FlushOut {} {
    global MailmodeVars
    message "Telling Eudora to flush messages"
    
    AEBuild 'CSOm' CSOm eCon eSen bool(01) eChk bool(00)
    message ""
}


# make message at end of mailbox "out" of mail folder ""
proc eudora::CreateMsg {} {
    createThingAtEnd 'CSOm' [mailboxByName Out] euMS
}

# set field "fld" of message 0 to "to"
proc eudora::SetField {fld to} {
    if {$fld == "body"} { set fld "" }
    AEBuild -r 'CSOm' core setd {----} "obj \{form:name, want:type(euFd), seld:$fld, from:obj \{form:indx, want:type(euMS), seld:1, from:'null'()\}\}" data "$to"
}


proc eudora::QueueLast {} {
    AEBuild -r 'CSOm' CSOm eQue ---- {obj {form:indx, want:type(euMS), seld:1, from:'null'()}}
}

#  Not yet dealt with  #

proc mailBoxProc {menu item} {
    global inboxMembers ALPHA eudoraBoxes
    global tileLeft tileTop tileHeight errorHeight defWidth
    
    if {$item == "Open Mailbox"} {
	return [openMailbox]
    }
    
    killMailboxWindow
    
    Mail::action checkSystem
    switchTo $ALPHA
    
    global trashedMsgs$item
    set trashedMsgs$item {}
    
    set inboxMembers {}
    set text {}
    set ind 1
    foreach msg [mailSenders $item] {
	set from [mailAddr [lindex $msg 0]]
	set tag {}
	while {[lsearch $inboxMembers $from$tag] >= 0} {
	    if {![string length $tag]} {
		set tag { <2>}
	    } else {
		regexp {[0-9]+} $tag tag
		set tag " <[expr $tag + 1]>"
	    }
	}
	append text [format "%-40s : %s\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$item$ind\r" "$from$tag" [lindex $msg 1]]
	lappend inboxMembers $from$tag
	incr ind
    }
    if {![string length $text]} {
	alertnote "No messages in '$item'!"
	return
    }
    
    new -n "* MAILBOX '$item' *" -g $tileLeft $tileTop $defWidth $errorHeight -m Brws
    insertText "(<cr> to go to message)\r-----\r$text"
    
    select [nextLineStart [nextLineStart [minPos]]] [nextLineStart [nextLineStart [nextLineStart [minPos]]]]
    winReadOnly
    message ""
}

proc eudoraNewMailbox {} {
}


proc eudoraDeleteMailbox {} {
}


proc killMailboxWindow {} {
    if {[set ind [lsearch [winNames] {*MAILBOX*}]] >= 0} {
	set win [lindex [winNames] $ind]
	bringToFront $win
	killWindow
    }
}


proc mailGotoMatch {} {
    if {[regexp {(.+)([0-9]+)} [getText [lineStart [getPos]] [nextLineStart [getPos]]] dummy folder ind]} {
	global trashedMsgs$folder
	
	set deleted 0
	for {set i 1} {$i < $ind} {incr i} {
	    if {[lsearch [set trashedMsgs$folder] $i] >= 0} {
		incr deleted
	    }
	}
	mailOpen $folder [expr {$ind - $deleted}] $ind
    }
}


proc mailAddr {name} {
    if {![regexp {<(.*)>} $name dummy addr]} {
	regexp {^[^ ]+} $name addr
    }
    return $addr
}


proc mailSenders {folder} {
    set cnt [mailCountMsgs $folder]
    set msgs {}
    for {set i 1} {$i <= $cnt} {incr i} {
	set subject {}
	if {![regexp -nocase {From: (.*)} [mailGetField from $folder $i] dummy from]} {
	    error "No from field!"
	}
	regexp {Subject: (.*)} [mailGetField subject $folder $i] dummy subject
	lappend msgs [list $from $subject]
    }
    return $msgs
}	


proc mailCheck {} {
    global MailmodeVars mailflushOnCheck
    message "Told Eudora to check for new mail"
    
    AEBuild 'CSOm' CSOm eCon eSen bool(0$mailflushOnCheck) eChk bool(01)
}


proc mailReplymsg {{toall 0}} {
    global inboxMembers MailmodeVars

    set prefix $MailmodeVars(prefixString)
    set result [search -s -f 1 -r 1 {^>==} [minPos]]
    set pos0   [lindex $result 0]
    set pos1   [nextLineStart $pos0]
    set header [getText [minPos] $pos1]
    
    set from [mailGetFrom]
    if {![regexp {Subject: ([^\r\n]*)} $header dummy subject]} {set subject ""}
    set body [getText $pos1 [maxPos]]
    
    regsub -all "\[\r\n\]" $body "\r$prefix" body
    mailNewMsg $from "" "" "Re: $subject" "\r$prefix$body\r"
}

proc transferProc {menu item} {
    if {$item == "Move To Trash"} {
	mailTrashmsg
    } elseif {$item == "Move To Mailbox"} {
	moveToMailbox
    } else {
	doFolderMove $item
    }
}


proc mailTrashmsg {} {
    global trashName
    doFolderMove $trashName
}

proc doFolderMove {toFolder} {
    # Is this a summary or msg window?
    if {[regexp "MAILBOX" [win::CurrentTail]]} {
	if {![regexp {(.*)([0-9]+)} [getText [getPos] [nextLineStart [getPos]]] dummy folder orig]} {
	    beep
	    return
	}
	if {$toFolder == $folder} return
	global trashedMsgs$folder
	
	set deleted 0
	for {set i 1} {$i < $orig} {incr i} {
	    if {[lsearch [set trashedMsgs$folder] $i] >= 0} {
		incr deleted
	    }
	}
	set number [expr {$orig - $deleted}]
	set summary 1
    } else {
	set pos [lindex [search -s -f 1 -r 1 {^Msg} [minPos]] 0]
	set text [getText $pos [nextLineStart $pos]]
	regexp {"([^"]+)" \(([0-9]+)\).*"([^"]+)"} $text dummy number orig folder
	if {$toFolder == $folder} return
	set summary 0
	
	global trashedMsgs$folder
    }
    
    message "Moving msg $number ($orig) of folder '$folder' to '$toFolder'"
    moveMsg $number $folder $toFolder
    if {!$summary} killWindow
    
    lappend trashedMsgs$folder $orig
    
    # Find summary info and delete it
    set win [win::CurrentTail]
    if {[regexp "MAILBOX" $win]} {
	setWinInfo read-only 0
	
	set inds [search -s -f 1 -r 1 "$folder$orig\$" [minPos]]
	set pos [lindex $inds 0]
	deleteText [lineStart $pos] [nextLineStart $pos]
	
	if {[string length [search -s -n -f 1 -r 0 {} [minPos]]]} {
	    setWinInfo dirty 0
	    setWinInfo read-only 1
	    
	    nextLine
	    browse::Up
	} else {
	    setWinInfo dirty 0
	    killWindow
	}
    }
}

	
proc mailGetFrom {} {
    set res [search -s -f 1 -r 1 {>==} [minPos]]
    set header [getText [minPos] [lindex $res 0]]
    if {[regexp {From: ([^\r\n]*)} $header dummy from]} {
	return [mailAddr $from]
    }
    error "No from line"
}

proc mailNewMsg {{to ""} {cc ""} {bcc ""} {subject ""} {body ""}} {
    new -n "New Mail" -m Mail \
      -text "To: $to\rSubject: $subject\r>==text follows this line==<\r$body" -dirty 0
    goto [minPos]
    refresh
    mailTab
}

proc mailTab {} {
    global Mail::Keywords
    if {[catch {search -s -f 1 -r 1 -i 1 "([join [set Mail::Keywords] {|}]|^>)" [getPos]} res]} {
	insertText "\t"
	return
    }
    if {[lookAt [lindex $res 0]] == ">"} {
	select [nextLineStart [lindex $res 1]] [maxPos]
    } else {
	goto [pos::math [lindex $res 1] + 1]
	endLineSelect
    }
}

proc mailOpen {folder i originalNum} {
    global tileLeft tileTop tileHeight trashedMsgs$folder defWidth
    
    set from [mailGetField From $folder $i]
    set to [mailGetField To $folder $i]
    set subject [mailGetField Subject $folder $i]
    set sender [mailGetField Sender $folder $i]
    set date [mailGetField Date $folder $i]
    set cc [mailGetField cc $folder $i]
    set contents [mailGetField "" $folder $i]
    
    #===============================================================================
    global tileHeight tileTop tileLeft errorHeight errorDisp defWidth
    set top $tileTop
    set geo [getGeometry]
    if {([lindex $geo 0] != $tileLeft) || ([lindex $geo 1] != $top) || ([lindex $geo 2] != $defWidth) || ([lindex $geo 3] != $errorHeight) } {
	moveWin $tileLeft $top
	sizeWin $defWidth $errorHeight
    }
    set mar 28
    incr top [expr {$errorHeight + $mar}]
    #===============================================================================
    set name [new -n "MAIL" -g $tileLeft $top $defWidth [expr {$errorDisp - 5}] -m Mail]
    
    set text {}
    if {[string length $subject]} {append text "$subject\r"}
    if {[string length $to]} {append text "$to\r"}
    if {[string length $from]} {append text "$from\r"}
    if {[string length $date]} {append text "$date\r"}
    if {[string length $cc]} {append text "$cc\r"}
    if {[string length $sender]} {append text "$sender\r"}
    append text "Msg \"$i\" ($originalNum) of mailbox \"$folder\"     Reply     Trash\r"
    insertText "${text}>===============================================================================
    $contents
    "
    goto [minPos]
    goto [lindex [search -s -f 1 -m 0 -r 1 {^Msg "} [minPos]] 0]
    set pos [getPos]
    regexp -indices {([0-9]+).*mailbox "(.+)".*(Reply).*(Trash)} [getText $pos [nextLineStart $pos]] dummy num fold rep tra
    text::color [pos::math $pos + [lindex $num 0]] [pos::math $pos + 1 + [lindex $num 1]] bold
    text::color [pos::math $pos + [lindex $fold 0]] [pos::math $pos + 1 + [lindex $fold 1]] bold
    hyperSelectPos [pos::math $pos + [lindex $rep 0]] [pos::math $pos + 1 + [lindex $rep 1]] "mailReplymsg"
    hyperSelectPos [pos::math $pos + [lindex $tra 0]] [pos::math $pos + 1 + [lindex $tra 1]] "mailTrashmsg"
    winReadOnly
    refresh
}

proc hyperSelectPos {from to text} {
    if {[pos::compare $from == $to]} {
	beep
	return
    }
    text::color $from $to 3
    text::hyper $from $to $text
}

proc mailFixMenus {} {
    set ins {}
    set outs {}
    
    set cnt [mailCountMsgs in]
    for {set i 1} {$i <= $cnt} {incr i} {
	lappend ins [mailGetField from in $i]
    }
    Menu -n in {}
    foreach item $ins {
	addMenuItem -l "" in $item
    }
    
    set cnt [mailCountMsgs out]
    for {set i 1} {$i <= $cnt} {incr i} {
	lappend outs [mailGetField from out $i]
    }
    Menu -n out {}
    foreach item $outs {
	addMenuItem -l "" out $item
    }
}

# Move msg w/ specified index between folders, including to Trash.
proc moveMsg {msg infolder outfolder} {
    AEBuild -r 'CSOm' core move {----} "obj \{form:indx, want:type(euMS), seld:$msg, from:obj \{form:name, want:type(euMB), seld:$infolder, from:obj \{form:name, want:type(euMF), seld:'TEXT'(), from:'null'()\}\}\}" {insh} "insl\{kobj:obj \{form:name, want:type(euMB), seld:$outfolder, from:obj \{form:name, want:type(euMF), seld:'TEXT'(), from:'null'()\}\}, kpos:end \}"
}


proc mailGetField {field folder msg} {
    if {[catch {getObjectData 'CSOm' euFd $field [eudoraMessage $msg $folder]} res]} {
	return ""
    } else {
	return $res
    }
}

# obj {want:type('prop'), from:obj {form:indx, want:type(euMS), seld:$msg, from:obj {form:name, want:type(euMB), seld:$folder, from:obj {form:name, want:type(euMF), seld:'TEXT'(), from:'null'()}}}, form:'prop', seld:type('euST')}

proc mailMsgStatus {folder msg} {
    if {[catch {AEBuild -r 'CSOm' core getd {----} "obj \{want:type('prop'), from:obj \{form:indx, want:type(euMS), seld:$msg, from:obj \{form:name, want:type(euMB), seld:$folder, from:obj \{form:name, want:type(euMF), seld:'TEXT'(), from:'null'()\}\}\}, form:'prop', seld:type('euST')\}"} text]} {
	return ""
    }
    set from [string first "" $text]
    set to [string first "" $text]
    return [string range $text [expr {$from + 1}] [expr {$to - 1}]]
}


# CSOm\eAtc{'----':obj {form:indx, want:type(euMS), seld:1, from:'null'()}, eDcl:[fss (486172643A4465736B746F7020466F6C6465723A62756773)]}
proc mailAttachmentDoesntWork {name} {
    AEBuild -r 'CSOm' CSOm eAtc ---- {obj {form:indx, want:type(euMS), seld:1, from:'null'()}} eDcl [makeAlis $name]
}

# core\setd{'----':obj {form:prop, want:type(prop), seld:type(euPY), from:obj {form:indx, want:type(euMS), seld:1, from:'null'()}}, data:1}
proc mailSetNumberProperty {prop to} {
    AEBuild -r 'CSOm' core setd ---- "obj \{form:prop, want:type(prop), seld:type($prop), from:obj \{form:indx, want:type(euMS), seld:1, from:'null'()\}\}" data $to
}	


proc mailCountMsgs {mbox} {
    return [countObjects 'CSOm' [mailboxByName "$mbox"]  euMS]
}

proc mailCountMailboxes {} {
    return [countObjects 'CSOm' [eudoraFolder] euMB]
}
	

proc updateMailboxLists {} {
    set num [mailCountMailboxes]
    for {set i 1} {$i <= $num} {incr i} {
	set name [file tail [mailboxPathIndex $i]]
	message $name
	lappend boxes $name
    }
    
    global eudoraBoxes
    prefs::modified eudoraBoxes
    set eudoraBoxes $boxes
    Menu -n open -p mailBoxProc -m $eudoraBoxes
    Menu -n moveTo -p transferProc -m $eudoraBoxes
    message "done."
}


#===============================================================================

# a sample 'msgs': obj {want:type(euMS), from:obj {want:type(euMB), from:obj {want:type(euMF), from:'null'(), form:name, seld:'TEXT'()}, form:name, seld:In}, form:indx, seld:18}, obj {want:type(euMS), from:obj {want:type(euMB), from:obj {want:type(euMF), from:'null'(), form:name, seld:'TEXT'()}, form:name, seld:In}, form:indx, seld:19}
proc eudoraHandler {it} {
    global blah mailalertOnIncoming
    message ""
    if {[regexp {eWHp:wArv.*\[(obj.*)\], &repq} $it dum1 msgs]} {
	set ids [getMsgIDs $msgs]
	if {$mailalertOnIncoming} {
	    mailBrowser In $ids
	}
    } else {
	message "No mail"
    }
}


proc mailBrowser {folder ids} {
    global inboxMembers ALPHA
    global tileLeft tileTop defWidth tileHeight errorHeight
    
    Mail::action checkSystem
    switchTo $ALPHA
    
    global trashedMsgs$folder
    set trashedMsgs$folder {}
    
    set inboxMembers {}
    set text {}
    
    foreach id $ids {
	regexp {From: (.*)} [mailGetField from $folder $id] dummy from
	set from [mailAddr $from]
	set subject {}
	regexp {Subject: (.*)} [mailGetField subject $folder $id] dummy subject
	set tag {}
	while {[lsearch $inboxMembers $from$tag] >= 0} {
	    if {![string length $tag]} {
		set tag { <2>}
	    } else {
		regexp {[0-9]+} $tag tag
		set tag " <[expr {$tag + 1}]>"
	    }
	}
	append text [format "%-40s : %s\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\tIn$id\r" "$from$tag" $subject]
	lappend inboxMembers $from$tag
    }
    if {![string length $text]} {
	alertnote "No messages!"
	return
    }
    
    if {[set ind [lsearch [winNames] "*Incoming MAILBOX*"]] >= 0} {
	bringToFront [lindex [winNames] $ind]
	setWinInfo read-only 0
	endOfBuffer
	insertText $text
    } else {
	killMailboxWindow
	new -n "* Incoming MAILBOX *" -g $tileLeft $tileTop $defWidth $errorHeight -m Brws
	
	insertText "(<cr> to go to message)\r-----\r$text"
    }
    select [nextLineStart [nextLineStart [minPos]]] [nextLineStart [nextLineStart [nextLineStart [minPos]]]]
    winReadOnly
    message ""
}


proc getMsgIDs {text} {
    if {[regexp -indices {seld:([0-9]+)} $text dummy ind]} {
	return [concat [string range $text [lindex $ind 0] [lindex $ind 1]] [getMsgIDs [string range $text [lindex $ind 1] end]]]
    }
}

proc startNotifying {} {
    global HOME ALPHA quitHooks
    
    AEBuild 'CSOm' CSOm nIns ---- [makeAlis [file join $HOME $ALPHA]]
}

proc stopNotifying {} {
    global HOME ALPHA
    
    AEBuild 'CSOm' CSOm nRem ---- [makeAlis [file join $HOME $ALPHA]]
}




