#=============================================================================
# Browser mode.
# 
# Alpha cannot do batch searches without this file
#=============================================================================

alpha::mode Brws 13.2 dummyBrws {} {
} {} help {
    During a batch search, a list of all matching lines are displayed in
    a Brws window.  By using the arrow keys and the return key, you can
    easily jump to the correct file and line of the match you desire.
    See the "Browser-Example" for a demonstration -- click on any
    fileset which appears in the dialog to see a Browser window.

    This mode is not intended for text editing.
}


namespace eval browse {}

Bind '\r'	{browse::Select [getPos];browse::Goto}  Brws
Bind enter	{browse::Select [getPos];browse::Goto}  Brws
ascii 0x3  	{browse::Select [getPos];browse::Goto}  Brws
Bind down 	browse::Down Brws
Bind up 	browse::Up   Brws
Bind 'n' <z>	browse::Down Brws
Bind 'p' <z>	browse::Up   Brws
ascii 0x20	browse::Down Brws
ascii 0x8	browse::Up   Brws
# this was below.  do we need it?
Bind 'c' <Cz>	{browse::Select [getPos];browse::Goto}

proc dummyBrws {} {}

# Set this to 1 to test dynamic code
if {${alpha::platform} == "alpha"} {
    set browse::enableDynamic 0
} else {
    set browse::enableDynamic 1
}

proc browse::lookUpToInfinity {loc} {
    # skip top two lines for browser message and divider
    set limit [nextLineStart [nextLineStart [minPos]]]
    
    set loc [lineStart $loc]
    
    set ind1 -1
    while {$ind1 < 0} {
	set loc [pos::math $loc - 1]
	set text [getText [lineStart $loc] [nextLineStart $loc]]
	set ind1 [string first "" $text]
	set loc [lineStart $loc]
	if {[pos::compare $loc <= $limit]} {
	    break
	}
    }
    
    if {$ind1 < 0} {
	error "can't see to infinity"
    } else {
	return $loc
    }
}

proc browse::lookDownToInfinity {loc} {
    set limit [pos::math [maxPos] - 1]
    
    set loc [lineStart $loc]
    
    set ind1 -1
    while {$ind1 < 0} {
	set text [getText $loc [nextLineStart $loc]]
	set ind1 [string first "" $text]
	set loc [nextLineStart $loc]
	if {[pos::compare $loc >= $limit]} {
	    break
	}
    }
    
    if {$ind1 < 0} {
	error "can't see to infinity"
    } else {
	return [lineStart [pos::math $loc - 1]]
    }
}

proc browse::Up {} {
    set pos [getPos]
    catch {set pos [browse::lookUpToInfinity [lineStart $pos]]}
    browse::Select $pos
}

proc browse::Down {} {
    set pos [selEnd]
    catch {set pos [browse::lookDownToInfinity [lineStart $pos]]}
    
    browse::Select $pos
}

proc browse::Select {pos} {
    if {[pos::compare $pos >= [maxPos]]} {
	set pos [lineStart [pos::math $pos - 1]]
    }
    
    set first [nextLineStart [nextLineStart [minPos]]]
    catch {
	set first [nextLineStart [browse::lookUpToInfinity $pos]]
    }
    
    set last [maxPos]
    catch {
	set last [nextLineStart [browse::lookDownToInfinity $pos]]
    }
    
    select $first $last
}

proc nextPrevMatch {{dir 1} {wname "*Batch Find*"}} {
    set wins [winNames]
    set res [lsearch $wins $wname]
    if {$res < 0} {
	set res [lsearch -regexp $wins {\*.*\*}]
	if {$res < 0} return
    }
    set win [lindex $wins $res]
    bringToFront $win
    if {$dir} {
	browse::Down
    } else {
	browse::Up
    }
    browse::Goto
    dispErr $win
}

proc nextMatch {{wname "*Batch Find*"}} {
    nextPrevMatch 1 $wname
}

proc prevMatch {{wname "*Batch Find*"}} {
    nextPrevMatch 0 $wname
}

proc dispErr {{win "* Compiler Errors *"}} {
    if {[string length $win]} {
	set text [getText -w $win [getPos -w $win] [selEnd -w $win]]
	if {[regexp {(Line.*)} $text dummy sub]} {
	    message "$sub"
	}
    }
}
		

##############################################################################
#  To be used in the windows created by "matchingLines" or by batch searches.
#
#  With the cursor positioned in a line corrsponding to a match, 
#  go back and select the line in the original file that 
#  generated this match.  (Like emacs 'Occur' functionality)
#
proc browse::Goto {} {
    global browse::GotoProc
    foreach pat [array names browse::GotoProc] {
	if {[string match $pat [win::CurrentTail]]} {
	    [set browse::GotoProc($pat)]
	    return
	}
    }
    global tileHeight tileWidth tileTop tileLeft tileHeight \
      errorHeight errorDisp tileMargin
    set loc [getPos]
    set ind1 -1
    while {$ind1 < 0} {
	set text [getText [lineStart $loc] [nextLineStart $loc]]
	set ind1 [string first "" $text]
	set loc [nextLineStart $loc]
	if {[pos::compare $loc == [maxPos]]} {break}
    }
    set ind2 [string last "" $text]
    if {$ind1 == $ind2} {
	set fname [string trim [string range $text $ind1 end] "\r\n"]
	set msg ""
    } else {
	set tmp [string trim [string range $text 0 $ind2] "\r\n"]
	if {[string last "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t" $tmp] < 0} {
	    set fname [string trim [string range $text $ind2 end] "\r\n"]
	    set msg ""
	} else {
	    set ind1 [string last "" $tmp]
	    set fname [string trim [string range $text $ind1 $ind2] "\r\n"]
	    set msg [string trim [string range $text $ind2 end] "\r\n"]
	}
    }
    set loc [getPos]
    set line -1
    while {1} {
	if {[regexp {Line ([0-9]+):} $text "" line]} {break}
	set text [getText [lineStart $loc] [nextLineStart $loc]]
	set loc [pos::math [lineStart $loc] - 1]
	if {[pos::compare $loc <= [minPos]]} {
	    # It's a browse window without line numbers, since we've
	    # backed up to the top of the window.
	    set line -1
	    break
	}
    }
    
    set top $tileTop
    set geo [getGeometry]
    if {([lindex $geo 0] != $tileLeft) || ([lindex $geo 1] != $top) \
      || ([lindex $geo 3] != $errorHeight) } {
	moveWin $tileLeft $top
	sizeWin $tileWidth $errorHeight
    }
    set mar $tileMargin
    incr top [expr {$errorHeight + $mar}]
    if {[browse::OpenWindow $fname]} {
	edit -c -w -g $tileLeft $top $tileWidth $errorDisp $fname
	set geo [getGeometry]
	if {([lindex $geo 0] != $tileLeft) || ([lindex $geo 1] != $top) \
	  || ([lindex $geo 2] != $tileWidth) || ([lindex $geo 3] != $errorDisp) } {
	    sizeWin $tileWidth $errorDisp
	    moveWin $tileLeft $top
	}
    } else {
	if {![string match "*Link*" \
	  [getText [minPos] [nextLineStart [minPos]]]]} {
	    alertnote "File \"$fname\" not found." 
	}
	return
    }
    if {$line >= 0} {
	set pos [rowColToPos $line 0]
	select $pos [nextLineStart $pos]
    }
    message $msg
}

proc browse::OpenWindow {fname} {
    global tileHeight tileWidth tileTop tileLeft tileHeight \
      errorHeight errorDisp tileMargin
    if {[file exists $fname]} {
	set top $tileTop
	set mar $tileMargin
	incr top [expr {$errorHeight + $mar}]
	edit -c -w -g $tileLeft $top $tileWidth $errorDisp $fname
	set geo [getGeometry]
	if {([lindex $geo 0] != $tileLeft) || ([lindex $geo 1] != $top) \
	  || ([lindex $geo 2] != $tileWidth) || ([lindex $geo 3] != $errorDisp) } {
	    sizeWin $tileWidth $errorDisp
	    moveWin $tileLeft $top
	}
	return 1
    } else {
	return 0
    }
}

set browse::lastMatchingLines ""

proc matchingLines {{reg ""} {for 1} {ign 1} {word 0} {regexp 1}} {
    global browse::lastMatchingLines
	
    if {![string length $reg] && \
      [catch {prompt "Regular expression:" [set browse::lastMatchingLines]} reg]} return
    set browse::lastMatchingLines $reg
    if {![string length $reg]} return
    if {!$regexp} {
	set reg [quote::Regfind $reg]
    }
    if {$word} {
	set reg "^.*\\b$reg\\b.*$"
    } else {
	set reg "^.*$reg.*$"
    }
    set pos [expr {$for ? [minPos] : [getPos]}]
    set fileName [win::StripCount [win::Current]]
    set matches 0
    browse::Start {* Matching Lines *} \
      "%d matching lines (<cr> to go to match)\r-----" 
    while {![catch {search -s -f 1 -r 1 -i $ign -- $reg $pos} mtch]} {
	browse::Add $fileName [eval getText $mtch] \
	  [lindex [posToRowCol [lindex $mtch 0]] 0] 0
	set pos [lindex $mtch 1]
	incr matches
    }
    browse::Complete
}

## 
 # -------------------------------------------------------------------------
 # 
 # "grepsToWindow" --
 # 
 #  'args' is a list of items
 # -------------------------------------------------------------------------
 ##
proc grepsToWindow {title args} {
    global tileLeft tileTop tileWidth tileHeight errorHeight
    win::SetProportions
    new -n $title -g $tileLeft $tileTop $tileWidth $errorHeight -m Brws \
      -tabsize 8 -info [join $args ""]
    browse::Select [minPos]
    message ""
}

## 
 # -------------------------------------------------------------------------
 # 
 # "browse::Format" --
 # 
 #  Can be used by external code to ensure browse information is in an
 #  acceptable format, and to simplify external code.
 # -------------------------------------------------------------------------
 ##
proc browse::Format {file match line {withname 1} {prefix ""}} {
    append res $prefix
    if {$withname} {
	set l [expr {40 - [string length [file tail $file]]}]
	append res "\"[file tail $file]\"; " [format "%$l\s" ""] " "
    } else {
	regsub -all "\t" $match "  " match
    }
    append res [format "Line %d:\r" $line] $match \
      "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$file"
    return $res
}

if {[info tclversion] < 8.0} {
    proc browse::RedoCount {} {
	global browse::count
	bringToFront [set browse::haveWindow]
	set oldcount [search -f 1 -m 0 -s -r 1 "\[0-9\]+" [minPos]]
	eval [list replaceText] $oldcount [list [set browse::count]]
    }
} else {
    proc browse::RedoCount {} {
	global browse::count
	set oldcount [search -w [set browse::haveWindow]\
	  -f 1 -m 0 -s -r 1 "\[0-9\]+" [minPos]]
	eval [list replaceText -w [set browse::haveWindow]]\
	  $oldcount [list [set browse::count]]
    }
}

proc browse::Complete {} {
    global browse::lines browse::none browse::haveWindow browse::count
    if {[string length [set browse::haveWindow]]} {
	bringToFront [set browse::haveWindow]
	browse::RedoCount
	goto [minPos]
	browse::Select [minPos]
	setWinInfo read-only 1
	return 0
    } else {
	if {[set browse::count]} {
	    browse::createWindow
	    setWinInfo -w [set browse::haveWindow] read-only 1
	    browse::Select [minPos]
	    return 0
	} else {
	    beep
	    message [set browse::none]
	    return 1
	}
    }
}

proc browse::createWindow {} {
    global tileLeft tileTop tileWidth tileHeight errorHeight \
      browse::lines browse::title browse::prefix browse::haveWindow \
      browse::backGround browse::count
    if {[set browse::backGround]} {set w [win::Current]}
    win::SetProportions
    set browse::haveWindow [new -n [set browse::title] \
      -g $tileLeft $tileTop $tileWidth $errorHeight -m Brws \
      -tabsize 8 -shell 1 \
      -text "[format [set browse::prefix] [set browse::count]]\r[join [set browse::lines] \r]\r"]
    set browse::lines {}
    if {[set browse::backGround]} {bringToFront $w}
    message ""
}

proc browse::updateWindow {} {
    global browse::haveWindow browse::lines
    placeText -w [set browse::haveWindow] [maxPos -w [set browse::haveWindow]] "[join [set browse::lines] \r]\r"
    browse::RedoCount
    set browse::lines {}
}

## 
 # -------------------------------------------------------------------------
 # 
 # "browse::Add" --
 # 
 #  Add the information to our list of browse items.  We can actually 
 #  add these dynamically to the window if we like.
 # -------------------------------------------------------------------------
 ##
proc browse::Add {file match line {withname 1} {prefix ""}} {
    global browse::lines browse::dynamic browse::haveWindow browse::count
    lappend browse::lines [browse::Format $file $match $line $withname $prefix]
    incr browse::count
    if {[set browse::dynamic]} {
	if {[string length [set browse::haveWindow]]} {
	    browse::updateWindow
	} else {
	    browse::createWindow
	}
	global alpha::platform
	if {${alpha::platform} != "alpha"} {update}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "browse::Dynamic" --
 # 
 #  Somewhat experimental.
 # -------------------------------------------------------------------------
 ##
proc browse::Dynamic {{backgd 0} {dyn 1}} {
    global browse::dynamic browse::haveWindow browse::backGround \
      browse::enableDynamic
    if {![set browse::enableDynamic]} {return}
    set browse::dynamic $dyn
    set browse::haveWindow ""
    set browse::backGround $backgd
}

proc browse::Start {{theTitle {* Matching Lines *}} \
  {thePrefix "%d matching lines (<cr> to go to match)\r-----"} \
  {ifNone "No matches found."}} {
    global browse::lines browse::title browse::prefix browse::none \
      browse::dynamic browse::haveWindow browse::backGround browse::count
    set browse::lines {}
    set browse::title $theTitle
    set browse::prefix $thePrefix
    set browse::none $ifNone
    set browse::dynamic 0
    set browse::haveWindow ""
    set browse::backGround 0
    set browse::count 0
}

