## -*-Tcl-*-  (nowrap)
 # ==========================================================================
 #  Scheme Mode - an extension package for Alpha
 # 
 #  FILE: "schemeMode.tcl"
 #                                    created: 07/03/1996 {02:19:49 pm} 
 #                                last update: 01-12-12 22.26.34 
 #  Description: 
 #  
 #  A mode for the shareware program Alpha, for Scheme files.
 # 
 #  Original by Oleg Kiselyov (oleg@ponder.csci.unt.edu) 
 #  
 #  Updated by Craig Barton Upright
 #  
 #  E-mail: <cupright@princeton.edu>
 #    mail: Princeton University,  Department of Sociology
 #          Princeton, New Jersey  08544
 #     www: <http://www.princeton.edu/~cupright>
 #     
 #  Craig will maintain this until somebody else steps forward.
 #  
 # -------------------------------------------------------------------
 #  
 # Copyright (c) 1996-2001  Oleg Kiselyov, Craig Barton Upright
 # All rights reserved.
 # 
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #   Redistributions of source code must retain the above copyright
 #    notice, this list of conditions and the following disclaimer.
 # 
 #   Redistributions in binary form must reproduce the above copyright
 #    notice, this list of conditions and the following disclaimer in the
 #    documentation and/or other materials provided with the distribution.
 # 
 #   Neither the name of Alpha/Alphatk nor the names of its contributors may
 #    be used to endorse or promote products derived from this software
 #    without specific prior written permission.
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # DAMAGE.
 # ==========================================================================
 ##

# ===========================================================================
#
#  Initialization of Scm mode  #
# 

alpha::mode Scm 2.1.1 schemeMode.tcl {*.scm} {
    schemeMenu electricReturn electricTab electricBraces 
} {
    # We require 7.5a3 for function procs.
    alpha::package require -loose AlphaTcl 7.5a3
    addMenu schemeMenu "Scheme" Scm
    set modeCreator(MrEd) Scm
    set modeCreator(MrSt) Scm
    set modeCreator(MzSt) Scm
} uninstall {
    catch {file delete [file join $HOME Tcl Modes schemeMode.tcl]}
    catch {file delete [file join $HOME Tcl Completions ScmCompletions.tcl]}
} help {
    The Scheme dialect of Lisp was created in 1975 by Guy Steele and Gerry
    Sussman to explore ideas in programming-language semantics.  They
    showed that a powerful language can be made "not by piling feature on
    top of feature, but by removing the weaknesses and restrictions that
    make additional features appear necessary".  Scheme pioneered lexical
    scope in Lisp, first-class continuations, and tail recursion, and more
    recently added an advanced macro system.  It's the best-known Lisp
    dialect after Common Lisp (which it influenced).  It is IEEE
    standardized and widely used in universities and in electronic CAD
    systems.
    
                                               -- <http://www.lisp.org>
    
    Alpha's Scheme mode includes a user-expandable dictionary of keywords,
    a full set of electric completions, and a sophisticated indentation
    procedure.  Normal features include Electric Return, Tab, and Braces.
    Scm mode also uses its own 'Scheme' menu, which is based on the Lisp
    menu.  See the "Lisp Help" file for more information.
    
    Click on this "Scheme Example.scm" link for an example syntax file.
} maintainer {
    "Craig Barton Upright" <cupright@princeton.edu> 
    <http://www.princeton.edu/~cupright/>
}

proc schemeMode.tcl {} {}

namespace eval Scm {}

# ===========================================================================
#
#  Setting Scm mode variables  #
#

#=============================================================================
#
# Standard preferences recognized by various Alpha procs
#

newPref var  fillColumn        {75}            Scm
newPref var  leftFillColumn    {0}             Scm
newPref var  prefixString      {;; }           Scm
newPref var  wordBreak         {[\w\-]+}       Scm
newPref var  wordBreakPreface  {([^\w\-])}     Scm
newPref flag wordWrap          {0}             Scm
newPref var  commentsContinuation 1            Scm "" \
     [list "only at line start" "spaces allowed" "anywhere"] index

#=============================================================================
#
# Flag preferences
#

# To automatically mark files when they are opened, turn this item on||To
# disable the automatic marking of files when they are opened, turn this
# item off
newPref flag autoMark          {0}             Scm

# To indent all continued commands (indicated by unmatched parantheses) by
# the full indentation amount rather than half, turn this item on|| To
# indent all continued commands (indicated by unmatched parantheses) by half
# of the indentation amount rather than the full, turn this item off
newPref flag fullIndent        {1}             Scm

# To primarily use a www site for help rather than the local Scheme
# application, turn this item on|| To primarily use the local Scheme
# application for help rather than on a www site turn this item off
newPref flag localHelp          {0}     Scm     {Scm::rebuildMenu lispHelp}

# If your keyboard does not have a "Help" key, turn this item on.  This will
# change some of the menu's key bindings|| If your keyboard has a "Help"
# key, turn this item off.  This will change some of the menu's key bindings
newPref flag noHelpKey          {0}     Scm     {Scm::rebuildMenu lispHelp}

# Set the list of flag preferences which can be changed in the menu.

set Scm::PrefsInMenu [list \
  "fullIndent" "localHelp" "noHelpKey" "(-)" "autoMark"]

#=============================================================================
#
# Variable preferences
# 

# Enter additional arguments to be colorized.
newPref var addArguments      {}              Scm    {Scm::colorizeScm}

# Enter additional Scm macros to be colorized.  
newPref var addMacros         {}              Scm    {Scm::colorizeScm}

# Command double-clicking on a Scheme keyword will send it to this url
# for a help reference page.
newPref url schemeHelp {http://www.harlequin.com:8000/xanalys_int/query.html?qt=} Scm

# Click on "Set" to find the local Scheme application.
newPref sig schemeSig         {}              Scm

# ===========================================================================
# 
# Color preferences
#

newPref color argumentColor     {magenta}       Scm    {Scm::colorizeScm}
newPref color commandColor      {blue}          Scm    {Scm::colorizeScm}
newPref color commentColor      {red}           Scm    {stringColorProc}
newPref color stringColor       {green}         Scm    {stringColorProc}
newPref color symbolColor       {magenta}       Scm    {Scm::colorizeScm}

regModeKeywords -C Scm {}
regModeKeywords -a -e {;} -c $ScmmodeVars(commentColor) \
  -s $ScmmodeVars(stringColor) Scm

# ==========================================================================
# 
# Comment Character variables for Comment Line / Paragraph / Box menu items.
# 

set Scm::commentCharacters(General)    ";; "
set Scm::commentCharacters(Paragraph)  [list ";; " " ;;" " ; "]
set Scm::commentCharacters(Box)        [list ";" 2 ";" 2 ";" 3]

# ===========================================================================
#
#  Keyword Dictionaries  #
#

# Making sure that ScmUserMacros and ScmUserArguments exist.
# These will be over-ridden if they are loaded from a ${mode}Prefs.tcl file.
#

set ScmUserMacros      ""
set ScmUserArguments   ""

set ScmCommands {
    abs and append apply assoc assq assv begin caar cadr
    call-with-current-continuation car case cdar cddr cdr cond cons declare
    define define-macro delay do else exact->inexact for-each if
    inexact->exact lambda length let let* letrec list list-refmake-vector
    map member memq memv number->string or peek-char read-char reverse set! 
    set-car!  set-cdr!  string string->number string-append string-length
    string-ref string-set!  substring vector vector-length vector-ref
    vector-set!
}

set ScmArguments {
    #f #t 
    char?  eof-object?  eq?  equal?  eqv?  even?  list?  negative?  not
    null?  odd?  pair?  positive?  procedure?  string=?  zero?
    
}


# ===========================================================================
# 
# Colorize Scheme.
# 
# Used to update preferences, and could be called in a <mode>Prefs.tcl file
# 

proc Scm::colorizeScm {{pref ""}} {
    
    global ScmmodeVars ScmCommands ScmArguments ScmUserMacros ScmUserArguments
    
    global ScmCommandList Scmcmds
    
    # First setting aside only the commands, for Scm::Completion::Command.
    set ScmCommandList [concat \
      $ScmCommands $ScmmodeVars(addMacros) $ScmUserMacros \
      ]
    
    # Then, create the list of all keywords for completions.
    set Scmcmds [lsort [lunique [concat \
      $ScmCommandList $ScmArguments \
      $ScmmodeVars(addArguments) $ScmUserArguments \
      ]]]
    # Commmands
    regModeKeywords -a -k $ScmmodeVars(commandColor) \
      Scm $ScmCommandList 
    
    # Arguments
    set ScmArgumentColorList [concat \
      $ScmArguments $ScmmodeVars(addArguments) $ScmUserArguments]
    regModeKeywords -a \
      -k $ScmmodeVars(argumentColor) Scm $ScmArgumentColorList
    
    # Symbols
    regModeKeywords -a \
      -i "+" -i "-" -i "*" -i "\\" -i "/" \
      -I $ScmmodeVars(symbolColor) Scm {}
    if {$pref != ""} {refresh}
}

# Call this now.
 
Scm::colorizeScm

# ===========================================================================
#
#  Key Bindings, Electrics  #
# 
# abbreviations:  <o> = option, <z> = control, <s> = shift, <c> = command
# 

Bind '\r'   <s>     {Scm::continueMacro} Scm
Bind '\)'           {Scm::electricRight "\)"} Scm

# For those that would rather use arrow keys to navigate.  Up and down
# arrow keys will advance to next/prev macro, right and left will also
# set the cursor to the top of the window.

Bind    up  <sz>    {Scm::searchFunc 0 0 0} Scm
Bind  left  <sz>    {Scm::searchFunc 0 0 1} Scm
Bind  down  <sz>    {Scm::searchFunc 1 0 0} Scm
Bind right  <sz>    {Scm::searchFunc 1 0 1} Scm

# ===========================================================================
# 
# Scheme Carriage Return
# 
# Inserts a carriage return, and indents properly.
# 

proc Scm::carriageReturn {} {
    
    global ScmmodeVars
    
    if {[isSelection]} {deleteSelection} 
    set pos1 [lineStart [getPos]]
    set pos2 [getPos]
    if {[regexp {^([\t ])*\)} [getText $pos1 $pos2]]} {
	createTMark temp $pos2
	bind::IndentLine
	gotoTMark temp ; removeTMark temp
    } 
    insertText "\r"
    bind::IndentLine
}

proc Scm::electricRight {{char "\}"}} {
    
    set pos [getPos]
    typeText $char
    if {![regexp {[^ \t]} [getText [lineStart $pos] $pos]]} {
	set pos [lineStart $pos]
	createTMark temp [getPos]
	bind::IndentLine
	gotoTMark temp ; removeTMark temp
	bind::CarriageReturn
    } 
    if {[catch {blink [matchIt $char [pos::math $pos - 1]]}]} {
	beep ; status::msg "No matching $char !!"
    } 
}

# ===========================================================================
#
# Continue Macro
# 
# Over-rides the automatic indentation of lines that begin with \) so that
# additional text can be entered.
# 

proc Scm::continueMacro {} {
    
    global ScmmodeVars indent_amounts
    
    Scm::carriageReturn
    if {[pos::compare [getPos] != [maxPos]]} {
	set nextChar [getText [getPos] [pos::math [getPos] + 1]]
	if {$nextChar == "\)"} {
	    set continueIndent [expr {$ScmmodeVars(fullIndent) + 1}]
	    insertText [text::indentOf $indent_amounts($continueIndent)]
	} 
    } 
}

proc Scm::searchFunc {direction args} {

    if {![llength $args]} {set args [list 0 2]}
    if {$direction} {
	eval function::next $args
    } else {
	eval function::prev $args
    }
}

# This is used preferentially by 'function::getLimits'

proc Scm::getLimits {pos direction} {
    
    set posBeg ""
    set posEnd ""
    set what   "macro"
    # The idea is to find the start of the closest macro (in the
    # specified direction, and based solely on indentation), the start
    # of the next, and then back up to remove empty lines.  Trailing
    # parens are not ignored backing up, so that they are retained as
    # part of the macro.
    set pat1 {^\([^\r\n\t \;]}
    set pat2 {^[\t ]*(;.*)?$}
    set pos1 $pos
    set posBeg ""
    set posEnd ""
    if {![catch {search -f $direction -s -r 1 -i 1 $pat1 $pos1} match]} {
	# This is the start of the closest function.
	set posBeg [lindex $match 0]
	set pos2   [lindex $match 1]
	if {![catch {search -s -f 1 -r 1 $pat1 $pos2} match]} {
	    # This is the start of the next one.
	    set posEnd [lindex $match 0]
	} else {
	    set posEnd [maxPos]
	} 
	# Now back up to skip empty lines, ignoring comments as well.
	while {1} {
	    set posEndPrev [pos::math $posEnd - 1]
	    set prevLine   [getText [lineStart $posEndPrev] $posEndPrev]
	    if {![regexp $pat2 $prevLine]} {break}
	    set posEnd     [lineStart $posEndPrev]
	}
    } 
    return [list $posBeg $posEnd $what]
}

# ===========================================================================
#
#  Indentation  #
# 
# Indenting a line of a Scheme code
#
# The idea is simple: the indent of a new line is the same as the indent of
# the previous non-empty non-comment-only line *plus* the paren balance of
# that line times two.
# 
# That is, if the last code line was paren balanced, the next line would
# have the same indent.  If the prev line opened an expression but didn't
# close it, the new line would be indented further
#

# This was the original procedure, with a little modification:

# proc Scm::indentLine {} {
# 
#     global mode indent_amounts ScmmodeVars
#     
#     set continueIndent [expr {$ScmmodeVars(fullIndent)  + 1}]
#     
#     set beg [lineStart [getPos]]
#     set end [nextLineStart [getPos]]
#     
#     # Find last previous non-comment line and get its leading whitespace
#     set pos $beg
#     set lst [search -s -f 0 -r 1 -i 0 {^[ \t]*[^ ;\t\r\n]} [pos::math $pos - 1]]   
#     set line [getText [lindex $lst 0] [pos::math [nextLineStart [lindex $lst 0]] - 1]]
#     set lwhite [getText [lindex $lst 0] [pos::math [lindex $lst 1] - 1]]
#     
#     # Computing the balance of parentheses within the 'line':
#     # 
#     # This appears to be utterly elementary.  One has to keep in mind
#     # however that parentheses might appear in comments and/or quoted
#     # strings, in which case they shouldn't count.  Although it's easy to
#     # detect a Scheme comment by a semicolon, a semicolon can also appear
#     # within a quoted string.  Note that a double quote isn't that sure a
#     # sign of a quoted string: the double quote may be escaped.  And the
#     # backslash can be escaped in turn...  Thus we face a full-blown
#     # problem of parsing a string according to a context-free grammar.  We
#     # note however that a TCL interpretor does similar kind of parsing all
#     # the time.  So, we can piggy-back on it and have it decide what is the
#     # quoted string and when a semicolon really starts a comment.  To this
#     # end, we replace all non-essential characters from the 'line' with
#     # spaces, separate all parens with spaces (so each paren would register
#     # as a separate token with the TCL interpretor), replace a semicolon
#     # with an opening brace (which, if unescaped and unquoted, acts as some
#     # kind of "comment", that is, shields all symbols that follows).  After
#     # that, we get TCL interpretor to convert thus prepared 'line' into a
#     # list, and simply count the balance of '(' and ')' tokens.
#     # 
#     
#     regsub -all -nocase {[^ ();\"\\]} $line { } line1
#     regsub -all {;} $line1 "\{" line
#     regsub -all {[()]} $line { \0 } line1
#     set line_list [eval "list $line1 \}"]
#     #alertnote ">$line_list<"
#     set balance 0
#     foreach i $line_list { 
# 	switch $i {
# 	    ( {incr balance $continueIndent} 
# 	    ) {incr balance -continueIndent}
# 	}
#     }
#     #alertnote "balance $balance, lwhite [string length $lwhite]"
#     if {$balance < 0} {
#       set lwhite [string range $lwhite 0 [expr [string length $lwhite] + 2 * $balance - 1]]
#     } else {
#       append lwhite [string range "              " 1 [expr 2 * $balance]]
#     }
#     #alertnote "new lwhite [string length $lwhite]"
#     
#     set text [getText $beg [nextLineStart $beg]]
#     regexp {^[ \t]*} $text white
#     set len [string length $white]
#     
#     if {$white != $lwhite} {
#       replaceText $beg [pos::math $beg + $len] $lwhite
#     }
#     goto [pos::math $beg + [string length $lwhite]]
#     return
#     
# }

proc Scm::correctIndentation {pos {next ""}} {
    
    global mode indent_amounts ScmmodeVars
    
    if {$mode == "Scm"} {
	set continueIndent [expr {$ScmmodeVars(fullIndent) + 1}]
    } else {
	set continueIndent ${indent_amounts(1)}
    } 
    
    set posBeg   [lineStart $pos]
    # Get information about this line, previous line ...
    set thisLine [Scm::getMacroLine $posBeg 1 1]
    set prevLine [Scm::getMacroLine [pos::math $posBeg - 1] 0 1]
    set lwhite   [lindex $prevLine 1]
    # If we have a previous line ...
    if {[pos::compare [lindex $prevLine 0] != $posBeg]} {
	# Find out if there are any unbalanced (,) in the last line.
	regsub -all {[^ \(\)\"\;\\]} $prevLine { } line
	# Remove all literals.
	regsub -all {\\\(|\\\)|\\\"|\\\;} $line { } line
	regsub -all {\\} $line { } line
	# If there is only one quote in a line, next to a closing brace,
	# assume that this is a continued quote from another line.  So add
	# a double quote at the beginning of the line (which will make us
	# ignore everything up to that point).  Not entirely foolproof ...
	if {![regexp {\".+\"} $line] && [regexp {\"[\t ]*\)} $line]} {
	    set line [concat \"$line]
	} 
	# Remove everything surrounded by quotes.
	regsub -all {\"[^\"]+\"} $line { } line
	regsub -all {\"} $line { } line
	# Remove all characters following the first valid comment.
	if {[regexp {\;} $line]} {
	    set line [string range $line 0 [string first {;} $line]]
	} 
	# Now turn all braces into "more" and "less"
	regsub -all {\(} $line { more } line
	regsub -all {\)} $line { less } line
	# Now indent based upon more and less.
	foreach i $line {
	    if {$i == "more"} {
		incr lwhite $indent_amounts($continueIndent)
	    } elseif {$i == "less"} {
		incr lwhite $indent_amounts(-$continueIndent)
	    }
	}
	# Did the last line start with a lone \) ?  If so, we want to keep the
	# indent, and not make call it an unbalanced line.
	if {[regexp {^[\t ]*\)} [lindex $prevLine 2]]} {
	    incr lwhite $indent_amounts($continueIndent)
	} 
    }
    # If we have a current line ...
    if {[pos::compare [lindex $thisLine 0] == $posBeg]} {
	# Reduce the indent if the first non-whitespace character of this
	# line is ) or \}.
	if {$next == ")" || [regexp {^[\t ]*\)} [lindex $thisLine 2]]} {
	    incr lwhite $indent_amounts(-$continueIndent)
	} 
    }
    # Now we return the level to the calling proc.
    return [expr {$lwhite > 0} ? $lwhite : 0]
}

# ===========================================================================
# 
# Get Macro Line
# 
# Find the next/prev command line relative to a given position, and return
# the position in which it starts, its indentation, and the complete text
# of the command line.  If the search for the next/prev command fails,
# return an indentation level of 0.
# 

proc Scm::getMacroLine {pos {direction 1} {ignoreComments 1}} {
    
    if {$ignoreComments} {
	set pat {^[\t ]*[^\t\r\n\; ]}
    } else {
	set pat {^[\t ]*[^\t\r\n ]}
    } 
    set posBeg [pos::math [lineStart $pos] - 1]
    if {[pos::compare $posBeg < [minPos]]} {
	set posBeg [minPos]
    } 
    set lwhite 0
    if {![catch {search -s -f $direction -r 1 $pat $pos} match]} {
	set posBeg [lindex $match 0]
	set lwhite [posX [pos::math [lindex $match 1] - 1]]
    }
    set posEnd [pos::math [nextLineStart $posBeg] - 1]
    if {[pos::compare $posEnd > [maxPos]]} {
	set posEnd [maxPos]
    } elseif {[pos::compare $posEnd < $posBeg]} {
	set posEnd $posBeg
    }
    return [list $posBeg $lwhite [getText $posBeg $posEnd]]
}

# ===========================================================================
# 
#  Command Double Click  #
#
# Checks to see if the highlighted word appears in any keyword list, and if
# so, sends the selected word to the www.Lisp.com help site.
# 
# Control-Command double click will insert syntax information in status bar.
# Shift-Command double click will insert commented syntax information in window.
# 
# (The above is not yet implemented -- need to enter all of the syntax info.)
# 

proc Scm::DblClick {from to shift option control} {
    
    global ScmmodeVars Scmcmds ScmSyntaxMessage
        
    select $from $to
    set command [getSelect]
    
    set varDef "(def|make)+(\[-a-zA-Z0-9\]+(\[\t\' \]+$command)+\[\t\r\n\(\) \])"
    
    if {![catch {search -s -f 1 -r 1 -m 0 $varDef [minPos]} match]} {
        # First check current file for a function, variable (etc)
        # definition, and if found ...
        placeBookmark
        goto [lineStart [lindex $match 0]]
        status::msg "press <Ctl .> to return to original cursor position"
        return
        # Could next check any open windows, or files in the current
        # window's folder ...  but not implemented.  For now, variables
        # (etc) need to be defined in current file.
    }
    if {![lcontains Scmcmds $command]} {
        status::msg "'$command' is not defined as a Scm system keyword."
        return
    }
    # Any modifiers pressed?
    if {$control} {
        # CONTROL -- Just put syntax status::msg in status bar window
        if {[info exists ScmSyntaxMessage($command)]} {
            status::msg "$ScmSyntaxMessage($command)"
        } else {
            status::msg "Sorry, no syntax information available for $command"
        }
    } elseif {$shift} {
        # SHIFT --Just insert syntax message as commented text
        if {[info exists ScmSyntaxMessage($command)]} {
            endOfLine
            insertText "\r"
            insertText "$ScmSyntaxMessage($command)
            comment::Line
        } else {
            status::msg "Sorry, no syntax information available for $command"
        }
    } else {
        # No modifiers -- Send command for on-line help.  This is the
        # "default" behavior.
        Scm::wwwMacroHelp $command
    }
}

# ===========================================================================
# 
# WWW Macro Help
# 
# Send macro to defined url, prompting for text if necessary.
# 

proc Scm::wwwMacroHelp {{macro ""}} {
    
    global ScmmodeVars
    
    if {![string length $macro]} {
	if {[catch {prompt "On-line SAS help for " ""} macro]} {
	    status::errorMsg "Cancelled."
	} 
    }
    status::msg "'$macro' sent to $ScmmodeVars(schemeHelp)"
    urlView $ScmmodeVars(schemeHelp)$macro
}

proc Scm::localMacroHelp {args} {Scm::betaMessage}

# ===========================================================================
#
#  Mark File and Parse Functions  #
#

# ===========================================================================
#
# Scm Mark File
# 

proc Scm::MarkFile {} {
    
    status::msg "Marking File "
    
    set count 0
    set pos [minPos]
    set pat {^[ \t]*[\(][#a-zA-z]*(define|define-[a-zA-Z]+) +[\(]*([^\(\) \t\r\n]+)}
    while {![catch {search -s -f 1 -r 1 -m 0 -i 1 $pat $pos} match]} {
        regexp -nocase -- $pat [eval getText $match] allofit defunname name
        set pos0 [lindex $match 0]
        set pos1 [nextLineStart $pos0]
	set mark [string trimleft [string trim [getText $pos0 $pos1]] "\("]
	while {[lcontains marks $mark]} {append mark " "}
	lappend marks $mark
        setNamedMark $mark $pos0 $pos0 $pos0
	incr count
        set pos $pos1
    }
    status::msg ""
}

# This was the original proc.

# proc Scm::MarkFile {} {
#     
#     set pat1 {^[ \t]*[\(][#a-zA-z]*(define|define-[a-zA-Z]+) +[\(]*([^\(\) \t\r\n]+)}
#     set end [maxPos]
#     set pos [minPos]
#     set l {}
#     while {![catch {search -s -f 1 -r 1 -m 0 -i 1 $pat1 $pos} mtch]} {
#       regexp -nocase -- $pat1 [eval getText $mtch] allofit defunname name
#       set start [lindex $mtch 0]
#       set end [nextLineStart $start]
#       set pos $end
#       set inds($name) [lineStart [pos::math $start - 1]]
#     }
#     
#     if {[info exists inds]} {
#       foreach f [lsort -ignore [array names inds]] {
#           set next [nextLineStart $inds($f)]
#           setNamedMark $f $inds($f) $next $next
#       }
#     }
# }

# ===========================================================================
#
# Scm Parse Functions
# 
# This will return only the Scm command names.
# 

proc Scm::parseFuncs {} {
    
    global sortFuncsMenu
    
    set pos [minPos]
    set m   [list ]
    while {![catch {search -s -f 1 -r 1 -i 0 {^\((\w+)} $pos} match]} {
	if {[regexp -- {(\w+)} [eval getText $match] "" word]} {
	    lappend m [list $word [lindex $match 0]]
	}
	set pos [lindex $match 1]
    }
    if {$sortFuncsMenu} {set m [lsort -ignore $m]}
    return [join $m]
}

# ===========================================================================
# 
#  --------------------  #
# 
#  Scheme Menu  #
# 

proc schemeMenu {} {}

# Tell Alpha what procedures to use to build all menus, submenus.

menu::buildProc schemeMenu Scm::buildMenu     Scm::postEval
menu::buildProc schemeHelp Scm::buildHelpMenu

# First build the main Scheme menu.

proc Scm::buildMenu {} {
    
    global schemeMenu Scm::PrefsInMenu

    set optionItems [set Scm::PrefsInMenu]
    set keywordItems [list \
      "listKeywords" "checkKeywords" "addNewMacros" "addNewArguments"]
    set menuList [list                                  \
      "schemeHomePage"                                  \
      "/S<U<OswitchToScheme"                            \
      "/P<U<OprocessFile"                               \
      "/P<U<O<BprocessSelection"                        \
      "(-)"                                             \
      [list Menu -n schemeHelp           -M Scm {}]     \
      [list Menu -n schemeModeOptions -p Scm::menuProc -M Scm $optionItems] \
      [list Menu -n schemeKeywords    -p Scm::menuProc -M Scm $keywordItems] \
      "(-)"                                             \
      "/b<UcontinueMacro"                               \
      "/'<E<S<BnewComment"                              \
      "/'<S<O<BcommentTemplate"                        \
      "(-)"                                             \
      "/N<U<BnextMacro"                                 \
      "/P<U<BprevMacro"                                 \
      "/S<U<BselectMacro"                               \
      "/I<B<OreformatMacro"                             \
      ]
    set submenus [list schemeHelp]
    return       [list build $menuList "Scm::menuProc -M Scm" $submenus $schemeMenu]
}

# Then build the "Scm Help" submenu.

proc Scm::buildHelpMenu {} {
    
    global ScmmodeVars
    
    # Determine which key should be used for "Help", with F8 as option.
    
    if {!$ScmmodeVars(noHelpKey)} {
	set key "/t"
    } else {
	set key "/l"
    } 
    
    # Reverse the local, www key bindings depending on the value of the
    # 'Local Help" variable.
    
    if {!$ScmmodeVars(localHelp)} {
	set menuList [list                  \
	  "${key}<OwwwMacroHelp"         \
	  "${key}<IlocalMacroHelp"       \
	  ]
    } else {
	set menuList [list                  \
	  "${key}<OlocalMacroHelp"       \
	  "${key}<IwwwMacroHelp"         \
	  ]
    } 
    lappend menuList "(-)"
    lappend menuList "setSchemeApplication"
    lappend menuList "(-)"
    lappend menuList "${key}<BschemeModeHelp"
    
    return [list build $menuList "Scm::menuProc -M Scm" {}]
}

# Mark or dim items as necessary.

proc Scm::postEval {args} {
    
    global ScmmodeVars Scm::PrefsInMenu
    
    foreach itemName [set Scm::PrefsInMenu] {
	if {[info exists ScmmodeVars($itemName)]} {
	    markMenuItem schemeModeOptions $itemName $ScmmodeVars($itemName) 
	} 
    } 
}

# Now we actually build the Scm menu.

menu::buildSome schemeMenu

proc Scm::rebuildMenu {{menuName "schemeMenu"}} {
    menu::buildSome $menuName
}

# Dim some menu items when there are no open windows.

proc Scm::registerOWH {{which "register"}} {
    
    global schemeMenu
    
    set menuItems {
	processFile processSelection continueMacro
	newComment commentTemplate
	nextMacro prevMacro selectMacro reformatMacro
    }
    foreach i $menuItems {
	hook::${which} requireOpenWindowsHook [list $schemeMenu $i] 1
    } 
}

# Call this now.
Scm::registerOWH register ; rename Scm::registerOWH ""

# ===========================================================================
# 
#  Scm menu support  #
# 
# We make some of these items "Scm Mode Only", in case Scheme mode also
# uses this menu.
# 

# This is the procedure called for all main menu items.

proc Scm::menuProc {menuName itemName} {
    
    global Scmcmds ScmmodeVars Scm::PrefsInMenu mode

    switch $menuName {
	"schemeHelp" {
	    switch $itemName {
		"setSchemeApplication"  {Scm::setApplication "Scheme"}
		"schemeModeHelp"        {package::helpFile "Scm"}
		default                 {Scm::$itemName}
	    } 
	}
	"schemeModeOptions" {
	    if {[getModifiers]} {
		cache::readContents index::prefshelp
		if {[catch {set prefshelp(Scm,$itemName)} helpText]} {
		    set helpText "Sorry, no information is available for '$itemName'"
		} else {
		    catch {unset index::prefshelp}
		}
		if {$ScmmodeVars($itemName)} {set end "on"} else {set end "off"}
		if {$end == "on"} {
		    regsub {^.*\|\|} $helpText {} helpText
		} else {
		    regsub {\|\|.*$} $helpText {} helpText
		}
		set msg "The '$itemName' preference for Scm mode is currently $end."
		dialog::alert "${helpText}."
	    } elseif {[lcontains Scm::PrefsInMenu $itemName]} {
		set ScmmodeVars($itemName) [expr {$ScmmodeVars($itemName) ? 0 : 1}]
		if {$mode == "Scm"} {
		    synchroniseModeVar $itemName $ScmmodeVars($itemName)
		} else {
		    prefs::modified $ScmmodeVars($itemName)
		}
		if {[regexp {Help} $itemName]} {Scm::rebuildMenu "schemeHelp"}
		Scm::postEval
		if {$ScmmodeVars($itemName)} {
		    set end "on"
		} else {
		    set end "off"        
		} 
		set msg "The '$itemName' preference is now $end."
	    } else {
		set msg "Don't know what to do with '$itemName'."
	    }
	    if {[info exists msg]} {status::msg $msg}
	}
	"schemeKeywords" {
	    if {$itemName == "listKeywords"} {
		set keywords [listpick -l -p "Current Scm mode keywords" $Scmcmds]
		foreach keyword $keywords {
		    Scm::checkKeywords $keyword
		}
	    } elseif {$itemName == "addNewMacros" || $itemName == "addNewArguments"} {
		set itemName [string trimleft $itemName "addNew"]
		Scm::addKeywords $itemName
	    } else {
		Scm::$itemName
	    } 
	    return
	}
	"markScmFileAs" {
	    removeAllMarks
	    switch $itemName {
		"source"    {Scm::MarkFile ".scheme"}
	    }
	}
	default {
	    switch $itemName {
		"schemeHomePage"    {url::execute $ScmmodeVars(schemeHomePage)}
		"switchToScm"    {app::launchFore $ScmmodeVars(schemeSig)}
		"newComment"      {comment::newComment 0}
		"commentTemplate" {comment::commentTemplate}
		"nextMacro"       {Scm::searchFunc 1 0 0}
		"prevMacro"       {Scm::searchFunc 0 0 0}
		"selectMacro"     {function::select}
		"reformatMacro"   {function::reformat}
		default           {Scm::$itemName}
	    }
	}
    }
}

# Give a beta message for untested features / menu items.

proc Scm::betaMessage {{item ""}} {
    
    if {![string length $item]} {
	if {[catch {info level -1} item]} {set item "this item"}
    }
    status::errorMsg "Sorry -- '$item' has not been implemented yet."
}

# Return the Scm signature.

proc Scm::sig {{app "Scheme"}} {
    
    global ScmmodeVars
    
    set lowApp [string tolower $app]
    set capApp [string toupper $app]
    if {$ScmmodeVars(${lowApp}Sig) == ""} {
	alertnote "Looking for the $capApp application ..."
	Scm::selectApplication $lowApp
    }
    return $ScmmodeVars(${lowApp}Sig)
}

# ===========================================================================
# 
# Select Application
# 
# Prompt the user to locate the local Scm application.
# 

proc Scm::selectApplication {{app "Scm"}} {
    
    global ScmmodeVars
    
    set lowApp [string tolower $app]
    set capApp [string toupper $app]
    
    set newSig ""
    set newSig [dialog::askFindApp $capApp $ScmmodeVars(${lowApp}Sig)]
    
    if {$newSig != ""} {
	set ScmmodeVars(${lowApp}Sig) "$newSig"
	prefs::modified ScmmodeVars(${lowApp}Sig)
	status::msg "The $capApp signature has been changed to '$newSig'."
    } else {
	status::errorMsg "Cancelled."
    }
}

# ===========================================================================
#
#  Keywords  #
# 

proc Scm::addKeywords {{category} {keywords ""}} {
    
    global ScmmodeVars
    
    if {$keywords == ""} {
        set keywords [prompt "Enter new Scm mode $category:" ""]
    }
    
    # Check to see if the keyword is already defined.
    foreach keyword $keywords {
        set checkStatus [Scm::checkKeywords $keyword 1 0]
        if {$checkStatus != 0} {
            alertnote "Sorry, '$keyword' is already defined\
              in the $checkStatus list."
            status::errorMsg "Cancelled."
        } 
    }
    # Keywords are all new, so add them to the appropriate mode preference.
    append ScmmodeVars(add$category) " $keywords"
    set ScmmodeVars(add$category) [lsort $ScmmodeVars(add$category)]
    synchroniseModeVar add$category $ScmmodeVars(add$category)
    Scm::colorizeScm
    status::msg "'$keywords' added to $category preference."
}

proc Scm::checkKeywords {{newKeywordList ""} {quietly 0} {noPrefs 0}} {
    
    global ScmmodeVars ScmCommands ScmUserMacros ScmUserArguments
    
    set type 0
    if {$newKeywordList == ""} {
        set quietly 0
        set newKeywordList [prompt "Enter Scm mode keywords to be checked:" ""]
    }
    # Check to see if the new keyword(s) is already defined.
    foreach newKeyword $newKeywordList {
        if {[lcontains ScmCommands $newKeyword]} {
            set type ScmCommands 
        } elseif {[lcontains ScmUserMacros $newKeyword]} {
            set type ScmUserMacros
        } elseif {[lcontains ScmUserArguments $newKeyword]} {
            set type ScmUserArguments
        } elseif {!$noPrefs && \
          [lcontains ScmmodeVars(addMacros) $newKeyword]} {
            set type ScmmodeVars(addMacros)
        } elseif {!$noPrefs && \
          [lcontains ScmmodeVars(addArguments) $newKeyword]} {
            set type ScmmodeVars(addArguments)
        }
        if {$quietly} {
            # When this is called from other code, it should only contain
            # one keyword to be checked, and we'll return it's type.
            return "$type"
        } elseif {!$quietly && $type == 0} {
            alertnote "'$newKeyword' is not currently defined\
              as a Scm mode keyword"
        } elseif {$type != 0} {
            # This will work for any other value for "quietly", such as "2"
            alertnote "'$newKeyword' is currently defined as a keyword\
              in the '$type' list."
        } 
        set type 0
    }
}

proc Scm::processFile {} {
    Scm::betaMessage
}

proc Scm::processSelection {} {
    Scm::betaMessage
}

# ===========================================================================
# 
#  --------------------  #
# 
#  version history  #
# 
#  modified by  rev    reason
#  -------- --- ------ -----------
#  10/01/96 ok  1.0    Original Scheme mode.
#  03/18/99 ??  - 1.3  Updates.
#  11/30/00 cbu 1.4    Updated to use the lispMenu better, including
#                      Added   Scm::colorizeScm 
#                      Added   Scm::carriageReturn 
#                      Added   Scm::electricRight 
#                      Revised Scm::indentLine
#                      Added   Scm::correctIndentation 
#                      Revised Scm::MarkFile
#                      Added   Scm::parseFuncs 
#                      Added   Scm::checkKeywords 
#                      Added   Scm::addKeywords 
#                      Added   Scm::searchFunc 
#  12/01/00 cbu 2.0    New url prefs handling requires 7.4b21
#  09/26/01 cbu 2.1    Big cleanup, enabled by new 'functions.tcl' procs.
#                      New 'Scheme' menu.
#  10/31/01 cbu 2.1.1  Minor bug fixes.
# 

# ===========================================================================
# 
# .