## -*-Tcl-*-
 # # ###################################################################
 #  Alpha - new Tcl folder configuration
 # 
 #  FILE: "tclMode.tcl"
 #                                    created: 5/4/97 {9:31:10 pm} 
 #                                last update: 11/13/2001 {12:03:29 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta
 #          Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 1997-2001 Vince Darley
 #  
 # Three procs from original: Tcl::DblClick listArray, getVarValue
 #	
 # Adds support for Tk, Itcl keywords and completions, plus 
 # numerous fixes, improvements and integration with Vince's
 # Additions.
 # ###################################################################
 ##

alpha::mode Tcl 1.9.13 tclMenu [list *.tcl *.itcl *.itk *.decls *.msg \
  *.tbc tclIndex* "\\* Trace '*' \\*" *.test] {
    tclMenu electricTab electricReturn electricBraces alphaDeveloperMenu
} {
    addMenu tclMenu "269" "Tcl" "Tcl menu\r\rnot very obvious..."
    set unixMode(wish) {Tcl}
    set unixMode(tclsh) {Tcl}
    set unixMode(itclsh) {Tcl}
    set unixMode(itkwish) {Tcl}
    set unixMode(prowish) {Tcl}
    set unixMode(protclsh) {Tcl}
    set unixMode(tclkit) {Tcl}
    set unixMode(tclkitsh) {Tcl}
    newPref sig tclshSig "WIsH"
    ensureset evaluateRemotely 0
    trace variable evaluateRemotely w tcltk::evaluateRemoteSynchronise
    menu::buildProc tclMenu menu::buildtclMenu
    lappend tclCmdColourings Tcl::colorTclKeywords \
      Tcl::colorTkKeywords \
      Tcl::colorItclKeywords Tcl::colorPseudoTclKeywords \
      Tcl::colorTkCommands 
    lappend tclExtraColourings \
      Tcl::colorSymbols Tcl::colorVariables
} maintainer {
    "Vince Darley" <vince@santafe.edu> <http://www.santafe.edu/~vince/>
} uninstall this-file help {
    This mode is for editing Tcl code.  You can edit code for internal
    use with Alpha, or use Alpha as an external editor for code destined
    for use with Tcl and Tk interpreters --- <http://www.tcltk.com>
    distributes the Tclsh/Wish applications and a tcl-tk browser plugin.
    <http://www.scriptics.com/> maintains a Tcl Developer's Xchange, and
    is the best source for the latest Tcl/Tk releases.

    Most of Alpha(tk)'s functionality is contained in its Tcl library.  See
    the "Tcl Example.tcl" file for an example.  You can 'evaluate' a
    procedure (or any Tcl code for that matter) to make changes on the
    fly.  If you select 'Evaluate Remotely' in the tcl-tk submenu, then
    such actions will actually send the code to a separately running
    Tclsh/Wish application to be evaluated.
    
    Alternatively, you can add .tcl files you wish to run in remote
    interpreters to the tcl-tk submenu.  Selecting them will startup
    a remote shell, source that .tcl file into it, and put Alpha(tk)
    into 'remote' mode. Subsequent loading of code will go to the
    remote shell automatically, so you can use Alpha to edit and
    reload procedures in the remote shell, very simply.

    Alpha includes the help files "Tcl Commands", and "Tcl Resources".
}

namespace eval tcltk {}
proc tclMenu {} {}

#  menu and prefs  #
# The menu.
proc menu::buildtclMenu {} {
    global tclMenu evaluateRemotely tcltk::executables
    set execs {}
    if {[info exists tcltk::executables]} {
	lappend execs "(-)"
	foreach ex ${tcltk::executables} {
	    lappend execs [file tail $ex]
	}
    }
    set ma [list \
      "/Levaluate" "/-<UswitchToTclsh" \
      [list Menu -n "tcl-tk" -p tcltk::menuProc [concat [list \
      "![lindex {{ } } $evaluateRemotely]evaluateRemotely" \
      executeCommand executeFileInRemoteShell addWindowToListOfExecutables] \
      $execs]] \
      "(-)" "/L<O<BreloadProc" "/I<O<BreformatProc" \
      "rebuildTclIndexForWin" "debugProc" "applyChanges" "(-)" \
      "<U/PfindProcDefinition" "/Q<O<BquickFindProc" "getVarValue" \
      "/4<BaddRemoveDollars" "/3<BinsertDivider" \
      "(-)" "regularExpressionColors" "defaultColors"]
    return [list build $ma Tcl::MenuProc "" $tclMenu]
}
menu::buildSome tclMenu

#===============================================================================
#
# Set up package-specific mode variables
# 

# Removing obsolete preferences from earlier versions.
prefs::removeObsolete TclmodeVars(alphaKeyWordColor) TclmodeVars(keywordColor) \
  TclmodeVars(tclHelp)
prefs::renameOld TclmodeVars(magicColor) TclmodeVars(variablesColor)

newPref v prefixString {# } Tcl
newPref f wordWrap {0} Tcl
#newPref v funcExpr {^proc *([-+a-zA-Z0-9:]+)} Tcl
#newPref v parseExpr {^proc *([-+a-zA-Z0-9:]+)} Tcl

newPref v wordBreak {(\$)?[\w:_]+} Tcl
newPref v wordBreakPreface {([^\w:_\$]|.\$)} Tcl
newPref f autoMark 0 Tcl
# Indentation scheme for lines following one ending in a backslash
newPref v indentSlashEndLines 1 Tcl "" indent::amounts varindex
# Mark files structurally, recognising the special comments
# entered by 'ctrl-3'
newPref f structuralMarks 0 Tcl
# Cmd-double-clicking on a Tcl command will provide web-based documentation
# from this location.
newPref url tclHelpLocation "http://dev.scriptics.com/man/tcl8.4/TclCmd/" Tcl
# How to handle comments continuation
newPref v commentsContinuation 2 Tcl {} \
  [list "only at line start" "spaces allowed" "anywhere"] index

# Directories in which to search for tclIndex files pointing to procedures
# for quick access to those procedures (e.g. by cmd-double-clicking).
newPref v procSearchPath "" Tcl

newPref f structuralElectricElseBrace 0 Tcl

# Tcl mode initialisations:
set Tcl::escapeChar "\\"
set Tcl::quotedstringChar "\""
set Tcl::lineContinuationChar "\\"

set Tcl::startPara {^(.*\{)?[ \t]*(#|$)}
set Tcl::endPara {^(.*\})?[ \t]*(#|$)}
set Tcl::commentRegexp {^[ \t]*#}
set Tcl::commentCharacters(General) "\#"
set Tcl::commentCharacters(Paragraph) [list "## " " ##" " # "]
set Tcl::commentCharacters(Box) [list "#" 1 "#" 1 "#" 3]

# This is used by completions...
ensureset Tclcmds [list \
  append array break case catch cd close concat continue eof error eval \
  exec exit expr file flush for foreach format gets glob global history if \
  incr info join lappend lindex linsert list llength lrange lreplace \
  lsearch lsort open pid proc puts pwd read regexp regsub rename return \
  scan seek set source split string switch tell time trace unknown unset \
  uplevel upvar while ]

#===============================================================================
#
# Colorization setup
#

# Colour Tk commands
newPref f recogniseTk		{1}	Tcl	{Tcl::updateColors}
# Colour [incr Tcl] commands
newPref f recogniseItcl		{1}	Tcl	{Tcl::updateColors}
# Recognise and colour some common procedures 'lunion' etc.
newPref f recognisePseudoTcl	{1}	Tcl	{Tcl::updateColors}
# Colour of all chosen commands.
newPref color commandColor	    {blue}	Tcl	{Tcl::updateColors}
# Color for Tcl comments
newPref color commentColor	    {red}	Tcl	{Tcl::updateColors}
# In Tcl, the colour of words started by '$'.  Tcl considers such 
# words to be variables.  A dark brown might be a good choice, 
# distinguishable, but not too distracting.  To try that, chose 
# 'Config:Redefine Colors:Color_9' and set it to brown (the 'raw sienna'
# crayon is a good choice). Then, when you are done, come back
# to this dialog. Color_9 will now be available as a choice.
newPref color variablesColor	    {black}	Tcl	{Tcl::updateColors}
# Colour for strings
newPref color stringColor	    {green}	Tcl	{Tcl::updateColors}
# Colour of symbols such as \, -, +, *, etc.  Can be useful for
# reading regular expressions.
newPref color symbolColor	    {black}	Tcl	{Tcl::updateColors}

## 
 # -------------------------------------------------------------------------
 # 
 # "Tcl::_updateKeywords" --
 # 
 #  This proc now includes support for optional separate colorization of 
 #  alpha commands. To use, set 'alphaKeyWordColor' to something other than 
 #  'none' in the Tcl Mode Preferences dialog. -trf
 # -------------------------------------------------------------------------
 # 
 # Now split into a series of procs, called in the end by colorizeTcl.  -cbu
 # 
 # -------------------------------------------------------------------------
 ##

#===============================================================================
#
# Color procs begin here #
#


#===============================================================================
#
# Color Tcl Keywords
#

proc Tcl::colorTclKeywords {} {
    global TclmodeVars

    # all except beep and echo are basic Tcl keywords

    set tclKeyWords {
	after append array auto_execok auto_import auto_load
	auto_load_index auto_qualify beep binary break case catch cd clock
	close concat continue dde default echo else elseif encoding eof
	error eval exec exit expr fblocked fconfigure fcopy file
	fileevent flush for foreach format gets glob global history if
	incr info interp join lappend lindex linsert list llength load
	lrange lreplace lsearch lsort namespace open package pid
	pkg_mkIndex proc puts pwd read regexp regsub rename resource
	return scan seek set socket source split string subst switch
	tclLog tclMacPkgSearch tclPkgSetup tclPkgUnknown tell then time
	trace unknown unset update uplevel upvar variable vwait while
    }
    
    regModeKeywords -a				\
      -e {#} -c $TclmodeVars(commentColor)      \
      -s $TclmodeVars(stringColor)              \
      -k $TclmodeVars(commandColor) Tcl  	\
      $tclKeyWords 

}

#===============================================================================
#
# Color Tk Keywords
#

proc Tcl::colorTkKeywords {} {
    global TclmodeVars

    set tkKeyWords {
	bell bind bindtags button canvas checkbutton console destroy
	entry event focus font frame grab grid image label listbox menu
	menubutton message pack place radiobutton raise scale scrollbar
	text tk tkwait toplevel winfo wm
    }

    if {$TclmodeVars(recogniseTk)} {
	 regModeKeywords -a 			\
	  -k $TclmodeVars(commandColor) Tcl	\
	  $tkKeyWords
    } else {
	 regModeKeywords -a 			\
	  -k {black} Tcl			\
	  $tkKeyWords
    }
}

#===============================================================================
#
# Color iTcl Keywords
#

proc Tcl::colorItclKeywords {} {
    global TclmodeVars

    set itclKeyWords {
	@scope body class code common component configbody constructor
	define destructor hull import inherit itcl itk itk_component
	itk_initialize itk_interior itk_option iwidgets keep method
	private protected public
    }

    if {$TclmodeVars(recogniseItcl)} {
	regModeKeywords -a 				\
	 -k $TclmodeVars(commandColor) Tcl	 	\
	 $itclKeyWords
   } else {
	regModeKeywords -a 				\
	 -k {black} Tcl				 	\
	 $itclKeyWords
   }
}

#===============================================================================
#
# Color Pseudo Tcl Keywords
#

proc Tcl::colorPseudoTclKeywords {} {
    global TclmodeVars
	
    set PseudoTclKeywords { 
	lcontains lunion lreverse lremove lunique
    }
    if {$TclmodeVars(recogniseTk)} {
	regModeKeywords -a 				\
	 -k $TclmodeVars(commandColor) Tcl	 	\
	 $PseudoTclKeywords
    } else {
	regModeKeywords -a 				\
	 -k {black} Tcl				 	\
	 $PseudoTclKeywords
    }
    
}

#===============================================================================
#
# Color Tk Commands
#
# add this line if we can handle double 'magic chars'
# -m {tk} 
#

proc Tcl::colorTkCommands {} {
    
    global TclmodeVars

    set TkCommands {
	tkButtonDown tkButtonEnter tkButtonInvoke tkButtonLeave tkButtonUp
	tkCancelRepeat tkCheckRadioInvoke tkDarken tkEntryAutoScan
	tkEntryBackspace tkEntryButton1 tkEntryClosestGap tkEntryInsert
	tkEntryKeySelect tkEntryMouseSelect tkEntryNextWord tkEntryPaste
	tkEntryPreviousWord tkEntrySeeInsert tkEntrySetCursor
	tkEntryTranspose tkEventMotifBindings tkFDGetFileTypes tkFirstMenu
	tkFocusGroup_BindIn tkFocusGroup_BindOut tkFocusGroup_Create
	tkFocusGroup_Destroy tkFocusGroup_In tkFocusGroup_Out tkFocusOK
	tkListboxAutoScan tkListboxBeginExtend tkListboxBeginSelect
	tkListboxBeginToggle tkListboxCancel tkListboxDataExtend
	tkListboxExtendUpDown tkListboxMotion tkListboxSelectAll
	tkListboxUpDown tkMbButtonUp tkMbEnter tkMbLeave tkMbMotion
	tkMbPost tkMenuButtonDown tkMenuDownArrow tkMenuDup tkMenuEscape
	tkMenuFind tkMenuFindName tkMenuFirstEntry tkMenuInvoke tkMenuLeave
	tkMenuLeftArrow tkMenuMotion tkMenuNextEntry tkMenuNextMenu
	tkMenuRightArrow tkMenuUnpost tkMenuUpArrow tkMessageBox
	tkPostOverPoint tkRecolorTree tkRestoreOldGrab tkSaveGrabInfo
	tkScaleActivate tkScaleButton2Down tkScaleButtonDown
	tkScaleControlPress tkScaleDrag tkScaleEndDrag tkScaleIncrement
	tkScreenChanged tkScrollButton2Down tkScrollButtonDown
	tkScrollButtonUp tkScrollByPages tkScrollByUnits tkScrollDrag
	tkScrollEndDrag tkScrollSelect tkScrollStartDrag tkScrollToPos
	tkScrollTopBottom tkTabToWindow tkTearOffMenu tkTextAutoScan
	tkTextButton1 tkTextClosestGap tkTextInsert tkTextKeyExtend
	tkTextKeySelect tkTextNextPara tkTextNextPos tkTextNextWord
	tkTextPaste tkTextPrevPara tkTextPrevPos tkTextResetAnchor
	tkTextScrollPages tkTextSelectTo tkTextSetCursor tkTextTranspose
	tkTextUpDownLine tkTraverseToMenu tkTraverseWithinMenu tk_bisque
	tk_chooseColor tk_dialog tk_focusFollowsMouse tk_focusNext
	tk_focusPrev tk_getOpenFile tk_getSaveFile tk_messageBox
	tk_optionMenu tk_popup tk_setPalette tk_textCopy tk_textCut
	tk_textPaste
    }
    
    if {[info tclversion] >= 8.4} {
	regsub -all {tk([A-Z])} $TkCommands {tk::\1} TkCommands
    }
    
    if {$TclmodeVars(recogniseTk)} {
	regModeKeywords -a 			\
	 -k $TclmodeVars(commandColor) Tcl 	\
	 $TkCommands
    } else {
	regModeKeywords -a 			\
	 -k {black} Tcl 			\
	 $TkCommands   

    }
    unset TkCommands    
}
    
#===============================================================================
#
# Color Symbols
# 

proc Tcl::colorSymbols {} {
    global TclmodeVars
    
    regModeKeywords -a                  \
      -i "+" -i "-" -i "*" -i "\\"      \
      -I $TclmodeVars(symbolColor) Tcl {}
}

#===============================================================================
#
# Color variables (i.e. words prefaced by '$')
# 

proc Tcl::colorVariables {} {
    
    global TclmodeVars
    
    regModeKeywords -a 				\
      -m {$}					\
      -k $TclmodeVars(variablesColor) Tcl {}        
}

#===============================================================================
#
# Colorize Tcl
# 

proc Tcl::colorizeTcl {} {
    global tclCmdColourings tclExtraColourings 
    foreach p $tclCmdColourings {
	$p
    }
    # for some reason, these want to be done last -trf
    foreach p $tclExtraColourings {
	$p
    }
    refresh
}

# This is a "dummy" command, necessary for the above proc so that all of
# the "regModeKeywords" commands in the called color procs can be "adds"
# (-a).  When the mode is first invoked, this has to occur before the color
# procs are called.

regModeKeywords -k {none} Tcl {}

# now we finally colorize

Tcl::colorizeTcl

#===============================================================================
#
# Tcl:: Update Colors -- 
# 
# This allows for changes to take effect without a restart.
# 
# Danger:  Don't include this proc in any {mode}Prefs.tcl file !!!
# 
# This will source the prefs file, and thus put Alpha in an endless loop.
# Instead, use the  Tcl::colorizeTcl  proc in the prefs file, so that
# "Load Prefs File" will update any local variables.     - cbu
# 

proc Tcl::updateColors {flag} {
    
    global mode PREFS $flag TclmodeVars 
    
    # If the mode has a {mode}Prefs.tcl file, we want to load that as 
    # well, otherwise any keywords contained therein won't be updated
    # without a manual "Load Prefs File".
    
    if {[file exists [file join ${PREFS} ${mode}Prefs.tcl]]} {
	uplevel #0 [list source [file join ${PREFS} ${mode}Prefs.tcl]]
    } 

    Tcl::colorizeTcl
}

#===============================================================================
#
# Regular Expression Colors --
# 
# Changes color scheme of current window to make it easier to read regular
# expressions.  Preferences aren't actually changed.  "defaultColors" will
# restore to the last stored values of the colors.  -cbu
# 

proc Tcl::regularExpressionColors {} {
    
    regModeKeywords -a -e {} -s {none} -m {$} -k {magenta} \
      -i "+" -i "-" -i "*" -i "\\" \
      -I {red} Tcl {}
    
    refresh
}

proc Tcl::defaultColors {} {Tcl::colorizeTcl}

#  end of keyword colorizing  #

#===============================================================================

proc Tcl::MenuProc {menu item} {
    switch -glob $item {
	"reformatProc" {
	    procs::reformatEnclosing [getPos]
	}
	"reloadProc" {
	    procs::loadEnclosing [getPos]
	}
	"debugProc" {
	    set func [procs::pick 1]
	    procs::debug $func
	}
	"applyChanges" {
	    set w [win::Current]
	    if {[regexp -- {\* Debug of (.*) \*( <[0-9]+>)?} $w "" proc]} {
		set f [procs::searchFor $proc]
		if {[string length $f]} {
		    if {![catch {procs::replace $f $proc 0}]} {
			bringToFront $w
			setWinInfo -w $w dirty 0
			killWindow
		    }
		} else {
		    message "Couldn't find $proc"
		}
	    } else {
		message "No debug window is foremost"
	    }
	}
	"findProcDefinition" {
	    procs::findDefinition
	}
	"quickFindProc" {
	    # use the status line
	    procs::quickFindDefn
	}
	"switch*" {
	    set v "[string tolower [string range $item 8 end]]Sig"
	    global $v
	    app::launchFore [set $v]
	}
	"addRemoveDollars" {
	    togglePrefix \$
	}
	default {
	    menu::generalProc Tcl $item 0
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "Tcl::rebuildTclIndexForWin" --
 # 
 #  If the file is in Alpha's source tree, use the currently loaded
 #  auto_mkindex.  If it is not, then fire up a separate Tcl application
 #  and use its auto_mkindex (i.e. the standard Tcl one).  It just occured
 #  to me that for Tcl >= 8.0, we could create a new interp, and 
 #  execute auto_mkindex within that to the same effect, but without
 #  the overhead of a whole new process (especially a Tk one!).
 # -------------------------------------------------------------------------
 ##
proc Tcl::rebuildTclIndexForWin {} {
    set dir [file dirname [win::Current]]
    if {[alpha::inAlphaHierarchy [win::Current]]} {
	auto_mkindex $dir
	auto_reset
    } else {
	# This will currently launch a Tk shell, which isn't ideal.
	tcltk::launchNewShell [list auto_mkindex $dir] "exit"
    }
    message "Tcl index for the directory \"$dir\" has been rebuilt."
}

proc tcltk::menuProc {menu item} {
    global tcl_platform tclshSig
    switch -- $item {
	"evaluateRemotely" {
	    global evaluateRemotely
	    set evaluateRemotely [expr {1 - $evaluateRemotely}]
	}
	"executeFileInRemoteShell" {
	    tcltk::executeInRemoteShell [win::Current]
	}
	"addWindowToListOfExecutables" {
	    global tcltk::executables
	    if {[lsearch -exact [set tcltk::executables] \
	      [win::StripCount [win::Current]]] == -1} {
		lappend tcltk::executables [win::Current]
		prefs::modified tcltk::executables
		menu::buildSome tclMenu
		message "Window [file tail [win::Current]] added\
		  to executable list"
	    } else {
		message "Window [file tail [win::Current]] is already\
		  in the list of executables!"
	    }
	}
	"executeCommand" {
	    set cmd [getline "Please enter the script to send to tcl-tk"]
	    if {$cmd == ""} {return}
	    if {$tcl_platform(platform) == "macintosh"} {
		set res [AEBuild -r -t 30000 '$tclshSig' misc dosc ---- "$cmd"]
	    } else {
		set res [tcltk::evaluate $cmd]
	    }
	    alertnote "Result was '$res'"
	}
	default {
	    global tcltk::executables
	    foreach ex ${tcltk::executables} {
		if {[file tail $ex] == $item} {
		    tcltk::executeInRemoteShell $ex
		    break
		}
	    }
	}
    }
}

proc tcltk::executeInRemoteShell {f} {
    global evaluateRemotely
    set realName [win::StripCount $f]
    tcltk::launchNewShell \
      "cd [file dirname $realName]" \
      "source [file tail $realName]"
    if {!$evaluateRemotely} {
	set evaluateRemotely 1
    }
}

if {[info tclversion] < 8.0} {
    proc tcltk::evaluateRemoteSynchronise {args} {
	global evaluateRemotely tclMenu
	catch {markMenuItem "tcl-tk" evaluateRemotely $evaluateRemotely}
	if {$evaluateRemotely} {
	    if {[info commands notRemoteEvaluate] == ""} {
		rename evaluate notRemoteEvaluate
		;proc ::evaluate {args} {uplevel 1 [list remoteEvaluate] $args}
	    }
	    menu::replaceRebuild tclMenu "320"
	} else {
	    if {[info commands notRemoteEvaluate] != ""} {
		rename evaluate {}
		rename notRemoteEvaluate evaluate
	    }
	    menu::replaceRebuild tclMenu "269"
	}
    }
} else {
    proc tcltk::evaluateRemoteSynchronise {args} {
	global evaluateRemotely tclMenu
	catch {markMenuItem "tcl-tk" evaluateRemotely $evaluateRemotely}
	if {$evaluateRemotely} {
	    if {[info commands ::notRemoteEvaluate] == ""} {
		rename ::evaluate ::notRemoteEvaluate
		;proc ::evaluate {args} {uplevel 1 [list remoteEvaluate] $args}
	    }
	    menu::replaceRebuild tclMenu "320"
	} else {
	    if {[info commands ::notRemoteEvaluate] != ""} {
		rename ::evaluate {}
		rename ::notRemoteEvaluate ::evaluate
	    }
	    menu::replaceRebuild tclMenu "269"
	}
    }
}

proc evaluate {{str ""}} {
    if {![string length $str]} {
	set str [getSelect]
	if {![string length $str]} {
	    set str [getText [minPos] [maxPos]]
	}
    }
    global tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	regsub -all "\r" $str "\n" str
    }
    message "Result: [uplevel \#0 $str]"
}

proc remoteEvaluate {{str ""}} {
    if {![string length $str]} {
	set str [getSelect]
	if {![string length $str]} {
	    set str [getText [minPos] [maxPos]]
	}
    }
    global tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	regsub -all "\r" $str "\n" str
    }
    message "Remote reply: [tcltk::evaluate $str]"
}

proc tcltk::quitRemote {} {
    global tclshSig tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	app::ensureRunning $tclshSig
	if {[catch {set r [tclAE::build::resultData -t 30000 '${tclshSig}' \
		misc dosc \
		---- [tclAE::build::TEXT exit] \
	    ]} res]} {
	    set res "Error: $res"
	}            
    } else {
	global tclshInterp
	if {![info exists tclshInterp]} {
	    if {[catch {tcltk::findTclshInterp}]} {
		return "No shell selected"
	    }
	}
	if {$tcl_platform(platform) == "windows"} {
	    if {[dde services TclEval $tclshInterp] == ""} {
		alertnote "The remote shell has died, please select a new one."
		unset tclshInterp
		return [tcltk::quitRemote exit]
	    }
	    catch {dde execute TclEval $tclshInterp "exit"}
	} else {
	    catch {send $tclshInterp exit}
	}
    }
}

proc tcltk::evaluate {what} {
    global tclshSig tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	app::ensureRunning $tclshSig
	if {[catch {set r [tclAE::build::resultData -t 30000 '${tclshSig}' \
            	misc dosc \
          	---- [tclAE::build::TEXT $what] \
            ]} res]} {
	    set res "Error: $res"
	}            
	#catch {dosc -c '${tclshSig}' -s $what} res
	#return $res
    } else {
	global tclshInterp
	if {![info exists tclshInterp]} {
	    if {[catch {tcltk::findTclshInterp}]} {
		return "No shell selected"
	    }
	}
	if {$tcl_platform(platform) == "windows"} {
	    if {[dde services TclEval $tclshInterp] == ""} {
		alertnote "The remote shell has died, please select a new one."
		unset tclshInterp
		return [tcltk::evaluate $what]
	    }
	    dde execute TclEval $tclshInterp [list catch $what alpha_result]
	    return [dde request TclEval $tclshInterp alpha_result]
	} else {
	    catch {send $tclshInterp $what} res
	}
    }
    return $res
}

proc tcltk::listInterps {} {
    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	set res {}
	foreach service [dde services TclEval ""] {
	    lappend res [lindex $service 1]
	}
	return $res
    } else {
	return [winfo interps]
    }
}

proc tcltk::findTclshInterp {} {
    global tclshInterp tclshSigs tclshSig
    set old [tcltk::listInterps]
    set shel [listpick -p "Use which Tcl shell?" [concat $old \
      [list "------------------" "Launch new shell"]]]
    if {$shel == "Launch new shell"} {
	tcltk::launchNewShell
    } else {
	set tclshInterp $shel
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "tcltk::launchNewShell" --
 # 
 #  Startup up a new Tcl shell, ensuring that we can communicate with that
 #  shell.  On Unix/MacOS this should be easy using 'send' or apple-events
 #  respectively.  On Windows we have to set up the new shell as a dde
 #  server.  We do this with the script 'winRemoteShell.tcl'.
 #  
 #  Any extra 'args' passed to this procedure are executed, one by one,
 #  in the new shell.
 # -------------------------------------------------------------------------
 ##
proc tcltk::launchNewShell {args} {
    global tclshInterp tclshSigs tclshSig tcl_platform HOME
    set old [tcltk::listInterps]
    if {$tcl_platform(platform) == "windows"} {
	app::runScript tclsh "Tcl shell" [file join $HOME Tools winRemoteShell.tcl] 2
    } else {
	app::launchElseTryThese $tclshSigs tclshSig "Please locate the remote Tcl application"
    }
    while {[tcltk::listInterps] == $old} {
	update
    }
    set tclshInterp [lremove -l [tcltk::listInterps] $old]
    # We're left with two items
    set tclshInterp [lindex $tclshInterp 0]
    
    if {[llength $args]} {
	foreach arg $args {
	    set res [tcltk::evaluate $arg]
	}
	return $res
    }
    return ""
}

#  Quick Find Proc  #

proc procs::quickFindDefn {} {
    Tcl::DblClickHelper [prompt::statusLineComplete "proc" procs::complete]
}

if {[info tclversion] < 8.0} {
    proc procs::complete {pref} {
	return [info commands ${pref}*]
    }
} else {
    proc procs::complete {pref} {
	if {[regexp {(.*)([^:]+)$} $pref "" start tail]} {
	    set cmds [info commands ${pref}*]
	    foreach child [namespace children ::$start] {
		if {[string match "::${tail}*" $child]} {
		    foreach cmd [info commands ${start}${child}::*] {
			lappend cmds [string trimleft $cmd :]
		    }
		}
	    }
	    return $cmds
	} else {
	    return [info commands ${pref}*]
	}
    }
}

#  electric behaviour  #
proc Tcl::electricLeft {} {
    if {[literalChar]} { insertText "\{"; return }
    set pat "\}\[ \t\r\n\]*(else(if)?)\[ \t\r\n\]*\$"
    set p [getPos]
    if { [set res [findPatJustBefore "\}" "$pat" $p word]] == "" } { 
	insertText "\{"
	return
    }
    # we have an if/else(if)/else
    global structuralElectricElseBrace
    # The behaviour here is optional, because some people may not
    # like this more structured entry.
    if {$structuralElectricElseBrace} {
	switch -- $word {
	    "else" {
		deleteText [lindex $res 0] $p
		elec::Insertion "\} $word \{\r\t\r\}\r"
	    }
	    "elseif" {
		deleteText [lindex $res 0] $p
		elec::Insertion "\} $word \{\} \{\r\t\r\}\r"
	    }
	}
    } else {
	switch -- $word {
	    "else" {
		replaceText [lindex $res 0] $p "\} $word \{\r"
		bind::IndentLine
	    }
	    "elseif" {
		replaceText [lindex $res 0] $p "\} $word \{"
	    }
	}
    }
}

proc Tcl::electricRight {} {
    if {[literalChar]} { insertText "\}"; return }
    set p [getPos]
    if { [regexp "\[^ \t\]" [getText [lineStart $p] $p]] } {
	insertText "\}"
	blink [matchIt "\}" [pos::math $p - 1]]
	return
    }
    set start [lineStart $p]
    insertText "\}"
    createTMark tcl_er [getPos]
    backwardChar
    bind::IndentLine
    gotoTMark tcl_er ; removeTMark tcl_er
    bind::CarriageReturn
    blink [matchIt "\}" [pos::math $start - 1]]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "Tcl::correctIndentation" --
 # 
 #  Returns the correct indentation for the line containing $pos, if that
 #  line were to contain ordinary characters only.  It is the 
 #  responsibility of the calling procedure to ensure that if we are to
 #  insert/have a line already, that that information is taken into
 #  account, by passing in the argument 'next'
 # -------------------------------------------------------------------------
 ##
proc Tcl::correctIndentation {pos {next ""}} {
    global indent_amounts indentSlashEndLines
    # preliminaries
    if {[pos::compare [set beg [lineStart $pos]] == [minPos]]} { return 0 }
    # if the current line is a comment, we have to check some
    # special cases
    if {[string index $next 0] == "\#"} {
	set p [prevLineStart $beg]
	if {[catch {set p [search -s -f 0 -r 1 -i 0 -m 0 "^\[ \t\]*\[^ \t\r\n\]" \
	  [pos::math $beg - 1]]}]} {
	    # check for search bug at beginning of file.
	    if {[pos::compare $p == [minPos]]} {
		if {[getText [minPos] [pos::math [minPos] + 2]] == "\#\#"} {
		    if {([string range $next 0 1] != "\#\#")} {
			return 1
		    } else {
			return 0
		    }
		}
	    }
	    return 0
	}
	set prev [pos::math [lindex $p 1] - 1]
	set p [lindex $p 0]
	if {[lookAt $prev] != "\#" || ($beg == [minPos])} {
	    # not a comment, so indent with code
	} else {
	    set lwhite [posX $prev]
	    # it's a comment
	    if {[getText $prev [pos::math $prev + 2]] == "\#\#" && \
	      [lookAt [pos::math $prev + 2]] != "\#" \
	      && ([string range $next 0 1] != "\#\#")} {
		# it's a comment paragraph
		incr lwhite 
	    }
	}
    }
    set next [string index $next 0]
    if {![info exists lwhite]} {
	if {![catch {search -s -f 0 -r 1 -i 0 -m 0 "^\[ \t\]*\[^\# \t\r\n\]" [pos::math $beg - 1]} lst]} {
	    # Find the last non-comment line and get its leading whitespace	
	    set lwhite [posX [pos::math [lindex $lst 1] - 1]]
	    set pe1 [lookAt [pos::math $beg - 2]]
	    set lst [lindex $lst 0]
	    set lastC [lookAt [lindex [search -s -f 0 -r 1 -i 0 -m 0 "\[^ \t\r\n\]" [pos::math [nextLineStart $lst] - 1]] 0]]
	    if {$next == "\}"} {
		incr lwhite $indent_amounts(-2)
		set pe2 [lookAt [pos::math [prevLineStart $beg] - 2]]
		if {$pe1 == "\\"} {
		    incr lwhite $indent_amounts(1)
		} else {
		    if {$pe2 == "\\"} {
			incr lwhite $indent_amounts(-1)
		    }
		}
		if {$lastC == "\{"} {incr lwhite $indent_amounts(2)}	
	    } else { 
		if {$pe1 == "\\"} {
		    if {[lookAt [pos::math [prevLineStart $beg] - 2]] != "\\"} {
			incr lwhite $indent_amounts($indentSlashEndLines)
		    }
		} else {
		    if {$lastC == "\{"} {incr lwhite $indent_amounts(2)}	
		    if {[lookAt [pos::math $lst - 2]] == "\\"} {
			incr lwhite $indent_amounts(-$indentSlashEndLines)
		    }
		}
	    }
	} else {
	    # basically failed in all the above, so keep current indentation
	    set lwhite [posX [text::firstNonWsLinePos $beg]]
	}
    }
    return [expr {$lwhite > 0 ? $lwhite : 0}]
}

#  Tcl Menu support  #

proc procs::reformatEnclosing {pos} {
    set p [procs::findEnclosing $pos "proc|((itcl::)?(body|configbody))" 0 1]
    eval select $p
    ::indentRegion
}

proc procs::loadEnclosing {pos} {
    if {[catch {procs::findEnclosing $pos "proc|((itcl::)?(body|configbody))" 0 1} p]} {
	evaluateLine $pos
    } else {
	eval select $p
	if {[catch {uplevel \#0 evaluate} err]} {
	    if {[regexp {can't create procedure "(.*)": unknown namespace} $err "" pr]} {
		if {[dialog::yesno "The procedure '$pr' couldn't be loaded, because\
		  it is in an unknown namespace.  Shall I create the namespace and\
		  try again?"]} {
		    ensureNamespaceExists $pr
		    return [procs::loadEnclosing $pos]
		}
	    }
	} else {
	    message "Loaded procedure '[procs::findEnclosingName $pos]'"
	}
    }
    goto $pos
}

proc procs::findDefinition {{func ""}} {
    # Select a function.
    if {$func == ""} {set func [procs::pick 1]}
    set fname [procs::find $func]
    if {$fname == ""} {
	global TclmodeVars
	set fname [procs::findIn $func $TclmodeVars(procSearchPath)]
    }
    if {$fname == ""} {
	status::errorMsg "Can't find the source file for '$func'."
    }
    # We have a filename, so try to find the mark, first using truncated
    # versions of 'func' so that we can look for the actual mark name. 
    # 'func' might be something like 'Bib::markFile', so first try to find
    # the '::markFile' mark.  This is used in Tcl < 8.0 only.
    if {[regexp {^[^:]+:+.*$} $func]} {
	regsub {^[^:]+} $func {} func1
	if {[string length $func1] && [editMark $fname $func1]} {return}
    } 
    # 'func' might be something like '::Bib::MarkFile', so try to
    # find the '::MarkFile' mark.
    if {[regexp {^::[^:]+::[^:]+.*$} $func]} {
	regsub {^::[^:]+} $func {} func1
	if {[string length $func1] && [editMark $fname $func1]} {return}
    } 
    # 'func' might be something like '::markFile', so try to find the
    # 'markFile' mark.  (This would pick up anything like '::electricLeft'
    # as well.)
    if {[regexp {^::[^:]+$} $func]} {
	regsub {^::} $func {} func1
	if {[string length $func1] && [editMark $fname $func1]} {return}
    } 
    # Still here ...  try removing all leading namespaces and '::'.  This
    # works for funcs like '::Bib::Completions::Entry'.
    if {[regexp {^.*::[^:]+.*$} $func]} {
	regsub {^.*::} $func {} func1
	if {[string length $func1] && [editMark $fname $func1]} {return}
    } 
    # Now try the full 'func' name, just so we can say we tried.  (If
    # everything else failed, this probably will as well.)
    if {[editMark $fname $func]} {return}
    # We tried ...
    set fname [file tail $fname]
    status::errorMsg "Couldn't find the definition of '$func' in '$fname'"
}

## 
 # -------------------------------------------------------------------------
 # 
 # "insertDivider" --
 # 
 #  Modified from Vince's original to allow you to just select part of
 #  an already written comment and turn it into a Divider. -trf
 # -------------------------------------------------------------------------
 ##
proc insertDivider {} {
    if {[isSelection]} {
	set enfoldThis [getSelect]
	beginningOfLine
	killLine
	insertText "#  $enfoldThis  #"
	return
    } 
    elec::Insertion "#    #"
}

#  Info providers  #

proc Tcl::DblClick {from to shift option control} {
    # if cmd and cntrl were pressed, we look to select part of
    # a combination word (less any leading dollar sign) -trf
    if {$control != 0} {
	set clickedPos [getPos]	
	if {[lookAt $from] == "\$"} {
	    set from [pos::math $from + 1]
	} 
	set sel_start $clickedPos 
	set selStartNotDetermined 1
	while {$selStartNotDetermined && ([pos::compare $sel_start > $from])} {
	    set char [lookAt $sel_start] 
	    if {[regexp {_} $char]} {
		set sel_start [pos::math $sel_start + 1]
		set selStartNotDetermined 0
	    } elseif {[regexp {[A-Z]} $char]} {
		set selStartNotDetermined 0
	    } else {
		set sel_start [pos::math $sel_start -1]
	    } 
	}
	set sel_end   $clickedPos 
	set selEndNotDetermined 1
	while {$selEndNotDetermined && ([pos::compare $sel_end <= $to])} {
	    set char [lookAt $sel_end] 
	    if {[regexp "\[A-Z_ \t\r\]" $char]} {
		set selEndNotDetermined 0
	    } else {
		set sel_end [pos::math $sel_end + 1]
	    } 
	}
	select $sel_start $sel_end 
	return
    } elseif {$shift != 0} {
	# Attempt to put the 'args' of the proc in the status window.
	set word [getText $from $to]
	if {[catch {info args $word} arguments]} {
	    set msg "Couldn't find the arguments for '$word'"
	} else {
	    set msg "$word arguments:  $arguments"
	}
	status::msg $msg
	return
    } 
    
    # otherwise, we try to impart some extra info
    select $from $to
    
    # This might have been called from within a Trace window, 
    # in which case we want to move this window to the right
    # and push its mark.
    if {[regexp "^\\* (Tcl error|Trace|Stack|Error Info)" [win::CurrentTail]]} {
	lappend selectionEndPoints [getPos] [selEnd]
	if {[Tcl::findErrorInfoLocation [getPos]]} {
	    return
	}
    } 

    # if this was called using the shortcut key, the selection 
    # may include more than just the proc name, so we dust off
    # the selection first.
    set procName [getSelect] 
    set procName [string trimright $procName "'"]
    regexp -- "^proc\[ \t\]+(\[^ \t\]+)" $procName all procName 
    set procName [string trim $procName]

    if {[catch [list Tcl::DblClickHelper $procName]]} {
	message "No docs $shift $control $option"
    } else {
	bringToFront [win::CurrentTail]
    }
}

# Return 1 if successful, which also implies we have placed a bookmark
proc Tcl::findErrorInfoLocation {pos} {
    set text [getText [lineStart $pos] [lineEnd $pos]]
    if {![regexp "^(\[ \t\]*)\\\((.*)\\\)\[ \t\]*\$" $text "" pre text]} {
	return 0
    }
    set start [pos::math [lineStart $pos] + [string length $pre]]
    set end [pos::math $start + [string length $text] + 2]
    select $start $end
    
    set words [parseWords $text]
    set line [lindex $words end]

    if {[lindex $words 0] == "procedure"} {
	set procName [lindex $words 1]
	placeBookmark
	if {![catch [list procs::debug $procName $line]]} {
	    return 1
	}
    } elseif {[lindex $words 1] == "arm"} {
	# a switch arm
	set procPos [lindex [search -s -f 1 -r 1 \
	  "^    \\\(procedure " [nextLineStart $pos]] 1]
	if {![Tcl::findErrorInfoLocation $procPos]} {
	    return 0
	}
	# we should have opened the procedure
	set swPos [lindex [search -s -f 1 -- [lindex $words 0] [getPos]] 0]
	set row [lindex [posToRowCol $swPos] 1]
	incr row $line
	set start [rowColToPos $row 0]
	select $start [nextLineStart $start]
	return 1
    }
    
    return 0
}

# Now finds commands in Alpha Commands,
# which has a <cr> immediately after them, e.g. beep, ticks.
proc Tcl::DblClickHelper {text} {
    global HOME auto_index auto_path TclmodeVars
    # Is it a loadable proc?
    if {![catch {procs::findDefinition $text}]} {
	return
    }
    
    if {[info exists "auto_index($text)"]} {
	if {[editMark "$auto_index($text)" $text]} {
	    # some marking schemes commonly used for Tcl modes
	    goto [lindex [search -s -f 1 -r 1 -m 0 -- "proc\[ \t\]+${text}\[ \t\]" [minPos]] 0]
	}
	return
    }
    # Is it a built-in Alpha command?
    set lines [grep "^ $text\( |\$)" [file join $HOME Help "Alpha Commands"]]
    if {[string length $lines]} {
	help::openFile "Alpha Commands"
	if {[catch {editMark [file join $HOME Help "Alpha Commands"] $text}]} {
	    # mark failed for some reason, but we have the line number
	    # anyway.
	    goto [rowColToPos [string trimright [lindex [lindex [split $lines "\n"] 1] 3] :] 0]
	}
	setWinInfo read-only 1
	return
    }
    # Is it a core Tcl command?
    if {[info tclversion] >= 8.0} {
	interp create sub
	set res [sub eval "info commands $text"]
	interp delete sub
	if {[string length $res]} {
	    urlView "$TclmodeVars(tclHelpLocation)${text}.htm"
	    return
	}
    }
    
    set lines [grep "^     $text -" [file join $HOME Help "Tcl Commands"]]
    if {[string length $lines]} {
	help::openFile "Tcl Commands"
	if {[catch {editMark [file join $HOME Help "Tcl Commands"] $text}]} {
	    # mark failed for some reason, but we have the line number
	    # anyway.
	    goto [rowColToPos [string trimright [lindex [lindex [split $lines "\n"] 1] 3] :] 0]
	}
	setWinInfo read-only 1
	return
    }
    # Is it a global variable?
    if {[llength [info globals [string trimleft $text {$}]]]==1} {
	showVarValue [string trimleft $text {$}]
	return
    } elseif {[array exists [string trimleft $text {$}]]} {
	showVarValue [string trimleft $text {$}]
	return
    }
    # (becoming desperate) is it a mark in the current file?
    if {[lsearch [getNamedMarks -n] ${text}] != -1} {
	gotoMark $text
	return
    }
    error ""
}

#############################################################################
#  Report the current value of a global variable, chosen interactively
#  from a list of all active variables.
#
#  If the variable is an array, or its value is too big to fit in an 
#  alertnote, then its contents are listed in a new window, otherwise 
#  the variable's value is displayed in an alertnote.
#
proc getVarValue {} {
    if {[catch {getText [getPos] [selEnd]} def]} {set def ""}
    set var [getVarFromList $def]
    if {[string length $var] == 0} return
    showVarValue $var
}

if {[info tclversion] < 8.0} {
    
    proc getVarFromList {{def ""}} {
	return [listpick -p {Which var?} -L $def [lsort -ignore [info globals]]]
    }
    
} else {
    
    proc getVarFromList {{def ""}} {
	set ns [namespace qualifiers $def]
	if {![namespace_exists $ns]} {
	    set def ""
	    set ns ""
	}
	set def [namespace tail $def]
	
	set items {}
	foreach var [info vars "${ns}::*"] {
	    lappend items [namespace tail $var]
	}
	foreach space [namespace children $ns] {
	    lappend items "[namespace tail $space]::"
	}
	
	set items [concat "::" [lsort -ignore $items]]
	set var [listpick -p "Which var in namespace ${ns}::?" -L $def $items]
	if {$var == "::"} {
	    set var [getVarFromList $ns]
	} elseif {[namespace qualifiers $var] != ""} {
	    set var [getVarFromList "${ns}::${var}"]
	} else {
	    set var "${ns}::${var}"
	}
	return $var
    }
}

#############################################################################
#  Report the current value of a global variable, chosen interactively
#  from a list of all active variables.
#
#  If the variable is an array, or its value is too big to fit in an 
#  alertnote, then its contents are listed in a new window, otherwise 
#  the variable's value is displayed in an alertnote.
#
proc showVarValue {var} {
    global $var
    if {![array exists $var]} {
        viewValue $var [set $var]
    } else {
	new -n "* $var *" -info [listArray $var]
	# if 'shrinkWindow' is loaded, call it to trim the output window.
	catch {shrinkWindow 2}
    }
} 

#############################################################################
#  List the name and value of each element of the array $arrName.
#  (Convenient to use as a shell command.)
#
proc listArray {arrName} {
    global $arrName
    if {[array exists $arrName]} {
	set lines {}
        foreach nm [array names $arrName] {
            lappend lines "\"$nm\"\t\{[set ${arrName}($nm)]\}"
        }
        return [join $lines \r]
    } else {
        alertnote "\"$arrName\" doesn't exist in this context"
    }
}


proc 1xTop2xShowComment {} {
    global __1x2xShowPos
    
    ensureset __1x2xShowPos [minPos]
    set pos [getPos]
    # if we have a proc definition selected
    if {[regexp -- proc [getSelect]]} {
	set startCommentPos [procs::getCommentPos $pos]
	if {[pos::compare $__1x2xShowPos == $pos]} {
	    # and, we already put it at the top
	    if {[pos::compare $pos == $startCommentPos]} {
		message "           No comment for proc"
	    } else {
		display $startCommentPos 
	    }
	    
	    
	} else {
	    # else, this is the first time the proc def is to be 
	    # to be moved
	    insertToTop
	    set __1x2xShowPos [getPos]
	    # just check and let user know if a comment is above
	    # this proc.
	    if {[pos::compare [getPos] == $startCommentPos]} {
		message "           No comment for proc"
	    } else {
		message "Comments are above, press again to see them"
	    }
	}
	
    } else {
	# user wants a normal line moved to the top
	insertToTop
	set __1x2xShowPos [minPos]
    }
}

Bind Clear 1xTop2xShowComment Tcl

#  Marking  #

## 
 # -------------------------------------------------------------------------
 #	 
 # "Tcl::parseFuncs" --
 #	
 # This proc is called by the "braces"	pop-up.	It returns a dynamically
 # created, alphabetical, list of "pseudo-marks".
 #	
 #	Author:	Tom Fetherston
 # -------------------------------------------------------------------------
 ## called by the "{}" button
proc Tcl::parseFuncs {} {
    global TclmodeVars
    set end [maxPos]
    set pos [minPos]
    set l {}
    set markExpr "^\[ \t\]*(itcl(::|_))?(class|body|proc|method|(config)?body)\[ \t\]"
    while {![catch {search -s -f 1 -r 1 -m 0 -i 0 "$markExpr" $pos} res]} {
	set start [lindex $res 0]
	set parsed [Tcl::FindEntireFunctionDeclaration $start]
	set t [lindex $parsed 0]
	set end [lindex $parsed 1]
	if {![string length $t]} {
	    # wasn't a well formed list
	    set pos $end
	    continue
	}
	set argLabel {}
	regsub "^itcl(::|_)" [lindex $t 0] "" what
	switch -- [lindex $t 0] {
	    "proc" {
		append argLabel [set word [lindex $t 1]]
		#get the list of arguments
		set argsList [lindex $t 2]
		if {![is::List $argsList]} { 
		    goto $start
		    return -code error "Bug in Tcl::parseFuncs; couldn't\
		      parse the function definition '$argsList' at\
		      position $start"
		}
		
		if {[llength $argsList] > 0} {
		    append argLabel " \{"
		    foreach arg $argsList {
			if {[llength $arg] == 2 } {
			    append argLabel ""
			} elseif {[set arg] != "args"} {
			    append argLabel ""
			} else {
			    append argLabel ""
			}
		    }
		    append argLabel "\}"					
		} 
	    }
	    default {
		append argLabel [set word [lindex $t 1]]
	    }
	}
	if {[info exists cnts($word)]} {
	    # This section handles duplicate. i.e., overloaded names
	    incr cnts($word)
	} else {
	    set cnts($word) 1
	}
	# if this is the only occurence of this proc, remember where it starts
	lappend indx($word) [lineStart [pos::math $start - 1]]
	# associate name and tag
	set tag($word) $argLabel
	
	# advance pos to where we want to start the next search from
	set pos $end
    }
    
    set rtnRes {}
    
    if {[info exists indx]} {
	foreach hn [lsort -ignore [array names indx]] {
	    set num [llength $indx($hn)]
	    if {$num > 1} {
		for {set i 0} {$i < $num} {} {
		    set next [nextLineStart [lindex $indx($hn) $i]]
		    incr i
		    lappend rtnRes "$tag($hn) \($i of $num\)" $next
		}
	    } else {
		set next [nextLineStart [lindex $indx($hn) 0]]
		lappend rtnRes $tag($hn) $next
	    }
	}
    }
    return $rtnRes 
}

proc Tcl::FindEntireFunctionDeclaration {start} {
    set end [nextLineStart $start]
    set t [getText $start $end]
    
    while {[lookAt [pos::math $end -2]] == "\\"} {
	set newEnd [nextLineStart $end]
	if {[pos::compare $newEnd == [maxPos]]} {
	    break
	}
	set t [string range $t 0 [expr {[string length $t] - 3}]]
	append t " " [string trimleft [getText $end $newEnd]]
	set end $newEnd
    }
    set t [string trimright $t]
    if {![regexp "\}\$" $t]} {
	append t "\}"
    }
    if {[is::List $t]} {
	return [list $t $end]
    } else {
	# parseWords actually as a bug when, say, t is 
	# proc Setx::electricRight {{char "\}"}} {}
	if {[catch {parseWords $t} parsed]} {
	    return [list "" $end]
	} else {
	    return [list $parsed $end]
	}
    }
}

# called by the "M" button
proc Tcl::MarkFile {} {
    
    status::msg "Marking File "
    set count 0
    
    global structuralMarks
    
    set end [maxPos]
    set pos [minPos]
    set l {}
    if {$structuralMarks} {
	set markExpr {^[; 	]*(itcl(::|_))?(class|namespace eval|proc|method|(config)?body|# )[ 	]}
    } else {
	set markExpr {^[; 	]*(itcl(::|_))?(class|namespace eval|proc|method|(config)?body)[ 	]}
    }
    set class ""
    set hasMarkers 0
    set already [list]
    while {![catch {search -s -f 1 -r 1 -m 0 -i 0 "$markExpr" $pos} res]} {
	set start [lindex $res 0]
	set parsed [Tcl::FindEntireFunctionDeclaration $start]
	set t [lindex $parsed 0]
	set end [lindex $parsed 1]
	if {![string length $t]} {
	    # wasn't a well formed list
	    set pos $end
	    continue
	}
	regsub "^itcl(::|_)" [lindex $t 0] "" what
	switch -glob [string trim $what ";"] {
	    "proc" -
	    "configbody" { set text [lindex $t 1] }
	    "method" { set text ${class}::[lindex $t 1] }
	    "body" { 
		regexp {[a-zA-Z_][a-zA-Z_/0-9]*::[a-zA-Z_][a-zA-Z_/0-9]* } \
		  "[lindex $t 1] " text
	    }
	    "namespace" {
		set ns [lindex $t 2]
		if {[regexp {[^a-zA-Z0-9:]} $ns]} {
		    set pos $end
		    continue
		}
		set text "${ns} 111" 
	    }
	    "*class" { 
		set class [lindex $t 1]
		set text "${class} 000" 
	    }
	    "#" { 
		regexp "#  (.*) " $t all text
		if {[regexp -- "---*" $text]} {
		    set text "-"
		} else {
		    if {[regexp "^(    )|(	)#  " $t]} {
			set text " $text"
		    } else {
			set text "$text"
		    }				
		}
		set hasMarkers 1
	    }
	}
	set pos $end
	while { [lsearch -exact $already $text] != -1 } {
	    set text "$text "
	}
	lappend already $text

	if {$structuralMarks} {
	    lappend asEncountered $text
	    set arr inds
	} else {
	    if {[string index $t 0] == ";"} {
		set arr iinds
	    } else {
		set arr inds
	    }
	}
	set ${arr}($text) [lineStart [pos::math $start - 1]]
    }
    
    set class "#"
    foreach arr {inds iinds} {
	if {[info exists $arr]} {
	    if {$arr == "iinds"} {
		# What is the goal of this line?
		setNamedMark "-" [minPos] [minPos] [minPos]
	    }
	    if {$structuralMarks} {
		set order $asEncountered
	    } else {
		set order [lsort -ignore [array names $arr]]
	    }
	    foreach f $order {
		if {[set el [set ${arr}($f)]] != 0} {
		    set next [nextLineStart $el]
		} else {
		    set next 0
		} 
		
		if { [string first "000" $f] != -1 } {
		    set ff "Class '[set class [lindex $f 0]]'"
		} elseif { [string first "111" $f] != -1 } {
		    set ff "Namespace '[set class [lindex $f 0]]'"
		} elseif { [string first "${class}::" $f] == 0 } {
		    set ff [string range $f [string length $class] end]
		} else {
		    set ff $f
		}
		if {$hasMarkers && ![string match "*" $ff] } {
		    set ff " $ff"
		} 
		if {[string trim $ff] == "-"} {set ff "-"}
 		while {[lcontains marks $ff]} {append ff " "}
		lappend marks $ff
		setNamedMark $ff $el $next $next
		if {![regexp {(^)|(.*[ ]+)|-} [string trim $ff]]} {incr count}
	    }
	}
    }
    # Return the number of commands in this file
    status::msg "'[win::CurrentTail]' contains $count defined procedures."
}

#  Misc.  #

proc evaluateLine { pos } {
    goto $pos
    beginningLineSelect
    endLineSelect

    uplevel \#0 evaluate
}

tcltk::evaluateRemoteSynchronise

