## -*-Tcl-*-
 # ==========================================================================
 # FILE: "addressBook.tcl"
 #                                   created: 10/22/2001 {10:28:23 AM}
 #                               last update: 10/31/2001 {10:31:07 AM}
 #                               
 # Description:
 # 
 # Manages an address book containing e-mail, snail addresses, as well as
 # other info for user defined entries.
 # 
 # Author: Craig Barton Upright
 # E-mail: <cupright@princeton.edu>
 #   mail: Princeton University
 #         2.N.1 Green Hall,  Princeton, New Jersey  08544
 #    www: <http://www.princeton.edu/~cupright>
 #
 # --------------------------------------------------------------------------
 #  
 # Copyright (c) 2001  Craig Barton Upright
 # 
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #   Redistributions of source code must retain the above copyright
 #    notice, this list of conditions and the following disclaimer.
 # 
 #   Redistributions in binary form must reproduce the above copyright
 #    notice, this list of conditions and the following disclaimer in the
 #    documentation and/or other materials provided with the distribution.
 # 
 #   Neither the name of Alpha/Alphatk nor the names of its contributors may
 #    be used to endorse or promote products derived from this software
 #    without specific prior written permission.
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # DAMAGE.
 # 
 # ==========================================================================
 ##

alpha::feature addressBook 0.3 "global-only" {
    # Initialization script.
    # We require 7.5b3 for new menu::buildProc, dialog::arrangeMenus,
    # composeEmail, ...
    alpha::package require -loose AlphaTcl 7.5b3
    namespace eval AddressBook {}
    namespace eval addressBook {}
    ensureset addressBook::Current "Default"
    set addressBook::DefaultEntryFields  [list \
      "Full Name"  "Organization" \
      "Address Line 1" "Address Line 2" "Address Line 3" \
      "Phone Number" "Email" "Home Page"]
    ensureset addressBook::EntryFields(Default) \
      [set addressBook::DefaultEntryFields]
    # This not only returns the current address book, it automatically makes
    # some of the common address book vars global, and creates several local
    # variable for use in the current procedure.
    proc addressBook::current {{newCurrent ""}} {
	global addressBook::Current addressBook::EntryFields
	# Set a new current addressBook if desired.
	if {[string length $newCurrent]} {set addressBook::Current $newCurrent} 
	# Upvar some local variables.
	upvar 1 current current
	upvar 1 book    book
	upvar 1 books   books
	upvar 1 entries entries
	upvar 1 fields  fields
	# Set these local variables.
	set current [set addressBook::Current]
	set book    AddressBook::$current
	global $book
	set books   [array names addressBook::EntryFields]
	set books   [concat "Default" [lremove $books "Default"]]
	set entries [array names $book]
	ensureset   addressBook::EntryFields($current) [list ]
	set fields  [set addressBook::EntryFields($current)]
	# Uplevel some global variables.
	uplevel 1 [list global \
	  addressBook::Current addressBook::EntryFields AddressBook::$current ]
    }
    # Register a build proc for the menu.
    menu::buildProc addressBook addressBook::buildMenu addressBook::postEval
    proc addressBook::buildMenu {} {
	addressBook::current
	set currentList [linsert $books 1 "(-)"]
	if {[llength $currentList] > 2} {lappend currentList "(-)"}
	lappend currentList "New Address Book" "Rename Address Book" \
	  "Copy Address Book" "Remove Address Book"
	set cProc "addressBook::currentProc"
	set menuList [list \
	  eMail insertMailingLabel createMailingList  (-) \
	  searchEntries searchFields displayAllEntries \
	  [list Menu -m -n currentAddressBook -p $cProc $currentList] (-) \
	  addEntry editEntry renameEntry removeEntry  \
	  collectAllEmails updateMailElectrics (-) \
	  addEntryField removeEntryField arrangeEntryFields \
	  addressBookHelp addressBookPrefs ]
	return [list build $menuList addressBook::menuProc]
    }
    proc addressBook::postEval {args} {
	addressBook::current
	set dim1 [expr {[llength $entries]} ? 1 : 0]
	set menuItems [list \
	  eMail insertMailingLabel createMailingList \
	  searchEntries searchFields \
	  displayAllEntries editEntry renameEntry removeEntry \
	  updateMailElectrics ]
	foreach item $menuItems {
	    enableMenuItem addressBook $item $dim1
	}
	set dim2 [expr {[llength $fields] > 0} ? 1 : 0]
	set dim3 [expr {[llength $fields] > 1} ? 1 : 0]
	enableMenuItem addressBook removeEntryField   $dim2
	enableMenuItem addressBook arrangeEntryFields $dim3
	# Mark the current book in the 'Current Address Book' menu.
	foreach book $books {
	    set mark [expr {$book == $current} ? 1 : 0]
	    markMenuItem -m currentAddressBook $book $mark 
	}
	set dim4 [expr {[llength $books] > 1} ? 1 : 0]
	enableMenuItem -m currentAddressBook "Remove Address Book" $dim4
    }
    # Register open windows hook.
    proc addressBook::registerOWH {{which "register"}} {
        set menuItems [list insertMailingLabel collectAllEmails]
	foreach item $menuItems {
	    hook::register requireOpenWindowsHook [list addressBook $item] 1
	}
    }
    # Call this now.
    addressBook::registerOWH ; rename addressBook::registerOWH ""
    # Register a new entry for the 'New Documents' menu item.
    set {newDocTypes(New Address Book Entry)} addressBook::addEntry
    # To include all entry names as Mail Menu electrics, turn this item on.
    # This will only create electrics for entry names which are a single word,
    # i.e. no spaces||To never use entry names as Mail Menu electrics, turn
    # this item off
    newPref flag autoUpdateMailElectrics 0 Mail
    if {$MailmodeVars(autoUpdateMailElectrics)} {
	addressBook::updateMailElectrics "" 1
    } 
} {
    # Activation script.
    menu::insert   Utils items 0 "(-)"
    menu::insert   Utils submenu "(-)" "addressBook"
    ensureset MailmodeVars(autoUpdateMailElectrics) 0
} {
    # De-activation script.
    menu::uninsert Utils submenu "(-)" "addressBook"
    prefs::removeObsolete MailmodeVars(autoUpdateMailElectrics)
} uninstall {
    this-file
} help {
    This package inserts an 'Address Book' submenu in the Utils menu,
    allowing for the management of a user modified address book which
    can be used to create customized mailing lists||
    This package inserts an 'Address Book' submenu in the Utils menu,
    allowing for the management of a user modified address book which
    can be used to create customized mailing lists||
    
    This package inserts an 'Address Book' submenu in the Utils menu,
    allowing for the management of a user modified address book which can
    be used to create customized mailing lists.  Activate this package by
    using the "Config --> Preferences --> Features" menu item.
    
	  	Creating an Address Book Database
    
    Use the "Utils --> Address Book --> Add Entry" menu item to create new
    entries.  If any e-mail address is selected in the current window, you
    have the option to use this for the 'Email' field.

    The 'Collect All Emails' menu item will scan the current window for all
    e-mail addresses, and allow you to create a new entry for each one. 
    You can easily experiment with this by opening the "Packages" help file
    and using this menu item.  
    
    The entries in the database can be modified at any time using the menu
    items "Utils --> Address Book --> Edit/Rename/Remove Entry".  
    
    The default entry fields used to create the database can be modified by
    using the "Utils --> Address Book --> Add/Remove Entry Field" menu item, 
    or rearranged to suit your tastes.  Note that a 'Comments' field will
    always be appended to the end of the default entry fields list.  (It's
    probably easiest to compose the 'comments' field in a regular window and
    then paste it into the dialog if it is multi-line ...)
  
	  	Custom Mailing Lists, Searching

    Once a database of entries has been created, you can use it to create a
    customized mailing list (including only those fields that you specify),
    or to search the entries or specific fields for specific text or a
    regular expression.  New mailing lists and search results are always
    displayed in a new window, while the "Insert Mailing Label" menu item
    will place the selected entry in the current window.
    
    Database entries selected using the menu item "Insert Mailing Label" and
    "Create Mailing List" will be listed like this:
    
	Craig Barton Upright
	Department of Sociology
	Princeton University
	Wallace Hall, # 127
	Princeton NJ  08544
	<cupright@princeton.edu>

    while search result entries will be displayed like:
    
	Entry: cbu

	Name:               Craig Barton Upright
	Organization:       Department of Sociology
	Address Line 1:     Princeton University
	Address Line 2:     Wallace Hall, # 127
	Address Line 3:     Princeton NJ  08544
	Email:              <cupright@princeton.edu>
    
    although you also given the option to return the search results using
    the first style ("Mailing List Format") if desired.
    
    The item "Utils --> Address Book --> Display All Entries" will list all
    fields for all entries (using the second style) in a new window.
    
	  	Interaction with the Mail Menu

    This package provides limited support for the Mail Menu.
    
    --- All entry names can be automatically added as Mail mode electrics,
    so long as the entry name does not contain any spaces.  Set the Mail 
    mode flag preference for 'Auto Update Mail Electrics' (available when
    the current window is in Mail mode ...) to create these electrics on
    start-up, or whenever a new entry is added.
    
    --- These electrics can be updated at any time using the menu item
    "Address Book --> Update Mail Electrics".
    
    --- The "Config --> Preferernces --> WWW" preference for "Email Using"
    determines if both the menu item "Address Book --> Email ..." and the
    mailing list hyperlinks will attempt to compose the e-mail using the
    Mail Menu or the default OS composing application.
} maintainer {
    "Craig Barton Upright" <cupright@princeton.edu> 
    <http://www.princeton.edu/~cupright/>
}

namespace eval addressBook {}

#  Address Book Prefs  #

# Entries can be added or edited using either a dialog or a text editing
# window.  This preference sets the default method.
newPref var editEntriesUsing "Choose each time" addressBook "" \
  addressBook::EditStyle array

# Search results can be displayed with full field headers, or simply
# Search results can be displayed with full field headers, or simply
# the field text.  This preference sets the default method.
newPref var displaySearchResults "Choose each time" addressBook "" \
  addressBook::SearchStyle array

array set addressBook::EditStyle {
    "Choose each time" {
	addressBook::chooseOption addressBook::EditStyle editEntriesUsing
    }
    "Dialog window"                {return "0"}
    "Text editing window"          {return "1"}
}

array set addressBook::SearchStyle {
    "Choose each time" {
	addressBook::chooseOption addressBook::SearchStyle displaySearchResults
    }
    "Mailing List Format"          {return "-2"}
    "Including Text Header Info"   {return "0"}
    "Delimited by carriage return" {return "1"}
    "Delimited by tab"             {return "2"}
    "Delimited by space"           {return "3"}
}

proc addressBook::findOption {prefName args} {

    global addressBookmodeVars 
    switch $prefName {
	editEntriesUsing          {set prefArray "addressBook::EditStyle"}
	displaySearchResults      {set prefArray "addressBook::SearchStyle"}
    }
    global $prefArray
    eval [set [set prefArray]($addressBookmodeVars($prefName))] $args
}

proc addressBook::chooseOption {prefArray prefName args} {
    global $prefArray
    set p "[quote::Prettify $prefName]"
    set options [lremove [array names $prefArray] "Choose each time"]
    set options [lsort -ignore $options]
    if {![llength $options]} {
	status::errorMsg "No options for $prefName available !!"
    } elseif {[llength $options] == "1"} {
	set val [lindex $options 0]
    } else {
	set setPref "(Set Address Book preferences to avoid this dialog )"
	lappend options $setPref
	set val [listpick -p $p $options]
	if {$val == $setPref} {
	    # This is the only difference from 'prefs::chooseOption'
	    dialog::pkg_options "addressBook"
	    global addressBookmodeVars
	    set val $addressBookmodeVars($prefName)
	}
    }
    eval [set [set prefArray]($val)] $args
}


# ===========================================================================
# 
#  --------  #
# 
#  Address Book menu, support  #
# 

proc addressBook::menuProc {menuName itemName} {
    
    addressBook::current
    
    switch $itemName {
	"eMail" {
	    set email ""
	    set p "E-mail which entry?"
	    if {[catch {addressBook::pickEntry $p} entryName]} {
		status::errorMsg $entryName
	    }
	    while {![string length $email]} {
		if {[catch {addressBook::getEntryField $entryName "Email"} email]} {
		    set email ""
		} elseif {[string length $email]} {
		    break
		}
		set    question "'$entryName' does not have an e-mail address. \r"
		append question "Would you like to edit that entry?"
		if {[askyesno $question] == "yes"} {
		    addressBook::editEntry $entryName
		} else {
		    status::errorMsg "Cancelled."
		}
	    }
	    composeEmail [url::mailto $email]
	}
	"insertMailingLabel" {
	    set p "Insert which entry?"
	    if {[catch {addressBook::pickEntry $p} entryName]} {
		status::errorMsg $entryName
	    } 
	    set p   "Use which fields?"
	    set all "Use All Fields"
	    if {[catch {addressBook::pickField $p 1 $all 0} fieldList]} {
		status::errorMsg $fieldList
	    } 
	    if {[catch {addressBook::createLabel $entryName $fieldList 1} entryText]} {
	        status::errorMsg $entryText
	    } 
	    elec::Insertion $entryText
	}
	"createMailingList" {
	    # Which entries shall we use?
	    set p   "Display which entries?"
	    set all "Use All Entries"
	    if {[catch {addressBook::pickEntry $p 1 $all} entryList]} {
		status::errorMsg $entryList
	    } elseif {[lcontains entryList "Use All Entries"]} {
		set entryList ""
	    } 
	    # Which fields shall we use?
	    set p   "Use which fields?"
	    set all "Use All Fields"
	    if {[catch {addressBook::pickField $p 1 $all 0} fieldList]} {
		status::errorMsg $fieldList
	    } 
	    # Create the entries text.
	    set entryText ""
	    foreach entryName [lsort -ignore $entryList] {
		if {![catch {addressBook::createLabel $entryName $fieldList 1} entryInfo]} {
		    append entryText "${entryInfo}\r"
		}
	    }
	    # Create a new window with the mailing list.
	    set title "Mailing List"
	    set intro "Customized mailing list."
	    addressBook::newWindow $entryText $title $intro
	}
	"searchEntries" {
	    set p "Enter alpha-numeric text or a Regular Expression to search:" 
	    if {[catch {prompt $p ""} pattern]} {
	        status::errorMsg "Cancelled."
	    } elseif {![string length $pattern]} {
		status::errorMsg "Cancelled -- nothing was entered."
	    }
	    addressBook::searchFor $pattern
	}
	"searchFields" {
	    set p   "Search which field?"
	    set all "Search All Fields"
	    if {[catch {addressBook::pickField $p 0 $all 1} field]} {
		status::errorMsg $field
	    }
	    set p "Enter alpha-numeric text or a Regular Expression to search:" 
	    if {[catch {prompt $p ""} pattern]} {
		status::errorMsg "Cancelled."
	    } elseif {![string length $pattern]} {
		status::errorMsg "Cancelled -- nothing was entered."
	    }
	    addressBook::searchFor $pattern $field
	}
	"displayAllEntries" {
	    set fields [concat {"Entry Name"} $fields "comments"]
	    set entryText ""
	    foreach entryName [lsort -ignore $entries] {
		if {![catch {addressBook::createLabel $entryName $fields "-1"} entryInfo]} {
		    append entryText "${entryInfo}\r"
		}
	    }
	    addressBook::newWindow $entryText
	}
	"addEntry"  {
	    set email ""
	    if {[llength [winNames]] && [isSelection]} {
	        set email [getSelect]
		set question "Would you like to add an entry for '$email'?"
		if {![regexp {[-_a-zA-Z0-9.]+@[-_a-zA-Z0-9.]+} $email]} {
		    set email ""
		} elseif {[askyesno $question] != "yes"} {
		    set email ""
		} else {
		    set email [string trim $email]
		    set email [string trimright $email >]
		    set email [string trimleft  $email <]
		}
	    } 
	    regsub {@.+$} $email {} nickName
	    addressBook::addEntry $nickName Email $email
	}
	"renameEntry" {
	    set p "Select an entry to rename:"
	    while {![catch {addressBook::pickEntry $p} oldName]} {
		set p "New name for $oldName"
		if {[catch {prompt $p $oldName} newName]} {
		    status::errorMsg "Cancelled."
		} elseif {$oldName == $newName} {
		    status::msg "Names were the same !!"
		    continue
		}
		set [set book]($newName) [set [set book]($oldName)]
		unset [set book]($oldName)
		prefs::modified [set book]($oldName)
		prefs::modified [set book]($newName)
		status::msg "'$oldName' has been renamed to '$newName'."
		set p "Select another entry to rename, or cancel:"
	    }
	}
	"removeEntry" {
	    set p "Remove which entries?"
	    if {[catch {addressBook::pickEntry $p 1} entryList]} {
		status::errorMsg $entryList
	    } 
	    foreach entry $entryList {
		catch {unset [set book]($entry)}
		prefs::modified [set book]($entry)
	    }
	    addressBook::postEval
	    if {[llength $entryList] == "1"} {
		status::msg "$entryList has been removed."
	    } else {
		status::msg "$entryList have been removed."
	    }
	}
	"collectAllEmails" {
	    requireOpenWindow
	    set pat {[-_a-zA-Z0-9.]+@[-_a-zA-Z0-9.]+}
	    set pos [minPos]
	    while {![catch {search -s -f 1 -r 1 $pat $pos} match]} {
	        lappend results [eval getText $match]
		set pos [lindex $match 1]
	    }
	    if {![info exists results]} {
	        status::msg "No e-mails found in '[win::CurrentTail]'."
		return
	    } 
	    set results [lunique $results]
	    set p "Add an entry for which e-mail?"
	    while {![catch {listpick -p $p $results} email]} {
		regsub {@.+$} $email {} nickName
		if {![catch {addressBook::addEntry $nickName Email $email}]} {
		    set results [lremove $results $email]
		    set p "Select another e-mail, or Cancel."
		}
		if {![llength $results]} {break}
	    }
	    addressBook::postEval
	}
	"addEntryField" {
	    set fields [concat [set addressBook::EntryFields] "comments"]
	    set p "New Address Book Field Name:"
	    if {[catch {prompt $p ""} newField]} {
	        status::errorMsg "Cancelled."
	    } elseif {![string length $newField]} {
	        status::errorMsg "Cancelled -- nothing was entered."
	    } elseif {[lcontains fields $newField]} {
	        status::errorMsg "Cancelled -- '$newField' was already a field."
	    }
	    lappend addressBook::EntryFields($current) [string trim $newField]
	    prefs::modified addressBook::EntryFields($current)
	    status::msg "'$newField' has been added as a default field."
	    set question "Would you like to re-arrange the field order?"
	    if {[askyesno $question] == "yes"} {
	        addressBook::menuProc "" arrangeEntryFields
	    } 
	}
	"removeEntryField" {
	    set p "Remove which fields?"
	    set r [list "comments"]
	    if {[catch {addressBook::pickField $p 1 "" 0 $r} fieldList]} {
		status::errorMsg $fieldList
	    } 
	    set fields [lremove -l $fields $fieldList]
	    set addressBook::EntryFields($current) $fields
	    prefs::modified addressBook::EntryFields($current)
	    addressBook::postEval
	    if {[llength $fieldList] == "1"} {
		status::msg "$fieldList has been removed as a default field."
	    } else {
		status::msg "$fieldList have been removed as default fields."
	    }
	}
	"arrangeEntryFields" {
	    if {[catch {dialog::arrangeItems $fields} newOrder]} {
		return
	    }
	    set addressBook::EntryFields($current) $newOrder
	    prefs::modified addressBook::EntryFields($current)
	    status::msg "The new order has been established."
	}
	"addressBookHelp"  {package::helpFile "addressBook"}
	"addressBookPrefs" {dialog::pkg_options "addressBook"}
	default            {addressBook::$itemName}
    }
}

proc addressBook::currentProc {menuName itemName} {
    
    global addressBook::DefaultEntryFields
    
    addressBook::current
    
    switch $itemName {
	"New Address Book" {
	    set p "New address book name:"
	    if {[catch {prompt $p ""} newBook]} {
		status::errorMsg "Cancelled."
	    } elseif {[lcontains books $newBook]} {
		status::errorMeg "Cancelled -- '$newBook' already exists."
	    }
	    set addressBook::EntryFields($newBook) \
	      [set addressBook::DefaultEntryFields]
	    addressBook::current $newBook
	    menu::buildSome addressBook
	    status::msg "The new address book '$newBook' has been created."
	}
	"Rename Address Book" {
	    set p "Rename which address book?"
	    if {[catch {addressBook::pickBook $p 0} oldName]} {
		status::errorMsg $oldName
	    } 
	    set p "New name for '$oldName':"
	    set newName $oldName
	    while {[lcontains books $newName]} {
		if {[catch {prompt $p $oldName} newName]} {
		    status::errorMsg "Cancelled."
		} elseif {[lcontains books $newName]} {
		    set    question "The address book '$newName' already exists. "
		    append question "Are you sure that you want to "
		    append question "over-write entries with the same names?"
		    set result [askyesno -c $question]
		    if {$result == "yes"} {
			break
		    } elseif {$result == "no"} {
			set newName $oldName
		    } else {
			status::errorMsg "Cancelled."
		    }
		}
	    }
	    set oldBook AddressBook::$oldName
	    set newBook AddressBook::$newName
	    set oldCurrent $current
	    addressBook::current $oldName
	    foreach entryName $entries {
		set entryData [set [set oldBook]($entryName)]
		set [set newBook]($entryName) $entryData
	    }
	    set addressBook::EntryFields($newName) [set addressBook::EntryFields($oldName)]
	    catch {unset $oldBook}
	    if {$oldName != "Default"} {
		unset addressBook::EntryFields($oldName)
	    } else {
		set addressBook::EntryFields(Default) ""
	    }
	    if {$oldCurrent == $oldName} {
		addressBook::current $newName
	    } else {
		addressBook::current $oldCurrent
	    }
	    menu::buildSome addressBook
	    status::msg "The addressBook '$oldName' has been renamed '$newName'."
	}
	"Copy Address Book" {
	    set p "Copy which address book?"
	    if {[catch {addressBook::pickBook $p 0} oldName]} {
		status::errorMsg $oldName
	    } 
	    set p "Copy '$oldName' to:"
	    set newName $oldName
	    while {[lcontains books $newName]} {
		if {[catch {prompt $p $oldName} newName]} {
		    status::errorMsg "Cancelled."
		} elseif {[lcontains books $newName]} {
		    set    question "The address book '$newName' already exists. "
		    append question "Are you sure that you want to "
		    append question "over-write entries with the same names?"
		    set result [askyesno -c $question]
		    if {$result == "yes"} {
			break
		    } elseif {$result == "no"} {
			set newName $oldName
		    } else {
			status::errorMsg "Cancelled."
		    }
		}
	    }
	    set oldBook AddressBook::$oldName
	    set newBook AddressBook::$newName
	    set oldCurrent $current
	    addressBook::current $oldName
	    foreach entryName $entries {
		set entryData [set [set oldBook]($entryName)]
		set [set newBook]($entryName) $entryData
	    }
	    foreach field $fields {
		lunion addressBook::EntryFields($newName) $field
	    }
	    addressBook::current $oldCurrent
	    menu::buildSome addressBook
	    status::msg "The address book '$newName' has been updated."
	}
	"Remove Address Book" {
	    set p   "Remove which address book?"
	    set all "Remove all address books"
	    if {[catch {addressBook::pickBook $p 1 $all 1} bookList]} {
		status::errorMsg $bookList
	    } 
	    foreach item $bookList {
		set    question "Are you sure that you want to permanently "
		append question "remove the address book '$item' and "
		append question "all of its contents?  This cannot be undone."
		if {[askyesno $question] == "yes"} {
		    catch {unset AddressBook::$item}
		    catch {unset addressBook::EntryFields($item)}
		    lappend removed $item
		} 
	    }
	    if {[info exists removed]} {
		if {[lcontains removed $current]} {
		    addressBook::current "Default"
		} 
		menu::buildSome addressBook
		if {[llength $removed] == 1} {
		    status::msg "The address book '$removed' has been removed."
		} else {
		    status::msg "The address books '$removed' have been removed."
		}
	    } 
	}
	default {
	    if {![lcontains books $itemName]} {
		status::errorMsg "'$itemName' is not a recognized address book."
	    } 
	    addressBook::current $itemName
	    prefs::modified addressBook::Current
	    addressBook::postEval
	    status::msg "The current address book is '$itemName'."
	}
    }
}

proc addressBook::addEntry {{entryName ""} args} {
    
    global MailmodeVars
    
    addressBook::current

    if {[catch {prompt "New Entry name:" $entryName} entryName]} {
	status::errorMsg "Cancelled."
    }
    if {[info exists [set book]($entryName)]} {
	set    question "'$entryName' already exists.\r"
	append question "Would you like to edit it?"
	if {[dialog::yesno $question]} {
	    set args [set [set book]($entryName)]
	} else {
	    status::errorMsg "Cancelled."
	}
    }
    set args [join $args]
    catch {addressBook::dialogOrWindow {} $entryName $args} result
    status::msg $result
}

proc addressBook::editEntry {{entryName ""} {addressbook ""}} {
    
    addressBook::current

    if {![string length $entryName] || ![string length $addressbook]} {
	# Called from the menu, offer a list of all possible entries.
	set p "Select an entry to edit:"
	while {![catch {addressBook::pickEntry $p} entryName]} {
	    catch {addressBook::dialogOrWindow {} $entryName} result
	    status::msg $result
	    set p "Select another entry to edit, or cancel:"
	}
    } else {
	# Most likely called from a hyperlink.
	# Make sure that we have the correct address book up front.
	if {![lcontains books $addressbook]} {
	    status::errorMsg "Cancelled -- '$addressbook' is not a recognized address book."
	} 
	set oldCurrent $current
	addressBook::current $addressbook
	if {[catch {set [set book]($entryName)} args]} {
	    status::errorMsg "Could not find any '$entryName' entry."
	} 
	set args [join $args]
	catch {addressBook::dialogOrWindow {} $entryName $args} result
	addressBook::current $oldCurrent
	status::msg $result
    }
}

proc addressBook::dialogOrWindow {{title ""} entryName args} {
    
    addressBook::current
    
    set which [addressBook::findOption editEntriesUsing]
    set args [join $args]
    switch $which {
	"0" {
	    # Use the dialog.
	    if {![catch {addressBook::entryDialog $title $entryName $args} result]} {
		set [set book]($entryName) $result
		prefs::modified [set book]($entryName)
		addressBook::postEval
		return "Changes to '$entryName' have been saved."
	    } else {
		error "Cancelled."
	    }
	}
	"1" {
	    # Use the text window.
	    addressBook::entryWindow $title $entryName $args
	    return "Edit the entry '$entryName' in this window."
	}
    }
}

proc addressBook::entryDialog {{title ""} entryName args} {

    addressBook::current
    
    if {![string length $title]} {set title "Edit the '$entryName' fields:"}
    set title [list $title]
    # '$args' is a list of field-text items.
    if {[llength $args]} {eval array set entryField $args}
    lappend fields "comments"
    # Set dialog size parameters.
    set left    10
    set right  120
    set top     10
    set bottom  30
    set incr    30
    
    set eleft  [expr {$left  + 120}]
    set eright [expr {$right + 270}]
    set height [expr {$incr * [llength $fields] + 110}]
    set width  [expr {$eright + 30}]
    
    set eD "dialog -w $width -h $height -t $title $left $top $width $bottom"
    
    # Add text fields.
    foreach fieldName $fields {
	ensureset entryField($fieldName) ""
	if {$fieldName == "comments"} {continue}
	incr top $incr
	incr bottom $incr
	lappend eD -t $fieldName               $left $top  $right $bottom
	lappend eD -e $entryField($fieldName) $eleft $top $eright $bottom
    }
    # We always put 'comments' last.
    incr top $incr
    incr bottom $incr
    incr bottom $incr
    lappend eD -t {comments}                   $left $top  $right $bottom
    lappend eD -e $entryField(comments)       $eleft $top $eright $bottom
    # Buttons
    incr top     [expr $incr + 40]
    incr bottom  [expr $incr + 10]
    eval lappend eD [dialog::okcancel -380 top]

    # Now present the dialog, return the results if not cancelled.
    set res [eval "$eD"]
    if {[lindex $res end]} {status::errorMsg "Cancelled."}
    # Add an array entry for each dialog field.
    set index 0
    foreach fieldText [lrange $res 0 [expr {[llength $res] - 3}]] {
	set entryField([lindex $fields $index]) $fieldText
	incr index
    }
    # Save all field info, including that which wasn't presented in the
    # dialog.  (We save all information, even for fields that might have
    # been removed after the entry was first created, so that the info
    # will still be there if the field is put back into play.)
    foreach fieldName [array names entryField] {
	if {![string length entryField($fieldName)]} {continue}
	lappend result [list $fieldName $entryField($fieldName)]
    }
    return $result
}

proc addressBook::entryWindow {{title ""} entryName args} {

    addressBook::current

    if {![string length $title]} {set title "Edit Address Book Entry"} 
    lappend fields "comments"
    set    intro "\rEdit the '$entryName' address book entry in this window.\r\r"
    append intro "When you are finished, click here: <<Save This Entry>>\r"
    append intro "to save the entry in the '$current' address book --\r"
    append intro "the text fields will appear in a dialog for confirmation.\r\r"
    append intro "Important:  Do NOT edit the blue \"Field:\" text !!!\r\r"
    append intro "\t______________________________________________________\r\r"
    set entryText [addressBook::createLabel $entryName $fields "-1"]
    new -n $title -text ${intro}${entryText} -m Text
    # Now add some color, and hyperize.
    win::searchAndHyperise "^Edit the '$entryName'"              {} 1 1 +10 -1
    win::searchAndHyperise "^\[a-zA-Z0-9\]\[a-zA-Z0-9 -\]+:   " {} 1 1
    win::searchAndHyperise "\"Field:\""                         {} 1 1
    win::searchAndHyperise "^Entry: +(\[^\r\n\]+)" \
      "addressBook::saveEntryWindow \"$entryName\" \"$current\"" 1 5 +7
    win::searchAndHyperise "<<Save This Entry>>" \
      "addressBook::saveEntryWindow \"$entryName\" \"$current\"" 1 5
    refresh
    # Try to position the cursor at the start of the first field text.
    set pos0 [minPos]
    foreach field $fields {
	set pat "^${field}: *"
	if {![catch {search -s -f 1 -r 1 $pat $pos0} match]} {
	    set pos0 [lindex $match 1]
	    break
	}
    }
    goto $pos0
    setWinInfo -w [win::Current] dirty 0
}

proc addressBook::saveEntryWindow {entryName addressbook} {

    global MailmodeVars
    
    addressBook::current
    set oldCurrent $current
    addressBook::current $addressbook
    
    set windowText [getText [minPos] [maxPos]]
    regsub {^.*________(\r|\n)} $windowText {} entryInfo
    set entryInfo [string trim $entryInfo]
    set args [list ]
    set lastField ""
    foreach field [concat $fields "comments" ""] {
	set pat "${lastField}:(.*)${field}:"
	set lastFieldText ""
	if {[regexp $pat $entryInfo allofit lastFieldText]} {
	    set lastFieldText [string trim $lastFieldText]
	    if {$lastField == "Email" || $lastField == "Home Page"} {
	        set lastFieldText [string trimleft  $lastFieldText "<"]
		set lastFieldText [string trimright $lastFieldText ">"]
	    } 
	    lappend args $lastField [string trim $lastFieldText]
	}
	regsub ".*${field}:" $entryInfo "${field}:" entryInfo
	set lastField $field
    }
    # Now make sure that we get the last one.
    regsub ".*${field}:" $entryInfo "" entryInfo
    lappend args $lastField [string trim $entryInfo]
    # Now confirm the changes.
    set title "Please confirm the '$entryName' changes."
    if {![catch {addressBook::entryDialog $title $entryName $args} result]} {
	set [set book]($entryName) $result
	prefs::modified [set book]($entryName)
	setWinInfo -w [win::Current] dirty 0
	killWindow
	addressBook::current $oldCurrent
	if {$MailmodeVars(autoUpdateMailElectrics)} {
	    addressBook::updateMailElectrics $entryName 1
	} 
	status::msg "Changes to '$entryName' have been saved."
    } else {
	addressBook::current $oldCurrent
	status::errorMsg "No changes have been saved."
    }
}
proc addressBook::searchFor {pattern {searchField ""}} {
    
    addressBook::current

    lappend fields "comments"
    if {$searchField == "all fields"} {set searchField ""}
    if {[string length $searchField]} {
        set allFields [list $searchField]
    } else {
	set allFields [list "Entry Name"]
    }
    # Set up the searching arrays.  We have to compensate here (and below)
    # for the possibility that the user has added/deleted entry fields that
    # might still exist (or not exist at all) in specific entries.
    foreach entryName $entries {
	catch {unset entryFields}
	set "Entry NameFields($entryName)" $entryName
	array set entryFields [join [set [set book]($entryName)]]
	foreach field [array names entryFields] {
	    ensureset entryFields($field) ""
	    if {![string length $searchField]} {
		# We're not looking for a specific field.
		set ${field}Fields($entryName) $entryFields($field)
		lunion allFields $field
	    } elseif {$field == $searchField} {
		set ${field}Fields($entryName) $entryFields($field)
	    }
	}
    }
    # Now we search in all of the fields for the pattern.
    if {[string length $searchField] && ![info exists ${searchField}Fields]} {
	status::errorMsg "No entries contain the field '$searchField'."
    } 
    foreach entryName $entries {
	foreach field $allFields {
	    if {![info exists ${field}Fields($entryName)]} {
	        continue
	    } elseif {[regexp -nocase $pattern [set ${field}Fields($entryName)]]} {
	        lappend entryList $entryName
	    } 
	}
    }
    # Find anything?
    if {![info exists entryList]} {
	status::errorMsg "Couldn't find '$pattern' in address book fields."
    }
    set entryList [lsort -ignore [lunique $entryList]]
    status::msg "matches found: [llength $entryList] (pattern: '$pattern')"
    # Which fields should be used?
    set resultStyle [addressBook::findOption "displaySearchResults"]
    if {$resultStyle == 0 || $resultStyle == 1} {
        set includeEntryName 1
    } else {
        set includeEntryName 0
    }
    set p   "Display which fields?"
    set all "Display All Fields"
    if {[catch {addressBook::pickField $p 1 $all $includeEntryName} fieldList]} {
	status::errorMsg $fieldList
    }
    # Create the entries text.
    set entryText ""
    foreach entryName [lsort -ignore $entryList] {
	if {![catch {addressBook::createLabel $entryName $fieldList $resultStyle} entryInfo]} {
	    append entryText "${entryInfo}\r"
	}
    }
    # Creat the title and intro.
    set    title "Address Book Search Results"
    set    intro "Address Book Search Results --\r\r"
    append intro "[format {%-20s} {Search Term:}]${pattern}\r"
    if {[string length $searchField]} {
        append intro "[format {%-20s} {Search Field:}]$searchField\r"
    } 
    append intro "[format {%-20s} {Entries Found:}][llength $entryList]"
    # Create a new window with the search results.
    addressBook::newWindow $entryText $title $intro
}

proc addressBook::createLabel {entryName {fieldList ""} {resultStyle 1}} {
 
    addressBook::current

    if {$resultStyle != "-1" && ![info exists [set book]($entryName)]} {
	error "No entry exists for '$entryName'."
    }
    set textHeaders 0
    switch -- $resultStyle {
	"-2" {set delimiter "\r" ; set includeEmpty 0}
	"-1" {set textHeaders 1 ; set delimiter "\r" ; set includeEmpty 1}
	"0"  {set textHeaders 1 ; set delimiter "\r"}
	"1"  {set delimiter "\r"}
	"2"  {set delimiter "\t"}
	"3"  {set delimiter " "}
    }
    if {![llength $fieldList]} {
	set fieldList $fields
	if {$delimiter == "\r"} {
	    set fieldList [concat [list "Entry Name"] $fieldList]
	} 
	lappend fieldList "comments"
    }
    if {[catch {join [set [set book]($entryName)]} args]} {
	set args [list ]
    }
    array set entryFields $args
    set entryText ""
    # Do we add the entry name?
    if {[lcontains fieldList "Entry Name"]} {
	if {$delimiter == "\r"} {
	    set entryText "Entry: ${entryName}\r\r"
	} else {
	    set entryText ${entryName}${delimiter}
	}
	set fieldList [lremove $fieldList "Entry Name"]
    }
    # Should we include empty fields?
    if {![info exists includeEmpty]} {
        if {[lcontains fieldList "Entry Name"] || $delimiter != "\r"} {
            set includeEmpty 1
        } else {
	    set includeEmpty 0
        }
    } 
    # Add the fields.
    foreach field $fieldList {
	ensureset entryFields($field) ""
	set fieldText [string trim $entryFields($field)]
	if {[string length $fieldText]} {
	    if {$field == "Email" || $field == "Home Page"} {
		set fieldText [string trimright $fieldText <]
		set fieldText [string trimleft  $fieldText >]
		set fieldText "<${fieldText}>"
	    } elseif {$field == "comments"} {
		set fieldText "\r\r${fieldText}"
	    }
	} elseif {!$includeEmpty} {
	    continue
	}
	if {$textHeaders} {
	    append entryText [format {%-20s} "${field}:"]
	} 
	append entryText ${fieldText}\r
    } 
    return $entryText
}

proc addressBook::newWindow {entryText {title ""} {intro ""}} {
 
    addressBook::current

    if {![string length $title]} {set title "'$current' Address Book Entries"} 
    if {![string length $intro]} {set intro "\"$current\" Address Book Entries"} 
    set    intro "\r${intro}\r\r"
    append intro "\t______________________________________________________\r\r"
    new -n $title -text ${intro}${entryText} -m Text
    # Now add some color, and hyperize.
    win::searchAndHyperise \
      "<(\[-_a-zA-Z0-9.\]+@\[-_a-zA-Z0-9.\]+)>" \
      {composeEmail "mailto:\1"} 1
    win::searchAndHyperise "<(\[^\r\n:\]+:\[^ >\]*)>" \
      {urlView "\1"} 1 
    win::searchAndHyperise "^\[a-zA-Z0-9\]\[a-zA-Z0-9 -\]+:   " {} 1 1
    win::searchAndHyperise "^Entry: +(\[^\r\n\]+)" \
      "addressBook::editEntry \"\\1\" \"$current\"" 1 5 +7
    refresh
    winReadOnly
}

proc addressBook::updateMailElectrics {{entryList ""} {quietly 0}} {
    
    global addressEntries Mailelectrics completions

    addressBook::current
    
    # Make sure that the electrics will actually work !!
    # (There should really be a "MailCompletions.tcl" file with this.)
    ensureset completions(Mail) [list completion::electric completion::word]
    if {![llength $entryList]} {set entryList $entries}
    foreach entryName $entryList {
	if {[regexp " |\t" $entryName]} {continue}
	if {[catch {addressBook::getEntryField $entryName "Email"} email]} {
	    set email ""
	}
	if {[string length $email]} {
	    set Mailelectrics($entryName) "kill0${email}"
	} 
	set added 1
    }
    if {[info exists added] && !$quietly} {
	status::msg "The Mail Menu electrics have been updated."
    } 
}

proc addressBook::pickBook {{dialogText ""} {listOkay 0} {selectAllText ""} {defaultOkay 1}} {
    
    global addressBook::LastBook
  
    addressBook::current
    
    set books [lsort -ignore $books]
    ensureset addressBook::LastBook [lindex $books 0]
    if {!$defaultOkay} {
	set books [lremove $books "Default"]
    } 
    if {![llength $books]} {
	error "There are no address books to list."
    }
    if {[string length $selectAllText]} {
	set pickList [concat [list $selectAllText] $books]
	set L $selectAllText
    } else {
	set pickList $books
	set L [set addressBook::LastBook]
    }
    if {![string length $dialogText]} {
	if {$listOkay} {
	    set dialogText "Select address books:"
	} else {
	    set dialogText "Select an address book:"
	}
    } 
    if {$listOkay} {
	if {[catch {listpick -p $dialogText -L $L -l $pickList} result]} {
	    error "Cancelled."
	} 
    } else {
	if {[catch {listpick -p $dialogText -L $L $pickList} result]} {
	    error "Cancelled."
	} 
    }
    # Return the results.  Return all entries if user selected the option.
    if {$listOkay} {
	if {[string length $selectAllText] && [lcontains result $selectAllText]} {
	    set addressBook::LastBook ""
	    return $books
	} else {
	    set addressBook::LastBook [lindex $result end]
	    return $result
	} 
    } else {
	if {[string length $selectAllText] && $result == $selectAllText} {
	    # We return an empty list to indicate that all address books
	    # were chosen.  (Calling code is expecting a single item ...)
	    set addressBook::LastBook ""
	    return ""
	} else {
	    set addressBook::LastBook $result
	    return $result
	} 
    }
}

proc addressBook::pickEntry {{dialogText ""} {listOkay 0} {selectAllText ""}} {

    global addressBook::LastEntry
    
    addressBook::current
  
    set entries [lsort -ignore $entries]
    ensureset addressBook::LastEntry [lindex $entries 0]
    if {![llength $entries]} {
	error "There are no address book entries to list."
    } elseif {[string length $selectAllText]} {
        set pickList [concat [list $selectAllText] $entries]
	set L $selectAllText
    } else {
        set pickList $entries
	set L [set addressBook::LastEntry]
    }
    if {![string length $dialogText]} {
	if {$listOkay} {
	    set dialogText "Select address book entries:"
	} else {
	    set dialogText "Select an address book entry:"
	}
    } 
    if {$listOkay} {
	if {[catch {listpick -p $dialogText -L $L -l $pickList} result]} {
	    error "Cancelled."
	} 
    } else {
	if {[catch {listpick -p $dialogText -L $L $pickList} result]} {
	    error "Cancelled."
	} 
    }
    # Return the results.  Return all entries if user selected the option.
    if {$listOkay} {
	if {[string length $selectAllText] && [lcontains result $selectAllText]} {
	    set addressBook::LastEntry ""
	    return $entries
	} else {
	    set addressBook::LastEntry [lindex $result end]
	    return $result
	} 
    } else {
	if {[string length $selectAllText] && $result == $selectAllText} {
	    # We return an empty list to indicate that all entries were
	    # chosen.  (Calling code is expecting a single item ...)
	    set addressBook::LastEntry ""
	    return ""
	} else {
	    set addressBook::LastEntry $result
	    return $result
	} 
    }
}

proc addressBook::pickField {{dialogText ""} {listOkay 0} {selectAllText ""} {includeEntryName 1} {removeList ""}} {

    global addressBook::LastField
    
    addressBook::current

    if {![string length $dialogText]} {
	if {$listOkay} {
	    set dialogText "Select some fields:"
	} else {
	    set dialogText "Select a field:"
	}
    } 
    # Add the 'comments' field at the end.
    lappend fields "comments"
    ensureset addressBook::LastField [lindex $fields 0]
    # Add "Entry Name" if desired.
    if {$includeEntryName} {
	set fields [concat [list "Entry Name"] $fields]
    } 
    # Remove fields if desired.
    if {[llength $removeList]} {
        set fields [lremove -l $fields $removeList]
    } 
    if {![llength $fields]} {
	error "There are no address book fields to list."
    }
    # Add an option to select all fields if desired.
    if {[string length $selectAllText]} {
	set pickList [concat [list $selectAllText] $fields]
	set L $selectAllText
    } else {
        set pickList $fields
	set L [set addressBook::LastField]
    }
    # Offer the list.
    if {$listOkay} {
	if {[catch {listpick -p $dialogText -L $L -l $pickList} result]} {
	    error "Cancelled."
	} 
    } else {
	if {[catch {listpick -p $dialogText -L $L $pickList} result]} {
	    error "Cancelled."
	} 
    }
    # Return the results.  Return all fields if user selected the option.
    if {$listOkay} {
	if {[string length $selectAllText] && [lcontains result $selectAllText]} {
	    set addressBook::LastField ""
	    return $fields
	} else {
	    set addressBook::LastField [lindex $result end]
	    return $result
	} 
    } else {
	if {[string length $selectAllText] && $result == $selectAllText} {
	    # We return an empty list to indicate that all fields were
	    # chosen.  (Calling code is expecting a single item ...)
	    set addressBook::LastField ""
	    return ""
	} else {
	    set addressBook::LastField $result
	    return $result
	} 
    }
}

proc addressBook::getEntryField {entryName field} {
    
    addressBook::current

    if {![info exists [set book]($entryName)]} {
	error "There is no entry named '$entryName'."
    } 
    array set entryFields [join [set [set book]($entryName)]]
    if {![info exists entryFields($field)]} {
        error "There is no '$field' field for the entry '$entryName'."
    }
    set fieldText $entryFields($field)
    if {$field == "Email" || $field == "Home Page"} {
	set fieldText [string trim      $fieldText]
	set fieldText [string trimright $fieldText >]
	set fieldText [string trimleft  $fieldText <]
    } 
    return $fieldText
}

# ===========================================================================
# 
#  --------  #
# 
#  version history  #
# 
# To Do:
# 
# -- Better integrate Address Book with Mail menu, esp completions.
#    Probably involves a bit of updating to mailMenu.tcl, unfortunately.
# -- Import, export functions for popular mailer address books.
# 
#  modified by  vers#  reason
#  -------- --- ------ -----------
#  10/21/01 cbu 0.1    Created package.
#  10/23/01 cbu 0.2    Better integration with Mail Menu electrics.
#                      Better mailing list creation procs.
#  10/26/01 cbu 0.3    Multiple address books now available.
# 

# ===========================================================================
# 
# .
