## -*-Tcl-*- (install)
 # ###################################################################
 #  alphatest - Tcl test suite for Alpha 7.x
 # 
 #  FILE: "alphaTest.tcl"
 #                                    created: 8/30/2000 {7:44:57 AM} 
 #                                last update: 04/20/2001 {11:49:06 AM} 
 #  Author: Jonathan Guyer
 #  E-mail: jguyer@his.com
 #    mail: POMODORO no seisan
 #     www: http://www.his.com/jguyer/
 #     
 # ========================================================================
 # This file contains support code for the Tcl test suite.  It is
 # normally sourced by the individual files in the test suite before
 # they run their tests.  This improved approach to testing was designed
 # and initially implemented by Mary Ann May-Pumphrey of Sun Microsystems.
 #
 # Copyright (c) 1990-1994 The Regents of the University of California.
 # Copyright (c) 1994-1996 Sun Microsystems, Inc.
 #
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 #
 # SCCS: @(#) defs 1.37 96/04/12 13:45:04
 # ========================================================================
 #  Description: 
 # 
 #   This Tcl test suite only makes sense for use in Alpha 7.x. 
 #   It was coopted from the old Tcl 7.5 "defs" code.
 #   In Alpha 8, use the standard [package require tcltest]
 #   
 #  History
 # 
 #  modified   by  rev reason
 #  ---------- --- --- -----------
 #  2000-09-09 JEG 1.0 original
 # ###################################################################
 ##


alpha::extension alphatest 1.0.1 {
    alpha::package require AlphaTcl 7
} maintainer {
    "Jon Guyer" <jguyer@his.com> <http://www.his.com/jguyer/>
} help {
    This Tcl test suite only makes sense for use in Alpha 7.x. It
    was coopted from the old Tcl 7.5 "defs" code.  In Alpha 8, use
    the standard [package require tcltest]
} requirements {
    if {[info tclversion] >= 8.0} {
	error "Tcl 8 or newer comes with a standard 'tcltest' package"
    }
}

    

if ![info exists VERBOSE] {
    set VERBOSE 0
}
if ![info exists TESTS] {
    set TESTS {}
}

if ![info exists alphatest::passed] {
    set alphatest::passed 0
}

if ![info exists alphatest::failed] {
    set alphatest::failed 0
}

if [info exists alphatest::skipped] {
    unset alphatest::skipped
}


# Some of the tests don't work on some system configurations due to
# differences in word length, file system configuration, etc.  In order
# to prevent false alarms, these tests are generally only run in the
# master development directory for Tcl.  The presence of a file
# "doAllTests" in this directory is used to indicate that the non-portable
# tests should be run.

set doNonPortableTests [file exists doAllTests]

# If there is no "memory" command (because memory debugging isn't
# enabled), generate a dummy command that does nothing.

if {[info commands memory] == ""} {
    proc memory args {}
}

# Check configuration information that will determine which tests
# to run.  To do this, create an array testConfig.  Each element
# has a 0 or 1 value, and the following elements are defined:
#	unixOnly -	1 means this is a UNIX platform, so it's OK
#			to run tests that only work under UNIX.
#	macOnly -	1 means this is a Mac platform, so it's OK
#			to run tests that only work on Macs.
#	pcOnly -	1 means this is a PC platform, so it's OK to
#			run tests that only work on PCs.
#	unixOrPc -	1 means this is a UNIX or PC platform.
#	macOrPc -	1 means this is a Mac or PC platform.
#	nonPortable -	1 means this the tests are being running in
#			the master Tcl/Tk development environment;
#			Some tests are inherently non-portable because
#			they depend on things like word length, file system
#			configuration, window manager, etc.  These tests
#			are only run in the main Tcl development directory
#			where the configuration is well known.  The presence
#			of the file "doAllTests" in this directory indicates
#			that it is safe to run non-portable tests.
#	tempNotPc -	The inverse of pcOnly.  This flag is used to
#			temporarily disable a test.
#	nonBlockFiles - 1 means this platform supports setting files into
#			nonblocking mode.
#	asyncPipeClose- 1 means this platform supports async flush and
#			async close on a pipe.
#	unixExecs     - 1 means this machine has commands such as 'cat',
#			'echo' etc available.

catch {unset alphatest::testConfig}
set alphatest::testConfig(unixOnly) 0
set alphatest::testConfig(tempNotPc) 1
set alphatest::testConfig(macOnly) 1
set alphatest::testConfig(pcOnly) 0
set alphatest::testConfig(unixOrPc) 0
set alphatest::testConfig(macOrPc) 1
set alphatest::testConfig(macOrUnix) 1
set alphatest::testConfig(nonPortable) [file exists doAllTests]

set alphatest::testConfig(nonBlockFiles) 1

set alphatest::testConfig(asyncPipeClose) 1

# Test to see if execed commands such as cat, echo, rm and so forth are
# present on this machine.

set alphatest::testConfig(unixExecs) 0

proc alphatest::print_verbose {name description script code answer} {
    tclLog "\n\n"
    tclLog "==== $name $description\n"
    tclLog "==== Contents of test case:\n"
    tclLog "$script\n"
    if {$code != 0} {
	if {$code == 1} {
	    tclLog "==== Test generated error:\n"
	    tclLog "$answer\n"
	} elseif {$code == 2} {
	    tclLog "==== Test generated return exception;  result was:\n"
	    tclLog "$answer\n"
	} elseif {$code == 3} {
	    tclLog "==== Test generated break exception\n"
	} elseif {$code == 4} {
	    tclLog "==== Test generated continue exception\n"
	} else {
	    tclLog "==== Test generated exception $code;  message was:\n"
	    tclLog "$answer\n"
	}
    } else {
	tclLog "==== Result was:\n"
	tclLog "$answer\n"
    }
}

# test --
# This procedure runs a test and prints an error message if the
# test fails.  If VERBOSE has been set, it also prints a message
# even if the test succeeds.  The test will be skipped if it
# doesn't match the TESTS variable, or if one of the elements
# of "constraints" turns out not to be true.
#
# Arguments:
# name -		Name of test, in the form foo-1.2.
# description -		Short textual description of the test, to
#			help humans understand what it does.
# constraints -		A list of one or more keywords, each of
#			which must be the name of an element in
#			the array "alphatest::testConfig".  If any of these
#			elements is zero, the test is skipped.
#			This argument may be omitted.
# script -		Script to run to carry out the test.  It must
#			return a result that can be checked for
#			correctness.
# answer -		Expected result from script.

proc test {name description script answer args} {
    global VERBOSE TESTS alphatest::testConfig
    if {[string compare $TESTS ""] != 0} then {
	set ok 0
	foreach test $TESTS {
	    if [string match $test $name] then {
		set ok 1
		break
	    }
        }
	if !$ok then return
    }
    set i [llength $args]
    if {$i == 0} {
	# Empty body
    } elseif {$i == 1} {
	# "constraints" argument exists;  shuffle arguments down, then
	# make sure that the constraints are satisfied.

	set constraints $script
	set script $answer
	set answer [lindex $args 0]
	foreach constraint $constraints {
	    if {![info exists alphatest::testConfig($constraint)]
		    || ![set alphatest::testConfig($constraint)]} {
			
		global alphatest::skipped
		if ![info exists alphatest::skipped($constraint)] {
		    set alphatest::skipped($constraint) 0
		}
		incr alphatest::skipped($constraint)
		
		return
	    }
	}
    } else {
	error "wrong # args: must be \"test name description ?constraints? script answer\""
    }
    memory tag $name
    set code [catch {uplevel $script} result]
    if {$code != 0} {
	global alphatest::failed
	incr alphatest::failed
	
	alphatest::print_verbose $name $description $script \
		$code $result
    } elseif {[string compare $result $answer] == 0} then { 
	global alphatest::passed
	incr alphatest::passed
	
	if $VERBOSE then {
	    if {$VERBOSE > 0} {
		alphatest::print_verbose $name $description $script \
		    $code $result
	    }
	    tclLog "++++ $name PASSED\n"
	}
    } else { 
	global alphatest::failed
	incr alphatest::failed
	
	alphatest::print_verbose $name $description $script \
		$code $result
	tclLog "---- Result should have been:\n"
	tclLog "$answer\n"
	tclLog "---- $name FAILED\n" 
    }
}

proc ::tcltest::cleanupTests {} {
    global alphatest::passed alphatest::failed alphatest::skipped
    
    set skipped 0
    foreach skip [array names alphatest::skipped] {
	incr skipped [set alphatest::skipped($skip)]
    }
    
    set totalTests [expr {${alphatest::passed} + ${alphatest::failed} + ${skipped}}]
    tclLog "\nTotal\t${totalTests}\tPassed\t${alphatest::passed}\tSkipped\t${skipped}\tFailed\t${alphatest::failed}\n"
    
    if {$skipped > 0} {
	tclLog "Number of tests skipped for each constraint:\n"
	foreach skip [array names alphatest::skipped] {
	    if {[set alphatest::skipped($skip)] > 0} {
		tclLog "\t${skip}\t[set alphatest::skipped($skip)]\n"
	    }
	}
    }
    
    unset alphatest::skipped
    set alphatest::failed 0
    set alphatest::passed 0    
}

proc alphatest::dotests {file args} {
    global TESTS
    set savedTests $TESTS
    set TESTS $args
    source $file
    set TESTS $savedTests
}

proc alphatest::normalizeMsg {msg} {
    regsub "\n$" [string tolower $msg] "" msg
    regsub -all "\n\n" $msg "\n" msg
    regsub -all "\n\}" $msg "\}" msg
    return $msg
}

proc alphatest::makeFile {contents name} {
    set fd [open $name w]
    fconfigure $fd -translation lf
    if {[string index $contents [expr [string length $contents] - 1]] == "\n"} {
	puts -nonewline $fd $contents
    } else {
	puts $fd $contents
    }
    close $fd
}

proc alphatest::removeFile {name} {
    global tcl_platform alphatest::testConfig
    if {$tcl_platform(platform) == "macintosh"} {
	catch {rm -f $name}
    } else {
	catch {exec rm -f $name}
    }
}

proc alphatest::makeDirectory {name} {
    global tcl_platform alphatest::testConfig
    if {$tcl_platform(platform) == "macintosh"} {
	catch {mkdir $name}
    } else {
	catch {exec mkdir $name}
    }
}

proc alphatest::removeDirectory {name} {
    global tcl_platform alphatest::testConfig
    if {$tcl_platform(platform) == "macintosh"} {
	catch {rmdir $name}
    } else {
	catch {exec rm -rf $name}
    }
}

proc alphatest::viewFile {name} {
    global tcl_platform alphatest::testConfig
    if {($tcl_platform(platform) == "macintosh") || \
		([set alphatest::testConfig(unixExecs)] == 0)} {
	set f [open $name]
	set data [read -nonewline $f]
	close $f
	return $data
    } else {
	exec cat $name
    }
}



