## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl
 # 
 #  FILE: "backup.tcl"
 #                                    created: 00-10-17 12.50.44
 #                                last update: 07/18/2001 {11:45:23 AM} 
 #  
 #  Author: Mostly Vince and Johan
 #  
 #  All backup related material is now in this one file, so if
 #  you want to modify things (to provide for saving of multiple
 #  backups, for instance), there's no need to search through
 #  all of AlphaTcl.
 #  
 #  Note: the 'activateBackup' procedure assumes the existence of
 #  win::Modified($name) is equivalent to checking whether the 
 #  window 'name' has been saved to disk.  I believe this is true,
 #  but I'm not 100% certain.
 # 
 # Distributed under a Tcl style license.  
 # 
 # ###################################################################
 ##

alpha::declare flag backup 0.1.4 global {
    # To make a backup every time a file is saved, click this box||To save disk
    # space and not make backup copies when saving files, click this box. \
    newPref flag backup 0
  
    # Folder in which to store backups.  A null value tells Alpha to use
    # the file's current directory.
    newPref var backupFolder ""
    # Extension to add to files when backing them up
    newPref var backupExtension "~"
    # If the previous backup is more recent than this number of hours,
    # then don't make a new backup.  If greater than zero, this means the 
    # backups Alpha has tend to be significantly different to the current file.
    newPref var backupAgeRequirementInHours 0.0
    lunion flagPrefs(Backups) backup
    lunion varPrefs(Backups) backupFolder backupExtension\
      backupAgeRequirementInHours
    set backup 0
} {
    set backup 1
    menu::replaceWith File "revert" items "<E<SrevertToBackup" "<Srevert"
    hook::register saveHook backupOnSave
    hook::register requireOpenWindowsHook [list File revertToBackup] 1
    hook::register activateHook activateBackup
} {
    set backup 0
    menu::removeFrom File "revert" items "<E<SrevertToBackup" "<Srevert"
    hook::deregister saveHook backupOnSave
    hook::deregister requireOpenWindowsHook [list File revertToBackup] 1
    hook::deregister activateHook activateBackup
} help {
    When savings files, occasionally make backup copies.  You can 
    fine-tune the frequency of backups, and their location via 
    the 'Preferences->Backups' dialog
}

proc activateBackup {name} {
    global win::Modified
    enableMenuItem File revertToBackup [expr {[backupFileExists \
      [win::StripCount $name]] && [info exists win::Modified($name)]}]
}

proc backupOnSave {name} {
    global backup backupExtension backupFolder \
      backupAgeRequirementInHours

    set realname $name
    if {![file exists $name]} {
	regsub { <[0-9]+>$} $name {} name
    }
    if {$backup} {
	if {$backupFolder != "" && ![file exists $backupFolder]} {
	    if {![dialog::yesno "Create backup folder '$backupFolder'?"]} {
		alertnote "Backup saved in document's folder."
		set backupFolder ""
		prefs::modified backupFolder
	    } elseif {[catch {file::ensureDirExists $backupFolder}]} {
		alertnote "Couldn't create backup folder.\
		  Backup saved in document's folder."
		set backupFolder ""
		prefs::modified backupFolder
	    }
	}
	set dir $backupFolder

	if {![string length $dir]} {
	    set dir [file dirname $name]
	}
	if {$backupExtension == "" && $backupFolder == ""} {
	    set backupExtension ~
	    prefs::modified backupExtension
	}
	set backfile [file join $dir [file tail $name]$backupExtension]
	if {$backupExtension == "" && [file dirname $name] == $backupFolder} {
	    append backfile ~
	}
	if {[file exists $backfile]} {
	    getFileInfo $name a
	    if {[expr {([now] - $a(modified) + 0.0)/3600}] \
	      < $backupAgeRequirementInHours} {
		return
	    }
	    catch {file delete $backfile}
	}
	if {[catch {file copy $name $backfile}]} {
	    message "Failed to backup $name to $backfile"
	} else {
	    message "Backed up $backfile"
	}
    }
    activateBackup $realname
}

namespace eval file {}

proc file::revertToBackup {} {
    global backupExtension backupFolder

    set fname [win::StripCount [win::Current]]
    set dir $backupFolder
    if {$dir == ""} {
        set dir [file dirname $fname]
    }
    set bname [file join $dir "[file tail $fname]$backupExtension"]
    if {![file exists $bname]} {
        beep
        message "Backup file '$bname' does not exist"
        return
    }

    if {[dialog::yesno -y "Revert" -n "Preview backup" -c \
      "Revert to backup dated '[join [mtime [file mtime $bname]]]'?"]} {
        killWindow
        edit $bname
        saveAs -f $fname
    } else {
	file::openAsTemplate $bname
    }
}

proc backupFileExists {name} {
    global backupExtension backupFolder

    set dir $backupFolder
    if {$dir == ""} {
        set dir [file dirname $name]
    }
    set bname [file join $dir "[file tail $name]$backupExtension"]
    return [file exists $bname]
}
