## -*-Tcl-*- (install)
 # ###################################################################
 #  binary - AlphaTcl emulator for Tcl 8's [binary]
 # 
 #  FILE: "binary.tcl"
 #                                    created: 9/26/2000 {7:36:14 PM} 
 #                                last update: 2/7/2001 {10:18:38 AM} 
 #  Author: Jonathan Guyer
 #  E-mail: jguyer@his.com
 #    mail: POMODORO no seisan
 #     www: http://www.his.com/jguyer/
 #  
 # ========================================================================
 #               Copyright  2000 Jonathan Guyer
 #                      All rights reserved
 # ========================================================================
 # Permission to use, copy, modify, and distribute this software and its
 # documentation for any purpose and without fee is hereby granted,
 # provided that the above copyright notice appear in all copies and that
 # both that the copyright notice and warranty disclaimer appear in
 # supporting documentation.
 # 
 # Jonathan Guyer disclaims all warranties with regard to this software,
 # including all implied warranties of merchantability and fitness.  In
 # no event shall Jonathan Guyer be liable for any special, indirect or
 # consequential damages or any damages whatsoever resulting from loss of
 # use, data or profits, whether in an action of contract, negligence or
 # other tortuous action, arising out of or in connection with the use or
 # performance of this software.
 # ========================================================================
 #  Description: 
 #  
 #   This package provides an emulator for the Tcl 8 [binary] command.
 #   Rather than forming a binary stream, this version formats and scans 
 #   hexadecimal character pairs.
 # 
 #  History
 # 
 #  modified   by  rev reason
 #  ---------- --- --- -----------
 #  2000-09-26 JEG 1.0 original
 # ###################################################################
 ##

alpha::extension binary 1.0b3 {
    if {[info tclversion] < 8.0} {
	;proc binary {cmd args} {
	    set options [list format scan]
	    if {[string length $cmd] == 0
	    ||	[set ix [lsearch -glob $options "${cmd}*"]] < 0} {
		error "bad option \"${cmd}\": must be format or scan"
	    } else {
		set cmd [lindex $options $ix]
	    }
	    switch $cmd {
		"format" {
		    if {[llength $args] < 1} {
			error "wrong # args: should be \"binary format formatString ?arg arg ...?\""
		    } 
		    return [eval Binary::format::doit $args]
		}
		"scan" {
		    if {[llength $args] < 2} {
			error "wrong # args: should be \"binary scan value formatString ?varName varName ...?\""
		    } 
		    return [eval Binary::scan::doit $args]
		}
	    }
	}
    }
} maintainer {
    "Jon Guyer" <jguyer@his.com> <http://www.his.com/jguyer/>
} help {
    This package provides an emulator for the Tcl 8 [binary]
    command.  Rather than forming a binary stream, this version
    formats and scans hexadecimal character pairs.
}
    

#  format  #

namespace eval Binary::format {}

proc Binary::format::doit {formatString args} {
    set output {}
    set cursor 0
    
    while {$formatString != ""} {
	set formatString [string trimleft $formatString]
	if {![regexp {^(.)([0-9]+|\*)?(.*)$} \
	  $formatString blah type count formatString]} {
	    break
	} 
	
	switch $type {
	    "a" - "A" - "b" - "B" - "h" - "H" {
		Binary::format::countChar
	    }
	    "c" - "s" - "S" - "i" - "I" {
		Binary::format::countNum "integer"
	    }
	    "f" - "d" {
		Binary::format::countNum "floating-point number"
	    }
	    "x" {
		Binary::format::countx
	    }
	    "X" {
		Binary::format::countX
		continue
	    }
	    "@" {
		Binary::format::count@
		continue
	    }
	    default {
		error "bad field specifier \"$type\""
	    }
	}
	
	set formatted {}
	set value 0
	
	switch $type {
	    "a" {
		set ascii [coerce TEXT [string range $arg 0 [expr {$count - 1}]] -x TEXT]
		eval lappend formatted [split $ascii ""]
		for {set i $len} {$i < $count} {incr i} {
		    lappend formatted 0 0
		}
	    }
	    "A" {
		set ascii [coerce TEXT [string range $arg 0 [expr {$count - 1}]] -x TEXT]
		eval lappend formatted [split $ascii ""]
		for {set i $len} {$i < $count} {incr i} {
		    lappend formatted 2 0
		}
	    }
	    "h" - "H" {
		for {set i 0} {$i < $count} {incr i 2} {
		    Binary::format::$type $i $arg [expr {$len > $count ? $count : $len}]
		}
	    }
	    default {
		for {set i 0} {$i < $count} {incr i} {
		    Binary::format::$type $i $arg $len
		}
	    }
	}
	
	# clean up any remainder
	switch $type {
	    "b" {
		if {[expr {$i % 8}] != 0} {
		    eval lappend formatted [split [format "%02X" $value] ""]
		}				
	    }
	    "B" {
		if {[expr {$i % 8}] != 0} {
		    set value [expr {$value << (8 - ($i % 8))}]
		    eval lappend formatted [split [format "%02X" $value] ""]
		}	
	    }
	}
	
	set formatLen [llength $formatted]
	
	if {$formatLen > 0} {
	    for {set i [llength $output]} {$i <= $cursor} {incr i 2} {
		lappend output 0 0
	    }
	    
	    set last [expr {$cursor + $formatLen - 1}]
	    set output [eval lreplace [list $output] $cursor $last $formatted]
	    incr cursor $formatLen
	} 
    }
    
    return [join $output ""]
}

#    formatters  #

# unused
proc Binary::format::a {index in length} {
    upvar formatted out
    
    if {$index < $length} {
	lappend out [split [coerce TEXT [string index $in $index] -x TEXT] ""]
    } else {
	lappend out 0 0
    }	
}

#unused
proc Binary::format::A {index in length} {
    upvar formatted out
    
    if {$index < $length} {
	lappend out [split [coerce TEXT [string index $in $index] -x TEXT] ""]
    } else {
	lappend out 2 0
    }	
}

proc Binary::format::b {index in length} {
    upvar formatted out
    upvar value val
    
    if {$index < $length} {
	set bit [string index $in $index]
    } else {
	set bit 0
    }
    
    if {$bit != 0 && $bit != 1} {
	error "expected binary string but got \"${in}\" instead"
    } 
    
    set val [expr {$val + ($bit << ($index % 8))}]
    
    if {[expr {($index + 1) % 8}] == 0} {
	eval lappend out [split [format "%02X" $val] ""]
	set val 0
    }		
}

proc Binary::format::B {index in length} {
    upvar formatted out
    upvar value val
    
    if {$index < $length} {
	set bit [string index $in $index]
    } else {
	set bit 0
    }
    
    if {$bit != 0 && $bit != 1} {
	error "expected binary string but got \"${in}\" instead"
    } 
    
    set val [expr {($val << 1) + $bit}]
    
    if {[expr {($index + 1) % 8}] == 0} {
	eval lappend out [split [format "%02X" $val] ""]
	set val 0
    }		
}

proc Binary::format::h {index in length} {
    upvar formatted out
    
    if {$index < $length} {
	set digit1 [string index $in $index]
    } else {
	set digit1 0
    }
    
    if {[incr index] < $length} {
	set digit2 [string index $in $index]
    } else {
	set digit2 0
    }
    
    set digit2 [string toupper $digit2]
    if {[regexp {[^0-9A-F]} $digit2]} {
	error "expected hexadecimal string but got \"${in}\" instead"
    }
    lappend out $digit2
    
    set digit1 [string toupper $digit1]
    if {[regexp {[^0-9A-F]} $digit1]} {
	error "expected hexadecimal string but got \"${in}\" instead"
    }
    lappend out $digit1
}

proc Binary::format::H {index in length} {
    upvar formatted out
    
    if {$index < $length} {
	set digit1 [string index $in $index]
    } else {
	set digit1 0
    }
    
    if {[incr index] < $length} {
	set digit2 [string index $in $index]
    } else {
	set digit2 0
    }
    
    set digit1 [string toupper ${digit1}]
    
    if {[regexp {[^0-9A-F]} $digit1]} {
	error "expected hexadecimal string but got \"${in}\" instead"
    }
    
    lappend out $digit1
    
    set digit2 [string toupper ${digit2}]
    
    if {[regexp {[^0-9A-F]} $digit2]} {
	error "expected hexadecimal string but got \"${in}\" instead"
    }
    
    lappend out $digit2
}

proc Binary::format::c {index in length} {
    upvar formatted out
    
    eval lappend out [Binary::format::integer [lindex $in $index] 2]
}

proc Binary::format::s {index in length} {
    upvar formatted out
    
    eval lappend out [Binary::format::integer [lindex $in $index] 4 1]
}

proc Binary::format::S {index in length} {
    upvar formatted out
    
    eval lappend out [Binary::format::integer [lindex $in $index] 4]
}

proc Binary::format::i {index in length} {
    upvar formatted out
    
    eval lappend out [Binary::format::integer [lindex $in $index] 8 1]
}

proc Binary::format::I {index in length} {
    upvar formatted out
    
    eval lappend out [Binary::format::integer [lindex $in $index] 8]
}

proc Binary::format::f {index in length} {
    upvar formatted out
    
    if {[catch {coerce TEXT [lindex $in $index] -x sing} sing]} {
	error "expected floating-point number but got \"${in}\""
    }
    
    eval lappend out [split $sing ""]
}

proc Binary::format::d {index in length} {
    upvar formatted out
    
    if {[catch {coerce TEXT [lindex $in $index] -x doub} doub]} {
	error "expected floating-point number but got \"${in}\""
    }
    
    eval lappend out [split $doub ""]
}

proc Binary::format::x {index in length} {
    upvar formatted out
    
    lappend out 0 0
}

proc Binary::format::integer {num size {reverse 0}} {
    set hex [format "%0${size}X" $num]
    set len [string length $hex]
    # Negative numbers and overflows may have too many.
    # We only want the lowest digits. 
    set bytes [string range $hex [expr {$len - $size}] $len]
    set bytes [split $bytes ""]
    if {$reverse} {
	return [Binary::reverse $bytes]
    } else {
	return $bytes
    }
} 

#    format counters  #

proc Binary::format::countChar {} {
    upvar arg next
    upvar args remainder
    upvar len length
    upvar count cnt
    
    if {[llength $remainder] == 0} {
	error "not enough arguments for all format specifiers"
    } 
    set next [lindex $remainder 0]
    set remainder [lrange $remainder 1 end]
    
    set length [string length $next]
    
    switch $cnt {
	"" {
	    set cnt 1
	}
	"*" {
	    set cnt $length
	}
    }	
}

proc Binary::format::countNum {type} {
    upvar arg next
    upvar args remainder
    upvar len length
    upvar count cnt
    
    if {[llength $remainder] == 0} {
	error "not enough arguments for all format specifiers"
    } 
    set next [lindex $remainder 0]
    set remainder [lrange $remainder 1 end]
    
    set length [llength $next]
    
    switch $cnt {
	"" {
	    set cnt 1
	    if {[llength $next] != 1} {
		error "expected ${type} but got \"${next}\""
	    }
	    
	}
	"*" {
	    set cnt $length
	}
    }	
    if {$cnt > [llength $next]} {
	error "number of elements in list does not match count"
    } 
}

proc Binary::format::countx {} {
    upvar count cnt
    
    switch $cnt {
	"" {
	    set cnt 1
	}
	"*" {
	    error "cannot use \"*\" in format string with \"x\""
	}
    }
}
			
proc Binary::format::countX {} {
    upvar count cnt
    upvar cursor csr
    
    switch $cnt {
	"" {
	    incr csr -2
	}
	"*" {
	    set csr 0
	}
	default {
	    incr csr [expr {-2 * $cnt}]
	}
    }
    if {$csr < 0} {
	set csr 0
    } 
}

				
proc Binary::format::count@ {} {
    upvar output out
    upvar count cnt
    upvar cursor csr
    
    set len [llength $out]
    switch $cnt {
	"" {
	    error "illegal cursor"
	}
	"*" {
	    set csr $len
	}
	default {
	    set csr [expr {2 * $cnt}]
	}
    }
    for {set i $len} {$i < $csr} {incr i 2} {
	lappend out 0 0
    }
}

#  scan  #

set Binary::sizeof(a) 2
set Binary::sizeof(A) 2
set Binary::sizeof(b) 2
set Binary::sizeof(B) 2
set Binary::sizeof(h) 1
set Binary::sizeof(H) 1
set Binary::sizeof(c) 2
set Binary::sizeof(s) 4
set Binary::sizeof(S) 4
set Binary::sizeof(i) 8
set Binary::sizeof(I) 8
set Binary::sizeof(f) 8
set Binary::sizeof(d) 16

namespace eval Binary::scan {}

proc Binary::scan::doit {str formatString args} {
    global Binary::sizeof
    
    set cursor 0
    set numScanned 0
    set length [string length $str]
    
    while {$formatString != ""} {
	set formatString [string trimleft $formatString]
	if {![regexp {^(.)([0-9]+|\*)?(.*)$} \
	  $formatString blah type count formatString]} {
	    break
	} 
	
	switch -- $type {
	    "a" - "A" - "h" - "H" -
	    "c" - "s" - "S" - "i" - "I" - "f" - "d" {
		Binary::scan::countItems
	    }
	    "b" - "B" {
		Binary::scan::countBits
	    }
	    "x" {
		Binary::scan::countx
		continue
	    }
	    "X" {
		Binary::scan::countX
		continue
	    }
	    "@" {
		Binary::scan::count@
		continue
	    }
	    default {
		error "bad field specifier \"$type\""
	    }
	}
	
	set scanned {}
	set value 0
	
	if {$last > $length} {
	    break
	} 
	
	switch $type {
	    "a" - "A" {
		# _much_ more efficient to do ascii in one fell swoop
		set text [string range $str $cursor [expr {$last - 1}]]
		if {[string length $text] == [expr {2 * $count}]} {
		    set scanned [coerce TEXT -x $text TEXT]
		    set cursor $last
		} else {
		    break
		}
		if {$type == "A"} {
		    set scanned [string trimright $scanned " \x00"]
		} 
	    }
	    default {
		for {} {$cursor < $last} {incr cursor} {
		    if {$cursor >= $length} {
			return $numScanned
		    } 
		    
		    Binary::scan::${type}
		}
	    }
	}
	
	set var $scanned
	incr numScanned
	
    }
    
    return $numScanned
}

#    scanners  #

proc Binary::scan::b {} {
    upvar str s
    upvar cursor csr
    upvar count cnt
    upvar scanned scn
    
    scan [string range $s $csr [incr csr]] "%x" temp
    for {set i 0} {($i < 8) && ($i < $cnt)} {incr i} {
	append scn [expr {($temp >> $i) & 1}]
    }
    incr cnt -8
}

proc Binary::scan::B {} {
    upvar str s
    upvar cursor csr
    upvar count cnt
    upvar scanned scn
    
    scan [string range $s $csr [incr csr]] "%x" temp
    for {set i 0} {($i < 8) && ($i < $cnt)} {incr i} {
	append scn [expr {($temp >> (7 - $i)) & 1}]
    }
    incr cnt -8
}

proc Binary::scan::h {} {
    upvar str s
    upvar cursor csr
    upvar last lst
    upvar scanned scn
    
    set low [string index $s $csr]
    append scn [string index $s [incr csr]]
    if {$csr < $lst} {
	append scn $low
    }
}

proc Binary::scan::H {} {
    upvar str s
    upvar cursor csr
    upvar last lst
    upvar scanned scn
    
    append scn [string index $s $csr]
    incr csr
    if {$csr < $lst} {
	append scn [string index $s $csr]
    }
}

proc Binary::scan::c {} {
    upvar str s
    upvar cursor csr
    upvar scanned scn
    
    scan [string range $s $csr [incr csr]] "%x" temp
    set temp [expr {$temp > 0x80 ? $temp - 0x100 : $temp}]
    lappend scn $temp
}

proc Binary::scan::s {} {
    upvar str s
    upvar cursor csr
    upvar scanned scn
    
    set temp [string range $s $csr [incr csr 3]]
    set temp [join [Binary::reverse [split $temp ""]] ""]
    scan $temp "%x" temp
    set temp [expr {$temp > 0x8000 ? $temp - 0x10000 : $temp}]
    lappend scn $temp
}

proc Binary::scan::S {} {
    upvar str s
    upvar cursor csr
    upvar scanned scn
    
    scan [string range $s $csr [incr csr 3]] "%x" temp
    set temp [expr {$temp > 0x8000 ? $temp - 0x10000 : $temp}]
    lappend scn $temp
}

proc Binary::scan::i {} {
    upvar str s
    upvar cursor csr
    upvar scanned scn
    
    set temp [string range $s $csr [incr csr 7]]
    set temp [join [Binary::reverse [split $temp ""]] ""]
    scan $temp "%x" temp
    lappend scn $temp
}

proc Binary::scan::I {} {
    upvar str s
    upvar cursor csr
    upvar scanned scn
    
    scan [string range $s $csr [incr csr 7]] "%x" temp
    lappend scn $temp
}

proc Binary::scan::f {} {
    upvar str s
    upvar cursor csr
    upvar scanned scn
    
    set sing [string range $s $csr [incr csr 7]]
    lappend scn [expr {[coerce sing -x $sing TEXT] + 0.0}]
}

proc Binary::scan::d {} {
    upvar str s
    upvar cursor csr
    upvar scanned scn
    
    set doub [string range $s $csr [incr csr 15]]
    lappend scn [expr {[coerce doub -x $doub TEXT] + 0.0}]
}

#    scan counters  #

proc Binary::scan::countItems {} {
    global Binary::sizeof
    
    upvar args varNames
    upvar count cnt
    upvar cursor csr
    upvar type typ
    upvar length len
    upvar last lst
    
    if {[llength $varNames] < 1} {
	error "not enough arguments for all format specifiers"
    }
    
    uplevel "upvar 2 [lindex $varNames 0] var"
    set varNames [lrange $varNames 1 end]
    
    switch -- $cnt {
	"" {
	    set lst [expr {$csr + [set Binary::sizeof($typ)]}]
	    set cnt 1
	}
	"*" {
	    set lst $len
	    set cnt [expr {($lst - $csr) / 2}]
	}
	default {
	    set lst [expr {$csr + $cnt * [set Binary::sizeof($typ)]}]
	}
    }
}

proc Binary::scan::countBits {} {
    global Binary::sizeof
    
    upvar args varNames
    upvar count cnt
    upvar cursor csr
    upvar type typ
    upvar length len
    upvar last lst
    
    if {[llength $varNames] < 1} {
	error "not enough arguments for all format specifiers"
    }
    
    uplevel "upvar 2 [lindex $varNames 0] var"
    set varNames [lrange $varNames 1 end]
    
    switch $cnt {
	"" {
	    set lst [expr {$csr + [set Binary::sizeof($typ)]}]
	    set cnt 1
	}
	"*" {
	    set lst $len
	    set cnt [expr {($len - $csr) * 4}]
	}
	default {
	    set lst [expr {$csr + $cnt / 8 + [set Binary::sizeof($typ)]}]
	}
    }
}
            
proc Binary::scan::countx {} {
    upvar count cnt
    upvar cursor csr
    upvar length len
    
    switch -- $cnt {
	"" {
	    incr csr 2
	}
	"*" {
	    set csr $len
	}
	default {
	    incr csr [expr {2 * $cnt}]
	}
    }
    if {$csr > $len} {
	set csr $len
    }
}
            
proc Binary::scan::countX {} {
    upvar count cnt
    upvar cursor csr
    
    switch -- $cnt {
	"" {
	    incr csr -2
	}
	"*" {
	    set csr 0
	}
	default {
	    incr csr [expr {-2 * $cnt}]
	}
    }
    if {$csr < 0} {
	set csr 0
    }
}

proc Binary::scan::count@ {} {
    upvar count cnt
    upvar cursor csr
    upvar length len
    
    switch -- $cnt {
	"" {
	    error "missing count for \"@\" field specifier"
	}
	"*" {
	    set csr $len
	}
	default {
	    set csr [expr {2 * $cnt}]
	}
    }
    if {$csr > $len} {
	set csr $len
    }
}

#  generic  #

proc Binary::reverse {bin} {
    set out {}
    for {set i [expr {[llength $bin] - 2}]} {$i >= 0} {incr i -3} {
	lappend out [lindex $bin $i] [lindex $bin [incr i]]
    }
    return $out
}


