## -*-Tcl-*-
 # ==========================================================================
 #  Electric Corrections -- an extension package for Alpha
 # 
 #  FILE: "elecCorrections.tcl"
 #                                    created: 10/05/2000 {02:18:05 pm} 
 #                                last update: 07/11/2001 {13:29:31 PM}
 #                                
 #  Author: Craig Barton Upright
 #
 #  E-mail: <cupright@princeton.edu>
 #    mail: Princeton University,  Department of Sociology
 #          Princeton, New Jersey  08544
 #     www: <http://www.princeton.edu/~cupright/>
 #     
 #  Description:
 #  
 #  Activating this feature allows Alpha to automatically correct spelling
 #  mistakes using keyword lists defined by the user.  Inspired by the proc
 #  "sql_electUpper", in sqlMode.tcl .  See the "Electric Corrections Help"
 #  file for details.
 #  
 #  Most of the procedures in this file concern the "Electric Corrections"
 #  menu, more specifically the manipulation of user defined corrections. 
 #  The correcting procedure, correction::correctTypo, is actually quite
 #  simple and speedy.
 #  
 # -------------------------------------------------------------------
 #  
 # Copyright (c) 2000-2001  Craig Barton Upright
 #  
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #   Redistributions of source code must retain the above copyright
 #    notice, this list of conditions and the following disclaimer.
 # 
 #   Redistributions in binary form must reproduce the above copyright
 #    notice, this list of conditions and the following disclaimer in the
 #    documentation and/or other materials provided with the distribution.
 # 
 #   Neither the name of Alpha/Alphatk nor the names of its contributors may
 #    be used to endorse or promote products derived from this software
 #    without specific prior written permission.
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # 
 # ==========================================================================
 ##

alpha::feature elecCorrections 0.4 "global-only" {
    # We require AlphaTcl 7.2.2 for preference manipulation ...
    alpha::package require AlphaTcl 7.2.2
    # Add the preferences to the Electric menu.
    lunion varPrefs(Electrics) [list "Electric Corrections:" \
      smartI correctTypos smartIModes typoModes]
    # Automatically convert " i " to " I ".
    newPref f smartI            {1} global {correction::rebuildMenu}
    # Automatically correct misspellings/typos for the specified modes.
    newPref f correctTypos      {1} global {correction::rebuildMenu}
    # The "Smart i" correction will only take place for the modes listed
    # here.  If the list is empty, the correction take place for all modes.
    newPref v smartIModes       {Text TeX Bib HTML} global
    # Electric corrections only take place for the modes listed here.  If
    # the list is empty, corrections take place for all modes.
    newPref v typoModes         {} global
    # Create the "Electric Corrections" menu.
    menu::buildProc electricCorrections menu::buildCorrectionsMenu
    proc menu::buildCorrectionsMenu {} {
	global  smartI correctTypos
	set     menulist "viewCorrections"
	lappend menulist "addCorrections"
	lappend menulist "editCorrections"
	lappend menulist "removeCorrections"
	lappend menulist "(-"
	# Toggleable menu item -- smart I preference.
	if {$smartI} {
	    lappend menulist "!smart i"
	} else {
	    lappend menulist "smart i"
	} 
	lappend menulist "smartIModes"
	# Toggle the use of corrections.
	if {$correctTypos} {
	    lappend menulist "turnCorrectionsOff"
	} else {
	    lappend menulist "turnCorrectionsOn"
	} 
	lappend menulist "typoModes"
	lappend menulist "(-"
	lappend menulist "correctionsHelp"
	return [list build $menulist correction::menuProc {} "electricCorrections"]
    }
} {
    # Insert the menu into "Config --> Packages".  Note that prior to
    # AlphaTcl library 7.4a8, the "menu::insert" required an integer.  
    if {[alpha::package vcompare [alpha::package versions AlphaTcl] 7.4a7] == "1"} {
        menu::insert packages submenu "(-)" electricCorrections
    } else {
        menu::insert packages submenu "1"   electricCorrections
    }
    # Bind space, return keys to spell correct check.
    Bind '\ ' {correction::correctTypo "\ "}
    Bind '\r' {correction::correctTypo "\r"}
} {
    # Deactivation script.
    if {[alpha::package vcompare [alpha::package versions AlphaTcl] 7.4a7] == "1"} {
	menu::uninsert packages submenu "(-)" electricCorrections
    } else {
	menu::uninsert packages submenu "1"   electricCorrections
    }
    # Bind space, return keys to spell correct check.
    unBind '\ ' {correction::correctTypo "\ "}
    unBind '\r' {correction::correctTypo "\r"}
    Bind   '\ ' {typeText " "}
    Bind   '\r' {bind::CarriageReturn}
} uninstall {
    catch {file delete [file join $HOME Tcl Packages elecCorrections.tcl]}
    catch {file delete [file join $HOME Help "Electric Corrections Help"]}
} maintainer {
    "Craig Barton Upright" <cupright@princeton.edu>
}  help {
    file "Electric Corrections Help"
}

namespace eval correction {}

proc elecCorrections.tcl {} {}

# ===========================================================================
# 
#  --------  #
# 
#  Electric Correction menu procs  #
# 

proc correction::rebuildMenu {} {menu::buildOne electricCorrections}

proc correction::menuProc {menu item} {
    
    global smartI correctTypos typoModes smartIModes
    
    if {$item == "smarti" || $item == "smart i"} {
        # Flip the value of the "smartI" preference.
        set smartI [expr {[set smartI] ? 0 : 1}]
        prefs::modified smartI
        correction::rebuildMenu
        if {$smartI} {
            set onOrOff "on"
        } else {
            set onOrOff "off"
        }
        set modes $smartIModes
        if {![llength $smartIModes]} {
            set modes "all modes"
        } 
        status::msg "The \"Smart i\" preference is currently $onOrOff for $modes."
    } elseif {[regsub {turnCorrections} $item {} onOrOff]} {
        # Flip the value of the "correctTypos" preference.
        if {$onOrOff == "Off"} {
            set correctTypos "0"
        } else {
            set correctTypos "1"
        } 
        prefs::modified correctTypos
        correction::rebuildMenu
        set onOrOff [string tolower $onOrOff]
        set modes $typoModes
        if {![llength $typoModes]} {
            set modes "all modes"
        } 
        status::msg "Automatic corrections have been turned $onOrOff for $modes ."
    } elseif {$item == "correctionsHelp"} {
         package::helpFile "elecCorrections"
    } else {
        # Just use the proc defined below.
        correction::$item
    } 
}

# ===========================================================================
# 
# View Corrections
# 
# Place the names and elements of the array in a new window, and shrink it.
# 

proc correction::viewCorrections {} {
    
    global mode userCorrections lowerTypos
    
    # Make sure that the list of lower typos is up to date.
    correction::lowerTypos
    
    set windows [winNames]
    foreach w $windows {
        # Close any open "* Corrections *" windows.
        if {[regexp "\\* Corrections \\*" [win::StripCount $w]]} {
            bringToFront $w
            killWindow
        }
    }
    if {[listArray userCorrections] == ""} {
        status::msg "There are currently no defined misspellings."
        return
    }
    # We only show the lower case typos / corrections
    new -n "* Corrections *" -text [listArray lowerTypos] -m $mode
    # if 'shrinkWindow' is loaded, call it to trim the output window.
    catch {
        goto [maxPos] ; insertText "\r"
        selectAll     ; sortLines 
    }
    goto [minPos]
    insertText "Use the \"Edit Corrections\" \rmenu item to re-define them.\r\r"
    catch {shrinkWindow 2}
    winReadOnly
    status::msg ""
}

proc correction::lowerTypos {} {
    
    global userCorrections lowerTypos

    foreach typo  [array names userCorrections] {
        set first [string index $typo 0]
        set First [string toupper $first]
        if {$first != $First} {
            set lowerTypos($typo) $userCorrections($typo)
        } 
    } 
}

# ===========================================================================
# 
# Add Corrections
# 
# Present the user with a dialog to create a new misspelling.
# 

proc correction::addCorrections {{title ""} {typo ""} {correction ""}} {
    
    set finish [correction::addCorrectionsDialog "" $typo $correction]
    # Offer the dialog again to add more.
    set title "Create another Correction, or press Finish:"
    while {$finish != "1"} {
        set finish [correction::addCorrectionsDialog $title]
    }
    correction::viewCorrections
    menu::buildOne Correct
}

proc correction::addCorrectionsDialog {{title ""} {typo ""} {correction ""}} {
    
    global userCorrections
    
    if {$title == ""} {
        set title "Create a new Correction, or redefine an existing one:"
    } 
    set y 10
    if {[info tclversion] < 8.0} {
        set aCD [dialog::text $title 10 y 90]
        set yb 45
    } else {
        set aCD [list -T $title]
        set yb 20
    }
    eval lappend aCD [dialog::button   "Finish"                    300 yb   ]
    eval lappend aCD [dialog::button   "More"                      300 yb   ]
    eval lappend aCD [dialog::button   "Cancel"                    300 yb   ]
    if {$typo == ""} {
        eval lappend aCD [dialog::textedit "Typo :" $typo           10  y 25]
    } else {
        eval lappend aCD [dialog::text     "Typo :"                 10  y   ]
        eval lappend aCD [dialog::menu 10 y $typo $typo                     ]
    } 
    eval lappend aCD [dialog::textedit "Correction :"  $correction  10  y 25]
    incr y 20
    set result [eval dialog -w 380 -h $y $aCD]
    if {[lindex $result 2]} {
        # User pressed "Cancel'
        status::errorMsg "Cancelled."
    }
    set finish     [lindex $result 0]
    set typo       [string trim [lindex $result 3]]
    set correction [lindex $result 4]
    if {$typo != "" && $correction != ""} {
        correction::capitalizeTypo $typo $correction
        status::msg "\"$typo -- $correction\" has been added."
        return $finish
    } elseif {$finish == "1"} {
        return $finish
    } else {
        status::errormsg "Cancelled -- one of the dialog fields was empty."
    } 
}

proc correction::capitalizeTypo {typo correction} {
    
    global userCorrections
    
    set word1 $typo
    set word2 $correction
    
    # First create capitalized and lower case misspelling.
    set first1     [string index   $word1 0]
    set first2     [string tolower $first1]
    set First3     [string toupper $first1]
    set typo       [concat $first2[string range $word1 1 end]]
    set Typo       [concat $First3[string range $word1 1 end]]
    # Then create capitalized and lower case correction.
    set first1     [string index   $word2 0]
    set first2     [string tolower $first1]
    set First3     [string toupper $first1]
    set Correction [concat $First3[string range $word2 1 end]]
    # Then add both of them to arrdefs.tcl
    set userCorrections($typo) $correction
    set userCorrections($Typo) $Correction
    prefs::addArrayElement userCorrections $typo $correction
    prefs::addArrayElement userCorrections $Typo $Correction
    # Make sure that the list of lower typos is up to date.
    correction::lowerTypos
    
}

# ===========================================================================
# 
# Edit Corrections
# 
# Present the user with a dialog to edit a current misspelling.
# 

proc correction::editCorrections {} {
    
    global userCorrections lowerTypos
    
    # Make sure that the list of lower typos is up to date.
    correction::lowerTypos
    
    set typo [listpick -p "Select a Typo to edit:" \
      [array names lowerTypos]]
    set correction $userCorrections($typo)
    set title "Edit the \"$typo\" correction:"
    set finish [correction::addCorrectionsDialog $title $typo $correction]
    # Offer the dialog again to add more.
    while {$finish != "1"} {
        set typo [listpick -p \
          "Select another Typo to edit, or Cancel:" \
          [array names lowerTypos]]
        set correction $userCorrections($typo)
        set title "Edit the \"$typo\" correction:"
        set finish [correction::addCorrectionsDialog $title $typo $correction]
    }
    correction::viewCorrections
}

# ===========================================================================
# 
# Remove Corrections
# 
# Present the user with a dialog to remove a current misspelling.
# 

proc correction::removeCorrections {{removeList ""}} {
    
    global userCorrections correctionsSet lowerTypos
    
    if {$removeList == ""} {
        # First list the user defined misspellings.
        set userTypos ""
        foreach typo  [array names userCorrections] {
            set first [string tolower [string index $typo 0]]
            set typo  [concat $first[string range $typo 1 end]]
            if {![info exists correctionsSet($typo)]} {
                # We know that this is user defined.
                lappend userTypos $typo
            } elseif {$userCorrections($typo) != $correctionsSet($typo)} {
                # We know that this has not been redefined.
                lappend userTypos $typo
            } 
        } 
        if {![llength $userTypos]} {
            status::msg "Cancelled -- there are no user defined misspellings to remove."
            correction::rebuildMenu
            return
        } 
        set removeList [listpick -l -p "Select some Typos to remove:" \
          [lunique $userTypos]]
    } 

    foreach typo $removeList {
        set word   [lindex $typo 0]
        # First create upper and lower case typo.
        set first1 [string index   $word 0]
        set first2 [string tolower $first1]
        set First3 [string toupper $first1]
        set typo   [concat $first2[string range $word 1 end]]
        set Typo   [concat $First3[string range $word 1 end]]
        # Then remove both of them from arrdefs.tcl
        catch {prefs::removeArrayElement userCorrections $typo}
        catch {prefs::removeArrayElement userCorrections $Typo}
        catch {unset userCorrections($typo)}
        catch {unset userCorrections($Typo)}
        catch {unset lowerTypos($typo)}
    }
    correction::viewCorrections
}

# ===========================================================================
# 
# Smart I Modes
# 
# Edit the "smartIModes" preference.
# 

proc correction::smartIModes {} {
    
    global smartIModes

    set newModesPref [prompt "Modify the list of modes\
      that use \"Smart i\"." $smartIModes]
    set smartIModes $newModesPref
    prefs::modified smartIModes
    if {![llength $newModesPref]} {
        set newModesPref "all modes"
    } 
    status::msg "\"Smart i\" modes include $newModesPref ."
}

# ===========================================================================
# 
# Typo Modes
# 
# Edit the "typoModes" preference.
# 

proc correction::typoModes {} {
    
    global typoModes

    set newModesPref [prompt "Modify the list of modes\
      that use \"Electric Corrections\"." $typoModes]
    set typoModes $newModesPref
    prefs::modified typoModes
    if {![llength $newModesPref]} {
        set newModesPref "all modes"
    } 
    status::msg "Correcting modes include $newModesPref"
}

# ===========================================================================
# 
# Unset Corrections List
# 
# Remove all of the pre-defined corrections.  This is not currently used
# in the menu, but if the pre-defined list was exanded, we would want to
# be able to add and remove the predefined list.
# 

proc correction::unsetCorrectionsList {} {
    
    global userCorrections correctionsSet
    
    set removeList ""
    foreach typo [array names correctionsSet] {
        if {![info exists userCorrections($typo)]} {
            # The list has already been unset
            status::msg "The pre-defined list of correction has already been unset."
            return
        } 
        if {$userCorrections($typo) == $correctionsSet($typo)} {
            # We know that this has not been redefined.
            lappend removeList $typo
        } 
    } 
    if {[llength $removeList] != "0"} {
        correction::removeCorrections $removeList
        correction::viewCorrections
        status::msg "The pre-defined list of corrections has been removed."

    } else {
        status::msg "There were no pre-defined corrections to remove."
    } 
}

# ===========================================================================
# 
# Restore Corrections List
# 
# Restore all of the pre-defined corrections.  This is not currently used
# in the menu, but if the pre-defined list was exanded, we would want to
# be able to add and remove the predefined list.
# 

proc correction::restoreCorrectionsList {} {
    
    global correctionsSet
    
    foreach typo [array names correctionsSet] {
        correction::capitalizeTypo $typo $correctionsSet($typo)
    } 
    correction::viewCorrections
    status::msg "The pre-defined list of corrections has been restored."
}

# ===========================================================================
# 
#  --------  #
# 
#  Correct Typo  #
# 
# This is all it takes to correct spelling mistakes as you type.  Note that
# even though this is only bound to <space> and <return>, intermediate
# characters (such as , .  ; etc) are generally ignored when finding the
# last "word".  This means that i.  should be corrected to I. as soon as
# the <space> or <return> keys are pressed, without disturbing the
# subsequent period.  This depends on the "wordBreak" preference of the
# current mode...
# 

proc correction::correctTypo {nextChar} {
    
    global mode smartI smartIModes correctTypos typoModes userCorrections
    
    if {[isSelection]} {
        deleteSelection
        set where [getPos]
    } else {
        # What was the word that was just entered?
        set where [getPos]
        backwardWord
        set start [getPos]
        forwardWord
        set end   [getPos]
        set word  [getText $start $end]
        # Do we need to delete the word and insert a correction?
        if {$word == "i" && $smartI} {
            if {[lsearch -exact $smartIModes $mode] != "-1" || ![llength $smartIModes]} {
                # Smart i -- automatically change to I
                replaceText $start $end "I"
            }
        } elseif {[info exists userCorrections($word)] && $correctTypos} {
            if {[lsearch -exact $typoModes $mode] != "-1" || ![llength $typoModes]} {
                # A defined typo -- automatically insert the correction.
                replaceText $start $end $userCorrections($word)
                set where [pos::math $where + [string length $userCorrections($word)] - [string length $word]]
            }
        }
    }
    # Now insert the calling character, which was bound to this proc.
    goto $where
    if {[is::Eol $nextChar]} {
        bind::CarriageReturn
    } else {
        insertText $nextChar
    }
}

# ===========================================================================
# ===========================================================================
# 
# Create the initial array of corrections.  These need to be lower-case. 
# More could be added ...

array set correctionsSet {
    hte "the"
}

# These are not yet defined as misspellings -- they're simply a list that
# we can add (or remove) using the "Electric Corrections" menu.  We only
# add them if the correction has not already been (re)defined in
# "arrdefs.tcl"

foreach typo [array names correctionsSet] {
    if {![info exists userCorrections($typo)]} {
        correction::capitalizeTypo $typo $correctionsSet($typo)
    } 
} 


# ===========================================================================
# 
#  Back Compatability  #
# 
# Ensuring the certain procs from the AlphaTcl library are in place. 
# 

# We need "is::Eol", only available in AlphaTcl 7.4b7 and up ...

if {[alpha::package vcompare [alpha::package versions AlphaTcl] 7.4b6] != "1"} {
    namespace eval is {}
    proc is::Eol {anyString} {
        return [regexp "^\[\r\n\]+$" $anyString]
    }
} 

# "win::StripCount" was renamed in 7.4b16

if {[alpha::package vcompare [alpha::package versions AlphaTcl] 7.4b15] != "1"} {
    namespace eval win {}
    proc win::StripCount {name} {
        regsub { <[0-9]+>} $name {} name
        return $name
    }
} 

# ===========================================================================
# 
#  Version History  #
# 
# 0.1   (10/00)   First version, named "mispellCorrect".
#                 "mispellCorrect" submenu is inserted in "Text" menu.
# 0.2   (10/00)   Renamed "elecCorrections".
#                 "electricCorrections" submenu is inserted in "Packages" menu.
#                 Corrections are only bound to <space>, <return>.
#                 Added "Electric Corrections Help" file.
# 0.3   (10/00)   Numerous fixes, many from Vince (thanks!).
#                 Namespace is now "correction".
#                 Calling character is now inserted before typo correction.
# 0.3.1 (10/00)   Better correction::correctTypo cursor movement.
# 0.3.2 (11/00)   Users can now define a lower case typo with a capitalized
#                   correction. (thanks Bernard!)
#                 Back compatability issue addressed with win::StripCount.
#                 Fixed "isSelection" bug, so that hilited selection will
#                   now delete without checking for a correction.
# 0.3.3 (04/01)   Better dialogs, especially with Alphatk/8.
# 0.4   (06/01)   Package is now a global-only feature, to allow for a
#                   de-activation script.
# 

# ===========================================================================
# 
# .