## -*-Tcl-*- (install)
 # ###################################################################
 # 
 #  FILE: "latexAccents.tcl"
 #                                    created: 14/1/1998 {6:57:41 pm} 
 #                                last update: 11/09/2001 {17:46:27 AM} 
 # Version:      1.05
 # Purpose:      Translates Mac characters to their LaTeX equivalents 
 #	         and vice-versa
 # Language:     TCL
 # Author:       F. Miguel Dionsio
 # Email:	 fmd@math.ist.utl.pt
 # Copying:      Free
 # 
 # The purpose of this tcl script is to translate mac characters to 
 # their LaTeX equivalents and vice-versa (from within the Mac 
 # editor Alpha).  The characters are those of the Courier font, 
 # with the exception of the ones that, in LaTeX, belong to math 
 # mode (see list below).  This is useful to  people who share 
 # their LaTeX sources with non-Mac users.
 # 
 # Two procedures are defined: accentsToLatexInRegion and 
 # latexToAccentsInRegion that convert all SELECTED text.  The 
 # conversion latexToAccentsInRegion tries to match (using regular 
 # expressions) various forms of the character definition in LaTeX.
 #
 # ###################################################################
 # 
 # Modified on 8/07/1999 by Dominique d'Humieres
 # E-mail: <dominiq@physique.ens.fr>
 # 
 # Added the choice between different possible syntax for accents and
 # ligatures:  replaced by \'{e} or \'e, and  by \oe{}, {\oe} or \oe
 # ('accentSyntax' in the 'Mode Prefs->Preferences...' dialog).
 # 
 # Added the binding of accented characters to their TeX form depending
 # on the above preference: typing  yields \'e (or \'{e}) (set by 
 # 'bindLatexAccents' in the 'Mode Prefs->Preferences...' dialog).
 # 
 # On 4/08/1999 added the Slovenian keyboard and a hook to the keyboard.
 # 
 # On 11/10/1999 added the option to keep the Mac accents when characters
 # are typed in comments. Note that this option does not work when
 # 'accentsToLatexInRegion' is used. The default setting of this flag is on,
 # it requires that 'bindLatexAccents' is on.
 # Added also a rather convoluted way to select and deselect the
 # 'latexAccents' feature in 'Config->Mode Prefs->Menus and Features'
 # {Select features for mode 'TeX'}.
 # 
 # On 21/10/1999 the default setting for 'bindLatexAccents' is now off. Small
 # change in 'TeX::bindOptionLatexAccents' in order to avoid to
 # bind option+key for keys already bound.
 # 
 # On 12/01/2000 added the flag 'accentSmartEscape'. When set this flag
 # allows one to delete the TeX accents with the backspace key. A single
 # character deletion is bound to SHIFT+backspace. The default setting
 # of this flag is off.
 # Works with any character for the \', \^, \`, \", \~, \c and \v accents,
 # but not for the \=, \., \u, \H, \r, \t, \d, \b, and \k ones.
 # 
 # On 1/02/2000 did some cleaning of the 'TeX::initAccents',
 # 'TeX::accentsActivate', and 'TeX::accentsDeactivate' procs.
 # NOTE: if the latexAccents package is activated as a global feature and
 # the 'bindLatexAccents' flag is on, the bindings are done only for the
 # TeX and Bib modes (this departs from the usual behavior of global settings,
 # but this is probably better than having the TeX accents for all modes).
 # See also the bug list at the end of these comments.
 # 
 # On 4/02/2000 made some corrections following the remarks from Pierre
 # Basso, Igor Kramberger, and Bart Truyen.
 # 
 # On 18/3/2000 added the 'Brasil' keyboard.
 # 
 # On 4/4/2000 added the 'Italian Pro' keyboard.
 # 
 # On 25/8/2000 modified 'TeX::bindLatexAccents' and 
 # 'TeX::bindOptionLatexAccents' and places using regexp for
 # word boundary for Alpha8.
 # 
 # On 11/9/2000 removed the modification of 'TeX::bindLatexAccents' and 
 # 'TeX::bindOptionLatexAccents' for Alpha8.
 # 
 ######################################################################
 #
 # List of characters on non-Slovenian keyboards (use Courier to see them):
 # 
 #	""	""	""	""	""	 ""
 #	""	""	""	""	""	 ""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""	""
 #	""	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""
 #	""	""
 #	""	""	""	""	""	""
 #	""	""	""
 #	""	""	""	""
 #	""	""	""
 # 
 #  WARNING:  appears different in Monaco
 # 
 # List of characters on Slovenian keyboards (use Slovenian Courier to see them):
 # 
 #	""	""	""	""	""	 ""
 #	""	""	""	""	""	 ""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""	""
 #	""	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""	""
 #	""	""	""
 #	""	""
 #	""	""	""	""	""	""
 #	""	""	""
 #	""	""	""	""
 #	""	""	""	""	""	""	""	""	""	""
 # 
 #  List of OMITTED characters from Courier:
 # 
 #  ,,, , , and the following symbols from mathematical mode:
 #  ,,,,,,,,,,,?,,,,,,,,,,
 # 
 # If you find some more problems please tell me. (fmd@math.ist.utl.pt)
 #
 ######################################################################
 # 
 # Alpha commands used: getText, getPos, selEnd, replaceText; 
 # all others are TCL built-ins
 # 
 # ####################################################################
 # 
 # The binding of accented characters is done for the TeX and Bib modes.
 # The characters obtained from keys without the option modifier
 # are bound without knowledge of the keyboard (provided the characters
 # in 'TeX::_asciiAccents' yield the TeX equivalent in 'TeX::_latexAccents').
 # I have been unable to do the same thing for the characters requiring
 # the option modifier. The binding is then done using the 'TeX_keyboards'
 # array which stores a list of four strings for each keyboard known by
 # Alpha: the first string stores the characters obtained with the option
 # modifier, the second one the corresponding keys without modifier,
 # the third one the characters obtained with the option + shift modifiers,
 # and the last one the corresponding keys without modifier.
 # There are many characters which can be accessed either through a 'dead-key' 
 # or with option (+ shift) modifier; most of the time I did not put in
 # the 'TeX_keyboards' table the corresponding characters with the option
 # modifier (with a few exceptions such as '' for the Flemish keyboard
 # which can be accessed through option+e).
 # I have done uneducated guesses for the non-French keyboards and I have only
 # done some (but non exhaustive) testing, so if you find missing characters
 # or errors let me know. 
 # If you have personnal reason to use your own keyboard map, you can put in
 # your 'TeXPrefs.tcl' file (replacing 'French' by the name of your keyboard
 # and doing the same modification in your 'BibPrefs.tcl' file if you
 # want the same binding in the Bib mode):
 # 
 #  TeX::bindOptionLatexAccents 0
 # 
 #  set "TeX_keyboards(French)" {
 #   {ϧ}
 #   {azobc!;}
 #   {}
 #   {azo,fm}
 #  }
 # 
 #  TeX::bindOptionLatexAccents 1
 # 
 # Bugs:
 # 
 # The following keyboards are set by default to the U.S. one: Croatian.
 # Addung the correct Croatian keyboard requires something similar to
 # the Slovenian one, but much more work. I shall be motivated to do it
 # only if someone shows some interest for it!
 # 
 # During the testing I switched between different keyboards without
 # restarting Alpha, unbinding and binding the accent keys.
 # Sometimes the binding function did not work; trying to trace the
 # procedure set the binding to work. This bug should not appear under
 # normal use (?).
 # 
 # When the package is activated at the global level, it does not know if
 # the activation/deactivation is done through the 'Preferences->Menus And Features'
 # or 'Modes Prefs->Menus And Features' dialogs. As a consequence: 
 # - if the package is activated at the global level while a window is open,
 #   the binding (if selected) will be done only for the mode of this window
 #   (while after a restart it will be done for the Bib and TeX modes);
 # - if the feature is deselected in the 'Modes Prefs->Menus And Features' dialog
 #   in a mode which is not Bib or TeX, the package will be deactivated at the global
 #   level (unless it appears in 'mode::features(Bib)' or  'mode::features(TeX)');
 # These problems (and possibly others) should not appear under normal use (?):
 # global activation without any open window, followed by a restart.
 # 
 ##

# extension declaration
alpha::feature latexAccents 1.1.9 {TeX Bib} {
    alpha::package require TeX 4.7
    alpha::package require AlphaTcl 7.5d28
    hook::register mode::init TeX::initAccents TeX Bib
} { TeX::accentsActivate
} { TeX::accentsDeactivate
} maintainer {
    {Dominique d'Humires} dominiq@physique.ens.fr
    {F. Miguel Dionsio} fmd@math.ist.utl.pt
} help {
    Converts between international symbols (available from the keyboard in use,
    presently Western Latin and Slovenian) and LaTeX codes.
    
    A choice between different possible syntax for accents and ligatures
    ( replaced by \'{e} or \'e, and  by \oe{}, {\oe} or \oe ) is available
    through the preference 'accentSyntax' in the  'Mode Prefs->Preferences...'
    dialog.

    Accented characters can be bound to their TeX form depending on the above
    preference: typing  yields \'e (or \'{e}) (set by 'bindLatexAccents'
    in the 'Mode Prefs->Preferences...' dialog, default is OFF).

    The Mac accents can be kept when characters are typed in comments (set by 
    'accentsInComments' in the 'Mode Prefs->Preferences...' dialog). 
    NOTE that this option requires 'bindLatexAccents' to be ON and works 
    only when typing. It does not work with the conversion procedures called 
    from the 'LaTeX Utilities' menu. 

    When the flag 'accentSmartEscape' ('Mode Prefs->Preferences...' dialog, 
    default is OFF) is ON, the TeX accents are deleted at once using the 
    backspace key.  A single character deletion is bound to SHIFT+backspace.
    Presently works with any character for the \', \^, \`, \", \~, \c and 
    \v accents, but not for the \=, \., \u, \H, \r, \t, \d, \b, and \k ones.
    WARNING! this feature deletes several characters at once and uses a
    general syntax to find accent or ligature patterns; it is not restricted 
    to the syntax chosen in 'accentSyntax' or the character set used
    in the conversion procedures.  For instance, '\^   {     v      }' will
    be recognized as a legal accented v and will be deleted at once. If you
    are impatient and press the backspace key several time, you will delete
    extra characters (possibly critical ones such as curly braces), 
    BE CAREFUL if you use this feature!.
    
} uninstall this-file

proc  TeX::initAccents  {args} {
	
    global TeX::_modeSeen TeXmodeVars mode

    if {![info exists {TeX::_modeSeen}]} {
	
	if {$mode == ""} {
	    set TeX::_modeSeen [list TeX Bib]
	} else {
	    set TeX::_modeSeen $mode
	}
	
	hook::register keyboard TeX::switchKeyboard
	menu::insert {LaTeX Utilities} items 5 \
	  "(-" {Convert Accents to LaTeX} {Convert LaTeX to Accents}
	
	# Set the TeX syntax for accents and ligatures.
	newPref variable accentSyntax 0 TeX TeX::setLatexAccents \
	  [list "\\'e & \\oe{}" "\\'e & {\\oe}" "\\'e & \\oe " "\\'{e} & \\oe{}" \
		"\\'{e} & {\\oe}" "\\'{e} & \\oe "] index
	# If set, keeps the Mac accents in comments.
	newPref f accentsInComments 1 TeX
	# Bind the TeX accents to the accented keys.
	newPref f bindLatexAccents 0 TeX TeX::setAccentsBindings
	# If set, allows the TeX accents to be deleted at once with the delete key.
	newPref f accentSmartEscape 0 TeX TeX::setAccentSmartEscape

	TeX::setLatexAccents 1
	TeX::setAccentSmartEscape 1 "$mode"
	TeX::addSmartEscape 1
    } elseif {[lsearch ${TeX::_modeSeen} $mode] < 0} {
	if {$mode != ""} { lappend TeX::_modeSeen "$mode" }
	if {$TeXmodeVars(bindLatexAccents)} { TeX::bindLatexAccents 1 }
	TeX::setAccentSmartEscape 1 "$mode"
   }
	
}

proc TeX::accentsActivate  {} {
    global alpha::changingMode TeX::_latexAccents TeXmodeVars TeX::_modeSeen mode
    if {!${alpha::changingMode}} { TeX::initAccents }
}

proc TeX::accentsDeactivate  {} {
    global alpha::changingMode TeXmodeVars TeX::_modeSeen TeX::_latexAccents \
           mode::features mode keyboard
    if {${alpha::changingMode}} { return }
    if {[lsearch ${TeX::_modeSeen} $mode] < 0} {
	foreach m ${TeX::_modeSeen} {
	    if {[lsearch -exact [set mode::features($m)] "latexAccents"] < 0} {
		set TeX::_modeSeen [lremove -l ${TeX::_modeSeen} $m]
		if {$TeXmodeVars(bindLatexAccents)} { TeX::bindLatexAccents 0 $keyboard $m }
		TeX::setAccentSmartEscape 0 $m
	    }
	}
	if {[llength ${TeX::_modeSeen}]} { return }
    } else {
	set TeX::_modeSeen [lremove -l ${TeX::_modeSeen} $mode]
	if {$TeXmodeVars(bindLatexAccents)} { TeX::bindLatexAccents 0 }
	TeX::setAccentSmartEscape 0 $mode
	if {[llength ${TeX::_modeSeen}]} { return }
	TeX::addSmartEscape 0
    } 
    unset TeXmodeVars(accentSyntax)
    prefs::removeArrayElement TeXmodeVars accentSyntax
    unset TeXmodeVars(accentsInComments)
    prefs::removeArrayElement TeXmodeVars accentsInComments
    unset TeXmodeVars(bindLatexAccents)
    prefs::removeArrayElement TeXmodeVars bindLatexAccents
    unset TeXmodeVars(accentSmartEscape)
    prefs::removeArrayElement TeXmodeVars accentSmartEscape
    unset TeX::_modeSeen
    unset TeX::_latexAccents
    menu::uninsert {LaTeX Utilities} items 5 \
      "(-" {Convert Accents to LaTeX} {Convert LaTeX to Accents}
    hook::deregister keyboard TeX::switchKeyboard
    hook::deregister mode::init TeX::initAccents TeX Bib
}

######################################################################
#
# Returns, for argument "a" the regular expression 
# [ \t]*(a|{[ \t]*a[ \t]*}),
# used to look for alternative ways of writing accents, for example :
# \`a, \` a, \`{a}, etc.
#
######################################################################
proc rexp  {c {pre ""}} {
    set ws "\[ \t\]*"
    return $ws\($pre$c|{$ws$c$ws}\)
}

######################################################################
#
# Returns, for argument "c" the regular expression 
# [ \t]*( c|{c}),
# used to look for alternative ways of writing cedilla, for example :
# \c c, \c{c}, \c {c} etc. Note that \c{}c, \c{ C}, or \c{C } do not
# yield the right glyph.
#
######################################################################
proc rexpc  {c {pre ""}} {
    set ws "\[ \t\]*"
    return $ws\($pre$c|{$c}\)
}

######################################################################
#
# Returns, for argument "\\i" the regular expression 
# [ \t]*(\i[ 	]|\i{}|\i\b|{[ 	]*\i[ 	]*}),
# used to look for alternative ways of writing accented i, for example :
# \'\i , \'{\i}, \'\i{} etc. 
#
######################################################################
proc rexpi  {c {post ""}} {
    set ws "\[ \t\]*"
    if {[info tclversion] >= 8.0} {
	return $ws\($c$post|$c\{\}|$c\\M|{$ws$c$ws}\)
    } else {
	return $ws\($c$post|$c\{\}|$c\\b|{$ws$c$ws}\)
    } 
}

######################################################################
#
# Returns, for argument "o" the regular expression 
# \\\\o$sep|{$ws\\\\o$ws},
# used to look for alternative ways of writting ligatures, for example :
# \o{}, {\o}, \o , etc.
#
######################################################################
proc rexpl  {c} {
    set ws "\[ \t\]*"
    if {[info tclversion] >= 8.0} {
	set sep {[ 	]*( |	|\{\}|\M)}
    } else {
	set sep {[ 	]*( |	|\{\}|\b)}
    } 
    return \(\\\\$c$sep|{$ws\\\\$c$ws}\)
}

######################################################################
#
# Returns, for argument "\\i" the regular expression 
# [ \t]*(\i[ 	]|\i{}|\i\b|{\i}),
# used to look for alternative ways of writing accented i, for example :
# \'\i , \'{\i}, \'\i{} etc. 
#
######################################################################
proc rexpv  {c {post ""}} {
    set ws "\[ \t\]*"
    if {[info tclversion] >= 8.0} {
	return $ws\($c$post|$c\{\}|$c\\M|{$c}\)
    } else {
	return $ws\($c$post|$c\{\}|$c\\b|{$c}\)
    } 
}

######################################################################
# 
# This does the rest: defines the list of chars (all), the list of their 
# LaTex equivalent (texall) and the list of corresponding regular 
# expressions (regall).  When translating to LaTeX replaces all 
# ocurrences of each char by the corresponding LaTeX equivalent (using 
# regsub, see the TCL manual).  In the other direction replaces the text 
# that matches the regular expression by the corresponding char.
# 
######################################################################

proc TeX::ConvertAccentstoLaTeX {} {TeX::replaceAccents 1}
proc TeX::ConvertLaTeXtoAccents {} {TeX::replaceAccents 0}

############################################################
# 
# List of characters to be replaced by LaTeX equivalent
# 
############################################################

proc  TeX::setAccentSmartEscape {args {listMode ""}} {
    global TeXmodeVars TeX::_modeSeen
    if {$args == "accentSmartEscape"} {
	TeX::addSmartEscape $TeXmodeVars(accentSmartEscape)
    }
    if {$listMode == ""} {
	set tmp ${TeX::_modeSeen}
    } else {
	set tmp $listMode
    }
    foreach m $tmp {
	if {$TeXmodeVars(accentSmartEscape)} {
	    if {$m != "TeX"} {
		ascii 0x08        escapeSmartStuff $m
	    }
	    Bind  0x33  <s>   backSpace $m
	} else {
	    if {$m != "TeX"} {
		unascii 0x08        escapeSmartStuff $m
	    }
	    unbind  0x33  <s>   backSpace $m
	}
    }
}

proc  TeX::setAccentsBindings {args} {
    global TeXmodeVars TeX::_modeSeen keyboard
    foreach m ${TeX::_modeSeen} {
	TeX::bindLatexAccents $TeXmodeVars(bindLatexAccents) $keyboard $m
    }
}

proc  TeX::setLatexAccents {args} {
    global TeXmodeVars TeX::_latexAccents TeX::_asciiAccents TeX::_globalBindings \
           TeX::_modeSeen keyboard mode::features
    if {![info exists {TeX::_globalBindings}]} {
	set TeX::_globalBindings ""
	set tmp [lsort -ignore [array names mode::features]]
	foreach b [split [bindingList] "\r"] {
	    if {[regexp "<s?os?>" $b]} {		
		set lst [lindex [split $b  " "] end]
		if {[lsearch $tmp $lst] < 0} {
		    append TeX::_globalBindings "$b\r"
		}
	    }
	}
    }
    set quote {"}  
    set seplater {\\\\sepsep//}
    if {$TeXmodeVars(bindLatexAccents) && $args!=1} {
	if {$args == "accentSyntax"} {
	    set key $keyboard
	} else {
	    set key $args
	}
	foreach m ${TeX::_modeSeen} {
	    TeX::bindLatexAccents 0 $key $m
	}
    }
    if {$keyboard == "Slovenian"} {
	set TeX::_asciiAccents "̀옗ͅ󆖄ƍ蹾ȩ޿Чɣ٠"
    } else {
	set TeX::_asciiAccents "̀蓒옗ͅ󆖄ٍξɣ"
    }
	
    ############################################################
    # 
    # List of the LaTex equivalents
    # 
    ############################################################
    
    set texis	[list  "\\`{\\i}" "\\'{\\i}" "\\^{\\i}"  "\\$quote{\\i}"]
    set texoth1  [list "\\dots$seplater" "\\pounds$seplater" "\\S$seplater" "\\P$seplater"]
    set texoth2  [list  "{\\leavevmode\\raise.585ex\\hbox{\\b{a}}}"  "{\\leavevmode\\raise.6ex\\hbox{\\b{o}}}" "\\accent'27"]
    if {$keyboard == "Slovenian"} {
	set texfix  [list  "\\c{c}"  "\\c{C}" "\\v{c}" "\\v{s}" "\\v{z}" "\\v{C}" "\\v{S}" "\\v{Z}"]
	set texoth3 [list  "\\copyright$seplater" "\\dag$seplater" "?`" "!`"]
    } else {
	set texfix  [list  "\\c{c}"  "\\c{C}"]
	set texoth3 [list "\\copyright$seplater" "\\dag$seplater" "?`" "!`" "\\ddag$seplater"]
    }
	    
    if {$TeXmodeVars(accentSyntax) > 2} {
	set texas    [list "\\`{a}" "\\'{a}" "\\^{a}" "\\~{a}" "\\$quote{a}"] 
	set texcas   [list "\\`{A}" "\\'{A}" "\\^{A}" "\\~{A}" "\\$quote{A}"]									 
	set texes    [list "\\`{e}" "\\'{e}" "\\^{e}" "\\$quote{e}"]				
	set texces   [list "\\`{E}" "\\'{E}" "\\^{E}" "\\$quote{E}"]   
	set texcis   [list "\\`{I}" "\\'{I}" "\\^{I}" "\\$quote{I}"]
	set texos    [list "\\`{o}" "\\'{o}" "\\^{o}" "\\~{o}" "\\$quote{o}"]
	set texcos   [list "\\`{O}" "\\'{O}" "\\^{O}" "\\~{O}" "\\$quote{O}"]
	set texus    [list "\\`{u}" "\\'{u}" "\\^{u}" "\\$quote{u}"]
	set texcus   [list "\\`{U}" "\\'{U}" "\\^{U}" "\\$quote{U}"]
	if {$keyboard == "Slovenian"} {
	    set texvar   [list "\\~{n}" "\\~{N}" "\\'{c}" "\\'{C}"]
	} else {
	    set texvar   [list "\\~{n}" "\\~{N}" "\\$quote{y}"  "\\$quote{Y}"]
	}
    } else {
	set texas    [list "\\`a" "\\'a" "\\^a" "\\~a" "\\${quote}a"] 
	set texcas   [list "\\`A" "\\'A" "\\^A" "\\~A" "\\${quote}A"]									 
	set texes    [list "\\`e" "\\'e" "\\^e" "\\${quote}e"]				
	set texces   [list "\\`E" "\\'E" "\\^E" "\\${quote}E"]   
	set texcis   [list "\\`I" "\\'I" "\\^I" "\\${quote}I"]
	set texos    [list "\\`o" "\\'o" "\\^o" "\\~o" "\\${quote}o"]
	set texcos   [list "\\`O" "\\'O" "\\^O" "\\~O" "\\${quote}O"]
	set texus    [list "\\`u" "\\'u" "\\^u" "\\${quote}u"]
	set texcus   [list "\\`U" "\\'U" "\\^U" "\\${quote}U"]
	if {$keyboard == "Slovenian"} {
	    set texvar  [list "\\~n" "\\~N" "\\'c" "\\'C"]
	} else {
	    set texvar   [list "\\~n" "\\~N" "\\${quote}y"  "\\${quote}Y"]
	}
    }
    switch -- [expr {$TeXmodeVars(accentSyntax) % 3}] {
       0    { set texlig1 [list "\\aa{}" "\\AA{}" "\\oe{}" "\\OE{}" \
				"\\ae{}" "\\AE{}" "\\o{}" "\\O{}"]
	      if {$keyboard == "Slovenian"} {
		  set texlig2 [list "\\dj{}" "\\DJ{}" "\\ss{}"]
	      } else {
		  set texlig2 [list "\\ss{}"]
	      }
	    }
       1    { set texlig1 [list "{\\aa}" "{\\AA}" "{\\oe}" "{\\OE}" \
				"{\\ae}" "{\\AE}" "{\\o}" "{\\O}"]
	      if {$keyboard == "Slovenian"} {
		  set texlig2 [list "{\\dj}" "{\\DJ}" "{\\ss}"]
	      } else {
		  set texlig2 [list "{\\ss}"]
	      }
	    }
       2    { set texlig1 [list "\\aa$seplater" "\\AA$seplater" "\\oe$seplater" "\\OE$seplater" \
				"\\ae$seplater" "\\AE$seplater" "\\o$seplater" "\\O$seplater"]
	      if {$keyboard == "Slovenian"} {
		  set texlig2 [list "\\dj$seplater" "\\DJ$seplater" "\\ss$seplater"]
	      } else {
		  set texlig2 [list "\\ss$seplater"]
	      }
	    }
    }
    
    set TeX::_latexAccents [concat $texas $texcas $texes $texces $texis $texcis $texos $texcos $texus $texcus \
				   $texvar $texfix $texlig1 $texlig2 $texoth1 $texoth2 $texoth3]

    if {$TeXmodeVars(bindLatexAccents)} {
	foreach m ${TeX::_modeSeen} {
	    TeX::bindLatexAccents 1 $keyboard $m
	}
    }
}

proc TeX::replaceAccents {ww} {
    global TeX::_asciiAccents TeX::_latexAccents keyboard
    if { [isSelection] } {
	set position [getPos]
	set endselection [selEnd]
    } else {
	switch [askyesno "Convert the entire document?"] {
	    "yes" {
		    set position [minPos]
		    set endselection [maxPos]
		   }
	    "no" { return }
	}
    }
    set text [getText $position $endselection]
    
    set all ${TeX::_asciiAccents}

    set quote {"}  
    set seplater {\\\\sepsep//}
	  
    ############################################################
    #                                                          #														   #
    #	List of regular expressions		               #
    #	For  the reg exp is \`[ \t]*(a|{[ \t]*a[ \t]*})       #
    #   \c c needs the space but \c{c} does not                #
    #   \i may have trailing spaces                            #
    #                                                          #														   #\\\\aa$sep|{$ws\\\\aa$ws}
    ############################################################

    if {$ww == "0"} {      
	set ws "\[ \t\]*"
	set sp "\[ \t\]"
	if {[info tclversion] >= 8.0} {
	    set sep { *( |\{\}|\M)}
	} else {
	    set sep { *( |\{\}|\b)}
	} 
	set a [rexp a]
	set regas    [list "\\\\`$a" "\\\\'$a" "\\\\\\^$a"  "\\\\~$a"  "\\\\\\$quote$a"]
	set a [rexp A]
	set regcas   [list "\\\\`$a" "\\\\'$a" "\\\\\\^$a"  "\\\\~$a"  "\\\\\\$quote$a"]
	set e [rexp e]
	set reges    [list "\\\\`$e" "\\\\'$e" "\\\\\\^$e"  "\\\\\\$quote$e"]
	set e [rexp E]
	set regces   [list "\\\\`$e" "\\\\'$e" "\\\\\\^$e"  "\\\\\\$quote$e"]
	set i [rexpi "\\\\i" {[ 	]}]
	set regis    [list "\\\\`$i" "\\\\'$i" "\\\\\\^$i"  "\\\\$quote$i"]
	set i [rexp I]
	set regcis   [list "\\\\`$i" "\\\\'$i" "\\\\\\^$i"  "\\\\$quote$i"]
	set o [rexp o]
	set regos    [list "\\\\`$o" "\\\\'$o" "\\\\\\^$o"  "\\\\~$o"  "\\\\$quote$o"]
	set o [rexp O]
	set regcos   [list "\\\\`$o" "\\\\'$o" "\\\\\\^$o"  "\\\\~$o"  "\\\\$quote$o"]
	set u [rexp u]
	set regus    [list "\\\\`$u" "\\\\'$u" "\\\\\\^$u"  "\\\\$quote$u"]
	set u [rexp U]
	set regcus   [list "\\\\`$u" "\\\\'$u" "\\\\\\^$u"  "\\\\$quote$u"]
	set reglig1  [list "[rexpl aa]" "[rexpl AA]" "[rexpl oe]" "[rexpl OE]" \
			   "[rexpl ae]" "[rexpl AE]" "[rexpl o]" "[rexpl O]"]
	set regoth1  [list "\\\\dots$sep" "\\\\pounds$sep" "[rexpl S]" "[rexpl P]"]
	set regoth2  [list "({\\\\leavevmode\\\\raise.585ex\\\\hbox{\\\\b{a}}})" \
			   "({\\\\leavevmode\\\\raise.6ex\\\\hbox{\\\\b{o}}})" "(\\\\accent'27)"]
	if {$keyboard == "Slovenian"} {
		set regvar  [list "\\\\~[rexp n]" "\\\\~[rexp N]" "\\\\\'[rexp c]" "\\\\\'[rexp C]"]
		set regfix  [list "\\\\c[rexpc c {[ 	]}]" "\\\\c[rexpc C {[ 	]}]" \
				  "\\\\v[rexpc c {[ 	]}]" "\\\\v[rexpc s {[ 	]}]" \
				  "\\\\v[rexpc z {[ 	]}]" "\\\\v[rexpc C {[ 	]}]" \
				  "\\\\v[rexpc S {[ 	]}]" "\\\\v[rexpc Z {[ 	]}]"]
		set reglig2 [list "[rexpl dj]" "[rexpl DJ]" "[rexpl ss]"]
		set regoth3 [list "\\\\copyright$sep" "\\\\dag$sep" "(\\?`)" "(\\!`)"]
	} else {
		set regvar  [list "\\\\~[rexp n]" "\\\\~[rexp N]" "\\\\\\$quote[rexp y]" "\\\\\\$quote[rexp Y]"]
		set regfix  [list "\\\\c[rexpc c {[ 	]}]" "\\\\c[rexpc C {[ 	]}]" ]
		set reglig2 [list "[rexpl ss]"]
		set regoth3 [list "\\\\copyright$sep" "\\\\dag$sep" "(\\?`)" "(\\!`)" "\\\\ddag$sep"]
	}
	
	set regall   [concat $regas $regcas $reges $regces $regis $regcis $regos $regcos $regus $regcus \
			     $regvar $regfix $reglig1 $reglig2 $regoth1 $regoth2 $regoth3]
    }

    ############################################################

    set mark {\\\\//}
    set space {[ ]+}
    set len [string length $all]
    set ltxt [string length $text]
    set i 0
    while {$i < $len} {
	set c [string index $all $i]
	if {$ww == "1"} {
	    set s [lindex ${TeX::_latexAccents} $i]
	    regsub -all "$c" $text "$s" text 
	    if {$ltxt > 10000} {message "pair $c $s"}
	} else {
	    set s [lindex $regall $i]
	    regsub -all "(${s})(\\\\( ) *)?" $text "$mark$c\\4" text
	    if {$ltxt > 10000} {message "pair $s $c"}
	}
	incr i
    }
    if {$ww == "1"} {   
	regsub -all "$seplater$space" $text "\\\\ " text  
	regsub -all "${seplater}(\\W)" $text {\1} text
	regsub -all "$seplater" $text " " text
    } else {
	regsub -all "$mark" $text "" text
    }
    # workaround Alpha bug
    goto $position
    replaceText $position $endselection $text	
}

set "TeX_keyboards(Australian)" {
 {Ϡɾ}
 {136790qtoasg;'c}
 {ί}
 {78qoa'c/}
}
set "TeX_keyboards(Brasil)" {
 {Ͼ}
 {46qeuoasc.}
 {ήு}
 {1'qeuoa}
}
set "TeX_keyboards(British)" {
 {Ϡɾ}
 {16790qtoasg;'c}
 {ί}
 {78qoa'c/}
}
set "TeX_keyboards(Canadian - CSA)" {
 {Ϧ}
 {13qroasdg;}
 {ίɠ}
 {67-qoaslcm}
}
set "TeX_keyboards(Canadian - ISO)" {
 {Ϧ}
 {13qroasdg}
 {ίɠ}
 {67-qoaslxm}
}
set "TeX_keyboards(Canadian - French)" {
 {Ͽ}
 {13790qoasg;,.}
 {ίɂ}
 {67qoascm.}
}
set "TeX_keyboards(Danish)" {
 {ϻɍ}
 {147qetuoasgxcn}
 {ن΀}
 {+teyuo'cn}
}
set "TeX_keyboards(Dutch)" {
 {Ϡɾ}
 {136790qtoasg;'c}
 {ί}
 {78qoa'c/}
}
set "TeX_keyboards(Espaol - ISO)" {
 {Ͼ}
 {qwoajkcb}
 {ή}
 {4qwoa.}
}
set "TeX_keyboards(Finnish)" {
 {ϧ}
 {13toskc.}
 {μ}
 {16+tokc}
}
set "TeX_keyboards(Flemish)" {
 {੧}
 {!aetuoqcb;}
 {ٻ}
 {azeyuo,}
}
set "TeX_keyboards(French)" {
 {ϧ}
 {aobctq!u;}
 {΁}
 {aoz,u}
}
set "TeX_keyboards(French - numerical)" {
 {ϧ}
 {aobctq!u;}
 {΁}
 {aoz,u}
}
set "TeX_keyboards(German)" {
 {Ͼ}
 {13toaghjc.}
 {ή}
 {4oayc}
}
set "TeX_keyboards(Italian)" {
 {ό}
 {toasjkc,;}
 {}
 {iuo$af}
}
set "TeX_keyboards(Italian - Pro)" {
 {όɩ}
 {yiaosjk,cx'}
 {΁}
 {yiao'x}
}
set "TeX_keyboards(Norwegian)" {
 {ϧ}
 {136etuoskc.}
 {μ}
 {16+qetuokc}
}
set "TeX_keyboards(Spanish)" {
 {Ͼ}
 {89qetoask;c.}
 {ή}
 {qeoaf}
}
set "TeX_keyboards(Swedish)" {
 {ϧ}
 {136etuoskc.}
 {μ}
 {16+aetuokc}
}
set "TeX_keyboards(Swiss French)" {
 {Ϡ}
 {4'qtuioashj$c.}
 {ί}
 {qoaf}
}
set "TeX_keyboards(Swiss German)" {
 {Ϡ}
 {4'qtuioashj$c.}
 {ί}
 {qoaf}
}
set "TeX_keyboards(U.S.)" {
 {Ϡɾ}
 {136790qtoasg;'c}
 {ί}
 {78qoa'c/}
}
set "TeX_keyboards(Slovenian)" {
 {Ϡٍ}
 {136790qtoasgc}
 {ί}
 {8qoac-}
}
set "TeX_keyboards(Croatian)" {
 {Ϡɾ}
 {136790qtoasg;'c}
 {ί}
 {78qoa'c/}
}
set "TeX_keyboards(Roman - JIS)" {
 {Ϡɾ}
 {136790qtoasg;:c}
 {ί}
 {78qoa:c/}
}

proc  TeX::switchKeyboard {args} {
    global TeXmodeVars TeX::_modeSeen oldkeyboard keyboard
    if {$oldkeyboard == "Slovenian" || $keyboard == "Slovenian"} {
	TeX::setLatexAccents $oldkeyboard
	return
    }
    if {$TeXmodeVars(bindLatexAccents)} {
	foreach m ${TeX::_modeSeen} {
	    TeX::bindOptionLatexAccents 0 $oldkeyboard $m
	    TeX::bindOptionLatexAccents 1 $keyboard $m
	}
    }
}

proc TeX::bindLatexAccents {flag {keys ""} {locmod ""}} {
    global TeX::_asciiAccents TeX::_latexAccents keyboard mode
        
    if {$flag == 0} {
	set func "unascii"
    } else {
	set func "ascii"
    } 
    if {$locmod == ""} {
	if {$mode == ""} { return}
	set locmod $mode
    } 
#     message "start ${func}ng for mode $locmod"
    set all ${TeX::_asciiAccents}
    set seplater {\\\\\\\\sepsep//}
    set len [string length $all]
    set i 0
    while {$i < $len} {
	set c [string index $all $i]
	set s [lindex ${TeX::_latexAccents} $i]
	regsub "$seplater" $s {\\/} s 
	set s [quote::Insert $s]
	set key "0x[format %x [text::Ascii $c]]"
	set pro "{TeX::insertAccent \"$s\" \"$c\"}"
	catch "$func $key $pro $locmod"
	incr i
    }
    if {$keys == ""} {
	TeX::bindOptionLatexAccents $flag $keyboard $locmod
    } else {
	TeX::bindOptionLatexAccents $flag $keys $locmod
    } 
#     message "finished ${func}ng for mode $locmod"
}

proc TeX::bindOptionLatexAccents {flag {keys ""} {locmod ""}} {
    global TeX::_asciiAccents TeX::_latexAccents TeX::_globalBindings TeX_keyboards keyboard mode
    if {$flag == 0} {
	set func2 "unbind"
    } else {
	set func2 "Bind"
    } 
    if {$locmod == ""} {
	set locmod $mode
    } 
    set all ${TeX::_asciiAccents}
    set seplater {\\\\\\\\sepsep//}
    if {$keys == ""} {
	set keys $keyboard
    } 
    set opt $TeX_keyboards($keys)
    set lopt [lindex $opt 0]
    set copt [lindex $opt 1]
    set len [string length $lopt]
    set i 0
    while {$i < $len} {
	set l [string index $lopt $i]
	set j [string first $l $all] 
	set s [lindex ${TeX::_latexAccents} $j]
	regsub "$seplater" $s {\\/} s 
	set s [quote::Insert $s]
	set key "'[string index $copt $i]'"
	set j $i
	incr i
	if {$func2 == "Bind" && [regexp "[quote::Regfind $key]\[ \t\]+<o>" ${TeX::_globalBindings}]} { continue } 
	set pro "{TeX::insertAccent \"$s\" \"$l\"}"
	if {[catch "$func2 $key <o> $pro $locmod"]} {
	    set key "'\\[string index $copt $j]'"
	    catch "$func2 $key <o> $pro $locmod"
	}
    }
    set lopt [lindex $opt 2]
    set copt [lindex $opt 3]
    set len [string length $lopt]
    set i 0
    while {$i < $len} {
	set l [string index $lopt $i]
	set j [string first $l $all] 
	set s [lindex ${TeX::_latexAccents} $j]
	regsub "$seplater" $s {\\/} s 
	set s [quote::Insert $s]
	set key "'[string index $copt $i]'"
	set j $i
	incr i
	if {$func2 == "Bind" && [regexp "[quote::Regfind $key]\[ \t\]+(<so>|<so>)" ${TeX::_globalBindings}]} { continue } 
	set pro "{TeX::insertAccent \"$s\" \"$l\"}"
	if {[catch "$func2 $key <os> $pro $locmod"]} {
	    set key "'\\[string index $copt $j]'"
	    catch "$func2 $key <os> $pro $locmod"
	}
    }
    if {$keys == "Slovenian"} {
	set j [string first "" $all] 
	set s [lindex ${TeX::_latexAccents} $j]
	regsub "$seplater" $s {\\/} s 
	set s [quote::Insert $s]
	set pro "{TeX::insertAccent \"$s\" \"\"}"
	catch "$func2 0x29 <o> $pro $locmod"
	set j [string first "" $all] 
	set s [lindex ${TeX::_latexAccents} $j]
	regsub "$seplater" $s {\\/} s 
	set s [quote::Insert $s]
	set pro "{TeX::insertAccent \"$s\" \"\"}"
	catch "$func2 0x29 <os> $pro $locmod"
	set j [string first "" $all] 
	set s [lindex ${TeX::_latexAccents} $j]
	regsub "$seplater" $s {\\/} s 
	set s [quote::Insert $s]
	set pro "{TeX::insertAccent \"$s\" \"\"}"
	catch "$func2 0x27 <os> $pro $locmod"
    } elseif {$keys == "Spanish"} {
	set j [string first "" $all] 
	set s [lindex ${TeX::_latexAccents} $j]
	regsub "$seplater" $s {\\/} s 
	set s [quote::Insert $s]
	set pro "{TeX::insertAccent \"$s\" \"\"}"
	catch "$func2 0x21 <o> $pro $locmod"
    }
}

proc  TeX::insertAccent {str1 {str2 ""}} {
    global TeXmodeVars
    if {[regsub {\\/} $str1 "" str1]} {
	set next [selEnd]
	if {[pos::compare $next <= [maxPos]]} {
	    set char [lookAt $next]
	    if {$char == " "} {
		set str1 "$str1\\"
	    } elseif {[regsub "\\w" $char "" tmp]} {
		set str1 "$str1 "
	    } 
	}
    }
    if {$str2 == ""} {
	insertObject $str1
	return
    }
    set pos [getPos]
    set beg [lineStart $pos]
    set txt [getText $beg $pos]
    if {$TeXmodeVars(accentsInComments) && [regexp {^%|[^\\]%} $txt]} {
	if {$str2 == ""} {
	    typeText $str2
	} else {
	    insertObject $str2
	}
    } else {
	insertObject $str1
    }
}

proc  TeX::addSmartEscape {flag} {
    global TeX::smartEscape TeXmodeVars	mode

    if {$TeXmodeVars(accentSmartEscape) || $flag == 0} {
	if {[info exists {TeX::smartEscape}]} {
	    set s [lindex ${TeX::smartEscape} 0]
	    set s [lindex $s 1]
	} else {
	    latexSmart.tcl
	    set s " "
	}
	if {$s != {[?!]`$} || $flag == 0} {
	    lappend TeXEscape {0 [?!]`$ {}}
	    lappend TeXEscape [list 0 "\\\\\['^`\"~\][rexp {[a-zA-Z]}]$"	{}]
	    lappend TeXEscape [list 0 "\\\\\['^`\"~\][rexpi {\\i} {[ 	]}]$"	{}]
	    lappend TeXEscape [list 0 "[rexpl {(aa|ae|oe?|dj|AA|AE|OE?|DJ|S|P|ss)}]$" {}]
	    lappend TeXEscape [list 0 "\\\\\[cv\][rexpc {[a-zA-Z]} {[ 	]}]$" {}]
	    lappend TeXEscape [list 0 "\\\\\[v\][rexpv {\\i} {[ 	]}]$" {}]
	    lappend TeXEscape {0 \{\\\\leavevmode\\\\raise\\.(6|585)ex\\\\hbox\{\\\\b\{[ao]\}\}\}$ {}}
	    lappend TeXEscape {0 \\\\(pounds|dots|accent'27|copyright|d?dag)(\{\}|\ )?$ {}}
	    if {$flag == 0} {
		set TeX::smartEscape [lremove -l ${TeX::smartEscape} $TeXEscape]
		return
	    } else {
		foreach l ${TeX::smartEscape} {
		    lappend TeXEscape $l
		}
		set TeX::smartEscape $TeXEscape
	    }
	}
    }
}

