## -*-Tcl-*-  (nowrap)
 # ==========================================================================
 #  Macros - an extension package for Alpha
 # 
 #  FILE: "macros.tcl"
 #                                    created: 07/27/1997 {11:39:46 pm} 
 #                                last update: 11/29/2001 {09:36:07 AM} 
 #  Description: 
 #  
 #  Support for user created macros.
 #                                 
 #  Author:  ??
 #  
 #  Includes contributions from Craig Barton Upright.
 #  
 # --------------------------------------------------------------------------
 #  
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #   Redistributions of source code must retain the above copyright
 #    notice, this list of conditions and the following disclaimer.
 # 
 #   Redistributions in binary form must reproduce the above copyright
 #    notice, this list of conditions and the following disclaimer in the
 #    documentation and/or other materials provided with the distribution.
 # 
 #   Neither the name of Alpha/Alphatk nor the names of its contributors may
 #    be used to endorse or promote products derived from this software
 #    without specific prior written permission.
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # DAMAGE.
 # ==========================================================================
 ## 

# ===========================================================================
# 
#  Initialization of package  # 
# 

alpha::feature macros 1.3.1 "global-only" {
    # Initialization script.
    # Make sure that this is compatible with older versions.
    foreach macro [array names macroArr] {
        if {![info exists macroNames($macro)]} {set macroNames($macro) 1}
    }
    # Move any old binding prefs into the macromodeVars array.
    prefs::renameOld startRecording   macromodeVars(startRecording)
    prefs::renameOld endRecording     macromodeVars(endRecording)
    prefs::renameOld executeLastMacro macromodeVars(executeLastMacro)
    # Key binding for the "Start Recording" menu item.
    newPref menubinding startRecording   "/q" macros {macro::rebuildMenu}
    # Key binding for the "End Recording" menu item.
    newPref menubinding endRecording     "/r" macros {macro::rebuildMenu}
    # Key binding for the "Execute Last Macro" menu item.
    newPref menubinding executeLastMacro "/s" macros {macro::rebuildMenu}

    namespace eval macro {}

    # Register the build procs for the menus.
    menu::buildProc macroRecording  macro::buildMacrosMenu macro::postEvalRecording
    menu::buildProc macroUtils      macro::buildUtilsMenu  macro::postEvalUtils
    menu::buildProc savedMacros     macro::buildSavedMenu
    # Define these menus here, so that this file does not need to be
    # sourced in its entirety on startup.
    proc macro::buildMacrosMenu {} {
        global macrosmodeVars 
        set menuList [list                                \
          "$macrosmodeVars(startRecording)startRecording" \
	  "$macrosmodeVars(endRecording)endRecording"     \
          "$macrosmodeVars(executeLastMacro)execute"      \
	  "(-)" "setBindings"                            ]
        Menu -n macroRecording -p macro::menuProc $menuList
    }
    proc macro::buildUtilsMenu {} {
        set menuList [list                             \
          "displayLastMacro" "saveLastMacro" "(-)"    \
          "listSavedMacros"  "hideSavedMacros"       \
          "showSavedMacros"  "(-)"                    \
          "editSavedMacro"   "deleteSavedMacro"      \
          "renameSavedMacro" "(-)"                    \
          "macrosHelp" "macrosTutorial"                ]
        return [list build $menuList macro::menuProc {}]
    }
    proc macro::buildSavedMenu {} {
	global macroNames macroArr macroKeys 
	set shownMacros ""
	foreach macro [lsort [array names macroNames]] {
	    if {$macroNames($macro)} {
		lappend shownMacros $macro
		eval $macroArr($macro)
	    } 
	}
	if {[llength $shownMacros]} {
	    set menuList {"assignBindings" "unsetBindings" "(-)"}
	    foreach macro $shownMacros {
		if {[info exists macroKeys($macro)]} {
		    lappend menuList $macroKeys($macro)$macro
		} else {
		    lappend menuList $macro
		} 
	    }
	} else {
	    set menuList "(noMacrosAvailable"
	}
	return [list build $menuList macro::menuProc {}]
    }
    proc macro::postEvalRecording {args} {
	set dim1 [expr {[macro::recording] < 1} ? 1 : 0]
	set dim2 [expr {$dim1} ? 0 : 1]
	set pat  {^proc +macroName[\{\}\r\n ]+$}
	set macro [macro::current]
	set dim4 [expr {![string length $macro] || [regexp $pat $macro]} ? 0 : 1]
	set dim3 [expr {$dim1 && $dim4} ? 1 : 0]
	enableMenuItem macroRecording startRecording $dim1
	enableMenuItem macroRecording endRecording   $dim2
	enableMenuItem macroRecording execute        $dim3
	# We can do these here because the macroUtils menu will have already
	# been inserted.
	enableMenuItem macroUtils displayLastMacro   $dim4
	enableMenuItem macroUtils saveLastMacro     $dim4
    }
    proc macro::postEvalUtils {} {
	global macroNames macroArr
	set allMacros    [array names macroNames]
	set hiddenMacros ""
	set shownMacros  ""
	foreach macro $allMacros {
	    if {!$macroNames($macro)} {
		lappend hiddenMacros $macro
	    } else {
		lappend shownMacros $macro
	    }
	}
	set dim1 [expr {[llength $allMacros   ]} ? 1 : 0]
	set dim2 [expr {[llength $shownMacros ]} ? 1 : 0]
	set dim3 [expr {[llength $hiddenMacros]} ? 1 : 0]
	enableMenuItem macroUtils listSavedMacros  $dim1
	enableMenuItem macroUtils editSavedMacro   $dim1
	enableMenuItem macroUtils deleteSavedMacro $dim1
	enableMenuItem macroUtils renameSavedMacro $dim1
	enableMenuItem macroUtils hideSavedMacros  $dim2
	enableMenuItem macroUtils showSavedMacros  $dim3
	set pat  "^proc +macroName\[\{\}\r\n \]+$"
	set macro [macro::current]
	if {[string length $macro] && ![regexp $pat $macro]} {
	    set dim4 1
	} else {
	    set dim4 0
	}
	enableMenuItem macroUtils displayLastMacro  $dim4
	enableMenuItem macroUtils saveLastMacro    $dim4
    }
    # This will automatically dim menu items when macros are started/stopped.
    trace variable macro::recording w macro::postEvalRecording
} {
    # Activation script: insert the menus.
    menu::insert   Utils items   end "(-) "
    menu::insert   Utils submenu end macroRecording
    menu::insert   Utils submenu end macroUtils
    menu::insert   Utils submenu end savedMacros
} {
    # Deactivation script: remove the menus.
    menu::uninsert Utils items   end "(-) " 
    menu::uninsert Utils submenu end macroRecording
    menu::uninsert Utils submenu end macroUtils
    menu::uninsert Utils submenu end savedMacros
} help {
    file "Macros Help"
}

proc macros.tcl {} {}

namespace eval macro {}

# ===========================================================================
# 
#  Menu Support  #
# 

proc macro::rebuildMenu {args} {

    set macrosList {startRecording endRecording executeLastMacro}
    if {[lcontains macrosList $args]} {
        # Called from the Macros Packages dialog.
        set args macroRecording 
    } elseif {![llength $args]} {
        set args [list macroUtils savedMacros]
    }
    foreach menuName $args {
        menu::buildOne $menuName
        message "The \"$menuName\" menu has been rebuilt."
    } 
}

proc macro::menuProc {menuName item} {
    
    switch $item {
        
        "displayLastMacro" -
        "saveLastMacro" -
        "listSavedMacros" -
        "hideSavedMacros" -
        "showSavedMacros" -
        "editSavedMacro" -
        "deleteSavedMacro" -
	"startRecording" -
	"endRecording" -
	"execute" -
	"assignBindings" - 
	"unsetBindings" -
        "renameSavedMacro"      { macro::$item                          }
        "macrosHelp"            { package::helpFile "macros"            }
        "macrosTutorial"        { help::openExample "Macros Example"    }
	"setBindings"           { dialog::pkg_options "macros"          }
        default                 { eval $item                            }
    }
}
    
# ===========================================================================
# 
#  Saved Macro Utilities  #
# 

proc macro::displayLastMacro {} {
    
    global mode
    
    # Do we have a valid macro yet?
    set macro [macro::current]
    if {![string length $macro] || [regexp {^proc +macroName[\{\}\r\n ]+$} $macro]} {
	# This should have been dimmed.
	macro::postEvalRecording
	status::errorMsg "Cancelled -- there is no 'last' macro to be displayed."
    }
    if {$mode != ""} {set m $mode} else {set m "Tcl"}
    new -n "* Last Macro *" -text $macro -m $m
    goto [minPos] ; shrinkWindow 0 ; winReadOnly
}

proc macro::saveLastMacro {} {
    
    global macroArr macroNames
    
    # Do we have a valid macro yet?
    set macro [macro::current]
    if {![string length $macro] || [regexp {^proc +macroName[\{\}\r\n ]+$} $macro]} {
	# This should have been dimmed.
	macro::postEvalRecording
        status::errorMsg "Cancelled -- there is no 'last' macro to be saved."
    }
    set name [macro::verifyName [prompt "New macro name " ""]]
    regsub macroName $macro $name macro
    regsub -all "\r" $macro ";\r" macro
    eval $macro
    set macroArr($name) $macro
    set macroNames($name) 1
    prefs::modified macroArr macroNames
    macro::rebuildMenu
    message "The macro \"$name\" has been saved and inserted into the \"Saved Macros\" menu."
}

proc macro::listSavedMacros {{macroList ""}} {
    
    global macroArr win::Active
    
    if {$macroList == ""} {
        set allMacros [lsort [array names macroArr]]
        set macros [listpick -l -p "All saved macros include " $allMacros]
    }
    if {![llength [set win::Active]]} {new -m Tcl}
    foreach macro $macros {
        regsub -all ";*(\r|\n)" $macroArr($macro) "\r" text
        insertText "\r"
        goto [lineStart [getPos]]
        insertText $text
    } 
}

proc macro::editSavedMacro {} {

    global macroArr
    
    set savedMacros [lsort [array names macroArr]]
    if {![llength $savedMacros]} {
        message "Cancelled -- there are no saved macros to edit." ; return
    } 
    foreach w [winNames] {
        # Find any open "* Last Macro *" windows.
        if {[regexp "\\* Edit Macro \\*" $w]} {
            bringToFront $w
            macro::saveEditedMacro
            return
        }
    }
    # Open a dialog explaining how this works, with a pop-up list
    # of all saved macros to choose.
    set y 10
    set title "Editing Saved Macros"
    if {[info tclversion] < 8.0} {
        set macroDialog [dialog::text $title 10 y 90]
    } else {
        set macroDialog [list -T $title]
    }
    eval lappend macroDialog [dialog::text \
      "Select a macro to edit from the pop-up menu below. \
      This will open a new window containing all of the procedures \
      contained in the macro.  When you are finished editing the macro, \
      select the \"Macro Utils --> Edit Saved Macro\" menu item again \
      to save it in your preferences folder." \
      10 y 60]
    eval lappend macroDialog [dialog::text "" 10 y 60]
    eval lappend macroDialog [dialog::menu 10 y $savedMacros]
    incr y -20
    eval lappend macroDialog [dialog::button "Edit" 360 y "Cancel" 265 y]
    
    set result [eval dialog -w 440 -h $y $macroDialog]
    if {[lindex $result 2]} {message "Cancelled." ; return}
    set macro  [lindex $result 0]

    regsub -all ";+(\r|\n)" $macroArr($macro) "\r" text
    new -n "* Edit Macro *" -text [string trim $text] -m "Tcl" -shell 1
    goto [minPos] ; insertText "\r"    
    catch {procs::reformatEnclosing [getPos]}
    shrinkWindow 0
    message "Edit the \"$macro\" macro, then select \"Edit Saved Macro\" again."
}

proc macro::saveEditedMacro {} {
    
    global macroArr 
    
    # A saved macro has already been edited, its window is in front,
    # and now it needs to be saved.
    set foundProc 0
    set pat {^proc *([-+a-zA-Z0-9:]+)}
    if {[catch {search -s -f 1 -r 1 $pat [minPos]} match]} {set foundProc 1}
    if {![isSelection]} {
        if {$foundProc} {
            dialog::alert "Could not find the macro's name.\r \
              Please highlight the name and try again."
            message "Cancelled -- could not find the macro's name." ; return
        } else {
            goto [lindex $match 1] ; backwardWord ; hiliteWord 
        } 
    } 
    regsub -all { } [getSelect] {} macro
    if {![dialog::yesno "Save this macro as \"$macro\" ?"]} {
        message "Cancelled." ; return
    } 
    # Select the entire window, trim it, make sure that each line ends with
    # a semi-colon, set the macroArr array, evaluate the procedure, and
    # then close the window.
    selectAll
    regsub -all ";* *(\r|\n|$)" [string trim [getSelect]] ";\r" text
    set  macroArr($macro) $text
    eval $macroArr($macro)
    prefs::modified macroArr 
    killWindow
    message "The edited  macro \"$macro\" has been saved."
}

proc macro::deleteSavedMacro {{macroList ""}} {
    
    global macroArr macroNames macroKeys

    set savedMacros [lsort [array names macroNames]]
    if {![llength $savedMacros]} {
        message "Cancelled -- there are no saved macros to delete." ; return
    } 
    if {$macroList == ""} {
        set macroList [listpick -l -p "Delete which macros?" $savedMacros]
    }
    foreach macro $macroList {
        unset macroArr($macro) macroNames($macro)
        catch {rename $macro ""}
        if {[info exists macroKeys($macro)]} {unset macroKeys($macro)} 
    } 
    prefs::modified macroArr macroNames macroKeys 
    macro::rebuildMenu
    if {[llength $macroList] == 1} {
        message "The macro $macroList has been deleted."
    } elseif {[llength $macroList] > 1} {
        message "The macros $macroList have been deleted."
    } 
}

proc macro::renameSavedMacro {{oldMacro ""}} {

    global macroArr macroNames macroKeys

    set allMacros [lsort [array names macroNames]]
    if {![llength $allMacros]} {
        message "Cancelled -- there are no saved macros to delete."
        return
    } 
    if {$oldMacro == ""} {
        set oldMacro [listpick -p "Rename which macro?" $allMacros]
    }
    set title "Rename \"$oldMacro\" to "
    set newMacro [macro::verifyName [prompt $title $oldMacro]]
    set macroArr($newMacro)   $macroArr($oldMacro)
    set macroNames($newMacro) $macroNames($oldMacro)
    unset macroArr($oldMacro) macroNames($oldMacro)
    # Now switch the binding if an old one exists.
    if {[info exists macroKeys($oldMacro)]} {
        set   macroKeys($newMacro) $macroKeys($oldMacro)
        unset macroKeys($oldMacro)
    } 
    prefs::modified macroArr macroNames macroKeys 
    macro::rebuildMenu
    catch {rename $oldMacro ""}
    message "The macro \"$oldMacro\" has been renamed to \"$newMacro\"."
}

proc macro::verifyName {name} {
    
    regsub -all " " $name "" name
    if {[info commands $name] != "" && ![dialog::yesno \
      "\"$name\" is already used by another defined procedure. \
      Are you sure that you want to continue? \
      This will remove the previously defined procedure "]} {
        message "Cancelled."
        error "New macro name is already used by another procedure."
    } else {
        return $name
    }
}

# ===========================================================================
# 
#  Saved Macro Menu  #
# 

proc macro::hideSavedMacros {{macroList ""}} {
    
    global macroNames
    
    set shownMacros  ""
    foreach macro [lsort [array names macroNames]] {
        if {$macroNames($macro)} {lappend shownMacros $macro} 
    }
    if {![llength $shownMacros]} {
        message "Cancelled -- all saved macros are currently hidden."
        return
    }
    if {$macroList == ""} {
        set title "Choose some macros to hide"
        set macroList [listpick -l -p $title $shownMacros]
    } 
    foreach macro $macroList {set macroNames($macro) 0}
    prefs::modified macroNames 
    macro::rebuildMenu
    if {[llength $macroList] == 1} {
        message "The macro $macroList has been removed from the \"Saved Macros\" menu."
    } elseif {[llength $macroList] > 1} {
        message "The macros $macroList have been removed from the \"Saved Macros\" menu."
    } 
}

proc macro::showSavedMacros {{macroList ""}} {
    
    global macroNames
    
    set hiddenMacros ""
    foreach macro [lsort [array names macroNames]] {
        if {!$macroNames($macro)} {lappend hiddenMacros $macro} 
    }
    if {![llength $hiddenMacros]} {
        message "Cancelled -- there are no hidden macros to show."
        return
    }
    if {$macroList == ""} {
        set title "Choose some macros to show"
        set macroList [listpick -l -p $title $hiddenMacros]
    } 
    foreach macro $macroList {set macroNames($macro) 1}
    prefs::modified macroNames 
    macro::rebuildMenu
    if {[llength $macroList] == 1} {
        message "The macro $macroList is now in the \"Saved Macros\" menu."
    } elseif {[llength $macroList] > 1} {
        message "The macros $macroList are now in the \"Saved Macros\" menu."
    } 
}

proc macro::assignBindings {} {
    
    global macroArr macroKeys macroNames 
    
    set shownMacros ""
    foreach macro [lsort [array names macroNames]] {
        if {$macroNames($macro)} {lappend shownMacros $macro} 
    }
    if {![llength $shownMacros]} {
        message "Cancelled -- there are no shown macros to assign"
        return
    } 
    foreach macro $shownMacros {
        if {[info exists macroKeys($macro)]} {
            set macroBindings($macro) $macroKeys($macro)
        } else {
            set macroBindings($macro) ""
        } 
    } 
    catch {dialog::arrayBindings "Saved Macro Key Bindings " macroBindings 1}

    foreach macro $shownMacros {
        if {[info exists macroBindings($macro)]} {
            set macroKeys($macro) $macroBindings($macro)
        } else {
            catch {unset macroKeys($macro)}
        } 
    } 
    prefs::modified macroKeys 
    macro::rebuildMenu savedMacros
    message "New bindings have been assigned, and appear in the \"Saved Macros\" menu."
}

proc macro::unsetBindings {{macroList ""}} {
    
    global macroKeys 
    
    set macrosWithKeys [lsort [array names macroKeys]]
    if {![llength $macrosWithKeys]} {
        message "Cancelled -- there are no assigned bindings to unset."
        return
    } 
    if {$macroList == ""} {
        set title "Choose some bindings to unset"
        set macroList [listpick -l -p $title $macrosWithKeys]
    } 
    foreach macro $macroList {catch {unset macroKeys($macro)}} 
    prefs::modified macroKeys 
    macro::rebuildMenu savedMacros
    if {[llength $macroList] == 1} {
        message "The binding for $macroList has been unset."
    } elseif {[llength $macroList] > 1} {
        message "The bindings for $macroList have been unset."
    } 
}

# ===========================================================================
# 
#  version history  #
# 
#  modified by  vers#  reason
#  -------- --- ------ -----------
#     ??    ??  0.1
#     ??    ??  0.2
#  02/26/01 cbu 1.0    Added support for assigning menu binding codes for
#                        user defined macros.
#                      Split the single menu into three, one for recording,
#                        one for utilities, one for saved macros.
#                      Added macro::renameSavedMacro menu item.
#                      Added ability to hide/show saved macros.
#                      Changed "Dump Saved Macro" to "List Saved Macros".
#                      All listpicks now take multiple lists.
#                      Added "Macros Help/Tutorial" menu items.
#  03/01/01 cbu 1.1    Added "Display Last Macro" menu item.
#                      Saved macros can now be easily edited by the user.
#  08/29/01 vmd 1.2    Renamed core macros procs, see coreFixes.tcl.
#  10/16/01 cbu 1.3    Using 'posteval' rather than trying to dim items as
#                        menus are being built.  (Much more robust, esp wrt
#                        Alphatk, where it's harder to both dim and have
#                        keyboard bindings embedded in menus.  Plus, enabling
#                        menu items is less intensive than rebuilding menus.)
#                      Macro keys can be set using menu item.
# 

# ===========================================================================
# 
# .
