## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "odbEditor.tcl"
 #                                    created: 5/2/01 {10:04:08 PM} 
 #                                last update: 09/11/2001 {13:37:36 PM} 
 #  Author: Jonathan Guyer
 #  E-mail: jguyer@his.com
 #    mail: Alpha Cabal
 #     www: http://www.his.com/jguyer/
 #  
 # ========================================================================
 # Copyright (c) 2001 Jonathan Guyer
 # All rights reserved.
 # 
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #      * Redistributions of source code must retain the above copyright
 #      notice, this list of conditions and the following disclaimer.
 # 
 #      * Redistributions in binary form must reproduce the above copyright
 #      notice, this list of conditions and the following disclaimer in the
 #      documentation and/or other materials provided with the distribution.
 # 
 #      * Neither the name of Alpha Cabal nor the names of its
 #      contributors may be used to endorse or promote products derived from
 #      this software without specific prior written permission.
 # 
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # DAMAGE.
 # ========================================================================
 #  Description: 
 # 
 # 	This package allows Alpha to act as an external editor for
 # 	applications that support the Bare Bones' ODB Editor suite
 # 	<http://www.barebones.com/support/developer/odbsuite.html>.
 # 	
 #  History
 # 
 #  modified   by  rev reason
 #  ---------- --- --- -----------
 #  2001-05-02 JEG 1.0 original
 # ###################################################################
 ##

alpha::feature ODBEditor 1.0b1 global {} {
    hook::register aevtodocHook odb::aevtodocHook 
    hook::register saveHook odb::modifiedHook
    hook::register saveasHook odb::modifiedHook
    hook::register closeHook odb::closeHook
} {
    hook::unregister aevtodocHook odb::aevtodocHook 
    hook::unregister saveHook odb::modifiedHook
    hook::unregister saveasHook odb::modifiedHook
    hook::unregister closeHook odb::closeHook
} maintainer {
    "Jon Guyer" <jguyer@his.com> <http://www.his.com/jguyer/>
} help {
    This package allows Alpha to act as an external editor for
    applications that support Bare Bones' ODB Editor suite
    <http://www.barebones.com/support/developer/odbsuite.html>.
} requirements {
    if {!(($tcl_platform(platform) == "macintosh")||($tcl_platform(platform) == "unix" && $tcl_platform(os) == "Darwin"))} {
	error "The ODB Editor suite is only supported on the Macintosh"
    }
    alpha::package require Alpha 8.0a3
    alpha::package require aeom 1.0a3
}


namespace eval odb {}

## 
 # -------------------------------------------------------------------------
 # 
 # "odb::aevtodocHook" --
 # 
 #  aevt/odoc AppleEvent hook for files opened via the ODB Editor suite.
 #  
 #  If the file is ODB Edited, then the file sender parameter 'FSnd' will
 #  always appear.
 #  
 #  The sender token 'FTok' and custom path 'Burl' parameters are optional.
 #  'FTok' is simply stored to resend to the file sender. 'Burl' is interpreted
 #  as well as can be to display the file properly.
 #  
 # 
 # Argument     Default In/Out Description
 # ------------ ------- ------ ---------------------------------------------
 # win                    In   full name of window that was opened by this event
 # index                  In   non-negative index if file was opened as one of a list
 # theAppleEvent          In   the AppleEvent descriptor that triggered this hook
 # theReplyAE           In/Out the reply AppleEvent that will be sent back to the caller
 # 
 # Results:
 #  1 if this was an ODB Edited file
 #  0 otherwise
 # 
 # Side effects:
 #  tracking information is stored so that proper events can be sent back
 #  to the server when the file is modified or closed. Also, if the file is
 #  edited via FTP, proper information is stored so Alpha treats it properly.
 # -------------------------------------------------------------------------
 ##
proc odb::aevtodocHook {win index theAppleEvent theReplyAE} {
    if {![catch {tclAE::getKeyData $theAppleEvent FSnd} sender]} {
	global odbedited fetched
	
	# sender token is optional
	set token ""
	catch {
	    set token [tclAE::getKeyDesc $theAppleEvent FTok]
	    if {$index >= 0} {
		# original file was opened from a list of files
		# extract the corresponding token
		set temp [tclAE::getNthDesc $token $index]
		tclAE::disposeDesc $token
		set token $temp
	    }
	}
	
	# custom path is optional
	catch {
	    set customPath [tclAE::getKeyData $theAppleEvent Burl]
	    if {$index >= 0} {
		# original file was opened from a list of files
		# extract the corresponding custom path
		set customPath [lindex $customPath $index]
	    }
	
	    # determine if the custom path is a url (remote FTP editing)
	    set url [url::parse $customPath]
	    if {[lindex $url 0] == "ftp"} {
		# custom path is an ftp url, so make sure Alpha treats
		# it accordingly
		alpha::package require ftpMenu
		url::parseFtp [lindex $url 1] a
		set fetched($win) [list $a(host) $a(path) $a(user) $a(pass) "ftp"]
	    }
	}
	
	# store ODB Edit information for later use
	set odbedited($win) [list $sender $token]
	
	return 1
    } else {
	# this file was not ODB Edited
	return 0
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "odb::modifiedHook" --
 # 
 #  save(as) hook for ODB Edited files.
 # 
 # Argument     Default In/Out Description
 # ------------ ------- ------ ---------------------------------------------
 # oldName                In   the original path of the file
 # newName        ""    In/Out the new path of the file (if saveAs invoked)
 # 
 # Results:
 #  1 if this was an ODB Edited file
 #  0 otherwise
 # 
 # Side effects:
 #  sender is notified of file change
 # -------------------------------------------------------------------------
 ##
proc odb::modifiedHook {oldName {newName ""}} {
    global odbedited
    
    if {[info exists odbedited($oldName)]} {
	set sender [lindex [set odbedited($oldName)] 0]
	set token [lindex [set odbedited($oldName)] 1]
	
	# build file-modified event
	set event [list tclAE::build::throw '${sender}' R*ch FMod]
	lappend event ---- [tclAE::build::alis $oldName]
	# attach token (if any)
	if {[string length $token] > 0} {
	    lappend event Tokn $token
	} 
	# attach new file name (if any)
	if {[string length $newName] > 0} {
	    lappend event New? [tclAE::build::alis $newName]
	    set odbedited($newName) [set odbedited($oldName)]
	    array unset odbedited($oldName)
	}
	
	eval $event
	
	return 1
    } else {
	# this file was not ODB Edited
	return 0
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "odb::closeHook" --
 # 
 #  close hook for ODB Edited files.
 # 
 # Argument     Default In/Out Description
 # ------------ ------- ------ ---------------------------------------------
 # name                   In   the path of the file
 # 
 # Results:
 #  1 if this was an ODB Edited file
 #  0 otherwise
 # 
 # Side effects:
 #  sender is notified of file closing
 # -------------------------------------------------------------------------
 ##
proc odb::closeHook {name} {
    global odbedited
    
    if {[info exists odbedited($name)]} {
	set sender [lindex [set odbedited($name)] 0]
	set token [lindex [set odbedited($name)] 1]
	
	# stored file info is no longer needed
	array unset odbedited($name)
	catch {array unset fetched($name)}
	
	# build file-closed event
	set event [list tclAE::build::throw '${sender}' R*ch FCls]
	lappend event ---- [tclAE::build::alis $name]
	
	# attach token (if any)
	if {[string length $token] > 0} {
	    lappend event Tokn $token
	} 
	
	eval $event

	if {[string length $token] > 0} {
	    tclAE::disposeDesc $token
	}
	return 1
    } else {
	# this file was not ODB Edited
	return 0
    }
}

proc odb::editFromFetch {} {
    global HOME ALPHA
    
    tclAE::build::throw 'FTCh' core setd \
      ---- [tclAE::build::propertyObject pETE] \
      data [tclAE::build::alis [file join $HOME $ALPHA]]
}

    