## -*-Tcl-*-
 # ==========================================================================
 # FILE: "recentDirs.tcl"
 #                                   created: 06/10/2001 {10:19:51 pm}
 #                               last update: 12/11/2001 {23:40:08 PM}
 #                               
 # Description:
 # 
 # Creates an 'Open Recent Dirs' subMenu to 'Files', including the
 # collection of all recently used volumes and directories.  Selecting any
 # menu item will open a standard OS Find File dialog, using the chosen
 # directory as the default.
 # 
 # Optionally inserts 'Save In' and 'Save A Copy In' menus as well.
 # 
 # Author: Craig Barton Upright
 # E-mail: <cupright@princeton.edu>
 #   mail: Princeton University
 #         2.N.1 Green Hall,  Princeton, New Jersey  08544
 #    www: <http://www.princeton.edu/~cupright>
 #
 # Many thanks to Darryl Smith and Bernard Desgraupes for several
 # suggestions and a lot of beta testing. 
 # --------------------------------------------------------------------------
 #  
 # Copyright (c) 2001  Craig Barton Upright
 # 
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #   Redistributions of source code must retain the above copyright
 #    notice, this list of conditions and the following disclaimer.
 # 
 #   Redistributions in binary form must reproduce the above copyright
 #    notice, this list of conditions and the following disclaimer in the
 #    documentation and/or other materials provided with the distribution.
 # 
 #   Neither the name of Alpha/Alphatk nor the names of its contributors may
 #    be used to endorse or promote products derived from this software
 #    without specific prior written permission.
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # DAMAGE.
 # 
 # ==========================================================================
 ##

alpha::feature recentDirsMenu 1.1.3 "global-only" {
    # Initialization script.
    namespace eval recent {}
    # We require 7.4a8 for the new 'menu::insert' proc.
    alpha::package require -loose AlphaTcl 7.4a8
    ensureset recent::Directories ""
    prefs::modified recent::Directories  recent::DirsMenuCache
    prefs::modified recent::DirsMarkList recent::DirsDimList 
    package::addPrefsDialog recentDirsMenu
    # Register build procs for each menu.
    menu::buildProc openRecentDir      recent::buildDirsMenu
    menu::buildProc saveACopyIn        recent::buildDirsMenu
    menu::buildProc saveIn             recent::buildDirsMenu
    menu::buildProc showDirInFinder    recent::buildDirsMenu
} {
    # Activation script.
    hook::register openHook            recent::pushDir
    hook::register saveasHook          recent::pushDir
    recent::insertDirsMenus            all insert
    foreach f [winNames -f]            {recent::pushDir $f} 
    catch {unset f}
} {
    # De-activation script.
    hook::deregister openHook          recent::pushDir
    hook::deregister saveasHook        recent::pushDir
    recent::insertDirsMenus            all uninsert
} uninstall {
    this-file
} help {
    This package inserts a 'Open Recent Dirs' submenu in the File menu,
    which keeps track of all recently used directories|
    |This package inserts a 'Open Recent Dirs' submenu in the File menu,
    which keeps track of all recently used directories||

    This feature is similar to the package: recentFilesMenu, but keeps
    track of the parent directories (and optionally volume names) of all
    recently used files rather than the actual filenames.
    

	  	Open Recent Dir menu
    
    Activating this package through the 'Config --> Packages --> Features'
    dialog will insert an 'Open Recent Dir' sub-menu of the 'File' menu. 
    This menu will contain the names of all 'remembered' directories, which
    are added whenever a file is opened in Alpha.  Selecting any directory
    or volume name will open a standard OS "Find File" dialog using the
    selected item as the default location.
    
    To set the max number of items listed in the menu (15 is the default)
    use the "Open Recent Dir --> Recent Dirs Prefs" menu item or simply
    click here:
    
	<<dialog::pkg_options recentDirsMenu>>.
    
    This dialog contains additional preferences which determine how the
    'Open Recent Dir' menu is built -- read the 'Recent Dirs Prefs' section
    below for more information.

    Tip: In Alpha (not Alphatk) holding down any modifier key while
    selecting a menu item will open it in the OS Finder instead of opening
    a Find File dialog.  To reverse this behavior, set the preference
    'openItemInFinder' in the preferences dialog.
    
    You also have the option to insert a 'Show Dir In Finder' menu in the
    'File --> File Utils' menu by setting the 'Show Dir In Finder Menu'
    preference.
    

	  	Save (A Copy) In menus
    
    This package can also insert two new 'Save' submenus into the File
    menu, named 'Save In' and 'Save A Copy In'.  These menus will contain
    the exact same list of directories/volumes as the 'Open Recent Dir'
    menu.  These are added if the 'Save (A Copy) In Menu' preferences are
    set in the preferences dialog.
    
    Selecting any menu item will either save a copy or save the current
    window IN the selected directory, bypassing the 'Save' dialog.  You
    will be prompted for the name of the file to be saved.  If the target
    file already exists, you will be asked if you want to replace it --
    pressing 'No' will cancel the operation.
    
    Tip: In Alpha (not Alphatk), holding down any modifier key while
    selecting a menu item in these 'Save' menus will automatically use the
    name of the current window, bypassing the prompt.  If the target file
    already exists, you will still be asked if you want to replace it.
    
    To reverse this behavior, so that you are prompted for a new filename
    ONLY if a modifier is pressed, set the 'saveUsingWindowName' preference
    in the preferences dialog.
    

	  	Directory Lists Preferences
    
    This package has a number of different preferences which can change how
    the list of recent directories will be built.  Changing any of these
    preferences will affect ALL of the submenus added by this package.
    

    This is a 'flag' preference, which can be turned on or off:
    
	Order Items Alphabetically
    
    Order all items alphabetically.  Otherwise, items are listed in the
    order in which they were last accessed, the most recent listed first.
    
    
    The remaining preferences can take on several different values:

	Number Of Recent Dirs
    
    The number of directories/volumes 'remembered' when building the list. 
    This will include items that might be ignored by the next three prefs.
    
	Display In Menu ...
   
    Items can be displayed with their full path name, only their tails, or
    with 'distinct' file tails.  With the second option, duplicate items
    are ignored, and only the most recent directory with the given name is
    used in the menu.  With the third option, duplicate directories are
    listed with their parent directories to indicate an unambiguous
    location.

	Missing Items Should Be ...
    
    When building the menu, the existence of directory/volume can be
    checked, and if found to be missing can be ignored, included, or dimmed
    in the menu.
    
    Note: In the Windows OS, directories which might exist over a network
    are never checked to see if they actually exist, since this might cause
    a big hangup -- they are always included in the menu.  In the MacOS, 
    networked items ARE checked for existence.
    
	Networked Items Should Be ...

    Since networked items (in the Windows OS) are never checked to see if
    they exist, and selecting a missing networked item might cause trouble,
    it might be useful to mark them in the menu so that you know which ones
    they are.  Networked items can also be simply ignored, or included in
    the menu.  This preference currently has no effect in the MacOS.

	Volumes Should Be ...

    Technically, any file in the top level of a volume (such as your hard
    drive or an ejectable disk) is not actually contained in a directory. 
    Volume names can be ignored, included, or placed in their own section
    in the menu.


	  	Recent Dirs Utilities
    
    The main 'Open Recent Dirs' menu contains several items which can also
    help maintain the list of directories.  Again, changing the list for
    one menu changes the lists for the others as well.
    
	Reset List
    
    Flushes the list of all remembered directories and volumes.
    
	Add Directory ...
    
    Opens a dialog allowing you to select directories which will be added
    to the menu.  Note that volumes can NOT be selected, only directories
    within them.  After choosing a directory, the same dialog will
    re-appear allowing you to select another -- press 'Cancel' when you are
    finished adding directories.
    
	Remove Menu Item ...
    
    Opens a list-pick dialog containing all of the current directories or 
    volumes listed in the menu.  Selecting one or more of them will remove 
    them from the menu(s).
    
	Rebuild Menu(s)
    
    This item should rarely be needed, because whenever a file is opened
    its parent directory/volume is always added to the list, and any
    missing items are dealt with according to the preferences described
    above.  If any of the menus seem out of date, however, use this item to
    rebuild them.
    
	Recent Dirs Help
    
    Opens this window.

	Recent Dirs Prefs ...
    
    Opens a dialog containing all of the preferences associated with this
    package.  Click on the 'Help' button in the dialog to obtain a short
    description of each preference listed.
    
	  	Performance Issues

    If you have a slower processor, activating this package might slow down
    this intitial startup of Alpha.  If this is the case, you probably want
    to set the 'Use Cache At Startup' preference, which will the menu list
    saved from when you last used Alpha, but won't check to see if the
    existence of any items has changed since your last editing session.
    
    If opening a window seems to take too long, you might also consider
    setting the following preferences to these values to make the Recent
    Dirs menus rebuild a little bit faster, listed in the order in which
    they'll have the greatest effect:

    Order Items Alphabetically --       Yes (turn this item on)
    Display In Menu --                  Any value except 'Distinct File Tails'
    Missing Items Should Be --          Included
    Networked Items Should Be --        Included
    Volumes Should Be --                Included
    
    Note that since all of the 'Save (A Copy) In' etc menus use the same
    list of menu items, turning any of them on or off probably will not
    have any noticable effect on startup or opening windows.
} maintainer {
    "Craig Barton Upright" <cupright@princeton.edu> 
    <http://www.princeton.edu/~cupright/>
}

namespace eval recent {}

# Recent Dirs Menu prefs

# By default, the 'Open Recent Dir' menu items open an OS 'Find File'
# dialog, but if any modifier key is held then the directory is opened in
# the OS Finder.  Turn this item on to reverse these behaviors||Turn this
# item off to restore the default behavior of modifier keys in the 'Open
# Recent Dir' menu item
newPref flag openItemInFinder           0 recentDirsMenu

# To listed recent directories alphabetically, turn this item on||To list
# directories in the order that they were most recently accessed, turn this
# item off
newPref flag orderItemsAlphabetically   1 recentDirsMenu recent::buildDirsMenu

# To insert a 'Save A Copy In' submenu in the File menu, turn this item on.
# Selecting any menu item will save a copy of the current window into the
# chosen directory or volume||To remove the 'Save A Copy In' menu, turn
# this item off
newPref flag saveACopyInMenu            0 recentDirsMenu recent::insertDirsMenus

# To insert a 'Save In' submenu in the File menu, turn this item on.
# Selecting any menu item will save the current window into the chosen
# directory or volume||To remove the 'Save In' menu, turn this item off
newPref flag saveInMenu                 0 recentDirsMenu recent::insertDirsMenus

# By default, the 'Save In' and 'Save A Copy In' menu items prompt the user
# for a new filename which will be saved in the given directory or volume,
# but if any modifier key is held then the name of the current window is
# used.  Turn this item on to reverse these behaviors||Turn this item off
# to restore the default behavior of modifier keys in the 'Save (A Copy) In'
# menus
newPref flag saveUsingWindowName        0 recentDirsMenu

# As of this writing, 'getModifiers' doesn't work in Alphatk, so we offer a
# different menu instead for opening a directory in the Finder.

# To insert a 'Show Dir In Finder' submenu in the 'File Utils' menu, turn
# this item on.  Selecting any menu item will save a copy of the current
# window into the chosen directory or volume||To remove the 'Show Dir In
# Finder' menu, turn this item off
newPref flag showDirInFinderMenu        0 recentDirsMenu recent::insertDirsMenus

# To use a cache of the Recent Dirs items when restarting Alpha (which is
# quicker but might not accurately reflect missing dirs) turn this item
# on||To always check the list of Recent Dirs items to see if they exist
# when restarting Alpha (which might take a little longer) turn this item
# off
newPref flag useCacheOnStartup          0 recentDirsMenu

# Recent items can be listed with their full path names, or only their file
# tails.  If 'Distinct File Tails' is selected, multiple items with the same
# tail will be identified with parent directories, displaying an unambiguous
# disk location.  The first two options rebuild the menu very quickly, the
# last takes the most time.
newPref var displayInMenu               2 recentDirsMenu recent::buildDirsMenu \
  [list "Full Path Names" "File Tails Only" "Distinct File Tails"] index

# Networked item (on the Windows OS) are never checked to see if they exist
# because if you are no longer on the network this can cause a big hangup.
# Networked items can be ignored, included, or marked as such in the menu.
newPref var networkedItemsShouldBe      2 recentDirsMenu recent::buildDirsMenu \
  [list "Ignored" "Included In Menu" "Marked In Menu"] index

# If a directory or volume cannot be found while rebuilding the menu, the
# item can be ignored or dimmed in the menu.  Note that networked items are
# never checked for existence.
newPref var missingItemsShouldBe        2 recentDirsMenu recent::buildDirsMenu \
  [list "Ignored" "Included In Menu" "Dimmed In Menu"] index

# The number of recent directories to remember when building the menu.
newPref var numberOfRecentDirs          15 recentDirsMenu recent::buildDirsMenu

# Technically, any file in the top level of a volume is not actually
# contained in a directory.  Volume names can be ignored, included, or
# placed in their own section in the menu.
newPref var volumesShouldBe             2 recentDirsMenu recent::buildDirsMenu \
  [list "Ignored" "Included In Menu" "Separated In Menu"] index

# Remove the menu cache if desired -- forces a call to 'recent::listDirs'
# during the building of the menu when Alpha is started up and activating
# this package.
if {!$recentDirsMenumodeVars(useCacheOnStartup)} {
    catch {unset recent::DirsMenuCache}
    catch {unset recent::DirsMarkList}
    catch {unset recent::DirsDimList}
} 

##
 # -------------------------------------------------------------------------
 #
 # "recent::insertDirsMenus" --
 #
 # Called by the (de)activate arguments in the package declaration, also by
 # the package prefs dialog when adding/removing optional menus.  Also
 # (de)registers Open Windows hooks for the optional menus.
 # 
 # -------------------------------------------------------------------------
 ##

proc recent::insertDirsMenus {pref {which ""}} {
    
    global recentDirsMenumodeVars
    
    if {$pref == "all"} {
	set menus "openRecentDir"
	foreach menuOption {saveACopyIn saveIn showDirInFinder} {
	    if {$recentDirsMenumodeVars(${menuOption}Menu)} {
		lappend menus $menuOption
	    } 
	} 
    } elseif {[regsub {Menu$} $pref {} menus]} {
	set which $recentDirsMenumodeVars($pref)
    } else {
	error "No valid menu could be defined."
    }
    foreach what $menus {
	switch $which {
	    "0"			{set which uninsert}
	    "1"			{set which insert}
	}
	# These last two will screw up some of the dynamic menu items, but
	# I'm not sure yet how to avoid this.  Maybe this is a feature
	# rather than a bug ...
	switch $what {
	    "openRecentDir" 	{set where "<S/Wclose"}
	    "showDirInFinder"	{set where "showInFinder"}
	    "saveACopyIn"	{set where "<SsaveACopyAs"}
	    "saveIn"		{set where "<SsaveAs"}
	}
	switch $what {
	    "showDirInFinder"	{set who "fileUtils"}
	    default             {set who "File"}
	}
	menu::$which $who submenu $where $what
	# Register open window hooks.
	if {[regexp {^save.*In} $what]} {
	    if {$which == "insert"} {
		hook::register requireOpenWindowsHook [list $who $what] 1
	    } else {
		hook::deregister requireOpenWindowsHook [list $who $what] 1
	    }
	}
	# Generate a message.
	if {$what != "openRecentDir"} {
	    if {$which == "insert"} {
		set msg "The '$what' menu has been inserted."
	    } else {
		set msg "The '$what' menu has been removed."
	    }
	} 
    }
    if {$pref != "all" && [info exists msg]} {status::msg $msg}
}

##
 # -------------------------------------------------------------------------
 #
 # "recent::pushDir" --
 #
 # Works with files whose name contained '[' or ']'.
 # Doesn't add to the menu the dir of any file which fails 'file exists'.
 # 
 # Called by either a hook or a menu item.  'args' is used in case this is
 # called by the 'saveas' hook.
 # 
 # -------------------------------------------------------------------------
 ##

proc recent::pushDir {fileName args} {
    
    global recent::Directories PREFS recentDirsMenumodeVars file::separator
    global recent::DirsMenuCache
    
    # Adding items to the menu.
    
    if {[llength $args]} {set fileName [lindex $args 0]}
    set rebuild 1
    set dirName [file dirname $fileName]
    set tmpDirs [file join $PREFS tmp]
    # More tests could be included here.
    if {![file exists $fileName] && [lindex $args 1] != "over-ride"} {
	# This isn't an actual file, and isn't being added by a menu item.
	return
    } elseif {[string first $tmpDirs $dirName] != "-1"} {
	# Don't include temporary directories (such as ftp ...)
	return
    } elseif {$fileName == [set file::separator] || ![llength $dirName]} {
	# Work-around for new 'file' command implementation in 7.5d1-d7.
	# There's probably a better way of handling this.
	set dirName ${fileName}${file::separator}
    } elseif {![string match *[set file::separator]* $dirName]} {
	# More work-around for new 'file' command implementation.
	append dirName [set file::separator]
    }
    # Make sure that the directory exists before we add it.  Once it's
    # in the list, it stays there unless the user removes it.
    if {![file exists $dirName]} {return}
    # Have we seen this one before?
    if {[set i [lsearch -exact [set recent::Directories] $dirName]] != "-1"} {
	# This dir/volume has already been included, so we just put it
	# at the end of the list.
	set recent::Directories [lreplace [set recent::Directories] $i $i]
	set rebuild 0
    }
    lappend recent::Directories $dirName
    if {[regexp {^[0-9]+$} $recentDirsMenumodeVars(numberOfRecentDirs)]} {
	set limit $recentDirsMenumodeVars(numberOfRecentDirs)
    } else {
	set limit 100
    }
    if {[llength ${recent::Directories}] > $limit} {
	set recent::Directories [lrange ${recent::Directories} 1 end]
    }
    if {$rebuild || !$recentDirsMenumodeVars(orderItemsAlphabetically)} {
	recent::buildDirsMenu 0
    }
}

##
 # -------------------------------------------------------------------------
 #
 # "recent::buildDirsMenu" --
 #
 # Building the menu.
 # 
 # Called by 'recent::pushDir', a menu item, or when some prefs change.
 # 'recent::listDirs' will determine the order of the items -- if anything
 # needs to be dimmed or marked, we do that here after creating the menus.
 # 
 # The variable 'recent::DirsMenuCache' is really only used when the package
 # is first activated, to make sure that we don't have to rebuild the same
 # menuList up to four times in a row.
 # 
 # -------------------------------------------------------------------------
 ##

proc recent::buildDirsMenu {{pref ""}} {
    
    global recent::Directories recentDirsMenumodeVars file::separator
    global recent::DirsMarkList recent::DirsDimList recent::DirsMenuCache
    
    if {[string length $pref]} {
	# Don't use the menu caches.
	catch {unset recent::DirsMenuCache}
	catch {unset recent::DirsMarkList}
	catch {unset recent::DirsDimList}
	if {$pref == "numberOfRecentDirs"} {
	    # Called from the prefs dialog.
	    set limit $recentDirsMenumodeVars(numberOfRecentDirs)
	    if {![regexp {^[0-9]+$} $limit]} {
		alertnote "The 'Number of Recent Dirs' preference should be a number."
		recent::dirsMenuProc "" "Recent Dirs Prefs"
		return
	    } else {
		set limit [expr $limit - 1]
		set recent::Directories [lrange [set recent::Directories] 0 $limit]
	    }
	} elseif {$pref == "rebuild"} {
	    # This is a "hard" rebuild, which will also remove all duplicates.
	    set recent::Directories [lunique [set recent::Directories]]
	} 
    }

    set menuNames "openRecentDir"
    foreach menuOption {saveACopyIn saveIn showDirInFinder} {
	if {$recentDirsMenumodeVars(${menuOption}Menu)} {
	    lappend menuNames $menuOption
	}
    }
    if {[info exists recent::DirsMenuCache]} {
	# This makes building the menus quicker when the package is first
	# activated.  Otherwise, the cache is usually unset above.
	set menuList1 [set recent::DirsMenuCache]
    } else {
	set menuList1 [recent::listDirs]
    }
    
    # Add the Recent Dirs Utils items.
    set menuList2 [list                                                        \
      "(-" "Reset List" "Remove Menu Item" "Add Directory" "Rebuild Menu"    \
      "(-" "Recent Dirs Help" "Recent Dirs Prefs"                             ]
    if {[llength $menuNames] != "1"} {
	set menuList2 [lreplace $menuList2 4 4 "Rebuild Menus"]
    }
    # Create the menus.
    set menuProc recent::dirsMenuProc
    if {[info tclversion] < 8.0} {
	# Bug in '-c' for Alpha 7.x
	Menu    -m -n openRecentDir   -p $menuProc "$menuList1 $menuList2"
	Menu    -m -n saveACopyIn     -p $menuProc $menuList1
	Menu    -m -n saveIn          -p $menuProc $menuList1
	Menu    -m -n showDirInFinder -p $menuProc $menuList1
    } else {
	Menu -c -m -n openRecentDir   -p $menuProc "$menuList1 $menuList2"
	Menu -c -m -n saveACopyIn     -p $menuProc $menuList1
	Menu -c -m -n saveIn          -p $menuProc $menuList1
	Menu -c -m -n showDirInFinder -p $menuProc $menuList1
    }
    # Do we have anything in the list?
    if {[lindex $menuList1 0] == "No Saved Dirs"} {
	# No.
	foreach menuName $menuNames {
	    enableMenuItem -m $menuName "No Saved Dirs" off
	}
	foreach item {"Reset List" "Remove Menu Item"} {
	    enableMenuItem -m openRecentDir $item off
	}
    } else {
	# Yes, so dim or mark as necessary.
	if {$recentDirsMenumodeVars(networkedItemsShouldBe) == "2"} {
	    foreach menuName $menuNames {
		if {![info exists recent::DirsMarkList]} {break}
		foreach item [set recent::DirsMarkList] {
		    markMenuItem -m $menuName $item on ~
		}
	    }
	} 
	if {$recentDirsMenumodeVars(missingItemsShouldBe) == "2"} {
	    foreach menuName $menuNames {
		if {![info exists recent::DirsDimList]} {break}
		foreach item [set recent::DirsDimList] {
		    enableMenuItem -m $menuName $item off
		}
	    }
	} 
    }
    if {[string length $pref] && $pref != "0"} {
	if {[llength $menuNames] == "1"} {
	    status::msg "The 'Open Recent Dir' menu has been rebuilt."
	} else {
	    status::msg "The Recent Dir menus have been rebuilt."
	}
    }
}

##
 # -------------------------------------------------------------------------
 #
 # "recent::listDirs" --
 #
 # Create a nice list for the menu, only adding parent directories of
 # remembered items if necessary to avoid ambiguity.  Possiblly Goes through
 # two more passes of the list to re-order or remove items as necessary,
 # determined by a few different package preferences.
 # 
 # -------------------------------------------------------------------------
 ##

proc recent::listDirs {} {
    
    global recent::Directories file::separator recentDirsMenumodeVars
    global recent::ItemDirConnect recent::DirsMarkList recent::DirsDimList
    global recent::DirsMenuCache
    
    set recent::DirsMarkList ""
    set recent::DirsDimList  ""
    set menuList ""
    set dirList   [set recent::Directories]
    set displayAs $recentDirsMenumodeVars(displayInMenu)
    set volumes   $recentDirsMenumodeVars(volumesShouldBe)
    set networked $recentDirsMenumodeVars(networkedItemsShouldBe)
    set missing   $recentDirsMenumodeVars(missingItemsShouldBe)

    if {$displayAs != "2"} {
	# Don't show distinct duplicate file tails in the menu.
	foreach d $dirList {
	    # Dealing with volumes.
	    if {[string match *[set file::separator] $d]} {
		# This Item is a volume.
		if {$volumes != "0"} {
		    # Include it and move on.
		    lappend menuList $d
		} 
		continue
	    }
	    # Dealing with networked or missing items.
	    if {[file::isNetworked $d]} {
		# Remove networked items from the menu list.
		if {$networked == "0"} {
		    continue
		}
	    } elseif {![file exists $d]} {
		# Remove missing items from the menu list .
		if {$missing == "0"} {
		    continue
		} 
	    }
	    if {$displayAs == "0"} {
		# Include full path names in the menu.
		lappend menuList $d
	    } else {
		# Include file tails in the menu.
		lappend menuList [file tail $d]
	    }
	}
    } else {
	# Display distinct duplicates in the menu.  We would use
	# 'file::minimalDistinctTails', but that automatically removes
	# files that don't exist.
	set level 1
	while {1} {
	    foreach d $dirList {
		# Dealing with volumes.
		if {[string match *[set file::separator] $d]} {
		    # This Item is a volume.
		    if {$volumes == "0"} {
			# Volumes should not be included.
			continue
		    } 
		}
		# Dealing with networked or missing items.
		if {[file::isNetworked $d]} {
		    # Remove networked items from the menu list.
		    if {$networked == "0"} {
			continue
		    }
		} elseif {![file exists $d]} {
		    # Remove missing items from the menu list .
		    if {!$missing} {
			continue
		    } 
		}
		set llen [llength [set tail [file split $d]]]
		if {$d == [set file::separator]} {
		    # Odd case, happens with volumes.
		    set i [lsearch -exact [set recent::Directories] $d]
		    set recent::Directories [lreplace [set recent::Directories] $i $i]
		} elseif {[regexp "${file::separator}$" $d]} {
		    # This item is a volume.
		    lappend menuList $d
		    continue
		} elseif {$llen < $level} {
		    # We've exceeded the top-level.  Must be an odd problem!
		    # Discard this problematic file.
		    continue
		}
		set tail [join [lrange $tail [expr {$llen - $level}] end] ${file::separator}]
		if {[info exists name($tail)]} {
		    lappend remaining $name($tail)
		    lappend remaining $d
		    set dup($tail) 1
		    set first [lsearch -exact $menuList $tail]
		    set menuList [lreplace $menuList $first $first $name($tail)]
		    if {$level == "1"} {lappend menuList $d}
		    unset name($tail)
		} elseif {[info exists dup($tail)]} {
		    lappend remaining $d
		    if {$level == "1"} {lappend menuList $d}
		} else {
		    set name($tail) $d
		    if {$level == "1"} {
			lappend menuList $tail
		    } else {
			set toolong [lsearch -exact $menuList $d]
			set menuList [lreplace $menuList $toolong $toolong $tail]
		    }
		}
	    }
	    if {![info exists remaining]} {break}
	    incr level
	    set dirList $remaining
	    unset remaining
	    unset dup
	}
    }
    # Make sure that we have a unique list
    set menuList [lunique $menuList]
    # Odd little bug with top-level folders.  Not sure where it happens above.
    regsub -all ${file::separator}${file::separator} $menuList ${file::separator} menuList
    # List either alphabetically, or in order of last appearance.
    if {$recentDirsMenumodeVars(orderItemsAlphabetically)} {
	set menuList [lsort -ignore $menuList]
    } else {
	set menuList [lreverse $menuList]
    }
    # Do we have anything in the menu list?
    if {![llength $menuList]} {
	# No.
	set recent::DirsMarkList "" 
	set recent::DirsDimList  ""
	return [set recent::DirsMenuCache [list "No Saved Dirs"]]
    } 
    # Yes, perform some tests to see if any items should be marked or dimmed.
    set test2 $recentDirsMenumodeVars(networkedItemsShouldBe) 
    set test3 $recentDirsMenumodeVars(missingItemsShouldBe) 
    if {$volumes == "2"} {
	# We have to first separate volumes if necessary, in order to properly
	# determine where marked or dimmed items will be located.
	set volsList  ""
	set dirsList  ""
	foreach item $menuList {
	    foreach d [set recent::Directories] {
		# Link it to the menu item ...
		set match1 [string match "*${file::separator}$item" $d]
		set match2 [string match "*[file nativename ${file::separator}]$item" $d]
		if {$item == $d || $match1 || $match2} {
		    # ...  if it matches the end of the directory we know it
		    # is unique (because of all of the work done above). If
		    # it doesn't match, we just continue.
		    set isVolume [string match *[set file::separator] $d]
		    if {$volumes == "2" && $isVolume} {
			# Separate volumes into their own section.
			lappend volsList $item
		    } else {
			lappend dirsList $item
		    }
		    continue
		}
	    }
	}
	# Now create the revised list of menu items.
	if {[llength $volsList]} {
	    set menuList "$volsList (- $dirsList"
	} else {
	    set menuList $dirsList
	}
    } 
    if {$networked == "2" || $missing == "2"} {
	# Determine where marked or dimmed items will be located.
	set recent::DirsMarkList ""
	set recent::DirsDimList  ""
	foreach item $menuList {
	    foreach d [set recent::Directories] {
		# Link it to the menu item ...
		set match1 [string match "*${file::separator}$item" $d]
		set match2 [string match "*[file nativename ${file::separator}]$item" $d]
		if {$item == $d || $match1 || $match2} {
		    # ...  if it matches the end of the directory we know it
		    # is unique (because of all of the work done above). If
		    # it doesn't match, we just continue.
		    if {[file::isNetworked $d]} {
			if {$networked == "2"} {
			    # Mark networked items.
			    lappend recent::DirsMarkList $item
			}
		    } elseif {![file exists $d]} {
			if {$missing == "2"} {
			    # Dim missing items.
			    lappend recent::DirsDimList  $item
			} 
		    }
		    continue
		}
	    }
	}
    } 
   return [set recent::DirsMenuCache $menuList]
}

##
 # -------------------------------------------------------------------------
 #
 # "recent::dirsMenuProc" --
 #
 # Works with menu items which contain '[', ']' and ''.
 #  
 # -------------------------------------------------------------------------
 ##

proc recent::dirsMenuProc {menuName item args} {
    
    global recentDirsMenumodeVars recent::Directories recent::DirsMenuCache
    global recent::ItemDirConnect alpha::platform
    
    if {[getModifiers]} {set modified 1} else {set modified 0}
    set rD [set recent::Directories]

    switch $item {
	"Reset List" {
	    if {$modified} {
		alertnote "Use this menu item to reset the list of remembered directories."
	    } else {
		set recent::Directories ""
		recent::buildDirsMenu 1
	    }
	}
	"Add Directory" -
	"Add Directory" {
	    if {$modified} {
		alertnote "Use this menu item to add to the list of remembered directories."
	    } elseif {[llength $args]} {
		foreach path $args {
		    recent::pushDir "" [file join $path dummy] "over-ride"
		}
	    } else {
		set newDir [get_directory]
		recent::pushDir "" [file join $newDir dummy] "over-ride"
		status::msg "'[file tail $newDir]' has been added to the 'Open Recent Dir' menu."
		set    question "Would you like to add more directories?\r"
		append question "If so, the dialog will reappear until you press 'Cancel'."
		if {[askyesno $question] == "no"} {return}
		set p "Select another, or cancel:"
		while {![catch {get_directory -p $p} newDir]} {
		    recent::pushDir "" [file join $newDir dummy] "over-ride"
		    status::msg "'[file tail $newDir]' has been added to the 'Open Recent Dir' menu."
		}
	    }
	}
	"Remove Menu Item" -
	"Remove Menu Item" {
	    if {$modified && ![llength $args]} {
		alertnote "Use this menu item to remove items from the list of remembered directories."
		return
	    } elseif {![llength $args]} {
		set p "Select items to remove from the menu:"
		if {![info exists recent::DirsMenuCache]} {recent::listDirs}
		set pickList [lremove -all [set recent::DirsMenuCache] "(-"]
		if {[catch {listpick -p $p -l $pickList} args]} {
		    status::msg "Cancelled." ; return
		}
	    }
	    foreach d $args {
		foreach dir $rD {
		    if {[string match *$d $dir]} {
			lappend removeList $dir
			break
		    }
		}
	    }
	    if {[info exists removeList]} {
		set recent::Directories [lremove -l $rD $removeList]
	    } else {
		# Should never get here ...
		status::msg "Sorry, couldn't remove '$args'." ; return
	    }
	    recent::buildDirsMenu 1
	}
	"Rebuild Menu" -
	"Rebuild Menus" {
	    if {$modified} {
		alertnote "Use this menu item to rebuild the Recent Dir menu(s)."
	    } else {
		recent::buildDirsMenu rebuild
	    }
	}
	"Recent Dirs Help" {
	    if {$modified} {
		alertnote "Use this menu item to open the 'Recent Dirs Help' window."
	    } else {
		package::helpFile   "recentDirsMenu"
	    }
	}
	"Recent Dirs Prefs" -
	"Recent Dirs Prefs" {
	    if {$modified} {
		alertnote "Use this menu item to set the 'Recent Dirs Menu' preferences."
	    } else {
		dialog::pkg_options "recentDirsMenu"
		requireOpenWindowsHook 1
	    }
	}
	default {
	    if {[info exists recent::ItemDirConnect($item)]} {
		set d [set recent::ItemDirConnect($item)]
	    } else {
		set d [file::pathEndsWith $item $rD]
	    }
	    if {$d != "" && [file exists $d]} {
		set dir $d
	    } elseif {[file exists $item]} {
		set dir $item
	    }
	    if {![info exists dir]} {
		set    question "Couldn't find '$item'.\r"
		append question "Would you like to remove it from the menu?"
		if {[askyesno $question] == "yes"} {
		    recent::dirsMenuProc "" "Remove Menu Item" $item
		} elseif {$recentDirsMenumodeVars(missingItemsShouldBe) == "2"} {
		    recent::buildDirsMenu 1
		}
		return
	    }
	    switch $menuName {
		"saveIn" -
		"saveACopyIn"   {
		    if {![llength [winNames]]} {
			status::msg "This item requires an open window."
			return
		    } 
		    if {$recentDirsMenumodeVars(saveUsingWindowName)} {
			set modified [expr -$modified + 1]
		    }
		    if {$modified} {
			# Save using the (stripped) current window name.
			file::$menuName $dir [win::StripCount [win::CurrentTail]]
		    } else {
			# Prompt the user for the window name.
			file::$menuName $dir ""
		    }
		}
		"showDirInFinder" -
		"openRecentDir" {
		    if {$menuName == "showDirInFinder"} {
			set modified 1
		    } elseif {$recentDirsMenumodeVars(openItemInFinder)} {
			set modified [expr -$modified + 1]
		    }
		    if {$modified} {
			# Try to show the directory in the Finder.
			file::showInFinder $dir
		    } else {
			# Open a 'Find File' dialog using this directory as
			# the default.
			findFile $dir
		    }
		}
	    }
	}
    }
}

# These probably should go in 'fileManipulation.tcl'

namespace eval file {}

##
 # -------------------------------------------------------------------------
 #
 # "file::saveIn" --
 #
 # Save the current window in a specified directory, bypassing the 'Save'
 # dialog, optionally prompting for a new fileName.
 #
 # -------------------------------------------------------------------------
 ##

proc file::saveIn {dirPath {fileName ""}} {

    requireOpenWindow

    global file::separator

    status::msg "Saving in '${dirPath}' ..."

    set currentWindow [win::StripCount [win::Current]]
    set currentTail   [win::StripCount [win::CurrentTail]]

    if {![string match *${file::separator} $dirPath]} {
	set dirName [file tail $dirPath]
    } else {
	set dirName $dirPath
    }
    if {![file exists $dirPath]} {
	# Make sure that the target directory/volume exists.
	status::msg "Sorry, '${dirPath}' could not be found."
	return
    } elseif {![string length $fileName]} {
	# Prompt the user for a new fileName.
	set p "Save in '${dirName}' as"
	if {[catch {prompt $p $currentTail} fileName]} {
	    status::msg "Cancelled." ; return
	}
    }
    # Save the current window in the desired location.
    set newFile [file join $dirPath $fileName]
    if {![file exists $newFile]} {
	# In order to use 'saveAs -f', the target must first exist.
	close [open $newFile w]
    } elseif {[file nativename $newFile] == $currentWindow} {
	status::msg "Cancelled -- target file is the same as the original !!"
	return
    } else {
	# Should we over-write the existing target?
	set    question "'${fileName}' already exists in '${dirName}'.\r"
	append question "Do you want to replace it?"
	if {[askyesno $question] == "no"} {status::msg "Cancelled." ; return}
    }
    saveAs -f $newFile ; save
}

##
 # -------------------------------------------------------------------------
 #
 # "file::saveACopyIn" --
 #
 # Save a copy of the current window in a specified directory, bypassing
 # the 'Save' dialog, optionally prompting for a new fileName.
 #
 # -------------------------------------------------------------------------
 ##

proc file::saveACopyIn {dirPath {fileName ""}} {

    requireOpenWindow

    global file::separator

    status::msg "Saving a copy in '${dirPath}' ..."

    set currentWindow [win::StripCount [win::Current]]
    set currentTail   [win::StripCount [win::CurrentTail]]

    if {![string match *${file::separator} $dirPath]} {
	set dirName [file tail $dirPath]
    } else {
	set dirName $dirPath
    }
    if {![file exists $dirPath]} {
	# Make sure that the target directory/volume exists.
	status::msg "Sorry, '${dirPath}' could not be found."
	return
    } elseif {![winIsFile $currentWindow]} {
	# Make sure that the current window is actually a file.
	status::msg "'$currentWindow' must first be saved."
	return
    } elseif {![string length $fileName]} {
	# Prompt the user for a new fileName.
	set p "Save a copy in '${dirName}' as"
	if {[catch {prompt $p $currentTail} fileName]} {
	    status::msg "Cancelled." ; return
	}
    }
    # Save a copy of the current window in the desired location.
    set newFile [file join $dirPath $fileName]
    if {[file nativename $newFile] == $currentWindow} {
	status::msg "Cancelled -- target file is the same as the original !!"
	return
    } elseif {[file exists $newFile]} {
	# Should we over-write the existing target?
	set question "'${fileName}' already exists in '${dirName}'.\r"
	append question "Do you want to replace it?"
	if {[askyesno $question] == "yes"} {
	    # In order to use 'file copy', the target can NOT first exist.
	    catch {file delete $newFile}
	} else {
	    status::msg "Cancelled."
	    return
	}
    }
    if {[catch {file copy $currentWindow $newFile}]} {
	status::msg "Failed to copy '$currentTail' to '${dirName}'"
    } else {
	status::msg "'${newFile}' saved."
    }
}


# ===========================================================================
#
#  ------------  #
# 
#  Version History  #
# 
#  modified by  rev    reason
#  -------- --- ------ -----------
#  06/10/01 cbu 0.1    First version of package written by Craig Barton Upright,
#                        based on 'recentFiles.tcl'.
#  06/12/01 cbu 0.2    Less restrictive inclusion of 'dir' names, can include
#                        volumes as well.
#                      If an item isn't found, the user is given the option
#                        to remove it from the menu.
#                      Items are added to the menu when a window is opened, not
#                        when closed.  (Using 'openHook', not 'closeHook'.)
#  06/13/01 cbu 0.3    Bug fix for removing non-existent menu items.
#                      Added pref to reverse modifier key behavior.
#                      'Remove Menu Item' now lists items as in menu.
#                      Better menu addition of volume names ??  Involves a few
#                        work-arounds to compensate for new 'file' command
#                        which was implemented in 7.4d1-d7, hopefully ensures
#                        compatibility back to 7.3.
#  06/15/01 cbu 1.0    Bug fix for menu declaration, 'menu::insert'.
#                      More user control over how missing items and volumes are
#                        listed (or ignored) in the menu.
#                      New 'Save A Copy In' and 'Save In' menus.
#                      Moved some code into new procs.
#                      Now requires 7.4.x for newer 'menu::insert' command.
#  06/18/01 cbu 1.0.1  Bug fix for file::saveIn.
#                      Bug fix when dimming missing items.
#                      Networked items are never checked for existence. (Can
#                        cause a hangup in the Windows OS).
#                      When adding directories, user is given the choice to
#                        add more items, with some explanation.
#  06/20/01 cbu 1.1    Bug fix for recent::pushDir when 'fileName' doesn't
#                        exist (as when called from 'Add Directory' menu item.)
#                      Using 'enableMenuItem' instead of embedding '(' in 
#                        menuLists so that it is compatible with 'Menu -c' 
#                        argument.  (Alphatk/8 issue).
#                      New pref for optionally marking networked items.
#                      New 'Show Dir In Finder' menu option.
#                      New option to display full path names in menu.
#                      Faster start-up, saving menulist in a cache.  (Cache
#                        is generally unset whenever 'recent::buildDirsMenu'
#                        is called otherwise.)
#                      'file::saveIn' and 'file::saveACopyIn' now check to
#                        make sure that target file is not same as original.
#                      Menu Cache can be used on start-up if desired.
#  06/27/01 cbu 1.1.1  Improved 'help' arg -- formatting, typos and such.
#  06/28/01 cbu 1.1.2  Minor bug for ignoring missing items in 'recent::listDirs'
#                      Improved ballon help for Alphatk.
#  09/25/01 cbu 1.1.3  Temporary dirs (in PREFS:tmp) are now finally ignored.
#                      Bug fix when adding dirs that are already there. 
#                        (Should have used 'end' instead of '$limit' in string
#                        command)
#                      Fix (??) to compensate for Alphatk enableMenuItem fixes,
#                        so we no longer need to find the index of the item
#                        names when creating the list to dim/mark.
# 

# ===========================================================================
#
# .
