## -*-Tcl-*- (install)
 # ###################################################################
 #  Alpha - new Tcl folder configuration
 #
 #  FILE: "recentFilesMulti.tcl"
 #                                    created: 01/28/2000 {18:45:34 PM}
 #                                last update: 10/11/2001 {17:15:03 PM}
 #
 # See 'help' section below for information.
 # 
 #     
 #  This code has gone through a couple of iterations, to simplify
 #  things internally, and to define a better way of specifying these
 #  groups of files.  It works well, (for me!).
 #  
 #  Known problem on Windoze with Alphatk.  If you edit network
 #  files (i.e. remote files), and then try to startup Alphatk
 #  when there is no network, it will hang for some time each
 #  time we try to rebuild this menu.  The code tries to workaround
 #  this issue.
 #  
 #  General buglet: if the list of recent files is manually edited
 #  somehow to contain the exact same file-path more than once,
 #  it will appear more than once in the menu.  You'll have to edit
 #  your arrdefs.tcl to remove it.
 #
 # Copyright (c) 2000-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # Distributable under Tcl-style (free) license.
 #  
 # ###################################################################
 ##

alpha::extension recentFilesMultiMenu 1.1.4 {
    namespace eval recentmulti {}
    hook::register saveasHook recentmulti::push
    hook::register closeHook recentmulti::push
    prefs::updateHome recent::Files list
    prefs::modified multilist::recent list
    menu::buildProc recent recentmulti::makeMenu
    # Place a submenu just before the 'close' menu item.
    menu::insert File submenu "<S/Wclose" recent
    # declare the fileset
    set "gfileSetsType(Recent Files)" "procedural"
    set "gfileSets(Recent Files)" recentmulti::listFiles
    lunion filesetsNotInMenu "Recent Files"
    lappend fileset::notChangeable "Recent Files"
    package::addPrefsDialog recentFilesMultiMenu
    multilist::create recent recentmulti::whichgroup
    # Add a space to avoid conflict with other menus
    set recentmulti::currentGroup "Miscellaneous "
    
    alpha::package require filesets
    # This allows us to use any fileset we like as a recent files group.
    fileset::attachNewInformation * flag "Use As Recent Files Group" 0 \
      "To have all recently used files from this set stored separately\
      in the recent files menu, click this box.||To remove this fileset\
      from the set considered for recent files, click this box." \
      recentmulti::filesetGroupInfoChanged
    hook::callAll fileset-new procedural "Recent Files"
    
    if {[package::active recentFilesMenu]} {
	error "The 'Recent Files Multi Menu' package\
	  cannot be used while the 'Recent Files Menu' package is active."
    }
} help {
    A recent files menu is already provided in the 'recentFiles'
    package, which this package can also emulate.  However it also
    adds a lot of new functionality:
     
    I find I am working on a number of different projects: Tcl code
    for Alpha (of course), various programming projects, perhaps
    some web-page design,...
     
    When I switch from working on one project to the next, I'm
    often annoyed by the fact that the recent files menu is full
    of files from the previous project, _and_ the particular 5 or
    10 files that I was working on before in this project have of
    course long ago vanished from the recent files menu.
    
    This package seeks to get around that problem, by using
    the concept of groups of recent files, defined by where they
    lie in your filesystem.  So, anything in the 'Alpha' hierarchy
    is in a separate group to anything in my 'Programming' hierarchy
    which is in a different group to my 'html' hierarchy.  Recently
    used files in these different groups are remembered separately.
    
    So what does the recent files menu display?  It displays the 
    items in the most recently used group in the usual 'Recent Files'
    menu, and places in submenus the items in all other groups. A
    'Miscellaneous' group contains items which don't fit into any other.
    
    It can handle files over a network, but it never checks if they
    exist, since if we're not on a network, that causes problems.
} maintainer {
    "Vince Darley" vince@santafe.edu <http://www.santafe.edu/~vince/>
} uninstall this-file


# The number of files to list in the 'Files->Recent' menu.
newPref variable numberOfRecentFiles 15 recentFilesMultiMenu
# The ordering scheme for items in the recent files menu.
newPref variable orderRecentFilesBy 0 recentFilesMultiMenu recentmulti::makeMenu [list \
  "Alphabetical Order" "Date"] index
# Use this key binding to edit the most recently used file.
newPref binding editLastUsedFile "" recentFilesMultiMenu "" recentmulti::editLastFile

# To show each copy of files with the same name, by identifying them with
# their disk location, click this box.||To list only the most recent version
# of all files with the same name, click this box.
newPref flag showDistinctDuplicates 1 recentFilesMultiMenu

# To use sets of recent files, click this box.||To use only one global
# list of recent files, click this box.
newPref flag useGroupsOfRecentFiles 0 recentFilesMultiMenu

##
 # -------------------------------------------------------------------------
 #
 # "recentmulti::push" --
 #
 #  Works with files whose name contained '[' or ']' which didn't before.
 #  Doesn't add any file which fails 'file exists' to the menu.
 #  
 #  We only need the 'args' since this may be called by saveasHook
 # -------------------------------------------------------------------------
 ##
proc recentmulti::push {name args} {
    global recentFilesMultiMenumodeVars file::separator

    regsub { <[0-9]+>$} $name {} name
    if {![file exists $name]} { return }
    set name [file nativename $name]
    if {$recentFilesMultiMenumodeVars(showDistinctDuplicates)} {
	if {[llength [set ind [multilist::find recent $name]]]} {
	    eval multilist::remove recent $ind
	    if {[recentmulti::add recent $name \
	      $recentFilesMultiMenumodeVars(numberOfRecentFiles)] \
	      || $recentFilesMultiMenumodeVars(orderRecentFilesBy)} {
		recentmulti::makeMenu
	    }
	    return
	}
    } else {
	if {[info tclversion] < 8.0} {
	    regsub -all {\\([][])} $name {\1} name
	    # this weird search handles a variety of unusual problems with
	    # Alpha's interpretation of menu items.
	    if {[llength [set ind [multilist::find recent "${file::separator}[quote::Regfind [file tail $name]]?$" -regexp]]]} {
		eval multilist::remove recent $ind
		if {[recentmulti::add recent $name \
		  $recentFilesMultiMenumodeVars(numberOfRecentFiles)] \
		  || $recentFilesMultiMenumodeVars(orderRecentFilesBy)} {
		    recentmulti::makeMenu
		}
		return
	    }
	} else {
	    if {[llength [set ind [multilist::find recent $name]]]} {
		eval multilist::remove recent $ind
		if {[recentmulti::add recent $name \
		  $recentFilesMultiMenumodeVars(numberOfRecentFiles)] \
		  || $recentFilesMultiMenumodeVars(orderRecentFilesBy)} {
		    recentmulti::makeMenu
		}
		return
	    }
	    if {0} {
		set ind 0
		foreach f [set recentmulti::Files] {
		    # perhaps we ought to test also for complications due to
		    # files which end in ''.
		    if {[file tail $f] == [file tail $name]} {
			set recentmulti::Files [lreplace ${recentmulti::Files} $ind $ind]
			lappend recentmulti::Files $name
			if {$recentFilesMultiMenumodeVars(orderRecentFilesBy)} {
			    recentmulti::makeMenu
			}
			return
		    }
		    incr ind
		}
	    }
	}
    }

    recentmulti::add recent $name $recentFilesMultiMenumodeVars(numberOfRecentFiles)
    recentmulti::makeMenu
}

proc recentmulti::getMenuEntries {filelist} {
    global recentFilesMultiMenumodeVars
    if {$recentFilesMultiMenumodeVars(showDistinctDuplicates)} {
	set menulist [file::minimalDistinctTails $filelist]
    } else {
	set menulist [list]
	foreach t $filelist {
	    if {[file::isNetworked $t] || [file exists $t]} {
		lappend menulist [file tail $t]
	    }
	    # else we just let the file disappear through lack of use
	}
    }
    if {$recentFilesMultiMenumodeVars(orderRecentFilesBy)} {
	return [lreverse $menulist]
    } else {
	return [lsort -ignore $menulist]
    }
}

proc recentmulti::makeMenu {args} {
    global recentFilesMultiMenumodeVars
    
    if {$recentFilesMultiMenumodeVars(useGroupsOfRecentFiles)} {
	global recentmulti::currentGroup
	set menuitems {}
	foreach group [multilist::getgroups recent] {
	    if {$group != [set recentmulti::currentGroup]} {
		lappend menuitems [list Menu -m -c -n $group -p recentmulti::menuProc \
		  [recentmulti::getMenuEntries [multilist::getgroup recent $group]]]
	    }
	}
	eval lappend menuitems [recentmulti::getMenuEntries [multilist::getgroup recent [set recentmulti::currentGroup]]]
    } else {
	set menuitems [recentmulti::getMenuEntries [multilist::getgroup recent "Miscellaneous "]]
    }

    set rest [list "(-" "Reset List" "Add Group"]
    if {[llength [recentmulti::listGroups]]} {
	lappend rest "Remove Group"
    }
    if {[info tclversion] < 8.0} {
	# Bug in '-c' for Alpha 7.2
	Menu -m -n recent -p recentmulti::menuProc \
	  [concat $menuitems $rest]
    } else {
	Menu -m -c -n recent -p recentmulti::menuProc \
	  [concat $menuitems $rest]
    }

    set enable [expr {[llength $menuitems] ? 1 : 0}]
    enableMenuItem File recent $enable
}

## 
 # -------------------------------------------------------------------------
 # 
 # "recentmulti::add" --
 # 
 #  Returns 1 if the current group has changed (and so the menu should
 #  probably be rebuilt).
 # -------------------------------------------------------------------------
 ##
proc recentmulti::add {tag what {max -1}} {
    global recentmulti::currentGroup
    set group [multilist::add $tag $what $max]
    if {$group != ${recentmulti::currentGroup}} {
	set recentmulti::currentGroup $group
	return 1
    } else {
	return 0
    }
}

##
 # -------------------------------------------------------------------------
 #
 # "recentmulti::menuProc" --
 #
 #  Works with menu items which contain '[', ']' and '' which didn't work
 #  before.
 # -------------------------------------------------------------------------
 ##
proc recentmulti::menuProc {menu name} {
    global recentmulti::currentGroup
    switch -- $name {
	"Reset List" {
	    multilist::remove recent [set recentmulti::currentGroup]
	    set recentmulti::currentGroup [lindex [multilist::getgroups recent] 0]
	    #Menu -m -n recent -p recentmulti::menuProc {}
	    recentmulti::makeMenu
	}
	"Add Group" {
	    if {![catch {prompt "Enter a name for the new group" ""} g]} {
		recentmulti::registerNewGroup $g [list recentmulti::standardGroup $g]
		newPref variable group[join $g ""]SearchPath "" recentFilesMultiMenu
		prefs::modified recentFilesMultiMenu(group[join $g ""]SearchPath)
		alertnote "You may now set the search path for this group"
		dialog::pkg_options recentFilesMultiMenu
		recentmulti::makeMenu
	    }
	}
	"Remove Group" {
	    if {[llength [recentmulti::listGroups]]} {
		set g [listpick -p "Remove which group" [recentmulti::listGroups]]
		if {$g != ""} {
		    multilist::remove recent $g
		    recentmulti::deregisterGroup $g
		    recentmulti::makeMenu
		}
	    }
	}
	default {
	    if {$menu == "recent"} {
		set menu [set recentmulti::currentGroup]
	    } else {
		set recentmulti::currentGroup $menu
		recentmulti::makeMenu
	    }
	    set f [file::pathEndsWith $name [multilist::getgroup recent $menu]]
	    if {$f != ""} {
		edit $f
		return
	    }
	    if {[file exists $name]} {
		edit $name
		return
	    }
	    dialog::errorAlert "Couldn't find a file '$name'.  Weird!"
	}
    }
}

##
 # -------------------------------------------------------------------------
 #
 # "recentmulti::listFiles" --
 #
 #  Used to retrieve the list of files in the 'recent files' fileset
 # -------------------------------------------------------------------------
 ##
proc recentmulti::listFiles {} {
    global recentmulti::currentGroup
    return [multilist::getgroup recent [set recentmulti::currentGroup]]
}

proc recentmulti::editLastFile {} {
    global recentmulti::Files
    if {[set rl [llength ${recentmulti::Files}]]} {
	incr rl -1
	edit -c -w [lindex ${recentmulti::Files} $rl]
    }
}

#  Group handling  #

proc recentmulti::listGroups {} {
    global recentmulti::groups
    array names recentmulti::groups
}

## 
 # -------------------------------------------------------------------------
 # 
 # "recentmulti::registerNewGroup" --
 # 
 #  The script you register for the new group may be evaluated in
 #  two different ways:
 #  
 #  (i) eval $testscript [list $filename]
 #  
 #  should return 1 or 0 depending on if the filename is in the group 
 #  or not
 #  
 #  (ii) eval $testscript
 #  
 #  should delete all information associated with the group. The user
 #  has decided to remove the group.
 #  
 #  We suggest using an 'args' parameter and [llength $args] to 
 #  differentiate between the two.
 # -------------------------------------------------------------------------
 ##
proc recentmulti::registerNewGroup {group testScript} {
    global recentmulti::groups
    set recentmulti::groups($group) $testScript
    prefs::modified recentmulti::groups($group)
}

proc recentmulti::deregisterGroup {group} {
    global recentmulti::groups
    if {[info exists recentmulti::groups($group)]} {
	eval [set recentmulti::groups($group)]
	unset recentmulti::groups($group)
	prefs::modified recentmulti::groups($group)
    }
}

proc recentmulti::whichgroup {name} {
    global recentFilesMultiMenumodeVars recentmulti::groups
    if {$recentFilesMultiMenumodeVars(useGroupsOfRecentFiles)} {
	foreach group [array names recentmulti::groups] {
	    # The array entry is a script to evaluate which 
	    # will tell us if the file is in that group
	    # (if given an argument).
	    if {[eval [set recentmulti::groups($group)] [list $name]]} {
		return $group
	    }
	}
	return "Miscellaneous "
    } else {
	return "Miscellaneous "
    }
}

proc recentmulti::filesetGroupInfoChanged {fset val} {
    if {$val} {
	recentmulti::registerNewGroup $fset [list recentmulti::filesetGroup $fset]
    } else {
	recentmulti::deregisterGroup $fset
    }
}

proc recentmulti::filesetGroup {fset args} {
    global gfileSets
    if {[llength $args]} {
	set name [lindex $args 0]
	if {[fileset::exists $fset]} {
	    return [fileset::isIn $fset $name]
	} else {
	    # fileset has been deleted!
	    recentmulti::deregisterGroup $fset
	    multilist::remove recent $fset
	    return 0
	}
    } else {
	# Deregister this fileset group
	fileset::setInformation $fset useAsRecentFilesGroup 0
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "recentmulti::standardGroup" --
 # 
 #  This is the stanard group handler, which has an associated prefs
 #  variable.  We create a new group and register this handler like this:
 #  
 #  recentmulti::registerNewGroup $g [list recentmulti::standardGroup $g]
 # 
 #  When we are called with one argument, it is a file, and we must check
 #  if it is in this group.  When called with no arguments, we delete
 #  all information associated with the group.
 # -------------------------------------------------------------------------
 ##
proc recentmulti::standardGroup {group args} {
    global recentFilesMultiMenumodeVars
    set pref "group[join ${group} ""]SearchPath"
    if {[llength $args]} {
	set name [lindex $args 0]
	foreach path $recentFilesMultiMenumodeVars($pref) {
	    if {[file::pathStartsWith $name $path]} {
		return 1
	    }
	}
	return 0
    } else {
	unset recentFilesMultiMenumodeVars($pref)
	prefs::modified recentFilesMultiMenu($pref)
    }
}


#  Multilist object  #

namespace eval multilist {}

proc multilist::add {tag what {max -1}} {
    global multilist::$tag
    set item [multilist::which $tag $what]
    lappend multilist::${tag}($item) $what
    if {$max > 0 && ([llength [set multilist::${tag}($item)]] > $max)} {
	set multilist::${tag}($item) [lrange [set multilist::${tag}($item)] 1 end]
    }
    return $item
}

proc multilist::which {tag what} {
    global multilist::decisions
    eval [set multilist::decisions($tag)] [list $what]
}

proc multilist::getgroup {tag item} {
    global multilist::$tag
    if {[info exists multilist::${tag}($item)]} {
	return [set multilist::${tag}($item)]
    } else {
	return ""
    }
}

proc multilist::getgroups {tag} {
    global multilist::$tag
    return [array names multilist::${tag}]
}

proc multilist::remove {tag item {index ""}} {
    global multilist::$tag
    if {[string length $index]} {
	set multilist::${tag}($item) [lreplace [set multilist::${tag}($item)] $index $index]
    } else {
	if {[info exists multilist::${tag}($item)]} {
	    unset multilist::${tag}($item)
	}
    }
}

proc multilist::find {tag what {how "-exact"}} {
    global multilist::$tag
    foreach arr [array names multilist::${tag}] {
	if {[set ind [lsearch $how [set multilist::${tag}($arr)] $what]] >= 0} {
	    return [list $arr $ind]
	}
    }
    return ""
}

proc multilist::create {tag decisionProc} {
    global multilist::decisions 
    set multilist::decisions($tag) $decisionProc
}
