## -*-Tcl-*- (auto-install)
 # ==========================================================================
 # FILE: "specialCharacters.tcl"
 #                                   created: 01-10-03 19.48.48
 #                               last update: 01-11-02 20.21.18
 #                               
 # Description:
 # 
 # Inserts a "Special Characters" submenu to the "Text" menu, allowing for
 # easy insertion of special characters into the current window.  Users can
 # define their own key bindings for any special character.
 # 
 # Note that all of these menus include a space after their names to help
 # ensure that they don't conflict with any mode-specific menu.
 # 
 # Author: Craig Barton Upright
 # E-mail: <cupright@princeton.edu>
 #   mail: Princeton University
 #         2.N.1 Green Hall,  Princeton, New Jersey  08544
 #    www: <http://www.princeton.edu/~cupright>
 #
 # --------------------------------------------------------------------------
 #  
 # Copyright (c) 2001  Craig Barton Upright
 # 
 # Redistribution and use in source and binary forms, with or without
 # modification, are permitted provided that the following conditions are met:
 # 
 #   Redistributions of source code must retain the above copyright
 #    notice, this list of conditions and the following disclaimer.
 # 
 #   Redistributions in binary form must reproduce the above copyright
 #    notice, this list of conditions and the following disclaimer in the
 #    documentation and/or other materials provided with the distribution.
 # 
 #   Neither the name of Alpha/Alphatk nor the names of its contributors may
 #    be used to endorse or promote products derived from this software
 #    without specific prior written permission.
 # 
 # THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 # AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 # IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 # ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
 # ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 # DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 # SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 # CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 # LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 # OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 # DAMAGE.
 # 
 # ==========================================================================
 ##

alpha::feature specialCharacters 0.1 "global-only" {
    # Initialization script.
    # We require 7.4a8 for the new 'menu::insert' proc.
    alpha::package require -loose AlphaTcl 7.4a8
    # Register a build proc for the menu.
    menu::buildProc "specialCharacters " specialChars::buildMenu
} {
    # Activation script.
    menu::insert     Text items 0 "(-)"
    menu::insert     Text submenu "(-)" "specialCharacters "
    hook::register   requireOpenWindowsHook [list "Text" "specialCharacters "] 1
} {
    # De-activation script.
    menu::uninsert   Text submenu "(-)" "specialCharacters "
    hook::deregister requireOpenWindowsHook [list "Text" "specialCharacters "] 1
} uninstall {
    this-file
} help {
    This package inserts a 'Special Characters' submenu in the Text menu,
    which allows for easy insertion of international letters and other symbols.
    All characters can be assigned a user defined key binding.
} maintainer {
    "Craig Barton Upright" <cupright@princeton.edu> 
    <http://www.princeton.edu/~cupright/>
}

namespace eval specialChars {}

# Define arrays containing the special characters.

array set specialChars::TypeChar {
    
    "a characters " {      (-)      }
    "e characters " {        (-)    }
    "i characters " {        (-)    }
    "o characters " {       (-)     }
    "u characters " {        (-)    }
    "misc chars "   {      (-)      }
    "symbols 1 "    {  ~                 }
    "symbols 2 "    {        ?   }
    "symbols 3 "    {          }
    "symbols 4 "    {             }
    "symbols 5 "    {  ?       }
}

# "array names specialChars::TypeChar" won't give us the order that we want.

set specialChars::Menus [list \
  "a characters " \
  "e characters " \
  "i characters " \
  "o characters " \
  "u characters " \
  "misc chars " \
  "(-)" \
  "symbols 1 " \
  "symbols 2 " \
  "symbols 3 " \
  "symbols 4 " \
  "symbols 5 " ]

# If I can figure out all of Alpha's default bindings (or is it in the MacOS
# ??)  then I could include these in the menu as well.  These would be
# presented in the dialog to change them as well, or could be removed by
# assigning "no binding" to them.  But then this might not be the best idea
# for international users ...

array set specialChars::Defaults {
    dummy ""
}

##
 # -------------------------------------------------------------------------
 # 
 # "specialChars::buildMenu" --
 # 
 # Using the arrays defined above, create the "Special Characters" menu.  User
 # defined bindings are used in the menu preferentially, otherwise any binding
 # defined in "specialChars::Defaults" will be used.
 # 
 # -------------------------------------------------------------------------
 ##

proc specialChars::buildMenu {} {
    
    global specialChars::Menus specialChars::TypeChar
    global specialChars::Keys  specialChars::Defaults
        
    set p "specialChars::menuProc"
    foreach menuName [set specialChars::Menus] {
	if {$menuName == "(-)"} {lappend menuList "(-)" ; continue}
	set charList ""
	foreach character [set specialChars::TypeChar($menuName)] {
	    if {[info exists specialChars::Keys($character)]} {
		lappend charList [set specialChars::Keys($character)]$character
	    } elseif {[info exists specialChars::Defaults($character)]} {
		lappend charList [set specialChars::Defaults($character)]$character
	    } else {
		lappend charList $character
	    } 
	}
	eval {lappend menuList "Menu -m -n {$menuName} -p $p {$charList}"}
    }
    lappend menuList "(-)" "assignBindings" "unsetBindings"
    return [list build $menuList specialChars::menuProc {}]
}

##
 # -------------------------------------------------------------------------
 # 
 # "specialChars::menuProc" --
 # 
 # Insert the special character, or adjust bindings.
 # 
 # Includes a special case for  , which gets converted to an empty string.
 # 
 # -------------------------------------------------------------------------
 ##

proc specialChars::menuProc {menuName itemName} {
    
    switch $itemName {
	assignBindings {specialChars::assignBindings}
	unsetBindings  {specialChars::unsetBindings}
	""             {typeText ""}
	default        {typeText $itemName}

    }
}

##
 # -------------------------------------------------------------------------
 # 
 # "specialChars::assignBindings" --
 # 
 # Offer the list of all menus to the user, to adjust bindings.
 # 
 # -------------------------------------------------------------------------
 ##

proc specialChars::assignBindings {{title "Select a menu:"}} {

    global specialChars::Menus specialChars::TypeChar 
    global specialChars::Keys  specialChars::Defaults
    
    set menus [lremove -all [set specialChars::Menus] "(-)"]
    if {[catch {listpick -p $title $menus} type]} {
        status::msg "Cancelled." ; return
    } 

    foreach character [set specialChars::TypeChar($type)] {
	if {$character == "(-)"} {
	    continue
	} elseif {[info exists specialChars::Keys($character)]} {
	    set specialCharBindings($character) [set specialChars::Keys($character)]
	} elseif {[info exists specialChars::Defaults($character)]} {
	    set specialCharBindings($character) [set specialChars::Defaults($character)]
	} else {
	    set specialCharBindings($character) ""
	} 
    } 
    set title "'[string trim $type]'  key bindings "
    catch {dialog::arrayBindings $title specialCharBindings 1}

    foreach character [set specialChars::TypeChar($type)] {
        if {[info exists specialCharBindings($character)]} {
	    set newBinding $specialCharBindings($character)
	    # Check to see if this is different from the default.
	    if {[info exists specialChars::Defaults($character)]} {
		set defaultBinding [set specialChars::Defaults($character)]
	    } else {
	        set defaultBinding ""
	    }
	    if {$newBinding != $defaultBinding} {
		set specialChars::Keys($character) $newBinding
	    }
	} else {
	    catch {unset specialChars::Keys($character)}
	} 
    } 
    prefs::modified specialChars::Keys 
    # Now offer the list pick again.
    catch {specialChars::assignBindings "Select another menu, or cancel:"}
    menu::buildSome "specialCharacters "
    status::msg "New bindings have been assigned, and appear in the \"Special Characters\" menus."
}

##
 # -------------------------------------------------------------------------
 # 
 # "specialChars::unsetBindings" --
 # 
 # Offer the list of all user-defined bindings to the user to unset.  After
 # unsetting them, any binding defined in "specialChars::Defaults" will be
 # used in the menus.
 # 
 # -------------------------------------------------------------------------
 ##

proc specialChars::unsetBindings {{characterList ""}} {
    
    global specialChars::Keys 
    
    set charactersWithKeys [lsort [array names specialChars::Keys]]
    if {![llength $charactersWithKeys]} {
	status::msg "Cancelled -- there are no assigned bindings to unset."
	return
    } 
    if {$characterList == ""} {
	set title "Choose some bindings to unset"
	set characterList [listpick -l -p $title $charactersWithKeys]
    } 
    foreach character $characterList {catch {unset specialChars::Keys($character)}} 
    prefs::modified specialChars::Keys 
    menu::buildSome "specialCharacters "
    if {[llength $characterList] == 1} {
	status::msg "The binding for $characterList has been unset."
    } elseif {[llength $characterList] > 1} {
	status::msg "The bindings for $characterList have been unset."
    } 
}

# ===========================================================================
# 
#  version history  #
# 
#  modified by  vers#  reason
#  -------- --- ------ -----------
#  08/14/01 cbu 0.1    Created package, based on characters contained in the
#                        var "text::_Ascii", as defined in "stringsLists.tcl"
# 

# ===========================================================================
# 
# .
