## -*-Tcl-*- (install)
 # ###################################################################
 #  Alpha extension 
 # 
 #  FILE: "vcLocal.tcl"
 #                                    created: 03/27/2000 {14:57:19 PM} 
 #                                last update: 10/05/2001 {17:09:23 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 2000  Vince Darley.
 # 
 # Redistributable under standard 'Tcl' license.
 # 
 # Not all of the logic here is 100% correct, so use with caution.
 # 
 # ###################################################################
 ##

# feature declaration
alpha::extension vcLocal 0.1a2 {
    # One time init
    vcs::register Local vclocal
    hook::register vcsSystemModified Local vclocal::attachToFileset
    fileset::attachAdditionalInformation "Version Control System" \
      Local folder "Local Repository"
} maintainer {
    {Vince Darley} vince@santafe.edu <http://www.santafe.edu/~vince/>
} help {
    Local version control is the simple situation in which you maintain
    one file-tree as the 'repository' which you do not edit, and another
    duplicate file-tree which you do edit.  Files in the duplicate tree
    may be checked in/out of the original when they are ready.  Also
    useful if you are trying to gradually merge your source tree with
    someone else's.
    
    It is an experimental way of dealing with development versus
    robust code.
} uninstall {this-file}

namespace eval vclocal {}

proc vclocal::attachToFileset {fset} {
    # no-op right now.
}

## 
 # -------------------------------------------------------------------------
 # 
 # "vclocal::getRepositoryFile" --
 # 
 #  For a file in the current fileset, under Local version control,
 #  find the equivalent file in the repository.  The actual file
 #  need not exist, since everything is handled via relative paths.
 # -------------------------------------------------------------------------
 ##
proc vclocal::getRepositoryFile {name} {
    set vcHierarchy [vcs::getFilesetInfo "Local Repository"]
    if {![string length $vcHierarchy]} {
	error "The local repository has not been set"
    }
    set relative [fileset::relativePath $name]
    set repositoryFile [file join $vcHierarchy $relative]
}

proc vclocal::getState {name} {
    set repositoryFile [vclocal::getRepositoryFile $name]
    if {[file exists $repositoryFile]} {
	# it's under version control, must check mod dates
	switch -- [file::compareModifiedDates $name $repositoryFile] {
	    -1 {
		return "needs-patch"
	    }
	    0 {
		return "up-to-date"
	    }
	    1 {
		return "checked-out"
	    }
	}
    } else {
	return "no-vcs"
    }
}

proc vclocal::otherCommands {state} {
    return [list updateAll]
}

proc vclocal::updateAll {name} {
    message "vclocal update all: unimplemented"
}

proc vclocal::lock {name} {
    vcs::lock $name
}

proc vclocal::unlock {name} {
    vcs::unlock $name
}

proc vclocal::checkIn {name} {
    set repositoryFile [vclocal::getRepositoryFile $name]
    if {[file exists $repositoryFile]} {
	file delete $repositoryFile
    }
    file copy $name $repositoryFile
    vclocal::lock $name
    message "vclocal checkin: done"
}

proc vclocal::checkOut {name} {
    vclocal::unlock $name
    message "vclocal checkout: done"
}

proc vclocal::undoCheckout {name} {
    vclocal::lock $name
    message "vclocal undoCheckout: done"
}

proc vclocal::add {name} {
    set repositoryFile [vclocal::getRepositoryFile $name]
    file copy $name $repositoryFile
    message "vclocal add: successful"
}

proc vclocal::showDifferences {name} {
    set repositoryFile [vclocal::getRepositoryFile $name]
    compare::files $name $repositoryFile 0
}

proc vclocal::refetchReadOnly {name} {
    set repositoryFile [vclocal::getRepositoryFile $name]
    set mod [file::compareModifiedDates $name $repositoryFile]
    if {$mod == 1} {
	if {[dialog::yesno -y "Overwite" -n "Cancel operation" "Are you sure?  The current file is newer"]} {
	    set mod 0
	} else {
	    message "vclocal refetchReadOnly: cancelled"
	    return
	}
    }
    if {!$mod} {
	file delete $name
	file copy $repositoryFile $name
	revert -w $name
	message "vclocal refetchReadOnly: replaced with repository version"
    } else {
	message "vclocal refetchReadOnly: no new version available"
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "vclocal::getMenuItems" --
 # 
 #  Returns menu items pertinant to local version control
 # -------------------------------------------------------------------------
 ##
proc vclocal::getMenuItems {state} {
    set res [vcs::generalMenuItems $state]
    if {[llength res]} {
        lappend res "(-)"
    } 
    lappend res updateAll
    
    return $res
}

