## -*-Tcl-*-
 # -------------------------------------------------------------------------
 # 
 #  FILE: "backCompatibility.tcl"
 #                                    created: 04/04/01 {07:13:29 pm} 
 #                                last update: 09/12/2001 {16:34:07 PM} 
 #                                
 # You should avoid calling any of these procs:
 # They are slower, and may be removed at some point in the future.
 # 
 # Some of these are currently called from Alpha's main menus.  This
 # will be changed in the future, so that this entire file can vanish.
 # 
 # just wrappers around other procs.
 # 
 # -------------------------------------------------------------------------
 ##

## 
 # -------------------------------------------------------------------------
 # 
 # "warningForObsProc" --
 # 
 # Open a new window with the name of the obsolete proc called, and
 # (hopefully) the proc which called it.  "warningForObsoleteProcedures" is
 # set in the alphaDeveloper menu.
 # 
 # Some of the procs contained in this file are called by Alpha's core, and
 # thus should NOT include the warning.  These include:
 # 
 # pushMark 
 # popMark 
 # pushPosition 
 # popPosition
 # getIncludeFiles
 # editIncludeFile
 # -------------------------------------------------------------------------
 ##

proc warningForObsProc {{type "obsolete"}} {
    global warningForObsoleteProcedures
    
    ensureset warningForObsoleteProcedures 0
    if {!$warningForObsoleteProcedures} {return}
    # Note: if we call this proc at the global level, this line
    # will be an error, since level -1 doesn't exist.
    set what [info level -1]
    if {[set which [info level 1]] == $what} {
	set which "Some code"
    } else {
	set which "'$which'"
    }
    if {$type == "obsolete"} {
	set depend "deprecated procedures"
    } else {
	set depend "unsupported procedures (on this platform)"
    }

    set msg "$which is trying to execute '$what'; please try to\
      remove that dependence on $depend, or inform\
      the alphatcl-developers mailing list."

    if {[dialog::yesno -y "Throw an error" -n "Continue" $msg]} {
	# We throw an error so the user can capture the entire
	# stack back-trace, to discover exactly how this code is
	# called.
	return -code error $msg
    } else {
	# The user already has a warning, so probably no need
	# to open a window for them.
	return
	if {[llength [winNames -f]]} {set currWin [win::Current]}
	global errorInfo
	set errorInfo $msg
	alphadev::dumpErrorInfo
	if {[info exists currWin]} {
	    bringToFront $currWin
	}
    }
}

proc realMenuName {name} {
    warningForObsProc
    global subMenuInfo
    return [lindex $subMenuInfo($name) 1]
}

proc killWindowStatus {} {
    warningForObsProc
    closeAWindow
}
proc chooseWindowStatus {} {
    warningForObsProc
    chooseAWindow
}

proc sPromptChoices {msg def choices} {
    warningForObsProc
    uplevel 1 prompt::fromChoices [list $msg $def -list $choices]
}

proc sPrompt {msg def} {
    warningForObsProc
    global useStatusBarForPrompts
    if {!$useStatusBarForPrompts} {return [prompt $msg $def]}
    if {[catch {statusPrompt "$msg ($def): "} ans]} {
	error "cancel"
    }
    if {![string length $ans]} {return $def}
    return $ans
}

# This is superceded by more general completion/prompt routines which
# do the same thing when told to use '-command winNames'.
proc winComp {curr c} {
    warningForObsProc
    if {$c != "\t"} {return $c}
    
    set matches {}
    foreach w [winNames] {
	if {[string match "$curr*" $w]} {
	    lappend matches $w
	}
    }
    if {![llength $matches]} {
	error "No match"
    } else {
	return [string range [largestPrefix $matches] [string length $curr] end]
    }
    return ""
}

#  File and Help  #

namespace eval file {}
namespace eval help {}

proc winIsFile {name} {
    warningForObsProc
    win::IsFile $name 
}

proc file::searchAndHyperise {args} {
    warningForObsProc
    uplevel 1 win::searchAndHyperise $args
}
proc file::multiSearchAndHyperise {args} {
    warningForObsProc
    uplevel 1 win::multiSearchAndHyperise $args
}
proc file::jumpToCode {args} {
    warningForObsProc
    uplevel 1 win::jumpToCode $args
}

proc file::hyperHelpFileOpen {name} {
    warningForObsProc
    help::openGeneral $name
}
proc file::hyperOpen {name} {
    warningForObsProc
    help::openHyper $name
}
proc file::hyperHelpOpen {name} {
    warningForObsProc
    help::openGeneral $name
}
proc file::hyperExampleOpen {{f ""}} {
    warningForObsProc
    help::openExample $f
}

proc openFolder {args} {
    warningForObsProc
    uplevel 1 file::showInFinder $args
}

proc pushMark {} {pushPosition}
proc popMark {} {popPosition}
proc pushPosition {args} {uplevel placeBookmark $args}
proc popPosition {args} {uplevel returnToBookmark $args}

proc absolutePath {filename} {
    warningForObsProc
    file::absolutePath $filename
}
proc openFileQuietly {filename} {
    warningForObsProc
    edit -c -w $filename
}
proc searchInFile {filename searchString {indices 0}} {
    warningForObsProc
    file::searchFor $filename $searchString $indices
}

proc readFile {fileName} {
    warningForObsProc
    file::readAll $fileName
}
proc writeFile {filename {text {}} {overwrite 0}} {
    warningForObsProc
    file::writeAll $filename $text $overwrite
}
proc gotoFileLine {fname line {mesg {}}} {
    warningForObsProc
    file::gotoLine $fname $line $mesg
}

proc buildSubMenu {args} {
    warningForObsProc
    eval menu::buildHierarchy $args
}

proc listSubfolders {folder {depth 3}} {
    warningForObsProc
    file::hierarchy $folder $depth
}

proc getIncludeFiles {} {
    optClickTB_List
}
proc editIncludeFile {item} {
    optClickTB_Pick $item
}
proc tryElseDump {name script} {
    warningForObsProc
    try::level \#0 $script -reporting log -while $name
}
proc text::genericIndent {} {
    warningForObsProc
    ::indentLine
}

proc revertTheseFiles {flist} {
    warningForObsProc
    eval file::revertThese $flist
}

#  Comment and Paragraph  #

proc file::commentTextBlock {args} {
    warningForObsProc
    eval comment::TextBlock $args
}
proc commentLine  {args} {
    warningForObsProc
    uplevel 1 comment::Line $args
}
proc uncommentLine  {args} {
    warningForObsProc
    uplevel 1 comment::undoLine $args
}
proc commentBox  {args} {
    warningForObsProc
    uplevel 1 comment::Box $args
}
proc uncommentBox  {args} {
    warningForObsProc
    uplevel 1 comment::undoBox $args
}
proc commentCharacters {args} {
    warningForObsProc
    uplevel 1 comment::Characters $args
}
proc commentGetRegion  {args} {
    warningForObsProc
    uplevel 1 comment::GetRegion $args
}
proc commentSameStart  {args} {
    warningForObsProc
    uplevel 1 comment::SameStart $args
}
proc commentTextBlock  {args} {
    warningForObsProc
    uplevel 1 comment::TextBlock $args
}
proc commentGetFillLines {args} {
    warningForObsProc
    uplevel 1 comment::GetFillLines $args
}
proc commentParagraph  {args} {
    warningForObsProc
    uplevel 1 comment::Paragraph $args
}
proc uncommentParagraph {args} {
    warningForObsProc
    uplevel 1 comment::undoParagraph $args
}
proc fillParagraph {args} {
    warningForObsProc
    uplevel 1 paragraph::fill $args
}
proc fillOneParagraph {args} {
    warningForObsProc
    uplevel 1 paragraph::fillOne $args
}
proc paraStart {args} {
    warningForObsProc
    uplevel 1 paragraph::start $args
}
proc paraFinish {args} {
    warningForObsProc
    uplevel 1 paragraph::finish $args
}
proc sentenceParagraph {args} {
    warningForObsProc
    uplevel 1 paragraph::sentence $args
}
proc selectParagraph {args} {
    warningForObsProc
    uplevel 1 paragraph::select $args
}
proc addArrDef {args} {
    warningForObsProc
    uplevel 1 prefs::addArrayElement $args
}
proc removeArrDef {args} {
    warningForObsProc
    uplevel 1 prefs::removeArrayElement $args
}
proc addDef {args} {
    warningForObsProc
    uplevel 1 prefs::add $args
}
proc removeDef {args} {
    warningForObsProc
    uplevel 1 prefs::remove $args
}
proc readDefs {args} {
    warningForObsProc
    uplevel 1 prefs::_read $args
}
proc writeDefs {args} {
    warningForObsProc
    uplevel 1 prefs::_write $args
}
proc addArr {args} {
    warningForObsProc
    uplevel 1 prefs::addArray $args
}
proc removeArr {args} {
    warningForObsProc
    uplevel 1 prefs::removeArray $args
}
proc saveModifiedVars {args} {
    warningForObsProc
    uplevel 1 prefs::saveModified $args
}
proc alpha::readUserDefs {} {
    warningForObsProc
    uplevel 1 prefs::readAll
}
proc alpha::readUserPrefs {} {
    warningForObsProc
    uplevel 1 prefs::tclRead
}
proc addUserLine {args} {
    warningForObsProc
    uplevel 1 prefs::tclAddLine $args
}

namespace eval mode {}

proc mode::addUserLine {args} {
    warningForObsProc
    uplevel 1 prefs::tclAddModeLine $args
}
proc getSavedSettings {} {
    warningForObsProc
    prefs::listAllSaved
}

proc upBrowse {} {
    warningForObsProc
    browse::Up
}
proc downBrowse {} {
    warningForObsProc
    browse::Down
}
proc gotoMatch {} {
    warningForObsProc
    browse::Goto
}

proc stripNameCount str {
    warningForObsProc
    win::StripCount $str
}

########################################
#			               #
#    A few random lisp'ish functions.  #
#				       #
########################################

proc car     {l} {lindex $l 0}
proc cadr    {l} {lindex $l 1}
proc caddr   {l} {lindex $l 2}
proc cadddr  {l} {lindex $l 3}
proc caddddr {l} {lindex $l 4}
proc cdr     {l} {lrange $l 1 end}
proc cddr    {l} {lrange $l 2 end}

proc mapcar args {return [eval map $args]}

proc map {func l} {
    set out {}
    foreach el $l {
	lappend out [eval $func [list $el]]
    }
    return $out
}

proc cons {e l} {concat [list $e] $l}

#  Legacy TclAE Definitions  #

## 
 # -------------------------------------------------------------------------
 # 
 # "aebuild::*" --
 # 
 #  Wrapper routines for tclAE::build::*. Don't call these in new code.
 # 
 # -------------------------------------------------------------------------
 ##

namespace eval aebuild {}

proc aebuild::result {args} {
    warningForObsProc
    return [eval tclAE::build::resultData $args]
}

proc aebuild::objectProperty {process property object} {
    warningForObsProc
    return [tclAE::build::objectProperty $process $property $object]
}

proc aebuild::coercion {type value} {
    warningForObsProc
    return [tclAE::build::coercion $type $value]
}

proc aebuild::list {l args} {
    warningForObsProc
    return [eval tclAE::build::List [list $l] $args]
}

proc aebuild::hexd {value} {
    warningForObsProc
    return [tclAE::build::hexd $value]
}

proc aebuild::bool {val} {
    warningForObsProc
    return [tclAE::build::bool $val]
}

proc aebuild::TEXT {str} {
    warningForObsProc
    return [tclAE::build::TEXT $str]
}

proc aebuild::alis {path} {
    warningForObsProc
    return [tclAE::build::alis $path]
}

proc aebuild::fss {value} {
    warningForObsProc
    return [tclAE::build::fss $value]
}

proc aebuild::name {name} {
    warningForObsProc
    return [tclAE::build::name $name]
}

proc aebuild::filename {name} {
    warningForObsProc
    return [tclAE::build::filename $name]
}

proc aebuild::winByName {name} {
    warningForObsProc
    return [tclAE::build::winByName $name]
}

proc aebuild::winByPos {absPos} {
    warningForObsProc
    return [tclAE::build::winByPos $absPos]
}

proc aebuild::lineRange {absPos1 absPos2} {
    warningForObsProc
    return [tclAE::build::lineRange $absPos1 $absPos2]
}

proc aebuild::absPos {posName} {
    warningForObsProc
    return [tclAE::build::absPos $posName]
}

proc aebuild::startupDisk {} {
    warningForObsProc
    return [tclAE::build::startupDisk]
}

proc aebuild::userName {} {
    warningForObsProc
    return [tclAE::build::userName]
}


namespace eval aeparse {}

proc aeparse::event {chars args} {
    warningForObsProc
    return [eval tclAE::parse::event [list $chars] $args]
}

proc aeparse::keywordValue {keyword record {typed 0}} {
    warningForObsProc
    if {$typed} {
	return [tclAE::getKeyDesc $record $keyword]
    } else {
	return [tclAE::getKeyData $record $keyword]
    }
}


namespace eval aecoerce {}

proc aecoerce::identity {value} {
    warningForObsProc
    return $value
}

proc aecoerce::hexd:bool {value} {
    warningForObsProc
    binary scan $value c bool
    
    if {![info exists bool]
    ||	($bool != 0 && $bool != 1)} {
	error::throwOSErr -1700
    }
    
    return $bool
}

proc aecoerce::hexd:TEXT {value} {
    warningForObsProc
    binary scan $value a* TEXT
    return $TEXT
}

proc aecoerce::hexd {value} {
    warningForObsProc
    error "I have no idea what this is supposed to do"
}

proc aecoerce::null:TEXT {value} {
    warningForObsProc
    return [tclAE::coerce::null>hexd $value]
}

proc aecoerce::hexd:alis {value} {
    warningForObsProc
    set resultDesc [tclAE::createDesc alis]
    tclAE::coerce::TEXT>alis TEXT $value alis $resultDesc
    set result [tclAE::print $resultDesc]
    tclAE::disposeDesc $resultDesc
    
    return $result
}

proc aecoerce::TEXT:alis {value} {
    warningForObsProc
    return [aecoerce::hexd:alis [binary format a* $value]]
}

proc aecoerce::register {args} {
    warningForObsProc
    return [eval tclAE::installCoercionHandler $args]
}

proc aecoerce::apply {AEDesc toType} {
    warningForObsProc
    set newDesc [tclAE::coerceDesc $AEDesc $toType]
    set result [tclAE::getData $newDesc]
    tclAE::disposeDesc $newDesc
    
    return $result
}
	
proc aecoerce::deregister {hook {procname ""} args} {
    warningForObsProc
    error "There is no way to make \[aecoerce::deregister\] compatible with modern TclAE. \
      The code must be changed to use \[tclAE::removeCoercionHandler\]"
}

# I don't believe these are used anywhere, but put them here 
# to avoid the 7.4b17 fiasco

namespace eval tclAE {}
namespace eval tclAE::coerce {}

proc tclAE::coerce::null>TEXT {value} {
    warningForObsProc
    return ""
}

## 
 # bool ::= bool(00|01)
 ##
proc tclAE::coerce::_hexd>bool {value} {
    warningForObsProc
    set value [tclAE::coerce::_long>hexd $value]
    set bool [expr {"0x$value"}]
    if {($bool != 0) && ($bool != 1)} {
	set msg "Can't coerce $value from 'hexd' to 'bool'"
	error $msg "" [list AECoerce -1700 $msg]
    } 
    return $bool
}

proc tclAE::coerce::_****>hexd {value} {
    warningForObsProc
    set newval $value
    if {[expr {[string length $newval] % 2}]} {
	# left pad with zero to make even number of digits
	set newval "0${newval}"
    } 
    if {![is::Hexadecimal $newval]} {
	set msg "Non-hex-digit in ${value}" 
	error $msg "" [list AECoerce 6 $msg]
    } else {
	return ${newval}
    }
}

proc tclAE::coerce::_hexd>TEXT {value} {
    warningForObsProc
    # make sure input is really hexd
    set value [tclAE::coerce::_****>hexd $value]
    
    set TEXT ""
    set length [string length $value]
    set i 0
    while {$i < $length} {
	append TEXT [uplevel 0 "set temp \\x[string range $value $i [incr i]]"]
	incr i
    }
    return $TEXT
}

proc tclAE::coerce::_long>hexd {value} {
    warningForObsProc
    set newval [format "%08X" $value]
    return [tclAE::coerce::_****>hexd $newval]
}

proc tclAE::coerce::_shor>hexd {value} {
    warningForObsProc
    set newval [format "%04X" $value]
    return [tclAE::coerce::_****>hexd $newval]
}

if {[info tclversion] < 8.0} {
    proc tclAE::coerce::_TEXT>hexd {value} {
	warningForObsProc
	set length [string length $value]
	set hexd ""
	for {set i 0} {$i < $length} {incr i} {
	    if {[set char [string index $value $i]] == "\x00"} {
		# scan doesn't work on \x00, so we handle it specially.
		append hexd "00"
	    } else {
		scan $char "%c" char
		append hexd [format "%02X" $char]                
	    } 
	}	
	
	return $hexd
    }    
} else {
    proc tclAE::coerce::_TEXT>hexd {value} {
	warningForObsProc
	binary scan $value H* hexd
	return $hexd
    }
}


# mtime doesn't understand hex notation,
# so we force to decimal first
proc tclAE::coerce::_hexd>ldt {hexd} {
    warningForObsProc
    return [join [mtime [tclAE::coerce::hexd>long $hexd] short]]
}

