#  AlphaTcl - core Tcl engine

namespace eval comment {}

proc comment::Line {} {
    global mode
    if {![catch {comment::Characters Paragraph} chars]} {
	set start [lindex $chars 0]
	set end [lindex $chars 1]
 	if {[string trim $start] == [string trim $end]} {
	    insertPrefix
	} else {
	    set ext  [file extension [win::CurrentTail]]
	    if {($mode == "C" || $mode == "C++") && $ext != ".h" && $ext != ".c"} {
		insertPrefix
	    } else {
                if {[pos::compare [getPos] == [selEnd]]} {
                    beginningOfLine
                    insertText $start
                    endOfLine
                    insertText $end
                    beginningOfLine
                } else {
                    comment::Paragraph
                }
	    }
	}
    } else {
	insertPrefix
    }
}

proc comment::undoLine {} {
    global mode
    if {![catch {comment::Characters Paragraph} chars]} {
	set cBeg [lindex $chars 0]
	set cEnd [lindex $chars 1]
	if {[string trim $cBeg] == [string trim $cEnd]} {
	    removePrefix
	} else {
	    set ext  [file extension [win::CurrentTail]]
	    if {($mode == "C" || $mode == "C++") && $ext != ".h" && $ext != ".c"} {
		removePrefix
	    } else {
		if {[pos::compare [set pos [getPos]] == [selEnd]]} {
		    set start [lineStart $pos]
		    set end [nextLineStart $pos]
		    set text [getText $start $end]
		    replaceText $start $end [doPrefixText remove $cBeg $text]
		    set end [nextLineStart $start]
		    set text [getText $start $end]
		    set text [doSuffixText remove $cEnd $text]
		    replaceText $start $end $text
		    goto $start
		} else {
		    comment::undoParagraph
		    
		}
	    }
	}
    } else {
	removePrefix
    }
}

proc comment::Box {} {

    # Preliminaries
    if {[comment::GetRegion Box]} { return }
	
    set commentList [comment::Characters Box]
    if {![llength $commentList]} { return }
    
    set begComment [lindex $commentList 0]
    set begComLen [lindex $commentList 1]
    set endComment [lindex $commentList 2]
    set endComLen [lindex $commentList 3]
    set fillChar [lindex $commentList 4]
    set spaceOffset [lindex $commentList 5]

    set aSpace " "

    # First make sure we grab a full block of lines and adjust highlight

    set start [getPos]
    set start [lineStart $start]
    set end [selEnd]
    set end [nextLineStart [pos::math $end - 1]]
    select $start $end

    # Now get rid of any tabs
	
    if {[pos::compare $end < [maxPos]]} {
	createTMark stopComment [pos::math $end + 1]
	tabsToSpaces
	gotoTMark stopComment
	set end [pos::math [getPos] - 1]
	removeTMark stopComment
    } else {
	tabsToSpaces
	set end [maxPos]
    }
    select $start $end
    set text [getText $start $end]
	
# Next turn it into a list of lines--possibly drop an empty 'last line'

    set lineList [split $text "\r\n"]
    set numLines [llength $lineList]
    if {[lindex $lineList end] == {} } {
	set lineList [lrange $lineList 0 [expr {$numLines -2}]]
	set numLines [llength $lineList]
    }

# Find the longest line length and determine the new line length

    set maxLength 0
    foreach thisLine $lineList {
	set thisLength [string length $thisLine]
	if { $thisLength > $maxLength } { 
	    set maxLength $thisLength 
	}
    }
    set newLength [expr {$maxLength + 2 + 2*$spaceOffset}]
	
    # Now create the top & bottom bars and a blank line

    set topBar $begComment
    for { set i 0 } { $i < [expr {$newLength - $begComLen}] } { incr i } {
	append topBar $fillChar
    }
    set botBar ""
    for { set i 0 } { $i < [expr {$newLength - $endComLen}] } { incr i } {
	append botBar $fillChar
    }
    append botBar $endComment
    set blankLine $fillChar
    for { set i 0 } { $i < [expr {$newLength - 2}] } { incr i } {
	append blankLine " "
    }
    append blankLine $fillChar
	
    # For each line add stuff on left and spaces and stuff on right for box sides
    # and concatenate everything into 'text'.  Start with topBar; end with botBar

    set text $topBar\r$blankLine\r
	
    set frontStuff $fillChar
    set backStuff $fillChar
    for { set i 0 } { $i < $spaceOffset } { incr i } {
	append frontStuff " "
	set backStuff $aSpace$backStuff
    }
    set backStuffLen [string length $backStuff]
	
    foreach thisLine $lineList {
	set thisLine $frontStuff$thisLine
	set thisLength [string length $thisLine]
	set howMuchPad [expr {$newLength - $thisLength - $backStuffLen}]
	for { set j 0 } { $j < $howMuchPad } { incr j } {
	    append thisLine " "
	}
	append thisLine $backStuff
	append text $thisLine \r
    }
    
    append text $blankLine \r $botBar \r
	
# Now replace the old stuff, turn spaces to tabs, and highlight

    replaceText	$start $end $text
    set	end [pos::math $start + [string length $text]]
    frontSpacesToTabs $start $end
}

proc comment::undoBox {} {

# Preliminaries
    if {[comment::GetRegion Box 1]} { return }
	
    set commentList [comment::Characters Box]
    if {![llength $commentList]} { return }
	
    set	begComment [lindex $commentList	0]
    set	begComLen [lindex $commentList 1]
    set	endComment [lindex $commentList	2]
    set	endComLen [lindex $commentList 3]
    set	fillChar [lindex $commentList 4]
    set	spaceOffset [lindex $commentList 5]
    
    set aSpace " "
    set aTab \t

    # First make sure we grab a full block of lines

    set start [getPos]
    set start [lineStart $start]
    set end [selEnd]
    set end [nextLineStart [pos::math $end - 1]]
    set text [getText $start $end]

    # Make sure we're at the start and end of the box

    set startOK [string first $begComment $text]
    set endOK [string last $endComment $text]
    set textLength [string length $text]
    if { $startOK != 0 || ($endOK != [expr {$textLength-$endComLen-1}] || $endOK == -1) } {
	alertnote "You must highlight the entire comment box, including the borders."
	return
    }
	
    # Now get rid of any tabs
	
    if {[pos::compare $end < [maxPos]] } {
	createTMark stopComment [pos::math $end + 1]
	tabsToSpaces
	gotoTMark stopComment
	set end [pos::math [getPos] - 1]
	removeTMark stopComment
    } else {
	tabsToSpaces
	set end [maxPos]
    }
    select $start $end
    set text [getText $start $end]
	
# Next turn it into a list of lines--possibly drop an empty 'last line'

# VMD May'95: changed this code segment because it previously had problems
# with empty lines in the middle of the text to be commented

    set lineList [split $text "\n\r"]
    set ll [llength $lineList]
    if { [lindex $lineList end] == {} } {
	set lineList [lrange $lineList 0 [expr {$ll -2}] ]
    }
    set numLines [llength $lineList]

# end changes.
	
# Delete the first and last lines, recompute number of lines

    set lineList [lreplace $lineList [expr {$numLines-1}] [expr {$numLines-1}] ]
    set lineList [lreplace $lineList 0 0 ]
    set numLines [llength $lineList]
	
    # Eliminate 2nd and 2nd-to-last lines if they are empty

    set eliminate $fillChar$aSpace$aTab
    set thisLine [lindex $lineList [expr {$numLines-1}]]
    set thisLine [string trim $thisLine $eliminate]
    if { [string length $thisLine] == 0 } {
	set lineList [lreplace $lineList [expr {$numLines-1}] [expr {$numLines-1}] ]
    }
    set thisLine [lindex $lineList 0]
    set thisLine [string trim $thisLine $eliminate]
    if { [string length $thisLine] == 0 } {
	set lineList [lreplace $lineList 0 0 ]
    }
    set numLines [llength $lineList]	
	
# For each line trim stuff on left and spaces and stuff on right and splice

    set dropFromLeft [expr {$spaceOffset+1}]
    set text ""
    foreach thisLine $lineList {
	set thisLine [string trimright $thisLine $eliminate]
	set thisLine [string range $thisLine $dropFromLeft end]
	append text $thisLine \r
    }
		
    # Now replace the old stuff, convert spaces back to tabs

    replaceText	$start $end $text
    set end [pos::math $start + [string	length $text]]
    frontSpacesToTabs $start $end
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "comment::Characters" --
 #	
 #  Adds the 'general' purpose characters which are used to check if we're
 #  in a comment block.  Also has a check for an array entry like this:
 #	
 #	set C++::commentCharacters(General) [list "*" "//"]
 #	
 #  If such an entry exists, it is returned.  This allows mode authors to
 #  keep everything self-contained. 
 # -------------------------------------------------------------------------
 ##
proc comment::Characters {purpose} {
    global mode
    global ${mode}::commentCharacters
    # Modes must define these things.
    if {[info exists ${mode}::commentCharacters(${purpose})]} {
	return [set ${mode}::commentCharacters(${purpose})]
    }
    switch -- $purpose {
	"General" {
	    return
	}		
	"Paragraph" {		
	    error "I don't know what 'paragraph' comments should look like in this mode.  Sorry."
	}
	"Box" {
	    error "I don't know what 'box' comments should look like in this mode.  Sorry."
	}
    }	
}

## 
 # Default is to look for a paragraph to comment out.  If sent '1', then we
 # look for a commented region to uncomment.
 ##
proc comment::GetRegion { purpose {uncomment 0 } } {
    if {[pos::compare [getPos] != [selEnd]]} {
	watchCursor
	return 0
    }
    
    # there's no selection, so we try and generate one
    
    set pos [getPos]
    if {$uncomment} {
	# uncommenting
	set commentList [comment::Characters $purpose]
	if { [llength $commentList] == 0 } { return 1}
	switch -- $purpose {
	    "Box" {
		set begComment [lindex $commentList 0]
		set begComLen [lindex $commentList 1]
		set endComment [lindex $commentList 2]
		set endComLen [lindex $commentList 3]
		set fillChar [lindex $commentList 4]
		set spaceOffset [lindex $commentList 5]
		
		# get length of current line
		set line [getText [lineStart $pos] [nextLineStart $pos] ]
		set c [string trimleft $line]
		set slen [expr {[string length $line] - [string length $c]}]
		set start [string range $line 0 [expr {$slen -1 }] ]
				
		set pos [getPos]
				
		if { $start == "" } {
		    set p $pos
		    while { [string first $fillChar $line] == 0 && \
		      [expr {[string last $fillChar $line] + [string length $fillChar]}] \
		      >= [string length [string trimright $line]] } {
			set p [nextLineStart $p]
			set line [getText [lineStart $p] [nextLineStart $p]]
		    }
		    set end [lineStart $p]
		    
		    set p $pos
		    set line "${fillChar}"
		    while { [string first $fillChar $line] == 0 && \
		      [expr {[string last $fillChar $line] + [string length $fillChar]}] \
		      >= [string length [string trimright $line]] } {
			set p [prevLineStart $p]
			set line [getText [prevLineStart $p] [lineStart $p] ]
		    }
		    set begin [prevLineStart $p]
		    
		} else {
		    set line "$start"
		    set p $pos
		    while { [string range $line 0 [expr {$slen -1}] ] == "$start" } {
			set p [nextLineStart $p]
			set line [getText [lineStart $p] [nextLineStart $p]]
		    }
		    set end [prevLineStart $p]
		    
		    set p $pos
		    set line "$start"
		    while { [string range $line 0 [expr {$slen -1}] ] == "$start" } {
			set p [prevLineStart $p]
			set line [getText [prevLineStart $p] [lineStart $p] ]
		    }
		    set begin [lineStart $p]
		}
		
		set beginline [getText $begin [nextLineStart  $begin]]
		if { [string first "$begComment" "$beginline" ] != $slen } {
		    message "First line failed"
		    return 1
		}
		
		set endline [getText $end [nextLineStart $end]]
		set epos [string last "$endComment" "$endline"]
		incr epos [string length $endComment]
		set s [string range $endline $epos end ]
		set s [string trimright $s]
		
		if { $s != "" } {
		    message "Last line failed"
		    return 1
		}
		
		set end [nextLineStart $end]
		select $begin $end
		#alertnote "Sorry auto-box selection not yet implemented"
	    }
	    "Paragraph" {
		set begComment [lindex $commentList 0]
		set endComment [lindex $commentList 1]
		set fillChar [lindex $commentList 2]
				
		## 
		 # basic idea is search back and forwards for lines that
		 # don't begin the same way and then see if they match the
		 # idea of the beginning and end of a block
		 ##
		
		set line [getText [lineStart $pos] [nextLineStart $pos] ]
		set chk [string range $line 0 [string first $fillChar $line]]
		if { [string trimleft $chk] != "" } {
		    message "Not in a comment block"
		    return 1
		}
		regsub -all {	} $line " " line
		set p [string first "$fillChar" "$line"]
		set start [string range "$line" 0 [expr {$p + [string length $fillChar] -1}]]
		set ll [comment::GetFillLines $start]
		set begin [lindex $ll 0]
		set end [lindex $ll 1]
		
		set beginline [getText $begin [nextLineStart  $begin]]
		if {[string first "$begComment" "$beginline" ] != $p } {
		    message "First line failed"
		    return 1
		}
				
		set endline [getText $end [nextLineStart $end]]
		set epos [string last "$endComment" "$endline"]
		incr epos [string length $endComment]
		set s [string range $endline $epos end ]
		set s [string trimright $s]
		
		if { $s != "" } {
		    message "Last line failed"
		    return 1
		}
		#goto $end
		set end [nextLineStart $end]
		select $begin $end
	    }
	}
    } else {
	# commenting out
	set searchString "^\[ \t\]*\$"
	set searchResult1 [search -s -f 0 -r 1 -n $searchString $pos]
	set searchResult2 [search -s -f 1 -r 1 -n $searchString $pos]
	if {[llength $searchResult1]} {
	    set posStart [pos::math [lindex $searchResult1 1] + 1]
	} else {
	    set posStart [minPos]
	}
	if {[llength $searchResult2]} {
	    set posEnd [lindex $searchResult2 0]
	} else {
	    set posEnd [pos::math [maxPos] + 1]
	    goto [maxPos]
	    insertText "\n"
	}
	select $posStart $posEnd
    }
    
    set str "Do you wish to "
    if {$uncomment} { append str "uncomment" } else { append str "comment out" }
    append str " this region?"
    return [expr {![dialog::yesno $str]}]
}



proc comment::SameStart { line start } {
    regsub -all "\t" $line " " line
    if {[string first "$start" "$line"] == 0 } {
	return 1
    } else {
	return 0
    }
}

proc comment::TextBlock {text} {
    set cc [comment::Characters "Paragraph"]
    set c [lindex $cc 2]
    regsub -all "\[\r\n\]" $text "\r${c}" text
    return "[lindex $cc 0]\r[lindex $cc 2]${text}\r[lindex $cc 1]\r"
}

proc comment::GetFillLines { start } {
    set pos [getPos]
    regsub -all "\t" $start " " start
    set line "$start"
    
    set p $pos
    while { [comment::SameStart "$line" "$start"] } {
	set p [nextLineStart $p]
	set line [getText [lineStart $p] [nextLineStart $p]]
    }
    set end [lineStart $p]
    
    set p $pos
    set line "$start"
    while { [comment::SameStart "$line" "$start"] } {
	set p [prevLineStart $p]
	set line [getText [prevLineStart $p] [lineStart $p] ]
    }
    set begin [prevLineStart $p]
    return [list $begin $end]
}

## 
 # Author: Vince Darley	<mailto:vince@santafe.edu> 
 ##

proc comment::Paragraph {} {

# Preliminaries
    if {[comment::GetRegion Paragraph]} { return }
	
    set commentList [comment::Characters Paragraph]
    if { [llength $commentList] == 0 } { return }

    set begComment [lindex $commentList 0]
    set endComment [lindex $commentList 1]
    set fillChar [lindex $commentList 2]
    
    
    # First make sure we grab a full block of lines and adjust highlight
    
    set start [getPos]
    set start [lineStart $start]
    set end [selEnd]
    set end [nextLineStart [pos::math $end - 1]]
    select $start $end
    
    # Now get rid of any tabs
    
    if {[pos::compare $end < [maxPos]] } {
    	createTMark stopComment [pos::math $end + 1]
    	tabsToSpaces
    	gotoTMark stopComment
    	set end [pos::math [getPos] - 1]
    	removeTMark stopComment
    } else {
    	tabsToSpaces
    	set end [maxPos]
    }
    select $start $end
    set text [getText $start $end]
	
    # Next turn it into a list of lines--possibly drop an empty 'last line'

    set lineList [split $text "\r\n"]
    set ll [llength $lineList]
    if { [lindex $lineList end] == {} } {
        set lineList [lrange $lineList 0 [expr {$ll -2}] ]
    }
    set numLines [llength $lineList]
    
    # Find left margin for these lines
    set lmargin 100
    foreach l $lineList {
    	set lm [expr {[string length $l] - [string length [string trimleft $l]]}]
    	if { $lm < $lmargin } { set lmargin $lm }
    }
    set ltext ""
    for { set i 0 } { $i < $lmargin } { incr i } {
    	append ltext " "
    }
    
    # For each line add stuff on left and concatenate everything into 'text'. 
    
    set text ${ltext}${begComment}\r
    
    foreach l $lineList {
    	append text ${ltext} ${fillChar} [string range $l $lmargin end] \r
    }
    append text ${ltext} ${endComment} \r
    
    # Now replace the old stuff, turn spaces to tabs, and highlight
    
    replaceText $start $end $text
    set end [pos::math $start + [string length $text]]
    frontSpacesToTabs $start $end
    select [lineStart $start] [pos::math [nextLineStart $end] -1]
}

## 
 # Author: Vince Darley	<vince@santafe.edu>
 ##

proc comment::undoParagraph {} {

    # Preliminaries
    if {[comment::GetRegion Paragraph 1]} { return }
    
    set commentList [comment::Characters Paragraph]
    if { [llength $commentList] == 0 } { return }
    
    set begComment [lindex $commentList 0]
    set endComment [lindex $commentList 1]
    set fillChar [lindex $commentList 2]
    
    set aSpace " "
    set aTab \t
    
    # First make sure we grab a full block of lines and adjust highlight
    
    set start [getPos]
    set start [lineStart $start]
    set end [selEnd]
    set end [nextLineStart [pos::math $end - 1]]
    select $start $end
    set text [getText $start $end]
    
    # Find left margin for these lines
    set l [string range $text 0 [string first "\r" $text] ]
    set lmargin [expr {[string length $l] - [string length [string trimleft $l]]}]
    
    # Make sure we're at the start and end of the paragraph

    set startOK [string first $begComment $text]
    set endOK [string last $endComment $text]
    set textLength [string length $text]
    if { $startOK != $lmargin || ($endOK != [expr {$textLength-[string length $endComment]-1}] || $endOK == -1) } {
    	alertnote "You must highlight the entire comment paragraph, including the tail ends."
    	return
    }
    
    # Now get rid of any tabs
    
    if {[pos::compare $end < [maxPos]]} {
    	createTMark stopComment [pos::math $end + 1]
    	tabsToSpaces
    	gotoTMark stopComment
    	set end [pos::math [getPos] - 1]
    	removeTMark stopComment
    } else {
    	tabsToSpaces
    	set end [maxPos]
    }
    select $start $end
    set text [getText $start $end]
    
    # Next turn it into a list of lines--possibly drop an empty 'last line'
    
    set lineList [split $text "\r\n"]
    set ll [llength $lineList]
    if { [lindex $lineList end] == {} } {
    	set lineList [lrange $lineList 0 [expr {$ll -2}] ]
    }
    set numLines [llength $lineList]
	
    # Delete the first and last lines, recompute number of lines
    
    set lineList [lreplace $lineList [expr {$numLines-1}] [expr {$numLines-1}] ]
    set lineList [lreplace $lineList 0 0 ]
    set numLines [llength $lineList]
    
    # get the left margin
    set lmargin [string first $fillChar [lindex $lineList 0]]
    set ltext ""
    for { set i 0 } { $i < $lmargin } { incr i } {
    	append ltext " "
    }
    
    # For each line trim stuff on left and spaces and stuff on right and splice
    set eliminate $fillChar$aSpace$aTab
    set dropFromLeft [expr {[string length $fillChar] + $lmargin}]
    set text ""
    foreach thisLine $lineList {
    	set thisLine [string trimright $thisLine $eliminate]
    	set thisLine ${ltext}[string range $thisLine $dropFromLeft end]
    	append text $thisLine \r
    }
    
    # Now replace the old stuff, turn spaces to tabs, and highlight
    
    
    replaceText	$start $end $text
    set	end [pos::math $start + [string length $text]]
    frontSpacesToTabs $start $end
    select [lineStart $start] [pos::math [nextLineStart $end] -1]
}

#  Comment Templates  #
# 
# Based on the package 'newJavaDocComment', these procs will insert a new
# 'paragraph' style comment in the current window.  The template can be easily
# modified by the user in a mode specific way.
# 
# Contributed by Craig Barton Upright.
# 

## 
 # -------------------------------------------------------------------------
 # 
 # "comment::newComment" --
 # 
 # Adds a new comment at the beginning of the current command, assuming that
 # we are currently inside one.  If we end up at the top of the file, then
 # assume that the "previous" command could not be found, so simply insert
 # the new comment at the beginning of this line.
 # 
 # The "insertWhere" argument can be set to 0 if the mode uses paragraphs
 # to determine the start/end of commands, as in Fort mode, or "1" to
 # insert it at the start of a command.  "-1" simply inserts at the
 # beginning of the line containing the current insertion point.
 # 
 # -------------------------------------------------------------------------
 ##

proc comment::newComment {{insertWhere "-1"}} {
    
    global mode
    global ${mode}modeVars ${mode}::commentCharacters
    
    # Should we go to the start of this command/paragraph?.
    set pos [getPos]
    if {$insertWhere == "-1"} {
	set pos [lineStart $pos]
    } elseif {!$insertWhere} {
	set results [function::inFunction $pos]
	set result  [lindex $results 0]
	set start   [lindex $results 1]
	if {$result} {set pos $start} else {set pos [lineStart $pos]}
    } else {
	set pos [paragraph::start $pos]
    }
    # Insert the new paragraph comment template.
    if {[info exists ${mode}modeVars(commentTemplate)]} {
	set elecInsert [set ${mode}modeVars(commentTemplate)]
    } elseif {[info exists ${mode}::commentCharacters(Paragraph)]} {
	set cc1 [lindex [set ${mode}::commentCharacters(Paragraph)] 0]
	set cc2 [lindex [set ${mode}::commentCharacters(Paragraph)] 1]
	set cc3 [lindex [set ${mode}::commentCharacters(Paragraph)] 2]
	set elecInsert "${cc1}\r${cc3}comment body\r${cc2}\r"        
    }
    if {[info exists elecInsert]} {
	placeBookmark
	goto $pos
	elec::Insertion $elecInsert
	status::msg "Press <Ctrl>-. to return to original cursor position."
    } else {
	comment::commentTemplate
	status::errorMsg "Cancelled."
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "comment::commentTemplate" --
 # 
 # Requires that a selection be highlighted.  Take the highlighted selection
 # and use it for the mode's "commentTemplate" preference.
 # 
 # -------------------------------------------------------------------------
 ##

proc comment::commentTemplate {} {
    
    global mode
    
    if {![string length $mode]} {status::msg "This item requires an open window." ; return}
    
    global ${mode}modeVars
    
    set question "Save selection as the $mode comment template?"
    if {![string length [set template [getSelect]]]} {
	alertnote "To create a custom template,\
	  type it in your document, highlight it, and then\
	  select the 'Comment Template' menu item.  To enter\
	  template stops, press '<option> 8' \
	  -- be sure to include TWO bullets for every stop."
    } elseif {[askyesno $question] == "yes"} {
	set ${mode}modeVars(commentTemplate) "$template"
	prefs::modified ${mode}modeVars(commentTemplate)
	status::msg "The new comment template for '$mode' mode has been added."
    } else {
	status::msg "Cancelled."
    }
}
