## -*-Tcl-*-
 # ==========================================================================
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "completions.tcl"
 #                                    created: 07/27/1997 {12:43:41 am}    
 #                                last update: 11/26/2001 {16:08:14 PM} 
 # 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #      
 # Copyright (c) 1997-2001 Vince Darley.
 # 
 # Distributed under a Tcl style license.  
 # 
 # Basic parts of the completion package -- to handle word and file
 # completion, but allowing very simple piggy-backing of advanced
 # completions.
 # 
 # ===========================================================================
 ##

namespace eval bind {}

## 
 # -------------------------------------------------------------------------
 #       
 # "bind::Completion" --
 #      
 # If we're already completing, jump to that procedure, else go through
 # a mode-dependent list of completion procedures given by the array
 # 'completions', these return either '1' to indicate termination, or
 # '0' to say either that they failed (or that they succeeded in a
 # simple unique way and that further completion procedures may be
 # applied).
 # 
 # Procedures which succeed, but may allow further completion
 # possibilities if the user presses the completion binding again (e.g.
 # a variety of possible word completions may exist) should call
 # 'completion::action' with an appropriate '-repeatCommand' flag
 # with the follow up procedure.
 #       
 # If no mode-dependent procedure list exists (as in a basic Alpha
 # installation), then just the 'user' completions and 'word'
 # completions are attempted.
 #       
 # The list of procedures to try is copied into 'completion::chain', so
 # completion procs can modify that list if they like.
 # 
 # The 'completion' helper proc is used to translate from an entry
 # in the completion(mode) array to an actual procedure call.
 # 
 # The completion procedures which are called should not, in general,
 # modify the contents of the window.  Instead, they should call
 # 'completion::action' to carry out any actual modifications.
 # 
 # -------------------------------------------------------------------------
 ##

proc bind::Completion {} {
    if {![completion::tabDeleteSelection]} {return}
    
    if {[completion::noCompletionInProgress]} {
	if {[completion::user]} {return}
	set m [modeALike]
	global completions mode completion::chain
	if {[info exists completions($mode)]} {
	    set completion::chain $completions($mode)
	    while 1 {
		if {[set c [lindex ${completion::chain} 0]] == ""} {break}
		set completion::chain [lreplace ${completion::chain} 0 0]
		if {[completion $m $c]} {return}
	    }
	    status::msg "No further completions exist, perhaps you\
	      should write your own."
	} else {
	    completion::word
	}
    }
}

## 
 # -------------------------------------------------------------------------
 #       
 # "completion" --
 # 
 # Call a completion, by trying in order:
 #    1) error
 #    2) 'Type' is actually a generic completion routine
 #    3) '${mode}::Completion::${Type}' is a mode-specific routine
 #    4) 'completion::${type}' is a generic routine.
 #    
 # We also check for expansion procedures of the forms:
 #    1) 'expansions::${type}'
 #    2) '${mode}::Expansion::${Type}', where Type begins with 'Ex'
 #
 # -------------------------------------------------------------------------
 ##

proc completion {mode Type args} {
    #puts [list $mode $Type $args]
    if { $Type == "error" } {error ""}
    if {[regexp {^(completion|expansions)::} $Type]} {
	return [eval $Type $args]
    } elseif {[llength [info commands ${mode}::Completion::${Type}]]} {
	return [eval ${mode}::Completion::${Type} $args]
    } elseif {[llength [info commands ${mode}::Expansion::${Type}]]} {
	return [eval ${mode}::Expansion::${Type} $args]
    } else {
	return [eval completion::[string tolower $Type] $args]
    }
}

namespace eval completion {}

# Setup this variable to say no completions are in progress.
ensureset completion::in_progress 0

## 
 # -------------------------------------------------------------------------
 #       
 # "completion::noCompletionInProgress" --
 #      
 # Call this to check if we should divert directly to a previously
 # registered completion procedure instead of starting from scratch.
 # 
 # If we should start from scratch, we just initialise a few global
 # variables and return '1'.
 # 
 # If we are continuing (and if the position hasn't changed at all), 
 # we evaluate the stored script: that returns either 1 or 0.  In the
 # latter case there are no more completions (so we select some text
 # and give a message).  In the former case we found some (and presumably
 # called completion::action to update the display appropriately).
 # 
 # If the script throws an error, we effectively ignore its existence
 # and the entire proc resets the completions and returns 1.
 #   
 # -------------------------------------------------------------------------
 ##
proc completion::noCompletionInProgress {} {
    global completion::in_progress_proc completion::in_progress_pos \
      completion::in_progress completion::original_position

    # Do the old completion if possible
    if {[set completion::in_progress]} {
	if {[pos::compare ${completion::in_progress_pos} == [getPos]]} {
	    # An existing completion is in progress.  Call the script
	    # which was registered to continue that completion.
	    
	    #puts "repeat ${completion::in_progress_proc}"
	    if {![catch {eval ${completion::in_progress_proc}} res]} {
		if {$res == 1} {
		    # We completed successfully; so tell our caller that
		    # there is a completion in progress
		    return 0
		} else {
		    # We failed.  This means we've actually run out of
		    # completions in this continuation.  We highlight
		    # the currently completed text so the user can
		    # easily delete it if they want to.
		    select ${completion::original_position} \
		      ${completion::in_progress_pos}
		    status::msg "All possible completions have been cycled."
		    # We don't want to start a new completion mechanism,
		    # so tell our caller to stop now.
		    return 0
		}
	    }
	}
    }
    # No completion is in progress, or an error resulted above.
    # We therefore reset everything.
    completion::reset
    return 1
}

proc completion::reset {} {
    global completion::in_progress_proc completion::in_progress_pos \
      completion::in_progress completion::original_position

    set completion::in_progress 0
    set completion::in_progress_proc error
    set completion::in_progress_pos [getPos]
    catch {unset completion::original_position}
    return
}

## 
 # -------------------------------------------------------------------------
 #       
 # "completion::action" --
 #      
 # Any completion procedure which is successful should call this. It is
 # best to call this rather than inserting/deleting text manually.
 # 
 # Current flags are:
 # 
 # -delete <len>:  number of chars before current cursor position to delete
 # -text <text>:   insert this text at the current position
 # -electric:      if given, then text insertions are via 'elec::Insertion' and
 #                 may therefore contain various tab stops.
 # -msg <msg>:     give this message in the status bar.
 # 
 # -repeatCommand <script>: command to be evaluated if the user asks for another
 #                 completion (provided the window/position have not been 
 #                 changed).  If this is not given, then the other flag
 #                 actions will still be taken, and it is assumed this
 #                 is a completion which cannot be 'cycled' through numerous
 #                 options in any way.
 # 
 # -------------------------------------------------------------------------
 ##
proc completion::action {args} {
    global completion::in_progress_proc completion::in_progress_pos \
      completion::in_progress completion::original_position

    #puts "success $args"
    getOpts {-delete -text -repeatCommand -msg -postInsertHook}
    # Just in case the completion messed with the position
    goto [set completion::in_progress_pos]
    # Delete anything we were asked for
    if {[info exists opts(-delete)]} {
	if {$opts(-delete) != 0} {
	    deleteText [pos::math [getPos] - $opts(-delete)] [getPos]
	}
    }
    if {![info exists completion::original_position]} {
	set completion::original_position [getPos]
    }
    # Insert the new text
    if {[info exists opts(-electric)]} {
	elec::Insertion $opts(-text)
    } else {
	insertText $opts(-text)
    }
    if {[info exists opts(-postInsertHook)] \
      && [string length $opts(-postInsertHook)]} {
	catch {eval $opts(-postInsertHook)}
    }
    
    # And give a message
    if {[info exists opts(-msg)]} {
	status::msg $opts(-msg)
    }
    # Now store anything required for a second go around, if that is wanted...
    if {[info exists opts(-repeatCommand)]} {
	set completion::in_progress_proc $opts(-repeatCommand)
	set completion::in_progress_pos  [getPos]
	set completion::in_progress      1
    } else {
	completion::reset
    }
    return 1
}

# This should really NOT be used.  We need to make the procs in
# this file sufficiently general so that it isn't required!
proc completion::_insert {text} {
    global completion::in_progress_pos
    insertText $text
    set completion::in_progress_pos [getPos]
}

proc completion::user {{cmd ""}} {return 0}

## 
 # -------------------------------------------------------------------------
 #       
 # "modeALike" --
 #      
 # Some modes are really equivalent as far as commands etc.  go, so we
 # don't bother with duplication. 
 # 
 # -------------------------------------------------------------------------
 ##

proc modeALike {{m ""}} {
    global mode completionsLike
    if {$m == ""} {set m $mode}
    if {[info exists completionsLike($m)]} {
	set m $completionsLike($m)
    }
    return $m
}

# OBSOLETE, and behaviour is not what a caller will be expecting --
# the callers need to call completion::action -repeatCommand instead.
proc completion::already {args} { completion::reset }

proc completion::word {} {
    completion::general [completion::lastWord] 
}

## 
 # -------------------------------------------------------------------------
 # 
 # "completion::general ?opts? lookFor" --
 # 
 #  Look for completions for the given word '$lookFor', to be inserted
 #  at the current cursor position (which is actually stored in
 #  'completion::in_progress_pos').  We look backwards first, and then
 #  forwards.
 #  
 #  Acceptable syntax:
 #  
 #  completion::general ?-excludeBefore <len>? ?-mustMatch <str>? \
 #    ?-postInsert <script>? ?-pattern <pat>? lookFor
 #  
 #  The excludeBefore 'len' is the number of characters before the
 #  current insertion point in which matches are not acceptable (the
 #  idea is we want to match anything but what the user is currently
 #  typing!).  If not given, this will be calculated as the length of
 #  'lookFor', but it could be different.
 #  
 #  The mustMatch 'str' is a string which will be added to 'lookFor'
 #  and which must be matched.
 #  
 #  If 'pat' is given, then it is used as the pattern to be matched by
 #  any completion.  If it is not given then the current
 #  (mode-specific) 'wordBreak' value is used.
 #  
 #  When matches are found, any characters beyond 'lookFor' are
 #  considered an appropriate completion, and are sent to
 #  'completion::action' with appropriate other parameters.  This will
 #  generally insert those characters into the window.  If a postInsert
 #  'script' is given, then it is evaluated just after the characters
 #  are inserted.
 #  
 # -------------------------------------------------------------------------
 ##
proc completion::general {args} {
    set opts(-postInsert) ""
    set opts(-excludeBefore) ""
    getOpts {-excludeBefore -postInsert -mustMatch -pattern}
    if {[llength $args] != 1} {
	return -code error "Bad arguments"
    }
    
    set lookFor [lindex $args 0]
    set len [string length $lookFor]

    if {$opts(-excludeBefore) != ""} {
	set excludeBefore $opts(-excludeBefore)
    } else {
	set excludeBefore $len
    }
    
    # Exclude the current position as a match result!
    set origPos [getPos]
    set start [pos::math $origPos - $excludeBefore - 1]

    # We want to find anything else which looks like 'lookFor' and
    # continues a 'word'
    if {[info exists opts(-mustMatch)]} {
	append lookFor $opts(-mustMatch)
    }
    set pat [quote::Regfind $lookFor]
    
    if {[info exists opts(-pattern)]} {
	append pat $opts(-pattern)
    } else {
	global wordBreak
	append pat $wordBreak
    }
    
    # Start the general completion mechanism.
    return [completion::generalRepeat $len $origPos $pat $start \
      0 "" $opts(-postInsert)]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "completion::generalRepeat" --
 # 
 #  Helper procedure for 'completion::general'.
 #  
 #  Returns '1' on success, 0 otherwise.  Calls completion::action on 
 #  success as well.
 # -------------------------------------------------------------------------
 ##
proc completion::generalRepeat {len origPos pat nextStart fwd prevHits postInsert} {
    global completion::in_progress_pos
    while 1 {
	if {![catch {search -s -f $fwd -r 1 -i 0 -m 1 -- $pat $nextStart} data]} {
	    set Hit [eval getText $data]
	    set beg [pos::math [lindex $data 0] + $len]
	    set end [lindex $data 1]
	    
	    if {[lsearch -exact $prevHits $Hit] == -1} {
		# This Hit is not the same as the last one
		
		# Add the hit to the list of previous hits
		lappend prevHits $Hit
		
		set extraTxt [getText $beg $end]
		set deleteLen [pos::diff $origPos [getPos]]
		if {$fwd} {
		    # Search Forwards
		    set nextStart $end
		    # End of found word
		} else {
		    # Search Backwards
		    set nextStart [pos::math [lindex $data 0] - $len]
		    # Before start of found word
		    if {[pos::compare $nextStart <= [minPos]]} {
			set fwd 1
			set nextStart ${completion::in_progress_pos}
		    }
		}
		if {$fwd} {
		    set fndMsg "Found below."
		} else {
		    set fndMsg "Found above."
		}
		# Return point (1)
		return [completion::action -repeatCommand \
		  [list completion::generalRepeat $len $origPos \
		  $pat $nextStart $fwd $prevHits $postInsert] \
		  -msg $fndMsg -text $extraTxt -delete $deleteLen \
		  -postInsertHook $postInsert]
	    } else {
		# Move start of search after finding string again
		if {$fwd} {
		    # Searching Forwards
		    set nextStart $end
		    # End of found word
		} else {
		    # Still Searching Backwards
		    set nextStart [pos::math [lindex $data 0] - $len]
		    # Before start of found word
		    if {[pos::compare $nextStart <= [minPos]]} {
			set fwd 1
			set nextStart ${completion::in_progress_pos}
		    }
		}
	    }
	    # End if hit is the same as a previous hit
	} else {
	    # Search string not found
	    if {$fwd} {
		# Return point (2) : We were already looking forward,
		# so the word is not in the file
		return 0
	    } else {
		# Start looking forward
		set fwd 1
		set nextStart ${completion::in_progress_pos}
	    }
	}
	# There are two ways we could have returned above.  Either
	# successfully (1), or we already searched forwards and backwards
	# and failed (2).  If we reach here, we're still trying to find
	# more matches.
    }
    # This is never reached.
    return 0
}

## 
 # -------------------------------------------------------------------------
 #       
 # "completion::lastWord" --
 #      
 # Return the last word, without moving the cursor.  If a variable name is
 # given, it is returned containing the position of the start of the last
 # word.
 #       
 # Future extensions to this proc (in packages) may include further
 # optional arguments. 
 # 
 # -------------------------------------------------------------------------
 ##

proc completion::lastWord {{st ""}} {
    set pos [getPos]
    backwardWord
    if {$st != ""} {upvar $st beg}
    set beg [getPos]
    goto $pos
    set test1 [pos::compare $beg < [lineStart $pos]]
    set test2 [pos::compare $beg == $pos]
    if {$test1 || $test2} {
	status::errorMsg "Cancelled -- could not find a hint."
    } else {
	return [getText $beg $pos]
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "completion::lastTwoWords" --
 # 
 # Get last two words: returns the previous word, and sets the given var to
 # the word before that.  Note that the 'word before that' actually means
 # all text from the start of that word up to the beginning of the word
 # which is returned.  i.e. 'prev' will normally end in some sort of
 # space/punctuation.
 #       
 # Future extensions to this proc (in packages) may include further
 # optional arguments. 
 # 
 # -------------------------------------------------------------------------
 ##

proc completion::lastTwoWords {prev} {
    set pos [getPos]
    backwardWord
    set beg_rhw [getPos]
    backwardWord
    set beg_lhw [getPos]
    goto $pos
    upvar $prev lhw
    set test1 [pos::compare $beg_lhw < [lineStart $pos]]
    set test2 [pos::compare $beg_lhw == $beg_rhw]
    if {$test1 || $test2} {
	set lhw { } 
    } else {
	set lhw [getText $beg_lhw $beg_rhw]
    }
    return [getText $beg_rhw $pos]
}

## 
 # -------------------------------------------------------------------------
 #       
 # "completion::tabDeleteSelection" --
 #      
 # If there is a selection, this procedure is called by completion routines
 # to ask the user if it should be deleted (or if the appropriate flag is
 # set, to delete automatically). 
 # 
 # -------------------------------------------------------------------------
 ##

proc completion::tabDeleteSelection {} {
    global askDeleteSelection elecStopMarker

    if {([regexp "^\$|^$elecStopMarker" [getSelect]] || !$askDeleteSelection)} {
	deleteText [getPos] [selEnd]
    } else {
	switch [askyesno -c "Delete selection before continuing?"] {
	    "yes" 	{deleteText [getPos] [selEnd]}
	    "no"  	{goto [selEnd]}
	    "cancel"	{status::errorMsg "Cancelled."}
	}
    }
    return 1
}

## 
 # -------------------------------------------------------------------------
 #       
 # "completion::file" --
 #      
 #  Look back, see if there's a file/dir name and try and extend it. 
 #  Useful for Shel mode.  This improves on the one that comes with Alpha
 #  by default, and is much simpler. 
 #  
 # -------------------------------------------------------------------------
 ##

proc completion::filename {{dummy ""}} {
    set pos0 [getPos]
    set pos1 [pos::math $pos0 - 1]
    set pat  "\[\"\{ \t\r\n\]"
    set res  [search -s -f 0 -i 0 -m 0 -r 1 -n -- $pat $pos1]
    if {[string length $res]} {
	set from [lindex $res 1]
	if {[pos::compare $from < $pos0]} {
	    set pre ""
	    set text [getText $from $pos0]
	    if {[catch {glob -path $text *} globbed]} {
		# this bit is for macos only
		if {[catch {glob -path ":${text}" *} globbed]} {return 0}
		set pre ":"
	    }
	    completion::Find "$pre$text" $globbed
	    return 1
	}
    }
    return 0
}

## 
 # -------------------------------------------------------------------------
 #       
 # "completion::Find" --
 #      
 # Insert the completion of 'cmd' from the list 'matches', and return the
 # complete match if there was one.
 #      
 # 'cmd' is what we have, 'matches' is a list of things which can complete
 # it, and 'forcequery' says don't bother with partial completions: if we
 # can't finish the command off, present the user with a list. 
 # 
 # -------------------------------------------------------------------------
 ##

proc completion::Find {cmd matches {isdbllist 0} {forcequery 0} \
  {addQuery ""} {addAction ""} {insertHook ""}} {
    global listPickIfMultCmps listPickIfNonUniqueStuckCmp
    
    set cmdlen [string length $cmd]
    set mquery [set match [lindex $matches 0]]
    if {$isdbllist} { set match [lindex [lindex $match 0] 0]}
    if { [set cmdnum [llength $matches]] == 1 || $match == $cmd } {
	# It's unique or already a command, so insert it 
	# and turn off cmd completion.
	if {$cmdnum != 1 && $listPickIfNonUniqueStuckCmp \
	  && (![catch { set match [listpick -p "Pick a completion" -L $mquery $matches]}])} {
	    if {$isdbllist} { set match [lindex [lindex $match 0] 0]}
	    set msg ""
	} else {
	    set msg "Text is now a maximal completion."
	    # so we move on
	}
	set maxcompletion [string range $match $cmdlen end]
	completion::action -text $maxcompletion -msg $msg
	# So we move on
	return $match
    } else {
	set item [lindex $matches [incr cmdnum -1]]
	if {$isdbllist} { set item [lindex [lindex $item 0] 0] }
	set p [string length [largestPrefix [list $match $item]]]
	# set p $cmdlen
	# while {[string index $match $p]==[string index $item $p]} {incr p}
	if { $p == $cmdlen || $forcequery } {
	    beep
	    if {$listPickIfMultCmps || $forcequery} {
		if {$addQuery != ""} {
		    lappend matches "" $addQuery
		}
		if {[catch {set match [listpick -p "Pick a completion" -L $mquery $matches]}] \
		  || $match == "" } {
		    status::msg "Cancelled"
		    return 1
		} else {
		    if {$match == $addQuery} {$addAction ; return 1}
		    if {$isdbllist} { set match [lindex [lindex $match 0] 0]}                                   
		    set maxcompletion [string range $match $cmdlen end]
		    completion::action -text $maxcompletion \
		      -postInsertHook insertHook
		    # so we move on
		    return $match
		}
		
	    } else {
		status::msg "Can't extend --- ${matches}"
	    }
	} else { 
	    set maxcompletion [string range $match $cmdlen [incr p -1]]
	    completion::action -text $maxcompletion \
	      -msg "Matching: ${matches}" -postInsertHook insertHook
	}               
	return ""
    }
}

proc completion::getChoices {type choices} {
    global [lindex [split $choices "\("] 0]

    switch -- $type {
	"-command"  {return [uplevel 1 $choices]}
	"-list"     {return $choices}
	"-variable" {return [set $choices]}
	default     {error "Bad option '$type' to completion::getChoices"}
    }
}

proc completion::fromChoices {type choices prefix} {
    switch -- $type {
	"-command" {
	    set matches {}
	    foreach w [uplevel 1 $choices] {
		if {[string match "[quote::Find $prefix]*" $w]} {
		    lappend matches $w
		}
	    }
	    return $matches
	}
	"-list" {
	    set matches {}
	    foreach w $choices {
		if {[string match "[quote::Find $prefix]*" $w]} {
		    lappend matches $w
		}
	    }
	    return $matches
	}
	"-variable" {
	    return [completion::fromList $prefix $choices]
	}
	default {error "Bad option '$type' to completion::fromChoices"}
    }
}

## 
 # -------------------------------------------------------------------------
 #       
 # "completion::fromList" --
 #      
 # Given a 'cmd' prefix and the name of a list to search, that list being
 # stored in alphabetical order and starting/ending with whitespace, this
 # proc returns a list of all matches with 'cmd', or "" if there were none. 
 # Updated so works with arrays too (Nov'96)
 #       
 # It's quite an important procedure for completions, and must handle
 # pretty large lists, so it's worth optimising.
 #       
 # Note '\\b' = word boundary, '\\s' = whitespace '\\S' = not-whitespace
 # 
 # -------------------------------------------------------------------------
 ##

if {[info tclversion] < 8.0} {
    proc completion::fromList {__cmd slist} {
	global [lindex [split $slist "\("] 0]
	# Find all matches as a list --- a v. clever trick if I say so myself
	if {[regexp "(^|\\s)(${__cmd}\[^\\S\]*(\\s|\$))+" [set "$slist"] matches]} {
	    return [string trim $matches]
	} else {
	    return ""
	}
    }
} else {
    proc completion::fromList {__cmd slist} {
	global [lindex [split $slist "\("] 0]
	set first [lsearch -glob [set $slist] "${__cmd}*"]
	if {$first == -1} { return "" }
	set first [lrange [set $slist] $first end]
	regexp {^(.*)(.)$} $__cmd "" _find _last        
	set _find "^[::quote::Regfind $_find]\[^$_last\].*"
	set last [lsearch -regexp $first $_find]
	if {$last == -1} {
	    incr last
	    while {[string match "${__cmd}*" [lindex $first $last]]} {
		incr last
	    }
	}
	return [lrange $first 0 [incr last -1]]
    }
}


