## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "fileManipulation.tcl"
 #                                    created: 24/2/98 {1:57:08 pm} 
 #                                last update: 11/26/2001 {17:50:28 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Mostly Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # These are various utility procedures which operate on windows
 # which represent files, on files directly, and on file paths.
 # 
 # Procedures which can operate on non-file windows should not be
 # in this file, and should not be in the 'file::' namespace.
 # ###################################################################
 ##

namespace eval file {}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::minimalDistinctTails" --
 # 
 #  Go over all of the given files (which should be complete paths),
 #  and return an ordered list of the tails, where if any tail is
 #  duplicated, we add in the preceeding directory names until we
 #  do get a unique set of items.  Order is preserved.
 #  
 #  If files do not exist, we ignore them.
 # -------------------------------------------------------------------------
 ##
proc file::minimalDistinctTails {filelist} {
    global file::separator
    set menulist [list]
    set level 1
    while {1} {
	foreach t $filelist {
	    # Don't do 'file exists' for networked files.  If we're
	    # not on a network Alpha may lock for some time
	    if {![file::isNetworked $t] && ![file exists $t]} {continue}
	    set llen [llength [set tail [file split $t]]]
	    if {$llen < $level} {
		# We've exceeded the top-level.  Must be an odd problem!
		# Discard this problematic file.
		continue
	    }
	    set tail [join [lrange $tail [expr {$llen - $level}] end] ${file::separator}]
	    if {[info exists name($tail)]} {
		lappend remaining $name($tail)
		lappend remaining $t
		set dup($tail) 1
		set first [lsearch -exact $menulist $tail]
		set menulist [lreplace $menulist $first $first $name($tail)]
		if {$level==1} {
		    lappend menulist $t
		}
		unset name($tail)
	    } elseif {[info exists dup($tail)]} {
		lappend remaining $t
		if {$level==1} {
		    lappend menulist $t
		}
	    } else {
		set name($tail) $t
		if {$level==1} {
		    lappend menulist $tail
		} else {
		    set toolong [lsearch -exact $menulist $t]
		    set menulist [lreplace $menulist $toolong $toolong $tail]
		}
	    }
	}
	if {![info exists remaining]} {
	    break
	}
	incr level
	set filelist $remaining
	unset remaining
	unset dup
    }
    return $menulist
}

proc file::coreCopy {from to} {
    set fromEnc [alpha::encodingFor $from]
    set toEnc [alpha::encodingFor $to]
    if {[string length $fromEnc] && [string length $toEnc]\
      && ($fromEnc != $toEnc) && ([info tclversion] > 8.0)\
      && [file::isText $from]} {
	set fin [alphaOpen $from r]
	set fout [alphaOpen $to w]
	fcopy $fin $fout
	close $fin ; close $fout
	file mtime $to [file mtime $from]
    } else {
	file copy $from $to
    }
}

proc file::isText {filename} {
    switch -- [string tolower [file extension $filename]] {
	".gif" - ".pdf" - ".ps" - ".eps" - 
	".icr" - ".png" - ".dll" - ".shlb" -
	".so" - ".exe" - ".zip" - 
	".ppm" {
	    return 0
	}
	default {
	    return 1
	}
    }
}

proc file::toUrl {file} {
    global file::separator
    regsub -all ${file::separator} $file / file
    return "file:///[string trimleft [quote::Url $file] ${file::separator}]"
}

# Certain characters are not allowed in file names.  This procedure
# takes a prospective file name (i.e just the tail, not a full
# directory specification) and removes all illegal characters.
proc file::makeNameLegal {name} {
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    regsub -all {[:]} $name "" name
	    return $name
	}
	"windows" {
	    regsub -all {[<>*?/\\:]} $name "" name
	    return $name
	}
	"unix" {
	    regsub -all {[/]} $name "" name
	    return $name
	}
    }
}

proc file::tryToFindInFolder {folder rest pattern \
  {prompt "Select the file to use"}} {
    # Loop in case we really fail
    while {1} {
	# If we have multiple extensions, loop over them, testing for names.
	# We might have to strip off: .sit.hqx or even more.
	
	while {1} {
	    set filepre [file root $rest]
	    # look for directories
	    set dirs [glob -nocomplain -types d \
	      -path [file join $folder ${filepre}] -- *]
	    set local [lindex $dirs 0]
	    if {![info exists bestGuessFolder]} {
		set bestGuessFolder $local
	    }
	    set files [lunique [glob -types TEXT -nocomplain \
	      -dir $local -- $pattern]]
	    set realfiles [list]
	    foreach f $files {
		if {![file isdirectory $f]} {
		    lappend realfiles $f
		}
	    }
	    set files $realfiles
	    if {[llength $files] != 0} { break }

	    # look for files
	    set files [glob -types TEXT -nocomplain \
	      -path [file join ${folder} ${filepre}] -- *]
	    set realfiles [list]
	    foreach f $files {
		if {![file isdirectory $f]} {
		    lappend realfiles $f
		}
	    }
	    set files $realfiles
	    if {[llength $files] != 0} { break }
	    set newrest [file root $rest]
	    if {![string length $newrest] || ($rest == $newrest)} {
		break
	    }
	    set rest $newrest
	}
	
	if {[llength $files] == 0} {
	    set shortLen [string length $filepre]
	    # While the length is 4 chars or more, strip 2 characters
	    # off and try again.  This should eventually work unless
	    # the package is very strangely named or packaged.
	    if {$shortLen > 4} {
		set rest [string range $filepre 0 [expr {$shortLen - 3}]]
		# try everything again.
		continue
	    }
	    alertnote "I can't find an obvious, suitable, unique file.\
	      Please try to find it in the following dialog."
	    
	    if {[info exists bestGuessFolder]} {
		set folder $bestGuessFolder
	    }
	    if {[catch [list getfile $prompt $folder] f]} {
		return ""
	    } else {
		return [file nativename $f]
	    }
	    # We used to display the folder to the user:
	    # but we never get here now.
	    file::showInFinder $local
	    return
	}
	if {[llength $files] > 1} {
	    set f [listpick -p $prompt $files]
	} else {
	    set f [lindex $files 0]
	}
	return [file nativename $f]
    }

}

# For Alpha 7.x we simply return the existing name.
proc file::ensureStandardPath {n} {
    if {[info tclversion] < 8.0} { return $n }
    if {$n == ""} {return ""}
    # So we can handle relative path names
    if {[file pathtype $n] == "relative"} {
	set n [file join [pwd] $n]
    }
    global tcl_platform
    if {$tcl_platform(platform) == "windows"} {
	# Catch this in case we're in a vfs.
	catch {set n [file attributes $n -longname]}
	if {[string is lower [string index $n 0]]} {
	    set n "[string toupper [string index $n 0]][string range $n 1 end]"
	}
    }
    set n [file nativename $n]
    return $n
}

proc file::saveResourceChanges {filename} {
    global alpha::platform backup
    if {[set alpha::platform] == "alpha"} {
	bringToFront $filename
	# Closing marked file, re-opening it.  This is
	# necessary to avoid changing the "last save" date in
	# the resource fork, so that newer Help file versions
	# can be installed later if necessary.  Marks and
	# hypers will now be retained in the file, and
	# subsequent calls to helpMenu won't have to go
	# through this routine.
       catch {setWinInfo read-only 0}
       setWinInfo state mpw
       set oldBackup $backup
       set backup 0
       catch {saveUnmodified}
       set backup $oldBackup
       menu::fileProc "File" "close"
       edit -r -c $filename
    }
}

proc file::openAsTemplate {filename {name ""} {readonly 1}} {
    if {$name == ""} {
	set name [file tail $filename]
    }
    set m [win::FindMode $filename]
    new -n $name -m $m -text [file::readAll $filename]
    if {$readonly} {
	setWinInfo dirty 0
	setWinInfo read-only 1
    }
    goto [minPos]
}

proc file::showInFinder {{f {}}} {
    if {$f == ""} {set f [win::Current]}
    set f [win::StripCount $f]
    if {![file exists "$f"]} {
	message "ERROR: FILE NOT FOUND: \"$f\""
	return
    }
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    if {[file isdirectory $f]} {
		switchTo Finder
		sendOpenEvent -r Finder $f
	    } else {
		switchTo Finder
		AEBuild Finder misc mvis "----" [makeAlis $f]
	    }
	}
	"windows" {
	    windows::Show $f
	}
	"unix" {
	    if {$tcl_platform(os) == "Darwin"} {
		if {[file isdirectory $f]} {
		    switchTo Finder
		    sendOpenEvent -r Finder $f
		} else {
		    switchTo Finder
		    AEBuild Finder misc mvis "----" [makeAlis $f]
		}	
	    } else {
		alertnote "file::showInFinder not yet implemented on Unix."
	    }
	}
    }
}

proc file::tryToOpen {{fname ""}} {
    if {$fname == ""} {set fname [getSelect]}
    set f [file join [file dirname [win::Current]] $fname]
    if {[file exists $f]} {
	win::OpenQuietly $f
    } else {
	alertnote "Sorry, I couldn't find the file '$fname'.\
	  You could install\
	  or activate the 'Mode Search Paths' feature, which includes\
	  better include-path handling."
    }
}

proc file::ensureDirExists {dir} {
    if {![file exists $dir]} {
	set parent [file dirname $dir]
	if {$dir == "" || ($parent == $dir)} {
	    error "Can't create the folder '$dir' because\
	      the disk doesn't exist."
	}
	file::ensureDirExists [file dirname $dir]
	file mkdir $dir
	return 1
    }
    return 0
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::isNetworked" --
 # 
 #  Calling 'file exists' on a networked file which may not exist is a 
 #  rather time consuming operation.  We can use this to avoid such
 #  calls in, for example, the recent files menu.
 # -------------------------------------------------------------------------
 ##
proc file::isNetworked {file} {
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    return 0
	}
	"windows" {
	    return [expr {[string range [file nativename $file] 0 1] == "\\\\"}]
	}
	"unix" {
	    return 0
	}
    }
}

if {$tcl_platform(platform) == "macintosh"} {
    proc file::isLocal {fname} {
	set vol [lindex [file split [file normalize $fname]] 0]
	if {[catch {
	    tclAE::build::resultData 'MACS' core getd ----\
	      [tclAE::build::propertyObject isrv\
		[tclAE::build::nameObject cdis\
		  [tclAE::build::TEXT $vol]]]
	  } res]} then {return 0}
	return $res
    }
} else {
    proc file::isLocal {file} {
	return [expr {![file::isNetworked $file]}]
    }
}

proc file::openInDefault {file} {
    set file [file::ensureStandardPath $file]
    if {[file isfile $file]} {
	global tcl_platform
	switch -- $tcl_platform(platform) {
	    "macintosh" {
		sendOpenEvent -noreply Finder "${file}"
	    }
	    "windows" {
		windows::Launch $file
	    }
	    "unix" {
		if {$tcl_platform(os) == "Darwin"} {
		    sendOpenEvent -noreply Finder "${file}"
		} else {
		    alertnote "Opening such a file not yet implemented"
		}
	    }
	}
    } else {
	file::showInFinder $file
    }
}

if {[info tclversion] < 8.4} {
    proc file_system {filename} { return "native" }
} else {
    proc file_system {filename} { file system $filename }
}

proc file::browseFor {filename} {
    set system [lindex [file_system $filename] 0]
    if {$system == "native"} {
	return [findFile $filename]
    } else {
	if {[file isfile $filename]} {
	    set filename [file dirname $filename]
	}
	return [file::openViaListpicks $filename]
    }
}

proc file::openViaListpicks {filename} {
    while {[file isdirectory $filename]} {
	set disp [list]
	foreach f [glob -dir $filename *] {
	    lappend disp [file tail $f]
	}
	set disp [concat [list ..] [lsort -ignore $disp]]
	if {[catch {listpick -p {File?} $disp} choice]} {return}
	if {$choice == ".."} {
	    set filename [file dirname $filename]
	} else {
	    set filename [file join $filename $choice]
	}
    }
    edit $filename
}

proc file::openAny {file} {
    if {[catch {
	set file [file::ensureStandardPath $file]
	getFileInfo $file a
	# If it's a file or an alias
	if {[file isfile $file] \
	  || ([file type $file] == "unknown" && $a(type) != "fdrp")} {
	    if {![info exists a(type)] || ($a(type) == "TEXT")} {
		edit $file
	    } else {
		global tcl_platform
		switch -- $tcl_platform(platform) {
		    "macintosh" {
			sendOpenEvent -noreply Finder "${file}"
		    }
		    "windows" {
			windows::Launch $file
		    }
		    "unix" {
			if {$tcl_platform(os) == "Darwin"} {
			    sendOpenEvent -noreply Finder "${file}"
			} else {
			    alertnote "Opening such a file not yet implemented"
			}
		    }
		}
	    }
	} else {
	    file::browseFor $file
	}
    } err]} {
	alertnote "There was an error trying to open '$file': $err"
    }
}

proc file::renameTo {} {
    set c [win::StripCount [win::Current]]
    if {![file exists $c]} { 
	alertnote "'$c' is not a file window!" ; return 
    }
    set new [prompt "New name for file:" [file tail $c]]
    if {[file exists [set to [file join [file dirname $c] $new]]]} {
	alertnote "File '$to' already exists!"
	return
    }
    killWindow
    if {[catch {file rename $c $to} err]} {
	alertnote "Rename unsuccessful: $err"
	edit [win::StripCount $c]
    } else {
	edit $to
    }
}

proc file::standardFind {f} {
    global HOME auto_path PREFS tclExtensionsFolder file::separator
    set dirs $auto_path
    lappend dirs [file join $HOME Tcl Completions] $PREFS \
      [file join $HOME Help] [file join $HOME Tools]
    if {[info exists tclExtensionsFolder]} { lappend dirs $tclExtensionsFolder }
    foreach dir $dirs {
	if {[file exists [file join ${dir} ${f}]]} {
	    return [file join ${dir} ${f}]
	}
    }
    if {[regexp -- ${file::separator} $f]} {
	foreach dir $dirs {
	    if {[file exists [file join [file dirname $dir] $f]]} {
		return [file join [file dirname $dir] $f]
	    }
	}
    }
    error "File '$f' not found"	
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::compareModifiedDates" --
 # 
 #  Return -1 if first file is older, 0 if they have equal dates,
 #  and 1 if the second file is older.
 # -------------------------------------------------------------------------
 ##
proc file::compareModifiedDates {a b} {
    getFileInfo [win::StripCount $a] infoa
    getFileInfo [win::StripCount $b] infob
    # bigger = newer
    set ma $infoa(modified)
    set mb $infob(modified)
    if {$ma < $mb} {
	return -1
    } elseif {$ma == $mb} {
	return 0
    } else {
	return 1
    }
}

proc file::compareDates {a op b} {
    getFileInfo [win::StripCount $a] infoa
    getFileInfo [win::StripCount $b] infob
    # bigger = newer
    set ma $infoa(modified)
    set mb $infob(modified)
    return [expr ($ma $op $mb)]
}

proc file::sameModifiedDate {a b} {
    return [expr {[file::compareModifiedDates $a $b] == 0}]
}

proc file::secondIsOlder {a b} {
    return [expr {[file::compareModifiedDates $a $b] == 1}]
}

proc file::replaceSecondIfOlder {a b {complain 1} {backup ""}} {
    if {![file exists $a]} { error "file::replaceSecondIfOlder -- first file '$a' does not exist!" }
    if {[file exists $b]} {
	if {[file::secondIsOlder $a $b]} {
	    file::remove [file dirname $b] [list [file tail $b]] $backup
	    file::coreCopy $a $b
	    install::log "Copied [file tail $a] to $b"
	    return 1
	} elseif {[file::secondIsOlder $b $a]} {
	    install::log "The pre-existing [file tail $a] is newer than the one which was to be installed."
	}
    } elseif {$complain} { 
	error "file::replaceSecondIfOlder -- second file '$b' does not exist!"
    } else {
	file::coreCopy $a $b
	install::log "Copied [file tail $a] to $b"
    } 
    return 0
}

proc file::removeCheckingWins {f} {
    install::log "Removed $f"
    if {[set i [lsearch -regexp [winNames -f] "^[quote::Regfind $f]( <\d+>)?$"]] != -1} {
	bringToFront [lindex [winNames -f] $i]
	killWindow
	file delete $f
	return 1
    }
    file delete $f
    return 0
}

proc file::remove {to files {backup ""}} {
    foreach f $files {
	if {[file exists [file join $to $f]]} {
	    file::removeOne [file join $to $f] $backup
	}
    }
}

proc file::removeOne {f {backup ""}} {
    set ff [file tail $f]
    message "Removing old '$ff'"
    if {${backup} != ""} {
	if {![file exists $backup]} { file mkdir $backup }
	set i ""
	while {[file exists [file join $backup $ff$i]]} {
	    if {$i == ""} { set i 0}
	    incr i
	}
	file copy $f [file join ${backup} $ff$i]
    }
    file::removeCheckingWins $f
}

proc file::getSig {f} {
    if {[catch {getFileInfo $f arr}]} { return "" }
    return $arr(creator)
}

proc file::setSig {f sig} {
    if {[file isfile $f] && ([getFileType $f] == "TEXT") && ([getFileSig $f] != $sig)} {
	message "Converting $f"
	setFileInfo $f creator $sig
    }	
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::findAllInstances" --
 # 
 #  Returns all instances of a given pattern in a file.  This is a regexp
 #  search, and the pattern must match all the way to the end of the 
 #  file.  Here is an example usage:
 #  
 #  	set pat2 {^.*\\(usepackage|RequirePackage)\{([^\}]+)\}(.*)$}
 #  	set subpkgs [file::findAllInstances $filename $pat2 1]
 #  
 #  Notice the pattern ends in '(.*)$', this is important.
 #  Notice that since there is one extra '()' pair in the regexp,
 #  we give '1' as the last argument.
 #  
 #  WARNING:  Calling this procedure incorrectly can easily result
 #  in an infinite loop.  This will tend to crash Alpha and is hard
 #  to debug using trace-dumps, because Alpha will tend to crash
 #  whilst tracing too!  To debug, modify the 'while' loop so that it
 #  also increments a counter, and stops after a few iterations.
 # -------------------------------------------------------------------------
 ##
proc file::findAllInstances {filename searchString {extrabrackets 0}} {
    # Get the text of the file to be searched:
    if {[lsearch -exact [winNames -f] $filename] >= 0} {
	set fileText [getText -w $filename [minPos] [maxPos -w $filename]]
    } elseif {[file exists $filename]} {
	set fd [alphaOpen $filename]
	set fileText [read $fd]
	close $fd
    } else {
	return ""
    }
    # Search the text for the search string:
    while {[string length $fileText]} {
	set dmy [lrange "d d d d d d" 0 $extrabrackets]
	if {[eval regexp -- [list $searchString] [list $fileText] $dmy match fileText]} {
	    lappend matches $match
	} else {
	    break
	}
    }
    if {[info exists matches]} {
	return $matches
    } else {
	return ""
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 #  "file::preOpeningConfigurationCheck" --
 #	
 #  This is an adaptation of Tom Pollard's emacs mode setting facility.
 #  I call it from activateHook, which means it takes effect before the
 #  window yet exists, so you don't get a double redraw.  Here are
 #  Tom's comments from the original:
 #	   
 #   # Emacs-style mode selection using first nonblank line of file
 #   #
 #   # Checks for interpreter line "#!/dir/subdir/command ...", or
 #   # explicit major mode election "-*-Mode: vars ...-*-".
 #   #
 #   # "command" or "Mode" is compared (case-insensitively) to Alpha mode
 #   # names and first matching mode is used for the file.
 #   #
 #   # Author:   Tom Pollard	<pollard@chem.columbia.edu>
 #   # Modified: 9/11/95
 #	
 #  Note: this proc actually opens the file for reading.  It _must_
 #  close the file before exiting.  If you modify this proc, make sure
 #  that happens!
 #  
 #  The 'name' argument is the name of the file, which must exist,
 #  the 'winname' argument is the name of the window which will/would
 #  be opened in Alpha, which is there $name with possible a string
 #  like ' <2>' appended.
 #  
 #  Results:
 #  
 #  This procedure returns the mode to be used for the given file, if
 #  it can be ascertained from the contents/type of the file.  Note
 #  that we do NOT look at the file's extension.  That is handled by
 #  win::FindMode.
 #  
 #  Side effects:
 #  
 #  The win::config($winname) array entry may be created with
 #  additional configuration options for this window (e.g. encoding,
 #  tabsize, etc.)
 #  
 #  Therefore if you want to call this procedure manually, you must
 #  afterwards cleanup the win::config($winname) array entry if it exists.
 #	
 # --Version--Author------------------Changes-------------------------------  
 #    1.0     <vince@santafe.edu> first modification from Tom Pollard's
 #    1.1     <vince@santafe.edu> copes with a common Tcl/Tk exec trick.
 #    1.2     <vince@santafe.edu> can map creators if desired.
 #    1.3     <vince@santafe.edu> revamped for better win/file separation
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.0} {
proc file::preOpeningConfigurationCheck {name {winname ""}} {
    if {![file exists "$name"]} {
	error "No such file '$name'"
    }
    global modeCreator win::config
    if {[catch [list open "$name" r] fid]} { return }
    # find first non-empty line. Return if we fail
    for { set line "" } { [string trim $line] == "" } {} {
	if { [gets $fid line] == -1} { close $fid ; return }
    }
    set ll $line
    if {$winname == ""} {set winname $name}
    while {[regexp "\\((\[a-zA-Z\]+):(\[^\\)\]+)\\)(.*)" $ll "" var val ll]} {
	lappend win::config($winname) [list $var $val]
    }
    if {[info exists modeCreator([set sig [getFileSig $name]])]} {
	return $modeCreator($sig)
    }
    if {[regexp -nocase {^[^\n\r]*[-# \(]install($|[- \)])} $line]} {
	global HOME
	if {![string match [file join ${HOME} Tcl *] $name]} {
	    if {[catch {file readlink [file join ${HOME} Tcl]} link] || ([string first $link $name] != 0)} {
		close $fid
		return "Inst"
	    }
	}
    }
    if {[regexp {^#![ 	]*([^ 	\n\r]+)} $line dmy mtch] } {
	if {[regexp {([^/]+)$} $mtch majorMode]} { 
	    # remove trailing version number
	    set majorMode [string trimright $majorMode "01234567890."]
	    if {$majorMode == "sh"} {
		# need to check if we're using a common unix trick
		if {[gets $fid ll] != -1} {
		    while {[string index [string trimleft $ll] 0] == "#"} {
			if {[gets $fid ll] == -1} { close $fid ; return }
		    } 
		} else {
		    if {[regexp "\[\n\r\]\[ \t\]*\[^#\]\[^\r\n\]*\[\r\n\]" $line ll]} {
			set ll [string trimleft $ll]
		    } else {
			set ll ""
		    }
		}
		if {[regexp {^exec +([^ ]+) } $ll dummy ll]} {
		    regexp {([^/]+)$} [string trimright $ll "01234567890."] majorMode
		}
	    }		
	} else {
	    close $fid
	    return 
	}
    } elseif {[regexp -- {-\*- *(Mode:)? *([^ 	:;]+).*-\*-} $line "" "" majorMode]} {
	# do nothing
    } else {
	close $fid
	return
    }
    close $fid
    
    global unixMode
    set majorMode [string tolower $majorMode]
    if {[info exists unixMode($majorMode)]} {
	set m $unixMode($majorMode)
	hook::callAll unixModeHook $majorMode $name
	return $m
    } else {
	set m [mode::listAll]
	if {[set i [lsearch -exact [string tolower $m] $majorMode]] != -1} {
	    return [lindex $m $i]
	}
    }
    return 
}
} else {
    proc file::preOpeningConfigurationCheck {name {winname ""}} {
	if {![file exists "$name"]} {
	    error "No such file '$name'"
	}
	if {[catch [list ::open "$name" r] fid]} { return }
	# find first non-empty line. Return if we fail
	for { set line "" } { [string trim $line] == "" } {} {
	    if { [gets $fid line] == -1} { ::close $fid ; return }
	}
	if {$winname == ""} {set winname $name}
	global win::config

	# Check for unicode.
	if {[regexp \xFE\xFF $line]||[regexp \xFF\xFE $line]} {
	    fconfigure $fid -encoding unicode
	    set encoding unicode
	    # rewind -- real reading is still to come
	    seek $fid 0 start
	    for { set line "" } { [string trim $line] == "" } {} {
		if {[gets $fid line] == -1} {::close $fid ; return}
	    }
	    lappend win::config($winname) [list encoding unicode]
	}
	
    	set ll $line
	while {[regexp "\\((\[a-zA-Z\]+):(\[^\\)\]+)\\)(.*)" $ll "" var val ll]} {
	    lappend win::config($winname) [list $var $val]
	}
	global modeCreator
	if {[info exists modeCreator([set sig [getFileSig $name]])]} {
	    return $modeCreator($sig)
	}
	if {[regexp -nocase {^[^\n\r]*[-# \(]install($|[- \)])} $line]} {
	    global HOME
	    if {![file::pathStartsWith $name [file join $HOME Tcl]]} {
		::close $fid
		return "Inst"
	    }
	}
	if {[regexp {^#![	]*([^	\n\r]+)} $line dmy mtch] } {
	    if {[regexp {([^/]+)$} $mtch majorMode]} { 
		# remove trailing version number
		set majorMode [string trimright $majorMode "01234567890."]
		if {$majorMode == "sh"} {
		    # need to check if we're using a common unix trick
		    if {[gets $fid ll] != -1} {
			while {[string index [string trimleft $ll] 0] == "#"} {
			    if {[gets $fid ll] == -1} { ::close $fid ; return }
			} 
		    } else {
			if {[regexp {[\n\r][ \t]*[^#][^\r\n]*[\r\n]} $line ll]} {
			    set ll [string trimleft $ll]
			} else {
			    set ll ""
			}
		    }
		    if {[regexp {^exec +([^ ]+) } $ll dummy ll]} {
			regexp {([^/]+)$} [string trimright $ll "01234567890."] majorMode
		    }
		}		
	    } else {
		::close $fid
		return 
	    }
	} elseif {[regexp -- {-\*- *(Mode:)? *([^	:;]+).*-\*-} $line "" "" majorMode]} {
	    # do nothing
	} else {
	    ::close $fid
	    return
	}
	::close $fid
	
	global unixMode
	set majorMode [string tolower $majorMode]
	if {[info exists unixMode($majorMode)]} {
	    set m $unixMode($majorMode)
	    hook::callAll unixModeHook $majorMode $name
	    return $m
	} else {
	    set m [mode::listAll]
	    if {[set i [lsearch -exact [string tolower $m] $majorMode]] != -1} {
		return [lindex $m $i]
	    }
	}
	return
    }
}
    
proc file::pathEndsWith {name filelist {optionalchar ""}} {
    # This stuff is necessary on Windows where there can be a
    # variety of possible file separators (back and forwards slashes).
    # On other platforms we should find the correct file first time
    # through the loop.
    global file::separator
    lappend separators ${file::separator}
    if {${file::separator} != [file nativename ${file::separator}]} {
	lappend separators [file nativename ${file::separator}]
    }
    foreach n [list $name [file nativename $name]] {
	foreach s $separators {
	    set flist $filelist
	    if {$optionalchar != ""} {
		set reg "[quote::Regfind $s$n]${optionalchar}?$"
	    } else {
		set reg "[quote::Regfind $s$n]$"
	    }
	    if {[file exists $n] && [set ind [lsearch -exact $flist $n]] >= 0} {
		return $n
	    }
	    while {[set ind [lsearch -regexp $flist $reg]] >= 0} {
		set f [lindex $flist $ind]
		if {[file exists $f]} {
		    return $f
		}
		set flist [lrange $flist [incr ind] end]
	    }
	}
    }
    return ""
}

# Below:
#		Expanded version of old 'DblClickAux.tcl'
# 
# Authors: Tom Pollard <pollard@chem.columbia.edu>
#	  Tom Scavo   <trscavo@syr.edu>
#	  Vince Darley <vince@santafe.edu>
# 
#  modified by  rev reason
#  -------- --- --- -----------
#  9/97     VMD 1.0 reorganised for new alpha distribution.
# ###################################################################
##

#############################################################################
# Take any valid Macintosh filespec as input, and return the
# corresponding absolute filespec.  Filenames without an explicit
# folder are resolved relative to the folder of the current document.
#
proc file::absolutePath {filename} {
    set name [file tail $filename]
    set subdir [file dirname $filename]
    if { [string length $subdir] > 0 && [string index $subdir 0] != ":" } {
	set dir ""
    } else {
	# avoid "::" in path when dir end in ':' and subdir is ':'
	if { $subdir == ":" } {
	    set subdir ""
	}
	set dir [file dirname [lindex [winNames -f] 0]]
	# when window has no path (tcl shell for instance), use pwd
	if { $dir == ":" } {
	    set dir "[pwd]"
	}
    }
    return [file join $dir $subdir $name]
}


#############################################################################
# Open the file specified by the full pathname "$filename"
# If it's already open, just switch to it without any fuss.
#
proc file::openQuietly {filename} {
    edit -c -w $filename
    if {[icon -q]} {icon -o} 
}

if {[info tclversion] < 8.0} {
#############################################################################
# Searches $filename for the given pattern $searchString.  If the 
# search is successful, returns the matched string; otherwise returns
# the empty string.  If the flag 'indices' is true and the search is
# successful, returns a list of two pos giving the indices of the
# found string; otherwise returns the list '-1 -1'.
#
proc file::searchFor {filename searchString {indices 0}} {
    # Get the text of the file to be searched:
    if {[lsearch -exact [winNames -f] $filename] >= 0} {
	set fileText [getText -w $filename [minPos] [maxPos -w $filename]]
    } elseif {[file exists $filename]} {
	set fd [open $filename]
	set fileText [read $fd]
	close $fd
    } else {
	if { $indices } {
	    return [list -1 -1]
	} else {
	    return ""
	}
    }
    # Search the text for the search string:
    if { $indices } {
	if {[regexp -indices $searchString $fileText mtch]} {
	    # Fixes an apparent bug in 'regexp':
	    return [list [lindex $mtch 0] [expr {[lindex $mtch 1] + 1}]]
	} else {		
	    return [list -1 -1]
	}
    } else {
	if {[regexp -- $searchString $fileText mtch]} {
	    return $mtch
	} else {		
	    return ""
	}
    }
}

#############################################################################
#  Read and return the complete contents of the specified file.
#
proc file::readAll {fileName} {
    if {[file exists $fileName] && [file readable $fileName]} {
	set fileid [open $fileName "r"]
	set contents [read $fileid]
	close $fileid
	return $contents
    } else {
	error "No readable file '$fileName' found"
    }
}


#############################################################################
#  Save $text in $filename.  If $text is null, create an empty file.
#  Overwrite if {$overwrite} is true or the file does not exist; 
#  otherwise, prompt the user.
#
proc file::writeAll {filename {text {}} {overwrite 0}} {
    if { $overwrite || ![file exists $filename] } {
	message "Saving $filename"
	set fd [open $filename "w"]
	puts -nonewline $fd $text
	close $fd
    } else {
	if {[dialog::yesno "File $filename exists!  Overwrite?"]} {
	    file::writeAll $filename $text 1
	} else {
	    message "No file written"
	}
    }
}
} else {
    #############################################################################
    # Searches $filename for the given pattern $searchString.  If the 
    # search is successful, returns the matched string; otherwise returns
    # the empty string.  If the flag 'indices' is true and the search is
    # successful, returns a list of two pos giving the indices of the
    # found string; otherwise returns the list '-1 -1'.
    #
    proc file::searchFor {filename searchString {indices 0}} {
	# Get the text of the file to be searched:
	if {[lsearch -exact [winNames -f] $filename] >= 0} {
	    set fileText [getText -w $filename [minPos] [maxPos -w $filename]]
	} elseif {[file exists $filename]} {
	    set fd [::open $filename]
	    set fileText [::read $fd]
	    ::close $fd
	} else {
	    if { $indices } {
		return [list -1 -1]
	    } else {
		return ""
	    }
	}
	# Search the text for the search string:
	if { $indices } {
	    if {[regexp -indices -- $searchString $fileText mtch]} {
		# Fixes an apparent bug in 'regexp':
		return [list [pos::math [minPos] + [lindex $mtch 0]] \
		  [pos::math [minPos] + [expr {[lindex $mtch 1] + 1}]]]
	    } else {		
		return [list -1 -1]
	    }
	} else {
	    if {[regexp -- $searchString $fileText mtch]} {
		return $mtch
	    } else {		
		return ""
	    }
	}
    }
    
#############################################################################
    #  Read and return the complete contents of the specified file.
    #
    proc file::readAll {fileName} {
	if {[file exists $fileName] && [file readable $fileName]} {
	    set fileid [::alphaOpen $fileName "r"]
	    set contents [::read $fileid]
	    ::close $fileid
	    return $contents
	} else {
	    error "No readable file '$fileName' found"
	}
    }
    

    #############################################################################
    #  Save $text in $filename.  If $text is null, create an empty file.
    #  Overwrite if {$overwrite} is true or the file does not exist; 
    #  otherwise, prompt the user.
    #
    proc file::writeAll {filename {text {}} {overwrite 0}} {
	if { $overwrite || ![file exists $filename] } {
	    message "Saving $filename"
	    set fd [::alphaOpen $filename "w"]
	    puts -nonewline $fd $text
	    ::close $fd
	} else {
	    if {[dialog::yesno "File $filename exists!  Overwrite?"]} {
		file::writeAll $filename $text 1
	    } else {
		message "No file written"
	    }
	}
    }
}
    

#############################################################################
#  Highlight (select) a particular line in the designated file, opening the
#  file if necessary.  Returns the full name of the buffer containing the
#  opened file.  If provided, a message is displayed on the status line.
#
proc file::gotoLine {fname line {mesg {}}} {
    if {[lsearch [winNames -f] "*$fname"] >= 0} {
	bringToFront $fname
    } elseif {[lsearch [winNames] "*$fname"] >= 0} {
	bringToFront $fname
    } elseif {[file exists $fname]} {
	edit $fname
	catch {shrinkWindow 2}
    } else {
	alertnote "File \" $fname \" not found."
	return
    }
    set pos [rowColToPos $line 0]
    select [lineStart $pos] [nextLineStart $pos]
    if {[string length $mesg]} { message $mesg }
    return [win::Current]
}

#############################################################################
# Return a list of all subfolders found within $folder,
# down to some maximum recursion depth.  The top-level
# folder is not included in the returned list.
#
proc file::hierarchy {folder {depth 3}} {
    set folders {}
    if {$depth > 0} {
	global file::separator
	incr depth -1
	if {[string length [file tail $folder]] > 0} {
	    set folder "$folder${file::separator}"
	}
	foreach m [glob -nocomplain -path $folder -- *] {
	    if {[file isdirectory $m]} {
		set folders [concat $folders [list $m]]
		set folders [concat $folders [file::hierarchy ${m}${file::separator} $depth]]
	    }
	}
    }
    return $folders
}

proc file::recurse {dir} {
    if {![file exists $dir]} {return {}}
    set files {}
    foreach f [glob -nocomplain -dir $dir -- *] {
	if {[file isdirectory $f]} {
	    eval lappend files [file::recurse $f]
	} else {
	    lappend files $f
	}
    }
    return $files
}

proc file::touch {f {depth 3}} {
    if {[file isfile $f]} {
	setFileInfo $f modified [now]
	return
    }
    if {$depth == 0} {return}
    foreach ff [glob -nocomplain -dir $f *] {
	file::touch $ff [expr {$depth -1}]
    }
}

proc file::revertThese {args} {
    foreach f $args {
	foreach w [winNames -f] {
	    set ww $w
	    regsub { <[0-9]+>$} $w {} w
	    if {$f == $w || ([file nativename $f] == $w)} {
		revert -w $ww
	    }
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "file::completeFromDir" --
 # 
 #  Here's a good example:
 # 
 #    set filename [prompt::statusLineComplete "Open which file" \
 #       [list file::completeFromDir [file join $HOME Help]] \
 #        -nocache -tryuppercase]
 #  
 #  Returns the list of files in '$dir' which start with '$f'.
 # -------------------------------------------------------------------------
 ##
proc file::completeFromDir {dir f} {
    # Assuming we have proper '-tails' support in glob, we could use
    # simply 'glob -nocomplain -path [file join $dir $f] -tails *'
    set old [pwd]
    cd $dir
    set res [glob -nocomplain -path $f *]
    cd $old
    return $res
}

proc file::translatePathDelimiters {path} {
    if {[file exists $path]} {
	return [file nativename $path]
    }
    global tcl_platform
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    regsub -all "/" $path ":" path
	}
	"windows" {
	    if {[regexp -nocase {[a-z]:} $path vol]} {
		regsub -all ":" [string range $path 2 end] "/" path
		set path $vol$path
	    } else {
		regsub -all ":" $path "/" path
	    }
	}
	"unix" {
	    regsub -all ":" $path "/" path
	}
    }
    return $path
}

if {$tcl_platform(platform) == "macintosh"} {
    proc file::decompress {file} {
	global unstuffSig
	set downloadFolder [file dirname $file]
	set file [file tail $file]
	# install
	set filepre [lindex [split $file .] 0]
	# decode the downloaded file (this may happen automatically)
	set f_encoded [glob -nocomplain -path [file join $downloadFolder $filepre] *{.hqx,.bin}]
	set f_stuffed [glob -nocomplain -path [file join $downloadFolder $filepre] *.sit]
	if {[llength $f_encoded] == 1} {
	    if {[llength $f_stuffed] == 1} {
		# downloader was set to decode automatically --- we must wait
		set ff [lindex $f_stuffed 0]
		while {![file writable $ff]} {
		    switchTo '$unstuffSig'
		}
		switchTo 'ALFA'
	    } else {
		# downloader not set to decode automatically
		set ff [lindex $f_encoded 0]
		message "Decoding [file tail $ff]"
		set name [file tail [app::launchFore $unstuffSig]]
		sendOpenEvent -r '$unstuffSig' $ff
	    }
	}
	# decompress the downloaded file (this may happen automatically)
	set f_stuffed [glob -nocomplain -path [file join $downloadFolder $filepre] *.sit]
	set f_results [glob -types TEXT -nocomplain -path [file join $downloadFolder $filepre] *]
	eval lappend f_results [glob -nocomplain -path [file join $downloadFolder $filepre] -types d *]
	set f_results [eval lremove [list $f_results] $f_stuffed $f_encoded]
	if {[llength $f_results] == 0} {
	    # we didn't decompress automatically
	    set ff [lindex $f_stuffed 0]
	    message "Decompressing [file tail $ff]"
	    set name [file tail [app::launchFore $unstuffSig]]
	    sendOpenEvent -r '$unstuffSig' $ff		
	}
    }
} else {
    proc file::decompress {file} {
	switch -- [file extension $file] {
	    ".tgz" -
	    ".gz" {
		app::runScript gunzip "Gunzip application" "" 1 0 $file
	    }
	    ".tar" {
		app::execute -op untar -flags "-xf" \
		  -filename $file -flagsFirst 1 -gotErrorVar err
		if {!$err} {
		    file delete $file
		    return 1
		} else {
		    return 0
		}
	    }
	    default {
		app::runScript unstuff "Unstuffing application\
		  (Stuffit Expander 5)" "" 1 0 $file
	    }
	}
	return 1
    }
}

proc file::iscompressed {file} {
    set compressed [list .sit .bin .hqx .tar .gz .tgz .zip]
    set ext [file extension $file]
    if {[lsearch -exact $compressed $ext] != -1} {
	return 1
    } else {
	return 0
    }
}

# IMPORTANT: With Tcl 8, use 'file rename' instead.
# 
# Move a file or folder. If overwrite is 1, a file with
# the same name in the destination folder will be replaced
# by the file we move.
#
# Works with files and folder, even across volumes. 
#
# The destination may be specified as an existing folder, in 
# which case the item will be moved into this folder, keeping 
# the same name.
#
# Examples:
#   file::move anItem ::
#     moves anItem to the folder above its enclosing folder
#   
#   file::move From To
#     if To is an existing folder, moves From to this folder, making a copy 
#     and deleting From if To is not on the same volume as From
#     
#     if To is not an existing folder, renames From to To, moving it if
#     necessary and making a copy and deleting the original if To and From
#     are not on the same volume.
#     
#   file::move From To 1
#     same as previous example, but overwrites To if it exists (and is 
#     not a folder).
#
# 2001-07-11 <Frederic.Boulanger@supelec.fr>
#
proc file::move {from to {overwrite 0}} {
    # First, get the full path of both items
    set from [file::absolutePath $from]
    set to [file::absolutePath $to]
    set to_dir [file dirname $to]
    set from_dir [file dirname $from]
    set to_name [file tail $to]
    set from_name [file tail $from]
    # If the item to move does not exist -> Error
    if {![file exists $from]} {
	return -code error "$from does not exist."
    }
    # Select the right AE object builder from the type of the source item
    if { [file isdir $from] } {
	set build_from tclAE::build::foldername
    } else {
	set build_from tclAE::build::filename
    }
    if { [file isdir $to] } {
	# Moving into an existing folder: the new location
	# is the folder, the name is unchanged.
	set to_dir $to
	set to_name $from_name
	set to [file join $to_dir $to_name]
    }
    # Select the right AE object builder from the type of the dest item
    if { [file isdir $to] } {
	set build_to tclAE::build::foldername
    } else {
	set build_to tclAE::build::filename
    }
    # If the destination folder does not exist -> Error
    if {![file exists $to_dir]} {
	return -code error "Destination folder $to_dir does not exist."
    }
    # If the destination item exists, we must verify that
    # it is not the same as the source item.
    if {[file exists $to]} {
	# Get the id of each item to check if they are the same.
	# Simply comparing the full pathes does not work since several
	# pathes may describe the same item:
	#     "foo:bar:item" is the same as "foo:bar::bar:item"
	set from_id [tclAE::build::objectProperty 'MACS' ID [$build_from $from]]
	set to_id [tclAE::build::objectProperty 'MACS' ID \
	  [$build_to $to]]
	# If they are the same, do nothing
	if {$from_id == $to_id} {
	    # 			return -code error "source and destination are the same."
	    return
	}
    }
    # Get the id of the source and destination folders to compare 
    # them (see note above about Finder item comparison) and see
    # whether we will have to move the item or not.
    set from_id [tclAE::build::objectProperty 'MACS' ID \
      [tclAE::build::foldername $from_dir]]
    set to_id [tclAE::build::objectProperty 'MACS' ID \
      [tclAE::build::foldername $to_dir]]
    # If the destination and source folders are different
    if {$from_id != $to_id } {
	set moving_item 1
    } else {
	set moving_item 0
    }
    # If the tail names of "from" and "to" are different, we must rename
    # "from". This is done *before* moving the file so that "overwrite"
    # is applied to the right file name in the destination folder.
    # However, this may prevent file::move to succeed if they is an item
    # with the destination name in the source folder.
    if { "$from_name" != "$to_name" } {
	# If an item with the new name already exists
	if {[file exists [file join $from_dir $to_name]]} {
	    # If this is the destination folder
	    if {!$moving_item} {
		# If we want to overwrite existing files, delete the file
		if {$overwrite} {
		    file delete -force [file join $from_dir $to_name]
		    # 					catch {AEBuild -r 'MACS' core delo ---- \
		    # 					 [tclAE::build::filename [file join $from_dir $to_name]]} err
		    # 					if {[regexp {err(s|n):} $err]} {
		    # 						regsub {aevt\\ansr\{.*errs:} $err "" err
		    # 						regsub {\}$} $err "" err
		    # 						return -code error \
		    # 						 "Finder error while deleting [file join $from_dir $to_name]: $err"
		    # 					}
		} else {
		    # We don't want to overwrite existing files -> error
		    return -code error \
		      "A file named [file join $from_dir $to_name] already exists."
		}
	    } else {
		# We will have to move the item, so this is not the destination
		# folder and we should therefore not overwrite a file in this
		# directory. file::move fails because it cannot move and rename
		# in a single step.
		return -code error \
		  "file::move limitation: [file join $from_dir $to_name] exists."
	    }
	}
	# Rename the file. It should be possible now since we either deleted
	# an already existing file or aborted file::move
	catch {AEBuild -r 'MACS' core setd ---- \
	  [tclAE::build::propertyObject pnam \
	  [$build_from "$from"]] \
	  data "$to_name"} err
	# Look for an error
	if {[regexp {err(s|n):} $err]} {
	    regsub {aevt\\ansr\{.*errs:} $err "" err
	    regsub {\}$} $err "" err
	    return -code error \
	      "Finder error while renaming $from to $to_name: $err"
	}
	# Update "from" to reflect the name change.
	# Do *not* update "from_name" so that we can revert to
	# it in case something fails in the move process.
	set from [file join "$from_dir" "$to_name"]
    }
    if {$moving_item} {
	# move the file to its new location
	catch {AEBuild -r 'MACS' core move ---- \
	  [$build_from "$from"] \
	  insh [tclAE::build::nameObject cfol [tclAE::build::TEXT $to_dir]] \
	  alrp 'bool'(0$overwrite)} err
	# Look for an error
	if {[regexp {err(s|n):} $err]} {
	    # move failed. If we renamed the file, we should try to
	    # restore its original name (we kept it in "from_name").
	    if {$from_name != $to_name} {
		catch {AEBuild -r 'MACS' core setd ---- \
		  [tclAE::build::propertyObject pnam \
		  [$build_from "$from"]] \
		  data "$from_name"}
	    }
	    #
	    regsub {aevt\\ansr\{.*errs:} $err "" err
	    regsub {\}$} $err "" err
	    return -code error "Finder error while moving $from to $to_dir: $err"
	}
	# If we moved a file across volumes, it was copied and we must
	# delete the original file.
	if { [file exists $from] } {
	    file delete -force "$from"
	    # 		  # This does not really delete the item, it just 
	    # 		  # moves it to the trash.
	    # 			catch {AEBuild -r 'MACS' core delo ---- \
	    # 			 [$build_from "$from"]} err
	    # 			if {[regexp {err(s|n):} $err]} {
	    # 				regsub {aevt\\ansr\{.*errs:} $err "" err
	    # 				regsub {\}$} $err "" err
	    # 				return -code error "Finder error while deleting $from: $err"
	    # 			}
	}
    }
}

if {[info tclversion] < 8.0} {
    # This version is for Alpha 7.
    proc file::convertLineEndings {f eolType} {
	set contents [file::readAll $f]
	if {[regexp {\n\r} $contents]} {
	    set thisType "win"
	} elseif {[regexp {\r} $contents]} {
	    set thisType "unix"
	} else {
	    set thisType "mac"
	}
	if {$thisType == $eolType} {return}
	switch $thisType {
	    "mac" {
		if {$eolType == "win"} {
		    regsub -all "\n" $contents "\r\n" contents
		}
	    }
	    "unix" {
		if {$eolType == "win"} {
		    regsub -all "\r" $contents "\r\n" contents
		}
	    }
	    "win" {
		switch -- $eolType {
		    "mac" {
			regsub -all "\n\r" $contents "\r" contents
		    }
		    "unix" {
			regsub -all "\n\r" $contents "\n" contents
		    }
		}
	    }
	}

	if {$eolType == "mac"} {
	    file::writeAll $f $contents 1
	} else {
	    set fid [tclAE::build::resultData 'MACS' rdwr open ---- [makeAlis $f] perm true()]
	    AEBuild 'MACS' rdwr writ ---- [tclAE::build::TEXT $contents] refn $fid
	    if {$thisType == "win"} {
		AEBuild 'MACS' rdwr seof ---- $fid set2 [string length $contents]
	    }
	    AEBuild 'MACS' rdwr clos ---- $fid
	}
    }
} else {
    # This version is for Alpha 8/Alphatk.
    proc file::convertLineEndings {f eolType} {
	set contents [file::readAll $f]
	set fid [open $f w]
	switch -- $eolType {
	    "mac" {
		fconfigure $fid -translation cr
	    }
	    "unix" {
		fconfigure $fid -translation lf
	    }
	    "win" {
		fconfigure $fid -translation crlf
	    }
	}
	puts -nonewline $fid $contents
	close $fid
    }
}
