## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #	FILE: "filesetsCoreTypes.tcl"
 #					created: 05/01/2000 {14:10:07 PM} 
 #				   last update: 12/04/2001 {15:39:02 PM} 
 #	Author:	Vince Darley
 #	E-mail:	<vince@santafe.edu>
 #	  mail:	317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #	   www:	<http://www.santafe.edu/~vince/>
 #	
 # This file contains the implementation of the basic fileset types
 # which are included in Alpha's core.  The filesets library code is
 # in the file filesets.tcl.
 # ###################################################################
 ##

# Make sure we've loaded the package.
alpha::package require filesets

# Procs for procedural filesets
proc procFilesetRecurseIn {} {
    return [file::recurse [get_directory -p "Search recursively\
      in which folder?"]]
}

proc procFilesetOpenWindows {} { 
    set res {}
    foreach f [winNames -f] {
	lappend res [win::StripCount $f]
    }
    return $res
}

proc procFilesetHierarchyTopWin {} {
    global win::Active
    foreach f [set win::Active] {
	set fn [win::StripCount $f]
	if {[file exists $fn]} {
	    return [file::recurse [file dirname $fn]]
	}
    }
    return ""
}

proc procFilesetDirTopWin {} { 
    global win::Active
    foreach f [set win::Active] {
	set fn [win::StripCount $f]
	if {[file exists $fn]} {
	    return [glob -types TEXT -nocomplain -dir [file dirname $fn] *]
	}
    }
    return ""
}

namespace eval fileset::fromDirectory {}
namespace eval fileset::fromHierarchy {}

proc fileset::fromDirectory::setDetails {name dir pat ignore} {
    global gfileSets fileSetsExtra
    set gfileSets($name) [list $dir $pat]
    set fileSetsExtra($name) $ignore
    modifyFileset $name
}

proc fileset::recurseIn::setDetails {name dir} {
    global gfileSets
    set gfileSets($name) [list $dir *]
    modifyFileset $name
}

proc fileset::fromHierarchy::setDetails {name dir pat depth ignore} {
    global gfileSets fileSetsExtra
    set gfileSets($name) [list $dir $pat $depth]
    set fileSetsExtra($name) $ignore
    modifyFileset $name
}

proc fileset::fromHierarchy::getDialogItems {name} {
    global gfileSets fileSetsExtra
    if {[info tclversion] < 8.0} {
	set depth {1 2 3 4}
    } else {
	set depth {1 2 3 4 5 6 7}
    }
    lappend res \
      [list folder "Fileset dir:" [lindex $gfileSets($name) 0] \
      "Base directory for fileset"] \
      [list variable "File pattern:" [lindex $gfileSets($name) 1] \
      "Only include files which match this pattern"] \
      [list [list menu $depth] "Depth of hierarchy?"\
      [lindex $gfileSets($name) 2] "Number of levels deep in the disk\
      hierarchy to include"]
    if {[info exists fileSetsExtra($name)]} {
	set cur $fileSetsExtra($name)
    } else {
	set cur ""
    }
    lappend res [list variable "List of file patterns to ignore:" $cur]
    set res
}

proc fileset::recurseIn::getDialogItems {name} {
    global gfileSets
    lappend res \
      [list folder "Fileset dir:" [lindex $gfileSets($name) 0]]
    return $res
}

proc fileset::fromDirectory::getDialogItems {name} {
    global gfileSets fileSetsExtra
    lappend res \
      [list folder "Fileset dir:" [lindex $gfileSets($name) 0]\
      "Base directory for fileset"] \
      [list variable "File pattern:" [lindex $gfileSets($name) 1]\
      "Only include files which match this pattern"]
    if {[info exists fileSetsExtra($name)]} {
	set cur $fileSetsExtra($name)
    } else {
	set cur ""
    }
    lappend res [list variable "List of file patterns to ignore:" $cur]
    set res
}

#  Creation of basic fileset types  #

proc fileset::procedural::create {} {
    global gfileSets gfileSetsType filesetsNotInMenu
    set name [getline "Name for this fileset"]
    if {![string length $name]} return
    set gfileSetsType($name) "procedural"
    set p procFileset[join $name ""]
    set gfileSets($name) $p
    prefs::tclAddLine "\# procedure to list files in fileset '$name' on the fly"
    prefs::tclAddLine "proc $p \{\} \{"
    prefs::tclAddLine "\t"
    prefs::tclAddLine "\}"
    if {[dialog::yesno "I've added a template for the procedure to your\
      'prefs.tcl'. Do you want to edit it now?"]} {
	prefs::tclEdit
	goto [maxPos]
	beep
	message "Make sure you 'load' the new procedure."
    }
    lappend filesetsNotInMenu $name
    return $name
}

# Return empty menu.
proc fileset::procedural::updateContents {name {andMenu 0}} {
    return
}

proc fileset::fromDirectory::create {args} {
    global gfileSets gfileSetsType fileSetsExtra
    
    if {[llength $args] == 3} {
	newforeach {name dir pat} $args {}
    } elseif {[llength $args] != 0} {
	return -code error "Bad args"
    } else {
	newforeach {name dir pat} [getFilesetDirectoryAndPattern] {}
	if {![info exists name] || ![string length $name]} return
	set filePatIgnore [getline "List of file patterns to ignore:" ""]
	if {$filePatIgnore != ""} {
	    set fileSetsExtra($name) $filePatIgnore
	}
    }
    set gfileSets($name) [list $dir $pat]
    set gfileSetsType($name) "fromDirectory"
    
    return $name
}

proc fileset::recurseIn::getRoot {name} {
    global gfileSets
    return [lindex $gfileSets($name) 0]
}

proc fileset::recurseIn::create {} {
    global gfileSets gfileSetsType
    
    set name [getline "New fileset name:" ""]
    if {![string length $name]} return
    
    set dir [get_directory -p "New fileset dir:"]
    if {![string length $dir]} return

    set gfileSets($name) [list $dir *]
    set gfileSetsType($name) "recurseIn"
    
    return $name
}

proc getFilesetDirectoryAndPattern {} {
    global gfileSets fileSetsExtra
    set name [getline "New fileset name:" ""]
    if {![string length $name]} return
    
    set dir [get_directory -p "New fileset dir:"]
    if {![string length $dir]} return
    
    set filePat [getline "File pattern:" "*"]
    if {![string length $filePat]} return
    
    return [list $name $dir $filePat]
}

proc fileset::fromDirectory::updateContents {name {andMenu 0}} {
    if {$andMenu} {
	set menu [list]
	foreach m [getFilesInSet $name] {
	    lappend menu "[file tail $m]&"
	}
	return [filesetMenu::makeSub $name $name fileset::openItemProc \
	  [lsort -increasing $menu]]
    } else {
	return [list]
    }
}

proc fileset::fromHierarchy::create {args} {
    global gfileSets gfileSetsType
    
    if {[llength $args] == 4} {
	newforeach {name dir pat depth} $args {}
    } elseif {[llength $args] != 0} {
	return -code error "Bad args"
    } else {
	newforeach {name dir pat} [getFilesetDirectoryAndPattern] {}
	if {![string length $name]} return
	if {[info tclversion] < 8.0} {
	    set depth [listpick -p "Depth of hierarchy?" -L 3 {1 2 3 4}]
	} else {
	    set depth [listpick -p "Depth of hierarchy?" -L 3 {1 2 3 4 5 6 7}]
	}
	if { $depth == "" } {set depth 3}
    }
	
    set gfileSetsType($name) "fromHierarchy"
    set gfileSets($name) [list $dir $pat $depth]
    
    return $name
}

proc fileset::fromHierarchy::getRoot {name} {
    global gfileSets
    return [lindex $gfileSets($name) 0]
}

proc fileset::fromHierarchy::updateContents {name {andMenu 0}} {
    global fileSets gfileSets fileSetsExtra
    set dir [lindex $gfileSets($name) 0]
    set patt [lindex $gfileSets($name) 1]
    set depth [lindex $gfileSets($name) 2]
    if {[info exists fileSetsExtra($name)]} {
	set ignore $fileSetsExtra($name)
    } else {
	set ignore ""
    }
    # we make the menu as a string, but can bin it if we like
    set menu [menu::buildHierarchy [list $dir] $name\
      fileset::openItemProc filesetTemp $patt $depth $name $ignore]
    
    # we need to construct the list of items
    set fileSets($name) {}
    if {[info exists filesetTemp]} {
	foreach n [array names filesetTemp] {
	    lappend fileSets($name) $filesetTemp($n)
	}
    }
    return $menu
}

proc fileset::fromHierarchy::selected {fset menu item} {
    global gfileSets
    set dir [lindex $gfileSets($fset) 0]
    set ff [getFilesInSet $fset]
    if { $fset == $menu } {
	# it's top level
	if {[set match [lsearch $ff [file join ${dir} $item]]] >= 0} {
	    autoUpdateFileset $fset
	    file::openAny [lindex $ff $match]
	    return
	}
    }
    # the following two are slightly cumbersome, but give us the best
    # chance of finding the correct file given any ambiguity (which can
    # certainly arise if file and directory names clash excessively).
    if {[set match [lsearch $ff [file join ${dir} ${menu} $item]]] >= 0} {
	autoUpdateFileset $fset
	file::openAny [lindex $ff $match]
	return
    }
    if {[set match [lsearch $ff [file join ${dir} * ${menu} $item]]] >= 0} {
	autoUpdateFileset $fset
	file::openAny [lindex $ff $match]
	return
    }
    if {[string range $item 0 1] == " -"} {
	set item [string range $item 1 end]
	return [fileset::fromHierarchy::selected $fset $menu $item]
    }
    error "Weird! Couldn't find it."
}

proc fileset::fromOpenWindows::create {} {
    global gfileSets gfileSetsType
    
    set name [prompt "Create fileset containing current\
      windows under what name?" "OpenWins"]
    
    set gfileSets($name) [winNames -f]
    set gfileSetsType($name) "list"

    return $name
}

proc fileset::fromOpenWindows::updateContents {name {andMenu 0}} {
     if {$andMenu} {
	set menu [list]
	foreach m [getFilesInSet $name] {
	    lappend menu "[file tail $m]&"
	}
	return [filesetMenu::makeSub $name $name fileset::openItemProc \
	  [lsort -increasing $menu]]
     } else {
	return [list]
     }
}





