## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "filesetsUtils.tcl"
 #					created: 05/01/2000 {15:08:49 PM} 
 #				   last update: 10/12/2001 {16:58:07 PM} 
 #	
 # ###################################################################
 ##


## 
 # -------------------------------------------------------------------------
 #	 
 #	"iterateFileset" --
 # 
 #  Utility procedure to iterate over all files in a project, calling some
 #  predefined function '$fn' for each member of project '$proj'.  The
 #  results of such a call are passed to '$resfn' if given.  Finally "done"
 #  is passed to 'resfn'.
 #	 
 # -------------------------------------------------------------------------
 ##
proc iterateFileset { proj fn { resfn \# } } {
    eval $resfn "first"
    
    set check [expr {![catch {[fileset::type $proj]IterateCheck check}]}]
    
    foreach ff [getFileSet $proj] {
	if { $check && [[fileset::type $proj]IterateCheck $proj $ff] } {
	    continue
	}
	set res [eval $fn [list $ff]]
	eval $resfn [list $res]
    }
    
    if {$check} {
	catch {[fileset::type $proj]IterateCheck done}
    }
    
    eval $resfn "done"
    
}

proc filesetRememberOpenClose { file } {
    global fileset_openorclosed
    set fileset_openorclosed [list "$file" [lsearch -exact [winNames -f] $file]]
}

proc filesetRevertOpenClose { file } {
    global fileset_openorclosed
    if { [lindex $fileset_openorclosed 0] == "$file" } {
	if { [lindex $fileset_openorclosed 1] < 0 } {
	    killWindow
	}
    }	
    catch {unset fileset_openorclosed}
}

#  Utils  #

proc printFileset { {fset ""}} {
    if {[catch {pickFileset $fset "Print which Fileset?"} fset]} {return}
    foreach f [getFilesInSet $fset] {
	print $f
    }
}

proc browseFileset {{fset ""}} {
    global tileLeft tileTop tileWidth errorHeight
    
    if {[catch {pickFileset $fset {Fileset?}} fset]} {return}
    
    foreach f [getFilesInSet $fset] {
	lappend text "\t[file tail $f]\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t$f"
    }
    new -n "* FileSet '$fset' Browser *" -g $tileLeft $tileTop 200 $errorHeight \
      -m Brws -info "(<cr> to go to file)\r-----\r[join $text \r]"
    select [nextLineStart [nextLineStart [minPos]]] [nextLineStart [nextLineStart [nextLineStart [minPos]]]]
    message ""
}	

proc saveEntireFileset { fset } {
    foreach f [getFilesInSet $fset] {
	if { ![catch {getWinInfo -w $f arr}] && $arr(dirty)} { 
	    bringToFront $f
	    save 
	}
    }
}

proc closeEntireFileset { {fset ""} } {
    if {[catch {pickFileset $fset "Close which fileset?"} fset]} {return}
    
    foreach f [getFilesInSet $fset] {
	if {![catch {getWinInfo -w $f arr}]} {
	    bringToFront $f
	    killWindow
	}
    }
}

proc fileToAlpha {f} {
    file::setSig $f ALFA
}

proc filesetToAlpha {} {
    if {[catch {pickFileset "" {Convert all files from which fileset?}} fset]} {return}
    iterateFileset $fset fileToAlpha
}

proc openEntireFileset {} {
    set fset [pickFileset "" "Open which fileset?"]
    
    # we use our iterator in case there's something special to do
    iterateFileset $fset "edit -c -w"
}

proc openFilesetFolder {{fset ""}} {
    set fset [pickFileset $fset "Open which fileset's folder?"]
    
    set dir [fileset::getBaseDirectory $fset]

    if {[string length $dir]} {
	file::showInFinder $dir
    } else {
	alertnote "Fileset not connected to a folder."
    }
}

if {$tcl_platform(platform) == "macintosh"} {

proc stuffFileset {{fset ""}} {
    set fset [pickFileset $fset "Which fileset shall I stuff?"]
    if {[string length $fset]} {
	set dir [fileset::getBaseDirectory $fset]
	if {[string length $dir] && [dialog::yesno "Stuff entire directory '$dir'?"]} {
	    app::launchFore DStf
	    global file::separator
	    sendOpenEvent reply 'DStf' "${dir}${file::separator}"
	} else {			
	    app::launchFore DStf
	    eval [list sendOpenEvents 'DStf'] [getFilesInSet $fset]
	}		
	sendQuitEvent 'DStf'
    }
}

proc zipFileset {{fset ""}} {
    set fset [pickFileset $fset "Which fileset shall I zip?"]
    if {[string length $fset]} {
	set dir [fileset::getBaseDirectory $fset]
	if {[string length $dir] && [dialog::yesno "Zip entire directory '$dir'?"]} {
	    app::launchFore DZip
	    global file::separator
	    sendOpenEvent reply 'DZip' "${dir}${file::separator}"
	} else {			
	    app::launchFore DZip
	    eval [list sendOpenEvents 'DZip'] [getFilesInSet $fset]
	}		
	sendQuitEvent 'DZip'
    }
}

} else {
    proc stuffFileset {{fset ""}} {
	set fset [pickFileset $fset "Which fileset shall I stuff?"]
	if {[string length $fset]} {
	    set dir [fileset::getBaseDirectory $fset]
	    if {[string length $dir] && [dialog::yesno "Stuff entire directory '$dir'?"]} {
		app::runScript stuff "Stuffing application (DropStuff)" "" 1 0 $dir
	    } else {			
		foreach f [getFilesInSet $fset] {
		    app::runScript stuff "Stuffing application (DropStuff)" "" 1 0 $f
		}
	    }		
	}
    }
    proc zipFileset {{fset ""}} {
	set fset [pickFileset $fset "Which fileset shall I zip?"]
	if {[string length $fset]} {
	    set dir [fileset::getBaseDirectory $fset]
	    if {[string length $dir] && [dialog::yesno "Zip entire directory '$dir'?"]} {
		app::runScript zip "Zipping application" "" 1 0 $dir
	    } else {			
		foreach f [getFilesInSet $fset] {
		    app::runScript zip "Zipping application" "" 1 0 $f
		}
	    }		
	}
    }
}

proc wordCountFileset {{fset ""}} {
    set fset [pickFileset $fset "Word count in which fileset?"]
    iterateFileset $fset wordCountProc filesetUtilWordCount
}

proc filesetUtilWordCount {count} {
    global fs_ccount fs_wcount fs_lcount
    switch $count {
	"first" {
	    set fs_ccount 0
	    set fs_wcount 0
	    set fs_lcount 0
	}       
	"done" {
	    alertnote "There were $fs_ccount lines, $fs_wcount words and $fs_lcount chars"
	    unset fs_ccount fs_wcount fs_lcount
	}
	default {
	    incr fs_ccount [lindex $count 2]
	    incr fs_wcount [lindex $count 1]
	    incr fs_lcount [lindex $count 0]
	}
    }
}


## 
 # -------------------------------------------------------------------------
 # 
 # "wordCountProc" --
 # 
 #  Completely new proc which does the same as the old one
 #  without opening lots of windows.
 #  *Very* memory comsuming for large files, though.
 #  But I think the old one was equally memory consuming.
 #  
 #  Ok, this is not exactly a bug fix. It's a IMHO better option.
 #  
 # -------------------------------------------------------------------------
 ##

proc wordCountProc {file} {
    message "Counting [file tail $file]"
    set fid [alphaOpen $file r]
    set filecont [read $fid]
    close $fid
    if {[regexp {\n\r} $filecont]} {
	set newln "\n\r"
    } elseif {[regexp {\n} $filecont]} {
	set newln "\n"
    } else {
	set newln "\r"
    }
    set lines [expr {[regsub -all -- $newln $filecont " " filecont] + 1}]
    set chars [string length $filecont]
    regsub -all {[!=;.,\(\#\=\):\{\"\}]} $filecont " " filecont
    set words [llength $filecont]
    return "$chars $words $lines"
}


#  From search dialog  #

proc findNewFileset {} {
    return [newFileset]
}


proc findNewDirectory {} {
    global currFileSet temporaryFilesets
    
    set dir [get_directory -p "Scan which folder?"]
    if {![string length $dir]} return
    
    set name [file tail $dir]
    fileset::fromDirectory::create $name $dir *

    lappend temporaryFilesets $name
    set currFileSet $name

    updateCurrentFileset
    return $name
}

## 
 # -------------------------------------------------------------------------
 # 
 # "replaceInFileset" --
 # 
 #  Quotes things correctly so searches work, and adds a check on
 #  whether there are any windows.
 #  
 #  This procedure is a little obsolete, given what's in the supersearch
 #  package.  However some people may find it useful.
 # -------------------------------------------------------------------------
 ##
proc replaceInFileset {} {
    global win::NumDirty
    set how [dialog::optionMenu "Search type:" \
      [list "Textual replace" "Case-independent textual replace" \
      "Regexp replace" "Case-independent regexp replace"] "" 1]
    set from [prompt "Search string:" [searchString]]
    searchString $from
    if {$how < 2} {set from [quote::Regfind $from]}
    
    set to [prompt "Replace string:" [replaceString]]
    replaceString $to
    if {$how < 2} {set to [quote::Regsub $to]}
    if {[catch {regsub -- $from "$from" $to dummy} err]} {
	alertnote "Regexp compilation problems: $err"
	return
    }
    set fsets [pickFileset "" "Which filesets?" "multilist"]
    
    if {${win::NumDirty}} {
	if {[buttonAlert "Save all windows?" "Yes" "Cancel"] != "Yes"} return
	saveAll
    }
    
    set cid [scancontext create]
    set changes 0
    if {$how & 1} {
	set case "-nocase"
    } else {
	set case "--"
    }
    
    scanmatch $case $cid $from {set matches($f) 1 ;incr changes}
    foreach fset $fsets {
	foreach f [getFileSet $fset] {
	    if {![catch {set fid [alphaOpen $f]}]} {
		message "Looking at '[file tail $f]'"
		scanfile $cid $fid
		close $fid
	    }
	}
    }
    
    scancontext delete $cid
    
    foreach f [array names matches] {
	message "Modifying ${f}"
	set cid [alphaOpen $f "r"]
	if {[regsub -all $case $from [read $cid] $to out]} {
	    set ocid [alphaOpen $f "w+"]
	    puts -nonewline $ocid $out
	    close $ocid
	}
	close $cid
    }
    
    eval file::revertThese [array names matches]
    message "Replaced $changes instances"
}
