## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "help.tcl"
 #                                    created: 07/21/2000 {18:31:50 PM} 
 #                                last update: 12/13/2001 {18:52:23 PM} 
 #                                
 # The last 4 procs in this file are copyright (c) Pete Keleher
 # ###################################################################
 ##

namespace eval help {}

hook::register openHook {help::openHook} Text

proc help::openHook {winName} {
    global HOME helpFileWindowSize
    
    if {[win::IsFile $winName]} {
	set test [file::pathStartsWith $filename [file join $HOME Help]]
    } else {
	set test [string match {\* '*' Help \*} $filename]
    }
    if {$test} {
	set size $helpFileWindowSize
	# Wish that we had 'quote::Unprettify' or 'quote::Unmenuify' ...
	set s [string tolower [string index $size 0]]
	regsub -all { +} [string range $size 1 end] {} ize
	eval ${s}${ize}
    } 
}


## 
 # -------------------------------------------------------------------------
 # 
 # "alphaHelp" --
 # 
 #  Called from about box
 # -------------------------------------------------------------------------
 ##

proc alphaHelp {} {
    global HOME alpha::platform
    
    set files [list "Alpha Manual" "Quick Start" "Readme"]
    foreach f $files {
        if {[file exists [file join $HOME Help $f]]} {
            helpMenu $f
            return
        }
    }
    # No help files present ...
    if {${alpha::platform} == "alpha"} {
        url::execute "http://www.alpha.olm.net/"
    } else {
        url::execute "http://www.santafe.edu/~vince/Alphatk.html"
    } 
}

proc register {} {
    global HOME alpha::platform
    
    if {${alpha::platform} == "alpha"} {
       if {[file exists [file join $HOME Register]]} {
            launch -f [file join $HOME Register]
        } else {
            url::execute "http://www.alpha.olm.net/"
        } 
    } else {
        alpha::finalStartup
    }
}

#  Help Menu procedures  #

namespace eval help {}

## 
 # -------------------------------------------------------------------------
 # 
 # "help::buildMenu" --
 # 
 # Note: All file extensions are removed from files in the Help folder when
 # building the Help menu, and no duplicates are inserted.  If there are
 # two similarly named files with different extensions, the user can be
 # presented with a list pick dialog (in the proc: helpMenu below).
 # 
 # Subdirectories --
 # 
 # If tcl version is less than 8.0, subdirectories are simply added as menu
 # items.  Selecting the item will present the user with a list-pick dialog
 # with the directory's contents, unless there is a .tcl file to be
 # evalutated first.  
 # 
 # 8.0 and greater, subdirectories are added as submenus, unless there is a
 # .tcl file present.   In this case only the subdirectory's name is added
 # to the menu, and selecting it evaluates the .tcl file.
 # -------------------------------------------------------------------------
 ##

proc help::buildMenu {} {
    global HOME alpha::platform alphaDevHelpFiles
    
    if {![catch {glob -dir [file join $HOME Help] *} helpFiles]} {
	foreach f $helpFiles {
	    lappend files [file rootname [file tail $f]]
	} 
    } else {
	addHelpMenu "No Help files found"
	return
    }

    lappend men "Alpha Home Page" "Alphatk Home Page" "Alpha's www FAQ" \
      "(-" "Readme" "Readme Alphatk"
    
    # We automatically remove any which aren't actually there.
    lappend men \
      "Quick Start" "Alpha Manual" "Packages" \
      "Examples Help" "Bug Reports and Debugging" "Report A Bug"\
      "(-" \
      "Filesets Help" "FTP menu Help" "HTML Help" "LaTeX Help" \
      "(-"

    set alphaDevHelpFiles [list \
      "Extending Alpha" "Alpha Developers FAQ" "Alpha Commands" \
      "Tcl Commands" "Tcl 8.4 Commands" \
      "Error Help" "AEGizmos" "TclAE Help" "Help Files Help" \
      "Dialogs Help"]

    lappend men "Changes - Alpha" "Changes - Alphatk" "Changes - AlphaTcl" "(-"

    set unwanted $alphaDevHelpFiles

    # In this list, and for the remaining files in the Help folder, even if
    # there are multiple formats included the filename root is only added
    # once.  If there are no files, we don't include them.
    foreach f $men {
        if {$f != "(-" && [lsearch -exact $files $f] < 0} {
            set men [lremove $men $f]
        }
    }
    # Add the remaining filename root-tails.
    foreach f [lsort -ignore $files] {
        if {[lsearch -exact $men $f] < 0} {
	    # Remove anything which might have been left by cvs
	    # or by some backup packages.
            if {[string index $f 0] != "."} {
		lappend men $f
	    }
        }
    }
    
    # Remove unwanted files
    lappend unwanted "Images" "CVS"
    if {${alpha::platform} != "alpha"} {
	lappend unwanted "Alpha Home Page" "Changes - Alpha" "Registering"
    } else {
	lappend unwanted "Alphatk Home Page" "Readme Alphatk" \
	  "Changes - Alphatk" "Registering Alphatk"
    }
    if {[info tclversion] < 8.0} {
	lappend unwanted "Tcl 8.4 Commands" "Unicode and Encodings Help"
    } else {
	lappend unwanted "Tcl Commands"
    }
    
    foreach f $unwanted {
	set idx [lsearch -exact $men $f]
	if {$idx >= 0} {set men [lreplace $men $idx $idx]}
    }
    
    regsub -all {\(-[ \t\r\n]+\(-} $men {\(-} men
    foreach f $men {
        if {[info tclversion] >= 8.0 && \
          [file isdirectory [file join $HOME Help $f]] && \
          ![file exists [file join $HOME Help ${f}.tcl]]} {
            # We only add sub-menus for tclversion 8.0 if a given
            # subdirectory doesn't have an associated .tcl file.
            set subfiles ""
            foreach sub [glob -nocomplain -dir [file join $HOME Help $f] *] {
                lappend subfiles [file root [file tail $sub]]
            }
            addHelpMenu [list Menu -n $f $subfiles]
        } else {
            addHelpMenu $f
        }
    }
}

proc help::MenuProc {menu item} {
    menu::generalProc help $item
}

proc helpMenu {args} {
    uplevel 1 help::openFile $args
}

## 
 # -------------------------------------------------------------------------
 # 
 # "help::openFile" --
 # 
 # Given the name delivered by the Help menu, find all files which contain
 # it (including the name itself, and any variations with file extensions). 
 # If there are multiple files, or if the item selected is actually a
 # directory containing other files, offer the list to the user.
 # 
 # Thus there can be multiple versions of "LaTeX Help" (for example), such
 # as "LaTeX Help", "LaTeX Help.html", "LaTeX Help.pdf", "LaTeX Help",
 # which will be dealt with by help::openDirect. 
 # 
 # If there is a "<something> Help.tcl" file, as in "LaTeX Help.tcl", then
 # that file is sourced immediately.  It's up to the script to decide if
 # more options will be presented. 
 # -------------------------------------------------------------------------
 ##

proc help::openFile {args} {
    global HOME alpha::platform
    
    # Set the filename delivered by the Help menu.
    set filename [eval [list file join $HOME Help] $args]
    if {[file exists ${filename}.tcl]} {
        # There's a .tcl script to evaluate.  It is up to the script to
        # decide if more options will be presented.
        help::openDirect ${filename}.tcl
        return
    } elseif {[file tail $filename] == "No Help files found"} {
        # The list was built without any files.
        if {[askyesno "No help files were found -- perhaps you need to \
          re-install them. Would you like to open Alpha's home page?"] != "no"} {
            if {${alpha::platform} == "alpha"} {
                url::execute "http://www.alpha.olm.net/"
            } else {
                url::execute "http://www.santafe.edu/~vince/Alphatk.html"            } 
        } else {
            error "No Help files found in the top level of Alpha's hierarchy."
        }
    } 
    # Find out what file options we have.
    set fileDir  [file dirname $filename]
    if {[file isdirectory $filename]} {
        # This item is actually a subdirectory.
        set fileDir $filename
        set files [glob -nocomplain -dir $filename *]
    } elseif {[catch {glob -path "${filename}." *} files]} {
        # There are no files with this name plus an extension.
        set files ""
    } 
    if {[file isfile $filename]} {
        # The filename exists without an extension, so we add that too.
        lappend files $filename
    } 
    if {![llength $files]} {
        # "filename" shouldn't have even been in the menu ...
        message "Sorry, no \"$filename\" files were found."
        error "No \"$filename\" files were found."
    }
    # Now that the list of potential files has been set, check some of the
    # "help menu" preferences to see if we have some default action, or if
    # we should adjust the list.
    eval [list help::askOrOpen [file tail $filename]] $files
}

proc help::askOrOpen {what args} {
    set files [eval help::getChoices $args]
    if {[llength $files] == 1} {
	set helpFile [lindex $files 0]
    }
    if {![info exists helpFile]} {
	# No file has been set yet.
	if {[llength $files] == 1} {
	    # Only one file found.
	    set helpFile [lindex $files 0]
	} else {
	    # There is more than one file, with different extensions.
	    foreach f $files {
		set ext [string range [file extension $f] 1 end]
		set tail [file tail $f]
		if {$ext == ""} { set ext "text" }
		if {$ext == "txt" } { set ext "text" }
		set ch "View $ext help ($tail)"
		lappend choices $ch
		set choice($ch) $f
	    }
	    lappend choices "(Set Help preferences to avoid this dialog )"
	    set helpFile [listpick -p "\"$what\" options:" $choices]
	    if {$helpFile == "(Set Help preferences to avoid this dialog )"} {
		dialog::preferences preferences "Help"
		eval helpMenu $args
		return
	    }
	    set helpFile $choice($helpFile)
	    # In case $helpFile is itself a directory ...  This will also
	    # help make sure that the help menu item can work even if a
	    # .tcl file has been deleted.
	    while {[file isdirectory $helpFile]} {
		set files [glob -nocomplain -dir $helpFile *]
		set fileTails ""
		foreach f $files {
		    lappend fileTails [file tail $f]
		}
		set fileDir $helpFile
		set helpFile [listpick -p \
		  "\"[file tail $helpFile]\"  options :" \
		  [lsort $fileTails]]
		set helpFile [file join $fileDir $helpFile]
	    }
	}
    }
    help::openDirect $helpFile
}

# Given a list of possible filenames or file-rootnames,
# find a list of files which exist and then of those
# find ones which satisfy the users preferences.
proc help::getChoices {args} {
    global helpChoices
    set choices {}

    # Build a list of files which exist and which satisfy
    # one of the known extension types, plus anything ending
    # in .txt or a straight filename with no extension.
    foreach f $args {
	if {[file exists $f] && ![file isdirectory $f]} {
	    if {[lsearch -exact $choices $f] == -1} {
		lappend choices $f
	    }
	}
	if {[file extension $f] == ""} {
	    foreach ext $helpChoices {
		if {[file exists ${f}.$ext]} {
		    if {[lsearch -exact $choices ${f}.$ext] == -1} {
			lappend choices ${f}.$ext
		    }
		}
	    }
	    if {[file exists ${f}.txt]} {
		if {[lsearch -exact $choices ${f}.txt] == -1} {
		    lappend choices ${f}.txt
		}
	    }
	}
    }
    # Now we need to just take items from this list of
    # choices which satisfy the user's help preferences
    global preferedHelpFormat helpExtensions
    
    if {$preferedHelpFormat == "Always offer options"} {
	return $choices
    }

    foreach pref {prefered secondChoice thirdChoice} {
	global ${pref}HelpFormat
	set val [set ${pref}HelpFormat]
	foreach f $choices {
	    set ext [file extension $f]
	    if {[lsearch -exact $helpExtensions($val) $ext] != -1} {
		# We found one, and since we're looking in order
		# of the user's preference, return it.
		return [list $f]
	    }
	}
    }
    return {}
}

## 
 # ----------------------------------------------------------------------
 #       
 # "help::mimicHelpMenu" --
 #      
 # This proc can be used in the Alpha Developer Menu's "Help File Marking"
 # menu, or called by other code.  Assume that this is for the current
 # open window unless called from a shell prompt.
 #      
 # "help::mimicHelpMenu -choose" will offer a list-pick dialog to select
 # files for pre-marking.
 # 
 # "help::mimicHelpMenu -all" will pre-mark all Help files, useful for
 # public releases of Alpha (not Alphatk).
 # 
 # Both options will only select valid (i.e. Text mode) files in the top
 # level of the Help folder. 
 # ----------------------------------------------------------------------
 ##

proc help::mimicHelpMenu {{files ""}} {
    
    global HOME alpha::platform
    
    set fileList ""
    # Should we select of list of files to mark/hyper ...
    if {$files == "-choose" || $files == "-all"} {
        # Create the list of all valid help files (those in Text mode).
        message "Creating the list of valid Help files for marking/hyperizing "
        foreach f [glob -dir [file join $HOME Help] *] {
            if {[file isfile $f] && [win::FindMode $f] == "Text"} {
                lappend helpFiles $f
            } 
        } 
        if {$files == "-choose"} {
            # Offer a list-pick of Help files to mark/hyper.
            foreach f $helpFiles {
                if {[file isfile $f] && [win::FindMode $f] == "Text"} {
                    lappend helpFileTails [file tail $f]
                } 
            }
            set helpTailsList [listpick -l \
              -p "Choose some Help files to pre-mark" $helpFileTails]
            foreach f $helpTailsList {
                lappend fileList [file join $HOME Help $f]
            } 
        } elseif {${alpha::platform} != "alpha"} {
            alertnote "Pre-marking/hyperizing help files is only useful\
              for the Macintosh platform."
            error "\"help::mimicHelpMenu -all\" is only useful on the Macintosh."
        } else {
            # Select all valid help files to mark.
            set fileList $helpFiles
        } 
    }
    # ...  or given a single file argument "f", which is surrounded either
    # by quotes or brackets, assume that the file is in the Help folder
    # unless the entire path is given ...
    if {$files != "" && $fileList == ""} {
        set f1 [list $files]
        set f2 [file join $HOME Help $files]
        if {[file isfile $f1]} {
            lappend fileList $f1
        } elseif {[file isfile $f2]} {
            lappend fileList $f2
        } 
    } 
    # ...  or given no arguments, use the current window.
    if {$files == ""} {
        set fileList [list [win::Current]]
    }
    # Do we have any files to mark/hyper?
    if {![llength $fileList]} {
        message "No valid files were selected."
        error "No valid files were selected."
    } 
    # Now we actually mark/hyper the file.
    foreach f $fileList {
        # We already know that all files in "f2" exist, are complete paths.
        help::removeCHMCleanly $f
        help::openDirect [win::Current]
        setWinInfo dirty 0
        setWinInfo read-only 0
        if {$files == "-all"} {
            # If only marking files for public release, close them.
            shrinkFull
            menu::fileProc "File" "close"
        }
    }
}

proc help::preMarkAllHelpFiles {} {
    if {[askyesno "Do you really want to mark and hyper all help files?\
      This could take a little while "] != "no"} {
        help::mimicHelpMenu -all
    } 
}

#  Open File procedures  #

## 
 # -------------------------------------------------------------------------
 # 
 # "help::openDirect" --
 # 
 #  Called from the Help menu to evaluate (.tcl), send (.html), or just
 #  open/mark/hyper (no extension) a file .  Other formats opened by OS.
 #  ------------------------------------------------------------------------
 ##

proc help::openDirect {filename} {
    if {![file exists $filename]} {
        global HOME
        set filename [file join $HOME Help $filename]
    }
    if {[file isfile $filename]} {
        switch -- [file extension $filename] {
            ".tcl" {
                uplevel \#0 [list source $filename]
            }
            ".html" {
		htmlView $filename
            }
	    ".txt" -
            "" {
                edit -r -c -tabsize 4 $filename
		if {[icon -q]} {icon -o} 
                global mode
                if {$mode == "Chng"} {
                    Chng::MarkFile
                    help::hyperiseUrls
                }
                if {$mode == "Text" && ![llength [getColors]]} {
                    catch {
                        message "Please wait: Colouring and marking the help file"
                        help::hyperiseEmailAddresses
                        help::hyperiseUrls
                        set commandFiles [list \
                          "Diff Help" "Error Help" "Perl Commands" \
                          "Regular Expressions" "Tcl *Commands*" \
                          ]
                        set f [win::CurrentTail]
                        if {$f == "Alpha Commands"} {
                            help::markAlphaCommands
                        } else {
			    set cmd 0
			    foreach pat $commandFiles {
				if {[string match $pat $f]} {
				    help::markTclCommands
				    set cmd 1
				}
			    }
			    if {!$cmd} {
				help::colourHeadingsEtc
			    }
			}
                        message ""
                        file::saveResourceChanges $filename
                    }
                }
            }
            default {
                file::openInDefault $filename
            }
        }
    } else {
        file::openAny $filename
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "help::openGeneral" --
 # 
 #  Called by embedded hyperlinks; look first for "package" help, and
 #  otherwise try to open the named help file (as if from Help menu.)
 #  ------------------------------------------------------------------------
 ##
proc help::openGeneral {name} {
    global HOME
    regsub -nocase { Help} $name {} package
    if {[catch {package::helpFile $package}]} {
	help::openFile $name
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "help::openExample" --
 # 
 # Called by embedded hyperlinks. 
 # 
 # Example files must be of the form "<something>-Example.sfx", with the
 # suffix optional.  If the hyperlink looks like "<something> Example.sfx",
 # it will open in a shell window, and inserts some explanatory text at the
 # beginning.  If there is a completions tutorial available for the mode,
 # that will be noted as well.
 # 
 # Exceptions (which are evaluated first):
 # 
 # -- Hyperlinks such as "<something>-Example.sfx" (the actual file name,
 # with the dash) open the actual example file, read-only.
 #  
 # -- If a package includes not only an example file, but wants to do
 # something special with it, then it should install two different files,
 # both a "<something>-Example.sfx" AND a "<something>-Example.sfx.tcl"
 # file, and make the hyperlink with "<something> Example.sfx" .
 # 
 # The presence of a "<something>-Example.sfx.tcl" file creates a special
 # case.  The hyperlink "Calculator Example", for example, evaluates the
 # file "Calculator-Example.tcl", which will then open a calculator window,
 # etc.  ("Tcl Example.tcl" will simply open as an example without being
 # sourced, because there is no "Tcl-Example.tcl.tcl" file.)
 # 
 # ------------------------------------------------------------------------
 ##

proc help::openExample {name} {
    global HOME
    
    regsub -all { } $name {-} name2
    set f  [file join $HOME "Examples" $name]
    set f2 [file join $HOME "Examples" $name2]
    
    if {$name == $name2 && [file exists $f]} {
        # Open as a read-only file
	edit -r -c $f
        return
    } elseif {[file exists ${f2}.tcl]} {
        # A special case -- evaluate the ${f2}.tcl file.
        uplevel \#0 [list source ${f2}.tcl]
        return
    } elseif {[file exists ${f}.tcl]} {
        # We'll also check to see if the file was erroneously named without 
        # the dash, as in "<something> example.tcl" .
        uplevel \#0 [list source ${f}.tcl]
        return
    } elseif {![file exists $f2]} {
        # Special cases done, but the file defined by "f" doesn't exist.
        beep
        message "Sorry, \"$name2\" is not in the Examples folder"
        return
    }
    # File exists, and it's not a special case, so open it in a shell window.
    set m [win::FindMode $f2]
    # Does Alpha know what mode this is?  If not, send an alertnote.
    if {$m == "Text"} {
        alertnote "Alpha doesn't recognize the mode for this example,\
          and will open it in as plain text."    
    }
    new -n "* $m Mode Example *" -m $m -text [file::readAll $f2] -shell 1
    goto [minPos]
    set    t "\r  $m mode example  --  Modify as much as you like ! \r\r"
    append t "  None of the changes you make will affect the actual file.  If you close \r"
    append t "  the window and then click on the hyperlink again, you will start with the \r"
    append t "  same example as before.  This also means that you cannot send this window \r"
    append t "  to other applications -- technically, it doesn't exist as a file. \r\r"
    append t "  Type \"control-Help\" to open any available help for $m mode. \r\r"
    # Find out if there's a tutorial available for this mode.
    set f3 [file join ${HOME} Tcl Completions "[modeALike] Tutorial"]
    if {[llength [glob -nocomplain -path $f3 *]] == 1} {
        append t "  $m mode also has a Completions Tutorial in the Config --> Mode Prefs menu.\r\r"
    }
    insertText  $t
    goto [minPos]
    if {$m == "Text"} {
        help::hyperiseEmailAddresses
        help::hyperiseUrls
    }
    # Now try to mark the file.
    markFile
}

## 
 # -------------------------------------------------------------------------
 # 
 # "help::openHyper" --
 # 
 #  Called by embedded hyperlinks; we look through an installation
 #  directory (and subdirs) if it is known, then the prefs directory, then
 #  all of the auto_path.  If it is a tutorial shell, find the proper mode
 #  and open it in a shell window.  Otherwise, if the file is of type TEXT
 #  we open it as read-only, else we ask the finder to open it. 
 #  -------------------------------------------------------------------------
 ##

proc help::openHyper {name} {
    global PREFS tclExtensionsFolder auto_path file::separator
    set currD [list [file dirname [win::Current]]]
    set dirs [glob -types d -dir $currD -nocomplain -- *]
    foreach d $dirs {
        lappend currD [string trimright $d ${file::separator}]
    }
    lappend currD $PREFS 
    if {[info exists tclExtensionsFolder]} {lappend currD $tclExtensionsFolder}
    foreach d [concat $currD $auto_path] {
        if {[regsub -nocase {Prefs.tcl} $name {} m]} {
            if {$m != ""} {
                # Edit a <mode>Prefs.tcl file, prompting to create if necessary.
                mode::editPrefsFile $m
            } else {
                # Edit a prefs.tcl file, creating one if necessary.
                prefs::tclEdit
            }
            return
        }
        set f [file join $d $name]
        if {[file exists $f]} {
            if {[regsub { +Tutorial.*$} $name {} m]} {
                # This is a tutorial, so open it in a shell window
                if {[mode::exists $m]} {
		    mode::completionsTutorial $m
		} else {
		    mode::completionsTutorial [win::FindMode $f]
		}
            } elseif {[getFileType $f] == {TEXT}} {
                # Type is Text, so open as read-only
                edit -r -c $f
            } else {
                # Unknown type, so prompt user
                file::openAny $f 
            }
            return
        }
    }
    beep
    message "Sorry, couldn't find $name"
}

#  File Marking / Hyperizing  #

## 
 # ----------------------------------------------------------------------
 #       
 #  "help::hyperiseUrls" --
 #  "help::hyperiseEmailAddresses" --
 #      
 #  This attaches hypertext links to all '<http:...  >' or '<mailto...>'
 #  strings in a document.  This procedure works best on files in Text
 #  mode; in other modes the colouring schemes can make the links invisible
 #  (although they still function).
 #      
 # ----------------------------------------------------------------------
 ##

proc help::hyperiseUrls {} {
    win::searchAndHyperise {<((http|news|mailto|ftp):[^ >]*)>} {url::execute "\1"} 1
    refresh
}

proc help::hyperiseEmailAddresses {} {
    win::searchAndHyperise \
      {<([-_a-zA-Z0-9.]+@([-_a-zA-Z0-9.]+))>} \
      {composeEmail "mailto:\1"} 1
    refresh
}

proc help::colourHeadingsEtc {{markFile 1}} {
    
    if {$markFile} {
        catch {help::markAlphaManual}
        catch {help::colorManualMarks}
    } 
    # Search for "<something>.tcl" and attach appropriate lookup.
    # Search for "<something >Tutorial<.sfx>" and attach appropriate lookup.
    win::searchAndHyperise \
      {"([-a-zA-Z_+0-9 ]*\.tcl|[-a-zA-Z0-9_+. ]+Tutorial[-a-zA-Z0-9_+. ]*)"} \
      {help::openHyper "\1"} 1 3 +1 -1
    # Search for "package: <something>" and attach appropriate lookup.
    win::searchAndHyperise {package: ([-a-zA-Z0-9+]*[-a-zA-Z0-9+])} \
      {help::openGeneral "\1"} 1 4 +9
    # search for "<something>Example" and attach appropriate lookup.
    win::searchAndHyperise {"([-a-zA-Z_+0-9 ]*Example(\.[a-zA-Z0-9_]+)?)"} \
      {help::openExample "\1"}  1 3 +1 -1
    # Search for "<<something>>" and embed as hypertext.
    # (Can only handle ] at end of string, not within ...)
    win::searchAndHyperise {<<([^>\r\n]+)>>} {\1} 1 4 +2 -2
    # Search for "proc: <something>" and attach appropriate lookup.
    win::searchAndHyperise {proc: ([-a-zA-Z:\+\.\_]+\w+)} \
      {Tcl::DblClickHelper "\1" ; setWinInfo read-only 1} 1 4 +6 
    # Search for "command: <something>" and attach appropriate lookup.
    win::searchAndHyperise {command: ([-\w+:\+\.\_]+\w+)} \
      {Tcl::DblClickHelper "\1" ; setWinInfo read-only 1} 1 4 +9 

    # Help file hyperlinks -- 
    # Don't make these lists too long, or there will be memory corruption.
    # search for "<something>Help" etc and attach appropriate lookup.
    win::searchAndHyperise {\"([-a-zA-Z_0-9+ ]+(Help|Commands))\"} \
      {help::openGeneral "\1"}  1 3 +1 -1
    
    set helpFileList [list "Alpha Manual" "Readme" "Extending Alpha" \
      "Quick Start" "Bug Reports and Debugging" "Changes" \
      "Changes - Alpha(|Tcl|tk)" "Alpha Developers FAQ" \
      "Default Key Bindings" "Internet Config" "Packages" \
      "Regular Expressions" "AEGizmos" \
      "Registering" "Shells" "Symantec" "Tcl Resources"]

    # search for specific Help files and attach appropriate lookup.
    win::searchAndHyperise "\"\([join $helpFileList |]\)\"" \
      {help::openGeneral "\1"}  1 3 +1 -1
    
    # Hyperlink section marks for the current window, anything in double
    # quotes that starts with "# " (similar to html in-file-target.)
    # Note that you need to include extra leading spaces within the quotes.
    win::searchAndHyperise {"\# ([][a-zA-Z0-9':;,._&()<>/+*?!#\t -]+)"} \
      {editMark [win::Current] "\1"} 1 3 +3 -1
    # Highlight IMPORTANT bits
    win::searchAndHyperise {IMPORTANT:} {} 0 5
    # Highlight "<something>-><something>" menu directions
    win::searchAndHyperise {"[^\r\n"]+->[^\r\n"]+"|'[^\r\n']+->[^\r\n']+'} \
      {} 1 5 +1 -1
    # make code inserts blue
    set bluestr {^[ \t]*[\r\n]\t[^" \t\r\n][^\n\r]*[\r\n](\t([ \t]*[\r\n]|[ \t]*[^ \t\r\n]+[^\n\r]*[\r\n]))*[ \t]*[\r\n]}
    win::searchAndHyperise $bluestr {} 1 1
    win::searchAndHyperise {CLICK[ A-Z]* INSTALL} "install::installThisPackage" 1 3
    refresh
    goto [minPos]
}

## 
 # ----------------------------------------------------------------------
 #       
 # "help::removeAllColoursAndHypers" --
 # "help::removeCHMCleanly" --
 #      
 # Remove all colors and hypers from the current window.  Removing them
 # "cleanly" also removes marks, but will not change the last save date
 # contained in the resource fork.  "help::removeCHMCleanly" is also used
 # by "help::mimicHelpMenu". 
 # ----------------------------------------------------------------------
 ##

proc help::removeAllColoursAndHypers {} {
    # get rid of the old stuff
    catch { removeColorEscapes }
    refresh
}

proc help::removeCHMCleanly {{filePath ""}} {

    global HOME win::Active

    # This proc can be used for open windows, or called by other code.
    if {$filePath != ""} {
        file::openQuietly $filePath
    } elseif {[llength [set win::Active]] < 1} {
        findFile [file join $HOME Help] 
    } 
    # Altered windows can not be saved "cleanly"
    if {![catch {getWinInfo arr}] && $arr(dirty)} {
        beep ; message "File must first be saved."
        error "File must first be saved."
    }
    setWinInfo read-only 0
    removeAllMarks
    help::removeAllColoursAndHypers
    setWinInfo dirty 0
}

## 
 # ----------------------------------------------------------------------
 #       
 # "help::markTclCommands" --
 #      
 # An alternative marking scheme for help files which mainly contain lists
 # of commands.  See "Tcl Commands" or "Error Help" for examples. 
 # ----------------------------------------------------------------------
 ##

proc help::markTclCommands {} {
    # Mark the Command file
    removeAllMarks
    status::msg "Marking commands in this file..."
    set pos [minPos]
    while {![catch {search -s -f 1 -r 1 -i 0 "^\[\t \]*NAME\[\t \]*\$" $pos} inds]} {
        set got [search -s -n -f 1 -r 1 "\[^ \t\r\n\]+" [lindex $inds 1]]
	if {[llength $got]} {
	    set pos [lindex $got 0]
	    eval text::color $got 1
	    setNamedMark [eval getText $got] $pos $pos $pos
	    set pos [nextLineStart $pos]
	} else {
	    set pos [nextLineStart [lindex $inds 1]]
	}
    }
    # Search and color red lines with all CAP words.
    set pos [minPos]
    while {![catch {search -s -f 1 -r 1 -i 0 {^[A-Z][-A-Z,\t ]+$} $pos} inds]} {
        eval [list text::color] $inds [list 5]
        set pos [nextLineStart [lindex $inds 0]]
    }
    # Color, underline the title (first alphanumeric line in file), and
    # then color / hyper any extras using help::colourHeadingsEtc.
    goto [minPos]
    hiliteWord
    endLineSelect
    text::color [getPos] [selEnd] 5
    text::color [getPos] [selEnd] 15
    help::colourHeadingsEtc 0
}

proc help::markAlphaCommands {} {
    global HOME alpha::platform

    if {[set alpha::platform] == "alpha"} {setWinInfo read-only 0}
    help::removeAllColoursAndHypers
    removeAllMarks "" 1
    status::msg "Marking commands in this file..."
    set pos [minPos]
    while {![catch {search -s -f 1 -r 1 {^ } $pos} inds]} {
	set pos1 [lindex $inds 1]
	set got [search -s -n -f 1 -r 1 "\[^ \t\r\n\]+" $pos1]
	if {[llength $got]} {
	    set pos [lindex $got 0]
	    eval text::color $got 1
	    eval setNamedMark [eval getText $got] [list $pos1] $got
	}
	set pos [nextLineStart $pos1]
    }
    select [minPos] [nextLineStart [nextLineStart [nextLineStart [minPos]]]]
    redWord
    goto [minPos]
    if {[set alpha::platform] == "alpha"} {save}
    # Sort the marks alphabetically.
    sortMarksFile
}

proc help::markAlphaChanges {} {
    set pos [minPos]
    while {[string length [set inds [search -s -f 1 -r 1 {^= } $pos]]]} {
	set pos1 [lindex $inds 1]
	goto $pos1
	endLineSelect
	redWord
	set pos [nextLineStart $pos1]
    }
}

#  Pete's manual-marking routines  #

proc help::markAlphaManual {} {
    set pos [minPos]
    set pat {^((\t  \t)|([\t ]*=+[\t ]*$))}
    while {![catch {search -f 1 -r 1 -s $pat $pos} match]} {
	set pos0  [lindex $match 0]
	set pos1  [lindex $match 1]
	set pos   [nextLineStart $pos1]
	if {[regexp {^[\t ]*=+$} [string trim [getText $pos0 $pos]]]} {
	    set label "-"
	} elseif {![string length [string trim [getText $pos1 $pos]]]} {
	    continue
	} else {
	    regsub -all "\t" [string trimright [getText $pos1 $pos]] " " label
	}
	while {[lcontains labels $label]} {append label " "}
	lappend labels $label
	set pos2 [lineStart [pos::math [lineStart $pos1] - 1]]
	setNamedMark $label $pos2 $pos0 $pos0
    }
}

proc help::colorManualMarks {} {
    goto [minPos]
    hiliteWord
    endLineSelect
    set from [getPos]
    set to [selEnd]
    text::color $from $to 5
    text::color $from $to 15
    
    foreach mk [getNamedMarks] {
	set name [lindex $mk 0]
	set disp [lindex $mk 2]
	set pos [lindex $mk 3]
	set end [lindex $mk 4]
	
	goto $disp
	hiliteWord
	endLineSelect
	set from [getPos]
	set to [selEnd]
	text::color $from $to 5
	text::color $from $to 15
    }
}

