## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "interactiveSearch.tcl"
 #                                    created: 11/08/2001 {20:01:53 PM}  
 #                                last update: 11/13/2001 {14:08:43 PM} 
 #  
 # Interactive search code.
 #  
 # ###################################################################
 ##

proc quickFind {} {search::interactive exact}
proc reverseQuickFind {} {search::interactive exact 0}
proc quickFindRegexp {} {search::interactive regexp}

namespace eval search {}

## 
 # -------------------------------------------------------------------------
 # 
 # "search::interactive" -- general interactive searching
 # 
 # This version allows class shorthands (\d \s \w \D \S \W), 
 # word anchors (\b), and some aliases of the machine dependent 
 # control characters (\a \f \e \n \r \t). Therefore, 
 # we need two prompts, one for when we have a valid pattern, and one 
 # for when the pattern has gone invalid (most likely due to starting 
 # to enter one of the above patterns). 
 # 
 # The Return key and unknown key combinations exit the search, leaving 
 # the point at its current position. You can then use 'exchangePointAndMark'
 # (cntrl-x, cntrl-x -in emacs keyset) to jump back and forth between where
 # the search started from and where the search ended.
 # 
 # Known key combinations (e.g., arrow keys, many emacs navigation keys)
 # exit the search and perform the appropriate action. The mark is set to
 # the last successful search, so 'exchangePointAndMark' does NOT take you
 # to the start of the search.
 # 
 # The Escape key or abortEm (cntrl-g in emacs) "aborts" the search,
 # returning the cursor to the point where the search started from.
 # Use 'exchangePointAndMark' to jump to the last found match.
 # 
 # The next occurrence of the current pattern can be matched by typing 
 # either control-s (to get the next occurence forward), or control-r 
 # (to get the the next occurrence backward)
 #
 # Also, after aborting or exiting, the search string is left in the Find
 # dialog, and so you can use 'findAgain' or cntrl-s or cntrl-r to continue
 # the search. Be aware that the Find dialog starts out with a default of 
 # <Grep=OFF>.
 #  
 # Original Author: Mark Nagata
 # modifications  : Tom Fetherston
 # modifications  : Vince Darley, so works with or without regexp
 # -------------------------------------------------------------------------
 ##

proc search::interactive {{type "exact"} {direction 1}} {
    set ignoreCase 1
    set interpretBackslash 0
    set patt ""
    set pos [getPos]
    lappend history [list "" [list $pos $pos] 1]
    
    set done 0
    while {!$done} {
	if {$type == "regexp"} {
	    # check pattern validatity
	    if {[catch {regexp -- $patt {} dmy} dmy]} {		
		set prompt "building->: $patt"
	    } else {
		set prompt "regIsearch: $patt"
	    }
	} else {
	    set prompt "search: $patt"
	}
	set proc [list search::interactiveKeypress $type $direction]
	set done 1
	switch -- [catch [list status::prompt -appendvar patt -command $proc -add anything $prompt] res] {
	    0 {
		# got a keystroke that triggered a normal end (e.g. <return>)
		set res "<return>"
		set tmp [getPos]
		goto $pos
		setMark
		goto $tmp
	    }
	    1 {
		# an error was generated
		if {[string match "missing close-brace" $res]} {
		    # must have typed a slash, so:
		    append patt "\\"
		    set done 0
		} elseif {[string match "invoked \"break\" outside of a loop" $res]} {
		    # do nothing
		} elseif {[string match "abort*" $res]} {
		    if {[package::active emacs]} { append res ". ctrl-x ctrl-x goes to last found" }
		    goto $pos
		} elseif {[string match "unknown*" $res]} {
		    if {[package::active emacs]} { append res ". ctrl-x ctrl-x goes to search start" }
		    set tmp [getPos]
		    goto $pos
		    setMark
		    goto $tmp
		} else {
		    # unknown error -- exit
		}
	    }
	    default {
		set done 1
	    }
	}
    }
    message "Search $patt: exited with $res."
}

## 
 # -------------------------------------------------------------------------
 # 
 # "search::interactiveKeypress" -- handle isearch, rsearch, regIsearch
 # 
 #  This proc handles each keypress while running a regIsearch. It has been 
 #  modified from Mark Nagata's original to provide next ocurrence 
 #  before/after current, and support for key bindings whose navigation or 
 #  text manipulation functionality makes sense with respect to a regIsearch.
 #  
 #  closest occurence before current match	
 #    - command-option g & cntrl-r (mnemonic 'reverse')
 #  closest occurence after current match
 #    - command g & cntrl-s (mnemonic 'successor')
 #  
 #                         Text Naviagation
 #  forwardChar (aborts and leaves cursor after last match)
 #    - right arrow & cntrl-f (emacs)
 #  backwardChar (aborts and leaves cursor before last match)
 #    - left arrow & cntrl-b (emacs)
 #  beginningOfLine (aborts and moves cursors to the start of the line 
 #  	containing the last match)
 #    - cmd left arrow & cntrl-a (emacs)
 #  beginningOfLine (aborts and moves cursors to the start of the line 
 #  	containing the last match)
 #    - cmd right arrow & cntrl-e (emacs)
 #  centerRedraw (moves selection to center, without aborting)
 #   - cntrl-l
 #  insertToTop (moves selection to top, without aborting)
 #   - cntrl-t
 #  ctrl-w adds the rest of the current word to the search string.
 #  
 #                         Text Manipulation
 #  deleteSelection (aborts and deletes selection)
 #    - cntrl-d (emacs)
 #  killLine (aborts and deletes from start of selection to end of line)
 #    - cntrl-k (emacs)
 #    
 #    Changing the search type:
 #    
 #  ctrl-i switches the case-sensitivity of the current search
 #  ctrl-backslash toggles interpretation of \n,\r,\t in non-regexp searches
 # -------------------------------------------------------------------------
 ##
proc search::interactiveKeypress {type dir {key 0} {mod 0}} {
    set direction {}

    set t [keys::modToString $mod]

    if {[string length $key]} {
	scan $key %c decVal
    } else {
	# No key showed up.  Probably running on Alphatk
	error "no key press"
    }
    #tclLog "\r$key $t $mod $decVal"
    upvar patt pat
    switch -- $t {
	"____" {
	    switch -- $decVal {
		8  {
		    set len [string length $pat]
		    if {$len > 0} {
			set pat [string range $pat 0 [expr {$len-2}]]
			set key ""
			set backtrack 1
		    } else {
			error "deletion of all characters"
		    }
		}
		1 { beginningOfBuffer;  error "navigation key"; # home; }
		4 { endOfBuffer;  error "navigation key"; # end; }
		11 { pageBack;  error "navigation key"; # page up; }
		12 { pageForward;  error "navigation key"; # page down; }
		29 { forwardChar; error "navigation key"; # right arrow; }
		28 { backwardChar; error "navigation key"; # left arrow; }
		30 { previousLine; error "navigation key"; # up arrow; }
		31 { nextLine; error "navigation key"; # down arrow; }
		27 { error "abort (esc key)"; # escape; }
		13 { error "<return> key"; }
	    }
	}
    }
    switch -- $t {
	"____" - 
	"_s__" {
	    if {0 && $curr != ""} {
		while {[string compare [string range $pat [string last $curr $pat] end] $curr] != 0} {
		    set newEnd [expr {[string length $pat] - 2}]
		    if {$newEnd < 0} {
			error "deletion of all characters"
		    } 
		    set pat [string range $pat 0 $newEnd] 
		    set backtrack 1
		}
	    } 
	    
	    set preAppend $pat
	    append pat $key
	    if {$type == "regexp"} {
		if {[catch {regexp -- $pat {} dmy} res]} {
		    message "building->: $preAppend"
		    return $key
		}
	    }
	    set direction $dir
	    # This is a continuing search from the current point
	    set inplace 1
	}
	"c___" {
	    switch -- $decVal {
		101 { 
		    # cmd-e = enter search string
		    searchString $pat
		    return {}
		}
		103 { set direction 1; 	   # (cmd g); }
		28 { beginningOfLine; error "navigation key"; # cmd left arrow; }
		29 { endOfLine; error "navigation key"; # cmd right arrow; }
		default { error "unknown cmd key" }
	    }
	    
	}
	"__o_" {
	    if {[package::active emacs]} {
		switch -- $decVal {
		    2 - 186 { backwardWord; error "emacs delete word (opt-d)"; # opt-b; }
		    4 - 182 { deleteWord; error "emacs delete word (opt-d)"; # opt-d; }
		    6 - 196 { forwardWord; error "emacs forward word (opt-f)"; # opt-f; }
		}
	    } 
	}
	"___z" {
	    # If the user is using the emacs key bindings, check for ones that 
	    # make sense. All other control key combinations abort
	    if {[package::active emacs]} {
		switch -- $decVal {
		    1 { beginningOfLine; error "emacs beginning of line (cnt-a)"; # cntrl-a; }
		    2 { backwardChar; error "emacs backward char (cnt-b)"; # cntrl-b; }
		    4 { deleteSelection; error "emacs delete selection (cnt-d)"; # cntrl-d; }
		    5 { endOfLine; error "emacs end of line (cnt-e)"; # cntrl-e; }
		    6 { forwardChar; error "emacs forward char (cnt-f)"; # cntrl-f; }
		    11 - 107 { killLine; error "emacs kill line (cnt-k)"; # cntrl-k; }
		    12 - 108 { centerRedraw; return {};	# cntrl-l; }
		    14 { backwardChar; nextLine; error "emacs next line (cnt-n)"; }
		    15 { openLine; error "emacs open line (cnt-o)"; # cntrl-o; }
		    16 { backwardChar; previousLine; error "emacs previous line (cnt-p)"; }
		}
	    } 
	    # See if user has requested to find another match, either searchForward 
	    # (cntrl-s) or reverseSearch (cntrl-r). Set flag accordingly
	    switch -- $decVal {
		18 - 114 - 19 - 115 { 
		    # (ctrl-r, ctrl-s)
		    if {![string length $pat]} { 
			# load previous search string if current is empty
			set pat [searchString]
		    }
		    switch -- $decVal {
			18 - 114 { set direction 0; # reverse; }
			19 - 115 { set direction 1; # forward; }
			default {}
		    }
		}
		20 - 116 {
		    insertToTop; #cntl-t; 
		}
		28 {
		    # ctrl-backslash : toggle \n\r\t interpretation
		    upvar interpretBackslash ib
		    set ib [expr {1-$ib}]
		    set direction $dir
		    set inplace 1
		}
		8 - 103 {
		    # cntrl-g
		    error "abort (ctrl-g)"
		}
		9 - 105 {
		    # ctrl-i : change case-sensitivity
		    upvar ignoreCase ign
		    set ign [expr {1-$ign}]
		    set direction $dir
		    set inplace 1
		}
		23 - 119 {
		    # ctrl-w : add next word
		    set _p [getPos]
		    set _q [pos::math $_p + [string length [getSelect]]]
		    goto $_q
		    forwardWord
		    append pat [getText $_q [getPos]]
		    goto $_p
		    set direction $dir
		    set inplace 1
		}
		default { error "unknown cntrl key" }
	    }
	}
	"c_o_" {
	    switch -- $decVal {
		169 { 
		    # (cmd-opt 'g')
		    set direction 0 
		}
		default { error "unknown cmd-option key" }
	    }
	    
	}
	"default" {
	    error "unknown modifier key"
	}
    }
    # handle direction flag if it got set above
    if {$direction != ""} {
	if {$type == "regexp"} {
	    message "regIsearch: $pat " 
	} else {
	    message "search: $pat " 
	}
	upvar ignoreCase ign
	if {![info exists inplace]} {
	    if {$direction} {
		set search_start [pos::math [getPos] + 1]
	    } else {
		set search_start [pos::math [getPos] - 1]
	    }
	} else {
	    set search_start [getPos]
	}
	upvar history hist
	if {[info exists backtrack]} {
	    while {[llength $hist] > 1} {
		set hist [lrange $hist 0 [expr {[llength $hist]} -2]]
		if {[llength $hist]} {
		    set last [lindex $hist end]
		    if {[llength $last] == 1} {
			# search failed
			set failed 1
			continue
		    }
		    # Only if we haven't failed do we check the in-place
		    # flag (list index 2).
		    if {![info exists failed]} {
			if {![lindex $last 2]} {
			    continue
			}
		    }
		    break
		} else {
		    # error "Probably shouldn't get here"
		    # Avoid infinite loop in some odd cases.
		    break
		}
	    }
	    set last [lindex $hist end]
	    set pat [lindex $last 0]
	    eval select [lindex $last 1]
	} else {
	    if {$type == "regexp"} {
		set searchResult [search -s -n -f $direction -m 0 \
		  -i $ign -r 1 -- $pat $search_start]
	    } else {
		upvar interpretBackslash ib
		if {$ib} {
		    set spat $pat
		    regsub -all "\\\\n" $spat "\n" spat
		    regsub -all "\\\\r" $spat "\r" spat
		    regsub -all "\\\\t" $spat "\t" spat
		    set searchResult [search -s -n -f $direction -m 0 \
		      -i $ign -r 0 -- $spat $search_start]
		} else {
		    set searchResult [search -s -n -f $direction -m 0 \
		      -i $ign -r 0 -- $pat $search_start]
		}
	    }
	    searchString $pat
	    if {[llength $searchResult] == 0} {
		lappend hist [list "failed"]
		beep
	    } else {
		lappend hist [list $pat $searchResult [info exists inplace]]
		eval select $searchResult
	    }
	}
	return {}
    }
}
