## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl 
 # 
 #  FILE: "isIn.tcl"
 #                                    created: 2001-05-05
 #                                last update: 2001-11-09 16:09:09 
 # File : "isIn.tcl"
 # Author : Bernard Desgraupes and Joachim Kock
 # e-mail : <berdesg@easynet.fr>  <kock@math.unice.fr>
 # www : <http://perso.easynet.fr/~berdesg/alpha.html>
 #
 #  Description: 
 # 
 # Procedures for handling comments - in particular to determine
 # whether a given position is in a comment or not.  The answer to this
 # question takes into account that logical line starts may not
 # coincide with physical line starts, that the meaning of a comment
 # char may be suspended, either by an escape char (like in Tcl: \#) or
 # when being part of a quoted string (e.g. "#").
 #
 # The mode specific aspects of these questions are controlled by one
 # preference flag
 #
 #   commentsContinuation
 #
 # and three series of mode specific global variables
 #
 #   set (mode)::escapeChar "\\"
 #   set (mode)::quotedstringChar "\""
 #   set (mode)::lineContinuationChar "\\"
 #
 # These variables must be defined by each mode individually.  If the
 # variable is not defined, the corresponding check is simply not
 # performed, so modes where the concept of escapeChar,
 # quotedstringChar, or lineContinuationChar has no meaning can (and
 # should) just leave that variable undefined. 
 #
 # The flag commentsContinuation controls whether every comment char
 # is detected (value 2), or if it is only detected at line start modulo
 # whitespace (value 1), or only at proper line start (value 0).  For
 # most modes, the global setting =2 is adequate according to syntax,
 # but the user may prefer to override this, according to typing habits.
 # For Text and Mail modes the flag should be set =1.  Fortran syntax
 # requires the flag to be =0.
 # Mode writers should declare this flag like in this example :
 #   newPref v commentsContinuation 0 TeX "" \
 #    [list "only at line start" "spaces allowed" "anywhere"] index
 #
 # ###################################################################
 ##

# The variable {mode}::escapeChar is used only in the proc
# text::isEscaped, which in turn is called by text::isInQuotedString,
# text::trueStartOfLine, text::InSingleComment, and
# text::InDoubleComment.  This variable holds the character that
# suspends the meaning of special characters...  In many modes it is \
# (single backslash), eg.  C, C++, TeX, Tcl...  In modes where there is
# no such thing, the variable should be left undefined.
proc text::isEscaped {{position ""}} {
  global mode
  global ${mode}::escapeChar
  if {![info exists ${mode}::escapeChar]} {
    return 0
  }
  set escapeChar [set ${mode}::escapeChar]
  if {$escapeChar == ""} {
    return 0
  }
  if {$position==""} {
    set position [getPos]
  }
  set prevposition [pos::math $position - 1]
  if {[lookAt $prevposition] == $escapeChar} {
    # now we've detected an escape char -
    # (provided this char has not previously been escaped!):
    if {![text::isEscaped $prevposition]} {
      return 1
    }
  }
  return 0
}


# The variable {mode}::quotedstringChar is used only in
# text::isInQuotedString, which in turn is called from
# text::InSingleComment and text::InDoubleComment.  It holds the
# character, in the middle of a pair of which certain special
# characters are given their literal meaning...  In many programming
# modes this is " (double quote).  In modes where there is no such
# thing, the variable should be left undefined.
proc text::isInQuotedString {{position ""}} {
  global mode
  global ${mode}::quotedstringChar
  if {![info exists ${mode}::quotedstringChar]} {
    return 0
  }
  set quotedstringChar [set ${mode}::quotedstringChar]
  if {$quotedstringChar == ""} {
    return 0
  }
  if {$position==""} {
    set position [getPos]
  }
  set start [text::trueStartOfLine $position]
  set qcount 0
  while  {![catch {search -f 1 -r 1 -s -l $position \
   -- "[quote::Regfind $quotedstringChar]" $start} res]} {
    #if not escaped
    if {![text::isEscaped [lindex $res 0]]} { 
      incr qcount;         #then count it
    }
    set start [lindex $res 1]
  }
  #if odd number of true quotes then we're in:
  return [expr {$qcount % 2}]
}

# The variable (mode)::lineContinuationChar is used only by
# text::trueStartOfLine, which in turn is called from
# text::isInQuotedString and text::isInSingleComment.  In modes where no
# such thing exists, leave the variable undefined.
# 
# Proc to find the true beginning of a line i-e the beginning of the
# logical line and not only of the physical line : this means that it
# has to take into account continued line endings with a _single_
# backslash.
proc text::trueStartOfLine {{position ""}} {
  if {$position==""} {
    set position [getPos]
  }
  set begpos [lineStart $position]
  global mode
  global ${mode}::lineContinuationChar
  if {![info exists ${mode}::lineContinuationChar]} {
    return $begpos
  }
  set lineContinuationChar [set ${mode}::lineContinuationChar]
  if {$lineContinuationChar == ""} {
    return $begpos
  }
  while {[lookAt [pos::math $begpos - 2]]=="$lineContinuationChar"} {
    if {![text::isEscaped [pos::math $begpos - 2]]} {
      set begpos [prevLineStart $begpos]
    } else {break}
  }
  return $begpos
}


# Proc to determine whether we are in a single line comment.  Answer: 0
# or 1.  In affirmative case, the variable commentPos will acquire the
# position of the active comment tag (list of two integers, e.g. 13 14)
proc text::isInSingleComment { pos {commentPos ""} } {
    global commentsContinuation
    if {$commentPos != ""} {
	upvar $commentPos foundPos
    }
    set startpos [text::trueStartOfLine $pos]
    if {![catch {comment::Characters "General"} commentCharList]} {
	foreach commentCh $commentCharList {
	    set commentCh [string trim $commentCh]
	    while  {![catch {search -f 1 -r 1 -s -l $pos \
	      -- "[quote::Regfind $commentCh]" $startpos} foundPos]} {
		if {[text::isEscaped [lindex $foundPos 0]] || \
		  [text::isInQuotedString [lindex $foundPos 0]] || \
		  [text::isInDoubleComment [lindex $foundPos 0]]} {
		    #escaped, quoted, or in double comment, so keep searching:
		    set startpos [lindex $foundPos 1]
		    continue  ;#looking for the same commentCh
		} elseif {$commentsContinuation == 1} { 
		    # 1 = "spaces allowed"
		    #(actually we should perform line-continuation substitution)
		    set txt [getText $startpos [lindex $foundPos 0]]
		    if {![is::Whitespace $txt]} {
			#found preceding nonwhite, so there can't possibly
			#be any comments (of this type) on this line:
			break  ;#and proceed to the next commentCh in the list
		    }
		} elseif {$commentsContinuation == 0} { 
		    # 0 = "only at line start"
		    #(comments can ONLY start in column 1)
		    if {![ pos::compare $startpos == [lindex $foundPos 0] ]} {
			#we are not at line start, so this comment is not valid:
			break  ;#and proceed to the next commentCh in the list
		    }
		}
		return 1  ;#this is a true comment
	    }
	}
    }
    return 0  ;#found none
}


# Proc to determine whether we are in a double comment, i.e. a comment
# defined by an opening and a closing tag.  In fact the proc does not
# bother whether a closing tag exists after the cursor position,
# because anyway the user might not have typed it yet...
# 
# Result: 0 or 1.  In affirmative case, the variable openingPos will
# acquire the position of the active opening comment tag (list of two
# integers, e.g. 13 15)
proc text::isInDoubleComment {pos {openingPos ""}} {
  if {$openingPos != ""} {
    upvar $openingPos foundPos
  }
  if {![catch {comment::Characters "Paragraph"} cpar]} {
    set lftComment [string trim [lindex $cpar 0]]
    set rtComment [string trim [lindex $cpar 1]]
    if { $lftComment != $rtComment } {
      #otherwise we don't care...
      set lftpos [pos::math $pos - 1]
      #adjust this if you use very long comments
      set searchSize 2000
      set searchlim [pos::math $lftpos - $searchSize]
      while {![catch {search -f 0 -r 1 -s -l $searchlim \
       -- "[quote::Regfind $lftComment]" $lftpos} foundPos]} {
        if {[text::isEscaped [lindex $foundPos 0]] \
         || [text::isInQuotedString [lindex $foundPos 0]]} {
          #escaped or quoted; keep searching:
          set lftpos [pos::math [lindex $foundPos 0] - 1]
          continue  ;#to look backwards for an opening
        } else {
          #found a true lftComment.  Now looking for a rtComment
          set rtpos [lindex $foundPos 1]
          if {![catch {search -f 1 -r 1 -s -l $pos \
           -- "[quote::Regfind $rtComment]" $rtpos} res]} {
            #found a closing rtComment, so there is no reason to
            #search further back for an opening:
            break  ;# = return 0
          }
        }
        return 1  ;#found lftComment without subsequent rtComment
      }
    }
  }
  return 0  ;#found none
}


# Proc to determine whether the given position is in a comment.
# Answer: 0 or 1. In affirmative case, the variable st will acquire the
# suggested way of prefixing the following line, as used by the proc
# bind::CarriageReturn
proc text::isInComment {pos {st ""}} {
  if {[pos::compare $pos == [minPos]]} {
    return 0
  }
  if {$st != ""} {
    upvar $st start
  }
  # First case : double comments
  if {[text::isInDoubleComment $pos]} {
    set txt [getText [lineStart $pos] $pos]
    set cpar [comment::Characters "Paragraph"]
    set openCom [string trim [lindex $cpar 0]]
    set midCom [string trim [lindex $cpar 2]]
    # determine the indent/comment prefix of next line:
    if {[regexp -- "^(.*)[quote::Regfind $openCom](\[ \t\]*)" \
     $txt "" start c]} {
      # the line has an opening comment, so next line gets a
      # middle comment:
      regsub -all "\[^ \t\]" $start " " start
      append start [string trimright [lindex $cpar 2]] $c
    } else {
      # no opening comment in this line, so just copy the leading
      # whitespace, possibly with an occurrence of the
      # middle-continuation-comment char:
      regexp -- "^\[ \t\]*[quote::Regfind $midCom]?\[ \t\]*" \
       $txt start
    }
    return 1
  }
  # Second case : single comments
  if {[text::isInSingleComment $pos commentPos]} {
    #set preString, commentString and postString:
    set previousText [getText [lineStart [lindex $commentPos 0]] \
     [lindex $commentPos 0]]
    regsub -all "\[^ \t\]" $previousText " " preString
    set commentString [getText [lindex $commentPos 0] \
     [lindex $commentPos 1]]
    set followingText [getText [lindex $commentPos 1] $pos]
    if {![regexp -- "([quote::Regfind $commentString]|\[ \t\])*" \
     $followingText postString]} {
      set postString ""
    }
    set start $preString$commentString$postString
    return 1
  }
  return 0
}

################################################################################


proc isAtLineEnd {pos} {
  set char [lookAt $pos]
  if {$char == "\r" || $char == "\n"} {return 1} else {return 0}
}

## 
 # -------------------------------------------------------------------------
 # 
 # "text::OLDisInComment" --
 # 
 # Are we in a block comment? Just checks if both the given line and the
 # next line commence with any of a set of known block-comment characters.
 # Not 100% satisfactory for C comments, but fine for all others.
 # 
 # If 'st' is given, we use it as a variable name in which to store
 # the prefix which will be needed in indenting the following line.
 # -------------------------------------------------------------------------
 ##
proc text::OLDisInComment {pos {st ""}} {
  if {[pos::compare $pos == [minPos]]} {
    return 0
  }
  set p [lineStart $pos]
  if {[pos::compare $pos == $p]} {
    set pos [pos::math $pos - 1] ; set p [lineStart $pos]
  }
  set q [nextLineStart $pos]
  set t [getText $p $q]
  if { $st != "" } {
    upvar $st a
  }
  if {![catch {comment::Characters "Paragraph"} cpar]} {
    if {[regexp -- "^(.*)[quote::Regfind [string trim [lindex $cpar 0]]](.*)" $t "" a rest]} {
      # We found an opening comment
      if {![regexp -indices -- "[quote::Regfind [string trim [lindex $cpar 1]]]" $rest where] \
       || (([string length $rest] - [pos::diff $pos $q] -1) < [lindex $where 0])} {
        # Either we didn't find a closing comment, or we did,
        # and the closing comment was after the position $pos,
        # in which case $pos was still inside the comment.
        # Either way we need to continue the comment.
        regsub -all "\[^ \t\]" $a " " a
        append a [lindex $cpar 2]
        return 1
      }
    }
  }
  # if the next line is a comment 
  set qq [text::firstNonWsLinePos $q]
  if {[pos::compare $qq == [maxPos]]} { 
    return 0 
  }
  foreach commentCh [comment::Characters "General"] {	
    if {[regexp -- "^\[ \t\]*[quote::Regfind ${commentCh}]\[ \t\]*" $t a]} {
      # if we hit return in the middle of a line
      if {[string trim [getText $pos $q]] != "" && [pos::compare $pos != $p]} { 
        return 1
      }
      if {[getText $qq [pos::math $qq + [string length $commentCh]]] == $commentCh} {
        return 1
      }
    }
  }
  return 0
}

proc text::OLDisInString {pos} {
  set t [getText [lineStart $pos] $pos]
  # Remove all backslash quote sequences.
  # We should really check for an odd number of backslashes here.
  regsub -all "\\\\\"" $t "" t
  # Now count the number of quotes.  If it is odd, we are most
  # likely in a string.
  set count [regsub -all "\"" $t X t]
  return [expr {$count % 2}]
}

proc OLDliteralChar {} {
  return [expr {[lookAt [pos::math [getPos] - 1]] == "\\"}]
}

######################################################################
# NOTE THAT the following two procs
#   text::isInString
#   literalChar
# are primitive equivalents of
#   text::isInQuotedString
#   text::isEscaped
# and they could probably be replaced or overwritten:
#
# proc text::isInString {pos} {
#   text::isInQuotedString $pos
# }
#
# proc literalChar {} {
#   text::isEscaped
# }
#
# CHECK OUT THIS CAREFULLY BEFORE CHANGING ANYTHING!
#
######################################################################

proc text::isInString {pos} { text::isInQuotedString $pos }
proc literalChar {} { text::isEscaped }
