# For use by Alpha 7 only.  Newer versions of Tcl have this command
# built in.

if {[info tclversion] >= 8.0} {
    return
}

# NOTE: In some ways this may be more stringent or more lenient than
# Tcl 8.4's version of clock.  In all respects 8.4 should be considered
# the reference implementation which this should be fixed to follow!
proc clock {cmd args} {
    switch -- $cmd {
	"clicks" {
	    # clock clicks ?-milliseconds?
	    if {[llength $args] > 0} {
		return -code error "milliseconds granularity not\
		  possible in Alpha 7"
	    }
	    return [ticks]
	}
	"format" {
	    # clock format clockValue ?-format string? ?-gmt boolean?
	    if {[llength $args] == 0} { error "Not enough args" }
	    if {[llength $args] > 1} {
		return -code error "Other arguments to clock\
		  format not supported"
	    }
	    set val [lindex $args 0]
	    #aclock_debug $val
	    return [mtime $val]
	}
	"scan" {
	    # clock scan dateString ?-base clockVal? ?-gmt boolean?
	    if {[llength $args] == 0} { error "Not enough args" }
	    if {[llength $args] > 1} {
		return -code error "Other arguments to clock\
		  scan not supported"
	    }
	    set datestr [lindex $args 0]
            if {![clock_scan_ISO_date $datestr] &&\
              ![clock_scan_long_date $datestr] &&\
              ![clock_scan_short_date $datestr]} then {
                return -code error "Failed to parse '$datestr' as a date.\
                  Try using ISO format, like 1969-07-20T22:56:00"
            }
            if {$theDate(year)<100} then {
                incr theDate(year) 2000
                if {$theDate(year)>2039} then {incr theDate(year) -100}
            }
            # Take care of the timezone difference
            if {[info exists theDate(zoneOfs)]} then {
                set theDate(zoneOfs) [expr {[ISOTime::ZoneOffset] - $theDate(zoneOfs)}]
            } else {
                set theDate(zoneOfs) 0
            }

	    # STEP 1: Find number of seconds since beginning of day
	    set secondsSinceBeginningOfDay\
              [expr {$theDate(seconds) + 60 * ($theDate(minutes) + 60 *\
                $theDate(hour)) + $theDate(zoneOfs)}]

	    # STEP 2: Find number of days since beginning of computer-time.
	    # Computer time starts at 00:00:00 1 January 1904 on MacOS.
	    set years [expr {$theDate(year) - 1904}]
	    # Number of leapyears up to Jan 1st of the given year
	    set leapyears [expr {($years + 3)/4}]
	    set nonleapyears [expr {$years - $leapyears}]

	    set daysSinceBot [expr {$leapyears * 366 + $nonleapyears * 365}]
	    incr daysSinceBot [lindex\
              {0 0 31 59 90 120 151 181 212 243 273 304 334 365} $theDate(month)]
	    # If leap year and after Feb, add a day
	    if {($theDate(year) % 4) == 0 && $theDate(month) > 2} then {
                incr daysSinceBot
	    }
	    incr daysSinceBot [expr {$theDate(day) -1}]

            # STEP 3: Compute number of seconds since beginning of computer-time
            # Need to work around arithmetic overflow
            # 0x80000000 = 24855 * 24 * 60 * 60 + 11648
	    set totalSeconds [expr {(($daysSinceBot - 24855) * 86400 +\
              ($secondsSinceBeginningOfDay - 11648)) ^ 0x80000000}]

            # STEP 4: Use ISOTime::brokenDate to check if we need to
            # compensate for Daylight Saving Time or the like
            ISOTime::brokenDate $totalSeconds testDate
            set offset [expr {(($testDate(hour) - $theDate(hour)) * 60 +\
              $testDate(minutes) - $theDate(minutes)) * 60 +\
              $testDate(seconds) - $theDate(seconds) - $theDate(zoneOfs)}]
            while {$offset>43200} {incr offset -86400}
            while {$offset<-43200} {incr offset 86400}
	    return [expr {$totalSeconds - $offset}]
	}
	"seconds" {
	    if {[llength $args] > 0} {
		return -code error "Too many args"
	    }
	    return [now]
	}
	default {
	    return -code error "bad option \"cmd\" to clock"
	}
    }
}

# The following procs checks to see whether $datestr conforms
# to some given date format and if so parses it by setting entries
# in the array named $var (by default "theDate") in the caller's
# local context as follows:
#   year     contains the year (may be in the range 0-99, which should then be normalized)
#   month    contains the month (1-12)
#   day      contains the day of month
#   hour     contains the hour (0-23)
#   minutes  contains the minutes
#   seconds  contains the seconds
#   zoneOfs  contains the timezone offset to UCT, in seconds
#            (if this field is unset then assume local time)
# With the exception of the zoneOfs entry, the format is the same
# as that of the ISOTime::brokenDate procedure.
#
# These procedures return 1 when they succeed in parsing the date and
# 0 otherwise.
#
# Scan ISO8601 dates.
set clock_scan_ISO_regexp {([0-9][0-9][0-9]?[0-9]?)}
append clock_scan_ISO_regexp {-?([0-1][0-9])-?([0-3][0-9])}
append clock_scan_ISO_regexp {T([0-2][0-9]):?([0-5][0-9]):?([0-6][0-9])}
append clock_scan_ISO_regexp {(Z|[+-][0-1][0-9](:?[0-5][0-9])?)?}
proc clock_scan_ISO_date {datestr {var theDate}} {
    regsub -all "\[ \t\r\n\]" $datestr {} datestr
    global clock_scan_ISO_regexp
    if {![regexp -- $clock_scan_ISO_regexp $datestr\
      "" year month day hour minutes seconds zone]} then {return 0}
    upvar 1 $var theDate
    foreach field {year month day hour minutes seconds} {
        scan [set $field] %d theDate($field)
    }
    if {"Z"=="$zone"} then {
        set theDate(zoneOfs) 0
    } elseif {[regexp {([+-])([0-9][0-9]):?([0-9][0-9])?} $zone\
      "" sign hour minutes]} then {
        scan $hour %d hour
        if {![scan $minutes %d minutes]} then {set minutes 0}
        set theDate(zoneOfs) [expr $sign {($hour*60 + $minutes)*60}]
    }
    return 1
}

# Scan dates where the month is written with letters
# First an array for translating three-letter abbreviations to
# month number. Since there are spelling variations between
# languages, some months have multiple entries, and more can be
# added.
array set clock_scan_months {jan 1 feb 2 mar 3 apr 4 may 5 maj 5\
  jun 6 jul 7 jui 7 aug 8 sep 9 oct 10 okt 10 nov 11 dec 12}
proc clock_scan_long_date {datestr {var theDate}} {
    regsub -all "\[ \t\r\n\]+" [string tolower $datestr] { } datestr

    set timeRegexp {([0-9][0-9]?)[:.]([0-9][0-9]?)([:.]([0-9][0-9]?))?}
    if {[regexp $timeRegexp $datestr time]} {
        regsub $timeRegexp $datestr "" datestr
    } else {# Assume midnight
        set time 00:00:00
    }
    global clock_scan_months
    set dateRegexp {([0-3]?[0-9]) }
    append dateRegexp "([join [array names clock_scan_months] |]" {)[a-z]*}
    append dateRegexp { ([0-9]+)}
    if {![regexp $dateRegexp $datestr "" day month year]} then {
        set dateRegexp "([join [array names clock_scan_months] |]"
        append dateRegexp  {)[a-z]* ([0-3]?[0-9]),? ([0-9]+)}
        if {![regexp $dateRegexp $datestr "" month day year]} then {
            return 0
        }
    }
    upvar 1 $var theDate
    scan $year %d theDate(year)
    set theDate(month) $clock_scan_months($month)
    scan $day %d theDate(day)
    set theDate(seconds) 0
    scan [split $time :.] "%d %d %d" theDate(hour) theDate(minutes) theDate(seconds)
    return 1
}

# Finally a proc that tries to scan short date forms. It relies on
# the code for this in the ISOTime package.
proc clock_scan_short_date {datestr {var theDate}} {
    global ISOTime::regdate ISOTime::matchdate
    if {(![info exists ISOTime::regdate]) ||\
      (![info exists ISOTime::matchdate])} then {
        ISOTime::parseLocalizedTime
    }
    if {![regsub ${ISOTime::regdate} $datestr ${ISOTime::matchdate} dateCmd] ||\
      [catch $dateCmd]} then {
        return 0
    }
    upvar 1 $var theDate
    foreach field {year month day hour minutes seconds} {
        scan $date($field) %d theDate($field)
    }
    return 1
}

return

# Code orphans:
#
#     if {[info exists ampm]} {
#         # Add 12 hours if its a pm time
#         if {[string tolower [string index $ampm 0]] == "p"} {
#             if {$t1 < 12} {
#                 incr t1 12
#             }
#         }
#     }
#
#puts "$month $day $year $t1 $t2 $t3, leapyears: $leapyears"
#
# Daylight Saving Time begins in the US at 2 a.m. on the
# first Sunday of April.  Time reverts to standard time at
# 2 a.m. on the last Sunday of October.
#
# Other parts of the world observe Daylight Saving Time as
# well.  While European nations have been taking advantage
# of the time change for decades, in 1996 the European
# Union (EU) standardized a EU-wide "summertime period."
# The EU version of Daylight Saving Time runs from the last
# Sunday in March through the last Sunday in October.
# During the summer, Russia's clocks are two hours ahead of
# standard time.  During the winter, all 11 of the Russian
# time zones are an hour ahead of standard time.  During
# the summer months, Russian clocks are advanced another
# hour ahead.  With their high latitude, the two hours of
# Daylight Saving Time really helps to save daylight.  In
# the southern hemisphere where summer comes in December,
# Daylight Saving Time is observed from October to March.
# Equatorial and tropical countries (lower latitudes) don't
# observe Daylight Saving Time since the daylight hours are
# similar during every season, so there's no advantage to
# moving clocks forward during the summer.
#
# Daylight savings: THIS IS WRONG BUT IT WORKS FOR MOST DATES!
#
# In the USA it fails between the April 1 and the first sunday, and
# between the last Sunday in October and the end of October.
# if {$month >= 4 && $month < 11} {
#     set offset -3600
# }



# procs below for debugging

proc aclock_debug {time} {
    puts [list $time days:[expr {$time/(24*60*60)}] seconds:[expr {$time % (24*60*60)}]]
}

# only works in Tcl 8
proc aclock_test {} {
    set time [expr {int(rand()*1000000000)}]
    set converted [aclock scan [aclock format $time]]
    if {$time == $converted} {
	#puts "Ok $time [aclock format $time]"
    } else {
	puts "Bad $time [aclock format $time] converted to $converted"
	set diff [expr {$converted - $time}]
	if {$diff < 0} {
	    set diff [expr -$diff]
	    set msg "behind"
	} else {
	    set msg "ahead"
	}
	puts "Difference was $diff secs $msg ([expr {$diff / 86400}] days, \
	  [expr {$diff % 86400}] secs)"
    }
}
