## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "paragraphs.tcl"
 #                                    created: 10/29/1999 {14:12:52 PM} 
 #                                last update: 10/04/2001 {19:12:59 PM} 
 #
 #  Author: largely Vince Darley; originals probably Pete Keleher
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # ###################################################################
 ##

namespace eval paragraph {}

## 
 # -------------------------------------------------------------------------
 # 
 # "paragraph::fill" --
 # 
 #  If there's a selection, then fill all paragraphs in that selection.
 #  If not then fill the paragraph surrounding the insertion point.
 #  The definition of a 'paragraph' may be mode dependent (see
 #  paraStart, paraFinish)
 #	   
 # -------------------------------------------------------------------------
 ##

proc paragraph::fill {} {
    if {[pos::compare [getPos] == [selEnd]]} {
	paragraph::fillOne
    } else {	
	set start [getPos]
	set end [selEnd]
	set p $start
	while {[pos::compare $p < $end] && [pos::compare $p < [maxPos]]} {
	    goto $p
	    set p [paragraph::fillOne 0 $start $end]
	}
	goto $start
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "paragraph::fillOne" --
 # 
 #  Fills the single paragraph surrounding the insertion point.  If called
 #  with parameter '0', it doesn't bother to remember where the insertion
 #  point was, which makes multiple paragraph fills quicker when called by
 #  'fillParagraph'.  Works for mode-dependent definitions of paragraphs and
 #  for commented out text (such as this paragraph here!).  
 #  Fixes: won't put a double-space after abbreviations like 'e.g.', 'i.e.'
 #  
 #  Works around the Alpha 'replaceText' bug.
 # -------------------------------------------------------------------------
 ##
proc paragraph::fillOne {{remember 1} {minstart ""} {maxend ""}} {
    global leftFillColumn fillColumn doubleSpaces

    set pos [getPos]
    # If we're on an empty line, do nothing.
    if {[string trim [getText [lineStart $pos] [nextLineStart $pos]]] == ""} {
	return
    }
    if {[set inComment [text::isInComment $pos ch]]} {
	# Find lines which contain just a comment char, but no actual text
	# (We want to flow the text in the comment in its constituent
	# paragraphs, not as one big block).
	set ch [string trim $ch]
	set chreg [quote::Regfind ${ch}]
	if {$ch == "*"} {
	    # We assume it's a C-style comment
	    set start [pos::math [lindex [search -s -f 0 -r 1 "^\[ \t\]*(${chreg}+|/\\*+)\[ \t\]*\$" $pos] 1] +1]
	    set end [lindex [search -s -f 1 -r 1 "^\[ \t\]*(${chreg}+|\\*+/)\[ \t\]*\$" $pos] 0]
	} else {
	    if {[isAtLineEnd $pos]} {
		set pos [pos::math $pos -1]
	    }
	    set start [lindex [search -s -n -f 0 -r 1 \
	      "^\[ \t\]*(${chreg}+\[ \t\]*${chreg}*\$|\[^${chreg} \r\n\t\]|\$)" $pos] 0]
	    set end [lindex [search -s -n -f 1 -r 1 \
	      "^\[ \t\]*(${chreg}+\[ \t\]*${chreg}*\$|\[^${chreg} \r\n\t\]|\$)" $pos] 0]
	    # The comment doesn't have a leading/trailing almost blank line
	    # Look for any line which is either blank, or starts with a 
	    # different character
	    if {$start == ""} {
		set start [lindex [search -s -n -f 0 -r 1 \
		  "^\[ \t\]*(\[^ \t[string index $ch 0]\]|\$)" $pos] 0]
		if {$start == ""} {
		    set start [minPos]
		} else {
		    set start [nextLineStart $start]
		}
	    } else {
		set start [nextLineStart $start]
	    }
	    if {$end == ""} {
		set end [lindex [search -s -f 1 -r 1 \
		  "^\[ \t\]*(\[^ \t[string index $ch 0]\]|\$)" $pos] 0]
	    }
	}
    } else {
	set start [paragraph::start $pos] 
	if {[pos::compare $start > $pos]} {
	    set end [paragraph::finish $start]
	} else {
	    set end [paragraph::finish $pos]
	}
    }
    # Extra arguments allow us to specify a region in which to operate
    if {$minstart != ""} {
	if {[pos::compare $minstart > $start]} {
	    set start $minstart
	}
    }
    if {$maxend != ""} {
	if {[pos::compare $maxend < $end]} {
	    set end $maxend
	}
    }
    if {$remember} {
	if {$inComment} {
	    set memory [rememberWhereYouAre $start $pos $end $chreg]
	} else {
	    set memory [rememberWhereYouAre $start $pos $end]
	}
    }
    
    if {$inComment} {
	set text [getText $start [nextLineStart $start]]
	if {[set boxComment [regexp -- "(${chreg}+)\[\r\n\]" $text "" commentSuffix]]} {
	    set boxWidth [posX [pos::math [nextLineStart $start] -1]]
	}
	regsub -all -- $chreg $text [string range "   " 1 [string length $ch]] fr
	regexp "^\[ \t\]*" $fr fr
	set left [string length [text::maxSpaceForm $fr]]
	if {$boxComment} {
	    set newFillColumn [expr {$boxWidth - $left - [string length $commentSuffix] -2}]
	} else {
	    set newFillColumn [expr {$fillColumn - $left}]
	}
	
	if {![regexp "^((\[ \t\]*${chreg}+)\[ \t\]*)" $text "" front commentPrefix]} {
	    alertnote "Sorry, I can't yet reflow the text inside this comment."
	    return $end
	}
	if {$boxComment} {
	    regsub -all "[quote::Regfind $commentSuffix](\r|\n|$)" \
	      [getText $start $end] "\\1" text
	    regsub -all "(^|\r|\n)[quote::Regfind $commentPrefix]" \
	      $text " " text
	} else {
	    regsub -all "(^|\r|\n)[quote::Regfind $commentPrefix]" \
	      [getText $start $end] " " text
	}
	
	regsub -all "\[ \t\r\n\]+" [string trim $text] " " text
    } else {
	# Get the leading whitespace of the current line and store length in 'left'
	set front [getLeadingIndent $pos left]
	# fill the text
	regsub -all "\[ \t\r\n\]+" [string trim [getText $start $end]] " " text
	set newFillColumn [expr {$fillColumn - $left}]
    }
    
    # turn single spaces at end of sentences into double
    if {$doubleSpaces} {regsub -all {(([^.][a-z]|[^a-zA-Z@]|\\@)[.?!]("|'|'')?([])])?) } $text {\1  } text}
    # 	if {$doubleSpaces} {regsub -all {(([^A-Z@]|\\@)[.?!][])'"]?) } $text {\1  } text}

    # temporarily adjust the fillColumns
    set ol $leftFillColumn
    set or $fillColumn
    set leftFillColumn 0
    set fillColumn $newFillColumn
		
    # break and indent the paragraph
    regsub -all " ?\r" "\r[string trimright [breakIntoLines $text]]" "\r${front}" text
    # reset columns
    set leftFillColumn $ol
    set fillColumn  $or
    if {$inComment && $boxComment} {
	global bind::_IndentSpaces
	set newtext ""
	foreach line [split $text "\r\n"] {
	    set pad [string range [set bind::_IndentSpaces] 0 [expr {$boxWidth- [string length $line] -2}]]
	    lappend newtext "$line$pad$commentSuffix"
	}
	set text "\r[join [lrange $newtext 1 end] \r]"
    }
    
    # don't replace if nothing's changed
    if {"$text\r" != "\r[getText $start $end]"} {
	# workaround an alpha bug
	if {$remember} { 
	    getWinInfo a
	    if {[pos::compare [rowColToPos $a(currline) 0] > $start]} { goto $start }
	}
	replaceText $start $end "[string range $text 1 end]\r"
	if {$remember} {
	    goBackToWhereYouWere $start [pos::math $start + \
	      [string length $text]] $memory
	}
    }
    
    # in case we wish to fill a region
    return $end
}

## 
 # -------------------------------------------------------------------------
 # 
 #	"paragraph::start" -- "paragraph::finish"
 # 
 #  "Start": It's pretty clear for non TeX modes how this works.  The only
 #  key is that we start at the beginning of the current line and look
 #  back.  We then have a quick check for whether we found that very
 #  beginning (in which case return it) or if not (in which case we have
 #  found the end of the previous paragraph) we move forward a line.
 # 
 #  "Finish": The only addition is the need for an additional check for
 #  stuff which explicitly ends lines.
 #	   
 # Results:
 #  The start/finish position of the paragraph containing the given 'pos'
 # 
 # --Version--Author------------------Changes-------------------------------
 #    1.1     <vince@santafe.edu> Cut down on '()' pairs
 #    1.2     Vince - March '96		  Better filling for TeX tables ('hline')
 #    1.3     Johan Linde - May '96   Now sensitive to HTML elements
 #    1.4     <vince@santafe.edu> Handle Tcl lists, top of file fix.
 # -------------------------------------------------------------------------
 ##
proc paragraph::start {pos} {
    global mode 
    global ${mode}::startPara
    if {[pos::compare $pos == [maxPos]]} {set pos [pos::math $pos - 1]}
    set pos [lineStart $pos]
    if {[info exists ${mode}::startPara]} {
	set startPara [set ${mode}::startPara]
    } else {
	set startPara {^([ \t]*|([\\%].*))$}
    }

    set res [search -s -n -f 0 -r 1 -l [minPos] -- "$startPara" $pos]
    if {![llength $res] || $res == "0 0" } {
	# bug work-around.  Alpha fails to match '^' with start of file.
	return [lineStart [lindex [search -s -f 1 -r 1 "\[^ \t\r\n\]" [minPos]] 0]]
    } elseif {[pos::compare [lindex $res 0] == $pos]} {
	return $pos
    } else {
	return [nextLineStart [lindex $res 0]]
    }
	
}

proc paragraph::finish {pos} {
    global mode
    global ${mode}::endPara
    set pos [lineStart $pos]
    set end [maxPos]
    if {[info exists ${mode}::endPara]} {
	set endPara [set ${mode}::endPara]
    } else {
	set endPara {^([ \t]*|([\\%].*))$}
    }
    
    set res [search -s -n -f 1 -r 1 -l $end -- "$endPara" $pos]
    if {![string length $res]} {return $end}
    set cpos [lineStart [lindex $res 0]]
    if {[pos::compare $cpos == $pos]} {
	return [nextLineStart $cpos]
    }
    # A line which ends in '\\', '%...', '\hline', '\hhline'
    # signifies the end of the current paragraph in TeX mode
    # (the above checked for beginning of the next paragraph).
    if { $mode == "TeX" || $mode == "Bib" } {
	set res2 [search -s -n -f 1 -r 1 -l $end {((\\\\|\\h+line)[ \t]*|[^\\]%.*)$} $pos]
	if {[string length $res2]} {
	    if {[pos::compare [lindex $res2 0] < $cpos] } {
		return [nextLineStart [lindex $res2 0]]
	    }
	}
    }

    return $cpos
    
}

proc paragraph::select {} {
    set pos [getPos]
    set start [paragraph::start $pos] 
    set finish [paragraph::finish $pos]
    goto $start
    if {[info tclversion] < 8.0} {
	select $start $finish
    } else {
	::select $start $finish
    }
}

proc paragraph::sentence {} {
    set pos [getPos]
    set start [paragraph::start $pos] 
    set finish [paragraph::finish $pos]
    
    set t [string trim [getText $start $finish]]
    set period [regexp {\.$} $t]
    regsub -all "\[ \t\r\n\]+" $t " " text
    regsub -all {\. } $text "" text
    set result ""
    foreach line [split [string trimright $text {.}] ""] {
	if {[string length $line]} {
	    append result [breakIntoLines $line] ".\r"
	}
    }
    if {!$period && [regexp {\.\r} $result]} {
	set result [string trimright $result ".\r"]
	append result "\r"
    }
    if {$result != [getText $start $finish]} {
	replaceText $start $finish $result
    }
    goto $pos
}

#  Paragraph Navigation  #
# 
# While these next procs are intended for true 'text' files, they can also
# be used for programming modes where it makes more sense to navigate blocks
# of commands (as delineated by empty lines) rather than by indentation, as
# in Fortran files.
# 
# Contributed by Craig Barton Upright.
# 

## 
 # --------------------------------------------------------------------------
 # 
 # "paragraph::next" --
 # 
 # Advance the cursor to the start of the next paragraph.  "quietly" will
 # simply return the position of the start of the previous paragraph. 
 # "insertTo" determines where to move the current insertion point --
 #  
 # 0  don't do anything
 # 1  move the insertion point to the top of the buffer
 # 2  move the insertion point to the middle of the buffer
 #  
 # If there is a selection already highlighted, "insertTo" is ignored, and
 # the selection is extended to the either the END of the current paragraph
 # (if the selection is not there already) or to the END of the next
 # paragraph.
 # 
 # -------------------------------------------------------------------------
 ##

proc paragraph::next {{quietly 0} {insertTo 0}} {
    
    global mode
    global ${mode}::paragraphName
    
    if {[info exists ${mode}::paragraphName]} {
	set what [set ${mode}::paragraphName]
    } else {
	set what "paragraph"
    }
    
    set pat  {[\r\n]+[\r\n\t ]*[\r\n]+[^\r\n]}
    set pos0 [getPos]
    if {[pos::compare [selEnd] == [maxPos]]} {
	set pos1 [maxPos]
    } elseif {[isSelection]} {
	set pos1 [pos::math [selEnd] + 1]
    } else {
	set pos1 [getPos]
    } 
    
    if {![catch {search -s -f 1 -r 1 $pat $pos1} match]} {
	set pos2 [lineStart [lindex $match 1]]
	if {[pos::compare $pos1 >= $pos2]} {
	    set pos2 [paragraph::finish [lindex $match 1]]
	}
    } else {
	set pos2 [maxPos]
    }
    if {$quietly} {
	return $pos2
    } elseif {[isSelection]} {
	set pos3 [nextLineStart [lindex $match 0]]
	if {[pos::compare $pos1 >= $pos3]} {
	    set pos3 [maxPos]
	} 
	if {[info tclversion] < 8.0} {
	    select   $pos0 $pos3
	} else {
	    ::select $pos0 $pos3
	}
    } else {
	goto $pos2
    }
    if {$insertTo == 1} {insertToTop} elseif {$insertTo == 2} {centerRedraw} 
    if {[pos::compare $pos2 == [maxPos]]} {
	status::msg "No further ${what}s in the file."
    } elseif {[isSelection]} {
	status::msg "Next ${what}: [getText $pos2 [nextLineStart $pos2]]"
    } else {
	status::msg [getText $pos2 [nextLineStart $pos2]]
    } 
}

## 
 # --------------------------------------------------------------------------
 # 
 # "paragraph::prev" --
 # 
 # Back up the cursor to the start of the previous paragraph.  If the cursor
 # is currently in a paragraph, move to the start if the current one. 
 # "quietly" will simply return the position of the start of the' previous
 # paragraph.  "insertTo" determines where to move the current insertion
 # point --
 #  
 # 0  don't do anything
 # 1  move the insertion point to the top of the buffer
 # 2  move the insertion point to the middle of the buffer
 #          
 # If there is a selection already highlighted, "insertTo" is ignored, and
 # the selection is extended to the either the beginning of the current
 # paragraph (if the selection is not there already) or to the beginning of
 # the previous paragraph.
 # 
 # --------------------------------------------------------------------------
 ##

proc paragraph::prev {{quietly 0} {insertTo 0}} {
    
    global mode
    global ${mode}::paragraphName
    
    if {[info exists ${mode}::paragraphName]} {
	set what [set ${mode}::paragraphName]
    } else {
	set what "paragraph"
    }
    
    set pat {[^\r\n\t ]+[^\r\n]+[\r\n]+[\r\n\t ]*[\r\n]+[^\r\n]}
    set pos0 [selEnd]
    if {[pos::compare [getPos] == [minPos]]} {
	set pos1 [minPos]
    } else {
	set pos1 [pos::math [getPos] - 1]
    } 
    if {![catch {search -s -f 0 -r 1 $pat $pos1} match]} {
	set pos2 [lineStart [lindex $match 1]]
	if {[pos::compare [getPos] <= $pos2]} {
	    set pos2 [paragraph::start [lindex $match 0]]
	}
    } else {
	set pos2 [minPos]
    }
    if {$quietly} {
	return $pos2
    } elseif {[isSelection]} {
	if {[info tclversion] < 8.0} {
	    select   $pos2 $pos0
	} else {
	    ::select $pos2 $pos0
	}
    } else {
	goto $pos2
    }
    if {$insertTo == 1} {insertToTop} elseif {$insertTo == 2} {centerRedraw} 
    if {[pos::compare $pos2 == [minPos]]} {
	status::msg "No further ${what}s in the file."
    } else {
	status::msg [getText $pos2 [nextLineStart $pos2]]
    } 
}

## 
 # --------------------------------------------------------------------------
 # 
 # "paragraph::reformat" --
 # 
 # This is NOT the same as paragraph::fill.  Instead, this procedure will
 # align the indentation of every line contained in the paragraph, so
 # something like this:
 #  
 # If there's a selection, then fill 
 #           all paragraphs in that selection.
 #           If not then fill the paragraph surrounding the insertion point.
 #  The definition of a 'paragraph' may be mode dependent 
 #                                      (see paraStart, paraFinish)
 #  
 #  will end up like this:
 #  
 # If there's a selection, then fill 
 # all paragraphs in that selection.
 # If not then fill the paragraph surrounding the insertion point.
 # The definition of a 'paragraph' may be mode dependent 
 # (see paraStart, paraFinish)
 #  
 # Following reformatting, the cursor is placed at the start of the next
 # paragraph.
 # 
 # -------------------------------------------------------------------------- 
 ##

proc paragraph::reformat {} {
    
    if {![isSelection]} {paragraph::select}
    
    status::msg "Reformatting "
    ::indentRegion
    goto [pos::math [getPos] -1]
    goto [paragraph::prev 1 0]
    goto [paragraph::next 1 0]
    status::msg "Reformatted."
}
