#  AlphaTcl - core Tcl engine

namespace eval quote {}

## 
 # -------------------------------------------------------------------------
 # 
 # "quote::" --
 # 
 # Manipulate string so search and insertion procedures work as expected.
 # When strings are passed to functions such as 'regexp', 'glob', 
 # 'lsearch -glob', etc. certain characters in those strings will be
 # interpreted as special (in some sense) unless they are preceded
 # by a backslash '\' character.  Exactly which characters have this
 # effect depends on the command in question.  These procedures allow
 # you to quote exactly the right characters so the commands work
 # as expected with arbitrary strings.
 # 
 # Of course, these procedures should only be used when you want to
 # avoid the effect of the special characters -- usually you don't!
 # 
 # quote::Find
 # 
 #  Use this for 'glob' type searches, but not 'glob' itself!  The
 #  commands 'string match', 'lsearch -glob' need their arguments
 #  quoted with this procedure.
 #  
 # quote::Glob
 #  
 #  Glob treats expressions like {a,b,c} specially, in addition to
 #  *,? etc, so requires a separate procedure.
 # 	
 # quote::Regfind
 # 
 #  Use this for regexp searches.  Note that this procedure hasn't
 #  been tested much with the advanced regexps in Tcl 8.2
 #  
 # quote::Regsub
 # 
 #  Use this for the replacement expression.  A common usage might look
 #  like this:
 #   
 #   regsub -all [quote::Regfind $from] [read $cid] [quote::Regsub $to] out
 #  
 # quote::Insert
 # 
 #  Quotes any block of text captured from a window so it can be used as a 
 #  Tcl string. e.g. 'set a [quote::Insert [getSelect]] ; eval insertText $a'
 #  will work correctly.  Can be used to generate procedures on the fly,
 #  especially to add to your prefs.tcl:
 #  
 #   set a [quote::Insert [getSelect]]
 #   prefs::tclAddLine "proc foo \{\} \{ return \"$a\" \}"
 # 
 # -------------------------------------------------------------------------
 ##
proc quote::Find  str {
    regsub -all {[][\\*?]} $str {\\&} str
    return $str
}

proc quote::Regfind str {
    regsub -all {[][\$?^|*+()\.\{\}\\]} $str {\\&} str
    return $str
}

proc quote::Regsub str {
    regsub -all {(\\|&)} $str {\\&} str
    return $str
}

proc quote::Glob str {
    regsub -all {[][*?\{\}\\]} $str {\\&} str
    return $str
}

proc quote::Insert str {
    regsub -all {[][\\$"\{\}]} $str {\\&} str
    regsub -all "\[\r\n\]" $str "\\r" str
    regsub -all "\t" $str "\\t" str
    return $str
}

## 
 # -------------------------------------------------------------------------
 # 
 # "quote::Url" --
 # 
 #  If you want a piece of arbitrary text to be part of a URL, then
 #  various characters needed to be turned into their hexadecimal
 #  equivalent.  This procedure does that.
 # -------------------------------------------------------------------------
 ##
proc quote::Url {str {slash 0}} {
    set hexa {0 1 2 3 4 5 6 7 8 9 A B C D E F}
    set nstr ""
    set exp "\[\001- \177-%<>\"#\?=&;|\\{\\}\\`^"
    if {$slash} {append exp "/"}
    append exp "\]"
    while {[regexp -indices $exp $str c]} {
	set asc [text::Ascii [string index $str [lindex $c 0]]]
	append nstr [string range $str 0 [expr {[lindex $c 0] - 1}]]
	append nstr % [lindex $hexa [expr {$asc / 16}]] [lindex $hexa [expr {$asc % 16}]]
	set str [string range $str [expr {[lindex $c 1] + 1}] end]
    }
    return "$nstr$str"
}

proc quote::UrlExceptAnchor {str} {
    set url ""
    regexp {[^#]*} $str url
    set anchor [string range $str [string length $url] end]
    return "[quote::Url $url]$anchor"
}

proc quote::Unurl {str} {
    set hexa {0 1 2 3 4 5 6 7 8 9 A B C D E F}
    set nstr ""
    while {[regexp -indices {%[0-9A-F][0-9A-F]} $str hex]} {
	append nstr [string range $str 0 [expr {[lindex $hex 0] - 1}]]
	append nstr [text::Ascii [expr {16 * [lsearch $hexa [string index $str [expr {[lindex $hex 0] + 1}]]] \
	  + [lsearch $hexa [string index $str [expr {[lindex $hex 0] + 2}]]]}] 1]
	set str [string range $str [expr {[lindex $hex 1] + 1}] end]
    }
    return "$nstr$str"
}

loadHtmlUtilities.tcl

proc quote::Unhtml {text} {
    global html::CharacterSpecial
    set entitylist [list "&amp;" "&lt;" "&gt;" "&#8482;" "&nbsp;" "&quot;"]
    foreach char $entitylist {
	set schar [string range $char 1 [expr {[string length $char] - 2}]]
	if {[info exists html::CharacterSpecial($schar)]} {
	    set rtext [set html::CharacterSpecial($schar)]
	} elseif {$schar == "amp"} {
	    set rtext "\\&"
	} elseif {$schar == "lt"} {
	    set rtext "<"
	} elseif {$schar == "gt"} {
	    set rtext ">"
	} elseif {$schar == "quot"} {
	    set rtext "\""
	} elseif {$schar == "nbsp"} {
	    set rtext " "
	} elseif {$schar == "#8482"} {
	    set rtext ""
	}
	
	set subNum [regsub -all $char $text $rtext text]
    }
    return $text
}

# These procs have been modified to avoid substitutions in TeX commands 
# starting with \n, \r and \t. The fix is based on replacing single \ by
# double \\ in 'quote::Display' and replacing \(n|r|t) by their ascii
# counterpart only if there is an odd number of \.
proc quote::Display str {
    regsub -all {\\} $str {\\\\} str
    regsub -all "\r" $str "\\r" str
    regsub -all "\n" $str "\\n" str
    regsub -all "\t" $str "\\t" str
    return $str
}

proc quote::Undisplay str {
    regsub -all {(^|[^\\]|(\\\\)+)\\r} $str "\\1\r" str
    regsub -all {(^|[^\\]|(\\\\)+)\\n} $str "\\1\n" str
    regsub -all {(^|[^\\]|(\\\\)+)\\t} $str "\\1\t" str
    regsub -all {\\\\} $str {\\} str
    return $str
}

## 
 # -------------------------------------------------------------------------
 # 
 # "quote::Prettify" --
 # 
 #  Since we're supposed to be a LaTeX editor, we handle symbols with
 #  TeX in a bit differently
 # -------------------------------------------------------------------------
 ##
proc quote::Prettify str {
    if {![regexp {^.+,([^,]+)$} $str b a]} then {
	set a [string toupper [string index $str 0]]
	regsub -all {([^A-Z])([A-Z])} [string range $str 1 end] {\1 \2} b
	regsub -all {((La|Bib|Oz|CMac) )?Te X} $a$b {\2TeX } a
	regsub -all {::} $a {-} a
    }
    return $a
}
proc quote::Menuify str {
    set a [string toupper [string index $str 0]]
    regsub -all { *([A-Z])} [string range $str 1 end] { \1} b
    append a $b
}
## 
 # -------------------------------------------------------------------------
 # 
 # "quote::WhitespaceReg" --
 # 
 #  Quote a string so you can search for it ignoring all problems with
 #  whitespace: all sequences of space/tab/cr are treated alike.
 # -------------------------------------------------------------------------
 ##
proc quote::WhitespaceReg { str } { 
    regsub -all "\[ \t\r\n\]+" $str {[ \t\r\n]+} str
    return $str
}
