## -*-Tcl-*-
 # ==========================================================================
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "reportABug.tcl"
 #                                    created: 01-11-04 12.37.11    
 #                                last update: 12/11/2001 {23:40:58 PM} 
 # 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #      
 #  Includes contributions from Craig Barton Upright
 # 
 #  Copyright (c) 2001 Vince Darley.
 # 
 #  Distributed under a Tcl style license.  
 # 
 # ===========================================================================
 # 
 # Issues to deal with:
 # 
 # Which package caused the bug?  look up maintainer name and email if not
 # obvious which package then what?
 # 
 # Can we do an automatic trace?  (see code below for some possible sample
 # code).  This would mean we'd want to interrupt the dialog, let the user
 # do the trace, and then come back to the dialog to complete the report.
 # Or, create a floating menu that would automatically do the trace,
 # assuming that we have somehow collected the necessary info on what should
 # be traced.
 # 
 # Assuming bug is in the core Alpha or AlphaTcl code, the report should go
 # to bugzilla.  (Currently all reports sent to bugzilla.)
 # 
 # If it's in a 'subsidiary package' then what?  Also go to bugzilla
 # or direct to the maintainer (probably best to bugzilla with a cc to
 # the maintainer? and assigned to the maintainer).
 # 
 # What pieces of information should the user provide?  Can we get better
 # information from the user if they have a few well-defined boxes (perhaps
 # called "Step by step actions to reproduce the problem", "Result I
 # expected", "Buggy result that actually occurred", "Patch/fix I think will
 # solve the problem", etc).
 # 
 # Given the inability to copy and paste effectively into Alpha's dialogs
 # from Alpha's windows (since dialogs are modal), might this be better
 # handled by a text window with a few hyper-links they can click on when
 # they are done?  In that case the dialog should just ask a few questions
 # and then open an appropriate window (could even be a 'Mail' window with
 # some additional embedded hyper links, or a floating-menu of buttons the
 # user can press: 'Perform trace', 'Cancel report', 'Send report', etc).
 # 
 # ===========================================================================
 ##

namespace eval reportABug {}

# Set some report fields.  Note that these are never unset, in case the
# user cancels midway and then starts over.

set reportABug(versionInfo) "[set alpha::version]\
  on [set alpha::platform] ($tcl_platform(platform)),\
  with Tcl[info patchlevel] and AlphaTcl [set alpha::tclversion]"

foreach reportField [list short_description details fix] {
    ensureset reportABug($reportField) ""
}
unset reportField

#  Bugzilla Fields  #

# The following fields are the only ones allowed by bugzilla as of this
# writing, lifted from the email generated by sending an empty message to
# <alpha-bugzilla+help@ics.mq.edu.au>.

set reportABug::EmailFields [list \
  product component version short_description rep_platform op_sys \
  bug_severity bug_status assigned_to cc_list keywords groupset ]

set reportABug::RequiredFields [list \
  product component version short_description ]
set reportABug::OptionalFields [list]

foreach bugzillaField [set reportABug::EmailFields] {
    ensureset reportABug($bugzillaField) ""
    ensureset reportABug::FieldOptions($bugzillaField) [list]
    if {![lcontains reportABug::RequiredFields $bugzillaField]} {
        lappend reportABug::OptionalFields $bugzillaField
    } 
}
unset bugzillaField

# If an incorrect field value is sent with a bug report, the user will
# receive a reply from bugzilla suggesting a list of possible correct
# values, which certainly won't lead to many new users using this item.
# 
# The goal here, then, is to make sure that the choices are at least valid,
# even if not completely up to date.  Of course, if the user upgrades,
# ideally this file will have already been updated to include the new
# allowed values.  Whenever the bugzilla fields are changed, then, this file
# should be updated as well.

set reportABug::Products [list Alpha Alpha-Bugzilla AlphaTcl Alphatk TclAE]
  
#  Product Components  #

set reportABug::Components(Alpha) {
    Core
    "Text Engine"
    Regexps
}
set reportABug::Components(Alpha-Bugzilla) {
    cgi
    html
    test
}
set reportABug::Components(AlphaTcl) {
    SystemCode
    "HTML & CSS"
    latex
    JavaScript
    Frontier
}

set reportABug::Components(Alphatk) {
    AlphatkCore
}
set reportABug::Components(TclAE) {
    AlphaTcl
    .shlb
}

# Make sure that these are proper lists.

foreach product [array names reportABug::Components] {
    set reportABug::Components($product) \
      [join [split [set reportABug::Components($product)]]]
}
unset product

#  Product Versions  #

set reportABug::Versions(Alpha) [alpha::package versions Alpha]

set reportABug::Versions(Alpha-Bugzilla) {
    2.9
    2.9.1
    2.9.2
    2.9.3
    2.9.4
    2.11
    2.11.1
    2.11.2
}
set reportABug::Versions(AlphaTcl) [alpha::package versions AlphaTcl]

set reportABug::Versions(Alphatk) [alpha::package versions Alpha]

set reportABug::Versions(TclAE) {
    2.0b1
    2.0b2
    2.0b3
    2.0b4
    2.0b5
    2.0b6
    2.0b7
    2.0b8
}

# Make sure that these are proper lists.

foreach product [array names reportABug::Versions] {
    set reportABug::Versions($product) \
      [join [split [set reportABug::Versions($product)]]]
}
unset product

#  Field Options  #

# More field options.  Set default values here as well.  Note that we never
# clear old values, in case the user cancelled and is trying again.

set reportABug::FieldOptions(rep_platform) {
    All
    "Macintosh PowerPC"
    "Macintosh 68k"
    Unix
    PC
    Other
}
set reportABug(rep_platform) "All"

set reportABug::FieldOptions(op_sys) {
    All
    "Mac System 7"
    "Mac System 7.5.5"
    "Mac System 7.6.1"
    "Mac System 8.0"
    "Mac System 8.1"
    "Mac System 8.5"
    "Mac System 8.6"
    "Mac System 9.0"
    "Mac System 9.0.4"
    "Mac System 9.1"
    "Mac System 9.2.1"
    "Mac System X"
    "Tcl/Tk 8.1"
    "Tcl/Tk 8.2"
    "Tcl/Tk 8.2.1"
    "Tcl/Tk 8.2.2"
    "Tcl/Tk 8.3"
    "Tcl/Tk 8.3.1"
    "Tcl/Tk 8.3.2"
    "Tcl/Tk 8.3.3"
    "Tcl/Tk 8.4a1"
    "Tcl/Tk 8.4a2"
    "Tcl/Tk 8.4a3"
    Other
}
set reportABug(op_sys) "All"

set reportABug::FieldOptions(bug_severity) {
    blocker
    critical
    major
    normal
    minor
    trivial
    enhancement
    faq
}
set reportABug(bug_severity) "normal"

set reportABug::FieldOptions(bug_status) {
    UNCONFIRMED
    NEW
}
set reportABug(bug_status) "NEW"

set reportABug::FieldOptions(keywords) {
    "(No keywords)"
    ALPHA-D
    alphaAlpha
    cabal
    crash
    dummy
    FAQ
    helpwanted
    mail-interface
    meta
    patch
    perf
    pp
    votesrequested
}

# This one is handled by checkboxes, so each group must be a list with two
# items: the actual group name, and the text that describes it.

set reportABug::FieldOptions(groupset) {
    {alphatesters {Only people in the "Alpha8 alpha Testers" group can see this bug}}
}

# Make sure that these are proper lists.

foreach field [array names reportABug::FieldOptions] {
    set reportABug::FieldOptions($field) \
      [join [split [set reportABug::FieldOptions($field)]]]
}
unset field

# ===========================================================================
# 
#  ----  #
# 
#  Bug Reporting Dialogs  #
# 

proc reportABug {} {
    
    global reportABug
    
    foreach page {
        welcome versionsAndComponents basics trace anyDetails possibleFix
        fields done
    } {
        reportABug::$page
    }
}

proc reportABug::start {title yy} {
    upvar $yy y
    if {[info tclversion] < 8.0} {
        return [dialog::text $title 10 y 100]
    } else {
        return [list -T $title]
    }
}

proc reportABug::welcome {} {
    
    global reportABug reportABug::Components
    
    set y 20
    set dialog [reportABug::start "Bug Reporter" y]
    eval lappend dialog [dialog::text \
      "Welcome to the Bug Reporter.  \nThis assistant will guide\
      you through a few screens to report a bug." \
      10 y 100]
    incr y  5

    set products [lsort -ignore [array names reportABug::Components]]
    eval lappend dialog [dialog::text \
      "First, you must pick a product on which to enter a bug." \
      10 y 60]
    eval lappend dialog [dialog::menu 10 y $products $reportABug(product)]
    
    incr y 10
    eval lappend dialog [dialog::button "Continue" 10 y "Cancel" 90 y] 
    eval lappend dialog [dialog::text \
      "___________________________________________________" \
      10 y 60]

    eval lappend dialog \
      [dialog::text "Have you checked whether a newer version is\
      available? \n(This bug might already have been fixed.)" \
      10 y 65]
    eval lappend dialog [dialog::button \
      "Check for new versions" 10 y]

    eval lappend dialog \
      [dialog::text "Have you read the \"Bug Reports and Debugging\"\
      help file (in the 'Help' menu)?  This might be a known problem\
      we are currently working on." 10 y 65]
    eval lappend dialog [dialog::button \
      "Read 'Bug Reports and Debugging'" 10 y \
      "Go to Alpha-Bugzilla" 250 y]

    incr y 10
    set res [eval dialog -w 440 -h $y $dialog]
    
    if {[lindex $res 2]} {status::errorMsg "Cancelled!"}
    set reportABug(product) [lindex $res 0]
    
    if {[lindex $res 3]} {
        # new versions
        urlView "http://alphatcl.sourceforge.net/cgi-bin/wikit/"
    } elseif {[lindex $res 4]} {
        # read bug reports
        help::openDirect "Bug Reports and Debugging"
    } elseif {[lindex $res 5]} {
        # go to bugzilla
        url::execute "http://www.maths.mq.edu.au/~steffen/Alpha/bugzilla/"
    } else {
        # continue
        return
    }
    status::errorMsg "Bug reporter postponed"
}
  
proc reportABug::versionsAndComponents {} {
    
    global reportABug reportABug::Components reportABug::Versions

    set product $reportABug(product)
    set components [set reportABug::Components($product)]
    set versions   [set reportABug::Versions($product)]
    set y 20
    set dialog [reportABug::start "Bug Reporter" y]
    # Select a component.
    set componentDefault $reportABug(component)
    eval lappend dialog [dialog::text \
      "Please select a component for '$reportABug(product)' :\n\
      (If you're not sure, make a good guess.)"\
      10 y 100]
    eval lappend dialog [dialog::menu 10 y $components $componentDefault]
    incr y  5
    # Select a version
    switch $product {
        Alpha - Alphatk {set versionDefault [alpha::package versions Alpha]}
        Alpha-Bugzilla  {set versionDefault 2.11}
        AlphaTcl        {set versionDefault [alpha::package versions AlphaTcl]}
        TclAE           {
            if {[catch {alpha::package versions TclAE} versionDefault]} {
                # Package might have thrown an error when loading.
                set versionDefault 2.0b8
            } 
        }
    }
    eval lappend dialog [dialog::text \
      "Please select a version number for '$reportABug(product)' :\n\
      (If you're not sure, make a good guess.)"\
      10 y 100]
    eval lappend dialog [dialog::menu 10 y $versions $versionDefault]
    
    incr y 10
    eval lappend dialog [dialog::okcancel -320 y 0 "Continue"] 
    set res [eval dialog -w 340 -h $y $dialog]
    
    if {[lindex $res end]} {status::errorMsg "Cancelled!"}
    set reportABug(component) [lindex $res 0]
    set reportABug(version)   [lindex $res 1]
}

proc reportABug::basics {} {
    
    global reportABug
    
    set y 20
    set dialog [reportABug::start "Bug Reporter Basic Information" y]

    eval lappend dialog \
      [dialog::text "Please enter a brief description for the bug" 10 y 60]\
      [dialog::edit $reportABug(short_description) 20 y 40]

    # How did this bug occur?
    eval lappend dialog \
      [dialog::text "Have you quit and restarted Alpha and are able\
      to reproduce the bug again with the same sequence of actions?" 10 y 60]\
      [dialog::checkbox "I can reproduce this bug" 0 20 y]
    
    incr y 20
    
    set actions [list \
      "Particular keypress or key-combination" \
      "Menu selection" \
      "Other"]
    
    eval lappend dialog \
      [dialog::text "What action did you take which showed the bug?" 10 y 60]\
      [dialog::menu 20 y $actions]
    
    
    # Ask the user which package or mode?
    # and if no package?
    # Ask user if particular key-combo binding caused the problem?

    eval lappend dialog [dialog::okcancel -420 y 0 "Continue"] 
    set res [eval dialog -w 440 -h $y $dialog]
    
    set reportABug(short_description) [lindex $res 0]
    set reportABug(reproducible)      [lindex $res 1]
    set reportABug(action)            [lindex $res 2]
    set reportABug(actionnum)         [lsearch -exact $actions [lindex $res 2]]

    if {[lindex $res end]} {status::errorMsg "Cancelled!"}
    if {!$reportABug(reproducible)} {
        alertnote "If you can't reproduce the bug, it is very unlikely\
          that anyone else will be able to do anything about it.  You\
          should probably investigate further before reporting it."
    }
}

proc reportABug::trace {} {
    
    global reportABug
    
    set reportABug(trace) ""
    set cmdname [lindex {keyPress menuSelection other} $reportABug(actionnum)]
    set reportABug(actionspec) [eval reportABug::$cmdname]
}

# We'll at least get some info as to what caused it.  Can add tracing
# capabilities later.

proc reportABug::keyPress {} {
    # This was a key press, so find out which key.
    if {[catch {dialog::getAKey "bug report" "" 0} result]} {
        return ""
    } elseif {$result == ""} {
        return ""
    } else {
        return [dialog::specialView::binding $result]
    }
}

proc reportABug::menuSelection {} {
    set p "Which menu item caused the bug?"
    if {[catch {prompt $p ""} result]} {
        set result ""
    }
    return $result
}

proc reportABug::other {} {
    set p "What action caused the bug?"
    if {[catch {prompt $p ""} result]} {
        set result ""
    }
    return $result
}

proc reportABug::anyDetails {} {
    
    global reportABug
    
    set y 20
    set dialog [reportABug::start "Bug Reporter Details" y]

    incr y 10
    eval lappend dialog [dialog::text \
      "Please include any further details about this bug which\
      might help a developer replicate the problem and resolve it."\
      10 y 65]
    eval lappend dialog \
      [dialog::edit $reportABug(details) 20 y 40 10]
    incr y 20

    eval lappend dialog [dialog::okcancel -420 y 0 "Continue"] 
    set res [eval dialog -w 440 -h $y $dialog]

    set reportABug(details) [lindex $res 0]
    
    if {[lindex $res end]} {status::errorMsg "Cancelled!"}
}

proc reportABug::possibleFix {} {
    
    global reportABug
    
    set y 20
    set dialog [reportABug::start "Bug Reporter Patch" y]

    incr y 10
    eval lappend dialog [dialog::text \
      "If you know of any possible solutions or patches for this bug,\
      please include them here, or simply press 'Continue'."\
      10 y 65]
    eval lappend dialog \
      [dialog::edit $reportABug(fix) 20 y 40 10]
    incr y 20

    eval lappend dialog [dialog::okcancel -420 y 0 "Continue"] 
    set res [eval dialog -w 440 -h $y $dialog]

    set reportABug(fix) [lindex $res 0]
    
    if {[lindex $res end]} {status::errorMsg "Cancelled!"}
}

proc reportABug::fields {} {
    
    global reportABug reportABug::OptionalFields reportABug::FieldOptions

    set y 20
    set dialog [reportABug::start "Bugzilla Fields" y]

    incr y 10
    eval lappend dialog [dialog::text \
      "Change any default bugzilla fields here if desired,\
      or simply press 'Continue'."\
      10 y 65]
    # These are handled differently.
    set optionalFields [lremove [set reportABug::OptionalFields] \
      keywords groupset]
    # These have edit fields, not menus.
    set textFields [list assigned_to cc_list]
    foreach field $optionalFields {
        set oldY $y
        eval lappend dialog [dialog::text $field 10 y 60]
        set y $oldY
        if {![lcontains textFields $field]} {
            set options [set reportABug::FieldOptions($field)]
            set option  $reportABug($field)
            eval lappend dialog [dialog::menu 100 y $options $option]
        } else {
            eval lappend dialog [dialog::edit $reportABug($field) 100 y 34]
            incr y 5
        }
    }
    # Now do the groupset items.
    foreach item [set reportABug::FieldOptions(groupset)] {
        eval lappend dialog [dialog::checkbox [lindex $item 1] 0 20 y]
    }

    incr y 20
    eval lappend dialog [dialog::okcancel -430 y 0 "Continue"] 
    set res [eval dialog -w 450 -h $y $dialog]
    if {[lindex $res end]} {status::errorMsg "Cancelled!"}

    set count 0
    foreach field $optionalFields {
        set reportABug($field) [lindex $res $count]
        incr count
    }
    # Now do the groupset items.
    set reportABug(groupset) [list]
    foreach item [set reportABug::FieldOptions(groupset)] {
        if {[lindex $res $count]} {
            lappend reportABug(groupset) [lindex $item 0]
        } 
        incr count
    }
    set reportABug(groupset) [join $reportABug(groupset) ", "]
    # Now we get the key word list in a separate dialog.
    set p "Select any keywords for this bug."
    set keywordOptions [set reportABug::FieldOptions(keywords)]
    if {[catch {listpick -p $p -l $keywordOptions} res]} {
        set res [list]
    } elseif {[lcontains res "(No keywords)"]} {
        set res [list]
    }
    set reportABug(keywords) [join $res ", "]
}

proc reportABug::done {} {
    reportABug::makeReport
    status::msg "Thanks for using the Bug Reporter !!"
    
    # global HOME
    # alertnote "Please contribute to complete the bug reporter"
    # file::openQuietly [file join $HOME Tcl SystemCode CorePackages reportABug.tcl]
}

# ===========================================================================
# 
#  ----  #
# 
#  Bug Reporting Support  #
# 

proc reportABug::makeReport {} {

    global reportABug reportABug::EmailFields

    set    report {
Bug Report

Please review the following information to ensure its accuracy.  

In the "DETAILS" section please explain what you were expecting to happen,
as well as the erroneous behavior that did happen.  Note that phrases like
"this didn't work" are generally worse than saying nothing at all, and are
unlikely to lead to a proper fix, while including information about any
error messages you might have encountered is extremely helpful.  Please
see the "Bugs and Bug Reporting" help file for more information.

Do NOT edit any of the blue "@field :" header names in the E-MAIL FIELDS
section below, which ensure that bugzilla's e-mail interface works properly.
(You can edit the values within these fields if you are sure that you know
what the correct ones should be, although this should not necessary, and if
you enter an invalid value the bug report will not be properly filed.)

When you are satisfied with your report, click here <<Send This Report>> to
open an e-mail composing window with the information below -- the report
will not be sent to bugzilla until you manually send it using your emailer
program.  Thanks for using the Bug Reporter !!
    _______________________________________________________________

}
    # Include the email field headers.  These have been set either by
    # using default values assigned above, or in reportABug::fields.
    append report "E-MAIL FIELDS"	              	\r\r
    foreach field [set reportABug::EmailFields] {
        ensureset reportABug($field) ""
        append report "@[format {%-20s} $field]: $reportABug($field)" \r
    }
    append report \r
    append report "BRIEF DESCRIPTION"              	\r\r
    append report $reportABug(short_description)        \r\r
    if {[string length $reportABug(actionspec)]} {
        append report "ACTION WHICH PRODUCED BUG"  	\r\r
        append report "$reportABug(action): "
        append report $reportABug(actionspec)       	\r\r
    } 
    append report "DETAILS"                        	\r\r
    append report $reportABug(details)                 	\r\r
    if {[string length $reportABug(fix)]} {
        append report "POSSIBLE SOLUTION"	  	\r\r
        append report $reportABug(fix)		       	\r\r
    } 
    append report "VERSION INFO"            		\r\r
    append report $reportABug(versionInfo)              \r\r
    append report "GENERAL SYSTEM INFO"            	\r\r
    append report [reportABug::addSystemInfo]       	\r\r
    if {[string length $reportABug(trace)]} {
        append report "TRACE"	  			\r\r
        append report $reportABug(trace)       		\r\r
    } 
    append report "This report was generated by the \"Report A Bug\" menu item.\r"
    
    new -n "* Bug Report *" -text $report
    
    # Now we colorize, hyperize this window.
    win::searchAndHyperise "^Bug Report$"                 {} 1 5 
    win::searchAndHyperise "\"Bugs And Bug Reporting\""   \
      {help::openGeneral "\1"}  1 3 +1 -1 
    win::searchAndHyperise "\"@field :\""                 {} 1 1 +1 -1
    win::searchAndHyperise "^@\[a-zA-Z0-9_\]+\[\\t \]+:"  {} 1 1
    win::searchAndHyperise "<<Send This Report>>" "reportABug::mailIt" 1 5
    refresh
    goto [set pos [minPos]]
    # A quick file marking.
    while {![catch {search -f 1 -r 1 -i 0 -s {^[A-Z][A-Z ]+$} $pos} match]} {
        set pos0 [lindex $match 0]
        set pos1 [lindex $match 1]
        set pos  [nextLineStart $pos0]
        setNamedMark [getText $pos0 $pos1] $pos0 $pos0 $pos0
    }
    setWinInfo -w [win::Current] dirty 0
}

proc reportABug::addSystemInfo {} {

    global alpha::tclversion mode global::features mode::features

    if {$mode == ""} {
        append trace "  mode: no mode active\r"
        append trace "  global features: [set global::features]\r"
    } else {
        append trace "  mode $mode\r"
        append trace "  global features: [set global::features]\r"
        append trace "  mode features: [set mode::features($mode)]\r"
    }
    append trace [global::listEnvironment]
    return $trace
}

# Could use this to trace a problem with a particular binding
#Bind 'key' [list reportABug::autoTrace $ofProc $oldBinding] <mode>

proc reportABug::autoTrace {ofProc oldBinding} {
    traceTclProc $ofProc
    set err [catch $ofProc res]
    set trace "Automatic capture of procedure '$ofProc'\r"
    append trace "return code was $err and result was $res\r"
    append trace [dumpTraces $ofProc "" dump]
    traceTclProc ""
    eval Bind $oldBinding
}

# This should only be called by a hyperlink or a floating menu item.

proc reportABug::mailIt {} {
    
    global reportABug tcl_platform

    # Find out where the intro ends and the report begins.
    set pat {^[\t ]*@[a-zA-Z0-9-]+[\t ]*:}
    if {![catch {search -f 1 -r 1 -s $pat [minPos]} match]} {
        set pos0 [lineStart [lindex $match 1]]
    } else {
        # Bother.  User messed with the introduction.
        set pos0 [minPos]
    }
    set report [string trim [getText $pos0 [maxPos]]]
    if {$tcl_platform(platform) == "windows"} {
        if {[string length $report] > 500} {
            putScrap $report
            set    report "The bug report was placed on the clipboard. "
            append report "(It was too long to handle automatically.)  "
            append report "Just paste it in."
        }
    }
    # Confirm the name of the bug.
    set pat "^@short_description *:"
    if {![catch {search -f 1 -r 1 -s $pat [minPos]} match]} {
        set pos1 [lindex $match 1]
        set pos2 [nextLineStart $pos1]
        set reportABug(short_description) [string trim [getText $pos1 $pos2]]
    }
    composeEmail [url::mailto alpha-bugzilla+submit@ics.mq.edu.au \
      subject $reportABug(short_description) body $report]
}

