## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "setupWizard.tcl"
 #                                    created: 02/29/2000 {09:40:40 AM} 
 #                                last update: 10/12/2001 {16:56:44 PM} 
 #  Author: Vince Darley
 #  E-mail: vince@santafe.edu
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 2000-2001 Vince Darley
 # 
 # Distributed under a Tcl style license.
 # 
 # ###################################################################
 ##

namespace eval setupAssistant {}


## 
 # -------------------------------------------------------------------------
 # 
 # "setupAssistant" --
 # 
 #  This is called each time you upgrade Alpha or AlphaTcl.
 #  
 #  'oldversions' is a list containing the previous version numbers
 #  of Alpha and AlphaTcl this user was using.
 #  
 #  'newversions' is a list for the current versions.
 #  
 #  We can use these lists to decide which setup wizards to run.
 #  
 #  Note: if the user has _never_ used Alpha before, then 'oldversions'
 #  will be an empty list.
 #  
 #  If both arguments are empty, the user called the assistant manually
 #  by using the Config menu item.
 # -------------------------------------------------------------------------
 ##
proc setupAssistant {{oldversions ""} {newversions ""}} {
    global tcl_platform
    set step 0
    
    set pages {
	welcome usualFeatures usualMenus webInformation 
	helpPreferences filesets 
	alphaDeveloper helpfulHints
    }
    
    if {$tcl_platform(platform) == "windows"} {
	set pages [linsert $pages 1 1 windowsGroup windowsAssociateFileTypes]
    }
    foreach page $pages {
	if {[catch {setupAssistant::$page $step} errMsg]} {
	    if {[string match "Exit Setup Assistant" $errMsg]} { break }
	}
	incr step
    }
    
    # Only update the package file if we weren't called manually
    if {[string length $oldversions] || [string length $newversions]} {
	# Update the "Packages" help file, then close it.
	global::listPackages ; menu::fileProc "File" "close"
    }
    message "Setup complete."
}

proc setupAssistant::start {title yy} {
    upvar $yy y
    set y 10
    if {[info tclversion] < 8.0} {
	return [dialog::text $title 10 y 90]
    } else {
	return [list -T $title]
    }
}

proc setupAssistant::welcome {step} {
    set y 0
    set dialog [setupAssistant::start "Welcome to the Alpha Setup Assistant" y]
    eval lappend dialog [dialog::text \
      "This assistant guides you through a few screens to help configure Alpha." \
      10 y 60]
    eval lappend dialog [dialog::button "Continue" 360 y "Cancel" 285 y] 
    set res [eval dialog -w 440 -h $y $dialog]
    if {[lindex $res end]} { error "Exit Setup Assistant" }
}

proc setupAssistant::windowsGroup {step} {
    set y 0
    set dialog [setupAssistant::start "Alpha Setup Assistant - Step $step" y]
    eval lappend dialog [dialog::text \
      "Would you like to create a Program Manager\
      group for Alphatk?  This way Alphatk will be\
      placed in your Start menu for easy access."\
      10 y 60]
    incr y 10
    eval lappend dialog [dialog::button "OK" 360 y "No thanks!" 265 y]
    set res [eval dialog -w 440 -h $y $dialog]
    if {![lindex $res end]} {
	windows::CreateGroup
    }
}

proc setupAssistant::windowsAssociateFileTypes {step} {
    set menu [list menu [list none edit open]]
    
    if {[catch {dialog::make -title "Alpha Setup Assistant - Step $step" \
      -cancel "No thanks!" \
      [list "File type associations"\
      [list static "Would you like to associated certain file\
      types with Alphatk?"]\
      [list static "This allows you to edit these types more easily."]\
      [list $menu ".tcl" edit]\
      [list $menu ".java" open]\
      [list $menu ".txt" open]\
      [list $menu ".tex" open]\
      [list $menu ".c" open]\
      [list $menu ".cpp" open]\
      [list $menu ".html" edit]\
      [list $menu ".js" open]\
      [list var "Other file extensions for 'open'" ""]\
      [list var "Other file extensions for 'edit'" ""]\
      [list static "An 'open' action is when you double-click on a file"]\
      [list static "An 'edit' action can be selected when you right-click on\
      a file"]]} res]} {
	# cancelled -- i.e. 'no thanks' pressed.
	return
    }
    
    set edit {}
    set open {}
    set none {}
    
    set i 0
    foreach ext [list .tcl .java .txt .tex .c .cpp .html .js] {
	set action [lindex $res $i]
	lappend $action $ext
	incr i
    }
    eval lappend open [lindex $res $i]
    incr i
    eval lappend edit [lindex $res $i]
    
    # Now set the Windows file associations.
    foreach action {open edit} {
	foreach ext [set $action] {
	    windows::AssociateActionWithAlphatk $ext $action
	}
    }
}

proc setupAssistant::usualFeatures {step} {
    set y 0
    set dialog [setupAssistant::start "Alpha Setup Assistant - Step $step" y]
    eval lappend dialog [dialog::text \
      "Alpha contains a lot of packages and features. \
      Most people find the following items useful. \
      Select those you would like to activate." \
      10 y 60]
    eval lappend dialog [dialog::checkbox \
      "Backup. " 1 10 y] \
      [dialog::text\
      "Makes a copy of saved files, with a tilde (~) added to the name." \
      28 y 65]
    incr y -10
    eval lappend dialog [dialog::checkbox \
      "Recent Files.  Lists recently accessed files under the 'File' menu." \
      1 10 y]
    eval lappend dialog [dialog::checkbox \
      "Emacs."  \
      0 10 y]\
      [dialog::text\
      "Adds keyboard navigation key-bindings familiar to Emacs users." \
      28 y 65]
    incr y -10
    eval lappend dialog [dialog::checkbox \
      "Macros." 1 10 y]\
      [dialog::text\
      "Adds the ability to record and execute keyboard macros."\
      28 y 65]
    incr y -10
    eval lappend dialog [dialog::checkbox \
      "Electric Bindings." 1 10 y]\
      [dialog::text\
      "Adds context-dependent keys that automatically format text." \
      28 y 70]
    incr y -10
    eval lappend dialog [dialog::checkbox \
      "Electric Expansions.  Allows acronym expansion." \
      1 10 y]
    eval lappend dialog [dialog::checkbox \
      "Electric Completions.  Allows word and command completion." \
      1 10 y]
    eval lappend dialog [dialog::checkbox \
      "Auto Continue Comment." 1 10 y]\
      [dialog::text\
      "Extends 'comments' across lines in certain file types."\
      28 y 65]
    incr y -10
    
    global tcl_platform
    if {(($tcl_platform(platform) == "macintosh")||($tcl_platform(platform) == "unix" && $tcl_platform(os) == "Darwin"))} {
	set mx 10
	set md 1
    } else {
	# Turn off, and move offscreen.
	set mx 5000
	set md 0
    }
    eval lappend dialog [dialog::checkbox \
      "Switch To.  Easy application switching." \
      $md $mx y]
    
    incr y 5
    eval lappend dialog [dialog::button \
      "Activate" 370 y \
      "Don't change" 260 y]
    
    set res [eval dialog -w 450 -h $y $dialog]
    if {![lindex $res end]} {
	global global::features
	set i 0
	foreach item { 
	    backup recentFilesMenu emacs macros elecBindings
	    elecExpansions elecCompletions autoContinueComment 
	    switchToMenu
	} {
	    if {[lindex $res $i]} {
		# Add to global list
		lunion global::features $item
		# Activate it now
		alpha::package require $item
	    }
	    incr i
	}
    }
}

proc setupAssistant::usualMenus {step} {
    set y 0
    set dialog [setupAssistant::start "Alpha Setup Assistant - Step $step" y]
    eval lappend dialog [dialog::text \
      "Alpha contains a lot of optional menus. \
      Most people find the following items useful. \
      Select those you would like to activate." \
      10 y 60]
    global tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	lappend usuals macMenu
	eval lappend dialog [dialog::checkbox \
	  "Mac Menu.  Control your mac's filesystem." \
	  1 10 y]
    }
    lappend usuals ftpMenu filtersMenu electricMenu internetConfigMenu mailMenu
    eval lappend dialog [dialog::checkbox \
      "FTP Menu.  To access and edit files on a remote ftp server." \
      1 10 y]
    eval lappend dialog [dialog::checkbox \
      "Filters Menu.  To do successive <Search And Replace> operations." \
      1 10 y]
    eval lappend dialog [dialog::checkbox \
      "Electric Menu.  For handling of text templates." \
      1 10 y]
    eval lappend dialog [dialog::checkbox \
      "Internet Config Menu.  Easy access to your Internet applications." \
      0 10 y]
    eval lappend dialog [dialog::checkbox \
      "Mail Menu.  To use Alpha as your mail editor. Works with Eudora." \
      0 10 y]
    if {[info tclversion] >= 8.0} {
	lappend usuals wwwMenu
	eval lappend dialog [dialog::checkbox \
	  "WWW Menu.  Browse the web inside Alpha (like 'lynx')." \
	  1 10 y]
    }

    incr y 10
    eval lappend dialog [dialog::button \
      "Activate" 370 y \
      "Don't change" 260 y]
    
    set res [eval dialog -w 450 -h $y $dialog]
    if {![lindex $res end]} {
	global global::features
	set i 0
	foreach item $usuals {
	    if {[lindex $res $i]} {
		# Add to global list
		lunion global::features $item
		# Activate it now
		alpha::package require $item
	    }
	    incr i
	}
    }
    
}

proc setupAssistant::webInformation { step } {
    set y 0
    set dialog [setupAssistant::start "Alpha Setup Assistant - Step $step" y]
    eval lappend dialog [dialog::text "Alpha has an active mailing list\
      for questions, help, and discussion. \
      Alpha also has a public bug reporting web-page and database. \
      Would you like to:" 10 y 60] \
      [dialog::checkbox "Go to the Alpha-users/developers mailing lists" 1 20 y] \
      [dialog::checkbox "Add the bug-reporting web-page as a browser bookmark" 1 20 y]
    incr y 10
    eval lappend dialog [dialog::button "OK" 360 y "No thanks!" 265 y]
    
    set res [eval dialog -w 440 -h $y $dialog]

    if {![lindex $res end]} {
	if {[lindex $res 0]} {
	    # join the mailing list
	    catch {url::execute "http://sourceforge.net/mail/?group_id=16416"}
	    alertnote "Please use the page which opens in your browser to subscribe\
	      to the 'users' and/or 'developers' mailing lists."
	}
	if {[lindex $res 1]} {
	    # add a web-page bookmark
	    catch {url::execute http://www.maths.mq.edu.au/~steffen/bugzilla/}
	    alertnote "Please bookmark the page which opens in your browser."
	}
    }
    
    # Add some dialogs for particular uses, e.g. the three main uses of
    # Alpha are probably:
    # (1) Programming in C/C++/Java
    # (2) Writing documents in LaTeX/BibTeX
    # (3) Writing HTML
}

proc setupAssistant::alphaDeveloper { step } {
    set y 0 
    set dialog [setupAssistant::start "Alpha Setup Assistant - Step $step" y]
    eval lappend dialog [dialog::text "If you are comfortable with\
      Tcl and wish to develop add-ons to Alpha (new\
      menus, modes, features, etc.), activate the following items:" 10 y 60]
    eval lappend dialog \
      [dialog::checkbox "Default filesets for Core, Modes, Menus and Packages" 0 10 y] \
      [dialog::checkbox "Make 'Alpha Developer Menu' package globally available" 0 10 y]
    incr y 10 
    eval lappend dialog [dialog::button \
      "Activate" 360 y \
      "Don't change" 250 y]
    set res [eval dialog -w 440 -h $y $dialog] 

    if {![lindex $res end]} {
	global global::features
	if {[lindex $res 0]} {
	    message "Creating your filesets"
	    global HOME
	    if {[lsearch -exact [set global::features] "filesetMenu"] == -1} {
		lappend global::features filesetMenu
	    }
	    fileset::fromHierarchy::create AlphaTclCore \
	      [file join $HOME Tcl SystemCode] *.tcl 2
	    fileset::fromHierarchy::create Menus \
	      [file join $HOME Tcl Menus] *.{tcl,flt} 3
	    fileset::fromHierarchy::create Modes \
	      [file join $HOME Tcl Modes] *.tcl 2
	    if {![file exists [file join $HOME Tcl Packages]]} { 
		file mkdir [file join $HOME Tcl Packages] 
	    }
	    fileset::fromHierarchy::create Packages \
	      [file join $HOME Tcl Packages] *.tcl 2

	    rebuildAllFilesets 1
	    # Register these to be saved.
	    foreach f {AlphaTclCore Menus Modes Packages} {
		modifyFileset $f
	    }
	}
	if {[lindex $res 1]} {
	    message "Activating 'Alpha Developer Menu' package"
	    if {[lsearch -exact [set global::features] "alphaDeveloperMenu"] == -1} {
		lappend global::features alphaDeveloperMenu
		alpha::package require alphaDeveloperMenu
	    }
	}
    }

}

proc setupAssistant::filesets { step } {
    set y 0 
    set dialog [setupAssistant::start "Alpha Setup Assistant - Step $step" y]
    eval lappend dialog [dialog::text \
      "Alpha uses 'filesets' to group related files. \
      Filesets provide an easy way to open, edit, search, and manipulate all files in the set.\
      \n\nWould you like to create filesets from..." 10 y 60]
    eval lappend dialog [dialog::text "All files in a directory" 20 y]
    incr y -20
    eval lappend dialog [dialog::button "Create" 360 y] 
    eval lappend dialog [dialog::text "All files in a hierarchy" 20 y] 
    incr y -20
    eval lappend dialog [dialog::button "Create" 360 y] 
    eval lappend dialog [dialog::text "All files from a multi-part LaTeX document" 20 y] 
    incr y -20
    eval lappend dialog [dialog::button "Create" 360 y]
    eval lappend dialog [dialog::text \
      "All files at an ftp site (edit them as if they were local:\
      they are automatically up/downloaded!)" 20 y 40]
    incr y -30
    eval lappend dialog [dialog::button "Create" 360 y]
    eval lappend dialog [dialog::text "Create as many filesets as you want\
      using these buttons." 10 y 60]
    # We put the continue button in front so it is the default
    set dialog [concat [dialog::button "Continue" 360 y] $dialog]
    while {1} {
	set res [eval dialog -w 440 -h $y $dialog] 
	if {[lindex $res 0]} {break}
	
	if {[lindex $res 1]} {
	    # directory
	    catch {newFileset fromDirectory}
	}

	if {[lindex $res 2]} {
	    # hierarchy
	    catch {newFileset fromHierarchy}
	}

	if {[lindex $res 3]} {
	    # latex
	    catch {newFileset tex}
	}

	if {[lindex $res 4]} {
	    # ftp
	    catch {newFileset ftp}
	}
    }
    
}

# This is a cheat.  We should really create a proper 'setup assistant'
# dialog with the same information in it.
proc setupAssistant::helpPreferences {step} {
    dialog::preferences preferences "Help"
}

proc setupAssistant::helpfulHints { step } {
    set y 0
    set dialog [setupAssistant::start "Alpha Setup Assistant - Done!" y]
    eval lappend dialog [dialog::text \
      "If you're new to Alpha, read Quick Start under the 'Help' menu. \
      Other items in this menu provide help with specific Features and Packages." \
      10 y 65]
    eval lappend dialog [dialog::text \
      "Look under the menu item 'Config->Preferences' for more global choices, and\
      'Config->Mode Prefs' for choices that apply to particular types of files. \
      'Config->Packages' lists additional extensions to Alpha. Use the 'Help' button in\
      Menus, Features, and Packages dialog boxes to explore Alpha." \
      10 y 65]
    set dialog [concat [dialog::button "Continue" 360 y] $dialog]
    set res [eval dialog -w 440 -h $y $dialog] 
}

