## -*-Tcl-*- (install)
 # ###################################################################
 #  Vince's Additions - an extension package for Alpha
 # 
 #  FILE: "superSearch.tcl"
 #                                    created: 1/10/97 {8:39:08 pm} 
 #                                last update: 11/28/2001 {09:45:50 AM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 1997-2001  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # Distributable under Tcl-style (free) license.
 # 
 # Thanks for: 
 # 
 # Package modified on 5/7/99 by Dominique d'Humieres
 # E-mail: <dominiq@physique.ens.fr>
 # following ideas from Pierre Basso (3/07/98)
 # E-mail: <basso@lim.univ-mrs.fr>
 # 
 # ###################################################################
 ##

# extension declaration
alpha::feature supersearch 1.0.3 global-only {
    alpha::package require AlphaTcl 7.2.2fc5
    set supersearchOn 0
    array set supersearch {
	f 1 i 1 m 0 r 0 b 0 multi 0 
	multifsets "" ismultifset 0 inselection 0 pattern "" 
	wrap 1 ignore 0 casereplace 0
    }
    namespace eval supersearch {}
    if {${alpha::platform} == "alpha"} {
	# For Alphatk, supersearch is the only option.
	# Use this key combination to toggle the new search dialog on and off
	newPref binding supersearch::toggle "<U<O/F" global "" 1
	alpha::addToPreferencePage Packages supersearch::toggle supersearchOn
    }
} {
    set supersearchOn 1
    supersearch::onoff
} {
    set supersearchOn 0
    supersearch::onoff
} maintainer {
    {Vince Darley} <vince@santafe.edu> <http://www.santafe.edu/~vince/>
} uninstall {this-file} help {
    Description:
	
    Togglable replacement for the 'Find' dialog box (shift-cmd-F is the
    default to toggle the dialog on/off)
	 
    Removes some deficiencies in Alpha's search dialog
    
    Allows single/multi-file regexp/batch/word-match search/replace/
    replace-all/in-selection from the same dialog box.
}

# Enter the number of rows of text to make room for in the 'Search For' and 
# 'Replace With' boxes in the search dialog
newPref var boxHeights 3 supersearch
# To clear the batch, multi-file and regexp flags when using shift-cmd-E 
# to enter a new search selection, click this box.||To leave all flags
# untouched when using shift-cmd-E, click this box.
newPref flag enterClearsBatch&MultiFlags 1 supersearch
# To turn Alpha's undo-memory off when doing 'replace all' (for added
# speed), click this box.||To remember all changes when doing 'replace all',
# click this box.
newPref flag undoOffForReplaceAll 1 supersearch
# To remember the current search fileset separately to Alpha's current
# fileset, click this box.||To synchronise the search fileset
# with Alpha's current fileset each time you open the search dialog, 
# click this box.
newPref flag separateSearchAndCurrentFileset 1 supersearch
# To set the 'in selection' flag automatically if there's a selection of at
# least 80 more characters than the length of the current search string,
# click this box.||To leave the 'in selection' flag in its original state,
# click this box.
newPref flag smartInSelectionFlag 1 supersearch
# To beep on a failed search, click this box.||To remain silent on
# failed searches, click this box.
newPref flag beepOnFailedSearch 1 supersearch 
# To display tabs, newlines etc in the search dialog as '\t', \n' etc,
# click this box.||To leave the displayed search and replace strings
# unmodified, click this box.
newPref flag quoteFunnyChars 0 supersearch 
# Enter the regular expression defining the set of characters for which
# preceding or following spaces are ignored when the 'IgnoreSpaces' is on.
# If the 'regexp' flag is on, the set is replaced by '\r'.
# The set must not contain 'space' or 'tab' characters.
newPref var checkSpacesAround "\[^ a-zA-Z0-9----\t\]" supersearch

#menu -n supersearch -p menu::generalProc {search replace+find replaceAll replace}
#float -m supersearch -n "" -z super
namespace eval supersearch {}

proc supersearch::toggle {} {
    global supersearchOn
    set supersearchOn [expr {1 - $supersearchOn}]
    supersearch::onoff
    message "Supersearch switched [expr {$supersearchOn ? {on} : {off}}]"
}
if {[info tclversion] < 8.0} {
    proc supersearch::onoff {} {
	global supersearchOn  supersearch
	global buffersearch bufferreplace
	set buffersearch ""
	set bufferreplace ""
	set alpha_cmds {find findAgain replace replaceAll replace&FindAgain \
	  findInNextFile findAgainBackward enterSearchString searchString \
	  replaceString performSearch}
	if {$supersearchOn} {
	    if {[info commands alpha::_find] != ""} {return}
	    foreach p $alpha_cmds {
		if {![llength [info commands $p]]} { auto_load $p }
		rename $p alpha::_$p
		;proc $p {args} "eval supersearch::$p \$args"
	    }
	    # Tcl 7.5 has a bug in 'eval ... $args'
	    proc searchString {args} {
		global buffersearch supersearch
		switch -- [llength $args] {
		    0 {
			return $buffersearch
		    }
		    1 {
			set t [lindex $args 0]
			set supersearch(search) $t
			supersearch::parseSearch $t
			return [set buffersearch $t]
		    }
		    default {
			error "Too many args"
		    }
		}
	    }
	    proc replaceString {args} {
		global bufferreplace supersearch
		switch -- [llength $args] {
		    0 {
			return $bufferreplace
		    }
		    1 {
			set t [lindex $args 0]
			set supersearch(replace) $t
			return [set bufferreplace $t]
		    }
		    default {
			error "Too many args"
		    }
		}
	    }
	} else {
	    if {[info commands alpha::_find] == ""} {return}
	    foreach p $alpha_cmds {
		rename $p {}
		rename alpha::_$p $p
	    }
	}
    }
    proc supersearch::searchString {args} {
	global buffersearch supersearch
	switch -- [llength $args] {
	    0 {
		return $buffersearch
	    }
	    1 {
		set t [lindex $args 0]
		set supersearch(search) $t
		supersearch::parseSearch $t
		return [set buffersearch $t]
	    }
	    default {
		error "Too many args"
	    }
	}
    }
    
    proc supersearch::replaceString {args} {
	global bufferreplace supersearch
	switch -- [llength $args] {
	    0 {
		return $bufferreplace
	    }
	    1 {
		set t [lindex $args 0]
		set supersearch(replace) $t
		return [set bufferreplace $t]
	    }
	    default {
		error "Too many args"
	    }
	}
    }

} else {
    proc supersearch::onoff {} {
	global supersearchOn
	set alpha_cmds {
	    find findAgain replace replaceAll replace&FindAgain
	    findInNextFile searchString enterSearchString findAgainBackward 
	    performSearch
	}
	if {$supersearchOn} {
	    if {[info commands ::alpha::_find] != ""} {return}
	    foreach p $alpha_cmds {
		if {![llength [info commands ::$p]]} { auto_load ::$p }
		rename ::$p ::alpha::_$p
		;proc ::$p {args} "eval ::supersearch::$p \$args"
	    }
	} else {
	    if {[info commands ::alpha::_find] == ""} {return}
	    foreach p $alpha_cmds {
		rename ::$p {}
		rename ::alpha::_$p ::$p
	    }
	}
	if {![info exists supersearch(search)]} {
	    ::searchString ""
	}
    }
    proc supersearch::searchString {args} {
	global supersearch
	switch -- [llength $args] {
	    0 {
		return [::alpha::_searchString]
	    }
	    1 {
		set t [lindex $args 0]
		set supersearch(search) $t
		supersearch::parseSearch $t
		return [::alpha::_searchString $t]
	    }
	    default {
		error "Too many args"
	    }
	}
    }
}

proc supersearch::enterSearchString {} {
    global supersearchmodeVars supersearch
    set t [getSelect]			
    searchString $t			
    set msg "Entered search '$t'"
    set supersearch(r) 0
    if {$supersearch(inselection)} {
	if {[pos::compare [getPos] >= $supersearch(start)] \
	  && [pos::compare [getPos] <= $supersearch(end)]} {
	    append msg ", still searching in previous selection."
	} else {
	    set supersearch(inselection) 0
	    append msg ", no longer searching in selection."
	}
    }
    if {$supersearchmodeVars(enterClearsBatch&MultiFlags)} {
	set supersearch(multi) 0
	set supersearch(b) 0
    }
    message [shorten $msg 80 45]
    supersearch::parseSearch
}

# set supersearch(start) 0
# set supersearch(end) 0

## 
 # -------------------------------------------------------------------------
 # 
 # "supersearch::performSearch" --
 # 
 #  Like 'search', but for use from Tcl code, therefore the equivalent
 #  of performSearch.  May have bugs still.
 # -------------------------------------------------------------------------
 ##
proc supersearch::performSearch {args} {
    global supersearch
    set opts(-b) 0
    set opts(-r) 0
    set opts(-f) 1
    set opts(-m) 0
    set opts(-i) 0
    getOpts {-f -r -m -b -i -l}
    if {[info exists opts(-s)]} {
	array set temp [array get supersearch]
    }
    set supersearch(multi) 0
    set supersearch(b) $opts(-b)
    set supersearch(r) $opts(-r)
    set supersearch(i) $opts(-i)
    set supersearch(m) $opts(-m)
    set supersearch(beep) [info exists opts(-beep)]
    if {[info exists opts(-l)]} {
	set supersearch(inselection) 1
	if {$opts(-f)} {
	    set supersearch(start) [lindex $args 1]
	    set supersearch(end) $opts(-l)
	} else {
	    set supersearch(start) $opts(-l)
	    set supersearch(end) [lindex $args 1]
	}
    } else {
	set supersearch(inselection) 0
    }
    searchString [lindex $args 0]
    goto [lindex $args 1]
    set res [supersearch::basicSearch $opts(-f)]
    if {[info exists opts(-s)]} {
	array set supersearch [array get temp]
    }
    unset supersearch(beep)
    
    if {[llength $res]} {
	return $res
    } else {
	error "Not found"
    }
}

proc supersearch::generalSearch {} {
    global supersearch
    if {$supersearch(b)} {
	supersearch::batchSearch
    } else {
	if {$supersearch(multi)} {
	    supersearch::getFiles
	    set supersearch(multiindex) -1
	    supersearch::findNextOkLocation
	    return
	} else {
	    if {$supersearch(inselection)} {
		goto $supersearch(start)
	    } else {
		goto [expr {$supersearch(f) ? [getPos] : [minPos]}]
	    }
	    supersearch::basicSearch
	}
    }
}

#  Basic searching  #
proc supersearch::findAgain {} {
    global supersearch
    if {$supersearch(b)} {set supersearch(b) 0}
    supersearch::basicSearch 1
}

proc supersearch::findAgainBackward {} {
    global supersearch
    if {$supersearch(b)} {
	dialog::errorAlert "You can't do backwards batch searches"
    } else {
	supersearch::basicSearch 0
    }
}

proc supersearch::basicSearch {{f 1} {rfrsh 0}} {
    global supersearch
    if {$supersearch(inselection) && !$supersearch(multi)} {
	set pos [getPos]
	set start $supersearch(start)
	set end $supersearch(end)
	set lstart [lindex [posToRowCol [lineStart $start]] 0]
	set lend [lindex [posToRowCol [nextLineStart $end]] 0]
	message "searching in selection between lines $lstart and $lend"
	if {[pos::compare $pos < $start] || [pos::compare $pos > $end]} {
	    goto [expr {$f ? $start : $end}]
	} 
    }
    set from $supersearch(reg)
    if {[catch {search -s -f $f -r 1 -i $supersearch(i) \
      -m $supersearch(m) -- $from [supersearch::searchStart $f]} p]} {
	supersearch::findInNextFile
    } else {
	if {$supersearch(inselection) && !$supersearch(multi)} {
	    if {$f} {
		if {[pos::compare [lindex $p 0] > $supersearch(end)]} {
		    supersearch::findNextOkLocation
		    return
		}
	    } else {
		if {[pos::compare [lindex $p 0] < $supersearch(start)]} {
		    supersearch::findNextOkLocation
		    return
		}
	    }
	}
	goto [lindex $p 0]

	getWinInfo wndw
	set wndwFrst $wndw(currline)
	set wndwDsp $wndw(linesdisp)
	set wndwln [lindex [posToRowCol [getPos]] 0]
	set wndwln [expr {4*(1-2*$f)*($wndwln-$wndwFrst)-(1-4*$f)*$wndwDsp}]
	if {$rfrsh||$wndwln < 0} {centerRedraw}
	eval select $p
	set supersearch(lastfound) $p
    }
}
	
proc supersearch::replace {} {
    global supersearch
    set s [getPos] ; set e [selEnd]
    if {[pos::compare $s != $e]} {
	if {$supersearch(r)} {
	    if {$supersearch(i)} {
		regsub -nocase -- $supersearch(reg) [getText $s $e] \
		  [replaceString] rep
	    } else {
		regsub -- $supersearch(reg) [getText $s $e] [replaceString] rep
	    }
	} else {
	    set rep [replaceString]
	}
	if {$supersearch(casereplace)} {
	    set char [lookAt $s]
	    if {[string toupper $char] == $char} {
		set rep "[string toupper [string index $rep 0]][string range $rep 1 end]"
	    }
	}
	replaceText $s $e $rep
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "supersearch::reg" --
 # 
 #  Take account of 'search' handling \t but regsub not handling it.
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.1} {
    proc supersearch::reg {str} {
	regsub -all -- {((^|[^\\])(\\\\)*)\\t} $str \\1\t str
	regsub -all -- {((^|[^\\])(\\\\)*)\\r} $str \\1\r str
	regsub -all -- {((^|[^\\])(\\\\)*)\\n} $str \\1\n str
	return ^$str\$
    }
    proc supersearch::regPart {str} {
	regsub -all -- {((^|[^\\])(\\\\)*)\\t} $str \\1\t str
	regsub -all -- {((^|[^\\])(\\\\)*)\\r} $str \\1\r str
	regsub -all -- {((^|[^\\])(\\\\)*)\\n} $str \\1\n str
	return ($str)
    }
} else {
    proc supersearch::reg {str} {
	return "^$str\$"
    }
    proc supersearch::regPart {str} {
	return "($str)"
    }
}

set supersearch(lastfound) [list]

proc supersearch::searchStart {{f 1}} {
    global supersearch
    if {[info exists supersearch(startfrom)]} {
	set p $supersearch(startfrom)
	unset supersearch(startfrom)
    } else {
	set p [getPos]
	# If there is any kind of selection it is either the last thing
	# found, or an initial selection from the user.  Either way we
	# want to change the start position.  Also, if there is no
	# selection, but the current position is equal to both the
	# start and end of the last thing found, then we successfully
	# found an empty string, and also want to adjust the start
	# position.
	if {[pos::compare $p != [selEnd]] || \
	  ([llength $supersearch(lastfound)] \
	  && [pos::compare $p == [lindex $supersearch(lastfound) 0]] \
	  && [pos::compare $p == [lindex $supersearch(lastfound) 1]])} {
	    if {$f} {
		set p [pos::math $p + 1]
	    } else {
		set p [pos::math $p - 1]
	    }
	}
    }
    set supersearch(laststart) $p
    return $p
}

proc supersearch::replace&FindAgain {} {
    supersearch::replace
    supersearch::findAgain
}

#  File switching  #
proc supersearch::nextFile {} {
    global supersearch
    set i 0
    supersearch::createContext nextFile
    while {$i < $supersearch(numfiles)} {
	if {[info exists supersearch(multiindex)] \
	  && $supersearch(multiindex) != ""} {
	    set f [lindex $supersearch(files) [incr supersearch(multiindex)]]
	    if {$f == ""} {
		set supersearch(files) ""
		unset supersearch(multiindex)
		return 0
	    } else {
		if {[file isdirectory $f]} {
		    incr i
		    continue
		}
		set remaining [expr {$supersearch(numfiles) \
		  - $supersearch(multiindex) -1}]
		if {[supersearch::_isInFile $f $remaining]} {
		    file::openQuietly $f
		    goto [minPos]
		    supersearch::deleteContext
		    return 1
		} else {
		    incr i
		}
	    }
	}
	continue
    }
    supersearch::deleteContext
    return 0
}

proc supersearch::createContext {type} {
    global supersearch
    set part $supersearch(firstLine)
    if {$supersearch(i)} {
	set case "-nocase"
	set scancase "-nocase"
    } else {
	set case "--"
	set scancase ""
    }
    set supersearch(cid) [scancontext create]
    if {$type == "multiBatchSearch" && $supersearch(singleline)} {
	if {[info tclversion] > 8.0} {
	    eval scanmatch $scancase \
	      [list $supersearch(cid) $supersearch(firstLine) {
		if {(!$supersearch(m) \
		  || [regexp $case "\\m$supersearch(firstLine)\\M" \
		  $matchInfo(line)])} {
		    browse::Add $f $matchInfo(line) $matchInfo(linenum)
		    incr matches
		}
	    }]
	} else {
	    eval scanmatch $scancase \
	      [list $supersearch(cid) $supersearch(firstLine) {
		if {(!$supersearch(m) \
		  || [regexp $case "\\b$supersearch(firstLine)\\b" \
		  $matchInfo(line)])} {
		    browse::Add $f $matchInfo(line) $matchInfo(linenum)
		    incr matches
		}
	    }]
	}
	return
    }
    
    eval scanmatch $scancase [list $supersearch(cid) $part {
	if {[info tclversion] > 8.0} {
	    set wmatch "\\m$supersearch(firstLine)\\M"
	} else {
	    set wmatch "\\b$supersearch(firstLine)\\b"
	}
	if {(!$supersearch(m) \
	  || [regexp [expr {$supersearch(i) ? {-nocase} : {--}}] \
	  $wmatch $matchInfo(line)]) \
	  && !$matches} {
	    set offset $matchInfo(offset)
	    set lines $matchInfo(line)
	    set matches 1
	    set mlines $supersearch(rcFirst)
	    if {$offset == 0 && $mlines == 0 && \
	      [llength $supersearch(rcList)] != 0} {
		set matches  0 ; return $matches
	    }
	    if {$mlines == 0} {return $matches}
	    if {$offset == 0 && $mlines == 2} {
		set matches  0 ; return $matches
	    }
	    set from $supersearch(regTail)
	    if {$fid != 0} {close $fid}
	    set fid [alphaOpen $f]
	    read $fid $offset
	    gets $fid lines
	    if {[eof $fid]} {set matches  0 ; return $matches}
	    while {$mlines > 0 } {
		if {$from == ""} {set mlines 0 ; return $matches}	 
		set mlines [regexp -indices "\\r" $from values]
		if {$mlines == 1} {
		    set rc [lindex $values 0]
		    set part [string range $from 0 [expr {$rc-1}]]
		    set from [string range $from [expr {$rc+1}] end]
		} else { 
		    set part $from
		}
		gets $fid lines
		if {[eof $fid]} {set matches  0 ; return $matches}
		if {$supersearch(ignore)} {
		    set lines [string trim $lines]
		}
		set part [supersearch::prepLine $part]
		set part "^$part"
		if {$mlines == 1} {set part "$part\$"}
		if {![regexp $case $part $lines]} {
		    set matches  0 ; return $matches
		}
		continue	
	    }
	}
    }]
    
}

proc supersearch::deleteContext {} {
    global supersearch
    if {[string length $supersearch(cid)]} {
	scancontext delete $supersearch(cid)
	set supersearch(cid) ""
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "supersearch::_isInFile" --
 # 
 #  This proc must be wrapped in calls to supersearch::createContext
 #  and supersearch::deleteContext.
 # -------------------------------------------------------------------------
 ##
proc supersearch::_isInFile {f {remaining ""}} {
    global supersearch
    # Are all of these necessary?
    set values 0
    set matches 0
    set lines {}
    set offset 0
    set fid 0
    
    if {$remaining == ""} {
	message "Searching [file tail $f]"
    } else {
	message "Searching ($remaining left) [file tail $f]"
    }
    
    if {![catch [list alphaOpen $f "r"] fid]} {
	catch {scanfile $supersearch(cid) $fid}
	close $fid
    }
    
    return $matches
}

proc supersearch::findNextOkLocation {} { supersearch::findInNextFile }
proc supersearch::findInNextFile {} {
    global supersearch
    if {$supersearch(multi)} {
	if {[supersearch::nextFile]} {
	    return [supersearch::basicSearch 1 1]
	}
	set str "Can't find '[searchString]', and there are no more files\
	  to search."
	message "[shorten $str 80 50]"
	set supersearch(numfiles) -1
    } else {
	if {$supersearch(inselection)} {
	    set str "No more instances of '[searchString]' in selected range."
	    message "[shorten $str 80 38]"
	} else {
	    if {$supersearch(wrap)} {
		if {$supersearch(b)} {
		    if {[pos::compare $supersearch(laststart) != [maxPos]]} {
			set supersearch(startfrom) [maxPos]
			message "Now searching backwards from end of document."
			return [supersearch::basicSearch 0]
		    }
		} else {
		    if {[pos::compare $supersearch(laststart) != [minPos]]} {
			set supersearch(startfrom) [minPos]
			message "Now searching forwards from start of document."
			return [supersearch::basicSearch 1]
		    }
		}
	    }
	    message "Can't find '[shorten [searchString] 65 30]'."
	}
    }
    global supersearchmodeVars
    if {$supersearchmodeVars(beepOnFailedSearch)} {
	if {![info exists supersearch(beep)] || $supersearch(beep)} {
	    beep
	}
    }
    return
}

#  Batch  #
proc supersearch::batchSearch {} {
    global supersearch
    if {$supersearch(multi)} {
	supersearch::multiBatchSearch
    } else {
	supersearch::basicBatchSearch
    }
}

proc supersearch::basicBatchSearch {} {	
    global supersearch
    set from $supersearch(reg)
    if {[info tclversion] < 8.0} {
	if {$supersearch(m)} {
	    set from "^.*\\b$from\\b.*$"
	} else {
	    set from "^.*$from.*$"
	}
    } else {
	# We have to use newline sensitive matching in case
	# we have a multiline-search aware text engine
	if {$supersearch(m)} {
	    set from "^\[^\r\n\]*\\m${from}\\M\[^\r\n\]*\$"
	} else {
	    set from "^\[^\r\n\]*${from}\[^\r\n\]*\$"
	}
    }
    set pos [expr {$supersearch(f) ? [getPos] : [minPos]}]
    set fileName [win::Current]
    browse::Start
    # Could use this, but it is ugly.
    #browse::Dynamic 1
    while {![catch {search -s -f 1 -r 1 -i $supersearch(i) -- $from $pos} mtch]} {
	browse::Add $fileName [eval getText $mtch] \
	  [lindex [posToRowCol [lindex $mtch 0]] 0] 0
	set next [lindex $mtch 1]
	if {[pos::compare $next == $pos]} {
	    set pos [pos::math $pos + 1]
	    if {[pos::compare $pos > [maxPos]]} {
		# odd circumstance can happen on Alphatk
		break
	    }
	} else {
	    set pos $next
	}
    }
    browse::Complete
}

proc supersearch::multiBatchSearch {} {
    global supersearch
    set from $supersearch(reg)
    if {$supersearch(m)} {
	if {[info tclversion] < 8.0} {
	    set from "\\b$from\\b"
	} else {
	    set from "\\m$from\\M"
	}
    }
    
    if {$supersearch(i)} {
	set case "-nocase"
    } else {
	set case "--"
    }
    
    supersearch::getFiles
    set supersearch(multiindex) -1
    set changes 0
    set i 0
    set values 0
    set matches 0
    supersearch::createContext multiBatchSearch
    browse::Start
    browse::Dynamic
    while {$i < $supersearch(numfiles)} {
	if {[info exists supersearch(multiindex)] \
	  && $supersearch(multiindex) != ""} {
	    set f [lindex $supersearch(files) [incr supersearch(multiindex)]]
	    if {$f == ""} {
		set supersearch(files) ""
		unset supersearch(multiindex)
		continue
	    } else {
		if {[file isdirectory $f]} {
		    incr i
		    continue
		}
		set remaining [expr {$supersearch(numfiles) \
		  - $supersearch(multiindex) -1}]
		if {$supersearch(singleline)} {
		    message "Searching ($remaining left) [file tail $f]"
		    if {![file exists $f]} {
			alertnote "File '$f' doesn't exist any more.  It\
			  won't be searched."
		    } elseif {![catch [list alphaOpen $f "r"] fid]} {
			if {[catch {scanfile $supersearch(cid) $fid}]} {
			    alertnote "Opened $f, but couldn't scan it!\
			      May not be able to find matching lines."
			}
			close $fid
		    } else {
			alertnote "Couldn't open $f with read permission!\
			  Will not be able to find matching lines."
		    }
		} else {
		    if {[supersearch::_isInFile $f $remaining] \
		      && !$supersearch(singleline)} {
			message "Looking at '[file tail $f]'"
			if {![catch [list alphaOpen $f "r"] cid]} {
			    if {[catch {read $cid} tmp]} {
				alertnote "Opened $f, but couldn't read it!\
				  Will not be able to find matching lines."
				close $cid
			    } else {
				close $cid
				regsub -all "\n" $tmp "\r" tmp
				set nl 1
				while {[regexp -indices $case "($from)" \
				  $tmp values]} {
				    set m0 [lindex $values 0]
				    set m1 [lindex $values 1]
				    set beg [string range $tmp 0 [expr {$m0-1}]]
				    set lrc [string last "\r" $beg]
				    if {$lrc < 0} {
					set m0 0
				    } else {
					set m0 [expr {$lrc+1}]
				    } 
				    set count [regsub -all "\r" $beg "\r" beg]
				    incr nl $count
				    set fol [string range $tmp $m1 end]
				    set frc [string first "\r" $fol]
				    if {$frc < 0} {
					incr m1 [string length $fol]
				    } else {
					incr m1 [expr {$frc-1}]
				    } 
				    set mtch [string range $tmp $m0 $m1]
				    set tmp [string range $tmp [expr {$m1+1}] end]
				    browse::Add $f $mtch $nl
				    incr matches
				    set count [regsub -all "\r" $mtch "\r" beg]
				    incr nl $count
				}
			    }
			} else {
			    alertnote "Couldn't open $f with read permission!\
			      Will not be able to find matching lines."
			}
		    }		
		}
		incr i
	    }
	    continue
	}
    }
    supersearch::deleteContext
    
    browse::Complete
}

#  Replace all  #
## 
 #  search  [options] <pattern> <pos> - 
 #   -f <num>      - go forward?
 #   -r <num>      - regular expression?
 #   -s            - save previous search string and search flags.
 #   -i <num>      - ignore case?
 #   -m <num>      - match words?
 #   -n            - failed search still returns TCL_OK, but null string.
 #   -l <limit>    - limit on how search goes.
 #   --            - next arg is the pattern.
 ##
proc supersearch::replaceAll {{force 0}} {
    supersearch::reallyReplaceAll $force
}

proc supersearch::reallyReplaceAll {{force 1}} {
    global supersearch
    if {$supersearch(multi)} {
	if {!$force} {
	    if {![dialog::yesno "Are you sure you want to perform\
	      a global replacement in many files?"]} {
		message "Cancelled"
		return
	    }
	}
	supersearch::multiReplaceAll
    } else {
	if {$supersearch(inselection)} {
	    goto $supersearch(start)
	} else {
	    goto [expr {$supersearch(f) ? [getPos] : [minPos]}]
	}
	supersearch::basicReplaceAll
    }
}

proc supersearch::basicReplaceAll {{f 1}} {
    global supersearch supersearchmodeVars undoOn
    set from $supersearch(reg)
    if {$supersearch(m)} {
	if {[info tclversion] < 8.0} {
	    set from "\\b$from\\b"
	} else {
	    set from "\\m$from\\M"
	}
    } 
    set savefrom [searchString]
    set to [replaceString]
    if {!$supersearch(r)} {
	set to [quote::Regsub $to]
	if {[catch {regsub -- $from "$savefrom" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}
	if {![llength {$dummy}]} {
	    alertnote "Regsub problem"
	    return
	}
    } else {
	if {[catch {regsub -- $from "$from" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}	
    }

    set pos [supersearch::searchStart]
    if {$supersearch(inselection) && !$supersearch(multi)} {
	set range [list $supersearch(start) $supersearch(end)]
    } else {
	if {$f} {
	    set range [list $pos [maxPos]]
	} else {
	    set range [list [minPos] $pos]
	}
    }
    if {$supersearch(i)} {
	set case "-nocase --"
    } else {
	set case "--"
    }
    if {[set count [eval regsub -all $case \
      [list $from [eval getText $range] $to out]]]} {
	if {$supersearchmodeVars(undoOffForReplaceAll)} {
	    set oldUndo $undoOn ; set undoOn 0
	}	
	eval replaceText $range [list $out]
	if {$supersearchmodeVars(undoOffForReplaceAll)} {
	    set undoOn $oldUndo
	}
	message "Replacements: $count in given range: $range"
    } else {
	message "No replacements in given range: $range"
    }
}

proc supersearch::multiReplaceAll {} {
    global supersearch
    set from $supersearch(reg)
    if {$supersearch(m)} {
	if {[info tclversion] < 8.0} {
	    set from "\\b$from\\b"
	} else {
	    set from "\\m$from\\M"
	}
    } 
    set savefrom [searchString]
    set to [replaceString]
    if {!$supersearch(r)} {
	set to [quote::Regsub $to]
	if {[catch {regsub -- $from "$savefrom" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}
	if {![llength {$dummy}]} {
	    alertnote "Regsub problem"
	    return
	}
    } else {
	if {[catch {regsub -- $from "$from" $to dummy} err]} {
	    alertnote "Regexp compilation problems: $err"
	    return
	}	
    }
    supersearch::ensureAllWindowsSaved
    
    supersearch::getFiles
    set supersearch(multiindex) -1
    set changes 0
    set i 0
    supersearch::createContext multiReplaceAll
    while {$i < $supersearch(numfiles)} {
	if {[info exists supersearch(multiindex)] \
	  && $supersearch(multiindex) != ""} {
	    set f [lindex $supersearch(files) [incr supersearch(multiindex)]]
	    if {$f == ""} {
		set supersearch(files) ""
		unset supersearch(multiindex)
		continue
	    } else {
		if {[file isdirectory $f]} {
		    incr i
		    continue
		}
		set remaining [expr {$supersearch(numfiles) \
		  - $supersearch(multiindex) -1}]
		if {[supersearch::_isInFile $f $remaining]} {
		    message "Modifying ${f}"
		    if {![catch [list alphaOpen $f "r"] cid]} {
			set tmp [read $cid]
			close $cid
			regsub -all "\n" $tmp "\r" tmp
			if {$supersearch(i)} {
			    set inc [regsub -all -nocase -- $from $tmp $to tmp]
			} else {
			    set inc [regsub -all -- $from $tmp $to tmp]
			}
			if {$inc > 0} {
			    if {![catch [list alphaOpen $f "w+"] ocid]} {
				puts -nonewline $ocid $tmp
				close $ocid
				set tmp ""
				incr changes $inc
				set matches($f) 1
			    } else {
				alertnote "Couldn't open $f with write\
				  permission!  Changes will not take place."
			    }
			}
		    } else {
			alertnote "Couldn't open $f with read permission!\
			  Changes will not take place."
		    }
		}
		incr i
	    }
	    continue
	}
    }
    supersearch::deleteContext
    
    eval file::revertThese [array names matches]
    message "Replaced $changes instances"
}



#  Search utilities  #
proc supersearch::getFiles {} {
    global supersearch
    set supersearch(files) ""
    foreach fset [supersearch::getfsets] {
	eval lappend supersearch(files) [getFileSet $fset]
    }
    set supersearch(numfiles) [llength $supersearch(files)]
    if {$supersearch(files) == ""} {
	dialog::errorAlert "You have selected an empty fileset.\
	  Please select another one.\r\
	  If you are trying to search multiple filesets, you must tell\
	  Alpha which ones to search by clicking on the\
	  'multiple filesets' button.  Once you've done that \
	  once, you can then use the checkbox to toggle\
	  single/multi for all subsequent searches."
    }
}

proc supersearch::ensureAllWindowsSaved {} {
    global win::NumDirty
    if {${win::NumDirty}} {
	if {[buttonAlert "Save all windows?" "Yes" "Cancel"] != "Yes"} {
	    error "Cancelled!"
	}
	saveAll
    }
}

proc supersearch::getfsets {} {
    global supersearch
    if {$supersearch(ismultifset)} {
	return $supersearch(multifsets)
    } else {
	return [list $supersearch(fileset)]
    }
}


#  Dialog box and helpers  #
proc supersearch::find {} {
    global supersearch searchPattern
    global supersearchmodeVars currFileSet
    set multimenu $supersearch(multi)
    # If there's no window, we don't want single file search
    if {$multimenu == 0 && ![llength [winNames -f]]} {
	set multimenu 1
    }
    if {!$supersearchmodeVars(separateSearchAndCurrentFileset) \
      || ![info exists supersearch(fileset)] \
      || ($supersearch(fileset) == "")} {
	set supersearch(fileset) $currFileSet
    }
    set loop 0
    while 1 {
	set haveWin [expr {[win::Current] != ""}]
	if {[info tclversion] < 8.0 && $loop} {
	    # set search and replace strings
	    set newQuote $supersearchmodeVars(quoteFunnyChars)
	    if {($oldQuote != $newQuote) || ($old != $str)} {
		if {$oldQuote} {
		    set str [quote::Undisplay $str]
		}
		if {[supersearch::updateStrings $str]} {return}
	    }
	}
	
	set y 5
	if {[info tclversion] < 8.0} {
	    set args [dialog::text "Search" 50 y]
	    incr y 5
	} else {
	    set args {}
	    lappend args -T "Search"
	    incr y 9
	}
	set yt $y
	if {$supersearchmodeVars(quoteFunnyChars)} {
	    eval lappend args [dialog::textedit "Search for:" \
	      [set supersearch(search) [quote::Display [searchString]]] \
	      20 y 27 $supersearchmodeVars(boxHeights)]
	    set y $yt
	    eval lappend args [dialog::textedit "Replace with:" \
	      [set supersearch(replace) [quote::Display [replaceString]]] \
	      320 y 27 $supersearchmodeVars(boxHeights)]
	} else {
	    eval lappend args [dialog::textedit "Search for:" \
	      [set supersearch(search) [searchString]] \
	      20 y 27 $supersearchmodeVars(boxHeights)]
	    set y $yt
	    eval lappend args [dialog::textedit "Replace with:" \
	      [set supersearch(replace) [replaceString]] \
	      320 y 27 $supersearchmodeVars(boxHeights)]
	}
	incr y 13
	set yr $y
	eval lappend args \
	  [dialog::checkbox "Ignore Case" $supersearch(i) 20 y] \
	  [dialog::checkbox "Regexp" $supersearch(r) 20 y] \
	  [dialog::checkbox "Batch" $supersearch(b) 20 y]
	set my $y
	set y $yr
	eval lappend args \
	  [dialog::checkbox "Word Match" $supersearch(m) 122 y] \
	  [dialog::checkbox "Keep capitals" $supersearch(casereplace) 122 y] \
	  [dialog::checkbox "Ignore spaces" $supersearch(ignore) 122 y]
	set y $my
	incr y 10
	eval lappend args [dialog::button "Search" 20 y "Don't Search" 90 y]
	# 10 pixel vertical spacing between buttons (we get 6 automatically)
	incr y 4
	eval lappend args [dialog::button "Replace All" 20 y "Cancel" 125 y]
	set ym [expr {$y + 10}]
	if {[info tclversion] < 8.0} {
	    set y 10
	} else {
	    set y 4
	}
	eval lappend args [dialog::button "prefs" 535 y]
	# single/multi files
	set y $yr
	set x 245
	global alpha::platform
	# Only Alphatk supports disabled items right now
	if {$haveWin || (${alpha::platform} != "tk")} {
	    set justOne "Top window"
	} else {
	    set justOne "\(Top window"
	}
	set ftypes [list $justOne "Multiple files" "Patterns"]
	eval lappend args [dialog::text "Options:" $x y]
	set yr2 $y
	eval lappend args \
	  [dialog::menu $x y $ftypes [lindex $ftypes $multimenu]] 
	set yr $y
	incr y 8
	if {$supersearchmodeVars(smartInSelectionFlag)} {
	    if {([win::Current] != "") && ([string length [getSelect]] - \
	      [string length $supersearch(search)] > 80)} {
		set select 1
	    } else {
		set select 0
	    }
	} else {
	    set select $supersearch(inselection)
	}
	lappend args -n $justOne
	
	# If there's no window/no selection move irrelevant checkboxes offscreen.
	eval lappend args \
	  [dialog::checkbox "From current pos" $supersearch(f) \
	  [expr {$haveWin ? $x : 3000}] y] \
	  [dialog::checkbox "Wrap " $supersearch(wrap) \
	  [expr {$haveWin ? $x : 3000}] y] \
	  [dialog::checkbox "In selection only" $select \
	  [expr {$haveWin && ([string length [getSelect]] > 0) ? $x : 3000}] y] \
	  [list -n "Multiple files"]
	set y $yr
	incr y 2
	eval lappend args \
	  [dialog::text "Select a single fileset" $x y]
	incr x 10
	# align button with text 
	set y $yr
	eval lappend args \
	  [dialog::menu [expr {$x +140}] y \
	  [lsort -ignore [fileset::names]] \
	  $supersearch(fileset)]
	set yr3 $y
	incr y 2
	set delx 0
	if {[llength $supersearch(multifsets)]} {
	    eval lappend args \
	      [dialog::checkbox "or pick" $supersearch(ismultifset) $x y]
	    incr x 12
	    incr delx -11
	} else {
	    # we don't want the check box, so we move it way offscreen.
	    eval lappend args \
	      [dialog::text "Or pick" $x y]
	    set y $yr3
	    eval lappend args \
	      [dialog::checkbox "or pick" $supersearch(ismultifset) \
	      [expr {$x -1000}] y]
	    incr x
	}
	set y $yr3
	eval lappend args \
	  [dialog::button "Multiple Filesets:" [expr {$x +55}] y] \
	  [dialog::text [join $supersearch(multifsets) ", "] $x y 50]
	set y $yr2
	eval lappend args \
	  [dialog::button "New fileset" [expr {$x +139 + $delx}] y \
	  "Dir scan" [expr {$x +264 + $delx}] y]
	set y $yr
	eval lappend args \
	  [list -n "Patterns"] \
	  [dialog::button "Save this pattern" $x y] \
	  [dialog::menu $x y [array names searchPattern] $supersearch(pattern)] 
	eval lappend args \
	  [dialog::button "Use pattern from menu" $x y]
	incr y 30
	if {[info tclversion] >= 8.0} {
	    lappend args -help [list "Enter the search string here" \
	      "Enter the replace string here" \
	      "To ignore upper/lower case differences when searching,\
	      click this box.||To use exact, case-sensitive matches only,\
	      click this box" \
	      "To interpret the above search string as a regular expression,\
	      click this box.||To match the above search string literally,\
	      click this box." \
	      "To display all possible matches in a separate control window,\
	      click this box.||To highlight each match individually as it is\
	      found, click this box." \
	      "To match only entire words,\
	      click this box.||To allow any sequence of characters which matches\
	      the above search string, click this box." \
	      "To try to match the case of leading replacement character\
	      with the leading found character,\
	      click this box.||To always replace with the exact, given\
	      replacement string, click this box" \
	      "To ignore whitespace differences when searching,\
	      click this box.||To use exact, character and space matching only,\
	      click this box" \
	      "Click here to begin the search" \
	      "Click here to remember my settings, but not actually search" \
	      "Click here to replace all matches at once without my intervention" \
	      "Click here to discard any changes you've made to the settings." \
	      "Click here to edit the supersearch preferences." \
	      "More search facilities are accessible from this menu." \
	      "To start the search at the current cursor position,\
	      click this box.||To search from the beginning of the document,\
	      click this box" \
	      "To continue searching from the beginning after reaching the end\
	      of the document, click this box.||To stop searching at the end\
	      of the document, click this box." \
	      "To search just in the current highlighted text,\
	      click this box||To search the entire text, click this box." \
	      "Select the fileset in which to search." \
	      "To search more than one fileset, use the 'Multiple filesets'\
	      button and click this box||To search just a single fileset,\
	      click this box." \
	      "Click here to select more than one fileset in which to search." \
	      "Click here to create a new fileset" \
	      "Click here to scan a particular directory." \
	      "Save the search and replace strings above for future use." \
	      "Select a pattern from this menu and then click the button below." \
	      "Use the pattern currently selected in this popup menu." \
	      ]
	}
	set res [eval dialog -w 610 -h $ym $args]
	set loop 1
	if {[info tclversion] < 8.0} {
	    set oldQuote $supersearchmodeVars(quoteFunnyChars)
	    if {$supersearchmodeVars(quoteFunnyChars)} {
		set old "\{[quote::Display [searchString]]\} \{[quote::Display [replaceString]]\}"
	    } else {
		set old "\{[searchString]\} \{[replaceString]\}"
	    }
	    # parse res, cor contains the flags and str contains the search 
	    # and replace strings enclosed between braces: 
	    # cor="{} {} flags"
	    # str="{search} {replace}"
	    if {[supersearch::parse $res cor str]} {return}
	    set res $cor
	}
	
	if {[lindex $res 11]} {
	    return
	}
	if {[info tclversion] < 8.0} {
	    set i 2
	    foreach text {i r b m casereplace ignore} {
		set supersearch($text) [lindex $res $i]
		incr i
	    }
	    # set search and replace strings
# 	    if {$old != $str} {
# 		if {[supersearch::updateStrings $str]} {return}
# 	    }
	} else {
	    # turn on other flags if any
	    set i 0
	    foreach text {search replace i r b m casereplace ignore} {
		set supersearch($text) [lindex $res $i]
		incr i
	    }
	}
    
	if {$supersearchmodeVars(quoteFunnyChars)} {
	    searchString [quote::Undisplay $supersearch(search)]
	    replaceString [quote::Undisplay $supersearch(replace)]
	} else {
	    searchString $supersearch(search)
	    replaceString $supersearch(replace)
	}
	
	# get buttons
	set j $i
	foreach but {search dontSearch replaceAll cancel prefs} {
	    if {[lindex $res $j]} {
		set button $but
		break
	    }
	    incr j
	}
	
	incr i 5
	if {[lindex $res $i] == "Top window"} {
	    set multimenu [set supersearch(multi) 0]
	    set supersearch(f) [lindex $res [incr i]]
	    set supersearch(wrap) [lindex $res [incr i]]
	    set supersearch(inselection) [lindex $res [incr i]]
	    if {$supersearch(inselection)} {
		if {[pos::compare [getPos] <= [selEnd]]} {
		    set supersearch(start) [getPos]
		    set supersearch(end) [selEnd]
		} else {
		    set supersearch(start) [selEnd]
		    set supersearch(end) [getPos]
		}
	    } 
	    incr i 6
	} elseif {[lindex $res $i] == "Multiple files"} {
	    set multimenu [set supersearch(multi) 1]
	    set supersearch(f) [lindex $res [incr i]]
	    set supersearch(wrap) [lindex $res [incr i]]
	    set supersearch(inselection) [lindex $res [incr i]]
	    
	    # ignore multi-fset for the moment
	    set supersearch(fileset) [lindex $res [incr i]]
	    set supersearch(ismultifset) [lindex $res [incr i]]
	    
	    if {[lindex $res [incr i]]} {supersearch::multifset ; continue }
	    if {[lindex $res [incr i]]} {supersearch::newfset ; continue }
	    if {[lindex $res [incr i]]} {supersearch::dirscan ; continue }
	} else {
	    incr i 9
	    set multimenu 2
	    set supersearch(pattern) [lindex $res [expr {$i +1}]]
	    if {[lindex $res $i]} {
		supersearch::storePattern
		continue
	    }
	    if {[lindex $res [incr i 2]]} {supersearch::usePattern ; continue}
	    incr i
	}
	
	if {[info exists button] && $button == "prefs"} {
	    supersearch::prefs
	    unset button
	    continue
	}
	
	if {[info exists button]} {
	    if {$button == "search"} { 
		set button "generalSearch" 
	    }
	    if {$button == "replaceAll"} { 
		set button "reallyReplaceAll" 
	    }
	    if {[info tclversion] < 8.0} {
		# set search and replace strings
		if {($old != $str)} {
		    if {$oldQuote} {
			set str [quote::Undisplay $str]
		    }
		    if {[supersearch::updateStrings $str]} {return}
		}
	    }
	    supersearch::parseSearch
	    supersearch::$button
	    return
# 	    if {$button != "prefs"} {return}
	}
	
	set loop 0
	# big while loop
    }
}

if {[info tclversion] < 8.0} {
    
    ## 
     #	This proc search backward for the following pattern:
     #		
     #	'{s1} {s2} f f f f f f f f f f {s3} f {s4} f f f f f {s5} f '
     #	
     #		
     #	or '{s1} {s2} f f f f f f f f f f f {s3} f {s4} f f f f f {s5} f '
     #	it the flag 'Ignore Spaces' is used.
     #	
     #	The only assumptions are: s3, s4 and s5 are strings which do not
     #	contain an opening brace ("\{") and there are 10 binary flags
     #	between s2 and s3.  The proc returns in st the original string
     #	after replacing s1 and s2 by empty strings and "{s1} {s2}" in hd. 
     #	If s1 and s2 do not contain the string "\} \{" the proc
     #	supersearch::updateStrings sets silently the search and replace
     #	strings, otherwise it asks the user to set the boundary between the
     #	two strings.
     #		
     #	The use of prompt::var requires to use the useStatusBar and
     #	useStatusBar flags defined in this version as newPref flags.  This
     #	is not working properly.  The segmentation between search and
     #	replace string could be made easier by using the Pierre Basso idea
     #	of using marks.  His idea of inserting the replace string instead
     #	of replacing it could also be implemented.
     ##
    proc supersearch::parse {pat {st ""} {hd ""}} {
	if { $st != "" } {
	    upvar $st cor
	}
	if { $hd != "" } {
	    upvar $hd beg
	}
	set a [string last "\{" $pat]
	set new [string range $pat 0 [expr {$a - 1}]]
	set a [string last "\{" $new]
	set new [string range $new 0 [expr {$a - 1}]]
	set a [string last "\{" $new]
	set new [string range $new 0 [expr {$a - 24}]]
	set len [string length $new]
	set tail [string range $pat $len end]
	set cor "\{\} \{\}$tail"
	set beg $new
	return 0
    }
    
    proc supersearch::updateStrings {str} {
	global supersearch buffersearch bufferreplace
	set new $str
	set count [regsub -all "\} \{" $new "\}\{" withBullets]
	set len [string length $new]
	if {$count > 0} {
	    set a [string last "\} \{" $new]
	    if {$count >1} {
		set tmp $new
		set c $count
		while {$c > 0} {
		    set from [string range $new 1 [expr {$a - 1}]]
		    set to [string range $new [expr {$a + 3}] [expr {$len - 2}]]
		    set fl [string length $from]
		    set tl [string length $to]
		    if {$tl < 20} {
			set fl [expr {40 - $tl}]
		    } elseif {$fl < 20} {
			set tl [expr {40 - $fl}]
		    } else {
			set fl 20
			set tl 20
		    }  
		    set from [shorten $from $fl 8]
		    set to [shorten $to $tl 8]
		    lappend items "\"$from\" and \"$to\""
		    set tmp [string range $tmp 0 [expr {$a - 1}]]
		    set a [string last "\} \{" $tmp]
		    incr c -1
		}
		set item [listpick -p "Select the pair of strings you want." $items]
		set n [expr {$count - [lsearch -exact $items $item]}]
		set a [string last "\} \{" $new]
		if {$count < $n } {return 1}
		set tmp $new
		while {$count != $n} {
		    set tmp [string range $tmp 0 [expr {$a - 1}]]
		    set a [string last "\} \{" $tmp]
		    incr count -1
		}
	    }
	    set from [string range $new 1 [expr {$a - 1}]]
	    set buffersearch $from
	    set supersearch(search) $from
	    set to [string range $new [expr {$a + 3}] [expr {$len - 2}]]
	    set bufferreplace $to
	    set supersearch(replace) $to
	    return 0
	} else {
	    return 1
	}
    }
}

proc supersearch::prefs {} {
    catch {dialog::pkg_options supersearch}
}

proc supersearch::usePattern {} {
    global supersearch searchPattern
    set pats $searchPattern($supersearch(pattern))
    searchString [lindex $pats 0]
    replaceString [lindex $pats 1]
}

proc supersearch::storePattern {} {
    global supersearch searchPattern
    if {[catch {set name [prompt "New pattern's name?" ""]}]} {
	return ""
    }
    prefs::modified searchPattern($name)
    set searchPattern($name) [list [searchString] [replaceString]]
    return $name
}

set {searchPattern(C++ to C Comments)} {{//(.*)} {/* \1 */}}

proc supersearch::dontSearch {} {}

proc supersearch::multifset {} {
    global supersearch
    if {![catch {listpick -p "Select filesets" -l -L $supersearch(multifsets) \
      [lsort -ignore [fileset::names]]} res]} {
	if {[llength $res]} {
	    set supersearch(multifsets) $res
	    set supersearch(ismultifset) 1
	} else {
	    set supersearch(multifsets) ""
	    set supersearch(ismultifset) 0
	} 
    }
}

proc supersearch::dirscan {} {
    global supersearch

    set supersearch(ismultifsets) 0
    if {[llength $supersearch(multifsets)] != 0} {
	set supersearch(multifsets) ""
    }
    
    set dir [get_directory -p "Scan which folder?"]
    if {![string length $dir]} return
    
    set name [file tail $dir]
    fileset::fromDirectory::create $name $dir *
    registerNewFileset $name fromDirectory
    updateCurrentFileset
    set supersearch(fileset) $name
}

proc supersearch::findNewDirectory {} {
    set dir [get_directory -p "Scan which folder?"]
    if {![string length $dir]} return
    
    set name [file tail $dir]
    fileset::fromDirectory::create $name $dir *

    registerNewFileset $name fromDirectory
    updateCurrentFileset
    return $name
}

proc supersearch::newfset {} {
    if {[catch newFileset]} return
    global supersearch currFileSet
    set supersearch(fileset) $currFileSet
}

proc shorten {str {len 40} {el 0}} {
    if {[set sl [string length $str]] > $len} {
	set hl [expr {$len - $el - 2}]
	set str "[string range $str 0 $hl][string range $str [expr {$sl - $el}] end]"
    }
    return $str
}

proc supersearch::parseSearch {args} {
    global supersearch supersearchmodeVars
    switch -- [llength $args] {
	0 {
	    set from [searchString]
	}
	1 {
	    set from [lindex $args 0]
	}
	default {
	    error "Too many args"
	}
    }
    set values 0
    set rcList {}
    set tmp $from
    set mlines [regexp -indices "\\r" $tmp values]
    while {$mlines == 1} {
	set rc [lindex $values 0]
	lappend rcList $rc
	set tmp [string range $tmp [expr {$rc + 1}] end]
	set mlines [regexp -indices "\\r" $tmp values]
    }
    set pre ""
    set post ""
    set rcFirst 0
    set regTail ""
    if {[llength  $rcList] > 0} {
	set rc [lindex  $rcList 0]
	set post "\$"
	if {$supersearch(ignore)} {set post "\[ \t\]*$post"}
	if {$rc == 0} {
	    set pre "^"
	    if {$supersearch(ignore)} {set pre "$pre\[ \t\]*"}
	    if {[llength  $rcList] > 1} {
		set rc [lindex  $rcList 1]
		if {$rc == 0} {
		    set part ""
		} else { 
		    set part [string range $from 1 $rc]
		}
		set rcFirst 2
		set regTail [string range $from [expr {$rc + 2}] end]
	    } else { 
		set part [string range $from 1 end]
		set post ""
	    }
	} else { 
	    set part [string range $from 0 [expr {$rc - 1}]]
	    set rcFirst 1
	    set regTail [string range $from [expr {$rc + 1}] end]
	}
	set supersearch(singleline) 0
    } else { 
	set part $from
	set supersearch(singleline) 1
    }
    
    set part [supersearch::prepLine $part]
    set tmp $from
    if {!$supersearch(r)} {
	if {$supersearch(ignore)} {
	    set chr $supersearchmodeVars(checkSpacesAround)
	    set tmpf [regsub -all "\[ \t\]*($chr)\[ \t\]*" $tmp " \\1 " tmp]
	    set tmpf [regsub -- "^ " $tmp "" tmp]
	    set tmpf [regsub -- " $" $tmp "" tmp]
	    set tmp [quote::Regfind $tmp]
	    set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	} else {
	    set tmp [quote::Regfind $tmp]
        }
    } elseif {$supersearch(ignore)} {
	regsub -all {(\[([^\\]|\\[^r])*)\\r([^]]*\])} $tmp "\\1\\3" tmp
	regsub -all {(\[([^\\]|\\[^t])*)\\t([^]]*\])} $tmp "\\1\\3" tmp
	regsub -all {(\[([^\\]|\\[^n])*)\\n([^]]*\])} $tmp "\\1\\3" tmp
	regsub -all {(\[[^ ]*) ([^]]*\])} $tmp "\\1\\2" tmp
	regsub -all {(\[[^ ]*)	([^]]*\])} $tmp "\\1\\2" tmp
	set tmpf [regsub -all "\[ \t\]*\r\[ \t\]*" $tmp " \r " tmp]
	set tmpf [regsub -- "^ " $tmp "" tmp]
	set tmpf [regsub -- " $" $tmp "" tmp]
	set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1\\r\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1\\t\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1\\n\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1 \\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1	\\3" tmp	
    }
    set supersearch(rcList) $rcList
    set supersearch(rcFirst) $rcFirst
    set supersearch(firstLine) $pre$part$post
    set supersearch(regTail) $regTail
    set supersearch(reg) $tmp
}

proc supersearch::prepLine {str} {
    global supersearch supersearchmodeVars
    set tmp $str
    if {!$supersearch(r)} {
	set tmp [quote::Regfind $str]
	if {$supersearch(ignore)} {
	    set chr $supersearchmodeVars(checkSpacesAround)
	    set tmpf [regsub -all "\[ \t\]*($chr)\[ \t\]*" $str " \\1 " tmp]
	    set tmp "[string trim $tmp]"
	    set tmp [quote::Regfind $tmp]
	    set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	}
    } elseif {$supersearch(ignore)} {
	regsub -all {(\[[^ ]*) ([^]]*\])} $tmp "\\1\\2" tmp
	regsub -all {(\[[^	]*)	([^]]*\])} $tmp "\\1\\2" tmp
	set tmpf [regsub -all "\[ \t\]+" $tmp "\[ \t\]*" tmp]
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1	\\3" tmp	
	regsub -all {(\[([^]*|[^]*|[^]*|[^]*))([^]]*\])} $tmp "\\1 \\3" tmp	
    }
    return [supersearch::regPart $tmp] 
}

