#  AlphaTcl - core Tcl engine

namespace eval win {}

# Similar to 'file::openQuietly' but will deal with
# windows which don't exist on disk.
proc win::OpenQuietly {name} {
    if {[catch {edit -c -w $name}]} {
	bringToFront $name
    }
    if {[icon -q]} {icon -o}
}

# Take a string and turn it into an acceptable window
# title.  This may involve removing illegal characters
# or shortening the string
if {${alpha::platform} == "alpha"} {
    proc win::MakeTitle {str} {
	if {[string length $str] > 31} {
	    set str "[string range $str 0 30]"
	}
	return $str
    }
} else {
    # Alphatk is happy with much longer titles
    # (so may Alpha 8, actually, but we haven't changed that).
    proc win::MakeTitle {str} {
	if {[string length $str] > 51} {
	    set str "[string range $str 0 50]"
	}
	return $str
    }
}

proc win::IsFile {name} {
    return [expr {[file exists $name] || \
      ([regsub { <[0-9]+>$} $name {} name] && [file exists $name])}]
}

proc win::Current {} {global win::Current ; return ${win::Current}}
proc win::CurrentTail {} {
    global win::Current ; return [file tail ${win::Current}]
}
proc win::TopNonProcessWindow {} {
    global win::Active
    foreach f [set win::Active] {
	if {![regexp {^\* .* \*( <[0-9]+>)?$} $f]} {
	    return $f
	}
    }
    return ""
}
proc win::TopFileWindow {} {
    global win::Active
    foreach f [set win::Active] {
	if {[file exists [win::StripCount $f]]} {
	    return $f
	}
    }
    return ""
}

proc win::StripCount {name} {
    regsub { <[0-9]+>} $name {} name
    return $name
}

# Find the count string required for a window with the given
# name.  If it is a file window, we should only be passed the
# tail of the name.
proc win::CountFor {name {excludeCurrent 0}} {
    if {$excludeCurrent} {
	set names [lrange [winNames] 1 end]
    } else {
	set names [winNames]
    }
    
    if {[lsearch -exact $names $name] != -1} {
	set num 2
	while {[lsearch -exact $names "$name <$num>"] != -1} { 
	    incr num 
	}
	return " <$num>"
    }
    return ""
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "win::setMode" --
 #	
 #  Copes with endings like '.orig' or the backup ending '~' or '
 #  copy'.
 #
 # -------------------------------------------------------------------------
 ##
proc win::setMode name {
    global win::Modes
    set win::Modes($name) [win::FindMode $name]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "win::setInitialMode" --
 # 
 #  Only to be called the very first time we set a window's mode, since
 #  it may have all sorts of side-effects.
 # -------------------------------------------------------------------------
 ##
proc win::setInitialMode {winname mode} {
    global win::Modes
    set win::Modes($winname) $mode
    
    # We need to _load_ the mode now (not 'changeMode'), but simply
    # load it all in.
    loadAMode $mode
    
    global tabSize ${mode}modeVars
    if {[info exists ${mode}modeVars(tabSize)]} {
	# The mode that the new window will open up in
	# has its own value for tabSize
	win::setInitialConfig $winname tabsize [set ${mode}modeVars(tabSize)]]
    }

    # If someone wants a hook here for some reason, please ask!
    #hook::callAll initialModeSetHook $mode $winname
}

proc win::setInitialConfig {winname option value} {
    global win::config
    lappend win::config($winname) [list $option $value]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "win::FindMode" --
 # 
 #  Copes with trailing '<2>', .orig, copy, '~',...
 # -------------------------------------------------------------------------
 ##
proc win::FindMode {name} {
    global ModeSuffixes
    regsub { <[0-9]+>$} $name "" nm
    regsub {( copy|~[0-9]*|.orig|.in)+$} [file tail $nm] "" nm
    case $nm in $ModeSuffixes
    return $winMode
}

if {[info tclversion] < 8.0} {
    proc win::Encoding {args} {
	switch -- [llength $args] {
	    0 {
		return "macRoman"
	    }
	    1 {
		# encoding of 'name = [lindex $args 0]'
		return "macRoman"
	    }
	    2 {
		# set encoding of [lindex $args 0] to [lindex $args 1]
		# not implemented in Alpha 7 or 8 yet.
		return ""
	    }
	    default {
		error "Wrong number of arguments"
	    }
	}
    }
} else {
    # Will be used by Alpha 8, and is shared with Alphatk.  If this
    # is broken or modified, Alphatk will cease to work.
    proc win::Encoding {args} {
	switch -- [llength $args] {
	    0 {
		return [win::getInfo [win::Current] encoding]
	    }
	    1 - 
	    2 {
		set n [lindex $args 0]
		set enc [lindex $args 1]
		set oldenc [win::getInfo $n encoding]
		if {$oldenc != $enc} {
		    setWinInfo -w $n encoding $enc
		    if {[win::getInfo $n dirty] || ($oldenc == "") \
		      || ![win::IsFile $n]} {
			message "'$enc' encoding now associated with this window."
		    } else {
			if {[dialog::yesno "Reread from disk?"]} {
			    ::revert $n
			    message "Synchronised with version on disk;\
			      using new $enc encoding."
			} else {
			    message "'$enc' encoding now associated with\
			      this window."
			}
		    }
		    displayEncoding $enc
		}
	    }
	    default {
		error "Wrong number of arguments"
	    }
	}
    }
}

proc win::cursorInWindow {{pos ""}} {
    if {![string length $pos]} {set pos [getPos]}
    getWinInfo winArray
    set top   $winArray(currline)
    set lines $winArray(linesdisp)
    # This is the top of the window.
    set pos0  [rowColToPos $top 0]
    # This is the bottom of the window.
    set pos1  [nextLineStart [rowColToPos [expr {$top + $lines}] 0]]
    # Find out if the cursor is somewhere in the window.
    set test0 [pos::compare $pos < $pos0]
    set test1 [pos::compare $pos >= [nextLineStart $pos1]]
    set isIn  [expr {$test0 || $test1} ? 0 : 1]
    return    [list $isIn $pos0 $pos1]
}

## 
 # ----------------------------------------------------------------------
 #	 
 #  "win::searchAndHyperise" --
 #	
 #  Scans through an entire file for a particular string or regexp, and
 #  attaches a hyperlink of the specified form (regsub'ed if desired)
 #  to the original string.
 #			
 #	Side effects:
 #	 Many hyperlinks will be embedded in your file
 #	
 #	Arguments:
 #	 Look for 'text', replace with 'link', doing both with a regexp
 #	 if signified (regexp = 1), using colour 'col', and offsetting
 #	 the link start and end by 'startoff' and 'endoff' respectively.
 #	 This last bit is so you can search for a large pattern, but only
 #	 embed a link in a smaller part of it.
 #	 
 #	Examples: 
 #	 see 'proc install::hyperiseUrls'
 # ----------------------------------------------------------------------
 ##
proc win::searchAndHyperise {text link {regexp 0} {col 3} {startoff 0} {endoff 0}} {
    set pos [minPos]
    catch {
	while 1 {
	    set inds [search -s -f 1 -r $regexp -- $text $pos]
	    set from [lindex $inds 0]
	    set to [lindex $inds 1]
	    set realfrom $from
	    set realto $to
	    set realfrom [pos::math $realfrom + $startoff]
	    set realto [pos::math $realto + $endoff]
	    text::color $realfrom $realto $col
	    if {$link != ""} {
		if {$regexp} {
		    regsub -- $text [getText $from $to] "$link" llink
		} else {
		    set llink $link
		}
		# hack to handle some links.
		regsub -- "<<" $llink "" llink
		regsub -- ">>" $llink "" llink
		if {[pos::diff $realfrom $realto] < 100} {
		    text::hyper $realfrom $realto $llink
		} else {
		    # Should turn this into an error in the future.
		    message "Tried to mark very large hyper."
		}
	    }
	    set pos $to
	}	
    }
    # The calling procedure should 'refresh' now.
}
proc win::multiSearchAndHyperise {args} {
    while 1 {
	set text [lindex $args 0]
	set link [lindex $args 1]
	set args [lrange $args 2 end]
	if {$text == ""} {return}
	win::searchAndHyperise $text $link
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "win::jumpToCode" --
 # 
 #  It creates a hyperlink to a specific string in a code file, without
 #  requiring a mark to be defined there. It was handy for identifying places 
 #  in other packages that potentially collide with my key-bindings.
 #  
 #  Author: Jon Guyer.
 # -------------------------------------------------------------------------
 ##
proc win::jumpToCode {text file code} {
    set hyper {edit -c }
    append hyper $file
    append hyper { ; set pos [search -s -f 1 -r 1 "}
    append hyper $code
    append hyper {"] ; select [lindex $pos 0] [lindex $pos 1]}
    win::searchAndHyperise $text $hyper 0 3
}

