## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "www.tcl"
 #                                    created: 01-10-03 19.48.58 
 #                                last update: 10/29/2001 {13:45:55 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  4/9/97   VMD 1.0 original
 # ###################################################################
 ##

proc forceLoadAE {} {
    return
    # Comment out the above line if your machine is very slow.
    alpha::package require tclAE
    global ALPHA HOME
    catch {makeAlis [file join $HOME $ALPHA]}
}

#  Handling html, url, mailto actions  #

proc htmlView {filename} {
    global viewHtmlUsing alpha::platform browserSig htmlViewer
    if {${alpha::platform} == "alpha"} {
	# Is the browserSig set yet?
	if {$browserSig == ""} {
	    app::getSig "Please locate your browser :" browserSig
	} 
	# Make sure that the .html file is of type $browserSig. 
	# We also need to delete any resource fork to ensure
	# that it gets sent to the browser, not Alpha.
	setFileInfo $filename type TEXT
	setFileInfo $filename creator $browserSig
	setFileInfo $filename resourcelen
    }
    eval $htmlViewer($viewHtmlUsing) [list $filename]
}

proc htmlHelpViewer {filename} {
    global tcl_platform
    if {!(($tcl_platform(platform) == "macintosh")||($tcl_platform(platform) == "unix" && $tcl_platform(os) == "Darwin")) || [catch {nameFromAppl hbwr}]} {
	# This shouldn't have been presented as an option ...
	alertnote "Sorry, the MacOS Help viewer could not be found."
	error "The MacOS Help Viewer could not be found."
    }
    app::launchBack hbwr
    sendOpenEvent noReply 'hbwr' $filename
    switchTo 'hbwr'
}

proc urlView {url} {
    global viewUrlsUsing alpha::platform browserSig urlViewer
    if {${alpha::platform} == "alpha"} {
	# Is the browserSig set yet?
	if {$browserSig == ""} {
	    app::getSig "Please locate your browser :" browserSig
	} 
    }
    eval $urlViewer($viewUrlsUsing) [list $url]
}

proc urlDefaultViewer {url {title ""}} {
    status::msg "'$url' sent to browser."
    url::execute $url
}

proc composeEmail {to} {
    global composeEmailUsing eMailer
    # Note that it is up to the calling proc to ensure that 'to' is in the
    # proper format, i.e. by first using 'url::mailto'
    eval $eMailer($composeEmailUsing) [list $to]
}

proc emailDefaultComposer {to} {
    # A little insurance to ensure that 'to' starts with 'mail::to'
    regsub {mailto:} $to {} to
    set to "mailto:$to"
    status::msg "'$to' sent to browser."
    url::execute $to
}


#  Basic url handling  #

namespace eval url {}

## 
 # -------------------------------------------------------------------------
 # 
 # "url::mailto" --
 # 
 #  Generate a mailto url from the given argument pairs.  You can then
 #  pass the result to 'url::execute' to take action.  Note that very
 #  long mailto urls seem not to be handled properly, so you may wish
 #  to check the length of the 'body' field, if given and take a different
 #  action (e.g. put the body on the clip board for the user to handle
 #  manually).
 #  
 #  A typical use is:
 #  
 #  url::execute [url::mailto vince@santafe.edu subject hello body goodbye]
 # -------------------------------------------------------------------------
 ##
proc url::mailto {address args} {
    set url "mailto:$address"
    set divider "?"
    newforeach {arg value} $args {
	append url $divider $arg = [quote::Url $value]
	set divider "&"
    }
    return $url
}

# This should carry out the default action of opening/clicking-on
# a url
proc url::execute {url} {
    icURL $url
}

# For urls which ought to be downloaded (e.g. files), this
# procedure will try to carry that out in preference to opening.
proc url::download {url} {
    global downloadFolder
    if {![file exists $downloadFolder] || ![file isdirectory $downloadFolder]} {
	global HOME
	alertnote "Your Download Folder does not exist.\
	  I'll download to Alpha's home directory."
	set downloadFolder $HOME
    }
    url::fetch $url $downloadFolder
}

# This is the same as url::download, except if we're given a directory
# url.  In that case we find a listing of that directory and then
# ask the user for a file in that directory.
proc url::getAFile {url} {
    while {1} {
	while {[url::isDirectory $url]} {
	    set url [url::pickFromDirectory $url]
	}
	if {[catch [list url::download $url] res]} {
	    alertnote "Fetch error '$res'"
	    error ""
	}
	set type [lindex $res 0]
	set name [lindex $res 1]
	
	# Check if it really was a directory without a trailing '/'
	if {[url::downloadedDirectoryListing $type $name]} {
	    set url [url::pickFromDirectory $url [file::readAll $name]]
	    file delete $name
	} else {
	    break
	}
    }
    return $res
}

# Optional argument is the contents of the dirurl if it has
# been previously downloaded (perhaps by mistake).
proc url::pickFromDirectory {dirurl {contents ""}} {
    if {![regexp {/$} $dirurl]} { append dirurl "/" }
    set listing [url::directoryListing $dirurl $contents]

    set names [list]
    foreach u $listing {
	lappend names [lindex $u 0]
    }
    set filechoice [listpick -p "Pick a file to install" $names]
    set index [lsearch -exact $names $filechoice]
    if {$index < 0} {
	return -code error "User selected a file I can't find in the list!"
    }
    set result [lindex [lindex $listing $index] 1]
    return $result
}

proc url::downloadedDirectoryListing {type name} {
    switch -- $type {
	"http" {
	    # Check if "<TITLE>Index of " is in the first 10 lines.
	    set fin [alphaOpen $name r]
	    for {set i 0} {$i < 10} {incr i} {
		if {[eof $fin]} { break }
		gets $fin line
		if {[regexp {<TITLE>Index of } $line]} {
		    close $fin
		    return 1
		}
	    }
	    close $fin
	}
    }
    return 0
}

# This works for anything 'url::fetch' can handle.
proc url::contents {url} {
    # Fetch the url contents into a temporary file
    temp::cleanup _urltmp
    set _tmp [temp::unique _urltmp tmp]
    url::fetch $url $_tmp
    
    # Get the contents of the url
    set fd [alphaOpen $_tmp "r"]
    set err [catch {read $fd} contents]
    close $fd
    
    if {$err} {
	return -code error $contents
    }
    return $contents
}

proc url::directoryListing {url {urlContents ""}} {
    if {![url::isDirectory $url]} {
	return -code error "\"$url\" is not a directory"
    }
    set t [url::parse $url]
    set type [lindex $t 0]
    set rest [lindex $t 1]
    
    # Should return list of sublists where each sublist
    # is "name url date ...".  This is used by install::fromRemoteUrl
    # amongst other places
    switch -- $type {
	"ftp" {
	    if {![string length $urlContents]} {
		set urlContents [url::contents $url]
	    }
	    set lines [split $urlContents "\n\r"]
	    set files {}
	    foreach f [lrange [lreplace $lines end end] 1 end] {
		set nm [lindex $f end]
		if {[string length $nm]} {
		    if {[string match "d*" $f]} {
			#lappend files "$nm/"
		    } else {
			regexp {[A-Z].*$} [lreplace $f end end] time
			set date [lindex $time end]
			if {[regexp : $date] || ![regexp {^19[89][0-5]$} $date]} {
			    # reject anything pre 1996
			    lappend files [list $nm $url$nm $time]
			}
		    }
		}
	    }
	    return $files
	}
	"file" {
	    set filename [string range $rest 1 end]
	    set files [list]
	    foreach f [glob -dir $filename *] {
		lappend files [list [file tail $f] [file::toUrl $f]]
	    }
	    return $files
	}
	"http" {
	    if {![string length $urlContents]} {
		set urlContents [url::contents $url]
	    }
	    set lines [split $urlContents "\n\r"]
	    set files {}
	    foreach f $lines {
		if {[regexp "<A (.*)<A " $f]} {
		    # lines with two links are ignored
		    continue
		}
		if {[regexp "<A HREF=\"(\[^\"\]*)\">\[^<\]*</A>\[ \t\]*(\[^ \t\]+)\[ \t\]" $f "" name date]} {
		    if {![regexp {/$} $name]} {
			if {![regexp {[89][0-5]$} $date]} {
			    # reject anything pre 1996
			    set date [split $date -]
			    set md "[lindex $date 1] [lindex $date 0] "
			    append md [expr {[lindex $date 2] < 80 ? 20 : 19}]
			    append md [lindex $date 2]
			    lappend files [list $name $url$name $md]
			}
		    }
		}
	    }
	    return $files
	}
	default {
	    alertnote "Don't know how to list '$type' url directories"
	    return -code error "Don't know how to list '$type' url directories"
	}
    }
}

# Should we perhaps check for 'index.html' urls?  If so then the
# above procedure should also be modified (we couldn't just get
# a file's url by appending the name onto the 'directory').
proc url::isDirectory {url} {
    return [regexp "/\$" $url]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "url::makeAbsolute" --
 # 
 #  Given a 'base' url and a potentially relative url, resolve the final
 #  absolute url.  It is up to any calling code to ensure that the base url
 #  is in proper format, i.e. contains a trailing '/' or 'index.html' if
 #  necessary.  A base url that is something like 'http://alpha.olm.net'
 #  will return a url which will most certainly throw an error down the
 #  line.  Similarly, relative urls that would normally break if parsed by
 #  a browser will most likely produce garbage output here. 
 #  -------------------------------------------------------------------------
 ##
proc url::makeAbsolute {baseUrl relUrl} {
    set parent [string range $baseUrl 0 [string last "/" $baseUrl]]
    
    if {[regexp {^[^/]+:} $relUrl]} {
	# This is already absolute
	return $relUrl
    } elseif {[regexp {^\#} $relUrl]} {
	# A simple anchor link in the same document.
        return ${baseUrl}${relUrl}
    } elseif {![regexp {^([./]+).*} $relUrl allofit leadingChars]} {
	# Something contained in the parent directory.
	return ${parent}${relUrl}
    } elseif {[string index $leadingChars 0] == "/"} {
	# Something contained in the server directory.
	if {[regexp {^([a-zA-Z0-9]+://[^/]+)} $parent allofit server]} {
	    return ${server}${relUrl}
	} else {
	    error "Could not identify server."
	}
    } else {
	# Up some levels in the family tree.
	set urlDirs  [split $parent "/"]
	set upTo     [regsub -all {\.\./} $relUrl {} theRest]
	set length   [expr {[llength $urlDirs] - $upTo - 2}]
	set ancestor [join [lrange $urlDirs 0 $length] "/"]/
	# Ignore any './' strings.
	regsub -all {\./} $theRest {} theRest
	return ${ancestor}${theRest}
    }
}

proc url::parse {url} {
    if {![regexp {^([^:]+)://(.*)$} $url dmy type rest]} {
	alertnote "I couldn't understand that url: '$url'"
	error ""
    }
    return [list $type $rest]
}

proc url::parseFtp {p array} {
    # format is user:pass@host/path
    regexp {(([^:]*)(:([^@]*))?@)?([^/]*)/(.*/)?([^/]*)$} $p \
      junk junk user junk pass host path file
    
    if {$user == ""} {
	set user "anonymous"
	if {[catch {set pass [icGetPref Email]}] || ![string length $pass]} {
	    set pass "anonymous"
	}
    }
    upvar $array a
    array set a [list user $user pass $pass host $host path $path file $file]
}

proc url::store {url file} {
    set t [url::parse $url]
    set type [lindex $t 0]
    set rest [lindex $t 1]	
    switch -- $type {
	"ftp" {
	    url::parseFtp $rest i
	    set i(file) [file tail $file]
	    ftpStore "$file" $i(host) "$i(path)$i(file)" $i(user) $i(pass)
	}
	"file" {
	    if {[file isdirectory $file]} {
		alertnote "Don't know how to store directories to file urls"
		error ""
	    } else {
		set urlfilename [string range $rest 1 end]
		file copy $file $urlfilename
	    }
	}
	default {
	    alertnote "Don't know how to put '$type' urls"
	    error ""
	}
    }
}

proc url::fetchFrom {url localdir {file ""}} {
    url::fetch ${url}${file} $localdir $file	
}


## 
 # -------------------------------------------------------------------------
 # 
 # "url::fetch" --
 # 
 #  Get a precise url into a localdir/file.  The url may be a directory,
 #  in which case we retrieve a listing.
 #  
 #  Use url::fetchFrom to fetch a file from a given url-location.
 #  
 #  Note 'Geni' is the sig of a wish applet I wrote which is augmented
 #  with a few procedures to download files via http. 
 #  Of course it needs the user to install Sun's latest
 #  release of Tcl/Tk
 # -------------------------------------------------------------------------
 ##
proc url::fetch {url localdir {file ""}} {
    set t [url::parse $url]
    set type [lindex $t 0]
    set rest [lindex $t 1]
    if {$file != ""} {
	set to [file join $localdir $file]
    } else {
	set to $localdir
    }
    
    switch -- $type {
	"ftp" {
	    url::parseFtp $rest i
	    catch {file mkdir [file dirname $localdir]}
	    if {[regexp "/$" "$i(path)$i(file)"]} {
		# directory
		ftpList $to $i(host) $i(path) $i(user) $i(pass)
	    } else {
		ftpFetch $to $i(host) "$i(path)$i(file)" $i(user) $i(pass)
	    }
	    set localname $to
	}
	"http" {
	    if {[file isdirectory $to]} {
		global file::separator
		if {[regexp "\\${file::separator}\$" $url]} {
		    set to [file join $to index.html]
		} else {
		    set to [file join $to [file tail $url]]
		}
	    }
	    httpFetch $url $to
	    set localname $to
	}
	"file" {
	    set filename [string range $rest 1 end]
	    if {[file isdirectory $filename]} {
		alertnote "Don't know how to fetch 'file' directory urls"
		error ""
	    } else {
		set localname [file join $to [file tail $filename]]
		file copy $filename $localname
	    }
	}
	default {
	    alertnote "Don't know how to fetch '$type' urls"
	    error ""
	}
    }
    return [list $type $localname]
}

proc url::browserWindow {} {
    global tcl_platform browserSig browserSigs
    switch -- $tcl_platform(platform) {
	"macintosh" {
	    # If several different browsers are running, we should
	    # really pick the frontmost, somehow.
	    if {![app::isRunning $browserSigs name sig]} {
		error "No browser running."
	    }
	    if {![regexp {\[([0-9]+)} [AEBuild -r '$sig' WWW! LSTW] "" winnum]} {
		error "No browser window."
	    }
	    # returns window info
	    regexp {\[([^ ]+)} [AEBuild -r '$sig' WWW! WNFO ---- $winnum] "" winurl
	    set winurl [string trim $winurl ","]
	    if {$winurl == "'TEXT'()"} {
		error "Empty browser window."
	    }
	    return $winurl
	}
	"windows" {
	    if {[info exists browserSig]} {
		set root [string tolower [file rootname [file tail $browserSig]]]
	    } else {
		set root iexplore
	    }
	    set root [string trim $root ".0123456789"]
	    # If multiple iexplore instances are running, this seems
	    # to pick the first?  This should work for 'iexplore' and
	    # 'netscape' names.
	    set info [dde request $root WWW_GetWindowInfo 1]
	    set url [lindex [split $info \"] 1]
	    return $url
	}
	"unix" {
	    if {$tcl_platform(os) == "Darwin"} {
		if {![app::isRunning $browserSigs name sig]} {
		    error "No browser running."
		}
		if {![regexp {\[([0-9]+)} [AEBuild -r '$sig' WWW! LSTW] "" winnum]} {
		    error "No browser window."
		}
		# returns window info
		regexp {\[([^ ]+)} [AEBuild -r '$sig' WWW! WNFO ---- $winnum] "" winurl
		set winurl [string trim $winurl "??,"]
		if {$winurl == "'TEXT'()"} {
		    error "Empty browser window."
		}
		return $winurl
	    } else {
		error "Sorry, this is unimplemented.  Please contribute\
		  a suitable implementation!"
	    }
	}
    }
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "GURLHandler" --
 #	
 #  Handle general GURL events by extracting the type 'ftp', 'http', and
 #  calling a procedure ${type}GURLHandler with a single parameter which is
 #  the extracted resource.  Can be put to more general use.  You must
 #  register this proc as an event handler if you want to use it.  Do this
 #  with:
 #   
 #    eventHandler GURL GURL GURLHandler
 #    
 # -------------------------------------------------------------------------
 ##
proc GURLHandler {msg} {
    if {![regsub {.*(.*).*} $msg {\1} gurl]} {
	alertnote "Didn't understand GURL: $msg"
	return
    }
    set GURLtype [lindex [split $gurl ":"] 0]
    set GURLvalue [string range $gurl [expr {1+[string length $GURLtype]}] end]
    if {[catch {${GURLtype}GURLHandler $GURLvalue} msg]} {
	message $msg
    }
}

proc httpFetch {url to} {
    global useTclServiceForHttp
    if {[info exists useTclServiceForHttp] && $useTclServiceForHttp} {
	httpCopy ${url} $to
	return
    }
    global httpDownloadSig httpDownloadSigs
    # force loading of AE code to avoid some timeout/ae problems
    if {[info tclversion] < 8.0} { forceLoadAE }
    app::launchAnyOfThese $httpDownloadSigs httpDownloadSig
    if {[file exists $to] && [file isfile $to]} {
	if {[dialog::yesno "Replace [file tail $to]?"]} {
	    file delete $to
	} else {
	    error "Abort download."
	}
    }
    set fid [alphaOpen $to w]
    close $fid
    if {$httpDownloadSig == "Geni"} {
	switchTo '$httpDownloadSig'
	set res [AEBuild -r -t 30000 '$httpDownloadSig' misc dosc ---- \
	  "[list Http_Copy ${url} $to]"]
	switchTo 'ALFA'
	if {[string match "*Not found*" $res]} {
	    catch {file delete $to}
	    error "File not found on http server."
	}
    } else {
	AEBuild -r -t 30000 '$httpDownloadSig' WWW! OURL ---- "${url}" \
	  INTO [makeAlis "$to"]
    }
}

# Copy a URL to a file and print meta-data
proc httpCopy { url file {chunk 4096} } {
    package require http
    # We declare ourselves otherwise some sites will
    # actually reject us because they view the 'http' library
    # as being most likely a robot, I guess.
    http::config -useragent "AlphaTcl WWW browser"
    
    set out [alphaOpen $file w]
    set token [http::geturl $url -channel $out -progress httpProgress \
      -blocksize $chunk]
    close $out
    upvar #0 $token state
    set max 0
    foreach {name value} $state(meta) {
	if {[string length $name] > $max} {
	    set max [string length $name]
	}
	if {[regexp -nocase ^location$ $name]} {
	    # Handle URL redirects
	    message "Location:$value"
	    return [httpCopy [string trim $value] $file $chunk]
	}
    }
    incr max
    foreach {name value} $state(meta) {
	#puts [format "%-*s %s" $max $name: $value]
    }
    return $token
}

proc httpProgress {args} {
    message $args
}

#  Core ftp code  #

## 
 # -------------------------------------------------------------------------
 # 
 # "ftpFetch" --
 # 
 #  Downloads a remote file to your disk. 
 #  
 # -------------------------------------------------------------------------
 ##
proc ftpFetch {localName host path user password {replyHandler ""}} {
    global useTclServiceForFtp
    file::ensureDirExists [file dirname $localName]
    if {[info exists useTclServiceForFtp] && $useTclServiceForFtp} {
	package require ftp
	set s [ftp::Open $host $user $password -output ftpDisplayMsg]
	if {$s == -1} {
	    error "Failed to open ftp connection to $host"
	}
	ftp::Type $s binary
	if {![ftp::Get $s $path $localName]} {
	    ftp::Close $s
	    error "Problem fetching file"
	}
	ftp::Close $s
	if {[string length $replyHandler]} {
	    eval $replyHandler
	}
	return
    }
    global ftpSig ftpSigs
    # force loading of AE code to avoid some timeout/ae problems
    if {[info tclversion] < 8.0} { forceLoadAE }
    app::launchAnyOfThese $ftpSigs ftpSig "Please locate your ftp application:"
    if {[file exists $localName]} {
	file delete $localName
    }
    if {$ftpSig == "FTCh" || $ftpSig == "Arch"} {
	set localName "[file dirname $localName]:"
	set flag -r
	if {$replyHandler != ""} {
	    currentReplyHandler $replyHandler
	    set flag -q
	}
    }
    switch -- $ftpSig {
	Arch -
	FTCh {AEBuild $flag -t 30000 '$ftpSig' Arch Ftch FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password" ---- [makeAlis $localName]}
	Woof {
	    if {$replyHandler == "" || ![checkNetFinderVersion]} {
		set flag -r
		if {$replyHandler != ""} {
		    currentReplyHandler $replyHandler
		    set flag -q
		}
		close [open $localName "w"]
		AEBuild $flag -t 30000 'Woof' GURL GURL ---- "ftp://${user}:${password}@${host}/${path}" dest [makeAlis $localName]
		return
	    }
	    global PREFS ALPHA
	    set Woof [temp::unique ftptmp Woof]
	    set fid [open $Woof "w"]
	    puts $fid "auto result;"
	    puts $fid "auto script;"
	    puts $fid "auto script1;"
	    puts $fid "auto ftpRef = NFCreateFTPInstance();"
	    puts $fid "NFLoadModuleConstants();"
	    puts $fid "do \{"
	    puts $fid "if (result = NFConnect(ftpRef, \"$host\", 21, \"$user\", \"$password\"), result != 0) break;"
	    puts $fid "if (result = NFReceiveFile(ftpRef, \"$path\", eASCIIType, \"$localName\", eText, NULL, NULL), result != 0) break;"
	    puts $fid "\} while(0);"
	    puts $fid "NFDisconnect(ftpRef);"
	    puts $fid "NFDeleteFTPInstance(ftpRef);"
	    puts $fid "script = \"tell app \\\"$ALPHA\\\"\\r ignoring application responses \\r DoScript \\\"$replyHandler aevt\\\\\\\\\\\\\\\\ansr\\\\\\\\\\\\\\\\{'----':\" + string(result) + \"\\\\\\\\\\\\\\\\}\";"
	    puts $fid "script1 = \"; file delete \{$Woof\}\\\"\\r end ignoring\\r end tell\";"
	    puts $fid "MICI.ExecuteScript(script + script1);"
	    close $fid
	    setFileInfo $Woof type ICI!
	    sendOpenEvent noReply 'Woof' $Woof  
	}
    }
}

proc ftpLibPut {s localName path} {
    global useFtpEolType tcl_platform
    set type $useFtpEolType
    if {$type == "mac"} { set type "macintosh" }
    if {$type == "ibm"} { set type "windows" }
    if {$type == "auto"} { set type $tcl_platform(platform) }
    
    if {[file::isText $localName] && ($tcl_platform(platform) != $type)} {
	set eol ""
	switch -- $type {
	    "unix" { set eol lf }
	    "windows" { set eol crlf }
	    "macintosh" { set eol cr }
	}
	set newLocal [temp::path ftpLibPut [file tail $localName]]
	set fout [open $newLocal w]
	set fin [open $localName r]
	fconfigure $fout -translation $eol
	fcopy $fin $fout
	close $fin
	close $fout
	ftp::Put $s $newLocal $path
	file delete $newLocal
    } else {
	ftp::Put $s $localName $path
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ftpStore" --
 # 
 #  Uploads a file to a remote ftp server.
 #  
 # -------------------------------------------------------------------------
 ##
proc ftpStore {localName host path user password {replyHandler ftpHandleReply}} {
    global useTclServiceForFtp
    if {[info exists useTclServiceForFtp] && $useTclServiceForFtp} {
	package require ftp
	set s [ftp::Open $host $user $password -output ftpDisplayMsg]
	if {$s == -1} {
	    error "Failed to open ftp connection to $host"
	}
	ftp::Type $s binary
	# Note that 'Put' will overwrite existing files.
	if {[catch {ftpLibPut $s $localName $path}]} {
	    # Most likely cause is sub-paths not existing.
	    set pieces [file split [file dirname $path]]
	    set sub {}
	    foreach piece $pieces {
		set sub [file join $sub $piece]
		ftp::MkDir $s $sub
	    }
	    ftpLibPut $s $localName $path
	}
	ftp::Close $s
	return
    }
    global ftpSig ftpSigs
    # force loading of AE code to avoid some timeout/ae problems
    if {[info tclversion] < 8.0} { forceLoadAE }
    app::launchAnyOfThese $ftpSigs ftpSig "Please locate your ftp application:"
    switch -- $ftpSig {
	Arch -
	FTCh {
	    currentReplyHandler $replyHandler
	    AEBuild -q -t 30000 '$ftpSig' Arch Stor ---- [makeAlis $localName] FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password"
	}
	Woof {
	    set dirpath [string range $path 0 [expr {[string last / $path] - 1}]]
	    if {![checkNetFinderVersion]} {
		currentReplyHandler $replyHandler
		AEBuild -q -t 30000 '$ftpSig' PURL PURL ---- [makeAlis $localName] dest "ftp://${user}:${password}@${host}/${dirpath}"
		return
	    }
	    global PREFS ALPHA
	    set Woof [temp::unique ftptmp Woof]
	    set fid [open $Woof "w"]
	    puts $fid "auto result;"
	    puts $fid "auto script;"
	    puts $fid "auto script1;"
	    puts $fid "auto ftpRef = NFCreateFTPInstance();"
	    puts $fid "NFLoadModuleConstants();"
	    puts $fid "do \{"
	    puts $fid "if (result = NFConnect(ftpRef, \"$host\", 21, \"$user\", \"$password\"), result != 0) break;"
	    puts $fid "if (result = NFChangeWorkingDirectory(ftpRef, \"$dirpath\"), result != 0) break;"
	    puts $fid "if (result = NFSendFile(ftpRef, \"$path\", eASCIIType, \"$localName\", eText, NULL, NULL), result != 0) break;"
	    puts $fid "\} while(0);"
	    puts $fid "NFDisconnect(ftpRef);"
	    puts $fid "NFDeleteFTPInstance(ftpRef);"
	    puts $fid "script = \"tell app \\\"$ALPHA\\\"\\r ignoring application responses \\r DoScript \\\"$replyHandler aevt\\\\\\\\\\\\\\\\ansr\\\\\\\\\\\\\\\\{'----':\" + string(result) + \"\\\\\\\\\\\\\\\\}\";"
	    puts $fid "script1 = \"; file delete \{$Woof\}\\\"\\r end ignoring\\r end tell\";"
	    puts $fid "MICI.ExecuteScript(script + script1);"
	    close $fid
	    setFileInfo $Woof type ICI!
	    sendOpenEvent noReply 'Woof' $Woof  
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "ftpList" --
 # 
 #  Saves the file listing of a remote directory to a file. Uses a trick 
 #  for Fetch when saving the file. First the files are listed in a text
 #  window in Fetch. This window is then saved to the disk.
 #  
 # -------------------------------------------------------------------------
 ##
proc ftpList {localName host path user password {replyHandler ""}} {
    global useTclServiceForFtp
    if {[info exists useTclServiceForFtp] && $useTclServiceForFtp} {
	package require ftp
	set s [ftp::Open $host $user $password -output ftpDisplayMsg]
	if {$s == -1} {
	    error "Failed to open ftp connection to $host"
	}
	ftp::Type $s binary
	if {[string length $path]} {
	    if {![regexp {/$} $path]} {append path "/"}
	}
	set res [ftp::List $s $path]
	ftp::Close $s
	set fd [alphaOpen $localName "w"]
	puts $fd [join [concat "dummy" $res "dummy"] "\n"]
	close $fd
	if {[string length $replyHandler]} {
	    eval $replyHandler
	}
	return
    }
    global ftpSig ftpSigs
    # force loading of AE code to avoid some timeout/ae problems
    if {[info tclversion] < 8.0} { forceLoadAE }
    app::launchAnyOfThese $ftpSigs ftpSig "Please locate your ftp application:"
    switch -- $ftpSig {
	Arch -
	FTCh {
	    close [open $localName "w"]
 	    set flag -r
	    if {$replyHandler != ""} {
		currentReplyHandler $replyHandler
		set flag -q
	    }
	    if {$ftpSig == "Arch"} {
		AEBuild $flag -t 30000 '$ftpSig' Arch List FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password" {----} [makeAlis $localName]
	    }
	    if {$ftpSig == "FTCh"} {
		AEBuild -r -t 3000 '$ftpSig' Arch List FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password" {----} [makeAlis $localName]
		AEBuild -r -t 3000 'FTCh' FTCh VwFL ---- "obj{want:type(cFWA), from:'null'(), form:name, seld:$host}"
		AEBuild -r -t 3000 'FTCh' core save ---- "obj{want:type(cFWC), from:'null'(), form:indx, seld:long(1)}" kfil [makeAlis $localName]
		AEBuild $flag -t 3000 'FTCh' core clos ---- "obj{want:type(cFWC), from:'null'(), form:indx, seld:long(1)}" savo "yes"
	    }
	    
	    if {$ftpSig == "Arch"} {
		set newname [file rootname $localName]#1[file extension $localName]
		getFileInfo $localName arr
		if {$arr(datalen) == 0 && [file exists $newname]} {
		    file delete $localName
		    file rename $newname $localName
		}
	    }
	}
	Woof {
	    if {$replyHandler == ""} {
		alertnote "This doesn't work with NetFinder."
		error "no reply handler"
	    }
	    global PREFS ALPHA
	    if {![checkNetFinderVersion]} {
		alertnote "NetFinder 2.1.2 or later required."
		error "too old NetFinder"
	    }
	    close [open $localName "w"]
	    set Woof [temp::unique ftptmp Woof]
	    set fid [open $Woof "w"]
	    puts $fid "auto file;"
	    puts $fid "auto result;"
	    puts $fid "auto item;"
	    puts $fid "auto script;"
	    puts $fid "auto script1;"
	    puts $fid {auto listing = [array];}
	    puts $fid "auto ftpRef = NFCreateFTPInstance();"
	    puts $fid "file = fopen(\"$localName\", \"w\");"
	    puts $fid "NFLoadModuleConstants();"
	    puts $fid "do \{"
	    puts $fid "if (result = NFConnect(ftpRef, \"$host\", 21, \"$user\", \"$password\"), result != 0) break;"
	    puts $fid "if (result = NFListDirectory(ftpRef, \"$path\", 1, &listing), result != 0) break;"
	    puts $fid "forall(item in listing) \{"
	    puts $fid "if ((item.kind & eDirectoryItem) == eDirectoryItem) fprintf(file, \"d \");"
	    puts $fid "else if ((item.kind & eLinkItem) == eLinkItem) fprintf(file, \"l \");"
	    puts $fid "else fprintf(file, \"  \");"
	    puts $fid "fprintf(file, \"Ab 0 0 %s\", item.name);"
	    puts $fid "if ((item.kind & eLinkItem) == eLinkItem) fprintf(file, \" -> %s\", item.link);"
	    puts $fid "fprintf(file, \"\\n\");"
	    puts $fid "\}"
	    puts $fid "\} while(0);"
	    puts $fid "NFDisconnect(ftpRef);"
	    puts $fid "NFDeleteFTPInstance(ftpRef);"
	    puts $fid "close(file);"
	    puts $fid "script = \"tell app \\\"$ALPHA\\\"\\r ignoring application responses \\r DoScript \\\"$replyHandler aevt\\\\\\\\\\\\\\\\ansr\\\\\\\\\\\\\\\\{'----':\" + string(result) + \"\\\\\\\\\\\\\\\\}\";"  
	    puts $fid "script1 = \"; file delete \{$Woof\}\\\"\\r end ignoring\\r end tell\";"
	    puts $fid "MICI.ExecuteScript(script + script1);"
	    close $fid
	    setFileInfo $Woof type ICI!
	    sendOpenEvent noReply 'Woof' $Woof  
	}
	default {
	    alertnote "This doesn't work with [file tail [nameFromAppl $ftpSig]]."
	}
    }
}

# Checks the version of NetFinder
proc checkNetFinderVersion {} {
     global NetFinderVersion
     if {![info exists NetFinderVersion]} {
	alpha::package require version
	# if error, assume recent enough.
	if {[catch {file::version -creator Woof} NetFinderVersion]} {
	    set NetFinderVersion "2.1.2"
	    return 1
	}
     }
     return [expr {[alpha::package vcompare $NetFinderVersion "2.1.2"] >= 0}]
}

ensureset ftpMultipleUploads 0

## 
 # -------------------------------------------------------------------------
 # 
 # "ftpHandleReply" --
 # 
 #  Handles the reply when using ftpStore.  If the 'makedir' parameter is
 #  1 (usually because we were called by 'ftpHandleReplyAndMakeDir'), then
 #  we interpret 'nonexistent directory' errors and call ftpMkDir instead
 #  of throwing an error.
 # -------------------------------------------------------------------------
 ##
proc ftpHandleReply {reply {makedir 0}} {
    global ftpMultipleUploads
    
    regsub {\\\{} $reply "{" reply
    regsub {\\\}} $reply "}" reply
    set ans [string range $reply 10 end]
    if {[regexp {^errs:([^]+)} $ans dum err]} {
	# Fetch error
	if {[regexp {Error: (.*)} $err dum err2]} {set err $err2}
	if {$makedir && ($err == "that file or directory is unavailable or non-existent.")} {
	    message "Creating new directory on server."
	    global ftp::_mkdir
	    eval ftpMkDir ${ftp::_mkdir}
	} else {
	    switchTo 'ALFA'
	    alertnote "Ftp error: $err"
	}
    } elseif {[regexp {^'----':-?([0-9]*)} $ans dum err]} {
	if {$err != "0"} {
	    # Anarchie error.
	    if {$makedir && ($err == "553" || $err == "550")} {
		message "Creating new directory on server."
		global ftp::_mkdir
		eval ftpMkDir ${ftp::_mkdir}
	    } else {
		switchTo 'ALFA'
		alertnote "Ftp error: $err"
	    }
	} else {
	    message "Document uploaded to ftp server."
	    if {$ftpMultipleUploads} {ftpUploadNextFile}
	}
    } elseif {$ans == "\}"} {
	message "Document uploaded to ftp server."
	if {$ftpMultipleUploads} {ftpUploadNextFile}
    } else {
	# Reset this flag.
	set ftpMultipleUploads 0
	return 0
    }
    # Reset this flag.
    set ftpMultipleUploads 0
    return 1
}

proc ftpHandleReplyAndMakeDir {reply} {
    ftpHandleReply $reply 1
}

# Used by Tcl's 'ftp' package.
proc ftpDisplayMsg {s msg {state ""}} {
    switch -- $state {
	data	{::message $msg}
	control	{::message $msg}
	error	{
	    # The way we use 'ftp::Open -output ftpDisplayMsg', we want
	    # this to throw an error whenever something goes wrong.
	    ::message $msg
	    error $msg
	}
	default {::message $msg}
    }	
}

proc ftpMkDir {host path user password {replyHandler ftpMkDirHandler}} {
    global useTclServiceForFtp
    if {[info exists useTclServiceForFtp] && $useTclServiceForFtp} {
	package require ftp
	set s [ftp::Open $host $user $password -output ftpDisplayMsg]
	if {$s == -1} {
	    error "Failed to open ftp connection to $host"
	}
	if {[string length $path]} {
	    if {![regexp {/$} $path]} {append path "/"}
	}
	# This will fail
	if {[catch [list ftp::Type $s $path] type]} {
	    ftp::Mkdir $s $path
	}
	ftp::Close $s
	
	if {[string length $replyHandler]} {
	    eval $replyHandler
	}
	return
    }
    global ftpSig 
    switch -- $ftpSig {
	Arch -
	FTCh {
	    currentReplyHandler $replyHandler
	    AEBuild -q -t 30000 '$ftpSig' Arch MkDr FTPh "$host" FTPc "$path" ArGU "$user" ArGp "$password"
	}
	Woof {
	    global ALPHA

	    set Woof [temp::unique ftptmp Woof]
	    set fid [open $Woof "w"]
	    set dirpath [string range $path 0 [expr {[string last / [string trimright $path /]] - 1}]]
	    puts $fid "auto result;"
	    puts $fid "auto script;"
	    puts $fid "auto ftpRef = NFCreateFTPInstance();"
	    puts $fid "NFLoadModuleConstants();"
	    puts $fid "do \{"
	    puts $fid "if (result = NFConnect(ftpRef, \"$host\", 21, \"$user\", \"$password\"), result != 0) break;"
	    puts $fid "if (result = NFChangeWorkingDirectory(ftpRef, \"$dirpath\"), result != 0) break;"
	    puts $fid "if (result = NFMakeDirectory(ftpRef, \"$path\"), result != 0) break;"
	    puts $fid "\} while(0);"
	    puts $fid "NFDisconnect(ftpRef);"
	    puts $fid "NFDeleteFTPInstance(ftpRef);"
	    # Does this work if replyHandler is "" ?
	    puts $fid "script = \"tell app \\\"$ALPHA\\\"\\r ignoring application responses \\r DoScript \\\"$replyHandler aevt\\\\\\\\\\\\\\\\ansr\\\\\\\\\\\\\\\\{'----':\" + string(result) + \"\\\\\\\\\\\\\\\\}\\\"\\r end ignoring\\r end tell\";"
	    puts $fid "MICI.ExecuteScript(script);"
	    close $fid
	    setFileInfo $Woof type ICI!
	    sendOpenEvent noReply 'Woof' $Woof
	}
    }
}

proc ftpMkDirHandler {reply} {
    global ftpCurrentUpload ftp::_mkdir ftp::_dirtomake ftp::_originaldirtomake
    regsub {\\\{} $reply "{" reply
    regsub {\\\}} $reply "}" reply
    set ans [string range $reply 10 end]
    if {[regexp {^errs:([^]+)} $ans dum err]} {
	# Fetch error
	if {[regexp {Error: (.*)} $err dum err2]} {set err $err2}
	if {$err == "that file or directory is unavailable or non-existent."} {
	    set ftp::_dirtomake [string range ${ftp::_dirtomake} 0 [string last / [string trimright ${ftp::_dirtomake} /]]] 
	    eval ftpMkDir [lreplace ${ftp::_mkdir} 1 1 ${ftp::_dirtomake}]
	} else {
	    switchTo 'ALFA'
	    alertnote "Ftp error: $err"
	}
    } elseif {[regexp {^'----':-?([0-9]*)} $ans dum err]} {
	if {$err != "0"} {
	    # Anarchie error
	    if {$err == "553" || $err == "550" || $err == "521"} {
		set ftp::_dirtomake [string range ${ftp::_dirtomake} 0 [string last / [string trimright ${ftp::_dirtomake} /]]] 
		eval ftpMkDir [lreplace ${ftp::_mkdir} 1 1 ${ftp::_dirtomake}]
	    } else {
		switchTo 'ALFA'
		alertnote "Ftp error: $err"
	    }
	} else {
	    message "Directory created on server."
	    set ftp::_dirtomake ${ftp::_originaldirtomake}
	    eval ${ftpCurrentUpload}
	}
    } elseif {$ans == "\}"} {
	message "Directory created on server."
	set ftp::_dirtomake ${ftp::_originaldirtomake}
	eval ${ftpCurrentUpload}
    } else {
	return 0
    }
    return 1
}


#  Ftp mirroring  #

ensureset anarchieMirrorWarn 1

proc ftpMirrorHierarchy {localDir host user password path} {
    global ftpSig ftpSigs tcl_precision
    app::launchAnyOfThese $ftpSigs ftpSig "Please locate your ftp application:"
    
    if {$ftpSig == "Arch"} {
	global anarchieMirrorWarn
	if {$anarchieMirrorWarn} {
	    set val [dialog -w 400 -h 100 -t "Warning! Files on your server not\
	      found on your disk will be deleted from the server." 10 10 390 40 \
	      -c "Don't warn me about this in the future." 0 10 50 390 65 \
	      -b OK 20 75 85 95 -b Cancel 110 75 165 95]
	    if {[lindex $val 0]} {
		set anarchieMirrorWarn 0
		prefs::modified anarchieMirrorWarn
	    }
	    if {[lindex $val 2]} {return}
	}
	AEBuild '$ftpSig' Arch MPut ---- "$localDir" FTPh "$host" \
	  FTPc "$path" ArGU "$user" \
	  ArGp "$password"
	return
    }
    if {$ftpSig == "Woof"} {
	ftpNetFinderMirror [list $localDir $host $user $password $path]
	return
    }
    set val [dialog -w 330 -h 100 -t "Upload files modified within the last" 10 10 290 30 -e "" 15 40 45 55 \
      -m {hours days hours minutes} 60 40 200 60 -b OK 20 70 85 90 -b Cancel 110 70 175 90 -b "Upload all files" 200 70 320 90]
    set age [string trim [lindex $val 0]]
    if {[lindex $val 3] || (![is::PositiveInteger $age] && ![lindex $val 4])} {
	message "Cancelled"
    }
    
    global ftpUploadLimit
    if {[lindex $val 4]} {
	set ftpUploadLimit 0
    } else {
	if {![info exists tcl_precision]} {
	    set old_precision 6
	} else {
	    set old_precision $tcl_precision
	} 
	set tcl_precision 17
	switch [lindex $val 1] {
	    days {set ftpUploadLimit [expr [now].0 - $age * 86400]}
	    hours {set ftpUploadLimit [expr [now].0 - $age * 3600]}
	    minutes {set ftpUploadLimit [expr [now].0 - $age * 60]}
	}
	set tcl_precision $old_precision
	regexp {[^\.]+} ${ftpUploadLimit} ftpUploadLimit
    }
    message "Building file list"
    set filelist [temp::unique ftptmp mirror]
    
    global ftpFid
    set ftpFid [alphaOpen $filelist w+]
    
    set folders [list $localDir]
    while {[llength $folders]} {
	set newFolders ""
	foreach fl $folders { 
	    foreach f [glob -nocomplain -dir $fl *] {
		if {[file isdirectory $f]} {
		    lappend newFolders $f
		} else {
		    puts ${ftpFid} $f
		}
	    }
	}
	set folders $newFolders
    }
    seek ${ftpFid} 0

    global ftpMultipleUploads ftpMultipleUploadInfo ftpBaselen

    set ftpBaselen [expr {[string length $localDir] + 1}]
    set ftpMultipleUploads 1
    
    set ftpMultipleUploadInfo [list $localDir $host $user $password $path]
    ftpUploadNextFile
}

proc ftpUploadNextFile {} {
    global ftpFid ftpBaselen ftpMultipleUploadInfo file::separator ftp::_mkdir useTclServiceForFtp \
      ftpCurrentUpload ftp::_dirtomake
    
    while {![eof ${ftpFid}] && [set f1 [gets ${ftpFid} f]] != -1 && ![ftpModifiedRecently $f]} {}
    
    if {$f1 != -1} {
	message "Uploading '[file tail $f]'"
	set path [string range $f ${ftpBaselen} end]
	regsub -all ${file::separator} $path {/} path
	if {[lindex ${ftpMultipleUploadInfo} 4] != ""} {set path [join [list [lindex ${ftpMultipleUploadInfo} 4] $path] /]}
	set ftp::_dirtomake [string range $path 0 [string last / $path]]
	set ftp::_mkdir [list [lindex ${ftpMultipleUploadInfo} 1] ${ftp::_dirtomake} [lindex ${ftpMultipleUploadInfo} 2] [lindex ${ftpMultipleUploadInfo} 3]]
	eval [set ftpCurrentUpload [list ftpStore $f [lindex ${ftpMultipleUploadInfo} 1] $path [lindex ${ftpMultipleUploadInfo} 2] [lindex ${ftpMultipleUploadInfo} 3] ftpHandleReplyAndMakeDir]]
	if {[info exists useTclServiceForFtp] && $useTclServiceForFtp} {
	    ftpUploadNextFile
	}
    } else {
	close ${ftpFid}
	message "All documents uploaded to ftp server."
    }
}

proc ftpModifiedRecently {f} {
    global ftpUploadLimit
    getFileInfo $f arr
    return [expr {${ftpUploadLimit} == 0 || ${ftpUploadLimit} < $arr(modified)}]
}

namespace eval ftp {}

array set ftp::NFpwList {
    " " %FA ! %FB \" %F8 # %F9 \$ %FE % %FF & %FC ' %FD ( %F2 ) %F3
    * %F0 + %F1 , %F6 - %F7 . %F4 / %F5 0 %EA 1 %EB 2 %E8 3 %E9
    4 %EE 5 %EF 6 %EC 7 %ED 8 %E2 9 %E3 : %E0 ; %E1 < %E6 = %E7
    > %E4 ? %E5 @ %9A A %9B B %98 C %99 D %9E E %9F F %9C G %9D
    H %92 I %93 J %90 K %91 L %96 M %97 N %94 O %95 P %8A Q %8B
    R %88 S %89 T %8E U %8F V %8C W %8D X %82 Y %83 Z %80 \[ %81
    \\ %86 \] %87 ^ %84 _ %85 ` %BA a %BB b %B8 c %B9 d %BE e %BF
    f %BC g %BD h %B2 i %B3 j %B0 k %B1 l %B6 m %B7 n %B4 o %B5
    p %AA q %AB r %A8 s %A9 t %AE u %AF v %AC w %AD x %A2 y %A3
    z %A0 \{ %A1 | %A6 \} %A7 ~ %A4 \177 %A5  Z  \[  X  Y  ^
     _  \\  \]  R  S  P  Q  V  W  T
     U  J  K  H  I  N  O  L  M  B
     C  %40  A  F  G  D  E  z  \{  x
     y  ~  %7F  |  \}  r  s  p  q  v
     w  t  u  j  k  h  i  n  o  l
     m  b  c  `  a  f ? g  d  e  %1A
     %1B  %18  %19  %1E  %1F  %1C  %1D  %12  %13  %10
     %11  %16  %17  %14  %15  %0A  %0B  %08  %09  %0E
     %0F  %0C  %0D  %02  %03 ? %01  %06  %07  %04
     %05  %3A  ;  8  9  >  %3F  <  =  2
     3  0  1  6  7  4  5  *  +  (
     )  .  /  ,  -  \"  #  " "  !  &
     '  \$  %25
}


proc ftpNetFinderMirror {server} {
    global ftp::NFpwList PREFS ftpNFmirrorFiles
    if {![info exists ftpNFmirrorFiles([lindex $server 0])] || 
    ([set ftpNFmirrorFiles([lindex $server 0])] != ":Generic" && ![file exists [set ftpNFmirrorFiles([lindex $server 0])]])} {
	set val [dialog -w 400 -h 100 -t "No NetFinder mirror file has been selected for this home page. Either select one or let Alpha use a generic one.\
	  If you select one it may only contain one single mirror item." 10 10 390 60 \
	  -b Select 20 70 85 90 -b "Use generic" 110 70 190 90 -b Cancel 215 70 280 90]
	if {[lindex $val 2]} {return}
	if {[lindex $val 0]} {
	    ftpPickNFmirrorFile [lindex $server 0]
	} else {
	    set ftpNFmirrorFiles([lindex $server 0]) ":Generic"
	    prefs::modifiedArrayElement [lindex $server 0] ftpNFmirrorFiles
	}
    }
    if {[set ftpNFmirrorFiles([lindex $server 0])] == ":Generic"} {
	set fil [temp::unique ftptmp NFmirror]
	set passw ""
	for {set i 0} {$i < [string length [lindex $server 3]]} {incr i} {
	    append passw [set ftp::NFpwList([string index [lindex $server 3] $i])]
	}
	set path [makeAlis "[lindex $server 0]:"]
	regexp {(.*)} $path "" path
	set out "<NFML>\n\n<head>\n\t<version=1.0>\n\t<encoding=Macintosh>\n</head>\n\n<body>\n\n<item>\n\t<attributes>\n\t\t<name=\"HTML mode mirror\">"
	append out "\n\t\t<type=MIRROR_ITEM>\n\t\t<source>\n\t\t\t<alias="
	for {set i 0} {$i < [string length $path]} {incr i 64} {
	    append out "\n[string range $path $i [expr {$i + 63}]]"
	}
	append out ">\n\t\t\t<path=\"[lindex $server 0]:\">\n\t\t</source>\n\t\t<target>"
	append out "\n\t\t\t<url=ftp://[lindex $server 2]:${passw}@[lindex $server 1]/[lindex $server 4]>"
	append out "\n\t\t</target>\n\t\t<mirror_options=by_name,by_size>\n\t\t<comment=\"\">"
	append out "\n\t\t<label=0>\n\t\t<lock_status=UNLOCKED>\n\t\t<stationery_status=NORMAL>\n\t</attributes>\n</item>\n\n</body>\n</NFML>"
	set fid [open $fil w]
	puts $fid $out
	close $fid
	setFileInfo $fil type Mirr
    } else {
	set fil [set ftpNFmirrorFiles([lindex $server 0])]
	ftpCheckNFmirrorFile $fil [lindex $server 0]
    }
    sendOpenEvent -r 'Woof' $fil
    switchTo 'Woof'
    # A little delay to make sure window is opened
    set t [ticks]
    while {[expr {[ticks] - $t < 30}]} {
	if {[info tclversion] >= 8.0} { update }
    }
    AEBuild 'Woof' NFAE SAll
    AEBuild 'Woof' NFAE OPEN
}

proc ftpNetFinderMirrorFiles {dir {for ""}} {
    global ftpNFmirrorFiles
    if {![info exists ftpNFmirrorFiles($dir)]} {
	set current "None"
    } else {
	set current [dialog::specialView::file [string trimleft [set ftpNFmirrorFiles($dir)] :]]
    }
    if {$for == ""} {
	set for "files in $dir"
    }
    set val [dialog -w 400 -h 170 -t "NetFinder mirror file for\r${for}" 10 10 390 40 \
      -t "Current file: $current" 10 50 390 65 \
      -t "You can either select a NetFinder mirror file for this folder or let Alpha use a generic one.\
      If you select one it may only contain one single mirror item." 10 75 390 135 \
      -b Select 20 140 85 160 -b "Use generic" 110 140 190 160 -b Cancel 215 140 280 160]
    if {[lindex $val 2]} {return}
    if {[lindex $val 0]} {
	ftpPickNFmirrorFile $dir
    } else {
	set ftpNFmirrorFiles($dir) ":Generic"
	prefs::modifiedArrayElement $dir ftpNFmirrorFiles
    }		
}

proc ftpPickNFmirrorFile {folder} {
    global ftpNFmirrorFiles
    set fil [getfile "NetFinder mirror file"]
    ftpCheckNFmirrorFile $fil $folder
    set ftpNFmirrorFiles($folder) $fil
    prefs::modifiedArrayElement $folder ftpNFmirrorFiles
}

proc ftpCheckNFmirrorFile {fil folder} {
    if {[getFileType $fil] == "Mirr"} {
	set fcont [file::readAll $fil]
	if {[regsub -all {<item>} $fcont "" ""] != 1} {
	    alertnote "The mirror file '[file tail $fil]' must contain one single item."
	    error ""
	}
	if {![regexp "<path=\"$folder:\">" $fcont]} {
	    alertnote "The file '[file tail $fil]' is not a mirror file for the folder '[file tail $folder]'."
	    error ""
	}
    } else {
	alertnote "'[file tail $fil]' is not a NetFinder mirror file."
	error ""
    }
}
