## -*-Tcl-*-
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "alphaHooks.tcl"
 #                                    created: 18/7/97 {5:10:18 pm} 
 #                                last update: 12/13/2001 {19:11:22 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Copyright (c) 1997-2001  Vince Darley
 #  
 # Description: 
 #  
 #  This file contains most of the Tcl procedures which are called
 #  by Alpha/Alphatk internally.  As such you should be very careful
 #  making any changes to these procedures.
 #  
 #  Here are the current hooks:
 #  
 #  activateHook changeMode closeHook deactivateHook modifyModeFlags 
 #  quitHook resumeHook saveasHook saveHook savePostHook suspendHook
 #  openHook
 #  
 #  There's also a 'mode::init' hook which will be called the first
 #  time a mode is started up.  Note that the mode exists, but its
 #  variables have not yet been made global, and its menus have not
 #  yet been inserted into the menu bar.
 #  
 #  There's also a 'startupHook' which is called when Alpha starts
 #  up, but after all other initialisation has taken place (before
 #  any files are opened though).
 #  
 #  There's also a 'launch' hook for when an app is launched.
 #  
 #  If you wish to attach code to any of these procedures,
 #  use hook::register.
 #  
 #  History
 # 
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  18/7/97  VMD 1.0 original
 #  22/7/97  VMD 1.1 fixed all bugs ;-) and added the above examples.
 # ###################################################################
 ##

namespace eval mode {}
namespace eval win {}

lappend mode::procs carriageReturn OptionTitleBar OptionTitleBarSelect \
  electricLeft electricRight electricSemi indentLine indentRegion \
  parseFuncs MarkFile

# Clients of the AlphaTcl library should call things as follows:
# 
# (1) Before opening a file window, the first step is to call
# 'filePreOpeningHook $filename $winname'  (this step is obviously
# skipped for non-file windows).
# 
# (2) When the window is created, and the appropriate structures
# are in place (so 'setWinInfo' works) call 'winCreatedHook $winname'
# (in Alphatk at least it is called after the window's contents
# are in place, but while the window is in a withdrawn state).
# 
# (3) Finally call 'registerWindowWithAlphaTcl' which will itself
# call 'bringToFront' (which calls 'activateHook'), and then
# 'openHook'
# 

# Used in Alphatk and probably Alpha 8.  We may wish to
# add some error handling to this procedure, since if it
# fails we may be in some trouble.
proc registerWindowWithAlphaTcl {name} {
    bringToFront $name
    openHook $name
}

proc saveHook name {
    global win::Modes
    hook::callAll saveHook [set win::Modes($name)] $name
}

proc saveUnmodified {} {
    set name [win::Current]
    if {[file exists $name] || \
      ([regsub { <[0-9]+>$} $name {} name] && [file exists $name])} {
	getFileInfo $name arr
	set mod $arr(modified)
	save
	setFileInfo $name modified $mod
	return
    }
    # shouldn't really get here!
    error "File doesn't exist"
}

proc winChangeMode {name newMode} {
    global win::Modes mode
    set oldmode $mode
    changeMode $newMode
    set win::Modes($name) $newMode
    refresh
}

## 
 # -------------------------------------------------------------------------
 # 
 # "filePreOpeningHook" --
 # 
 #  Called by Alpha(tk)'s core inside things like 'edit' just before
 #  we read in and open a file. 'name' is the name of the file on disk,
 #  'winname' is the name of the window we will use (e.g. it may have
 #  trailing <2>...)
 #  
 #  The idea is that we can use this proc and the hooks it calls
 #  to set the window's mode, to adjust tab-size, encoding, etc, before
 #  the window is properly created, and before the contents of the
 #  file are read in (this is particularly important if we want to
 #  set a particular encoding to use for that reading!)
 #  
 #  In Alpha 7 and Alpha 8 at present it is called in activateHook which
 #  is a little too late.
 # -------------------------------------------------------------------------
 ##
proc filePreOpeningHook {name winname} {
    set m [file::preOpeningConfigurationCheck $name $winname]
    if {![string length $m]} { set m [win::FindMode $name] }

    win::setInitialMode $winname $m
    hook::callAll preOpeningHook $m $winname
    return 0
}

## 
 # -------------------------------------------------------------------------
 # 
 # "winCreatedHook" --
 # 
 #  Called by Alpha(tk)'s core as soon as the window structures are
 #  created so that 'setWinInfo' can work.  In Alpha 7 and Alpha 8 at
 #  present it is only called from openHook which is not early enough.
 #  
 #  Note that, in most respects, the window does not yet exist.
 #  
 #  This proc can be used to set any characteristics of the window
 #  which we somehow determined were necessary while it was being
 #  opened/created (e.g. mode-dependent values).
 #  
 # -------------------------------------------------------------------------
 ##
if {[info tclversion] < 8.0} {
    proc winCreatedHook {winname} {
	global win::config
	if {[info exists win::config($winname)]} {
	    foreach opt [set win::config($winname)] {
		if {([lindex $opt 0] == "tabsize") && (![winDirty])} {
		    catch [list eval setWinInfo $opt]
		    setWinInfo dirty 0
		} else {
		    catch [list eval setWinInfo $opt]
		}
	    }
	    unset win::config($winname)
	}
    }
} else {
    proc winCreatedHook {winname} {
	global win::config
	if {[info exists win::config($winname)]} {
	    foreach opt [set win::config($winname)] {
		catch {eval [list setWinInfo -w $winname] $opt}
	    }
	    unset win::config($winname)
	}
    }
}

# This procedure is used to check whether a global variable
# is currently shadowed by a mode var.  It must be kept
# in sync with what the 'changeMode' procedure does below.
proc globalVarIsShadowed {var} {
    global global::_varMem
    info exists global::_varMem($var)
}

# Works just like 'set' but for a global variable which
# is currently hidden/shadowed by a mode-specific variable.
proc globalVarSet {var args} {
    global global::_varMem
    eval [list set global::_varMem($var)] $args
}

## 
 # -------------------------------------------------------------------------
 # 
 # "changeMode" --
 # 
 #  A very important procedure.  It handles all switching from one mode
 #  to another.  This means it has to adjust menus, floating windows,
 #  global variables, mode prefs, and call a number of hooks.
 #  
 #  It maintains a list of variables which the new mode over-rides from
 #  the global scope, and recreates them.  This allows a mode to have
 #  its own value for a global variable without messing anything up.
 #  
 #  This is not a procedure which should be called by extensions.
 #  To change the mode associated with the current window, call 
 #  win::ChangeMode.
 # -------------------------------------------------------------------------
 ##
proc changeMode {newMode} {
    global mode seenMode global::_varMem \
      global::features global::_oldTabSize \
      alpha::changingMode alpha::changingModeFrom
	
    if {$mode == $newMode} {
	if {$newMode != ""} {
	    displayMode $newMode
	}
	return
    }

    if {${alpha::changingMode}} {
	# This is not something that should happen.  It can happen
	# if a badly constructed mode is being loaded for the first time
	# due to a 'changeMode $badmode', but then during startup the
	# mode decides it wants to switch mode (for some unfathomable
	# reason).
	set msg "A mode or package is causing a serious problem.\
	  It is illegal to change mode while we\
	  are already in the process of changing mode\
	  (currently changing from mode ${alpha::changingModeFrom}\
	  to $mode, attempted mode $newMode)."
	alertnote $msg
	# If we don't written, global variables could get mangled below.
	return -code error $msg
    }
    # Some code would like to know whether we're in the process
    # of changing mode or not (e.g. complex package activation/deactivation
    # sequences).
    set alpha::changingMode 1

    set lastMode $mode
    set mode $newMode
    set alpha::changingModeFrom $lastMode
    
    # This section should restore any internally shadowed globals, 
    # currently only tabSize may be stored by 'new'.  This code
    # is only used for Alpha 7, and will be removed in the future.
    if {[info exists global::_oldTabSize]} {
	global tabSize
	set tabSize [set global::_oldTabSize]
	unset global::_oldTabSize
    }
    
    if {$lastMode == ""} {
	renameMenuItem -m Config "Mode Prefs" "${mode} Mode Prefs"
	catch {menuEnableHook 1}
    } elseif {$mode == ""} {
	renameMenuItem -m Config "${lastMode} Mode Prefs" "Mode Prefs"
	catch {menuEnableHook 0}
    } else {
	renameMenuItem -m Config "${lastMode} Mode Prefs" "${mode} Mode Prefs"
    }
    
    # Get rid of all the old mode's variables, but only if it is necessary
    # (Else we screw up traces on those variables)
    global ${lastMode}modeVars
    if {[info exists ${lastMode}modeVars]} {
        foreach v [array names ${lastMode}modeVars] {
	    if {![info exists global::_varMem($v)]} {
		global $v
		catch {unset $v}
	    }
        }
    }
    floatShowHide off $lastMode
    if {[info exists global::_varMem]} {
	foreach v [array names global::_varMem] {
	    global $v
	    set $v [set global::_varMem($v)]
	}
	unset global::_varMem
    }
    set onoff [package::onOrOff $mode $lastMode]
    
    eval package::deactivate [lindex $onoff 0]
    
    loadAMode $mode

    if {![info exists seenMode($mode)]} {
	eval package::initialise [lindex $onoff 1]
	hook::callAll mode::init $mode
    }
    # once the vars are in mode-var scope (= the <mode>modeVars array),
    # they can be transfered to the global scope.  A future version of
    # Alpha with Tcl8.0 namespaces may not need to do this.
    global ${mode}modeVars
    if {[info exists ${mode}modeVars]} {
        foreach v [array names ${mode}modeVars] {
            global $v
	    if {[info exists $v]} { 
		set global::_varMem($v) [set $v]
	    }
            set $v [set ${mode}modeVars($v)]
        }
    }
    
    eval package::activate [lindex $onoff 1]
    floatShowHide on $mode

    if {![info exists seenMode($mode)]} {
	global mode::procs
	#foreach p ${mode::procs} {
	#    if {[info commands ${mode}::${p}] == ""} {
	#	auto_load ${mode}::${p}
	#    }
	#}
	set seenMode($mode) 1
	global PREFS
	set mprefs [file join $PREFS ${mode}Prefs.tcl]
	if {($mode != "") && [file exists $mprefs]} {
	    if {[catch {uplevel \#0 [list source $mprefs]}]} {
                alertnote "Your preferences file '${mode}Prefs.tcl\
		  has an error."
            } 
        }
    }
        
    if {$newMode != ""} {
	displayMode $newMode
    } else {
	displayEncoding ""
    }

    hook::callAll changeMode $mode $mode

    # Reset this.
    set alpha::changingMode 0
    set alpha::changingModeFrom ""
}

proc loadAMode {mode} {
    global dummyProc
    # These lines must load the mode vars into the mode var scope.
    if {[info exists dummyProc($mode)]} { 
	if {[catch {uplevel \#0 $dummyProc($mode)} err]} {
	    alertnote "There was a BAD problem loading '$mode': $err.\
	      Please contact the mode's maintainer for a fix."
	}
	unset dummyProc($mode)
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "requireOpenWindowsHook" --
 # 
 #  En-/disable meaningless menu items which would require the presence
 #  of a certain number of windows to be active
 #  
 #  This proc should only be called from 'openHook' and 'closeHook'.
 #  
 #  You can register with it using 
 #  
 #  'hook::register requireOpenWindowsHook [list menu item] N'
 #  
 #  where 'N' is the number of windows required (1 or 2 usually)
 #  (and deregister etc using hook::deregister).
 #  
 #  We only really need the catch in here for two reasons:
 #  (i) in case bad menus are registered accidentally
 #  (ii) so startup errors can open a window without hitting another error
 #  in the middle of doing that!
 # -------------------------------------------------------------------------
 ##
proc requireOpenWindowsHook {requiredNum} {
    global win::Active
    set enable [expr {[llength [set win::Active]] >= $requiredNum ? 1 : 0}]
    foreach i [hook::information requireOpenWindowsHook $requiredNum] {
	catch "enableMenuItem $i $enable"
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "menuEnableHook" --
 # 
 #  This hook is called to turn menu items on or off.  It is called 
 #  whenever there are no windows, or when we go from 0->1 window.
 #  
 #  It should deal with all standard menus.  It does not deal with
 #  special menu items like 'save', 'revert',.. which require more
 #  information.
 #  
 #  It is called from changeMode.
 #  
 #  Andreas wrote most of this proc.
 #  
 #  Due to a deficiency in MacOS/MercutioMDEF/Alpha (not sure who
 #  the culprit is!), key-bindings attached to menu items are still
 #  triggered even if the menu item is inactive.
 # -------------------------------------------------------------------------
 ##
proc menuEnableHook {{haveWin 1}} {
    # we only get here if there are no windows, or 1 window which we
    # just opened.  Otherwise nothing will be different to last time.
    enableMenuItem File close $haveWin
    enableMenuItem File closeAll $haveWin
    enableMenuItem File saveAs $haveWin
    enableMenuItem File saveACopyAs $haveWin
    if {[package::active printerChoicesMenu]} {
	enableMenuItem File print $haveWin
    } else {
	enableMenuItem File print $haveWin
    }
    enableMenuItem File printAll $haveWin
    eval [lindex [list un {}] $haveWin]Bind 'p' <c> print

    if {[info tclversion] < 8.0} {
	enableMenuItem Edit undo $haveWin
	enableMenuItem Edit redo $haveWin
	enableMenuItem Edit cut $haveWin
	enableMenuItem Edit copy $haveWin
	enableMenuItem Edit paste $haveWin
	enableMenuItem Edit selectAll $haveWin
	enableMenuItem Edit selectParagraph $haveWin
	enableMenuItem Edit clear $haveWin
	enableMenuItem Edit twiddle $haveWin
	enableMenuItem Edit twiddleWords $haveWin
	enableMenuItem Edit shiftLeft  $haveWin
	enableMenuItem Edit shiftLeftSpace  $haveWin
	enableMenuItem Edit shiftRight  $haveWin
	enableMenuItem Edit shiftRightSpace  $haveWin
	enableMenuItem Edit balance  $haveWin
	
        enableMenuItem Text fillParagraph $haveWin
        enableMenuItem Text wrapParagraph $haveWin
        enableMenuItem Text sentenceParagraph $haveWin
        enableMenuItem Text fillRegion $haveWin
        enableMenuItem Text wrapRegion $haveWin
        enableMenuItem Text sentenceRegion $haveWin
        enableMenuItem Text paragraphToLine $haveWin
        enableMenuItem Text lineToParagraph $haveWin
        enableMenuItem Text reverseSort $haveWin
        enableMenuItem Text sortLines $haveWin
        enableMenuItem Text sortParagraphs $haveWin
        enableMenuItem Text zapInvisibles $haveWin
        enableMenuItem Text tabsToSpaces $haveWin
        enableMenuItem Text spacesToTabs $haveWin
        enableMenuItem Text indentLine $haveWin
        enableMenuItem Text indentSelection $haveWin
        enableMenuItem Text upcaseRegion $haveWin
        enableMenuItem Text downcaseRegion $haveWin
        enableMenuItem Text strings $haveWin
        enableMenuItem Text commentLine $haveWin
        enableMenuItem Text uncommentLine $haveWin
        enableMenuItem Text commentBox $haveWin
        enableMenuItem Text uncommentBox $haveWin
        enableMenuItem Text commentParagraph $haveWin
        enableMenuItem Text uncommentParagraph $haveWin
	enableMenuItem Config "Mode Prefs" $haveWin
    } else {
	global mode
	enableMenuItem Edit "" $haveWin
	enableMenuItem Text "" $haveWin
	if {$mode == ""} {
	    enableMenuItem -m Config "Mode Prefs" $haveWin
	} else {
	    enableMenuItem -m Config "${mode} Mode Prefs" $haveWin
	}
    }
    
    enableMenuItem Search searchStart $haveWin
    enableMenuItem Search findAgain $haveWin
    enableMenuItem Search findAgainBackward $haveWin
    if {!$haveWin && ![string compare [searchString] ""]} {
	enableMenuItem Search findInNextFile $haveWin
    } else {
	enableMenuItem Search findInNextFile 1
    }
    enableMenuItem Search enterSearchString $haveWin
    enableMenuItem Search enterReplaceString $haveWin
    enableMenuItem Search quickFind $haveWin
    enableMenuItem Search quickFindRegexp $haveWin
    enableMenuItem Search reverseQuickFind $haveWin
    enableMenuItem Search replace $haveWin
    enableMenuItem Search replace&FindAgain $haveWin
    enableMenuItem Search replaceAll $haveWin
    enableMenuItem Search placeBookmark $haveWin
    enableMenuItem Search returnToBookmark $haveWin
    enableMenuItem Search gotoLine $haveWin
    enableMenuItem Search matchingLines $haveWin
    enableMenuItem Search gotoMatch $haveWin
    enableMenuItem Search nextMatch $haveWin
    enableMenuItem Search gotoFunc $haveWin
    enableMenuItem Search gotoFileMark $haveWin
    enableMenuItem Search markHilite $haveWin
    enableMenuItem Search namedMarks $haveWin
    enableMenuItem Search unnamedMarks $haveWin
    
    enableMenuItem Utils AsciiEtc $haveWin
    enableMenuItem Utils cmdDoubleClick $haveWin
    enableMenuItem Utils winUtils $haveWin
    enableMenuItem Utils spellcheckWindow $haveWin
    enableMenuItem Utils spellcheckSelection $haveWin
    enableMenuItem Utils wordCount $haveWin
    
    enableMenuItem Config setFontsTabs $haveWin
    
    if {!$haveWin} {
	enableMenuItem File save 0
	enableMenuItem File saveUnmodified 0
	enableMenuItem File revert 0
	enableMenuItem File renameTo 0
	enableMenuItem File saveAll 0
    }

    enableMenuItem fileUtils showInFinder $haveWin

    requireOpenWindowsHook 1
}

proc savePostHook name {
    # So modified date is ok
    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
      && (![catch {getFileInfo $nm info}]))} {
	global win::Modified
	set win::Modified($name) $info(modified)
    } else {
	if {[info tclversion] < 8.0} {
	    # Alpha bug workaround
	    set name [subst $name]
	    if {([file exists $name] && (![catch {getFileInfo $name info}]))\
	      || ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
	      && (![catch {getFileInfo $nm info}]))} {
		global win::Modified
		set win::Modified($name) $info(modified)
	    } else {
		alertnote "Weird, file '$name' doesn't seem to exist: please\
		  report the circumstances of this problem to the\
		  alphatcl-developers mailing list."
	    }
	} else {
	    alertnote "Weird, file '$name' doesn't seem to exist: please\
	      report the circumstances of this problem to the\
	      alphatcl-developers mailing list."
	}
    }
    hook::callAll savePostHook "" $name
}

proc closeHook name {
    global win::Modes win::Active win::Current win::Dirty \
      win::NumDirty win::Modified
    hook::callAll closeHook [set win::Modes($name)] $name

    if {[info exists win::Dirty($name)]} {
	incr win::NumDirty -1
	unset win::Dirty($name)
	enableMenuItem File saveAll [expr {${win::NumDirty} ? 1 : 0}]
    }
		
    unset win::Modes($name)
    if {[info exists win::Modified($name)]} {
	unset win::Modified($name)
    }
    
    if {[set ind [lsearch -exact ${win::Active} $name]] >= 0} {
        set win::Active [lreplace ${win::Active} $ind $ind]
    }
    if {![llength [winNames]]} {
	set win::Current ""
	changeMode {}
    }
    requireOpenWindowsHook 2
}

proc deactivateHook name {
    hook::callAll deactivateHook "" $name
}

proc suspendHook name {
    hook::callAll suspendHook "" $name
}

## 
 # -------------------------------------------------------------------------
 # 
 # "resumeHook" --
 # 
 #  The parameter 'name' is not used, so please ignore it.
 # -------------------------------------------------------------------------
 ##
proc resumeHook {name} {
    # Check if the foremost window needs to be have its modified
    # status adjusted, and calls all resumeHooks with the 
    # modified status (1 or 0) as an extra argument.
    
    hook::callAll resumeHook ""
    
    global win::Active
    foreach win [set win::Active] {
	hook::callAll resumeModifiedHook $win $win [modifiedCheck $win]
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "modifiedCheck" --
 # 
 #  Check whether $name has been modified on disk, and ensure that the
 #  save and revert menu items are correctly dimmed (if this is the front-
 #  most window).
 #  
 #  Returns 1 if the window has been modified on disk since last save.
 # -------------------------------------------------------------------------
 ##
proc modifiedCheck {name} {
    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
      && (![catch {getFileInfo $nm info}]))} {
	set ret 0
	if {[catch {getWinInfo -w $name arr}]} {
	    set mod 0
	} else {
	    set dirty $arr(dirty)
	    if {!$dirty} {
		global win::Modified
		set diff [expr {[set win::Modified($name)] - $info(modified)}]
		set mod [expr {$diff != 0}]
		if {$mod} { 
		    diskModifiedHook $name 1 $diff
		    set ret 1
		}
	    } else {
		set mod 1
	    }
	}
	if {$name == [win::Current]} {
	    enableMenuItem File save $mod
	    enableMenuItem File revert $mod
	}
	return $ret
    }
    return 0
}

proc diskModifiedHook {name {mod 1} {diff 0}} {
    if {$mod} {
	set msg "File has changed on disk since last save"
	if {$diff > 0} {
	    append msg " (the version on disk is older)"
	} elseif {$diff < 0} {
	    append msg " (the version on disk is newer)"
	}
	message $msg
	hook::callAll diskModifiedHook 1 $name 1 $diff
    } else {
	# Unmodified (i.e. reverted)
	hook::callAll diskModifiedHook 0 $name 0 $diff
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "saveasHook" --
 # 
 #  Called when saving a window which doesn't yet exist as a file
 #  (in particular 'untitled' windows) or when the user selects
 #  saveAs.
 # -------------------------------------------------------------------------
 ##
proc saveasHook {oldName newName} {
    global win::Modes win::Active win::Current win::Modified
    if {$oldName == $newName} return
    win::setMode $newName
    changeMode [set win::Modes($newName)]
    
    if {[set ind [lsearch -exact ${win::Active} $oldName]] >= 0} {
	set win::Active [linsert [lreplace ${win::Active} $ind $ind] 0 $newName]
    } else {
	# hmmm! this is bad.  The old window has gone!
	set win::Active [linsert ${win::Active} 0 $newName]
    }
    
    set win::Current $newName
    if {[info exists win::Modes($oldName)]} {
	unset win::Modes($oldName)
    }
    if {[info exists win::Modified($oldName)]} {
	unset win::Modified($oldName)
    }

    hook::callAll saveasHook [set win::Modes($newName)] $oldName $newName
    refresh
}

if {0 && [info tclversion] < 8.0} {
    hook::register saveasHook callSavePostHook *
    proc callSavePostHook {old new} {
	savePostHook $new
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "saveACopyAs" --
 # 
 # (This proc actually has nothing to do with hooks, but seemed to fit here)
 # -------------------------------------------------------------------------
 ##
proc saveACopyAs {} {
    if {[file exists [set nm [win::StripCount [win::Current]]]]} {
	global alpha::platform
	if {${alpha::platform} == "alpha"} {
	    set nm2 [putfile "Save a copy as:" [file tail $nm]]
	} else {
	    set nm2 [putfile "Save a copy as:" $nm]
	}
	if {$nm2 == $nm} {
	    dialog::errorAlert "You can't save a copy on top of the\
	     original!"
	}
	if {[string length $nm2]} {
	    if {[file exists $nm2]} {file delete $nm2}
	    file copy $nm $nm2
	}
    }
}

ensureset win::Active ""

proc activateHook {name} {
    global win::Modes win::Active win::Current win::Modified encoding \
      alpha::platform
    
    # if the file exists (this seems to be the quickest way to check)
    set isfile [expr {[file exists $name] || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm])}]

    if {![info exists win::Modes($name)]} {
	# Ideally this should happen internal to Alpha when a file-window
	# is opened.  These calls check the actual contents of the file
	# (the first few lines) to see whether a particular tabsize, encoding,
	# mode, etc are specified.
	if {${alpha::platform} == "alpha"} {
	    if {$isfile} {
		if {[info exists nm]} {
		    filePreOpeningHook $nm $name
		} else {
		    filePreOpeningHook $name $name
		}
	    }
	}
	if {![info exists win::Modes($name)]} {
	    win::setMode $name
	}
    }
    if {[set ind [lsearch -exact ${win::Active} $name]] == -1} {
	set win::Active [linsert ${win::Active} 0 $name]
    } elseif {$ind >= 1} {
	set win::Active [linsert [lreplace ${win::Active} $ind $ind] 0 $name]
    }
    set win::Current $name

    # Change mode and encoding
    changeMode [set win::Modes($name)]
    displayEncoding [set encoding [win::Encoding]]
    
    hook::callAll activateHook [set win::Modes($name)] $name

    # This fails if the window is just opening, but then we know it's clean
    # (on Alpha 8/Tk it doesn't fail).
    if {[catchNoClobber {getWinInfo -w $name arr}]} {
	set dirty 0
	set mod 0
    } else {
	set dirty $arr(dirty)
    }
    
    if {$isfile} {
	if {!$dirty} {
	    if {![info exists mod]} {
		if {[info exists win::Modified($name)]} {
		    if {[info exists nm]} {
			getFileInfo $nm modarr
		    } else {
			getFileInfo $name modarr
		    }
		    set diff [expr {[set win::Modified($name)] \
		      - $modarr(modified)}]
		    set mod [expr {$diff != 0}]
		    if {$mod} { 
			diskModifiedHook $name 1 $diff
		    }
		} else {
		    set mod 0
		}
	    }
	} else {
	    set mod 1
	}
	enableMenuItem File save $mod
	enableMenuItem File saveUnmodified $dirty
	enableMenuItem File revert $mod
	enableMenuItem File renameTo 1
    } else {
	if {${alpha::platform} == "alpha"} {
	    # On Alphatk this prevents save acting as save-as when the 
	    # window has never been saved.
	    enableMenuItem File save 0
	} else {
	    enableMenuItem File save 1
	}
	enableMenuItem File saveUnmodified 0
	enableMenuItem File revert 0
	enableMenuItem File renameTo 0
    }
    enableMenuItem Edit undo $dirty
}

proc quitHook {} {
    global alpha::tracingChannel
    catch {close ${alpha::tracingChannel}}
    hook::callAll quitHook
    prefs::saveModified
}

proc uninstall {{force 0}} {
    if {!$force} {
	if {![dialog::yesno "Are you sure you wish to remove Alpha from\
	  your system"]} {
	    message "Cancelled"
	    return
	}
    }
    hook::register quitHook uninstallHook
    quit
}

proc uninstallHook {} {
    # Call any registered hooks
    hook::callAll uninstallHook
    # Delete all prefs
    prefs::deleteEverything
    
    # So we don't save any new prefs
    global skipPrefs
    set skipPrefs 2
    
    alertnote "You may now delete the entire Alpha distribution."
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dirtyHook" --
 # 
 #  This proc currently has to keep track in the array 'win::Dirty' of
 #  the dirty status of windows.  Its only use is if we close a dirty
 #  window and select 'discard', we would otherwise have a faulty
 #  'win::NumDirty' count.  If there's a different solution we should
 #  get rid of the win::Dirty array.
 #  
 #  Note: closeHook is called after the window is gone, and killWindow
 #  isn't called if you click in the close-box, so they don't solve
 #  the problem.
 # -------------------------------------------------------------------------
 ##
proc dirtyHook {name dirty} {
    global win::NumDirty win::Dirty win::Modes
    if {$dirty == "on" || $dirty == 1} {
	set win::Dirty($name) 1
	incr win::NumDirty 1
    } else {
	if {[info exists win::Dirty($name)]} {
	    unset win::Dirty($name)
	}
	incr win::NumDirty -1
    }
    enableMenuItem File save $dirty
    enableMenuItem File saveUnmodified $dirty
    enableMenuItem File revert $dirty
    enableMenuItem File saveAll [expr {${win::NumDirty} ? 1 : 0}]
    enableMenuItem Edit undo $dirty
    
    hook::callAll dirtyHook [set win::Modes($name)] $name $dirty
}

proc openHook {name} {
    global win::Modes autoMark mode screenHeight screenWidth \
      forceMainScreen win::Modified PREFS

    changeMode [set win::Modes($name)]
    message ""

    if {[info tclversion] < 8.0} {
	# This should be called much earlier in Alpha 8/tk
	winCreatedHook $name
    }
    
    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
      && (![catch {getFileInfo $nm info}]))} {
        if {[info exists info(creator)] && ($info(creator) == {ttxt})} {
            setWinInfo dirty 0
        }
        if {[info exists info(type)] && ($info(type) == {ttro})} {
            catch {setWinInfo read-only 1}
            message "Read-only!"
        }
	set win::Modified($name) $info(modified)
    }
    
    global ${mode}modeVars
    
    if {$forceMainScreen} {
        set geo [getGeometry]
        newforeach {l t w h} $geo {break}
        if {($l < 0) || ($t < 35) || ([expr {$l + $w}] > $screenWidth) \
	  || ([expr {$t + $h + 18}] > $screenHeight)} {
	    defaultSize
        }
    }
    getWinInfo arr
    if {!$arr(read-only)} {
	if {[info exists ${mode}modeVars(autoMark)] \
	  && [set ${mode}modeVars(autoMark)] \
	  && ![llength [getNamedMarks -n]]} {
	    markFile
	}
    }
    
    if {[string match "${PREFS}*defs.tcl" $name]} {setWinInfo read-only 1}
	
    requireOpenWindowsHook 2
    
    hook::callAll openHook [set win::Modes($name)] $name
}

## 
 # -------------------------------------------------------------------------
 # 
 # "fileMovedHook" --
 # 
 #  Called by Alpha when a window's file has been moved behind our back.
 #  (Only for Alpha using Tcl 8.0)
 # -------------------------------------------------------------------------
 ##
proc fileMovedHook {from to} {
    global win::Active win::Modes win::Modified
    if {![info exists win::Modes($from)]} {
	alertnote "Can't find old window.  Bad error."
    }
    set win::Modes($to) [set win::Modes($from)]
    set win::Modified($to) [set win::Modified($from)]
    unset win::Modes($from)
    unset win::Modified($from)
    set idx [lsearch -exact ${win::Active} $from]
    if {$idx >= 0} {
	set win::Active [lreplace ${win::Active} $idx $idx $to]
    } else {
	alertnote "Can't find the old window! Bad error in fileMovedHook."
    }
    hook::callAll fileMovedHook $from $to
}

proc changeTextHook {name} {
    global win::Modes
    hook::callAll changeTextHook [set win::Modes($name)] $name
}

proc revertHook {name} {
    global win::Modified
    if {([file exists $name] && (![catch {getFileInfo $name info}])) || \
      ([regsub { <[0-9]+>$} $name {} nm] && [file exists $nm] \
      && (![catch {getFileInfo $nm info}]))} {
	set win::Modified($name) $info(modified)
	diskModifiedHook $name 0
    }
    enableMenuItem File save 0
    enableMenuItem File revert 0
}

proc encodingMenuHook {} {
    menu::buildOne encodingMenu
    return "encodingMenu"
}
menu::buildProc encodingMenu buildEncodingMenu

proc contextualMenuHook {pos} {
    global menu::additions mode

    set menuName "ctxtlMenu"

    catch {unset menu::additions(${menuName})}

    set selectionDesc [tclAE::createDesc null]

    hook::callAll contextualMenuHook $mode $menuName $pos $selectionDesc

    menu::buildSome $menuName

    return [list $menuName $selectionDesc]
}

proc vcsMenuHook {} {
    if {[package::active vcs]} {
	menu::buildOne vcsMenu
	return "vcsMenu"
    } else {
	dialog::errorAlert "The VCS popup menu requires the package 'vcs'"
    }
}

proc marksMenuHook {} {
    catch {unset menu::additions(marksMenu)}
    menu::buildOne marksMenu
    
    return "marksMenu"
}
menu::buildProc marksMenu buildMarksMenu

proc parseMenuHook {} {
    catch {unset menu::additions(parseMenu)}
    menu::buildOne parseMenu
    return "parseMenu"
}
menu::buildProc parseMenu buildParseMenu

proc modeMenuHook {} {
    catch {unset menu::additions(modeMenu)}
    menu::buildOne modeMenu
    return "modeMenu"
}
menu::buildProc modeMenu buildModeMenu


proc wrapMenuHook {} {
    catch {unset menu::additions(wrapMenu)}
    menu::buildOne wrapMenu
    return "wrapMenu"
}
menu::buildProc wrapMenu buildWrapMenu

proc fileInfoMenuHook {} {
    catch {unset menu::additions(fileInfoMenu)}
    menu::buildOne fileInfoMenu
    return "fileInfoMenu"
}
menu::buildProc fileInfoMenu buildFileInfoMenu

