## -*-Tcl-*- (nowrap)
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "dialogs.tcl"
 #                                    created: 01-10-03 19.48.52 
 #                                last update: 01-12-12 20.57.49 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Much copyright (c) 1997-2001  Vince Darley
 # rest Pete Keleher, Johan Linde.
 # 
 # Reorganisation carried out by Vince Darley with much help from Tom
 # Fetherston, Johan Linde and suggestions from the alphatcl-developers
 # mailing list.  Alpha is shareware; please register with the author
 # using the register button in the about box.
 #  
 #  Description: 
 # 
 # Much more flexible dialogs for querying the user about flags and
 # vars.  These may be global, mode-dependent, or package-dependent.
 # 
 # Things you may wish to do:
 # 
 #  dialog::pkg_options Pkg
 #  
 # creates a dialog for all array entries 'PkgmodeVars'.  These
 # must have been previously declared using 'newPref'.  These
 # variables are _not_ copied into the global scope; only
 # existing as array entries.
 # 
 # Note that rather than setting up traces on variables, you are
 # often better off using the optional proc argument to newPref;
 # the name of a procedure to call if that element is changed by
 # the user.
 # 
 # Use the procedure 'newPref' to declare preferences.  Why?  It has
 # optional arguments which allow you to declare:
 # 
 #  lists
 #  indexed lists
 #  folders
 #  files
 #  bindings
 #  menu-bindings
 #  applications
 #  variable-list elements
 #  array elements
 #  
 # all of which can be set using the same central mode/global
 # dialogs.
 #  
 # It also lets you add an optional procedure to call when an
 # item changes...  Also if Alpha upgrades to Tcl 8 and namespaces, 
 # it is easy to modify that central procedure to fit everything 
 # with the new scheme.
 # 
 # Most modes will just want to declare their vars using newPref.  
 # There is usually no need to do _anything_ else.
 # 
 # ---
 # 
 # The prefs dialog procs below were based upon Pete Keleher's 
 # originals.
 # ###################################################################
 ##

namespace eval dialog {}
namespace eval global {}
namespace eval flag {}

#  Toplevel dialog procedures  #

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::pkg_options" --
 # 
 #  Make a dialog for the given package, with 'title' for the dialog box.
 #  'not_global' indicates the variables are never copied into the global
 #  scope, remaining in their array ${pkg}modeVars (or '$var' if it is given)
 # 
 #  There is now some support for '$var' not to represent an array, but
 #  rather to be a namespace inside which the variables are placed. 
 #  However this hasn't been tested much.
 #  
 # Results:
 #  Nothing
 # 
 # Side effects:
 #  May modify any of the given package's variables.
 # 
 # --Version--Author------------------Changes-------------------------------
 #    1.0     <vince@santafe.edu> original
 # -------------------------------------------------------------------------
 ##
proc dialog::pkg_options {pkg {title ""} {not_global 1} {var ""} {listOfVars ""}} {
    if {!$not_global} {
	# make sure the package variables are global
	global ${pkg}modeVars
	if {[info exists ${pkg}modeVars]} {
	    foreach v [array names ${pkg}modeVars] {
		global $v
		set $v [set ${pkg}modeVars($v)]
	    }
	}
    }
    if {$title == ""} { 
	set title "Preferences for the '[quote::Prettify $pkg]' package" 
    }
    if {$not_global} {
	global dialog::_not_global_flag
	if {$var == ""} {
	    set dialog::_not_global_flag ${pkg}modeVars
	} else {
	    set dialog::_not_global_flag $var
	}
    }
    if {[llength $listOfVars]} {
	global dialog::_variablesForEditing
	set dialog::_variablesForEditing $listOfVars
    }
    set err [catch {dialog::modifyModeFlags $title $not_global $pkg} result]
    if {$not_global} {
	global dialog::_not_global_flag
	set dialog::_not_global_flag ""
    }
    if {[info exists dialog::_variablesForEditing]} {
	unset dialog::_variablesForEditing
    }
    if {$err} {
	error $result
    }
}
proc dialog::edit_array {var {title ""}} {
    if {$title == ""} {set title "Contents of '$var' array"}
    dialog::pkg_options "" $title 1 $var
}

proc dialog::chooseOption {prefArray prefName prefPage args} {
    global $prefArray
    set p "[quote::Prettify $prefName]"
    set options [lremove [array names $prefArray] "Choose each time"]
    if {![llength $options]} {
        status::errorMsg "No options for $prefName available !"
    } elseif {[llength $options] == "1"} {
        set val [lindex $options 0]
    } else {
        set setPref "(Set $prefPage preferences to avoid this dialog )"
        lappend options $setPref
        set val [listpick -p $p $options]
        if {$val == $setPref} {
            dialog::preferences preferences $prefPage
            global $prefName
            set val [set $prefName]
        }
    }
    eval [set [set prefArray]($val)] $args
}

proc dialog::editOneOfMany {title var store tempStore {what ""}} {
    global $tempStore $store
    if {[regexp {(.*)\(.*\)$} $var "" arr elt]} {
	global $arr
    } else {
	global $var
    }
    set oldInfo [array get $tempStore]
    if {[catch {dialog::pkg_options "" $title 1 $tempStore}] \
      || ($oldInfo == [array get $tempStore])} {
	return
    }
    set oldId [set $var]
    if {![dialog::yesno -y "Update" -n "New $what" \
      "Update [set $var] $what, or make a new one?"]} {
	# Ask for new name
	set name [eval prompt [list "Enter tag for new $what" \
	  "<Tag>" "Old ids:"] [array names $store]]
	set ${store}($name) [array get $tempStore]
	set $var $name
	# Have to store Usual id too.
	prefs::modified ${store}($name)
	prefs::modified $var
    } else {
	set ${store}($oldId) [array get $tempStore]
    }
    prefs::modified ${store}($oldId)
}

proc helperApps {} {
    set sigs [info globals *Sig]
    regsub -all {Sig} $sigs {} sigs
    set sig [listpick -p "Change/inspect which helper?" [lsort -ignore $sigs]]
    set sig ${sig}Sig
    global $sig
    if {![info exists $sig]} { set $sig "" }
    set nsig [dialog::askFindApp $sig [set $sig]]
    if {$nsig != "" && [set $sig] != $nsig} {
	set $sig $nsig
	prefs::modified $sig
    }
}

proc suffixMappings {} {
    global filepats modessuffices pagesindices
    set modes [lsort -ignore [array names filepats]]
    set len [llength $modes]
    set modessuffices {}
    set pagesindices [list 0]
    set pagenumber 1
    foreach m $modes {
	lappend modessuffices $filepats($m)
    }
    set startidx 0
    set rest $len
    if {![sufficesDialog $pagenumber $startidx $rest]} {return} 
    mode::updateSuffixes
}

proc sufficesDialog {{page 1} {startidx 0} {rest 0}} {
    global filepats modessuffices pagesindices
    set modeslist [lsort -ignore [array names filepats]]
    # Set params
    set dim [getMainDevice]
    set screenwidth [expr {[lindex $dim 2] - [lindex $dim 0]}]
    set screenheight [expr {[lindex $dim 3] - [lindex $dim 1]}]
    set l1 5
    set w1 38
    set l2 [expr {$l1 + $w1 + 5}]
    set w2 [expr {($screenwidth - 200)/2}]
    if {$w2 > 400 || $screenwidth <= 750} { set w2 400 }
    set h 18
    set top 5
    set mar 10
    set colht $top
    set idx $startidx
    set dialoght [expr $screenheight - 100]
    set dialogwd [expr {$l2 + $w2 + 10}]
    set buttonline [expr $dialoght - 28]
    set prevht $buttonline
    set nextht $buttonline

    # Build first column
    while {[expr {$screenheight - 160 > $colht}] && $rest} {
	set m [lindex $modeslist $idx]
	lappend items -t $m $l1 $colht [expr {$l1 + $w1}] [expr {$colht + $h}]
	lappend items -e [lindex $modessuffices $idx] $l2 $colht [expr {$l2 + $w2}]
	if {[string length [lindex $modessuffices $idx]] > 60} {
	    lappend items [expr {$colht + 2*$h - 2}]
	    incr colht [expr {2*$h + $mar}]
	} else {
	    lappend items [expr {$colht + $h - 2}]
	    incr colht [expr {$h + $mar}]
	}
	incr idx
	incr rest -1
    }
    # Build second column if necessary and if screen is large enough
    if {$rest && [expr {$screenwidth > 750}]} {
	set l1 [expr {$l2 + $w2 + 20}]
	set l2 [expr {$l1 + $w1 + 5}]
	set colht $top
	while {[expr {$screenheight - 160 > $colht}] && $rest} {
	    set m [lindex $modeslist $idx]
	    lappend items -t $m $l1 $colht [expr {$l1 + $w1}] [expr {$colht + $h}]
	    lappend items -e [lindex $modessuffices $idx] $l2 $colht [expr {$l2 + $w2}]
	    if {[string length [lindex $modessuffices $idx]] > 60} {
		lappend items [expr {$colht + 2*$h - 2}]
		incr colht [expr {2*$h + $mar}]
	    } else {
		lappend items [expr {$colht + $h - 2}]
		incr colht [expr {$h + $mar}]
	    }
	    incr idx
	    incr rest -1
	}
    } 
    
    set endidx $idx
    lunion pagesindices $endidx

    # Build buttons
    lappend buts -b OK [expr {$l2 + $w2 - 65}] $buttonline \
      [expr {$l2 + $w2 - 5}] [expr {$buttonline + 20}]
    lappend buts -b Cancel [expr {$l2 + $w2 - 140}] $buttonline \
      [expr {$l2 + $w2 - 80}] [expr {$buttonline + 20}]
    if {!$rest} {
	incr nextht 10000
    } 
    if {$page==1} {
	incr prevht 10000
    } 
    lappend buts -b "<< Prev" [expr {$l2 + $w2 - 310}] $prevht [expr {$l2 + $w2 - 250}] \
      [expr {$prevht + 20}]
    lappend buts -b "Next >>" [expr {$l2 + $w2 - 240}] $nextht \
      [expr {$l2 + $w2 - 180}] [expr {$nextht + 20}]

    # Display dialog
    if {[info tclversion] < 8.0} {
	set res [eval [list dialog -w [expr {$l2 + $w2 + 10}] \
	  -h $dialoght] $buts $items]
    } else {
	set res [eval [list dialog -w [expr {$l2 + $w2 + 10}] \
	  -h $dialoght -T "Suffix mappings"] $buts $items]
    }
    # Retrieve values from the dialog
    #   Cancel button
    if {[lindex $res 1]} {return 0}
    #   Other button : record the data of the current page
    set locallist ""
    for {set theidx $startidx} {$theidx < $endidx} {incr theidx} {
	lappend locallist [lindex $res [expr {$theidx - $startidx + 4}]]
    }
    set modessuffices [concat \
      [lrange $modessuffices 0 [expr $startidx - 1]] \
      $locallist \
      [lrange $modessuffices $endidx end]]
    #   OK button
    if {[lindex $res 0]} {
	# Validate the changes
	set midx 0
	foreach m $modeslist {
	    set exts [lindex $modessuffices $midx]
	    if {$filepats($m) != $exts} {
		if {[is::List $exts]} {
		    set filepats($m) $exts
		    prefs::modified filepats($m)
		} else {
		    lappend errors $m
		}
	    }
	    incr midx
	}
	if {[info exists errors]} {
	    alertnote "[join $errors ,]\
	      mode[expr {[llength $errors] > 1 ? {s} : {}}] had illegal\
	      lists of patterns. Please make sure \\\{,\\\} are properly\
	      quoted.  Changes to those modes have been ignored."
	}
	return 1
    }
    #   Prev or Next button was hit : record the changes in the 'modessuffices' temp list
    set delta [expr {[lindex $res 3]-[lindex $res 2]}]
    incr page $delta
    set startidx [lindex $pagesindices [expr $page-1]]
    if {$delta=="-1"} {
	set rest [expr {$rest + $endidx - $startidx}]
    } 
    return [sufficesDialog $page $startidx $rest]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::flagsAndVars" --
 # 
 #  Takes a list of flags and variables (where the latter can in fact
 #  contain sublists of stuff), and creates the dialog basing the
 #  title for each item on its name, extracting help text as required,
 #  and basing the type of each item on either its name or the type
 #  which has been registered.
 #  
 #  The procedure used to be named 'dialog::mode', but has been 
 #  renamed to reflect it's broader purpose.  It will return two lists,
 #  the first of the values returned, the second of the variables which
 #  should be set to those values.  The calling procedure should ensure
 #  that the variables are actually set to these values!
 #  
 #  The current values for the flags/vars must be accessible, since
 #  they will be required by the code this procedure calls.  By default
 #  it is assumed each flag/var is a global variable, unless various
 #  other information is declared in various globals.  The method
 #  which is actually called to get the values is dialog::getFlag,
 #  which usually calls dialog::getOldFlag.
 #  
 #  Please see those methods for details if you want to call this
 #  procedure without using global variables.
 # -------------------------------------------------------------------------
 ##
proc dialog::flagsAndVars {mod flags vars {title ""}} {
    set lim [expr {10 - [llength $flags]/4}]
    if {[llength $vars] > $lim } {
	set args {}
	set nvars [llength $vars]
	set j 0
	for {set i 0} {$i < $nvars} {incr i $lim ; set lim 10} {
	    lappend args [list "Page [incr j] of ${title}" $flags \
	      [lrange $vars $i [expr {$i+$lim -1}]]]
	    set flags ""
	}
	dialog::multipage $mod $title $args
    } else {
	dialog::onepage $mod $flags $vars $title
    }
}
## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::modifyModeFlags" --
 # 
 #  Currently 'not_global == 0' implies this is a mode, or at least that
 #  the variables are stored in ${mm}modeVars(...)
 #  
 #  'not_global == 1' implies that the variables are stored in the
 #  array/namespace given by the value of the variable
 #  'dialog::_not_global_flag'
 #  
 # -------------------------------------------------------------------------
 ##
proc dialog::modifyModeFlags {{title ""} {not_global 0} {mm ""}} {
    global mode invisibleModeVars \
      dialog::_not_global_flag allFlags 
    # Check whether this is a mode or package, and where variable values
    # are stored, and whether that's at the global level as well as in
    # an array...
    if {$not_global} {
	set storage ${dialog::_not_global_flag}
	if {$title == ""} {
	    set title "Preferences for '${mm}' package"
	}
    } else {
	if {$mm == ""} { 
	    set mm $mode 
	    if {$mm == ""} {
		alertnote "No mode set!"
		return
	    }
	}
	set storage ${mm}modeVars
	if {$title == ""} {
	    set title "Preferences for '${mm}' mode"
	}
    }
    # check for mode specific proc
    if {[info commands ${mm}modifyFlags] != ""} {${mm}modifyFlags; return}
    if {[info tclversion] >= 8.0} { set storage ::$storage }
    set unsortedNames {}
    global $storage ${storage}Invisible index::flags index::feature\
      dialog::_variablesForEditing
    if {[info exists dialog::_variablesForEditing]} {
	set unsortedNames ${dialog::_variablesForEditing}
    } elseif {[array exists $storage]} {
	set unsortedNames [array names $storage]
    } elseif {[namespace_exists $storage]} {
	foreach var [info vars ${storage}::*] {
	    # Caution: Tcl 8 or newer only.
	    lappend unsortedNames [namespace tail $var]
	}
    }
    if {[llength $unsortedNames]} {
	set mflags {}
	set mvars {}
	set colors {}
	set rest {}
	foreach i $unsortedNames {
	    if {[regexp {Colou?r$} $i]} {
		lappend colors $i
	    } else {
		lappend rest $i
	    }
	}
	
	foreach v [concat [lsort $rest] [lsort $colors]] {
	    if {[info exists invisibleModeVars($v)] \
	      || [info exists ${storage}Invisible($v)]} continue
	    
	    if {[lsearch -exact $allFlags $v] >= 0} {
		lappend mflags $v
	    } else {
		lappend mvars $v
	    }
	}
	foreach pkg [set index::flags] {
	    if {[lsearch -exact [lindex [set index::feature($pkg)] 1] $mm] != -1} {
		lappend mflags $pkg
	    }
	}
	if {![llength $mflags] && ![llength $mvars]} {
	    alertnote "There are no preferences!"
	    return
	}
	set dial [dialog::create]
	if {[catch {dialog::flagsAndVars $dial $mflags $mvars $title} values_items]} {
	    dialog::cleanup $dial
	    return
	}
	dialog::adjust_flags $dial \
	  [expr {$not_global ? "arraynamespace" : "arrayglobal"}] \
	  $values_items $storage
	dialog::cleanup $dial
    } else {
	alertnote "The '$mm' mode/package has no preference settings."
    }
    
    hook::callAll dialog::modifyModeFlags $mm $title
    
}

proc global::allPrefs {{which "AllPreferences"}} {
    global flagPrefs varPrefs
    
    set dial [dialog::create]

    global::updateHelperFlags
    global::updatePackageFlags
    set AllPreferences [array names flagPrefs]
    set InterfacePreferences {
	Appearance Completions Electrics Help Text Tiling Window
    }
    set Input-OutputPreferences {Backups Files Printer Tags WWW}
    set SystemPreferences [lremove -l $AllPreferences \
      $InterfacePreferences ${Input-OutputPreferences} Packages]
    foreach nm [set [join ${which} ""]] {
	lappend args [list $nm $flagPrefs($nm) $varPrefs($nm)]
    }
    
    dialog::is_global {
	dialog::adjust_flags $dial global [dialog::multipage $dial $which $args]
    }
    dialog::cleanup $dial
}

proc dialog::preferences {menu nm} {
    global flagPrefs varPrefs
    if {[string match "Suffix Mappings" $nm]} {
	return [suffixMappings]
    } elseif {[string match "Menus And Features" $nm]} {
	return [global::menusAndFeatures]
    } elseif {[string match "Arrange Menus" $nm]} {
	return [dialog::arrangeMenus]
    } elseif {[string match "Menus" $nm]} {
	return [global::menus]
    } elseif {[string match "Features" $nm]} {
	return [global::features]
    } elseif {[string match "Save Preferences Now" $nm]} {
	return [prefs::saveNow]
    } elseif {[string match "Edit Prefs File" $nm]} {
	return [prefs::tclEdit]
    }
    if {![info exists flagPrefs($nm)]} { 
	set nm "[string toupper [string index $nm 0]][string range $nm 1 end]" 
    }
    if {[string match "*Preferences" $nm]} { return [global::allPrefs $nm] }
    if {$nm == "Packages"} { global::updatePackageFlags }
    if {$nm == "Helper Applications"} { global::updateHelperFlags }
    set dial [dialog::create]
    dialog::is_global {
	dialog::adjust_flags $dial global [dialog::onepage $dial $flagPrefs($nm) $varPrefs($nm) "$nm preferences"]
    }
    dialog::cleanup $dial
}

# Dominique's nice proc to handle all packages at once.
proc global::allPackages {} {
    global package::prefs allFlags dialog::_not_global_pkg dialog::_not_global_flag
    global flagPrefs varPrefs alpha::prefs
    global::updatePackageFlags
    
    set dial [dialog::create]
    
    set args {}
    set dialog::_not_global_pkg {}
    if {[info exists package::prefs]} {
	set pkglist [concat ${package::prefs} [list miscellaneousPackages]]
    } else {
	set pkglist [list miscellaneousPackages]
    }
    foreach pkg [lsort -ignore $pkglist] {
	if {$pkg == "miscellaneousPackages"} {
	    lappend args [list miscellaneousPackages $flagPrefs(Packages) $varPrefs(Packages)]
	    continue
	}
	if {[info exists alpha::prefs($pkg)]} {
	    set pkg [set alpha::prefs($pkg)]
	}
	global ${pkg}modeVars
	set mflags {}
	set mvars {}
	if {[array exists ${pkg}modeVars]} {
	    lappend dialog::_not_global_pkg ${pkg}modeVars
	    foreach v [lsort [array names ${pkg}modeVars]] {
		if {[lsearch -exact $allFlags $v] >= 0} {
		    lappend mflags $v
		} else {
		    lappend mvars $v
		}
	    }
	}
	lappend args [list $pkg $mflags $mvars]
    }

    set values_items [dialog::multipage $dial "Packages preferences" $args]
    set dialog::_not_global_flag {}
    set res [lindex $values_items 0]
    set editItems [lindex $values_items 1]
    set i 0
    set values {}
    set items {}
    foreach item $editItems {
	global $item
	if {[info exists $item]} {
	    lappend values [lindex $res $i]
	    lappend items $item
	} 
	incr i
    }
    dialog::adjust_flags $dial global [list $values $items]
    foreach pkg ${dialog::_not_global_pkg} {
	set i 0
	set values {}
	set items {}
	foreach item $editItems {
	    if {[info exists ${pkg}($item)]} {
		lappend values [lindex $res $i]
		lappend items $item
	    } 
	    incr i
	}
	set dialog::_not_global_flag $pkg
	dialog::adjust_flags $dial arraynamespace [list $values $items] $pkg
    }
    unset dialog::_not_global_pkg
    set dialog::_not_global_flag {}
    
    dialog::cleanup $dial
}

#  Simple queries and alerts  #

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::value_for_variable" --
 # 
 #  Ask for a value, with default given by the given variable, and using
 #  that variable's type (list, file, ...) as a constraint.
 #  
 #  Currently assumes the variable is a list var, but this will change.
 # -------------------------------------------------------------------------
 ##
proc dialog::value_for_variable {var {title ""}} {
    if {$title == ""} { set title [quote::Prettify $var] }
    return [dialog::optionMenu $title [flag::options $var] \
      [uplevel [list set $var]]]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::getAKey" --
 # 
 #  Returns a keystring to be used for binding a key in a menu, 
 #  using a nice dialog box to ask the user.
 # 
 #  Possible improvements: we could replace the dialog
 #  box with a status-line prompt (which would allow the use of
 #  getModifiers to check what keys the user pressed).
 #  
 #  Now handles 'prefixChar' bindings for non-menu items.
 #  i.e. you can use this dialog to bind something to 'ctrl-x ctrl-s',
 #  for instance.
 # 
 #  If the name contains '/' it is considered to be two items,
 #  separated by that '/', which are to take the same binding,
 #  except that one of them will use the option key.
 #  
 #  Similarly '//' means use shift, '///' means shift-option,
 #  For instance 'dialog::getAKey close/closeAll//closeFloat /W<O'
 #  would give you the menu-item for 'close' in the file menu. 
 #  except these last two aren't implemented yet ;-)
 # --Version--Author------------------Changes-------------------------------
 #    1.0     Johan Linde		 original
 #    1.1     <vince@santafe.edu> can do non-menu bindings too
 #    1.2     <vince@santafe.edu> handles arrow keys
 #    1.2.1   Johan Linde        handles key pad keys
 # -------------------------------------------------------------------------
 ##
proc dialog::getAKey {{name {}} {keystr {}} {for_menu 1}} {
    global keys::func
    # two lists for any other keys which look better with a text description
    set otherKeys {"<No binding>" "-" Space}
    set otherKeyChars [list "" "" " "]
    if {!$for_menu} {
	lappend otherKeys Left Right Up Down "Key pad =" \
	  "Key pad /" "Key pad *" "Key pad -" "Key pad +" "Key pad ."
	lappend otherKeyChars "" "" "\x10" "" Kpad= \
	  Kpad/ Kpad* Kpad- Kpad+ Kpad.
	for {set i 0} {$i < 10} {incr i} {
	    lappend otherKeys "Key pad $i"
	    lappend otherKeyChars Kpad$i
	}
    }
    set nname $name
    set shift-opt [expr {![regsub {///} $nname { so-} $nname]}]
    set shift  [expr {![regsub {//} $nname { s-} $nname]}]
    set option [expr {![regsub {/} $nname { o-} $nname]}]
    if {[string length $keystr]} {
	set values "0 0"
	set mkey [keys::verboseKey $keystr normal]
	if {$normal} {
	    lappend values "Normal Key"
	} else {
	    lappend values $mkey
	    set mkey {}
	}
	lappend values [regexp {<U} $keystr]
	lappend values [regexp {<B} $keystr]
	if {!$for_menu} {
	    if {[regexp "(.*)" $keystr "" i]} {
		if {$i == "e"} {
		    lappend values "escape"
		} else {
		    lappend values "ctrl-$i"
		}
	    } else {
		lappend values "<none>"
	    }
	}
	if {$option} {lappend values [regexp {<I} $keystr]}
	lappend values [regexp {<O} $keystr]
	lappend values $mkey
    } else {
	set values {0 0 "" 0 0}
	if {!$for_menu} { lappend values <none> }
	if {$option} {lappend values 0}
	lappend values 0 ""
    }
    if {$for_menu} {
	set title "Menu key binding"
    } else {
	set title "Key binding"
	set prefixes [keys::findPrefixChars]
	foreach i $prefixes {
	    lappend prefix "ctrl-$i"
	}
	lappend prefixes e
	lappend prefix "escape"
    }
    if {$name != ""} { append title " for '$name'" }
    set usep [info exists prefix]
    global alpha::modifier_keys
    while {1} {
	set box ""
	# Build box
	if {[info tclversion] < 8.0} {
	    lappend box -t $title 10 10 315 25
	} else {
	    lappend box -T $title
	}
	lappend box -t Key 10 40 40 55 \
	  -m [concat [list [lindex $values 2]] \
	  [list "Normal key"] $otherKeys ${keys::func}] 80 40 180 57 \
	  -c Shift [lindex $values 3] 10 70 60 85 \
	  -c Control [lindex $values 4] 80 70 150 85
	if {$usep} {
	    lappend box -t Prefix 190 40 230 55  \
	      -m [concat [list [lindex $values 5]]  "<none>" "-" $prefix] \
	      235 40 315 57
	}
	if {$option} {
	    lappend box -c [lindex ${alpha::modifier_keys} 2] \
	      [lindex $values [expr {5 + $usep}]] 160 70 220 85
	}
	lappend box -c [lindex ${alpha::modifier_keys} 0] \
	  [lindex $values [expr {5 + $option +$usep}]] 230 70 315 85
	lappend box -n "Normal key" -e [lindex $values [expr {6 + $option +$usep}]] 50 40 70 55
	set values [eval [concat dialog -w 330 -h 130 -b OK 250 100 315 120 -b Cancel 170 100 235 120 $box]]
	# Interpret result
	if {[lindex $values 1]} {error "Cancel"}
	# work around a little Tcl problem
	regsub "\{\{\}" $values "\\\{" values
	set elemKey [string toupper [string trim [lindex $values [expr {6 + $option +$usep}]]]]
	set special [lindex $values 2]
	set keyStr ""
	if {[lindex $values 3]} {append keyStr "<U"}
	if {[lindex $values 4]} {append keyStr "<B"}
	if {$option && [lindex $values [expr {5 + $usep}]]} {append keyStr "<I"}
	if {[lindex $values [expr {5 + $option +$usep}]]} {append keyStr "<O"}
	if {$usep} {
	    set pref [lindex $values 5]
	    if {$pref != "<none>"} {
		set i [lsearch -exact $prefix $pref]
		append keyStr "[lindex $prefixes $i]"
	    }
	}
	if {[string length $elemKey] > 1 && $special == "Normal key"} {
	    alertnote "You should only give one character for key binding."
	} else {
	    if {$for_menu} {
		if {$special == "Normal key" && [text::Ascii $elemKey] > 126} {
		    alertnote "Sorry, can't define a key binding with $elemKey."
		} elseif {$elemKey != "" && $special == "Normal key" \
		  && ($keyStr == "" || $keyStr == "<U")} {
		    alertnote "You must choose at least one of\
		      the modifiers control, option and command."
		} elseif {![regexp {F[0-9]} $special] && $special != "Tab"\
		  && $special != "Normal key" && $special != "<No binding>"\
		  && $keyStr == ""} {
		    alertnote "You must choose at least one modifier."
		} else {
		    break
		}
	    } else {
		break
	    }
	}
    }
    if {$special == "<No binding>"} {set elemKey ""}
    if {$special != "Normal key" && $special != "<No binding>"} {
	if {[set i [lsearch -exact $otherKeys $special]] != -1} {
	    set elemKey [lindex $otherKeyChars $i]
	} else {
	    set elemKey [text::Ascii [expr {[lsearch -exact ${keys::func} $special] + 97}] 1]
	}
    }
    if {![string length $elemKey]} {
	set keyStr ""
    } else {
	append keyStr "/$elemKey"
    }	
    return $keyStr
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::optionMenu" --
 # 
 #  names is the list of items.  An item '-' is a divider, and empty items
 #  are not allowed.
 # -------------------------------------------------------------------------
 ##
proc dialog::optionMenu {prompt names {default ""} {index 0}} {
    if {$default == ""} {set default [lindex $names 0]}
    
    set y 5
    set w [expr {[string length $prompt] > 20 ? 350 : 200}]
    if {[string length $prompt] > 60} { set w 500 }
    
    # in case we need a wide pop-up area that needs more room
    set popUpWidth [eval dialog::_reqWidth $names]
    set altWidth [expr {$popUpWidth + 60}]
    set w [expr {$altWidth > $w ? $altWidth : $w}]
    
    set dialog [dialog::text $prompt 5 y [expr {int($w/6.7)}]]
    incr y 10
    eval lappend dialog [dialog::menu 30 y $names $default $popUpWidth]
    incr y 20
    eval lappend dialog [dialog::okcancel [expr {20 - $w}] y 0]
    set res [eval dialog -w $w -h $y $dialog]
    
    if {[lindex $res 2]} { error "Cancel" } 
    # cancel was pressed
    if {$index} {
	# we have to take out the entries correponding to pop-up 
	# menu separator lines -trf
	set possibilities [lremove -all $names "-"]
	return [lsearch -exact $possibilities [lindex $res 0]]
    } else {
	return [lindex $res 0]
    }
}

proc dialog::getDate {{prompt "Please type your date, or use the\
  button below"} {date ""}} {
    while {1} {
	set y 5
	set w 400
	if {[info tclversion] >= 8.0} {
	    set dialog [list -T "Select Date"]
	} else {
	    set dialog [list]
	}
	
	eval lappend dialog [dialog::text $prompt 5 y [expr {int($w/6.7)}]]
	incr y 10
	if {![info exists formattedDate]} {
	    if {$date != ""} {
		set formattedDate [clock format $date]
	    } else {
		set formattedDate ""
	    }
	}
	eval lappend dialog [dialog::edit $formattedDate 10 y 35]
	incr y 35
	eval lappend dialog [dialog::okcancel [expr {20 - $w}] y 0]
	incr y -55
	eval lappend dialog [dialog::button "Get modification\
	  date from a file" 10 y \
	  "Check date format" 260 y]

	incr y 30
	set res [eval dialog -w $w -h $y $dialog]
	
	set formattedDate [string trim [lindex $res 0]]

	if {[lindex $res 3]} {
	    # pick file
	    if {![catch {getfile "Pick file from which to get\
	      modification date"} file]} {
		set date [file mtime $file]
		unset formattedDate
	    }
	} elseif {[lindex $res 4]} {
	    # check format
	    if {[catch {clock scan $formattedDate} newdate]} {
		alertnote "There was an error interpreting your date: $newdate"
	    } else {
		alertnote "I understood '[clock format $newdate]'"
	    }
	} elseif {[lindex $res 1]} { 
	    # ok, trim the result in case it was pasted in with
	    # spaces/new-lines before or after
	    if {[catch {clock scan $formattedDate} newdate]} {
		alertnote "There was an error interpreting your date: $newdate"
	    } else {
		return $newdate
	    }
	} elseif {[lindex $res 2]} { 
	    # cancel
	    error "Cancel" 
	}
    }
}

proc dialog::getUrl {{prompt "Please type your url, or use one\
  of the buttons below"} {url ""}} {
    while {1} {
	set y 5
	set w 380
	if {[info tclversion] >= 8.0} {
	    set dialog [list -T "Select URL"]
	} else {
	    set dialog [list]
	}
	
	eval lappend dialog [dialog::text $prompt 5 y [expr {int($w/6.7)}]]
	incr y 10
	eval lappend dialog [dialog::edit $url 10 y 35]
	incr y 35
	eval lappend dialog [dialog::okcancel [expr {20 - $w}] y 0]
	incr y -55
	eval lappend dialog [dialog::button "Pick local file?" 10 y \
	  "Use foremost browser page" 150 y]
	incr y 30
	set res [eval dialog -w $w -h $y $dialog]
	
	if {[lindex $res 3]} {
	    # pick local file
	    if {[string range $url 0 6] == "file://"} {
		set default [string range $url 7 end]
	    } else {
		set default ""
	    }
	    if {![catch {getfile "Pick local file to use as url" $default} file]} {
		set url [file::toUrl $file]
	    }
	} elseif {[lindex $res 4]} {
	    # use browser page
	    if {[catch {url::browserWindow} res]} {
		alertnote "Can't get that information: $res"
	    } else {
		set url $res
	    }
	} elseif {[lindex $res 1]} { 
	    # ok, trim the result in case it was pasted in with spaces/new-lines
	    # before or after
	    return [string trim [lindex $res 0] ]
	} elseif {[lindex $res 2]} { 
	    # cancel
	    error "Cancel" 
	}
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::alert" --
 # 
 #  Identical to 'alertnote' but copes with larger blocks of text, and
 #  resizes to that text as appropriate.
 # -------------------------------------------------------------------------
 ##
proc dialog::alert {args} {
    if {[catch {eval [list dialog::yesno -y "OK" -n ""] $args}]} {
	# probably ran into a problem with Alpha 7's dialogs
	alertnote [string range [join $args " "] 0 250]
    }
}

proc dialog::errorAlert {args} {
    eval dialog::alert $args
    error [lindex $args 0]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::yesno" --
 # 
 #  Make a dialog with between 1 and 3 buttons, representing '1', '0' and
 #  error "Cancel" respectively.  The names of the first two can be given
 #  with '-y name' and '-n name' respectively.  The cancel button is
 #  only used if a '-c' flag is given (and its name is fixed).
 #  
 #  The procedure automatically sizes the dialog and buttons to fit the
 #  enclosed text.
 # -------------------------------------------------------------------------
 ##
proc dialog::yesno {args} {
    global alpha::platform
    
    # too long for Alpha's standard dialog
    getOpts {-y -n}
    set prompt [lindex $args 0]
    
    # Where possible, use the Mac alert dialog
    if {(${alpha::platform} == "alpha") \
      && [llength [info commands ::alert]] > 0 \
      && [string length $prompt] < 256} {
	# We can't use [askyesno] because it doesn't allow
	# changing the button names
	set cmd [list ::alert -t caution]
	if {[info exists opts(-y)] && $opts(-y) != ""} {
	    set yes $opts(-y)
	} else {
	    set yes "Yes"
	}
	lappend cmd -k $yes
	
	if {[info exists opts(-n)] && $opts(-n) != ""} {
	    set no $opts(-n)
	} else {
	    set no "No"
	}
	lappend cmd -c $no
	
	if {[info exists opts(-c)]} {
	    lappend cmd -o "Cancel" -C other
	} else {
	    lappend cmd -o "" -C none
	}
	lappend cmd $prompt
	
	switch -- [eval $cmd] [list	\
	    $yes {			\
		return 1		\
	    }				\
	    $no {			\
		return 0		\
	    }				\
	    default {			\
		error "cancel"		\
	    }				\
	]
    } else {
	set y 5
	set w [expr {[string length $prompt] > 20 ? 350 : 200}]
	if {[string length $prompt] > 60} { set w 500 }
	
	set dialog [dialog::multipagetext $prompt 5 y [expr {int($w/6.7)}]]
	incr y 10
	set y0 $y
	while {1} {
	    set buttons ""
	    set y $y0
	    set x [expr {-($w - 20)}]
	    if {[info exists opts(-y)] && $opts(-y) != ""} {
		lappend buttons $opts(-y) "" y
	    } else {
		lappend buttons "Yes" "" y
	    }
	    if {[info exists opts(-c)]} {
		lappend buttons "Cancel" "" y
	    }
	    if {[info exists opts(-n)]} {
		if {$opts(-n) != ""} {
		    lappend buttons $opts(-n) "" y
		}
	    } else {
		lappend buttons "No" "" y
	    }
	    set butts [eval dialog::button $buttons]
	    if {$x > 0} {
		incr w $x
	    } else {
		break 
	    }
	}
	set dialog [concat $butts $dialog]
	set res [eval dialog -w $w -h $y $dialog]
	if {[lindex $res 0]} {
	    return 1
	} elseif {(![info exists opts(-c)] && [lindex $res 1]) || ([info exists opts(-c)] && [lindex $res 2])} {
	    return 0
	} else {
	    error "cancel"
	}
    }
}

proc dialog::password {{msg "Please enter password:"}} {
    set values [dialog -w 300 -h 90 -t $msg 10 20 290 35 \
      -e "" 10 40 290 42 -b OK 220 60 285 80 -b Cancel 140 60 205 80]
    if {[lindex $values 2]} {error "Cancel"}
    return [lindex $values 0]
}

proc dialog::logon {pkg {msg "Log on as"} {connect "Connect"} \
  {cancel "Cancel"} {var ""}} {
    set y 20

    global dialog::_not_global_flag
    if {$var == ""} {
	set dialog::_not_global_flag ${pkg}modeVars
    } else {
	set dialog::_not_global_flag $var
    }
    
    if {$msg != ""} {
	set dialog [dialog::title $msg 480]
	incr y 25
    }

    set dial [dialog::create]

    eval lappend dialog [dialog::buildSection $dial {userName userPassword} y]
    incr y 10
    set x 300
    eval lappend dialog [dialog::button $connect "" y $cancel "" y ]
    set res [eval dialog -w 480 -h $y $dialog]
    
    dialog::modified $dial userName [lindex $res 0]
    dialog::modified $dial userPassword [lindex $res 1]        
    
    dialog::cleanup $dial
    
    # 1 if "connect", 0 if "cancel"
    return [lindex $res 2]
}

#  Finding applications  #


proc dialog::askFindApp {var sig} {
    if {$sig == ""} {
	set text "Currently unassigned.   Set?"
    } elseif {[catch {nameFromAppl '$sig'} name]} {
	set text "App w/ sig '$sig' doesn't seem to exist.   Change?"
    } else {
	set text "Current value is '$name'.   Change?"
    }
    if {[dialog::yesno $text]} {
	set nsig [dialog::findApp $var "" $sig]
	set app [nameFromAppl $nsig]
	if {[dialog::yesno "Are you sure you want to set $var to '$nsig'\
	  (mapped to '$app')?"]} {
	    return $nsig
	}
    }
    return ""
}

# The optional third argument can be used to prompt the user
# with the 'old' value 
proc dialog::findApp {var {prompt ""} {sig ""}} {
    global $var ${var}s

    set s 0
    
    lappend sigs $var
    if {[info exists ${var}s]} {
	# have a list of items
	eval [list lappend sigs] [set ${var}s]
    }
    
    set itempaths [list]
    set items [list]
    
    foreach f $sigs {
	set paths [app::getPathsFromSig $f]
	eval [list lappend itempaths] $paths
	incr s [llength $paths]
	# This could be problematic if this call drops some
	# of the paths.
	eval [list lappend items] [file::minimalDistinctTails $paths]
    }
    
    if {$s} {
	lappend items "-" "Locate manually"
	if {![string length $prompt]} {
	    set prompt "Select a new helper for '$var':"
	}
	if {[catch {dialog::optionMenu $prompt \
	  $items "" 1} p]} {
	    return ""
	}
	# we removed a bunch of items above, so have to look here
	if {$p < $s} {
	    return [getFileSig [lindex $itempaths $p]]
	}
    }
    if {!$s || $p >= $s} {
	set nsig [dialog::_findApp $var $prompt $sig]
	if {$nsig != ""} {
	    if {[lsearch $sigs $nsig] == -1} {
		lappend ${var}s $nsig
		prefs::modified ${var}s
	    }
	}
    } else {
	set nsig [lindex $sigs $p]
    }
    return $nsig
}

proc dialog::findAnyApp {{prompt "Locate application:"}} {
    if {[catch {getfile $prompt} path]} {return ""}
    return $path
}

proc dialog::_findApp {var prompt {sig ""}} {
    global alpha::platform
    if {${alpha::platform} == "alpha"} {
	set dir ""
    } else {
	set dir [file dirname $sig]
    }
    if {[catch {getfile $prompt $sig} path]} { return "" }
    set nsig [getFileSig $path]
    set app [nameFromAppl $nsig]
    if {$app != $path} {
	alertnote "Appl sig '$nsig' is mapped to '$app', not '$path'.\
	  Remove the former, or rebuild your desktop."
	return ""
    }
    return $nsig
}

#  Global/mode menus  #

proc dialog::arrangeMenus {} {
    global global::features index::feature
    set globalMenus {}
    set menus [eval concat [lrange [package::partition global] 0 2]]
    foreach pkg [set global::features] {
	if {[lsearch -exact $menus $pkg] == -1} { continue }
	if {![alpha::isPackageInvisibleToUser $pkg]} {
	    lappend globalMenus $pkg
	}
    }
    # Make sure we don't have any duplicates for bizarre reasons
    set globalMenus [lunique $globalMenus]
    # Now we have the global menus in the current order, and
    # we need to arrange them
    if {[catch {dialog::arrangeItems $globalMenus} newOrder]} {
	return
    }
    if {$newOrder != $globalMenus} {
	# Now if we didn't cancel, we remove them all from the list and
	# then add them in the new order at the end
	set global::features [lremove -l -all [set global::features] $newOrder]
	eval lappend global::features $newOrder
	# Now replace the menus in the correct order.
	status::msg "Re-ordering the menus"
	foreach menuName $newOrder {
	    menu::moveToEnd $menuName
	}
	# Now shove any mode menus to the rear.  There are quicker
	# ways to do this than changing mode, but changing mode is
	# probably the easiest way to code.
	if {[llength [winNames]]} {
	    global mode
	    set oldMode $mode
	    changeMode ""
	    changeMode $oldMode
	}
	status::msg "The new order has been established."
    }
}

proc dialog::arrangeItems {origItems} {
    set msg "(Use the following order:)"
    set items [list $msg]
    set items [concat $items $origItems]
    set p1 "Select item to move:"
    set L  [lindex $items 1]
    while {1} {
	if {[catch {listpick -p $p1 -L $L $items} item]} {
	    break
	}
	if {$item == "" || $item == $msg} {
	    break
	}
	set p2 "Move '$item' to position:"
	set where [lsearch -exact $items $item]
	if {![catch {prompt $p2 $where or first last} res]} {
	    switch -- $res {
		0 - first {set res 1}
		last      {set res [llength $items]}
		default   {
		    if {![is::UnsignedInteger $res]} {
			alertnote "Please enter a position number."
			set L $item
			continue
		    } elseif {$res > [llength $items]} {
			set res [llength $items]
		    }
		}
	    }
	    set index [lsearch -exact $items $item]
	    set items [lreplace $items $index $index]
	    set items [linsert $items $res $item]
	    set p1 "Move another item?"
	    set L $msg
	}
    }
    set items [lrange $items 1 end]
    if {($items == $origItems) \
      || ![dialog::yesno -y "Accept" -n "Cancel" "New order is: $items"]} {
	status::errorMsg "Cancelled"
    }
    return $items
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::pickMenusAndFeatures" --
 # 
 #  Prompt the user to select menus and features either globally or
 #  for a given mode.  We need to make sure that those items in
 #  the mode-list which are also in the global list aren't forgotten
 #  (since they are removed from the dialog).
 #  
 #  'mfb' is 0 for both menus and features
 #           1 for just menus
 #           2 for just features
 #           
 #  This procedure should be pretty clear now, having been rewritten.
 #  However, here are a few tips:
 #  
 #  Each page of the dialog may contain 2 or 3 sections.  The items
 #  to use in these sections are taken from the variables:
 #  
 #  menus1, menus2, menus3
 #  features1, features2, features3
 #  off1, off2
 #  
 #  Where if the variable is empty, the entire section is omitted.
 #  Furthermore, all 'always on' items are ignored, and for mode
 #  dialogs, anything which is globally on is moved from the menus
 #  or features pages to the 'off' pages.
 # -------------------------------------------------------------------------
 ##
proc dialog::pickMenusAndFeatures {formode {mfb 0}} {
    global mode::features global::features index::flags index::feature
    set all [package::partition $formode]
    set menus1 [lindex $all 0]
    set menus2 [lindex $all 1]
    set menus3 [lindex $all 2]
    set features1 [lindex $all 3]
    set features2 [lindex $all 4]
    set features3 [lindex $all 5]
    unset all
    
    if {[info tclversion] >= 8.0} {
	set help {}
    }
    # decide on two or three column
    #set endw [expr [llength $all] > 50 ? 560 : 380]
    set endw 560

    if {$formode == "global"} {
	set chosen ${global::features}
	set prefix "Select global #"
	set maintypes [list Usual "" "Other possible"]
    } else {
	set chosen {}
	set extras_off {}
	set off1 {}
	set off2 {}
	foreach pkg [mode::getFeatures $formode] {
	    if {[string index $pkg 0] == "-"} {
		set pkg [string range $pkg 1 end]
		if {[lsearch -exact ${global::features} $pkg] != -1} {
		    # these are the ones which are disabled
		    lappend extras_off $pkg
		}
	    } else {
		# These are items which are on for this mode.  Any of these
		# which are also on globally go in the first group of 'off' 
		# items.  The rest in the first pages.
		lappend chosen $pkg
	    }
	}
	foreach pkg [set global::features] {
	    if {[lsearch -exact $chosen $pkg] != -1} {
		# The top group of items
		if {[lindex [set index::feature($pkg)] 2] == 1} {
		    # it's a menu
		    if {$mfb != 2} {lappend off1 $pkg}
		} else {
		    # it's not a menu
		    if {$mfb != 1} {lappend off1 $pkg}
		}
	    } else {
		# The second group of items
		if {[lindex [set index::feature($pkg)] 2] == 1} {
		    # it's a menu
		    if {$mfb != 2} {lappend off2 $pkg}
		} else {
		    # it's not a menu
		    if {$mfb != 1} {lappend off2 $pkg}
		}
	    }
	}
	
	set prefix "Select # for mode '$formode'"
	set maintypes [list Usual General "Other possible"]
	set multipage 1
    }
    while 1 {
	set maxh 0
	set box ""
	#set names {}
	foreach type {menus features off} {
	    if {$formode == "global" && $type == "off"} {continue}
	    if {$mfb > 0} {
		if {$mfb == 1 && $type == "features"} {continue}
		if {$mfb == 2 && $type == "menus"} {continue}
	    }
	    set w 20
	    set h 45
	    set i 0
	    if {$type == "off"} {
		set subm "Turn items off"
		set types [list "Usually on for this mode" "Uncheck to disable"]
	    } else {
		regsub "\#" $prefix $type subm
		set types $maintypes
	    }
	    set page 1
	    if {![info exists names0]} {
		lappend names0 $subm
		lappend names $subm
	    }
	    lappend names $subm
	    lappend box "-n" $subm
	    if {$type == "off"} {
		lappend box -t "These items are currently globally on.\
		  You can turn them off just for this mode here." \
		  10 $h [expr {$endw -20}] [expr {$h +15}]
		incr h 20
	    }
	    foreach block $types {
		incr i
		if {[llength [set ${type}$i]] == 0} {
		    continue
		}
		if {($type == "off")} {
		    lappend box -t "$block:"
		} else {
		    lappend box -t "$block $type:" 
		}
		lappend box 10 $h [expr {$w +160}] [expr {$h +15}]
		incr h 20
		foreach m [set ${type}$i] {
		    if {[alpha::isPackageInvisibleToUser $m]} {
			continue
		    }
		    if {$h > 360} {
			if {$h > $maxh} {set maxh $h}
			incr page
			lappend names "$subm page $page"
			lappend box "-n" "$subm page $page"
			set h 45
			lappend box -t "$block $type continued..." 10 $h \
			  [expr {$w +260}] [expr {$h +15}]
			incr h 20
		    }
		    set name [quote::Prettify $m]
		    if {[info exists tmpcurrent]} {
			# Second or more times through we just recreate what we
			# have so far
			set tick [lindex $tmpcurrent $ii]
			incr ii
		    } else {
			# First time through, we need to work out whether each item
			# is on or off, and rememeber all the items.
			if {$type == "off"} {
			    set tick [expr {([lsearch -exact $extras_off $m] < 0)}]
			    lappend orig [list "off" $m $tick]
			} else {
			    set tick [expr {([lsearch -exact $chosen $m] >= 0)}]
			    lappend orig [list "on" $m $tick]
			}
		    }
		    lappend box -c $name $tick $w $h  [expr {$w + 160}] [expr {$h + 15}]
		    if {[info tclversion] >= 8.0} {
			lappend help [dialog::packagehelp $m 1]
		    }
		    incr w 180
		    if {$w == $endw} {set w 20; incr h 20}
		}
		if {$w != 20} {
		    incr h 30 ; set w 20
		}
	    }
	    if {$h > $maxh} {set maxh $h}
	    
	}
	set h $maxh
	incr h 20
	
	if {[llength $names] == 2} {
	    set offset 4
	    set name_piece [list -t [lindex $names 0]]
	    if {[set nindex [lsearch -exact $box -n]] != -1} {
		set box [lreplace $box $nindex [incr nindex]]
	    }
	    set singlepage 1
	} else {
	    set offset 5
	    set name_piece [list -m $names]
	    set singlepage 0
	}
	
	if {[info tclversion] >= 8.0} {
	    if {$formode == "global"} {
		set title "Global"
	    } else {
		set title "$formode"
	    }
	    switch -- $mfb {
		0 { append title " menus and features" }
		1 { append title " menus" }
		2 { append title " features" }
	    }
	    lappend box -T $title
	    set values [eval [concat dialog -w $endw -h [expr {$h + 30}] \
	      -b OK [expr {$endw -80}] $h [expr {$endw -15}] [expr {$h + 20}] \
	      -b Cancel [expr {$endw -160}] $h [expr {$endw -95}] [expr {$h + 20}]  \
	      -b Help [expr {$endw -400}] $h [expr {$endw - 340}] [expr {$h + 20}] \
	      -b Descriptions [expr {$endw -320}] $h [expr {$endw -220}] [expr {$h + 20}] \
	      $name_piece [expr {($endw - 220)/2}] 10 $endw 30 $box\
	      -help] [list [concat [list \
	      "Click here to save the current settings." \
	      "Click here to discard any changes you've made to the settings." \
	      "Click here to access help on each item in this dialog." \
	      "Click here to access descriptions of each item in this dialog."] \
	      [expr {$singlepage ? "" : {"Use this popup menu, or the cursor keys to select a \
	      different page of preferences."}}] \
	      $help]]]
	} else {
	    set values [eval [concat dialog -w $endw -h [expr {$h + 30}] \
	      -b OK [expr {$endw -80}] $h [expr {$endw -15}] [expr {$h + 20}] \
	      -b Cancel [expr {$endw -160}] $h [expr {$endw -95}] [expr {$h + 20}]  \
	      -b Help [expr {$endw -400}] $h [expr {$endw - 340}] [expr {$h + 20}] \
	      -b Descriptions [expr {$endw -320}] $h [expr {$endw -220}] [expr {$h + 20}] \
	      $name_piece [expr {($endw - 220)/2}] 10 $endw 30 $box]]
	}
	
	if {[llength $names] > 2} {
	    set names0 [list [lindex $values 4]]
	}
	set names $names0
	if {[lindex $values 0]} {
	    set tmpcurrent [lrange $values $offset end]
	    # Ok
	    break
	}
	if {[lindex $values 1]} {
	    # Cancel
	    return
	}
	if {[lindex $values 2]} {
	    dialog::describeMenusAndFeatures Help
	}
	if {[lindex $values 3]} {
	    dialog::describeMenusAndFeatures Describe
	}	
	set tmpcurrent [lrange $values $offset end]
	set ii 0
	#unset names0
    }
    set res_on {}
    set res_off {}
    
    global mode
    
    for {set iter 0} {$iter < 2} {incr iter} {
	for {set i 0} {$i < [llength $tmpcurrent]} {incr i} {
	    set choice [lindex $tmpcurrent $i]
	    set original [lindex $orig $i]
	    set onoff [lindex $original 0]
	    set m [lindex $original 1]
	    set tick [lindex $original 2]
	    if {$onoff == "on"} {
		# From the 'on' section of the dialog
		if {$choice && !$tick} {
		    # Only turn items on on the second iteration
		    if {$iter} {
			lappend res_on $m
			if {![package::do_activate $m]} {
			    if {$formode == "global"} {
				lappend global::features $m
			    } else {
				lappend mode::features($formode) $m
				prefs::modified mode::features($formode)
			    }
			} else {
			    package::throwActivationError
			}
		    }
		} elseif {!$choice && $tick} {
		    # Only turn items off on the first iteration
		    if {!$iter} {
			lappend res_off $m
			if {![package::tryToDeactivate $m terse]} {
			    if {$formode == "global"} {
				set global::features [lremove [set global::features] $m]
			    } else {
				set mode::features($formode) [lremove [set mode::features($formode)] $m]
				prefs::modified mode::features($formode)
			    }
			}
		    }
		}
	    } else {
		# From the 'off' section of the dialog
		if {$formode == "global"} {error "Shouldn't be here"}
		if {$choice && !$tick} {
		    # Only turn items on on the second iteration
		    if {$iter} {
			lappend res_on $m
			if {![package::do_activate $m]} {
			    # It is on globally, and we previously turned it off for this mode
			    set mode::features($formode) [lremove -- [set mode::features($formode)] "-$m"]
			    prefs::modified mode::features($formode)
			} else {
			    package::throwActivationError
			}
		    }
		} elseif {!$choice && $tick} {
		    # Only turn items off on the first iteration
		    if {!$iter} {
			lappend res_off $m
			if {![package::tryToDeactivate $m terse]} {
			    # It is on globally, and we now turn it off for this mode
			    lappend mode::features($formode) "-$m"
			    prefs::modified mode::features($formode)
			}
		    }
		}
	    }
	}
    }
}

proc dialog::describeMenusAndFeatures {{what "Help"}} {
    set all [package::partition]
    set okmenu [lindex $all 0]
    set okfeature [lindex $all 1]
    set okmode [lindex $all 2]
    set all [eval concat $all]
    # decide on two or three column
    set endw [expr {[llength $all] > 50 ? 560 : 380}]
    if {$what == "Help"} {
	set prefix "Read help for a #"
    } else {
	set prefix "Describe a #"
    }
    foreach m {menu feature mode} {
	regsub "\#" $prefix $m subm
	lappend names $subm
    }
    lappend box -m [concat [list [lindex $names 0]] $names] \
      [expr {($endw - 150)/2}] 10 $endw 30
    set maxh 0
    set wincr 160
    foreach type {menu feature mode} {
	set w 20
	set h 45
	regsub "\#" $prefix $type subm
	lappend box "-n" $subm
	if {$type == "mode"} {set wincr 70}
	foreach m [set ok$type] {
	    set name [quote::Prettify $m]
	    lappend box -b $name $w $h [expr {$w + $wincr}] [expr {$h + 15}]
	    incr w [expr {$wincr +20}]
	    if {$w == $endw} {set w 20; incr h 20}
	}
	if {$w > 20} {set w 20; incr h 20}
	if {$h > $maxh} {set maxh $h}
    }
    set h $maxh
    incr h 20
    while 1 {
	if {[catch {set values [eval [concat [list dialog -w $endw -h [expr {$h + 30}] \
	  -b OK [expr {$endw - 80}] $h [expr {$endw - 15}] [expr {$h + 20}]] $box]]}]} {
	    return [dialog::_simpleDescribeMenusAndFeatures $what]
	}
	if {[lindex $values 0]} {return}
	# we hit a button
	for {set i 0} {$i < [llength $all]} {incr i} {
	    if {[lindex $values [expr {$i + 2}]]} {
		if {$what == "Help"} {
		    package::helpFile [lindex $all $i]
		} else {
		    package::describe [lindex $all $i]
		}
		break
	    }
	}
    }
}

proc dialog::_simpleDescribeMenusAndFeatures {{what "Help"}} {
    if {$what == "Help"} {
	set prefix "Read help for"
    } else {
	set prefix "Describe"
    }
    set types [list Menus Packages Modes]
    set i 0
    foreach set [package::partition] {
	lappend choices "               [lindex $types $i]"
	eval lappend choices $set
	incr i
    }
	
    if {![catch {listpick -p $prefix $choices} pkg]} {
	if {[string index $pkg 0] == " "} { return }
	if {$what == "Help"} {
	    package::helpFile $pkg
	} else {
	    package::describe $pkg
	}
    }
}

#  Multiple bindings dialogs  #

proc dialog::arrayBindings {name array {for_menu 0}} {
    upvar $array a
    foreach n [array names a] {
	lappend l [list $a($n) $n]
    }
    if {[info exists l]} {
	eval dialog::adjustBindings [list $name modified "" $for_menu] $l
    }
    array set a [array get modified]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::adjustBindings" --
 # 
 #  'args' is a list of pairs.  The first element of each pair is the 
 #  menu binding, and the second element is a descriptive name for the
 #  element. 'array' is the name of an array in the calling proc's
 #  scope which is used to return modified bindings.
 # 
 # Results:
 #  
 # --Version--Author------------------Changes-------------------------------
 #    1.0     Johan Linde			   original for html mode
 #    1.1     <vince@santafe.edu> general purpose version
 #    1.2     Johan Linde              split into two pages when many items
 # -------------------------------------------------------------------------
 ##
proc dialog::adjustBindings {name array {rmod {}} {for_menu 1} args} {
    global screenHeight
    regsub -all {\"\(-\"} $args "" items
    upvar $array key_changes
    
    foreach it $items {
	if {[info exists key_changes([lindex $it 1])]} {
	    set tmpKeys([lindex $it 1]) $key_changes([lindex $it 1])
	} else {
	    set tmpKeys([lindex $it 1]) [lindex $it 0]
	}
    }
    # do we return modified stuff?
    if {$rmod != ""} { upvar $rmod modified }
    set modified ""
    set page "Page 1 of $name"
    # Can't currently set in place, so this is ok.
    set mod ""
    while {1} {
	# Build dialog.
	set twoWindows 0
	set box ""
	set h 30
	foreach it $items {
	    if {$it == "(-"} {continue}
	    set w 210
	    set w2 370
	    set key $tmpKeys([lindex $it 1])
	    set key1 [dialog::specialView::binding $key]
	    set it2 [split [lindex $it 1] /]
	    if {[llength $it2] == 1} {
		lappend box -t [lindex $it2 0] 65 $h 205 [expr {$h + 15}] -t $key1 $w $h $w2 [expr {$h + 15}]
		eval lappend box [dialog::buttonSet $mod 10 $h]
		incr h 17
	    } else {
		lappend box -t [lindex $it2 0] 65 $h 205 [expr {$h + 15}] -t $key1 $w $h $w2 [expr {$h + 15}]
		eval lappend box [dialog::buttonSet $mod 10 [expr {$h +8}]]
		incr h 17
		if {$key1 != "<no binding>"} {regsub {((ctrl-)?(shift-)?)(.*)} $key1 {\1opt-\4} key1}
		lappend box -t [lindex $it2 1] 65 $h 205 [expr {$h + 15}] -t $key1 $w $h $w2 [expr {$h + 15}]
		incr h 17
	    }
	    if {$it != [lindex $items [expr {[llength $items] -1}]] && !$twoWindows && [set twoWindows [expr {$h + 100 > $screenHeight}]]} {
		set box " -n [list [concat Page 1 of $name]] $box -n [list [concat Page 2 of $name]] "
		set hmax $h; set h 30
	    }
	}
	if {[info exists hmax]} {set h $hmax}
	if {$twoWindows} {
	    set top "-m [list [list $page [concat Page 1 of $name] [concat Page 2 of $name]]] 10 10 370 25"
	} else {
	    set top [dialog::title $name 250]
	}
	set buttons "-b OK 300 [expr {$h + 10}] 365 [expr {$h + 30}] -b Cancel 220 [expr {$h + 10}] 285 [expr {$h + 30}]"
	set values [eval [concat dialog -w 380 -h [expr {$h + 40}] $buttons $top $box]]
	if {$twoWindows} {set page [lindex $values 2]}
	if {[lindex $values 1]} {
	    # Cancel
	    return "Cancel"
	} elseif {[lindex $values 0]} {
	    # Save new key bindings
	    foreach it $modified {
		set key_changes($it) $tmpKeys($it)
	    }
	    return
	} else {
	    # Get a new key.
	    set it [lindex [lindex $items [expr {[lsearch $values 1] - 2 - $twoWindows}]] 1]
	    if {![catch {dialog::getAKey $it $tmpKeys($it) $for_menu} newKey]  && $newKey != $tmpKeys($it)} {
		set tmpKeys($it) $newKey
		lappend modified $it
	    }
	}
    }
}


