## -*-Tcl-*- (nowrap)
 # ###################################################################
 #  AlphaTcl - core Tcl engine
 # 
 #  FILE: "dialogsUtils.tcl"
 #                                    created: 01-08-13 17.38.38 
 #                                last update: 11/12/2001 {17:44:05 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Much copyright (c) 1997-2001  Vince Darley, all rights reserved, 
 # rest Pete Keleher, Johan Linde.
 # 
 # Reorganisation carried out by Vince Darley with much help from Tom 
 # Fetherston, Johan Linde and suggestions from the alphatcl-developers mailing list.  
 # Alpha is shareware; please register with the author using the register 
 # button in the about box.
 # 
 # This file contains helper procedures used by the other dialogs
 # code.  Most likely you will not need to call these procedures
 # directly in your code, unless you are really building your own
 # dialogs piece by piece.
 # ###################################################################
 ##

namespace eval dialog {}
namespace eval global {}
namespace eval flag {}

#  Dialog sub-panes  #

ensureset dialog::_not_global_flag ""

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::flag" --
 # 
 #  Helper procedure.
 #  
 #  Builds a dialog-box page to be used for setting global/mode/package
 #  preferences.  It can contain preferences for flags (on/off), variables,
 #  list items, mode items, files, folders, apps,...
 # 
 # Results:
 #  part of a script to generate the dialog
 # 
 # Side effects:
 #  sets maxT to the maximum height desired by the dialog
 # 
 # --Version--Author------------------Changes-------------------------------
 #    1.0     Pete Keleher             original
 #    2.0     <vince@santafe.edu> much more sophisticated (and complex!)
 # -------------------------------------------------------------------------
 ##
proc dialog::flag {mod mflags mvars {left 20} {top 40} {title {}}} {
    global maxT spelling alpha::prefNames dialog::_not_global_flag mode \
      includeDescriptionsInDialogs index::flags
    if {$includeDescriptionsInDialogs || [info tclversion] >= 8.0} {
	cache::readContents index::prefshelp
	if {[info tclversion] >= 8.0} {
	    upvar help help
	}
	if {[regsub {(modeVars)?$} ${dialog::_not_global_flag} "" vprefix]} {
	    append vprefix ","
	}
    }
    
    set args [list]
    if {$title != ""} {
	lappend args "-t" $title 30 10 400 25
	incr top 25
    } else {
	# Make room for the help button
	incr top 15
    }
    
    # if variable names are very long, switch to 2 columns
    if {$includeDescriptionsInDialogs} {
	set perRow 1
	set width 450
    } else {
	if {([maxListItemLength $mflags] > 18)} {
	    set perRow 2
	    set width 225
	} else {
	    set perRow 3
	    set width 150
	
	}
    }
    set height	15
    
    set ind 0
    set l $left
    foreach f $mflags {
	set fname [quote::Prettify $f]
	if {$spelling} {text::british fname}
	if {$includeDescriptionsInDialogs} {
	    if {[info exists prefshelp($vprefix$f)]} {
		incr top 10
		eval lappend args [dialog::text \
		  [dialog::helpdescription $prefshelp($vprefix$f)] $l top 90]
		incr top -14
	    } elseif {[info exists prefshelp($mode,$f)]} {
		incr top 10
		eval lappend args [dialog::text \
		  [dialog::helpdescription $prefshelp($mode,$f)] $l top 90]
		incr top -14
	    }
	}
	if {[info tclversion] < 8.0} {
	    lappend args "-c" $fname [dialog::getFlag $mod $f] \
	      $l $top [incr l $width] [expr {$top + $height}]
	} else {
	    lappend args "-c" $fname [dialog::getFlag $mod $f] -font 2 \
	      $l $top [incr l $width] [expr {$top + $height}]
	}
	if {[incr ind] % $perRow == 0} { set l $left ; incr top $height }
	if {[info tclversion] >= 8.0} {
	    if {[info exists prefshelp($vprefix$f)]} {
		lappend help $prefshelp($vprefix$f)
	    } elseif {[info exists prefshelp($mode,$f)]} {
		lappend help $prefshelp($mode,$f)
	    } elseif {[lsearch -exact [set index::flags] $f] != -1} {
		lappend help [dialog::packagehelp $f 1]
	    } else {
		lappend help ""
	    }
	}
    }
    
    if {$ind} {
	set top [expr {$top + 20}]
	lappend args -p 100 [expr {$top + 27}] 300 [expr {$top + 28}]
    } 
    
    dialog::buildSection $mod $mvars top 440 $left args alpha::prefNames
    incr top 16
    
    if {$top > $maxT} {set maxT $top}
    return $args
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::buildSection" --
 # 
 #  Build a dialog box section for a bunch of preferences.  If 'flag_check'
 #  is set the prefs can be flags or vars, else just vars.
 #  
 #  'yvar' is a variable which contains the current y-pos in the box,
 #  and should be incremented as appropriate by this procedure.
 #  'width' is the width of the dialog box (default 420)
 #  'l' is the left indent of all the items (default 20)
 #  'dialogvar' is the variable onto which all the construction code
 #  should be lappended.  If it is not given, then this proc will
 #  return the items.
 #  'names', if given, is an array containing textual replacements for
 #  the names of the variables to be used in the box.
 #  
 #  A minimal call would be:
 #  
 #  set y 20
 #  set build [dialog::buildSection dial1 [list fillColumn] y]
 #  eval lappend build [dialog::okcancel 20 y]
 #  set res [eval dialog -w 480 -h $y $build]
 #  
 # -------------------------------------------------------------------------
 ##
proc dialog::buildSection {mod vars yvar {width 420} {l 20} {dialogvar ""} {names ""} {flag_check 1}} {
    global flag::list flag::type allFlags spelling alpha::colors \
      includeDescriptionsInDialogs dialog::_not_global_flag mode alpha::platform

    if {$includeDescriptionsInDialogs || [info tclversion] >= 8.0} {
	cache::readContents index::prefshelp
	if {[info tclversion] >= 8.0} {
	    upvar help help
	}
    }
    if {[regsub {(modeVars)?$} ${dialog::_not_global_flag} "" vprefix]} {
	append vprefix ","
    }
    upvar $yvar t
    if {$dialogvar != ""} {upvar $dialogvar args}
    if {$names != ""} { upvar $names name }
    set height 17
    set lf 135
    set r [expr {$l + $width}]
    set rb [expr {$r -45}]
    foreach vset $vars {
	if {[llength $vset] > 1} {
	    incr t 5
	    if {[lindex $vset 0] != ""} {
		lappend args "-t" "[lindex $vset 0]" [expr {$l -10}] $t $r [expr {$t +15}]
		incr t 20
	    }
	    set vset [lrange $vset 1 end]
	}
	foreach v $vset {
	    if {$includeDescriptionsInDialogs} {
		if {[info exists prefshelp($vprefix$v)]} {
		    incr t 10
		    eval lappend args [dialog::text $prefshelp($vprefix$v) $l t 90]
		    incr t -14
		}
	    }
	    if {[info tclversion] >= 8.0} {
		if {[info exists prefshelp($vprefix$v)]} {
		    lappend help $prefshelp($vprefix$v)
		} elseif {[info exists prefshelp($mode,$v)]} {
		    lappend help $prefshelp($mode,$v)
		} else {
		    lappend help ""
		}
	    }
	    
	    set vv [dialog::getFlag $mod $v]
	    if {[info exists name($v)]} {
		set vname $name($v)
	    } else {
		set vname [quote::Prettify $v]
	    }
	    if {$spelling} {
		text::british vname
	    }
	    if {$flag_check && [lcontains allFlags $v]} {
		if {[info tclversion] < 8.0} {
		    lappend args "-c" $vname $vv $l $t $r [expr {$t + 15}]
		} else {
		    lappend args "-c" $vname $vv -font 2 $l $t $r [expr {$t + 15}]
		}
		incr t 15
		continue
	    }
	    # attempt to indent correctly
	    set len [string length $vname] 
	    if {$len > 40} {
		lappend args "-t" "$vname:" $l $t [expr {$r -30}] [expr {$t + $height}]
		incr t 15
		set indent 100
		set tle ""
	    } elseif {$len > 17} {
		set indent [expr {11 + 7 * $len}]
		set tle {"-t" "$vname:" $l $t [expr {$l + $indent}] [expr {$t + $height}]}
	    } else {
		set indent $lf
		set tle {"-t" "$vname:" $l $t [expr {$l + $indent}] [expr {$t + $height}]}
	    }
	    
	    if {[info exists flag::list($v)]} {
		incr t 5
		eval lappend args $tle
		set litems [flag::options $v]
		if {[regexp "index" [lindex [set flag::list($v)] 0]]} {
		    # set item to index, making sure bad values don't error
		    if {[catch {lindex $litems $vv} vv]} { set vv [lindex $litems 0] }
		}
		lappend args "-m" [concat [list $vv] $litems] [expr {$l + $indent -2}] [expr {$t -2}] [expr {$r - 14}] [expr {$t + $height +1}]
		incr t 17
	    } elseif {[regexp "Colou?r$" $v]} {
		incr t 5
		eval lappend args $tle
		lappend args "-m" [concat [list $vv] ${alpha::colors}] [expr {$l + $indent -2}] [expr {$t -2}] [expr {$r - 14}] [expr {$t + $height +1}]
		incr t 17
	    } elseif {[regexp "Mode$" $v]} {
		incr t 5
		eval lappend args $tle
		if {$vv == ""} { set vv "<none>" }
		lappend args "-m" [concat [list $vv] [concat "<none>" [mode::listAll]]] [expr {$l + $indent -2}] $t [expr {$r - 14}] [expr {$t + $height +1}]
		incr t 17
	    } elseif {[regexp "Sig$" $v]} {
		eval lappend args $tle
		set vv [dialog::specialView::sig $vv]
		lappend args "-t" $vv [expr {$l + $indent}] $t $rb [expr {$t + $height +1}]
		eval lappend args [dialog::buttonSet $mod $rb $t sig $v]
		incr t 17
	    } elseif {[regexp "SearchPath$" $v]} {
		if {${alpha::platform} == "alpha"} {
		    eval lappend args $tle
		    set origt $t
		    if {$vv == ""} {
			lappend args "-t" "No search paths currently set." \
			  [expr {$l + $indent}] $t $rb [expr {$t + $height +1}]
			incr t 17
		    } else {
			set view {}
			foreach ppath $vv {
			    lappend view [dialog::specialView::file $ppath]
			    incr t 17
			}
			lappend args "-t" [join $view "\r"] \
			  [expr {$l + $indent}] $origt $rb [expr {$t - 17 + $height +1}]
		    }
		    # Note: you can test the result of adding 'searchpath $v' as two
		    # more arguments to dialog::buttonSet here.  The problem arises
		    # when we increase the number of paths - there isn't room in the
		    # dialog for more of them!  This problem is solved by adding a
		    # dialog list item type with the '-l' flag, currently only supported
		    # by Alphatk.
		    eval lappend args [dialog::buttonSet $mod $rb $origt]
		} else {
		    eval lappend args $tle
		    set origt $t
		    lappend args "-l" $vv 3
		    lappend args "-dnd" $mod [list $v searchpath]
		    lappend args [expr {$l + $indent}] $origt $rb [expr {$t + 34 + $height +1}]
		    eval lappend args [dialog::buttonSet $mod $rb $origt searchpath $v]
		    incr t 51
		}
	    } elseif {[regexp "(Path|Folder)$" $v]} {
		eval lappend args $tle
		set vv [dialog::specialView::file $vv]
		lappend args "-t" $vv 
		if {${alpha::platform} != "alpha"} {
		    lappend args "-dnd" $mod [list $v folder]
		}
		lappend args [expr {$l + $indent}] $t $rb [expr {$t + $height +1}]
		eval lappend args [dialog::buttonSet $mod $rb $t folder $v]
		incr t 17
	    } elseif {[info exists flag::type($v)]} {
		set theType [set flag::type($v)]
		if {$theType == "funnyChars"} {
		    set vv [quote::Display $vv]
		    set eh [expr {1 + [string length $vv] / 60}]
		    incr t [expr {7 * $eh}]
		    eval lappend args $tle
		    incr t [expr {5 -7 * $eh}]
		    lappend args "-e" $vv [expr {$l + $indent}] $t $r [expr {$t + $eh * $height}]
		    incr t [expr {5 + 17 * $eh}]
		} else {
		    eval lappend args $tle
		    set vv [dialog::specialView::$theType $vv]
		    lappend args "-t" $vv
		    if {${alpha::platform} != "alpha" && $theType == "url"} {
			lappend args "-dnd" $mod [list $v $theType]
		    }
		    lappend args [expr {$l + $indent}] $t $rb [expr {$t + $height +1}]
		    eval lappend args [dialog::buttonSet $mod $rb $t $theType $v]			
		    incr t 17
		}
	    } else {
		set eh [expr {1 + [string length $vv] / 60}]
		incr t [expr {7 * $eh}]
		eval lappend args $tle
		incr t [expr {5 -7 * $eh}]
		lappend args "-e" $vv [expr {$l + $indent}] $t $r [expr {$t + $eh * $height}]
		incr t [expr {5 + 17 * $eh}]
	    }
	}
    }
    if {$dialogvar == ""} {return $args}
}

proc dialog::multipage {mod title data} {
    global maxT dialog::_not_global_pkg dialog::_not_global_flag
    # in case internal 'command-buttons' are used in the dialog
    while 1 {
	
	set left 20   
	
	set names {}
	set editItems {}
	set cmd ""
	set maxT 0
	if {![info exists dialog::_not_global_pkg]} {
	    set data [lsort $data]
	}
	foreach arg $data {
	    if {[llength $arg] != 3} {error "Bad structure"}
	    set name [lindex $arg 0]
	    lappend names $name
	    # set the appropriate value of dialog::_not_global_flag
	    if {[info exists dialog::_not_global_pkg] \
		&& [lsearch -exact ${dialog::_not_global_pkg} ${name}modeVars] >= 0} {
		set dialog::_not_global_flag ${name}modeVars
	    } else {
		set dialog::_not_global_flag {}
	    }
	    set flags [lindex $arg 1]
	    set vars [lindex $arg 2]
	    lappend editItems [eval list $flags $vars]
	    eval lappend cmd "-n" [list [lindex $arg 0]] [dialog::flag $mod $flags $vars]
	}
	
	set buttons [dialog::okcancel -460 maxT]
	set height [expr {$maxT + 14}]
	# set the appropriate value of dialog::_not_global_flag
	if {![info exists chosenName]} {
	    set chosenName [lindex $names 0]
	    if {[info exists dialog::_not_global_pkg] \
		&& [lsearch -exact ${dialog::_not_global_pkg} $chosenName] >= 0} {
		set dialog::_not_global_flag $chosenName
	    } else {
		set dialog::_not_global_flag {}
	    }
	}
	if {[info exists help]} {
	    set res [eval [concat dialog -w 480 -h $height [list -T $title] \
	      -t "Preferences:" 40 10 125 30 $buttons \
	      -b "Help" 410 10 460 28 \
	      [list -m [concat [list $chosenName] $names] 140 8 405 30] \
	      $cmd -help] [list [concat [list \
	      "Click here to save the current settings." \
	      "Click here to discard any changes you've made to the settings." \
	      "Click here to display textual help on each item in this dialog." \
	      "Use this popup menu, or the cursor keys to select a \
	      different page of preferences."] $help]]]
	} else {
	    set res [eval [concat dialog -w 480 -h $height \
	      -t "Preferences:" 40 10 125 30 $buttons \
	      -b "Help" 410 10 460 28 \
	      [list -m [concat [list $chosenName] $names] 140 8 405 30] \
	      $cmd]]
	}
	
	set chosenName [lindex $res 3]
	# set the appropriate value of dialog::_not_global_flag
	if {[info exists dialog::_not_global_pkg] \
	    && [lsearch -exact ${dialog::_not_global_pkg} ${chosenName}modeVars] >= 0} {
	    set dialog::_not_global_flag ${chosenName}modeVars
	} else {
	    set dialog::_not_global_flag {}
	}
	if {[lindex $res 0]} {
	    return [list [lrange $res 4 end] [eval concat $editItems]]
	} else {
	    if {[lindex $res 1]} {
		unset dialog::_not_global_pkg
		set dialog::_not_global_flag {}
		error "cancel"
	    }
	    dialog::rememberChanges $mod [list [lrange $res 4 end] [eval concat $editItems]]
	    # Either help, or some set or describe type button was pressed
	    # We need to ensure we remember anything the user has already
	    # changed.
	    if {[lindex $res 2]} {
		# help pressed
		set i [lsearch -exact $names [lindex $res 3]]
		dialog::describe $mod [lindex $editItems $i] "Description of [lindex $res 3]"
	    } else {
		# a 'set...' button was pressed
		dialog::handleSet $mod [lrange $res 4 end] [eval concat $editItems]
	    }
	}
	# end of large while loop
    }
}

proc dialog::rememberChanges {mod values_items} {
    set res [lindex $values_items 0]
    set editItems [lindex $values_items 1]
    unset values_items
    foreach fset $editItems {
	if {[llength $fset] > 1} {
	    set fset [lrange $fset 1 end]
	}
	foreach flag $fset {
	    set val [lindex $res 0]
	    set res [lrange $res 1 end]
	    # May need to 'catch' this postManipulate for Dominique's
	    # package changes.
	    dialog::postManipulate $mod 0
	    dialog::modified $mod $flag $val
	}
    }
}

proc dialog::onepage {mod flags vars {title ""}} {
    global maxT
    while 1 {
	set left 20
	set maxT 0
	if {[info tclversion] < 8.0} {
	    set args [dialog::flag $mod $flags $vars 20 10 $title]
	} else {
	    set args [dialog::flag $mod $flags $vars 20 10]
	}
	set buttons [dialog::okcancel -460 maxT]
	set height [expr {$maxT + 14}]
	if {[info exists help]} {
	    set res [eval [concat dialog -w 480 -h $height $buttons \
	      -T [list $title] -b "Help" 410 5 460 23 $args -help] \
	      [list [concat [list \
	      "Click here to save the current settings." \
	      "Click here to discard any changes you've made to the settings." \
	      "Click here to display textual help on each item in this dialog." \
	      ] $help]]]
	} else {
	    set res [eval [concat dialog -w 480 -h $height $buttons \
	      -b "Help" 410 10 460 28 $args]]
	}
	
	if {[lindex $res 0]} {
	    return [list [lrange $res 3 end] [concat $flags $vars]]
	} else {
	    
	    if {[lindex $res 1]} {
		error "cancel"
	    } 
	    dialog::rememberChanges $mod [list [lrange $res 3 end] [concat $flags $vars]]
	    if {[lindex $res 2]} {
		# help
		dialog::describe $mod [concat $flags $vars] $title
	    } else {
		dialog::handleSet $mod [lrange $res 3 end] [concat $flags $vars]
	    }
	}
	# big while loop end
    }
}

if {[info tclversion] >= 8.0} {
proc dialog::describe {mod vars {title ""}} {
    if {$title == ""} {
	set title "Preferences description"
    }
    global flag::list flag::type spelling alpha::colors \
      dialog::_not_global_flag mode index::flags
    if {[regsub {(modeVars)?$} ${dialog::_not_global_flag} "" vprefix]} {
	append vprefix ","
    }
    cache::readContents index::prefshelp
    set height 17
    set lf 135
    set l 20
    set width 420
    set r [expr {$l + $width}]
    set rb [expr {$r -45}]
    set args {}
    set t 35
    set height 0
    set page 1
    set pages {}
    set lst {}
    foreach vset $vars {
	if {$t > 360} {
	    # make another page
	    eval lappend pages -n [list "Page $page"] $args
	    eval lappend lst [list $args]
	    set args {}
	    incr page
	    if {$t > $height} {set height $t}
	    set t 35
	}
	if {[llength $vset] > 1} {
	    incr t 5
	    if {[lindex $vset 0] != ""} {
		lappend args "-t" "[lindex $vset 0]" [expr {$l -10}] $t $r [expr {$t +15}]
		incr t 20
	    }
	    set vset [lrange $vset 1 end]
	} else {
	    # Do this so that vars that have whitespace padding (used to
	    # force dialog position) are not stripped of that space in the
	    # next "foreach" statement
	    set vset [list [set vset]]
	}
	foreach v $vset {
	    set vv [dialog::getFlag $mod $v]
	    if {[info exists name($v)]} {
		set vname $name($v)
	    } else {
		set vname [quote::Prettify $v]
	    }
	    if {$spelling} {
		text::british vname
	    }
	    if {[info exists prefshelp($vprefix$v)]} {
		append vname ": " [dialog::helpdescription $prefshelp($vprefix$v)]
	    } elseif {[info exists prefshelp($mode,$v)]} {
		append vname ": " [dialog::helpdescription $prefshelp($mode,$v)]
	    } elseif {[lsearch -exact [set index::flags] $v] != -1} {
		append vname ": " [dialog::helpdescription [dialog::packagehelp $v]]
	    } else {
		append vname ": no description"
	    }
	    eval lappend args [dialog::text $vname $l t 60]
	}
    }
    if {$page > 1} {
	set t $height
	set height [expr {$t + 40}]
	for {set i 1} {$i <= $page} {incr i} {
	    lappend names "Page $i"
	}
	eval lappend pages [list -n "Page $page"] $args		
	set res [eval [concat dialog -w 480 -h $height \
	  -t [list $title] 60 10 $width 30 \
	  -b "OK" 30 [expr {$t + 10}] 95 [expr {$t +30}] \
	  [list -m [concat [list [lindex $names 0]] $names] 400 10 475 30] $pages]]
    } else {
	set height [expr {$t + 40}]
	set res [eval [concat dialog -w 480 -h $height \
	  -t [list $title] 60 10 $width 30 \
	  -b "OK" 30 [expr {$t + 10}] 95 [expr {$t +30}] $args]]
    }
}
} else {
    # old version for Alpha 7.x
    proc dialog::describe {mod vars {title ""}} {
	if {$title == ""} {
	    set title "Preferences description"
	}
	global flag::list flag::type spelling alpha::colors \
	  dialog::_not_global_flag mode index::flags
	global defHeight
	if {[regsub {(modeVars)?$} ${dialog::_not_global_flag} "" vprefix]} {
	    append vprefix ","
	}
	cache::readContents index::prefshelp
	set height 17
	set lf 135
	set l 20
	set width 420
	set r [expr {$l + $width}]
	set rb [expr {$r -45}]
	set args {}
	set t 35
	set height [expr {$defHeight - 60}]
	set page 1
	set lst {}
	foreach vset $vars {
	    if {[llength $vset] > 1} {
		incr t 5
		if {[lindex $vset 0] != ""} {
		    lappend args "-t" "[lindex $vset 0]" [expr {$l -10}] $t $r \
				      [expr {$t +15}]
		    incr t 20
		}
		set vset [lrange $vset 1 end]
	    } else {
		# Do this so that vars that have whitespace padding (used to
		# force dialog position) are not stripped of that space in the
		# next "foreach" statement
		set vset [list [set vset]]
	    }
	    foreach v $vset {
		set vv [dialog::getFlag $mod $v]
		if {[info exists name($v)]} {
		    set vname $name($v)
		} else {
		    set vname [quote::Prettify $v]
		}
		if {$spelling} {
		    text::british vname
		}
		if {[info exists prefshelp($vprefix$v)]} {
		    append vname ": " [dialog::helpdescription \
					$prefshelp($vprefix$v)]
		} elseif {[info exists prefshelp($mode,$v)]} {
		    append vname ": " [dialog::helpdescription $prefshelp($mode,$v)]
		} elseif {[lsearch -exact [set index::flags] $v] != -1} {
		    append vname ": " [dialog::helpdescription \
					 [dialog::packagehelp $v]]
		} else {
		    append vname ": no description"
		}
		set newarg [dialog::text $vname $l t 60]
		if {$t <= $height} {
		    eval lappend args $newarg
		} else {
		    # make another page
		    eval lappend lst [list $args] 
		    incr page
		    set lngth [llength $newarg]
		    set disp [expr {[lindex $newarg 3] - 35}]
		    set bot [expr {$t - [lindex $newarg [expr {$lngth - 1}]]}]
		    set k 3
		    for {set i 3} {$i < $lngth} {incr i 6} {
			# shift 'newarg' vertically.  The shifting is based 
			# on 'newarg' build as '-x text h1 v1 h2 v2'.
			set j [expr {$k + 2}]
			set pntr [expr {[lindex $newarg $j] - $disp}]
			if {$pntr > [expr {$height - $bot}]} {
			    # page too long, split 'newarg'. 
			    # The splitting is based on 'newarg' build as
			    # -x text h1 v1 h2 v2.  It assumes that the elementary 
			    # piece of text fits in a single page.
			    set tmp [lrange $newarg 0 [expr {$k - 4}]]
			    eval lappend lst [list $tmp] 
			    incr page
			    set newarg [lrange $newarg [expr {$k - 3}] end]
			    set disp [expr {[lindex $newarg 3] - 35}]
			    set pntr [expr {[lindex $newarg 5] - $disp}]
			    set k 3
			    set j 5
			}
			set newarg [lreplace $newarg $k $k \
			  [expr {[lindex $newarg $k] - $disp}]]
			set newarg [lreplace $newarg $j $j $pntr]
			incr k 6
		    }
		    if {![llength $newarg]} {
			set t 35
			set args {}
		    } else {
			set t [expr {$t - $disp}]
			set args $newarg
		    }		
		}
	    }
	    
	}
	if {![llength $args]} {
	    incr page -1
	} else {
	    lappend lst $args
	}		
	if {$page > 1} {
	    set t $height
	    set height [expr {$t + 40}]
	    for {set i 1} {$i <= $page} {incr i} {
		eval lappend pages -n [list "Page $i"] [lindex $lst \
								  [expr {$i - 1}]]
		lappend names "Page $i"
	    }
	    eval lappend pages -n [list "Page $page"] $args		
	    if {($page > 1) || \
		[catch { set res [eval [concat dialog -w 480 -h $height \
		  -t [list $title] 60 10 $width 30 \
		  -b "OK" 395 [expr {$t + 10}] 460 [expr {$t +30}] \
		  [list -m [concat [list [lindex $names 0]] $names] 400 10 475 30] \
		  $pages]]}]} {
		# Dominque's code to work around Alpha 7.x problem with
		# dialogs containing too much stuff.
		set pgnmbr 0
		set v1 [expr {$t + 10}]
		set v2 [expr {$t +30}]
		while 1 {
		    set pages [lindex $lst $pgnmbr]
		    set h1 110
		    set h2 130
		    set names {}
		    for {set i 1} {$i <= $page} {incr i} {
			if {$i == [expr {$pgnmbr+ 1}]} {
			    lappend names -t $i [expr {$h1 + 4}] $v1 $h2 $v2
			} else {
			    lappend names -b $i $h1 [expr {$v1 - 2}] \
			      $h2 [expr {$v2 - 2}]
			}
			incr h1 25
			incr h2 25
		    }
		    set res [eval [concat dialog -w 480 -h $height \
		      -t [list $title] 60 10 $width 30 -b "OK" 395 $v1 460 $v2 \
		      -t {Choose\ Page:} 20 $v1 110 $v2 $names $pages]]
		    if {[lindex $res 0]} {
			return
		    }
		    for {set i 1} {$i < $page} {incr i} {
			if {[lindex $res $i]} {
			    if {$i < [expr {$pgnmbr+ 1}]} {
				set pgnmbr [expr {$i - 1}] 
			    } else {
				set pgnmbr $i
			    }
			    break
			}
		    }
		}
	    }
	} else {
	    set height [expr {$t + 40}]
	    set res [eval [concat dialog -w 480 -h $height \
	      -t [list $title] 60 10 $width 30 \
	      -b "OK" 395 [expr {$t + 10}] 460 [expr {$t +30}] $args]]
	}
    }
}

proc dialog::helpdescription {hlp} {
    set hlp [split $hlp |]
    if {[llength $hlp] <= 1} {
	return [lindex $hlp 0]
    }
    set res ""
    for {set hi 0} {$hi < [llength $hlp]} {incr hi} {
	set hitem [lindex $hlp $hi]
	if {$hitem != ""} {
	    if {$hi == 0} {
		regsub "click this box\\.? *" $hitem "turn this item on" hitem
	    } elseif {$hi == 2} {
		regsub "click this box\\.? *" $hitem "turn this item off" hitem
	    }
	    append res $hitem ". "
	}
    }
    return $res
}

if {${alpha::platform} == "alpha"} {
    set dialog::strlength 253
} else {
    set dialog::strlength 2000
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::packagehelp" --
 # 
 #  Return help string useful for balloon help for a given package.
 #  If 'balloon' is 1, we only allow 253 characters in each substring.
 #  (Balloon help has trouble with large quantities...)
 # -------------------------------------------------------------------------
 ##
proc dialog::packagehelp {pkg {balloon 0}} {
    regsub -all "\[ \t\r\n\]+" [string trim [package::helpFile $pkg 1]] " " hlp
    if {$balloon} {
	global dialog::strlength
	set hlp [string range $hlp 0 [set dialog::strlength]]
	return "${hlp}||${hlp}"
    } else {
	return $hlp
    }
}

#  Dialog utilities  #
proc dialog::handleSet {mod res names} {
    # to account for sub-lists in the list of names
    foreach n $names {
	if {[llength $n] > 1} {
	    eval lappend newnames [lrange $n 1 end]
	} else {
	    lappend newnames $n
	}
    }
    set names $newnames
    unset newnames
    global flag::type
    # a 'set' button was pressed
    for {set i 0} {$i < [llength $names]} {incr i} {
	if {[lindex $res $i] == 1} {
	    set v [lindex $names $i]
	    if {[regexp "SearchPath$" $v]} {
		dialog::specialSet::searchpath $mod $v
		break
	    } elseif {[regexp "(Path|Folder)$" $v]} {
		# this set pressed
		if {![catch {get_directory -p "New [quote::Prettify $v]"} newval]} {
		    dialog::modified $mod $v $newval
		}
		break
	    } elseif {[info exists flag::type($v)]} {
		dialog::specialSet::[set flag::type($v)] $mod $v
		break
	    } elseif {[regexp "Sig$" $v]} {
		global $v
		if {[info exists $v]} {
		    set newval [dialog::findApp $v "" [set $v]]
		} else {
		    set newval [dialog::findApp $v]
		}
		if {$newval != ""} {
		    dialog::modified $mod $v $newval
		}
		break
	    }  
	}
    }
}

proc dialog::setFlag {name val} {
    global dialog::_not_global_flag
    if {${dialog::_not_global_flag} != ""} {
	global ${dialog::_not_global_flag}
	if {[array exists ${dialog::_not_global_flag}]} {
	    set ${dialog::_not_global_flag}($name) $val
	} else {
	    global ${dialog::_not_global_flag}::${name}
	    set ${dialog::_not_global_flag}::${name} $val
	}
    } else {
	global index::feature dialog::_is_global mode
	if {[info exists index::feature($name)]} {
	    if {[info exists dialog::_is_global]} {
		global global::features
		if {$val} {
		    if {[package::do_activate $name]} {
			package::throwActivationError
		    } else {
			lunion global::features $name
			package::activate $name
		    }
		} else {
		    set global::features [lremove [set global::features] $name]
		    global mode
		    if {$mode != ""} {
			if {![mode::isFeatureActive $mode $name]} {
			    package::deactivate $name
			}
		    } else {
			package::deactivate $name
		    }
		}
	    } else {
		mode::adjustFeatures $name $val
	    }
	} else {
	    global $name
	    set $name $val
	}
    }	
}

proc dialog::getFlag {mod name} {
    global dialog::${mod}
    if {[info exists dialog::${mod}($name)]} { 
	return [set dialog::${mod}($name)] 
    } else {
	return [dialog::getOldFlag $name]
    }
}

proc dialog::getOldFlag {name} {
    global dialog::_not_global_flag dialog::_not_global_pkg
    # Added this block to handle flags and variables in dialog::_not_global_pkg
    if {[info exists dialog::_not_global_pkg]} {
	foreach pkg ${dialog::_not_global_pkg} {
	    global $pkg
	    if {[info exists ${pkg}($name)]} {
		return [set ${pkg}($name)]
	    }
	}
	# For items in the miscellaneous page.
	global $name
	if {[info exists $name]} {
	    return [set $name]
	}
    }
    if {${dialog::_not_global_flag} != ""} {
	global ${dialog::_not_global_flag}
	if {[array exists ${dialog::_not_global_flag}]} {
	    return [set ${dialog::_not_global_flag}($name)]
	} elseif {[namespace_exists ${dialog::_not_global_flag}]} {
	    global ${dialog::_not_global_flag}::${name}
	    return [set ${dialog::_not_global_flag}::${name}]
	} else {
	    error "No variable storage for '${dialog::_not_global_flag}' exists for flag $name"
	}
    }
    global dialog::_is_global mode index::feature
    if {[info exists dialog::_is_global]} {
	global global::features
	if {[globalVarIsShadowed $name]} {
	    return [globalVarSet $name]
	}
	if {[info exists index::feature($name)]} {
	    return [expr {[lsearch -exact ${global::features} $name] != -1}]
	}
    } else {
	if {[info exists index::feature($name)]} {
	    return [mode::isFeatureActive $mode $name]
	}
    }
    global $name
    if {[info exists $name]} { 
	return [set $name]
    } else { 
	alertnote "Global variable '$name' in the dialog isn't set.\r\
	  I'll try to fix that."
	return [set $name ""]
    }
}

proc dialog::is_global {script} {
    global dialog::_is_global
    set dialog::_is_global 1
    catch [list uplevel $script]
    unset dialog::_is_global
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::adjust_flags" --
 # 
 #  This is used after the results are in from a prefs dialog, to 
 #  make sure those results are stored in the correct places/variables.
 #  
 #  'global' items need to be stored in global variables
 #  'arraynamespace' items are stored either inside the given array
 #  (if it is an array) or inside the given namespace.
 #  'arrayglobal' items are normally used for mode-vars, and are stored
 #  both inside an array, and (when the mode is active) globally.
 #  
 #  The latter two items use the optional 'storage' argument to pass
 #  the name of the array/namespace to be used.
 # -------------------------------------------------------------------------
 ##
proc dialog::adjust_flags {mod how values_items {storage ""}} {
    global flag::procs index::feature
    set res [lindex $values_items 0]
    set editItems [lindex $values_items 1]
    unset values_items

    if {[string length $storage]} {
	global $storage
    }
    
    foreach fset $editItems {
	if {[llength $fset] > 1} {
	    set fset [lrange $fset 1 end]
	}
	foreach flag $fset {
	    set val [lindex $res 0]
	    set res [lrange $res 1 end]
	    dialog::postManipulate $mod
	    if {[info exists index::feature($flag)]} {
		dialog::setFlag $flag $val
		continue
	    }
	    switch -- $how {
		"global" {
		    if {[globalVarIsShadowed $flag]} {
			set orig [globalVarSet $flag]
			if {$orig != $val} {
			    globalVarSet $flag $val
			    lappend warn_global $flag
			    prefs::modified $flag
			}
		    } else {
			global $flag
			set orig [set $flag]
			if {$orig != $val} {
			    set $flag $val
			    prefs::modified $flag
			}
		    }
		}
		"arraynamespace" {
		    # it's a package which keeps its vars in the array
		    # or in a namespace.
		    if {[array exists $storage]} {
			if {[set orig [set ${storage}($flag)]] != $val} {
			    set ${storage}($flag) $val
			    prefs::modified ${storage}($flag)
			}
		    } else {
			global ${storage}::$flag
			if {[set orig [set ${storage}::$flag]] != $val} {
			    set ${storage}::$flag $val
			    prefs::modified ${storage}::$flag
			}
		    }
		}
		"arrayglobal" {
		    # modes keep a copy of their vars at the global 
		    # level when active
		    global $flag
		    if {[set orig [set $flag]] != $val} {
			set $flag $val
			set ${storage}($flag) $val
			prefs::modified ${storage}($flag)
		    }
		}
	    }
	    # End of switch
	    if {$orig != $val} {
		if {[info exists flag::procs($flag)]} {
		    set proc [set flag::procs($flag)]
		    if {([info procs $proc] != "") && ([llength [info args $proc]] == 0)} {
			eval $proc
		    } else {
			eval $proc [list $flag]
		    }
		}
	    }
	}
    }
    if {[info exists warn_global]} {
	if {[llength $warn_global] == 1} {
	    set msg "is a global pref"
	} else {
	    set msg "are global prefs"
	}
	alertnote "You modified [join $warn_global {, }] which $msg,\
	  but currently over-ridden by mode-specific values.  If you meant to\
	  modify the latter values, use the mode prefs dialog."
    }
}

proc dialog::postManipulate {mod {modify 1}} {
    global flag::list flag::type
    upvar flag f
    upvar val v
    
    if {[info exists flag::list($f)]} {
	switch -- [lindex [set l [set flag::list($f)]] 0] {
	    "index" {
		set v [lsearch -exact [lindex $l 1] $v]
	    }
	    "varindex" {
		set itemv [lindex $l 1]
		global $itemv
		set v [lsearch -exact [set $itemv] $v]
	    }
	}
    }
    if {$v == "<none>" && [regexp "Mode$" $f]} { set v "" }
    # This check also captures any 'dialog::modified' items
    # This allows flags which are somehow already set by the
    # dialog (for instance if called recursively, or if set by embedded
    # 'Set' buttons) to be registered as modifed by our calling procedure.
    if {[regexp "(Path|Folder|Sig)$" $f]} {
	set v [dialog::getFlag $mod $f]
    } elseif {[info exists flag::type($f)]} {
	switch -- [set flag::type($f)] {
	    "binding" {
		# setup the changed binding
		set old [dialog::getOldFlag $f]
		set v [dialog::getFlag $mod $f]
		if {$modify && ($old != $v)} {
		    global flag::binding
		    if {[info exists flag::binding($f)]} {
			set m [lindex [set flag::binding($f)] 0]
			if {[set proc [lindex [set flag::binding($f)] 1]] == 1} {
			    set proc $f
			}
			catch "unBind [keys::toBind $old] [list $proc] $m"
			catch "Bind [keys::toBind $v] [list $proc] $m"
		    }
		}
	    }
	    "funnyChars" {
		set v [quote::Undisplay $v]
	    }
	    default {
		set v [dialog::getFlag $mod $f]
	    }
	}
    }
}

# Called by Alphatk/X/8 to find the allowed mime type
# which can be dropped on this command string.
# (This function will probably need to be updated
# as the varying requirements of Alphatk/X/8 become
# better known).
proc dialog::valGetMimeType {varInfo} {
    switch -- [lindex $varInfo 1] {
	"file" -
	"folder" -
	"url" {
	    return "text/uri-list"
	}
	"searchpath" {
	    return "text/uri-list"
	}
	default {
	    return ""
	}
    }
}

# Called by Alphatk/X/8 to find the appropriate command string
# to call when dropping item(s) on this 'varInfo'
proc dialog::valGetDropAction {varInfo} {
    set type [lindex $varInfo 1]
    if {$type == "searchpath"} {
	return [list dialog::modifiedAdd]
    } else {
	return [list dialog::modifiedAdjust]
    }
}

# Called by internals of Alphatk/8/X to ask AlphaTcl to abbreviate
# an item.  The 'width' is in pixels
proc dialog::abbreviate {varInfo val width} {
    # 'varInfo' is defined below in 'dialog::setControlValue'.
    return [dialog::width_abbrev $val $width]
}

proc dialog::modified {mod name val {type ""}} {
    dialog::valChanged $mod $name $val
    if {[string length $type]} {
	# We have some code registered which would like to know what
	# changed. Alphatk uses such hooks to update dialog items from
	# Set... buttons automatically, but it would be better if
	# the code that called dialog::modified could do that
	# explicitly.
	hook::callAll dialog modified $name $val $type
    }
}

# Called by drag'n'drop or set callbacks.
proc dialog::modifiedAdjust {mod varInfo args} {
    # Double-check we can accept it
    #tclLog [list dialog::modifiedAdjust $mod $varInfo $args]
    switch -- [llength $args] {
	0 {
	    return -code error "No value argument to dialog::modifiedAdjust"
	}
	1 {
	    set newValue [lindex $args 0]
	}
	default {
	    return -code error "Can only drop one item at a time."
	}
    }
    set err [dialog::itemAcceptable $varInfo $newValue]
    if {[string length $err]} {
	message "Drop failed for '$newValue' : $err"
    } else {
	set var [lindex $varInfo 0]
	set type [lindex $varInfo 1]
	dialog::modified $mod $var $newValue $type
	message "Dropped '$newValue'"
    }
}

proc dialog::modifiedAdd {mod varInfo args} {
    # Double-check we can accept it
    #tclLog [list dialog::modifiedAdd $mod $varInfo $args]
    foreach newValue $args {
	set err [dialog::itemAcceptable $varInfo $newValue]
	if {[string length $err]} {
	    message "Drop failed for '$newValue' : $err"
	    return
	}
    }
    
    set var [lindex $varInfo 0]
    set type [lindex $varInfo 1]
    
    set new [concat [dialog::getFlag $mod $var] $args]
    dialog::modified $mod $var $new $type
    
    message "Dropped '$new'"
}

# When a dialog item is modified in place, this procedure will
# be called by AlphaTcl.  It is called after the internal storage
# for the variable has been updated, and cannot cancel the change.
# 
# '$dialItemSetCmd' is a command/id string given to us by our
# editing environment (Alpha(tk)) which can be used to change
# the dialog in place.
# 
# We don't need to know any more than that.
# 
# Currently, we just use that command string, with
# 'eval $dialItemSetCmd [list $varInfo $val]', to modify
# the gui directly.
# 
# For Alpha 8, therefore, '$dialItemSetCmd' is a C command which
# modifies the dialog shown to the user directly.  For Alphatk it is
# some Tcl/Tk code which does a similar thing.
# 
# Alpha 8 needs updating to do what Alphatk does.  Here is a sample
# procedure from Alphatk so we can see how the callback should work
# 
#     proc dialog::labelSetContents {w varInfo val} {
#         set width [winfo width $w]
#         set valwidth [font measure [$w cget -font] $val]
#         #puts [list $w $varInfo $val $width $valwidth]
#         if {$valwidth > $width} {
#             set view [dialog::abbreviate $varInfo $width]
#         } else {
#             set view $val
#         }
#         $w configure -text $view
#     }
# 
# The 'dialItemSetCmd' already contains the first argument '$w',
# so we only need add the other two.  These are 'varInfo' which
# is an arbitrary tag for the current variable which
# Alpha tk/X/8 knows nothing about except that it should be
# passed on to appropriate other routines when necessary.
# 
# The last argument is of course the value we want to set
# the control to.
proc dialog::setControlValue {dialItemSetCmd name val type} {
    #tclLog "dialog::setControlValue $dialItemSetCmd $name $val $type"
    set varInfo [list $name $type]
    
    # Alpha 8 current expects this:
    # 
    # eval $dialItemSetCmd [list $val]
    # 
    # So it will fail here.
    eval $dialItemSetCmd [list $varInfo $val]
}

# Dialog code internal to Alpha(tk) can call this procedure when a
# 'set...'  button is pressed to interact with AlphaTcl in the
# process of modifying the dialog in place.  The 'cmd'
# is the command registered with the '-set' flag, and 'dialItemSetCmd'
# is a command which will be used in the procedure
# 'dialog::setControlValue' above.  These item-set cmds must be created
# and removed by the internal dialog code when appropriate.
# 
# While this procedure looks a little like a 'black box', it is
# actually quite simple.  We register a hook so that any change to
# a variable is trapped by the procedure above, then we call any
# standard procedure in AlphaTcl which might have the result of
# changing the value of a variable.  (The reason we need to have
# the obscure '$cmd' is that there are a few possibilities for
# that action: set... , drag'n'drop, drag'n'drop with an 'add' 
# rather than 'replace' action, etc).  Finally, we de-register the
# hook.
# 
# If the middle call changed the variable value, that change will
# always be routed through to 'dialog::modified', and dialog::modified
# will call all the hooks which have asked to be notified of such
# changes.  Since we've registered the above hook
# 'dialog::setControlValue', it will be called.
proc dialog::itemSet {dialItemSetCmd cmd args} {
    #tclLog [list dialog::itemSet $dialItemSetCmd $cmd $args]
    hook::register dialog [list dialog::setControlValue $dialItemSetCmd] \
      modified
    # Use of this proc by 'set...'  buttons will have an empty 'args'
    # list, and the 'cmd' will usually call one of the
    # dialog::specialSet procedures which brings up a new dialog to ask
    # the user for the value to use.
    # 
    # Uses by drag-n-drop or other direct methods will have a single
    # element in the args list -- the new value -- and the 'cmd' will
    # be something simple like [list dialog::modifiedAdjust $var $type]
    namespace eval :: $cmd $args
    hook::deregister dialog [list dialog::setControlValue $dialItemSetCmd] \
      modified
}

# Note: currently lots of pieces of code assume they know the 
# structure of the 'varInfo' object.  But really much of it
# shouldn't need to know?  Anyway, at the least we can be
# sure Alpha tk/8/X do not know anything about its internal
# structure.
# 
# I believe the next major changes to the dialogs code should
# be to abstract away the preferences information into the
# varInfo object.  Right now if we look at a procedure like
# 'dialog::getOldFlag' we can see just how much information
# about a flag/var is stored separately.
proc dialog::_getType {varInfo} {
    return [lindex $varInfo 1]
}

# Is the given variable of given type allowed to be set to the
# given value?  This procedure is called by things like drag-n-drop
# code, which means we can assume the newValue is already of an
# appropriate general type.  Any return value signifies an error
# with the value being a helpful error message.  To signify
# success, return nothing at all.
proc dialog::itemAcceptable {varInfo args} {
    #tclLog [list dialog::itemAcceptable $varInfo $newValue]
    set type [dialog::_getType $varInfo]
    if {$type != "searchpath"} {
	if {[llength $args] > 1} {
	    return "Can only drop one item!"
	}
    }
    if {[llength $args] == 0} {
	return "No item -- strange!"
    }
    set newValue [lindex $args 0]
    
    switch -- $type {
	"searchpath" {
	    if {[file isdirectory $newValue]} {
		return
	    }
	    global flag::extraOptions
	    # This 'flag::extraOptions' stuff needs embedding in
	    # varInfo
	    set var [lindex $varInfo 0]
	    if {[info exists flag::extraOptions($var)]} {
		foreach ext [set flag::extraOptions($var)] {
		    if {[string match $ext $newValue]} {
			return
		    }
		}
		return "Bad file extension"
	    }
	    return "Not a directory"
	}
	"folder" {
	    if {![file isdirectory $newValue]} {
		return "Not a directory"
	    }
	}
	"file" {
	    if {![file isfile $newValue]} {
		return "Not a file"
	    }
	}
	"url" {
	}
    }
    # Default is to accept a drop, since the code behind the scenes
    # will make sure we only get reasonable things dropped.
    return
}

# Used on modified mode flags.
set flag::procs(stringColor) "stringColorProc"
set flag::procs(commentColor) "stringColorProc"
set flag::procs(keywordColor) "stringColorProc"
set flag::procs(funcColor) "stringColorProc"
set flag::procs(sectionColor) "stringColorProc"
set flag::procs(bracesColor) "stringColorProc"

proc global::updateHelperFlags {} {
    uplevel #0 {
	set "flagPrefs(Helper Applications)" {}
	set "varPrefs(Helper Applications)" [lsort -ignore [info globals *Sig]]
    }
}

proc global::updatePackageFlags {} {
    global flagPrefs varPrefs allFlags modeVars allVars
    # flags can be in either flagPrefs or varPrefs if we're grouping
    # preferences according to function
    set all {}
    set flagPrefs(Packages) {}
    set varPrefs(Packages) {}
    foreach v [array names flagPrefs] {
	eval lappend all $flagPrefs($v)
	if {[info exists varPrefs($v)]} {
	    if {[regexp {[{}]} $varPrefs($v)]} {
		# we're grouping
		foreach i $varPrefs($v) {
		    if {[llength $i] > 1} {
			eval lappend all [lrange $i 1 end]
		    } else {
			lappend all $i
		    }
		}
	    } else {
		eval lappend all $varPrefs($v)
	    }
	}
    }
    foreach f $allFlags {
	if {([lsearch $modeVars $f] < 0)} {
	    if {[lsearch -exact $all $f] == -1} {
		lappend flagPrefs(Packages) $f
	    }
	}
    }
    
    foreach f $allVars {
	if {([lsearch $modeVars $f] < 0)} {
	    if {[lsearch -exact $all $f] == -1} {
		if {[regexp {Sig$} $f]} {
		    lappend "varPrefs(Helper Applications)" $f
		} else {
		    lappend varPrefs(Packages) $f
		}
	    }
	}
    }
}

#================================================================================

proc maxListItemLength {l} {
    set m 0
    foreach item $l {
	if {[set mm [string length $item]] > $m} { set m $mm }
    }
    return $m
}

proc stringColorProc {flag} {
    global $flag mode
    
    if {[set $flag] == "none"} {
        set $flag "foreground"
    }
    if {$flag == "stringColor"} {
        regModeKeywords -a -s $stringColor $mode
    } elseif {$flag == "commentColor"} {
        regModeKeywords -a -c $commentColor $mode
    } elseif {$flag == "funcColor"} {
        regModeKeywords -a -f $funcColor $mode
    } elseif {$flag == "bracesColor"} {
        regModeKeywords -a -I $bracesColor $mode
    } elseif {($flag == "keywordColor") || ($flag == "sectionColor")} {
        alertnote "Change in keyword color will take effect after Alpha restarts."
        return
    } else {
        alertnote "Change in $flag color will take effect after Alpha restarts."
        return
    }
    refresh
}

#  Dialog sub-items  #

if {[info tclversion] <= 8.0} {
    proc dialog::buttonSet {mod x y args} {
	return [list -b Set $x $y [expr {$x + 45}] [expr {$y + 15}]]
    }
} else {
    # Alphatk and Alpha8 can cope with setting dialog elements in place, via an extra
    # '-set' flag.  This means we don't need to destroy and recreate the entire dialog
    # when using a 'Set...' button.
    proc dialog::buttonSet {mod x y args} {
	if {[llength $args]} {
	    return [list -b Set -set [list [eval list [list dialog::specialSet::[lindex $args 0] $mod] [lrange $args 1 end]] -1] \
	      $x $y [expr {$x + 45}] [expr {$y + 15}]]
	} else {
	    return [list -b Set $x $y [expr {$x + 45}] [expr {$y + 15}]]
	}
    }
}

# if (x < 0) then it specifies the leftmost edge.
# Place items in their /correct/ locations.
proc dialog::okcancel {x yy {vertical 0} {ok OK} {cancel Cancel}} {
    upvar $yy y
    set yr $y
    if {$x < 0} {
	set i [dialog::button "$ok" "" y]
	if {$vertical} {
	    # want 10 pixel margin, but only get 6
	    incr y 4
	} else {
	    set y $yr
	}
	eval lappend i [dialog::button "$cancel" "" y]
    } else {
	set i [dialog::button "$ok" $x y]
	if {$vertical} {
	    # want 10 pixel margin, but only get 6
	    incr y 4
	} else {
	    set y $yr
	    incr x 80
	}
	eval lappend i [dialog::button "$cancel" $x y]
    }
    
    return $i
}

proc dialog::menu {x yy item {def "def"} {requestedWidth 0}} { 
    upvar $yy y
    set m [concat [list $def] $item]
    if {$requestedWidth == 0} {
	set popUpWidth 340
    } else {
	set popUpWidth $requestedWidth 
    }
    
    if {[info tclversion] < 8.0} {
	incr y 1
	set res [list -m $m $x $y [expr {$x + $popUpWidth}] [expr {$y +20}]]
	incr y 25
    } else {
	set res [list -m $m $x $y [expr {$x + $popUpWidth}] [expr {$y +20}]]
	incr y 26
    }
    return $res
}
## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::button" --
 # 
 #  Create a dialog string encoding one or more buttons.  'name' is the
 #  name of the button ("Ok" etc), x is the x position, or if x is null,
 #  then we use the variable called 'x' in the calling procedure.  yy is
 #  the name of a variable containing the y position of the button, which
 #  will be incremented by this procedure.  if args is non-null, it
 #  contains further name-x-yy values to be lined up next to this button.
 #  For sequences of default buttons, a spacing of '80' is usual, but
 #  it's probably best if you just set the 'x' param to "" and let this
 #  procedure calculate them for you.  See dialog::yesno for a good
 #  example of calling this procedure.
 # -------------------------------------------------------------------------
 ##
proc dialog::button {name x yy args} { 
    upvar $yy y
    if {$x == ""} {
	unset x
	upvar x x
    }
    set add 58
    if {[set i [expr {[string length $name] - 7}]] > 0} { 
	incr add [expr {$i * 7}]
    }
    if {$x < 0} {
	set res [list -b $name [expr {-($x +$add)}] $y [expr {-$x}] [expr {$y +20}]]
    } else {
	set res [list -b $name $x $y [expr {$x +$add}] [expr {$y +20}]]
    }
    
    incr x $add
    incr x 12
    if {[llength $args]} {
	eval lappend res [eval dialog::button $args]
	return $res
    }
    incr y 26
    
    return $res
}

proc dialog::title {name w} {
    set l [expr {${w}/2 - 4 * [string length $name]}]
    if {$l < 0} {set l 0}
    if {[info tclversion] < 8.0} {
	return [list -t $name $l 10 [expr {$w - $l}] 25]
    } else {
	return [list -T $name]
    }
}

# Can be used like 'dialog::text' but is capable of producing multiple
# pages if the text is really long.
proc dialog::multipagetext {text x yy width} {
    upvar $yy height
    
    set ystart $height

    set height 17
    set args {}
    set t $ystart
    set page 1
    set pages {}
    set lst {}
    
    #incr width -1
    
    foreach line [dialog::splitText $text $width] {
	if {$t > 360} {
	    # make another page
	    eval lappend pages -n [list "Page $page"] $args
	    eval lappend lst [list $args]
	    set args {}
	    incr page
	    if {$t > $height} {set height $t}
	    set t $ystart
	}
	eval [list lappend args] [dialog::text $line $x t $width]
	incr t -10
    }
    if {$t > $height} {set height $t}
    if {$page > 1} {
	set t $height
	set height [expr {$t + 40}]
	for {set i 1} {$i <= $page} {incr i} {
	    lappend names "Page $i"
	}
	eval lappend pages [list -n "Page $page"] $args

	return [concat [list -m [concat [list [lindex $names 0]] $names] 400 10 475 30] $pages]
    } else {
	return $args
    }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "dialog::text" --
 # 
 #  Creates a text box wrapping etc the text to fit appropriately.
 #  In the input text 'name', "\r" is used as a paragraph delimiter,
 #  and "\n" is used to force a linebreak.  Paragraphs have a wider
 #  spread.
 # -------------------------------------------------------------------------
 ##
proc dialog::text {name x yy {split 0} args} {
    upvar $yy y
    if {$split <= 0 || ![string length $name]} {
	if {$split < 0} {
	    set height [expr {15*[lindex $args 0]}]
	    set res [list -t $name $x $y [expr {$x - $split}] \
	      [incr y $height]]
	    incr y 3
	} else {
	    # 16 pixels high
	    set res [list -t $name $x $y [expr {$x + 8 * [string length $name]}] \
	      [incr y 16]]
	    # 6 pixel gap from baseline to next element (-4 pixel descender)
	    incr y 2
	}
    } else {
	global fillColumn dialog::strlength
	if {[info exists fillColumn]} {
	    set f $fillColumn
	}
	set fillColumn $split
	set name [string trim $name]
	set paragraphList [split $name "\r"]
	foreach para $paragraphList {
	    set lines ""
	    foreach line [split $para "\n"] {
		lappend lines [breakIntoLines $line]
	    }
	    set lines [join $lines "\r"]
	    set curline {}
	    set curlinecount 0
	    set curmax 0
	    foreach line [split $lines "\r"] {
		# Each '-t' dialog item can only be 255 characters long, and in Alpha 7
		# there are a limited number or total possible dialog items, so we try
		# to squash as much as possible into one -t item.
		if {([string length $curline] + [string length $line]) < [set dialog::strlength]} {
		    if {[string length $curline]} {append curline "\r"}
		    append curline $line
		    incr curlinecount
		    #set xx [dialog::_reqWidth $line]
		    set xx [expr {8 * [string length $line]}]
		    if {$xx > $curmax} { set curmax $xx }
		} else {
		    eval lappend res [list -t $curline $x $y \
		      [expr {$x + 4 + $curmax}] [expr {$y + 16 * $curlinecount -3}]]
		    incr y [expr {16 * $curlinecount}]
		    set curline $line
		    set curlinecount 1
		    #set curmax [dialog::_reqWidth $line]
		    set curmax [expr {8 * [string length $line]}]
		}
	    }
	    # handle the last item.
	    eval lappend res [list -t $curline $x $y \
	      [expr {$x + 4 + $curmax}] [expr {$y + 17 * $curlinecount -3}]]
	    incr y [expr {16 * $curlinecount}]
	    incr y 10
	}
	if {[info exists f]} {
	    set fillColumn $f
	} else {
	    unset fillColumn
	}
	if {![info exists res]} {
	    set res [list -t $name $x $y [expr {$x + 7 * [string length $name]}] \
	      [expr {$y +15}]]
	    incr y 18
	}
    }
    return $res
}

proc dialog::splitText {name split} {
    global fillColumn
    if {[info exists fillColumn]} {
	set f $fillColumn
    }
    set fillColumn $split
    set name [string trim $name]
    set paragraphList [split $name "\r"]
    
    set res [list]
    
    foreach para $paragraphList {
	set lines ""
	foreach line [split $para "\n"] {
	    lappend lines [breakIntoLines $line]
	}
	set lines [join $lines "\r"]
	eval [list lappend res] [split $lines "\r"]
    }
    if {[info exists f]} {
	set fillColumn $f
    } else {
	unset fillColumn
    }
    return $res
}

proc dialog::edit {name x yy chars {rows 1}} {
    upvar $yy y
    set res [list -e $name $x $y [expr {$x + 10 * $chars}] [expr {$y + 15 * $rows}]]
    incr y [expr {5 + 15*$rows}]
    return $res
}
proc dialog::textedit {name default x yy chars {height 1} {horiz 0}} {
    upvar $yy y
    set xx [dialog::_reqWidth $name]
    set res [list -t $name $x $y [expr {$x + $xx}]\
      [expr {$y +16}] -e $default]
    if {$horiz} {
	incr x $horiz
    } else {
	incr y 22
    }
    lappend res $x $y [expr {$x + 10 * $chars}] \
      [expr {$y + 16*$height}]
    incr y [expr {9 + 16*$height}]
    return $res
}

if {[info tclversion] < 8.0} {
    proc dialog::checkbox {name default x yy} {
	upvar $yy y
	set res [list -c $name $default $x $y]
	set wd [dialog::_reqWidth $name]
	incr wd 20
	if {[string length $name] < 5} {incr wd 10}
	lappend res [expr {$x + $wd}] [expr {$y +15}]
	incr y 18
	return $res
    }
} else {
    if {${alpha::platform} == "alpha"} {
	proc dialog::checkbox {name default x yy} {
	    upvar $yy y
	    set res [list -c $name $default -font 2 $x $y]
	    lappend res [expr {$x + 20 + [dialog::_reqWidth $name]}] [expr {$y +15}]
	    incr y 18
	    return $res
	}
    } else {
	proc dialog::checkbox {name default x yy} {
	    upvar $yy y
	    set res [list -c $name $default -font 2 $x $y]
	    lappend res [expr {$x + [dialog::_reqWidth $name]}] [expr {$y +15}]
	    incr y 18
	    return $res
	}
    }
}

if {${alpha::platform} == "alpha"} {
    proc dialog::_reqWidth {args} {
	set w 0
	foreach name $args {
	    set c [regsub -all -nocase {[wm]} $name "" ""]
	    set d [regsub -all {[ iIl',;:.]} $name "" ""]
	    set len [expr {10 * [string length $name] + 6 * $c - 5 * $d}]
	    if {[string length $name] < 7} {incr len 6}
	    if {$len > $w} {
		set w $len
	    }
	}
	return $w
    }
    # new version
    proc dialog::_reqWidth {args} {
       set width 0
       foreach str $args {
	  set w [dialog::text_width $str]
	  if {$w>$width} then {set width $w}
       }
       set width
    }
} else {
    proc dialog::_reqWidth {args} {return 0}
}


#  Manipulation of special pref types  #

namespace eval dialog::specialView {}
namespace eval dialog::specialSet {}

proc dialog::specialView::binding {key} {
    append key1 [keys::modifiersTo $key "verbose"]
    append key1 [keys::verboseKey $key]
    if {$key1 == ""} { return "<no binding>" }
    return $key1
}

proc dialog::specialSet::binding {mod v {menu 0}} {
    # Set pressed
    set oldB [dialog::getFlag $mod $v]
    if {![catch {dialog::getAKey [quote::Prettify $v] $oldB $menu} newKey] && $newKey != $oldB} {
	dialog::modified $mod $v $newKey binding
    }
}

proc dialog::specialSet::sig {mod v} {
    set old [dialog::getFlag $mod $v]
    set newval [dialog::findApp $v "" $old]
    if {($newval != "") && ($newval != $old)} {
	dialog::modified $mod $v $newval Sig
    }
}

proc dialog::specialSet::folder {mod v} {
    global alpha::platform
    if {${alpha::platform} == "alpha"} {
	if {![catch {get_directory -p "New [quote::Prettify $v]"} newval]} {
	    dialog::modified $mod $v $newval folder
	}
    } else {
	set old [dialog::getFlag $mod $v]
	if {![catch {get_directory -p "New [quote::Prettify $v]" $old} newval]} {
	    dialog::modified $mod $v $newval folder
	}
    }
}

proc dialog::specialView::menubinding {key} {
    dialog::specialView::binding $key
}

proc dialog::specialSet::menubinding {mod v} {
    dialog::specialSet::binding $mod $v 1
}

proc dialog::specialView::searchpath {vv} {
    if {[llength $vv]} {
	foreach ppath $vv {
	    lappend view [dialog::specialView::file $ppath]
	}
	return [join $view "\r"]
    } else {
	return "No search paths currently set."
    }
}

proc dialog::specialView::sig {vv} {
    if {$vv != ""} {
	if {[catch {nameFromAppl $vv} path]} {
	    return "Unknown application with sig '$vv'"
	} else {
	    return [dialog::specialView::file $path]
	}
    }
    return ""
}

proc dialog::specialView::folder {vv} {
    dialog::specialView::file $vv
}

proc dialog::specialView::io-file {vv} {
    dialog::specialView::file $vv
}
proc dialog::specialView::File {vv {maxLen 33}} {
    if {[info tclversion] < 8} {
	set legs [lreverse [file split $vv]]
    } else {
	set legs [lreverse [::file split $vv]]
    }
    set name [lindex $legs 0]
    set root [lindex $legs end]
    set legs [lreplace $legs 0 0]
    set legs [lreplace $legs end end] 
    
    incr maxLen [expr {-[string length $name]-1}]
    incr maxLen [expr {-[string length $root]-1}]
    
    set path {}

    foreach leg $legs {
	if {[string length $leg] <= $maxLen} {
	    lappend path $leg
	    incr maxLen [expr {-[string length $leg]}]
	} else {
	    lappend path ""
	    break
	}
    }
    if {[info tclversion] < 8} {
	return [eval file join $root [lreverse $path] $name]
    } else {
	return [eval ::file join $root [lreverse $path] $name]
    }
}
proc dialog::specialView::file {vv} {
    if {[set sl [string length $vv]] > 33} {
	set vv "[string range $vv 0 8]...[string range $vv [expr {$sl -21}] end]"
    }
    return $vv
}
proc dialog::specialView::url {vv} {
    if {[set sl [string length $vv]] > 33} {
	set vv "[string range $vv 0 8]...[string range $vv [expr {$sl -21}] end]"
    }
    return $vv
}
proc dialog::specialView::date {vv} {
    return [clock format $vv]
}

proc dialog::specialSet::file {mod v} {
    # Set pressed
    set old [dialog::getFlag $mod $v]
    if {![catch {getfile [quote::Prettify "New $v"] $old} ff] \
      && $ff != $old} {
	dialog::modified $mod $v $ff file
    }
}
proc dialog::specialSet::url {mod v} {
    # Set pressed
    set old [dialog::getFlag $mod $v]
    if {![catch {dialog::getUrl "New URL for [quote::Prettify $v]" $old} ff] \
      && $ff != $old} {
	dialog::modified $mod $v $ff url
    }
}
proc dialog::specialSet::date {mod v} {
    # Set pressed
    set old [dialog::getFlag $mod $v]
    if {![catch {dialog::getDate "New Date for [quote::Prettify $v]" $old} ff] \
      && $ff != $old} {
	dialog::modified $mod $v $ff date
    }
}

proc dialog::specialSet::io-file {mod v} {
    # Set pressed
    set old [dialog::getFlag $mod $v]
    if {![catch {putfile [quote::Prettify "New $v"] $old} ff] \
      && $ff != $old} {
	dialog::modified $mod $v $ff io-file
    }
}
proc dialog::specialSet::searchpath {mod v} {
    # Set pressed
    global flag::extraOptions
    if {[info exists flag::extraOptions($v)]} {
	set options [list "Add File" "Add Directory"]
    } else {
	set options [list "Add"]
    }
    lappend options "Remove" "Change" "Cancel"
    set res [eval [list buttonAlert "Perform what action to one of\
      the [quote::Prettify $v]s"] $options]
    switch -- $res {
	"Add File" {
	    # add a file
	    if {![catch {getfile -types [set flag::extraOptions($v)] \
	      "New [quote::Prettify $v]"} newval]} {
		set newval [concat [dialog::getFlag $mod $v] [list $newval]] 
		dialog::modified $mod $v $newval searchpath
	    }
	}
	"Add" -
	"Add Directory" {
	    # add a directory
	    if {![catch {get_directory -p "New [quote::Prettify $v]"} newval]} {
		set newval [concat [dialog::getFlag $mod $v] [list $newval]] 
		dialog::modified $mod $v $newval searchpath
	    }
	}
	"Remove" {
	    if {![catch {set remove [listpick -p \
	      "Remove which items from [quote::Prettify $v]" -l [dialog::getFlag $mod $v]]}]} {
		# remove them
		set newval [lremove -l [dialog::getFlag $mod $v] $remove] 
		dialog::modified $mod $v $newval searchpath
	    }
	}
	"Change" {
	    if {![catch {set change [listpick -p \
	      "Change which item from [quote::Prettify $v]" [dialog::getFlag $mod $v]]}]} {
		# change it
		if {[file isdirectory $change]} {
		    if {![catch {get_directory -p "Replacement [quote::Prettify $v]:"} newval]} {
			set old [dialog::getFlag $mod $v]
			set i [lsearch -exact $old $change]
			set old [lreplace $old $i $i $newval]
			dialog::modified $mod $v $old searchpath
		    }
		} else {
		    if {![catch {getfile "Replacement [quote::Prettify $v]:"} newval]} {
			set old [dialog::getFlag $mod $v]
			set i [lsearch -exact $old $change]
			set old [lreplace $old $i $i $newval]
			dialog::modified $mod $v $old searchpath
		    }
		}
	    }
	}
    }
}




